/*
 * Seven-segment LED emulator
 *
 * Stephen A. Edwards, Columbia University
 */

module VGA_LED_Emulator(
 input logic 	    clk50, reset,
// input logic [10:0]  soft_in_x, soft_in_y,
// input reg [1:0] array [0:100];

 /*input logic [3:0] first_sel, second_sel, third_sel,  //selection input 
 input logic first_used, second_used, third_used, //if used input
 input logic [1:0] arrow_sel,*/
 
 input logic [7:0] array_address,
 input logic [7:0] array_value,
 input logic [7:0] first_sel,second_sel,third_sel,arrow_sel, score1, score2, time1, time2, time3, finish,
 input write_en,
 output logic [7:0] VGA_R, VGA_G, VGA_B,
 output logic 	    VGA_CLK, VGA_HS, VGA_VS, VGA_BLANK_n, VGA_SYNC_n);

/*
 * 640 X 480 VGA timing for a 50 MHz clock: one pixel every other cycle
 * 
 * HCOUNT 1599 0             1279       1599 0
 *             _______________              ________
 * ___________|    Video      |____________|  Video
 * 
 * 
 * |SYNC| BP |<-- HACTIVE -->|FP|SYNC| BP |<-- HACTIVE
 *       _______________________      _____________
 * |____|       VGA_HS          |____|
 */
   // Parameters for hcount
   parameter HACTIVE      = 11'd 1280,
             HFRONT_PORCH = 11'd 32,
             HSYNC        = 11'd 192,
             HBACK_PORCH  = 11'd 96,   
             HTOTAL       = HACTIVE + HFRONT_PORCH + HSYNC + HBACK_PORCH; // 1600
   
   // Parameters for vcount
   parameter VACTIVE      = 10'd 480,
             VFRONT_PORCH = 10'd 10,
             VSYNC        = 10'd 2,
             VBACK_PORCH  = 10'd 33,
             VTOTAL       = VACTIVE + VFRONT_PORCH + VSYNC + VBACK_PORCH; // 525

   logic [10:0]			     hcount; // Horizontal counter
                                             // Hcount[10:1] indicates pixel column (0-639)
   logic 			     endOfLine;
   
   always_ff @(posedge clk50 or posedge reset)
     if (reset)          hcount <= 0;
     else if (endOfLine) hcount <= 0;
     else  	         hcount <= hcount + 11'd 1;

   assign endOfLine = hcount == HTOTAL - 1;

   // Vertical counter
   logic [9:0] 			     vcount;
   logic 			     endOfField;
   
   always_ff @(posedge clk50 or posedge reset)
     if (reset)          vcount <= 0;
     else if (endOfLine)
       if (endOfField)   vcount <= 0;
       else              vcount <= vcount + 10'd 1;

   assign endOfField = vcount == VTOTAL - 1;

   // Horizontal sync: from 0x520 to 0x5DF (0x57F)
   // 101 0010 0000 to 101 1101 1111
   assign VGA_HS = !( (hcount[10:8] == 3'b101) & !(hcount[7:5] == 3'b111));
   assign VGA_VS = !( vcount[9:1] == (VACTIVE + VFRONT_PORCH) / 2);

   assign VGA_SYNC_n = 1; // For adding sync to video signals; not used for VGA
   
   // Horizontal active: 0 to 1279     Vertical active: 0 to 479
   // 101 0000 0000  1280	       01 1110 0000  480
   // 110 0011 1111  1599	       10 0000 1100  524
   assign VGA_BLANK_n = !( hcount[10] & (hcount[9] | hcount[8]) ) &
			!( vcount[9] | (vcount[8:5] == 4'b1111) );   

   /* VGA_CLK is 25 MHz
    *             __    __    __
    * clk50    __|  |__|  |__|
    *        
    *             _____       __
    * hcount[0]__|     |_____|
    */
   assign VGA_CLK = hcount[0]; // 25 MHz clock: pixel latched on rising edge
   //hardware and software distance
  // logic[10:0] soft_hard_x, soft_hard_y;
	
	//assign soft_hard_x = (soft_in_x >= hcount)? ((soft_in_x - hcount)>>1): ((hcount - soft_in_x)>>1);
//	assign soft_hard_y = (soft_in_y >= vcount)? (soft_in_y - vcount): (vcount - soft_in_y);


		/////logic of select////
	logic grid; //all grid area
	assign grid = ((hcount[10:6] >= 5'd2)&(hcount[10:6]<= 5'd11)) & ((vcount[9:0]>= 10'd64)&(vcount[9:0]<=10'd383)); 
	//h:000 1000 0000  v:00 0100 0000
	//  011 0000 0000    01 1000 0000


	///////calculate read data address///////
	logic [7:0] read_address;
	logic [9:0] num_add;
	logic [9:0] time_add;
	logic [23:0] data_num0;
	logic [23:0] data_num1;
	logic [23:0] data_num2;
	logic [23:0] data_num3;
	logic [23:0] data_num4;
	logic [23:0] data_num5;
	logic [23:0] data_num6;
	logic [23:0] data_num7;
	logic [23:0] data_num8;
	logic [23:0] data_num9;


	
	//////color bits ////
	logic [7:0] color_data;
	logic color_on;

	////memory connection//////
	grid_array grid_array(.clock(clk50),.data(array_value),.wraddress(array_address),.rdaddress(read_address),.q(color_data),.wren(write_en));
	num0 num0 (.clock(clk50), .address(num_add), .q(data_num0));
	num1 num1 (.clock(clk50), .address(num_add), .q(data_num1));
	num2 num2 (.clock(clk50), .address(num_add), .q(data_num2));
	num3 num3 (.clock(clk50), .address(num_add), .q(data_num3));
	num4 num4 (.clock(clk50), .address(num_add), .q(data_num4));
	num5 num5 (.clock(clk50), .address(num_add), .q(data_num5));
	num6 num6 (.clock(clk50), .address(num_add), .q(data_num6));
	num7 num7 (.clock(clk50), .address(num_add), .q(data_num7));
	num8 num8 (.clock(clk50), .address(num_add), .q(data_num8));
	num9 num9 (.clock(clk50), .address(num_add), .q(data_num9));

	
	////grid controller///
	always_comb
		begin 
		read_address <= 0;	
		color_on <=0;
		if(grid) begin
			read_address <= (hcount[10:6]-2)  +(vcount[9:5]-2)*10;	
			color_on <= 1;
		end

	end 

	/////score controller/////
	logic[23:0] num_1000,num_100,num_10,num_1, clock_100, clock_10, clock_1;
	logic num_1000_on,num_100_on, num_10_on, num_1_on, clock_100_on, clock_10_on, clock_1_on;
	/*
	always_comb begin
		num_1000 <= data_num0;
		num_100 <= data_num0;
		num_10 <= data_num0;
		num_1 <= data_num0;
	end
	*/
	
	always_comb begin
		case (score1[7:4])
		4'h9: num_1000 <= data_num9;		
		4'h8: num_1000 <= data_num8;
		4'h7: num_1000 <= data_num7; 
		4'h6: num_1000 <= data_num6;
		4'h5: num_1000 <= data_num5;
		4'h4: num_1000 <= data_num4;
		4'h3: num_1000 <= data_num3;
		4'h2: num_1000 <= data_num2;
		4'h1: num_1000 <= data_num1;				
		4'h0: num_1000 <= data_num0;
		default: num_1000 <= data_num0;
		endcase

		case (score1[3:0])
		4'h9: num_100 <= data_num9;		
		4'h8: num_100 <= data_num8;
		4'h7: num_100 <= data_num7;
		4'h6: num_100 <= data_num6;
		4'h5: num_100 <= data_num5;
		4'h4: num_100 <= data_num4;
		4'h3: num_100 <= data_num3;
		4'h2: num_100 <= data_num2;
		4'h1: num_100 <= data_num1;				
		4'h0: num_100 <= data_num0;
		default: num_100 <= data_num0;
		endcase
		
		case (score2[7:4])
		4'h9: num_10 <= data_num9;		
		4'h8: num_10 <= data_num8;
		4'h7: num_10 <= data_num7;
		4'h6: num_10 <= data_num6;
		4'h5: num_10 <= data_num5;
		4'h4: num_10 <= data_num4;
		4'h3: num_10 <= data_num3;
		4'h2: num_10 <= data_num2;
		4'h1: num_10 <= data_num1;				
		4'h0: num_10 <= data_num0;
		default: num_10 <= data_num0;
		endcase

		case (score2[3:0])
		4'h9: num_1 <= data_num9;		
		4'h8: num_1 <= data_num8;
		4'h7: num_1 <= data_num7;
		4'h6: num_1 <= data_num6;
		4'h5: num_1 <= data_num5;
		4'h4: num_1 <= data_num4;
		4'h3: num_1 <= data_num3;
		4'h2: num_1 <= data_num2;
		4'h1: num_1 <= data_num1;				
		4'h0: num_1 <= data_num0;
		default: num_1 = data_num0;
		endcase
	end

	always_comb begin
		case (time1[3:0])
		4'h9: clock_100 <= data_num9;		
		4'h8: clock_100 <= data_num8;
		4'h7: clock_100 <= data_num7;
		4'h6: clock_100 <= data_num6;
		4'h5: clock_100 <= data_num5;
		4'h4: clock_100 <= data_num4;
		4'h3: clock_100 <= data_num3;
		4'h2: clock_100 <= data_num2;
		4'h1: clock_100 <= data_num1;				
		4'h0: clock_100 <= data_num0;
		default: clock_100 <= data_num0;
		endcase
		
		case (time2[3:0])
		4'h9: clock_10 <= data_num9;		
		4'h8: clock_10 <= data_num8;
		4'h7: clock_10 <= data_num7;
		4'h6: clock_10 <= data_num6;
		4'h5: clock_10 <= data_num5;
		4'h4: clock_10 <= data_num4;
		4'h3: clock_10 <= data_num3;
		4'h2: clock_10 <= data_num2;
		4'h1: clock_10 <= data_num1;				
		4'h0: clock_10 <= data_num0;
		default: clock_10 = data_num0;
		endcase
		
		case (time3[3:0])
		4'h9: clock_1 <= data_num9;		
		4'h8: clock_1 <= data_num8;
		4'h7: clock_1 <= data_num7;
		4'h6: clock_1 <= data_num6;
		4'h5: clock_1 <= data_num5;
		4'h4: clock_1 <= data_num4;
		4'h3: clock_1 <= data_num3;
		4'h2: clock_1 <= data_num2;
		4'h1: clock_1 <= data_num1;				
		4'h0: clock_1 <= data_num0;
		default: clock_1 <= data_num0;
		endcase
	end
	
/*	
	
	
	
	always_comb begin 

		
		if ((hcount[10:0]>=500)&(hcount[10:0]<=545)&(vcount[9:0]>=416)&(vcount[9:0]<=447))
			begin
				num_add <= (hcount[10:1]-250)+(vcount[4:0])*23;
				clock_100_on <= 1;
				clock_10_on <= 0;
				clock_1_on <=0;
			end
			
		else	if ((hcount[10:0]>=546)&(hcount[10:0]<=591)&(vcount[9:0]>=416)&(vcount[9:0]<=447))
			begin
				num_add <= (hcount[10:1]-273)+(vcount[4:0])*23;
				clock_100_on <= 0;
				clock_10_on <= 1;
				clock_1_on <=0;
			end	
			
		else	if ((hcount[10:0]>=592)&(hcount[10:0]<=637)&(vcount[9:0]>=416)&(vcount[9:0]<=447))
			begin
				num_add <= (hcount[10:1]-296)+(vcount[4:0])*23;
				clock_100_on <= 0;
				clock_10_on <= 0;
				clock_1_on <=1;
			end	
		else
			begin
				clock_100_on <=0;
				clock_10_on <=0;
				clock_1_on <=0;
				num_add <= 0;
			end
		end	
			*/
			
			
			
	always_comb begin
			num_1000_on <= 0;
			num_100_on <= 0;
			num_10_on <= 0;
			num_1_on <= 0;
			num_add <= 0;
			clock_100_on <=0;
			clock_10_on <=0;
			clock_1_on <=0;

		if((hcount[10:0]>=192)&(hcount[10:0]<=237)&(vcount[9:0]>=416)&(vcount[9:0]<=447))
			begin
			num_add <= (hcount[10:1]-96)+(vcount[4:0])*23;
			num_1000_on <= 1;
			num_100_on <= 0;
			num_10_on <= 0;
			num_1_on <= 0;
			end
			
		else if ((hcount[10:0]>=238)&(hcount[10:0]<=283)&(vcount[9:0]>=416)&(vcount[9:0]<=447))
			begin
			num_add <= (hcount[10:1]-119)+(vcount[4:0])*23;
			num_1000_on <= 0;
			num_100_on <= 1;
			num_10_on <= 0;
			num_1_on <= 0;
			end
		else if ((hcount[10:0]>=284)&(hcount[10:0]<=329)&(vcount[9:0]>=416)&(vcount[9:0]<=447))
			begin
			num_add <= (hcount[10:1]-142)+(vcount[4:0])*23;
			num_1000_on <= 0;
			num_100_on <= 0;
			num_10_on <= 1;
			num_1_on <= 0;
			end
		else if ((hcount[10:0]>=330)&(hcount[10:0]<=375)&(vcount[9:0]>=416)&(vcount[9:0]<=447))
			begin
			num_add <= (hcount[10:1]- 165)+(vcount[4:0])*23;
			num_1000_on <= 0;
			num_100_on <= 0;
			num_10_on <= 0;
			num_1_on <= 1;
			end
			
		else if ((hcount[10:0]>=500)&(hcount[10:0]<=545)&(vcount[9:0]>=416)&(vcount[9:0]<=447))
			begin
				num_add <= (hcount[10:1]-250)+(vcount[4:0])*23;
				clock_100_on <= 1;
				clock_10_on <= 0;
				clock_1_on <=0;
			end
			
		else	if ((hcount[10:0]>=546)&(hcount[10:0]<=591)&(vcount[9:0]>=416)&(vcount[9:0]<=447))
			begin
				num_add <= (hcount[10:1]-273)+(vcount[4:0])*23;
				clock_100_on <= 0;
				clock_10_on <= 1;
				clock_1_on <=0;
			end	
			
		else	if ((hcount[10:0]>=592)&(hcount[10:0]<=637)&(vcount[9:0]>=416)&(vcount[9:0]<=447))
			begin
				num_add <= (hcount[10:1]-296)+(vcount[4:0])*23;
				clock_100_on <= 0;
				clock_10_on <= 0;
				clock_1_on <=1;
			end	
	end



	
	logic [5:0] linex; // coordinates for drawing the black line
	logic [4:0] liney;

	assign linex = hcount[6:1];
	assign liney = vcount[4:0];

	logic line; // vertical and horizontal line

	assign line = ((linex[4:0] == 5'd31) | (liney[4:0] == 5'b11111)); 
	//assign line = (linex[5:0] == 6'b111111);

	//14 difference kinds of tetris
	//central for each tetris

	logic one_range;
	logic two_range;
	logic three_range;

	assign one_range = ((vcount[8:0] >= 9'd40)&(vcount[8:0] <= 9'd120));
	assign two_range = ((vcount[8:0]>= 9'd200)&(vcount[8:0]<=9'd280));
	assign three_range = ((vcount[8:0]>= 9'd360)&(vcount[8:0]<=9'd440));
	
	
	logic tetris_line_v;
	logic tetris_line_h;

	assign tetris_line_v = (hcount[4:0] == 5'd0);
	assign tetris_line_h = (vcount[3:0] == 4'd8);
	
		/*896 928 960 992 1024 1055
	
 	 _ _ _ _ _   40 (200) (360)
	|_|_|_|_|_|  56 (216) (376)
	|_|_|_|_|_|  72 (232) (392)
	|_|_|_|_|_|  88 (248) (408)
	|_|_|_|_|_|  104 (264) (424)
	|_|_|_|_|_|  120 (280) (440)
	
	reference of the block 
*/	
	
	//1*5
	logic one_five_x;
	logic one_five_y;

	assign one_five_x = ((hcount[10:0]>=11'd896)&(hcount[10:0]<=11'd1056));
	assign one_five_y = ((vcount[8:0]>=9'd72)&(vcount[8:0]<=9'd88)) | ((vcount[8:0]>=9'd232)&(vcount[8:0]<=9'd248)) | ((vcount[8:0]>=9'd392)&(vcount[8:0]<=9'd408));

	//1*4
	logic one_four_x; 
	logic one_four_y;
	
	assign one_four_x =((hcount[10:0]>=11'd896)&(hcount[10:0]<=11'd1024));
	assign one_four_y = ((vcount[8:0]>=9'd72)&(vcount[8:0]<=9'd88)) | ((vcount[8:0]>=9'd232)&(vcount[8:0]<=9'd248)) | ((vcount[8:0]>=9'd392)&(vcount[8:0]<=9'd408));

	//5*1
	logic five_one_x;
	logic five_one_y;
	
	assign five_one_x = ((hcount[10:0]>=11'd960)&(hcount[10:0]<=11'd992));
	assign five_one_y = ((vcount[8:0]>=9'd40)&(vcount[8:0]<=9'd120)) | ((vcount[8:0]>=9'd200)&(vcount[8:0]<=9'd280)) | ((vcount[8:0]>=9'd360)&(vcount[8:0]<=9'd440));

	//4*1
	logic four_one_x;
	logic four_one_y;
	
	assign four_one_x = ((hcount[10:0]>=11'd960)&(hcount[10:0]<=11'd992));
	assign four_one_y = ((vcount[8:0]>=9'd56)&(vcount[8:0]<=9'd120)) | ((vcount[8:0]>=9'd216)&(vcount[8:0]<=9'd280)) | ((vcount[8:0]>=9'd376)&(vcount[8:0]<=9'd440));
	
	//square
	logic two_two_x;
	logic two_two_y;
	
	assign two_two_x = ((hcount[10:0]>=11'd928)&(hcount[10:0]<=11'd992));
	assign two_two_y = ((vcount[8:0]>=9'd72)&(vcount[8:0]<=9'd104)) | ((vcount[8:0]>=9'd232)&(vcount[8:0]<=9'd264)) | ((vcount[8:0]>=9'd392)&(vcount[8:0]<=9'd424));
	
	// inverse_7
	logic invert_7_top;
	logic invert_7_bottom;
	
	assign invert_7_top = ((hcount[10:0]>=11'd928)&(hcount[10:0]<=11'd1024)& (((vcount[8:0]>=9'd56)&(vcount[8:0]<=9'd72)) | ((vcount[8:0]>=9'd216)&(vcount[8:0]<=9'd232)) | ((vcount[8:0]>=9'd376)&(vcount[8:0]<=9'd392))));
	assign invert_7_bottom = ((hcount[10:0]>=11'd928)&(hcount[10:0]<=11'd960)& (((vcount[8:0]>=9'd72)&(vcount[8:0]<=9'd104)) | ((vcount[8:0]>=9'd232)&(vcount[8:0]<=9'd264)) | ((vcount[8:0]>=9'd392)&(vcount[8:0]<=9'd424))));
	
	//seven
	logic seven_top;
	logic seven_bottom;
	
	assign seven_top = ((hcount[10:0]>=11'd928)&(hcount[10:0]<=11'd1024)& (((vcount[8:0]>=9'd56)&(vcount[8:0]<=9'd72)) | ((vcount[8:0]>=9'd216)&(vcount[8:0]<=9'd232)) | ((vcount[8:0]>=9'd376)&(vcount[8:0]<=9'd392))));
	assign seven_bottom = ((hcount[10:0]>=11'd992)&(hcount[10:0]<=11'd1024)& (((vcount[8:0]>=9'd72)&(vcount[8:0]<=9'd104)) | ((vcount[8:0]>=9'd232)&(vcount[8:0]<=9'd264)) | ((vcount[8:0]>=9'd392)&(vcount[8:0]<=9'd424))));
	
	//2*1
	logic two_one_x;
	logic two_one_y;
	
	assign two_one_x = ((hcount[10:0]>=11'd960)&(hcount[10:0]<=11'd992));
	assign two_one_y = ((vcount[8:0]>=9'd72)&(vcount[8:0]<=9'd104)) | ((vcount[8:0]>=9'd232)&(vcount[8:0]<=9'd264)) | ((vcount[8:0]>=9'd392)&(vcount[8:0]<=9'd424));
	
	//3*1
	logic three_one_x;
	logic three_one_y;
	
	assign three_one_x = ((hcount[10:0]>=11'd960)&(hcount[10:0]<=11'd992));
	assign three_one_y = ((vcount[8:0]>=9'd56)&(vcount[8:0]<=9'd104)) | ((vcount[8:0]>=9'd216)&(vcount[8:0]<=9'd264)) | ((vcount[8:0]>=9'd376)&(vcount[8:0]<=9'd424));
	
	//1*1
	logic one_one_x;
	logic one_one_y;
	
	assign one_one_x = ((hcount[10:0]>=11'd960)&(hcount[10:0]<=11'd992));
	assign one_one_y = ((vcount[8:0]>=9'd72)&(vcount[8:0]<=9'd88)) | ((vcount[8:0]>=9'd232)&(vcount[8:0]<=9'd248)) | ((vcount[8:0]>=9'd392)&(vcount[8:0]<=9'd408));
	
	//s
	logic s_top;
	logic s_bottom;
	
	assign s_top = ((hcount[10:0]>=11'd960)&(hcount[10:0]<=11'd1024)& (((vcount[8:0]>=9'd72)&(vcount[8:0]<=9'd88)) | ((vcount[8:0]>=9'd232)&(vcount[8:0]<=9'd248)) | ((vcount[8:0]>=9'd392)&(vcount[8:0]<=9'd408))));
	assign s_bottom = ((hcount[10:0]>=11'd960)&(hcount[10:0]<=11'd992)& (((vcount[8:0]>=9'd88)&(vcount[8:0]<=9'd104)) | ((vcount[8:0]>=9'd248)&(vcount[8:0]<=9'd264)) | ((vcount[8:0]>=9'd408)&(vcount[8:0]<=9'd424))));
	
	//invert_s
	logic invert_s_top;
	logic invert_s_bottom;
	
	assign invert_s_top = ((hcount[10:0]>=11'd928)&(hcount[10:0]<=11'd992)& (((vcount[8:0]>=9'd72)&(vcount[8:0]<=9'd88)) | ((vcount[8:0]>=9'd232)&(vcount[8:0]<=9'd248)) | ((vcount[8:0]>=9'd392)&(vcount[8:0]<=9'd408))));
	assign invert_s_bottom = ((hcount[10:0]>=11'd960)&(hcount[10:0]<=11'd1024)& (((vcount[8:0]>=9'd88)&(vcount[8:0]<=9'd104)) | ((vcount[8:0]>=9'd248)&(vcount[8:0]<=9'd264)) | ((vcount[8:0]>=9'd408)&(vcount[8:0]<=9'd424))));
	
	//1*3
	logic one_three_x;
	logic one_three_y;
	
	assign one_three_x =((hcount[10:0]>=11'd928)&(hcount[10:0]<=11'd1024));
	assign one_three_y = ((vcount[8:0]>=9'd72)&(vcount[8:0]<=9'd88)) | ((vcount[8:0]>=9'd232)&(vcount[8:0]<=9'd248)) | ((vcount[8:0]>=9'd392)&(vcount[8:0]<=9'd408));
	
	//1*2
	logic one_two_x;
	logic one_two_y;

	assign one_two_x =((hcount[10:0]>=11'd928)&(hcount[10:0]<=11'd992));
	assign one_two_y = ((vcount[8:0]>=9'd72)&(vcount[8:0]<=9'd88)) | ((vcount[8:0]>=9'd232)&(vcount[8:0]<=9'd248)) | ((vcount[8:0]>=9'd392)&(vcount[8:0]<=9'd408));

	//arrow
	logic arrow_one_x;
	logic arrow_one_y;
	
	assign arrow_one_x = ((hcount[10:0]>= 1120-(vcount-71)*2)&(hcount[10:0]<=1120));
	assign arrow_one_y = ((vcount[8:0]>= 9'd72)&(vcount[8:0]<=9'd88));
	
	logic arrow_two_x;
	logic arrow_two_y;
	
	assign arrow_two_x = ((hcount[10:0]>= 1120-(vcount-231)*2)&(hcount[10:0]<=1120));
	assign arrow_two_y = ((vcount[8:0]>= 9'd232)&(vcount[8:0]<=9'd248));
	
	logic arrow_three_x;
	logic arrow_three_y;
	
	assign arrow_three_x = ((hcount[10:0]>= 1120-(vcount-391)*2)&(hcount[10:0]<=1120));
	assign arrow_three_y = ((vcount[8:0]>= 9'd392)&(vcount[8:0]<=9'd408));

	
	////arrow controller//////
	logic curr_arrow;
	assign curr_arrow = arrow_sel == 8'd0 ? (arrow_one_x & arrow_one_y):
						     arrow_sel == 8'd1 ? (arrow_two_x & arrow_two_y):
						    (arrow_three_x & arrow_three_y);
	
	/////type select controller////
	logic cur_first;
	logic cur_second;
	logic cur_third;
	
	assign cur_first = first_sel == 8'd0 ? 0:
						first_sel == 8'd1 ? (one_five_x & one_five_y):
					   first_sel == 8'd3 ? (one_four_x & one_four_y):
					   first_sel == 8'd2 ? (five_one_x & five_one_y):
					   first_sel == 8'd4 ? (four_one_x & four_one_y):
					   first_sel == 8'd5 ? (two_two_x & two_two_y):
					   first_sel == 8'd6 ? (invert_7_top | invert_7_bottom):
					   first_sel == 8'd7 ? (seven_top | seven_bottom):
					   first_sel == 8'd8 ? (two_one_x & two_one_y):
					   first_sel == 8'd9 ? (three_one_x & three_one_y):
					   first_sel == 8'd10 ? (one_one_x & one_one_y):
					   first_sel == 8'd11 ? (s_top | s_bottom):
					   first_sel == 8'd12 ? (invert_s_top | invert_s_bottom):
					   first_sel == 8'd13 ? (one_three_x & one_three_y):
					   (one_two_x & one_two_y);

	assign cur_second = second_sel == 8'd0 ? 0:
						 second_sel == 8'd1 ? (one_five_x & one_five_y):
					    second_sel == 8'd3 ? (one_four_x & one_four_y):
					    second_sel == 8'd2 ? (five_one_x & five_one_y):
					    second_sel == 8'd4 ? (four_one_x & four_one_y):
					    second_sel == 8'd5 ? (two_two_x & two_two_y):
					    second_sel == 8'd6 ? (invert_7_top | invert_7_bottom):
					    second_sel == 8'd7 ? (seven_top | seven_bottom):
					    second_sel == 8'd8 ? (two_one_x & two_one_y):
					    second_sel == 8'd9 ? (three_one_x & three_one_y):
					    second_sel == 8'd10 ? (one_one_x & one_one_y):
					    second_sel == 8'd11 ? (s_top | s_bottom):
					    second_sel == 8'd12 ? (invert_s_top | invert_s_bottom):
					    second_sel == 8'd13 ? (one_three_x & one_three_y):
					    (one_two_x & one_two_y);
	
	
	assign cur_third = third_sel == 8'd0 ? 0:
						third_sel == 8'd1 ? (one_five_x & one_five_y):
					   third_sel == 8'd3 ? (one_four_x & one_four_y):
					   third_sel == 8'd2 ? (five_one_x & five_one_y):
					   third_sel == 8'd4 ? (four_one_x & four_one_y):
					   third_sel == 8'd5 ? (two_two_x & two_two_y):
					   third_sel == 8'd6 ? (invert_7_top | invert_7_bottom):
					   third_sel == 8'd7 ? (seven_top | seven_bottom):
					   third_sel == 8'd8 ? (two_one_x & two_one_y):
					   third_sel == 8'd9 ? (three_one_x & three_one_y):
					   third_sel == 8'd10 ? (one_one_x & one_one_y):
					   third_sel == 8'd11 ? (s_top | s_bottom):
					   third_sel == 8'd12 ? (invert_s_top | invert_s_bottom):
					   third_sel == 8'd13 ? (one_three_x & one_three_y):
					   (one_two_x & one_two_y);
						
	logic finish_on;
	
	always_comb begin	
		finish_on = 0;
		if(finish[7:0] ==8'd1)
			finish_on=1;
			
	end


	
	always_comb begin
	
		{VGA_R, VGA_G, VGA_B} = {8'hff, 8'hff, 8'hff}; // white
		
		if(finish_on)begin
			if (grid)
				{VGA_R, VGA_G, VGA_B} = {8'hff, 8'h00, 8'h00}; // 
			if(line & grid)
				{VGA_R, VGA_G, VGA_B} = {8'hff, 8'hff, 8'hff}; // grey      	
			if(curr_arrow)
			{VGA_R, VGA_G, VGA_B} = {8'h20, 8'h20, 8'h20}; //grey	
		end
		
		
				


				if (color_on) begin

					if (color_data==1)
						{VGA_R, VGA_G, VGA_B} = {8'h80, 8'h80, 8'h80}; //grey 
					else if (color_data==2)
						{VGA_R, VGA_G, VGA_B} = {8'h00, 8'hff, 8'h00}; // green
					else if (color_data==3)
						{VGA_R, VGA_G, VGA_B} = {8'hff, 8'h00, 8'h00}; //red
					else
						{VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'h00}; // black
				end
				
				
				
				if(clock_100_on)begin
					{VGA_R, VGA_G, VGA_B} = clock_100;
				end
				else if(clock_10_on)begin
					{VGA_R, VGA_G, VGA_B} = clock_10;
				end
				else if(clock_1_on)begin
					{VGA_R, VGA_G, VGA_B} = clock_1;
				end
				
				
				
				
				
				if(num_1000_on)begin
					{VGA_R, VGA_G, VGA_B} = num_1000;
				end
				else if(num_100_on)begin
					{VGA_R, VGA_G, VGA_B} = num_100;
				end
				else if(num_10_on)begin
					{VGA_R, VGA_G, VGA_B} = num_10;
				end
				else if(num_1_on)begin
					{VGA_R, VGA_G, VGA_B} = num_1;
				end
		
		
			  if(line & grid)
				{VGA_R, VGA_G, VGA_B} = {8'hff, 8'hff, 8'hff}; // grey    
		
			  if (cur_first & one_range)
					if(tetris_line_h | tetris_line_v)
						{VGA_R, VGA_G, VGA_B} = {8'hff, 8'hff, 8'hff}; //white
					else
						{VGA_R, VGA_G, VGA_B} = {8'h20, 8'h20, 8'h20}; //grey
					
		
			  if (cur_second & two_range)
					if(tetris_line_h | tetris_line_v)
						{VGA_R, VGA_G, VGA_B} = {8'hff, 8'hff, 8'hff}; //white
					else
						{VGA_R, VGA_G, VGA_B} = {8'h20, 8'h20, 8'h20}; //grey

			  if (cur_third & three_range)
					if(tetris_line_h | tetris_line_v)
						{VGA_R, VGA_G, VGA_B} = {8'hff, 8'hff, 8'hff}; //white
					else
						{VGA_R, VGA_G, VGA_B} = {8'h20, 8'h20, 8'h20}; //grey		
			
			  if(curr_arrow)
					{VGA_R, VGA_G, VGA_B} = {8'h20, 8'h20, 8'h20}; //grey	
	end
endmodule
