/*
 * Avalon memory-mapped peripheral for the VGA LED Emulator
 *
 * Stephen A. Edwards
 * Columbia University
 */

module VGA_LED(	input 	logic        		clk,
		       		input 	logic 	  			reset,
		       		input 	logic 	[31:0]  	writedata,
		       		input 	logic 		  		write,
				 	   input 	logic 	  			read,
		       		input 						   chipselect,
		       		input 	logic 	[2:0]  	address,
				 	   output 	logic 	[31:0]  	readdata);

						logic 	signed 	[31:0]  	   output_data;
						logic 			   [5:0] 		addr_write;
						logic 			   [5:0] 		addr_read;
						logic 			   [5:0] 		addr_dct_input;
						logic  			   [4:0]		   stages;

						// FLAGS
						logic 							read_before;
						logic 							input_loaded;
						logic 							dct_finished;
						logic 							dct1_finished;

						// 64 Addresses Memory Blocks
						logic 	signed 	[13:0] 	input_memory [63:0];
						logic 	signed 	[19:0] 	dct_memory   [63:0];
						logic   signed    [19:0]   zz_memory    [63:0];
						// DCT INPUT OUTPUT
						logic 	signed 	[13:0]	x1, x2, x3, x4, x5, x6, x7, x8;
						logic 	signed 	[19:0]	y1, y2, y3, y4, y5, y6, y7, y8;
						logic 	signed 	[19:0]	d1, d2, d3, d4, d5, d6, d7, d8;
					
						logic 	       	[31:0]	data_out_wire;
						logic    			[31:0]	neg;			
					
						// RLC
						logic signed [31:0] buckets [63:0];
						logic [5:0] count;
						logic [22:0] rlc [63:0];
						logic signed [19:0] data;
						logic [5:0]  rlc_index;
						logic [3:0]  zero_count;
						logic [3:0]  overhead;
						logic [5:0]  last_index;
						logic [19:0] data_new;
						logic [3:0]  category;
						logic [3:0]  category2;
						logic [3:0]  run;
						logic [14:0] rlc_data;
						logic data_before_zero;         // 1 if data before is zero, 0 if nonzero
						// HUFFMAN
						logic [5:0]  count_ac;
						logic [15:0] ac_codeword;
						logic [4:0]  ac_codeword_length;
						logic [31:0] bucket_out;
						logic [19:0] data_neg;
						logic [4:0]	bucket_length; 
						//*****FINAL OUTPUT *****//
						logic [511:0] final_out;
						logic [9:0]	final_length;
						assign run  = rlc[count_ac][22:19];
						assign category2 = rlc[count_ac][18:15];
						assign rlc_data = rlc[count_ac][14:0];
						assign bucket_out = (ac_codeword << category2) + rlc_data;
						assign bucket_length = category2 + ac_codeword_length;
						assign data = zz_memory[count];     
						assign data_neg = ~((~($unsigned(data))) + 1);
						assign data_new = (data >= 0)?data: 
												(data >= -1)?		 0:
												(data >= -3)?		({18'd0, data_neg[1:0]}):
												(data >= -7)?		({17'd0, data_neg[2:0]}):
												(data >= -15)?		({16'd0, data_neg[3:0]}):
												(data >= -31)?		({15'd0, data_neg[4:0]}):
												(data >= -63)?		({14'd0, data_neg[5:0]}):
												(data >= -127)?	({13'd0, data_neg[6:0]}):
												(data >= -255)?	({12'd0, data_neg[7:0]}):
												(data >= -511)?	({11'd0, data_neg[8:0]}):
												(data >= -1023)?	({10'd0, data_neg[9:0]}):
												(data >= -2047)?	({9'd0, data_neg[10:0]}): 0;
						
						getcategory gc0(data, category);
						ac_huffman_table ac1 (run, category2, ac_codeword, ac_codeword_length);	
						// DC 
						logic  signed [19:0]    data_old;
						logic  signed [19:0]	 dc_diff;
						logic  [8:0]     dc_codeword;
						logic  [4:0]     dc_codeword_length;
						assign dc_diff = zz_memory[0] - data_old;
						dc_huffman_table dc1 (category, dc_codeword, dc_codeword_length);
						logic [19:0] diff_neg;
						logic [19:0] diff_new;
						logic [19:0] diff_new_reg;
						assign diff_neg = ~((~($unsigned(dc_diff))) + 1);
						assign diff_new = (dc_diff >= 0)?dc_diff: 
												(dc_diff >= -1)?		 0:
												(dc_diff >= -3)?		({18'd0, diff_neg[1:0]}):
												(dc_diff >= -7)?		({17'd0, diff_neg[2:0]}):
												(dc_diff >= -15)?		({16'd0, diff_neg[3:0]}):
												(dc_diff >= -31)?		({15'd0, diff_neg[4:0]}):
												(dc_diff >= -63)?		({14'd0, diff_neg[5:0]}):
												(dc_diff >= -127)?	({13'd0, diff_neg[6:0]}):
												(dc_diff >= -255)?	({12'd0, diff_neg[7:0]}):
												(dc_diff >= -511)?	({11'd0, diff_neg[8:0]}):
												(dc_diff >= -1023)?	({10'd0, diff_neg[9:0]}):
												(dc_diff >= -2047)?	({9'd0, diff_neg[10:0]}): 100;	
												
						assign 	readdata = data_out_wire;
						//assign	data_out_wire = {12'b000000000000, zz_memory[addr_read]};
						//assign	data_out_wire = {9'b000000000, rlc[addr_read]};
						//assign	data_out_wire = buckets[addr_read];
						assign	data_out_wire = final_out[511:480];
						//assign data_out_wire = final_length;
						//assign	data_out_wire = zz_memory[addr_read];
						assign	x1 = (dct1_finished)?d1[13:0]:input_memory[addr_dct_input];	
						assign	x2 = (dct1_finished)?d2[13:0]:input_memory[addr_dct_input+1];
						assign	x3 = (dct1_finished)?d3[13:0]:input_memory[addr_dct_input+2];
						assign	x4 = (dct1_finished)?d4[13:0]:input_memory[addr_dct_input+3];
						assign	x5 = (dct1_finished)?d5[13:0]:input_memory[addr_dct_input+4];
						assign	x6 = (dct1_finished)?d6[13:0]:input_memory[addr_dct_input+5];
						assign	x7 = (dct1_finished)?d7[13:0]:input_memory[addr_dct_input+6];
						assign	x8 = (dct1_finished)?d8[13:0]:input_memory[addr_dct_input+7];
						
						assign 	d1 = dct_memory[addr_dct_input];
						assign 	d2 = dct_memory[addr_dct_input+1];
						assign 	d3 = dct_memory[addr_dct_input+2];
						assign 	d4 = dct_memory[addr_dct_input+3];
						assign 	d5 = dct_memory[addr_dct_input+4];
						assign 	d6 = dct_memory[addr_dct_input+5];
						assign 	d7 = dct_memory[addr_dct_input+6];
						assign 	d8 = dct_memory[addr_dct_input+7];
						dct1 DCT_1(clk, x1, x2, x3, x4, x5, x6, x7, x8, y1, y2, y3, y4, y5, y6, y7, y8);
			

		       	always_ff @(posedge clk) begin
			      	if (reset) begin
								addr_write 			<= 6'b000000;
								addr_read 			<= 6'b111111;
								addr_dct_input  	<= 6'd0;
								input_loaded 		<= 1'b0;
								stages 				<= 5'b00000;
								dct_finished 		<= 1'b0;
								dct1_finished 		<= 1'b0;
            					
								count               <=  0;
            				data_before_zero    <=  0;
            				rlc_index           <=  0;
            				zero_count          <=  0;
            				overhead            <=  0;
            				count_ac            <=  1;
								
								data_old 			  <=  0;
								diff_new_reg        <=  0;
								final_out			  <=  0;
								final_length		  <=  0;
			     		end else if (chipselect && write && !input_loaded) begin //loads inputs to buffer
								if (addr_write == 6'b111100) begin
									input_loaded <= 1'b1;
								end
								input_memory[addr_write] 	<= writedata[7:0] -   8'd128;
								input_memory[addr_write+1] <= writedata[15:8] -  8'd128;
								input_memory[addr_write+2] <= writedata[23:16] - 8'd128;
								input_memory[addr_write+3] <= writedata[31:24] - 8'd128;
								addr_write <= addr_write + 4;
				  		end else if(chipselect && read && !read_before) begin 
								read_before <= 1'b1;
								addr_read <= addr_read + 1;
								if (addr_read != 6'b111111) begin
									final_out <= (final_out << 32);
								end
						end else if (input_loaded) begin	// DCT
							if (stages == 5'd0) begin							// state 0  
									addr_dct_input <= 0;
									stages <= 5'd1;
									final_out			  <=  0;
									final_length		  <=  0;
							end else if(stages == 5'd1) begin				// state 1
									addr_dct_input <= 8;
									stages <= 5'd2;
							end else if(stages == 5'd2) begin				// state 2
									addr_dct_input <= 16;
									stages <= 5'd3;
							end else if(stages == 5'd3) begin				// state 3
									addr_dct_input <= 24;
									stages <= 5'd4;
							end else if(stages == 5'd4) begin				// state 4
									addr_dct_input 	<= 32;
									dct_memory[0] 	<= y1;
									dct_memory[8] 	<= y2;
									dct_memory[16] 	<= y3;
									dct_memory[24] 	<= y4;
									dct_memory[32] 	<= y5;
									dct_memory[40] 	<= y6;
									dct_memory[48] 	<= y7;
									dct_memory[56] 	<= y8;
									stages <= 5'd5;
							end else if(stages == 5'd5) begin				// state 5
									addr_dct_input 	<= 40;
									dct_memory[1] 	<= y1;
									dct_memory[9] 	<= y2;
									dct_memory[17] 	<= y3;
									dct_memory[25] 	<= y4;
									dct_memory[33] 	<= y5;
									dct_memory[41] 	<= y6;
									dct_memory[49] 	<= y7;
									dct_memory[57] 	<= y8;
									stages <= 5'd6;
							end else if(stages == 5'd6) begin				// state 6
									addr_dct_input 	<= 48;
									dct_memory[2] 	<= y1;
									dct_memory[10] 	<= y2;
									dct_memory[18] 	<= y3;
									dct_memory[26] 	<= y4;
									dct_memory[34] 	<= y5;
									dct_memory[42] 	<= y6;
									dct_memory[50] 	<= y7;
									dct_memory[58] 	<= y8;
									stages <= 5'd7;
							end else if(stages == 5'd7) begin				// state 7
									addr_dct_input 	<= 56;
									dct_memory[3] 	<= y1;
									dct_memory[11] 	<= y2;
									dct_memory[19] 	<= y3;
									dct_memory[27] 	<= y4;
									dct_memory[35] 	<= y5;
									dct_memory[43] 	<= y6;
									dct_memory[51] 	<= y7;
									dct_memory[59] 	<= y8;
									stages <= 5'd8;
							end else if(stages == 5'd8) begin				// state 8
									dct_memory[4] 	<= y1;
									dct_memory[12] 	<= y2;
									dct_memory[20] 	<= y3;
									dct_memory[28] 	<= y4;
									dct_memory[36] 	<= y5;
									dct_memory[44] 	<= y6;
									dct_memory[52] 	<= y7;
									dct_memory[60] 	<= y8;
									stages <= 5'd9;
							end else if(stages == 5'd9) begin				// state 9
									dct_memory[5] 	<= y1;
									dct_memory[13] 	<= y2;
									dct_memory[21] 	<= y3;
									dct_memory[29] 	<= y4;
									dct_memory[37] 	<= y5;
									dct_memory[45] 	<= y6;
									dct_memory[53] 	<= y7;
									dct_memory[61] 	<= y8;
									stages <= 5'd10;
							end else if(stages == 5'd10) begin				// state 10
									dct_memory[6] 	<= y1;
									dct_memory[14] 	<= y2;
									dct_memory[22] 	<= y3;
									dct_memory[30] 	<= y4;
									dct_memory[38] 	<= y5;
									dct_memory[46] 	<= y6;
									dct_memory[54] 	<= y7;
									dct_memory[62] 	<= y8;
									stages <= 5'd11;
							end else if(stages == 5'd11) begin				// state 11  dct1 end
									dct_memory[7] 	<= y1;
									dct_memory[15] 	<= y2;
									dct_memory[23] 	<= y3;
									dct_memory[31] 	<= y4;
									dct_memory[39] 	<= y5;
									dct_memory[47] 	<= y6;
									dct_memory[55] 	<= y7;
									dct_memory[63] 	<= y8;
									dct1_finished 	<= 1'b1;	
									stages 			<= 5'd12;
							end else if(stages == 5'd12) begin				// state 12
									addr_dct_input <= 0;
									stages <= 5'd13;
							end else if(stages == 5'd13) begin				// state 13
									addr_dct_input <= 8;
									stages <= 5'd14;
							end else if(stages == 5'd14) begin				// state 14
									addr_dct_input <= 16;
									stages <= 5'd15;
							end else if(stages == 5'd15) begin				// state 15
									addr_dct_input <= 24;
									stages <= 5'd16;
							end else if(stages == 5'd16) begin				// state 16
									addr_dct_input 	<= 32;
									zz_memory[0] 	<= (y1 < 0) ? ((y1 >>> 4) + 1) : (y1 >>> 4);
									zz_memory[2] 	<= (y2 < 0) ? ((y2 >>> 4) + 1) : (y2 >>> 4);
									zz_memory[3] 	<= (y3 < 0) ? ((y3 >>> 4) + 1) : (y3 >>> 4);
									zz_memory[9] 	<= (y4 < 0) ? ((y4 >>> 4) + 1) : (y4 >>> 4);
									zz_memory[10] 	<= (y5 < 0) ? ((y5 >>> 5) + 1) : (y5 >>> 5);
									zz_memory[20] 	<= (y6 < 0) ? ((y6 >>> 6) + 1) : (y6 >>> 6);
									zz_memory[21] 	<= (y7 < 0) ? ((y7 >>> 7) + 1) : (y7 >>> 7);
									zz_memory[35] 	<= (y8 < 0) ? ((y8 >>> 7) + 1) : (y8 >>> 7);
									stages <= 5'd17;
							end else if(stages == 5'd17) begin				// state 17
									addr_dct_input 	<= 40;
									zz_memory[1] 	<= (y1 < 0) ? ((y1 >>> 4) + 1) : (y1 >>> 4);
									zz_memory[4] 	<= (y2 < 0) ? ((y2 >>> 4) + 1) : (y2 >>> 4);
									zz_memory[8] 	<= (y3 < 0) ? ((y3 >>> 4) + 1) : (y3 >>> 4);
									zz_memory[11] 	<= (y4 < 0) ? ((y4 >>> 4) + 1) : (y4 >>> 4);
									zz_memory[19] 	<= (y5 < 0) ? ((y5 >>> 5) + 1) : (y5 >>> 5);
									zz_memory[22] 	<= (y6 < 0) ? ((y6 >>> 6) + 1) : (y6 >>> 6);
									zz_memory[34] 	<= (y7 < 0) ? ((y7 >>> 7) + 1) : (y7 >>> 7);
									zz_memory[36] 	<= (y8 < 0) ? ((y8 >>> 7) + 1) : (y8 >>> 7);
									stages <= 5'd18;
							end else if(stages == 5'd18) begin				// state 18
									addr_dct_input 	<= 48;
									zz_memory[5] 	<= (y1 < 0) ? ((y1 >>> 4) + 1) : (y1 >>> 4);
									zz_memory[7] 	<= (y2 < 0) ? ((y2 >>> 4) + 1) : (y2 >>> 4);
									zz_memory[12] 	<= (y3 < 0) ? ((y3 >>> 4) + 1) : (y3 >>> 4);
									zz_memory[18] 	<= (y4 < 0) ? ((y4 >>> 5) + 1) : (y4 >>> 5);
									zz_memory[23] 	<= (y5 < 0) ? ((y5 >>> 5) + 1) : (y5 >>> 5);
									zz_memory[33] 	<= (y6 < 0) ? ((y6 >>> 6) + 1) : (y6 >>> 6);
									zz_memory[37] 	<= (y7 < 0) ? ((y7 >>> 7) + 1) : (y7 >>> 7);
									zz_memory[48] 	<= (y8 < 0) ? ((y8 >>> 7) + 1) : (y8 >>> 7);
									stages <= 5'd19;
							end else if(stages == 5'd19) begin				// state 19
									addr_dct_input <= 56;
									zz_memory[6] 	<= (y1 < 0) ? ((y1 >>> 4) + 1) : (y1 >>> 4);
									zz_memory[13] 	<= (y2 < 0) ? ((y2 >>> 4) + 1) : (y2 >>> 4);
									zz_memory[17] 	<= (y3 < 0) ? ((y3 >>> 5) + 1) : (y3 >>> 5);
									zz_memory[24] 	<= (y4 < 0) ? ((y4 >>> 5) + 1) : (y4 >>> 5);
									zz_memory[32] 	<= (y5 < 0) ? ((y5 >>> 6) + 1) : (y5 >>> 6);
									zz_memory[38] 	<= (y6 < 0) ? ((y6 >>> 6) + 1) : (y6 >>> 6);
									zz_memory[47] 	<= (y7 < 0) ? ((y7 >>> 7) + 1) : (y7 >>> 7);
									zz_memory[49] 	<= (y8 < 0) ? ((y8 >>> 7) + 1) : (y8 >>> 7);
									stages <= 5'd20;
							end else if(stages == 5'd20) begin				// state 20
									zz_memory[14] 	<= (y1 < 0) ? ((y1 >>> 5) + 1) : (y1 >>> 5);
									zz_memory[16] 	<= (y2 < 0) ? ((y2 >>> 5) + 1) : (y2 >>> 5);
									zz_memory[25] 	<= (y3 < 0) ? ((y3 >>> 5) + 1) : (y3 >>> 5);
									zz_memory[31] 	<= (y4 < 0) ? ((y4 >>> 5) + 1) : (y4 >>> 5);
									zz_memory[39] 	<= (y5 < 0) ? ((y5 >>> 7) + 1) : (y5 >>> 7);
									zz_memory[46] 	<= (y6 < 0) ? ((y6 >>> 7) + 1) : (y6 >>> 7);
									zz_memory[50] 	<= (y7 < 0) ? ((y7 >>> 7) + 1) : (y7 >>> 7);
									zz_memory[57] 	<= (y8 < 0) ? ((y8 >>> 7) + 1) : (y8 >>> 7);
									stages <= 5'd21;
							end else if(stages == 5'd21) begin				// state 21
									zz_memory[15] 	<= (y1 < 0) ? ((y1 >>> 6) + 1) : (y1 >>> 6);
									zz_memory[26] 	<= (y2 < 0) ? ((y2 >>> 6) + 1) : (y2 >>> 6);
									zz_memory[30] 	<= (y3 < 0) ? ((y3 >>> 6) + 1) : (y3 >>> 6);
									zz_memory[40] 	<= (y4 < 0) ? ((y4 >>> 6) + 1) : (y4 >>> 6);
									zz_memory[45] 	<= (y5 < 0) ? ((y5 >>> 7) + 1) : (y5 >>> 7);
									zz_memory[51] 	<= (y6 < 0) ? ((y6 >>> 7) + 1) : (y6 >>> 7);
									zz_memory[56] 	<= (y7 < 0) ? ((y7 >>> 7) + 1) : (y7 >>> 7);
									zz_memory[48] 	<= (y8 < 0) ? ((y8 >>> 7) + 1) : (y8 >>> 7);
									stages <= 5'd22;
							end else if(stages == 5'd22) begin				// state 22
									zz_memory[27] 	<= (y1 < 0) ? ((y1 >>> 6) + 1) : (y1 >>> 6);
									zz_memory[29] 	<= (y2 < 0) ? ((y2 >>> 6) + 1) : (y2 >>> 6);
									zz_memory[41] 	<= (y3 < 0) ? ((y3 >>> 6) + 1) : (y3 >>> 6);
									zz_memory[44] 	<= (y4 < 0) ? ((y4 >>> 6) + 1) : (y4 >>> 6);
									zz_memory[52] 	<= (y5 < 0) ? ((y5 >>> 7) + 1) : (y5 >>> 7);
									zz_memory[55] 	<= (y6 < 0) ? ((y6 >>> 7) + 1) : (y6 >>> 7);
									zz_memory[59] 	<= (y7 < 0) ? ((y7 >>> 7) + 1) : (y7 >>> 7);
									zz_memory[62] 	<= (y8 < 0) ? ((y8 >>> 7) + 1) : (y8 >>> 7);
									stages <= 5'd23;
							end else if(stages == 5'd23) begin				// state 23
									zz_memory[28] 	<= (y1 < 0) ? ((y1 >>> 6) + 1) : (y1 >>> 6);
									zz_memory[42] 	<= (y2 < 0) ? ((y2 >>> 6) + 1) : (y2 >>> 6);
									zz_memory[43] 	<= (y3 < 0) ? ((y3 >>> 6) + 1) : (y3 >>> 6);
									zz_memory[53] 	<= (y4 < 0) ? ((y4 >>> 6) + 1) : (y4 >>> 6);
									zz_memory[54] 	<= (y5 < 0) ? ((y5 >>> 7) + 1) : (y5 >>> 7);
									zz_memory[60] 	<= (y6 < 0) ? ((y6 >>> 7) + 1) : (y6 >>> 7);
									zz_memory[61] 	<= (y7 < 0) ? ((y7 >>> 7) + 1) : (y7 >>> 7);
									zz_memory[63] 	<= (y8 < 0) ? ((y8 >>> 7) + 1) : (y8 >>> 7);
									stages <= 5'd24;
							end else if(stages == 5'd24) begin				// state 24 DC Calculation
									   stages 			<= 	5'd25;
										rlc_index 		<= 	0;
										count		 		<= 	0;
							end else if(stages == 5'd25) begin				// state 25   ****RLC******
						            if (data != 0 && !data_before_zero) begin // if data is nonzero, and data before is nonzero, put [0, data]
											 rlc[rlc_index]      <= {zero_count[3:0], category[3:0], data_new[14:0]};                
						                rlc_index           <= rlc_index + 1;
						                zero_count          <= 0;
						                data_before_zero    <= 0;
						                overhead            <= 0;
						            end else if (data == 0 && !data_before_zero) begin  // if data is zero, data before is nonzero, zero_count=1,
						                rlc_index           <= rlc_index;  
						                zero_count          <= 1;
						                data_before_zero    <= 1;
						                overhead            <= overhead;
						            end else if (data != 0 && data_before_zero) begin   //if data is nonzero,data before is zero, then put [zero_count, data]
						                rlc[rlc_index + overhead]    <= {zero_count[3:0], category[3:0], data_new[14:0]};
						                rlc_index           <= rlc_index + overhead + 1;
						                zero_count          <= 0;
						                data_before_zero    <= 0;
						                overhead            <= 0;
						            end else if (data == 0 && data_before_zero) begin   // if data is zero, data before is also zero, zero_count ++ 
						                if (zero_count == 15 && count != 63) begin
						                    rlc[rlc_index + overhead]    <= {4'b1111, 4'd0, 15'd0};
						                    rlc_index           <= rlc_index;
						                    zero_count          <= 0;
						                    data_before_zero    <= 0;
						                    overhead            <= overhead + 1;
						                end else begin
						                    if (count == 63) begin                      // end EOB if last number is 0, put [0, 0]
						                        rlc[rlc_index]    <= {4'd0, 4'd0, 15'd0};       
						                        rlc_index           <= rlc_index;
						                        zero_count          <= 0;
						                        data_before_zero    <= 1;    
						                        overhead            <= overhead;
						                        last_index          <= rlc_index;
						                    end else begin               
						                        rlc_index           <= rlc_index;               
						                        zero_count          <= zero_count + 1;
						                        data_before_zero    <= 1;
						                        overhead            <= overhead;
						                    end
						                end    
						            end
						            if (count == 63) begin
						                stages <= 5'd26;                                     // RLC FINISHED
						                count_ac <= 1;
						            end
						            count <= count + 1;
							end else if(stages == 5'd26) begin				// state 26
									zz_memory[0] 	<= 	dc_diff;
									data_old 		<= 	zz_memory[0];
									count				<= 	0;
									diff_new_reg   <=    diff_new;
									stages 			<= 5'd27;
							end else if(stages == 5'd27) begin				// state 27
									buckets[0]   <= dc_codeword;
									final_out    <= (dc_codeword << category) + diff_new_reg;
									final_length <= dc_codeword_length + category;
								   //final_out    <= dc_codeword;
									//final_length <= dc_codeword_length;
 									stages <= 5'd28;
							end else if(stages == 5'd28) begin				// state 28
									if (count_ac == last_index || bucket_out == 10) begin
						            	 stages <= 5'd29;
									end	else begin
						                count_ac    <= count_ac  + 1;
						         end
						         buckets[count_ac] <= bucket_out;
									final_out <= (final_out << bucket_length) + bucket_out; 
									final_length <= final_length + bucket_length;
							end else if(stages == 5'd29) begin				// state 29  shift final output to the MSB
								   stages <= 5'd30;
									//final_out <= (final_out << 20);
									final_out <= (final_out << (512 - final_length)); 
							end else if(stages == 5'd30) begin				// state 30
									addr_write 			<= 6'b000000;
									addr_read			<= 6'b111111;
									addr_dct_input  	<= 6'd0;
									input_loaded 		<= 1'b0;
									stages 				<= 5'd0;
									dct_finished 		<= 1'b1;
									dct1_finished		<= 1'b0;	
								
            					
								   count               <=  0;
            				   data_before_zero    <=  0;
            				   rlc_index           <=  0;
            				   zero_count          <=  0;
            				   overhead            <=  0;
            				   count_ac            <=  1;
									
									//final_out			  <=  0;
									//final_length		  <=  0;
							end
						end else begin
									read_before <= 1'b0;
						end
					end

endmodule


module dct1 (clk, x1, x2, x3, x4, x5, x6, x7, x8, y1, y2, y3, y4, y5, y6, y7, y8);
	input clk;
	input signed [13:0] x1, x2, x3, x4, x5, x6, x7, x8;	
	output signed [19:0] y1, y2, y3, y4, y5, y6, y7, y8;
	//output signed [26:0] temp0, temp1, temp2, temp3;
	wire signed [14:0] a1, a2, a3, a4, a5, a6, a7, a8;
	reg signed [15:0] a5_reg, a6_reg, a7_reg, a8_reg;
		

	reg signed [15:0] sb1_reg, sb2_reg, sb3_reg, sb4_reg, sb5_reg, sb6_reg, sb7_reg, sb8_reg, sb9_reg, sb10_reg, sb11_reg, sb12_reg, sb13_reg;  
	wire signed [15:0] sb1, sb2, sb3, sb4, sb5, sb6, sb7, sb8, sb9, sb10, sb11, sb12, sb13;

	// y1
	wire signed [16:0] y1_1_1;
	reg signed [24:0] y1_3_1_reg;	
	wire signed [26:0] y1_5_1;
	reg signed [26:0] y1_5_1_reg;
	wire signed [26:0] y1_6_1, y1_6_2;
	wire signed [19:0] y1_7_1;

	// y2
	wire signed [16:0] y2_1_1, y2_1_2, y2_1_3, y2_1_4, y2_1_5;
	wire signed [23:0] y2_2_1, y2_2_2, y2_2_3, y2_2_4, y2_2_5, y2_2_6;
	wire signed [24:0] y2_3_1, y2_3_2, y2_3_3;
	reg signed [24:0] y2_3_1_reg, y2_3_2_reg, y2_3_3_reg;
	wire signed [25:0] y2_3_3_new;
	wire signed [25:0] y2_4_1;
	wire signed [26:0] y2_5_1;	
	reg signed [26:0] y2_5_1_reg;
	wire signed [26:0] y2_6_1, y2_6_2;
	wire signed [19:0] y2_7_1;

	// y3
	wire signed [23:0] y3_2_1, y3_2_2, y3_2_3, y3_2_4, y3_2_5;
	wire signed [23:0] sb5_reg_new;
	wire signed [24:0] y3_3_1, y3_3_2, y3_3_3;
	reg signed [24:0] y3_3_1_reg, y3_3_2_reg, y3_3_3_reg;
	wire signed [25:0] y3_3_3_new;
	wire signed [25:0] y3_4_1;
	wire signed [26:0] y3_5_1;
	reg signed [26:0] y3_5_1_reg;
	wire signed [26:0] y3_6_1;
	wire signed [19:0] y3_7_1;
	// y4
	wire signed [16:0] y4_1_1, y4_1_2;
	wire signed [23:0] sb12_reg_new;
	wire signed [23:0] y4_2_1, y4_2_2, y4_2_3, y4_2_4, y4_2_5;
	wire signed [24:0] y4_3_1, y4_3_2, y4_3_3;
	reg signed  [24:0] y4_3_1_reg, y4_3_2_reg, y4_3_3_reg;
	wire signed [25:0] y4_3_3_new;
	wire signed [25:0] y4_4_1;
	wire signed [26:0] y4_5_1;
	reg signed [26:0] y4_5_1_reg;
	wire signed [26:0] y4_6_1;
	wire signed [19:0] y4_7_1;
	// y5
	wire signed [16:0] y5_1_1;
	reg signed  [24:0] y5_3_1_reg;
	wire signed [26:0] y5_5_1;
	reg signed [26:0] y5_5_1_reg;
	wire signed [26:0] y5_6_1, y5_6_2;
	wire signed [19:0] y5_7_1;
	
	// y6
	wire signed [16:0] y6_1_1, y6_1_2;
	wire signed [23:0] y6_2_1, y6_2_2, y6_2_3, y6_2_4, y6_2_5;
	wire signed [23:0] sb10_reg_new;
	wire signed [24:0] y6_3_1, y6_3_2, y6_3_3;
	reg signed  [24:0] y6_3_1_reg, y6_3_2_reg, y6_3_3_reg;
	wire signed [25:0] y6_3_3_new;
	wire signed [25:0] y6_4_1;
	wire signed [26:0] y6_5_1;	
	reg signed [26:0] y6_5_1_reg;
	wire signed [26:0] y6_6_1;
	wire signed [19:0] y6_7_1;
	// y7
	wire signed [23:0] y7_2_1, y7_2_2, y7_2_3, y7_2_4, y7_2_5;
	wire signed [23:0] sb6_reg_new;
	wire signed [24:0] y7_3_1, y7_3_2, y7_3_3;
	reg signed [24:0] y7_3_1_reg, y7_3_2_reg, y7_3_3_reg;
	wire signed [25:0] y7_3_3_new;
	wire signed [25:0] y7_4_1;
	wire signed [26:0] y7_5_1;
	reg signed [26:0] y7_5_1_reg;
	wire signed [26:0] y7_6_1;
	wire signed [19:0] y7_7_1;
	// y8
	wire signed [16:0] y8_1_1,y8_1_2;
	wire signed [23:0] y8_2_1,y8_2_2,y8_2_3,y8_2_4,y8_2_5;
	wire signed [23:0] sb11_reg_new;
	wire signed [24:0] y8_3_1,y8_3_2,y8_3_3;
	reg  signed [24:0] y8_3_1_reg,y8_3_2_reg,y8_3_3_reg;
	wire signed [25:0] y8_3_3_new;
	wire signed [25:0] y8_4_1;
	wire signed [26:0] y8_5_1;
	reg signed [26:0] y8_5_1_reg;
	wire signed [26:0] y8_6_1;
	wire signed [19:0] y8_7_1;
/////////////////////////////////////////////////////////////////////////////
// stage 1
/////////////////////////////////////////////////////////////////////////////
	adder_14 add0(x1, x8, 0, a1);
	adder_14 add1(x2, x7, 0, a2);
	adder_14 add2(x3, x6, 0, a3);
	adder_14 add3(x4, x5, 0, a4);
	adder_14 add4(x4, x5, 1, a5);
	adder_14 add5(x3, x6, 1, a6);
	adder_14 add6(x2, x7, 1, a7);
	adder_14 add7(x1, x8, 1, a8);

	adder_15 add8(a5, a8, 0, sb1);
	adder_15 add9(a8, a5, 1, sb2);
	adder_15 add10(a6, a7, 1, sb3);
	adder_15 add11(a6, a7, 0, sb4);

	adder_15 add12(a2, a3, 1, sb5);
	adder_15 add13(a1, a4, 1, sb6);
	adder_15 add14(a1, a4, 0, sb7);
	adder_15 add15(a2, a3, 0, sb8);

	adder_15 add16(a6, a5, 1, sb9);
	adder_15 add17(a6, a8, 1, sb10);
	adder_15 add18(a7, a8, 0, sb11);
	adder_15 add19(a5, a7, 1, sb12);
	adder_15 add20(a8, a8, 0, sb13);
/////////////////////////////////////////////////////////////////////////////
// stage 2
/////////////////////////////////////////////////////////////////////////////
	// y1 adders:
	adder_16 addy1_1_1(sb7_reg, sb8_reg, 0, y1_1_1);
	assign y1_5_1 = y1_3_1_reg;

	assign y1_6_1 = y1_5_1_reg >>> 2;
	assign y1_6_2 = y1_5_1_reg >>> 3;
	adder_19 addy1_7_1(y1_6_1[18:0],y1_6_2[18:0],0,y1_7_1);

	// y2 adders:
	adder_16 addy2_1_1(sb1_reg, sb4_reg, 0, y2_1_1);
	adder_16 addy2_1_2(sb1_reg, sb3_reg, 0, y2_1_2);
	adder_16 addy2_1_3(sb1_reg, sb4_reg, 1, y2_1_3);
	adder_16 addy2_1_4(sb2_reg, sb3_reg, 0, y2_1_4);
	adder_16 addy2_1_5(sb13_reg, sb3_reg, 0, y2_1_5);
	assign y2_2_1 = y2_1_1 <<< 7;
	assign y2_2_2 = sb2_reg <<< 6;	  
	assign y2_2_3 = y2_1_2 <<< 5;
	assign y2_2_4 = y2_1_5 <<< 3;
	assign y2_2_5 = y2_1_3 <<< 1;
	assign y2_2_6 = y2_1_4;
	adder_24 addy2_3_1(y2_2_1, y2_2_2, 0, y2_3_1);
	adder_24 addy2_3_2(y2_2_3, y2_2_4, 1, y2_3_2);
	adder_24 addy2_3_3(y2_2_5, y2_2_6, 1, y2_3_3);
	assign y2_3_3_new = y2_3_3_reg;		// 25 bits -> 26 bits
	adder_25 addy2_4_1(y2_3_1_reg, y2_3_2_reg, 1, y2_4_1);
	adder_26 addy2_5_1(y2_4_1, y2_3_3_new, 0, y2_5_1);

	assign y2_6_1 = y2_5_1_reg >>> 9;
	assign y2_6_2 = y2_5_1_reg >>> 10;
	adder_19 addy2_7_1(y2_6_1[18:0],y2_6_2[18:0],0,y2_7_1);

	// y3 adders:
	assign y3_2_1 = sb5_reg <<< 6;
	assign y3_2_2 = sb5_reg <<< 4;
	assign y3_2_3 = sb6_reg <<< 7;
	assign y3_2_4 = sb6_reg <<< 3;
	assign y3_2_5 = sb6_reg <<< 1;
	adder_24 addy3_3_1(y3_2_1, y3_2_2, 1, y3_3_1);
	adder_24 addy3_3_2(y3_2_3, y3_2_4, 1, y3_3_2);
	assign sb5_reg_new = sb5_reg;
	adder_24 addy3_3_3(sb5_reg_new, y3_2_5, 1, y3_3_3);
	assign y3_3_3_new = y3_3_3_reg;		// 25 bits -> 26 bits
	adder_25 addy3_4_1(y3_3_1_reg, y3_3_2_reg, 0, y3_4_1);
	adder_26 addy3_5_1(y3_4_1, y3_3_3_new, 0, y3_5_1);
	assign y3_6_1 = y3_5_1_reg >>> 8;
	assign y3_7_1 = y3_6_1[19:0];

	// y4 adders:
	adder_16 addy4_1_1(sb11_reg, a5_reg, 1, y4_1_1);
	adder_16 addy4_1_2(a6_reg, a8_reg, 0, y4_1_2);
	assign y4_2_1 = sb10_reg <<< 7;
	assign y4_2_2 = a5_reg <<< 6;
	assign y4_2_3 = sb11_reg <<< 5;
	assign y4_2_4 = y4_1_1 <<< 3;
	assign y4_2_5 = y4_1_2 <<< 1;
	assign sb12_reg_new = sb12_reg;
	adder_24 addy4_3_1(y4_2_2, y4_2_1, 0, y4_3_1);
	adder_24 addy4_3_2(y4_2_4, y4_2_3, 1, y4_3_2);
	adder_24 addy4_3_3(y4_2_5, sb12_reg_new, 0, y4_3_3);
	adder_25 addy4_4_1(y4_3_2_reg, y4_3_1_reg, 1, y4_4_1);
	assign y4_3_3_new = y4_3_3_reg;	 
	adder_26 addy4_5_1(y4_4_1, y4_3_3_new, 0, y4_5_1);
	assign y4_6_1 = y4_5_1_reg >>> 8;
	assign y4_7_1 = y4_6_1[19:0];
	// y5 adders:
	
	adder_16 addy5_1_1(sb7_reg, sb8_reg, 1, y5_1_1);
	assign y5_5_1 = y5_3_1_reg;
	assign y5_6_1 = y5_5_1_reg >>> 2;
	assign y5_6_2 = y5_5_1_reg >>> 3;
	adder_19 addy5_7_1(y5_6_1[18:0],y5_6_2[18:0],0,y5_7_1);

	// y6 adders:
	adder_16 addy6_1_1(sb1_reg, a6_reg, 1, y6_1_1);
	adder_16 addy6_1_2(a5_reg, a7_reg, 0, y6_1_2);
	assign y6_2_1 = sb12_reg <<< 7;
	assign y6_2_2 = a8_reg <<< 6;
	assign y6_2_3 = sb9_reg <<< 5;
	assign y6_2_4 = y6_1_1 <<< 3;
	assign y6_2_5 = y6_1_2 <<< 1;
	assign sb10_reg_new = sb10_reg;
	adder_24 addy6_3_1(y6_2_1, y6_2_2, 0, y6_3_1);
	adder_24 addy6_3_2(y6_2_3, y6_2_4, 0, y6_3_2);
	adder_24 addy6_3_3(y6_2_5, sb10_reg_new, 0, y6_3_3);
	adder_25 addy6_4_1(y6_3_1_reg, y6_3_2_reg, 0, y6_4_1);
	assign y6_3_3_new = y6_3_3_reg;	 
	adder_26 addy6_5_1(y6_4_1, y6_3_3_new, 0, y6_5_1);
	assign y6_6_1 = y6_5_1_reg >>> 8;
	assign y6_7_1 = y6_6_1[19:0];
	// y7 adders:
	assign y7_2_1 = sb6_reg <<< 6;
	assign y7_2_2 = sb6_reg <<< 4;
	assign y7_2_3 = sb5_reg <<< 7;
	assign y7_2_4 = sb5_reg <<< 3;
	assign y7_2_5 = sb5_reg <<< 1;
	assign sb6_reg_new = sb6_reg;
	adder_24 addy7_3_1(y7_2_1, y7_2_2, 1, y7_3_1);
	adder_24 addy7_3_2(sb6_reg_new, y7_2_3, 1, y7_3_2);
	adder_24 addy7_3_3(y7_2_4, y7_2_5, 0, y7_3_3);
	assign y7_3_3_new = y7_3_3_reg;		// 25 bits -> 26 bits
	adder_25 addy7_4_1(y7_3_1_reg, y7_3_2_reg, 0, y7_4_1);
	adder_26 addy7_5_1(y7_4_1, y7_3_3_new, 0, y7_5_1);
	assign y7_6_1 = y7_5_1_reg >>> 8;
	assign y7_7_1 = y7_6_1[19:0];

	// y8 adders:
	adder_16 addy8_1_1(sb10_reg,a7_reg, 1 , y8_1_1);
	adder_16 addy8_1_2(a6_reg, a5_reg, 0 , y8_1_2);
	assign y8_2_1 = sb9_reg <<< 7;
	assign y8_2_2 = a7_reg <<< 6;
	assign y8_2_3 = sb10_reg <<< 5;
	assign y8_2_4 = y8_1_1 <<< 3;
	assign y8_2_5 = y8_1_2 <<< 1;
	adder_24 addy8_3_1(y8_2_1,y8_2_2, 1 , y8_3_1);
	adder_24 addy8_3_2(y8_2_4,y8_2_3, 1 , y8_3_2);
	assign sb11_reg_new = sb11_reg;
	adder_24 addy8_3_3(y8_2_5,sb11_reg_new, 0 , y8_3_3);
	adder_25 addy8_4_1(y8_3_1_reg,y8_3_2_reg, 0 , y8_4_1);
	assign y8_3_3_new = y8_3_3_reg;
	adder_26 addy8_5_1(y8_3_3_new,y8_4_1, 0 , y8_5_1);
	assign y8_6_1 = y8_5_1_reg >>> 8;
	assign y8_7_1 = y8_6_1[19:0];
/////////////////////////////////////////////////////////////////////////////
// Pipline Registers
/////////////////////////////////////////////////////////////////////////////

	always @(posedge clk) begin
		// Pipline 1
		a5_reg <= a5; 
		a6_reg <= a6; 
		a7_reg <= a7; 
		a8_reg <= a8;

		sb1_reg <= sb1;
		sb2_reg <= sb2;
		sb3_reg <= sb3;
		sb4_reg <= sb4;
      		sb5_reg <= sb5;
		sb6_reg <= sb6;
		sb7_reg <= sb7;
		sb8_reg <= sb8;
		sb9_reg <= sb9;
		sb10_reg <= sb10;
		sb11_reg <= sb11;
		sb12_reg <= sb12;		
		sb13_reg <= sb13;
	
		// Pipline 2
		// y1 
		y1_3_1_reg  <= y1_1_1;	// 16 bits -> 24 bits
		// y2
		y2_3_1_reg <= y2_3_1;
		y2_3_2_reg <= y2_3_2;
		y2_3_3_reg <= y2_3_3;
		// y3
		y3_3_1_reg <= y3_3_1;
		y3_3_2_reg <= y3_3_2;
		y3_3_3_reg <= y3_3_3;
		// y4
		y4_3_1_reg <= y4_3_1;
		y4_3_2_reg <= y4_3_2;
		y4_3_3_reg <= y4_3_3;
		// y5
		y5_3_1_reg <= y5_1_1;
		// y6
		y6_3_1_reg <= y6_3_1;
		y6_3_2_reg <= y6_3_2;
		y6_3_3_reg <= y6_3_3;
		// y7
		y7_3_1_reg <= y7_3_1;
		y7_3_2_reg <= y7_3_2;
		y7_3_3_reg <= y7_3_3;
		// y8
		y8_3_1_reg <= y8_3_1;
		y8_3_2_reg <= y8_3_2;
		y8_3_3_reg <= y8_3_3;
		
		// Pipline 3
		y1_5_1_reg <= y1_5_1;
		y2_5_1_reg <= y2_5_1;
		y3_5_1_reg <= y3_5_1;
		y4_5_1_reg <= y4_5_1;
		y5_5_1_reg <= y5_5_1;
		y6_5_1_reg <= y6_5_1;
		y7_5_1_reg <= y7_5_1;
		y8_5_1_reg <= y8_5_1;
	end


	assign y1 = y1_7_1;
	assign y2 = y2_7_1;
	assign y3 = y3_7_1;
	assign y4 = y4_7_1;
	assign y5 = y5_7_1;
	assign y6 = y6_7_1;
	assign y7 = y7_7_1;
	assign y8 = y8_7_1;
	//assign temp0 =  0;
	//assign temp1 =  0;
	//assign temp2 =  0;
	//assign temp3 =  0; 

endmodule


module adder_14(a, b, cin, sum);
	input signed [13:0]a, b;
	input cin;
	output signed [14:0]sum;
	wire cout;
	wire c1, c2, c3, c4, c5, c6, c7, c8, c9, c10, c11, c12, c13;
	wire overFlow;
	wire signed [13:0] b_b;
	
   assign b_b[0] = (cin)? !b[0]: b[0];
	assign b_b[1] = (cin)? !b[1]: b[1];
	assign b_b[2] = (cin)? !b[2]: b[2];
	assign b_b[3] = (cin)? !b[3]: b[3];
	assign b_b[4] = (cin)? !b[4]: b[4];
	assign b_b[5] = (cin)? !b[5]: b[5];
	assign b_b[6] = (cin)? !b[6]: b[6];
	assign b_b[7] = (cin)? !b[7]: b[7];
	assign b_b[8] = (cin)? !b[8]: b[8];
	assign b_b[9] = (cin)? !b[9]: b[9];
	assign b_b[10] = (cin)? !b[10]: b[10];
	assign b_b[11] = (cin)? !b[11]: b[11];
	assign b_b[12] = (cin)? !b[12]: b[12];
	assign b_b[13] = (cin)? !b[13]: b[13];

	//approxAdder a0 (a[0], b_b[0], cin, sum[0], c1);
	//approxAdder a1 (a[1], b_b[1], c1, sum[1], c2);
	//approxAdder a2 (a[2], b_b[2], c2, sum[2], c3);
	fullAdder a0 (a[0], b_b[0], cin, sum[0], c1);
	fullAdder a1 (a[1], b_b[1], c1, sum[1], c2);
	fullAdder a2 (a[2], b_b[2], c2, sum[2], c3);
	fullAdder f3 (a[3], b_b[3], c3, sum[3], c4);
	fullAdder f4 (a[4], b_b[4], c4, sum[4], c5);
	fullAdder f5 (a[5], b_b[5], c5, sum[5], c6);
	fullAdder f6 (a[6], b_b[6], c6, sum[6], c7);
	fullAdder f7 (a[7], b_b[7], c7, sum[7], c8);
	fullAdder f8 (a[8], b_b[8], c8, sum[8], c9);
	fullAdder f9 (a[9], b_b[9], c9, sum[9], c10);
	fullAdder f10 (a[10], b_b[10], c10, sum[10], c11);
	fullAdder f11 (a[11], b_b[11], c11, sum[11], c12);
	fullAdder f12 (a[12], b_b[12], c12, sum[12], c13);
	fullAdder f13 (a[13], b_b[13], c13, sum[13], cout);
	xor x0(overFlow, cout, c13);
	assign sum[14] = (overFlow)? cout:sum[13];
	
endmodule


module adder_15(a, b, cin, sum);
	input signed [14:0] a, b;
	input cin;
	output signed [15:0] sum;
	wire cout;
	wire c1, c2, c3, c4, c5, c6, c7, c8, c9, c10, c11, c12, c13, c14;
	wire overFlow;
	wire signed [14:0] b_b;
	
   assign b_b[0] = (cin)? !b[0]: b[0];
	assign b_b[1] = (cin)? !b[1]: b[1];
	assign b_b[2] = (cin)? !b[2]: b[2];
	assign b_b[3] = (cin)? !b[3]: b[3];
	assign b_b[4] = (cin)? !b[4]: b[4];
	assign b_b[5] = (cin)? !b[5]: b[5];
	assign b_b[6] = (cin)? !b[6]: b[6];
	assign b_b[7] = (cin)? !b[7]: b[7];
	assign b_b[8] = (cin)? !b[8]: b[8];
	assign b_b[9] = (cin)? !b[9]: b[9];
	assign b_b[10] = (cin)? !b[10]: b[10];
	assign b_b[11] = (cin)? !b[11]: b[11];
	assign b_b[12] = (cin)? !b[12]: b[12];
	assign b_b[13] = (cin)? !b[13]: b[13];
	assign b_b[14] = (cin)? !b[14]: b[14];

	fullAdder a0 (a[0], b_b[0], cin, sum[0], c1);
	fullAdder a1 (a[1], b_b[1], c1, sum[1], c2);
	fullAdder a2 (a[2], b_b[2], c2, sum[2], c3);

	fullAdder f3 (a[3], b_b[3], c3, sum[3], c4);
	fullAdder f4 (a[4], b_b[4], c4, sum[4], c5);
	fullAdder f5 (a[5], b_b[5], c5, sum[5], c6);
	fullAdder f6 (a[6], b_b[6], c6, sum[6], c7);
	fullAdder f7 (a[7], b_b[7], c7, sum[7], c8);
	fullAdder f8 (a[8], b_b[8], c8, sum[8], c9);
	fullAdder f9 (a[9], b_b[9], c9, sum[9], c10);
	fullAdder f10 (a[10], b_b[10], c10, sum[10], c11);
	fullAdder f11 (a[11], b_b[11], c11, sum[11], c12);
	fullAdder f12 (a[12], b_b[12], c12, sum[12], c13);
	fullAdder f13 (a[13], b_b[13], c13, sum[13], c14);
	fullAdder f14 (a[14], b_b[14], c14, sum[14], cout);
	xor x0(overFlow, cout, c14);
	assign sum[15] = (overFlow)? cout:sum[14];
endmodule

module adder_16(a, b, cin, sum);
	input signed [15:0]a, b;
	input cin;
	output signed [16:0]sum;
	wire cout;
	wire c1, c2, c3, c4, c5, c6, c7, c8, c9, c10, c11, c12, c13, c14, c15, c16;
	wire overFlow;
	wire signed [15:0]b_b;
	
   assign b_b[0] = (cin)? !b[0]: b[0];
	assign b_b[1] = (cin)? !b[1]: b[1];
	assign b_b[2] = (cin)? !b[2]: b[2];
	assign b_b[3] = (cin)? !b[3]: b[3];
	assign b_b[4] = (cin)? !b[4]: b[4];
	assign b_b[5] = (cin)? !b[5]: b[5];
	assign b_b[6] = (cin)? !b[6]: b[6];
	assign b_b[7] = (cin)? !b[7]: b[7];
	assign b_b[8] = (cin)? !b[8]: b[8];
	assign b_b[9] = (cin)? !b[9]: b[9];
	assign b_b[10] = (cin)? !b[10]: b[10];
	assign b_b[11] = (cin)? !b[11]: b[11];
	assign b_b[12] = (cin)? !b[12]: b[12];
	assign b_b[13] = (cin)? !b[13]: b[13];
	assign b_b[14] = (cin)? !b[14]: b[14];
	assign b_b[15] = (cin)? !b[15]: b[15];

	fullAdder a0 (a[0], b_b[0], cin, sum[0], c1);
	fullAdder a1 (a[1], b_b[1], c1, sum[1], c2);
	fullAdder a2 (a[2], b_b[2], c2, sum[2], c3);

	fullAdder f3 (a[3], b_b[3], c3, sum[3], c4);
	fullAdder f4 (a[4], b_b[4], c4, sum[4], c5);
	fullAdder f5 (a[5], b_b[5], c5, sum[5], c6);
	fullAdder f6 (a[6], b_b[6], c6, sum[6], c7);
	fullAdder f7 (a[7], b_b[7], c7, sum[7], c8);
	fullAdder f8 (a[8], b_b[8], c8, sum[8], c9);
	fullAdder f9 (a[9], b_b[9], c9, sum[9], c10);
	fullAdder f10 (a[10], b_b[10], c10, sum[10], c11);
	fullAdder f11 (a[11], b_b[11], c11, sum[11], c12);
	fullAdder f12 (a[12], b_b[12], c12, sum[12], c13);
	fullAdder f13 (a[13], b_b[13], c13, sum[13], c14);
	fullAdder f14 (a[14], b_b[14], c14, sum[14], c15);
	fullAdder f15 (a[15], b_b[15], c15, sum[15], cout);
	xor x0(overFlow, cout, c15);
	assign sum[16] = (overFlow)? cout:sum[15];
endmodule

module adder_19(a, b, cin, sum);
	input signed [18:0]a, b;
	input cin;
	output signed [19:0]sum;
	wire cout;
	wire c1, c2, c3, c4, c5, c6, c7, c8, c9, c10, c11, c12, c13, c14, c15, c16, c17, c18;
	wire overFlow;
	wire signed [18:0]b_b;
	
   assign b_b[0] = (cin)? !b[0]: b[0];
	assign b_b[1] = (cin)? !b[1]: b[1];
	assign b_b[2] = (cin)? !b[2]: b[2];
	assign b_b[3] = (cin)? !b[3]: b[3];
	assign b_b[4] = (cin)? !b[4]: b[4];
	assign b_b[5] = (cin)? !b[5]: b[5];
	assign b_b[6] = (cin)? !b[6]: b[6];
	assign b_b[7] = (cin)? !b[7]: b[7];
	assign b_b[8] = (cin)? !b[8]: b[8];
	assign b_b[9] = (cin)? !b[9]: b[9];
	assign b_b[10] = (cin)? !b[10]: b[10];
	assign b_b[11] = (cin)? !b[11]: b[11];
	assign b_b[12] = (cin)? !b[12]: b[12];
	assign b_b[13] = (cin)? !b[13]: b[13];
	assign b_b[14] = (cin)? !b[14]: b[14];
	assign b_b[15] = (cin)? !b[15]: b[15];
	assign b_b[16] = (cin)? !b[16]: b[16];
	assign b_b[17] = (cin)? !b[17]: b[17];
	assign b_b[18] = (cin)? !b[18]: b[18];

	fullAdder a0 (a[0], b_b[0], cin, sum[0], c1);
	fullAdder a1 (a[1], b_b[1], c1, sum[1], c2);
	fullAdder a2 (a[2], b_b[2], c2, sum[2], c3);

	fullAdder f3 (a[3], b_b[3], c3, sum[3], c4);
	fullAdder f4 (a[4], b_b[4], c4, sum[4], c5);
	fullAdder f5 (a[5], b_b[5], c5, sum[5], c6);
	fullAdder f6 (a[6], b_b[6], c6, sum[6], c7);
	fullAdder f7 (a[7], b_b[7], c7, sum[7], c8);
	fullAdder f8 (a[8], b_b[8], c8, sum[8], c9);
	fullAdder f9 (a[9], b_b[9], c9, sum[9], c10);
	fullAdder f10 (a[10], b_b[10], c10, sum[10], c11);
	fullAdder f11 (a[11], b_b[11], c11, sum[11], c12);
	fullAdder f12 (a[12], b_b[12], c12, sum[12], c13);
	fullAdder f13 (a[13], b_b[13], c13, sum[13], c14);
	fullAdder f14 (a[14], b_b[14], c14, sum[14], c15);
	fullAdder f15 (a[15], b_b[15], c15, sum[15], c16);
	fullAdder f16 (a[16], b_b[16], c16, sum[16], c17);
	fullAdder f17 (a[17], b_b[17], c17, sum[17], c18);
	fullAdder f18 (a[18], b_b[18], c18, sum[18], cout);
	xor x0(overFlow, cout, c18);
	assign sum[19] = (overFlow)? cout:sum[18];
endmodule

module adder_24(a, b, cin, sum);
	input signed [23:0]a, b;
	input cin;
	output signed [24:0]sum;
	wire cout;
	wire c1, c2, c3, c4, c5, c6, c7, c8, c9, c10, c11, c12, c13, c14, c15, c16, c17, c18, c19, c20, c21, c22, c23;
	wire overFlow;
	wire signed [23:0]b_b;
	
   assign b_b[0] = (cin)? !b[0]: b[0];
	assign b_b[1] = (cin)? !b[1]: b[1];
	assign b_b[2] = (cin)? !b[2]: b[2];
	assign b_b[3] = (cin)? !b[3]: b[3];
	assign b_b[4] = (cin)? !b[4]: b[4];
	assign b_b[5] = (cin)? !b[5]: b[5];
	assign b_b[6] = (cin)? !b[6]: b[6];
	assign b_b[7] = (cin)? !b[7]: b[7];
	assign b_b[8] = (cin)? !b[8]: b[8];
	assign b_b[9] = (cin)? !b[9]: b[9];
	assign b_b[10] = (cin)? !b[10]: b[10];
	assign b_b[11] = (cin)? !b[11]: b[11];
	assign b_b[12] = (cin)? !b[12]: b[12];
	assign b_b[13] = (cin)? !b[13]: b[13];
	assign b_b[14] = (cin)? !b[14]: b[14];
	assign b_b[15] = (cin)? !b[15]: b[15];
	assign b_b[16] = (cin)? !b[16]: b[16];
	assign b_b[17] = (cin)? !b[17]: b[17];
	assign b_b[18] = (cin)? !b[18]: b[18];
	assign b_b[19] = (cin)? !b[19]: b[19];
	assign b_b[20] = (cin)? !b[20]: b[20];
	assign b_b[21] = (cin)? !b[21]: b[21];
	assign b_b[22] = (cin)? !b[22]: b[22];
	assign b_b[23] = (cin)? !b[23]: b[23];

	fullAdder a0 (a[0], b_b[0], cin, sum[0], c1);
	fullAdder a1 (a[1], b_b[1], c1, sum[1], c2);
	fullAdder a2 (a[2], b_b[2], c2, sum[2], c3);

	fullAdder f3 (a[3], b_b[3], c3, sum[3], c4);
	fullAdder f4 (a[4], b_b[4], c4, sum[4], c5);
	fullAdder f5 (a[5], b_b[5], c5, sum[5], c6);
	fullAdder f6 (a[6], b_b[6], c6, sum[6], c7);
	fullAdder f7 (a[7], b_b[7], c7, sum[7], c8);
	fullAdder f8 (a[8], b_b[8], c8, sum[8], c9);
	fullAdder f9 (a[9], b_b[9], c9, sum[9], c10);
	fullAdder f10 (a[10], b_b[10], c10, sum[10], c11);
	fullAdder f11 (a[11], b_b[11], c11, sum[11], c12);
	fullAdder f12 (a[12], b_b[12], c12, sum[12], c13);
	fullAdder f13 (a[13], b_b[13], c13, sum[13], c14);
	fullAdder f14 (a[14], b_b[14], c14, sum[14], c15);
	fullAdder f15 (a[15], b_b[15], c15, sum[15], c16);
	fullAdder f16 (a[16], b_b[16], c16, sum[16], c17);
	fullAdder f17 (a[17], b_b[17], c17, sum[17], c18);
	fullAdder f18 (a[18], b_b[18], c18, sum[18], c19);
	fullAdder f19 (a[19], b_b[19], c19, sum[19], c20);
	fullAdder f20 (a[20], b_b[20], c20, sum[20], c21);
	fullAdder f21 (a[21], b_b[21], c21, sum[21], c22);
	fullAdder f22 (a[22], b_b[22], c22, sum[22], c23);
	fullAdder f23 (a[23], b_b[23], c23, sum[23], cout);
	xor x0(overFlow, cout, c23);
	assign sum[24] = (overFlow)? cout:sum[23];
endmodule

module adder_25(a, b, cin, sum);
	input signed [24:0]a, b;
	input cin;
	output signed [25:0]sum;
	wire cout;
	wire c1, c2, c3, c4, c5, c6, c7, c8, c9, c10, c11, c12, c13, c14, c15, c16, c17, c18, c19, c20, c21, c22, c23, c24;
	wire overFlow;
	wire signed [24:0]b_b;
	
   assign b_b[0] = (cin)? !b[0]: b[0];
	assign b_b[1] = (cin)? !b[1]: b[1];
	assign b_b[2] = (cin)? !b[2]: b[2];
	assign b_b[3] = (cin)? !b[3]: b[3];
	assign b_b[4] = (cin)? !b[4]: b[4];
	assign b_b[5] = (cin)? !b[5]: b[5];
	assign b_b[6] = (cin)? !b[6]: b[6];
	assign b_b[7] = (cin)? !b[7]: b[7];
	assign b_b[8] = (cin)? !b[8]: b[8];
	assign b_b[9] = (cin)? !b[9]: b[9];
	assign b_b[10] = (cin)? !b[10]: b[10];
	assign b_b[11] = (cin)? !b[11]: b[11];
	assign b_b[12] = (cin)? !b[12]: b[12];
	assign b_b[13] = (cin)? !b[13]: b[13];
	assign b_b[14] = (cin)? !b[14]: b[14];
	assign b_b[15] = (cin)? !b[15]: b[15];
	assign b_b[16] = (cin)? !b[16]: b[16];
	assign b_b[17] = (cin)? !b[17]: b[17];
	assign b_b[18] = (cin)? !b[18]: b[18];
	assign b_b[19] = (cin)? !b[19]: b[19];
	assign b_b[20] = (cin)? !b[20]: b[20];
	assign b_b[21] = (cin)? !b[21]: b[21];
	assign b_b[22] = (cin)? !b[22]: b[22];
	assign b_b[23] = (cin)? !b[23]: b[23];
	assign b_b[24] = (cin)? !b[24]: b[24];

	fullAdder a0 (a[0], b_b[0], cin, sum[0], c1);
	fullAdder a1 (a[1], b_b[1], c1, sum[1], c2);
	fullAdder a2 (a[2], b_b[2], c2, sum[2], c3);

	fullAdder f3 (a[3], b_b[3], c3, sum[3], c4);
	fullAdder f4 (a[4], b_b[4], c4, sum[4], c5);
	fullAdder f5 (a[5], b_b[5], c5, sum[5], c6);
	fullAdder f6 (a[6], b_b[6], c6, sum[6], c7);
	fullAdder f7 (a[7], b_b[7], c7, sum[7], c8);
	fullAdder f8 (a[8], b_b[8], c8, sum[8], c9);
	fullAdder f9 (a[9], b_b[9], c9, sum[9], c10);
	fullAdder f10 (a[10], b_b[10], c10, sum[10], c11);
	fullAdder f11 (a[11], b_b[11], c11, sum[11], c12);
	fullAdder f12 (a[12], b_b[12], c12, sum[12], c13);
	fullAdder f13 (a[13], b_b[13], c13, sum[13], c14);
	fullAdder f14 (a[14], b_b[14], c14, sum[14], c15);
	fullAdder f15 (a[15], b_b[15], c15, sum[15], c16);
	fullAdder f16 (a[16], b_b[16], c16, sum[16], c17);
	fullAdder f17 (a[17], b_b[17], c17, sum[17], c18);
	fullAdder f18 (a[18], b_b[18], c18, sum[18], c19);
	fullAdder f19 (a[19], b_b[19], c19, sum[19], c20);
	fullAdder f20 (a[20], b_b[20], c20, sum[20], c21);
	fullAdder f21 (a[21], b_b[21], c21, sum[21], c22);
	fullAdder f22 (a[22], b_b[22], c22, sum[22], c23);
	fullAdder f23 (a[23], b_b[23], c23, sum[23], c24);
	fullAdder f24 (a[24], b_b[24], c24, sum[24], cout);
	xor x0(overFlow, cout, c24);
	assign sum[25] = (overFlow)? cout:sum[24];
endmodule

module adder_26(a, b, cin, sum);
	input signed [25:0]a, b;
	input cin;
	output signed [26:0]sum;
	wire cout;
	wire c1, c2, c3, c4, c5, c6, c7, c8, c9, c10, c11, c12, c13, c14, c15, c16, c17, c18, c19, c20, c21, c22, c23, c24, c25;
	wire overFlow;
	wire signed [25:0]b_b;
	
   assign b_b[0] = (cin)? !b[0]: b[0];
	assign b_b[1] = (cin)? !b[1]: b[1];
	assign b_b[2] = (cin)? !b[2]: b[2];
	assign b_b[3] = (cin)? !b[3]: b[3];
	assign b_b[4] = (cin)? !b[4]: b[4];
	assign b_b[5] = (cin)? !b[5]: b[5];
	assign b_b[6] = (cin)? !b[6]: b[6];
	assign b_b[7] = (cin)? !b[7]: b[7];
	assign b_b[8] = (cin)? !b[8]: b[8];
	assign b_b[9] = (cin)? !b[9]: b[9];
	assign b_b[10] = (cin)? !b[10]: b[10];
	assign b_b[11] = (cin)? !b[11]: b[11];
	assign b_b[12] = (cin)? !b[12]: b[12];
	assign b_b[13] = (cin)? !b[13]: b[13];
	assign b_b[14] = (cin)? !b[14]: b[14];
	assign b_b[15] = (cin)? !b[15]: b[15];
	assign b_b[16] = (cin)? !b[16]: b[16];
	assign b_b[17] = (cin)? !b[17]: b[17];
	assign b_b[18] = (cin)? !b[18]: b[18];
	assign b_b[19] = (cin)? !b[19]: b[19];
	assign b_b[20] = (cin)? !b[20]: b[20];
	assign b_b[21] = (cin)? !b[21]: b[21];
	assign b_b[22] = (cin)? !b[22]: b[22];
	assign b_b[23] = (cin)? !b[23]: b[23];
	assign b_b[24] = (cin)? !b[24]: b[24];
	assign b_b[25] = (cin)? !b[25]: b[25];

	fullAdder a0 (a[0], b_b[0], cin, sum[0], c1);
	fullAdder a1 (a[1], b_b[1], c1, sum[1], c2);
	fullAdder a2 (a[2], b_b[2], c2, sum[2], c3);

	fullAdder f3 (a[3], b_b[3], c3, sum[3], c4);
	fullAdder f4 (a[4], b_b[4], c4, sum[4], c5);
	fullAdder f5 (a[5], b_b[5], c5, sum[5], c6);
	fullAdder f6 (a[6], b_b[6], c6, sum[6], c7);
	fullAdder f7 (a[7], b_b[7], c7, sum[7], c8);
	fullAdder f8 (a[8], b_b[8], c8, sum[8], c9);
	fullAdder f9 (a[9], b_b[9], c9, sum[9], c10);
	fullAdder f10 (a[10], b_b[10], c10, sum[10], c11);
	fullAdder f11 (a[11], b_b[11], c11, sum[11], c12);
	fullAdder f12 (a[12], b_b[12], c12, sum[12], c13);
	fullAdder f13 (a[13], b_b[13], c13, sum[13], c14);
	fullAdder f14 (a[14], b_b[14], c14, sum[14], c15);
	fullAdder f15 (a[15], b_b[15], c15, sum[15], c16);
	fullAdder f16 (a[16], b_b[16], c16, sum[16], c17);
	fullAdder f17 (a[17], b_b[17], c17, sum[17], c18);
	fullAdder f18 (a[18], b_b[18], c18, sum[18], c19);
	fullAdder f19 (a[19], b_b[19], c19, sum[19], c20);
	fullAdder f20 (a[20], b_b[20], c20, sum[20], c21);
	fullAdder f21 (a[21], b_b[21], c21, sum[21], c22);
	fullAdder f22 (a[22], b_b[22], c22, sum[22], c23);
	fullAdder f23 (a[23], b_b[23], c23, sum[23], c24);
	fullAdder f24 (a[24], b_b[24], c24, sum[24], c25);
	fullAdder f25 (a[25], b_b[25], c24, sum[25], cout);
	xor x0(overFlow, cout, c25);
	assign sum[26] = (overFlow)? cout:sum[25];
endmodule


module fullAdder (a, b, cin, sum, cout);
	input a, b, cin;
	output sum, cout;
	wire w1,w2,w3;
	xor x1(w1,a,b);
	xor x2(sum,w1,cin);
	and a1(w2,a,b);
	and a2(w3,w1,cin);
	or o1(cout,w2,w3);
endmodule 

module getcategory (data, category);
    input  logic signed [19:0] data;
    output logic [3:0] category;
    always @(data)
    begin
        if      (data <= 1    &&  data >= -1) begin category = 4'd1; end 
        else if (data <= 3    &&  data >= -3) begin category = 4'd2; end
        else if (data <= 7    &&  data >= -7) begin category = 4'd3; end
        else if (data <=15    &&  data >= -15) begin category = 4'd4; end
        else if (data <=31    &&  data >= -31) begin category = 4'd5; end
        else if (data <=63    &&  data >= -63) begin category = 4'd6; end
        else if (data <=127   &&  data >= -127) begin category = 4'd7; end
        else if (data <=255   &&  data >= -255) begin category = 4'd8; end
        else if (data <=511   &&  data >= -511) begin category = 4'd9; end
        else if (data <=1023  &&  data >= -1023) begin category = 4'd10; end
        else if (data <=2047  &&  data >= -2047) begin category = 4'd11; end
        else if (data <=4095  &&  data >= -4095) begin category = 4'd12; end
        else if (data <=8191  &&  data >= -8191) begin category = 4'd13; end
        else if (data <=16383 &&  data >= -16383) begin category = 4'd14; end
        else if (data <=32767 &&  data >= -32767) begin category = 4'd15; end
        else begin category = 4'd0; end 
    end
endmodule

module dc_huffman_table(category, codeword, code_length);
    input           [3:0] category;
    output          [8:0] codeword;
    output logic    [4:0] code_length;
    always @(category) begin
        case(category)
            4'd0:   begin codeword = 9'b00; code_length = 2;end
            4'd1:   begin codeword = 9'b010; code_length = 3;end
            4'd2:   begin codeword = 9'b011; code_length = 3;end
            4'd3:   begin codeword = 9'b100; code_length = 3;end
            4'd4:   begin codeword = 9'b101; code_length = 3;end
            4'd5:   begin codeword = 9'b110; code_length = 3;end
            4'd6:   begin codeword = 9'b1110; code_length = 4;end
            4'd7:   begin codeword = 9'b11110; code_length = 5;end
            4'd8:   begin codeword = 9'b111110; code_length = 6;end
            4'd9:   begin codeword = 9'b1111110; code_length = 7;end
            4'd10:  begin codeword = 9'b11111110; code_length = 8;end
            4'd11:  begin codeword = 9'b111111110; code_length = 9;end
				default : begin codeword = 9'd0; code_length = 0;end
        endcase
    end
endmodule
// Huffman AC Table
module ac_huffman_table(run, category, ac_codeword, ac_codeword_length);
	input [3:0]run;
	input [3:0]category;
	output logic [15:0] ac_codeword;
	output logic [4:0] ac_codeword_length;

	always @(run or category)
	begin
		case({run, category})
			{4'd0, 4'd0} : begin ac_codeword = 16'b1010; ac_codeword_length = 5'd4; end
			{4'd0, 4'd1} : begin ac_codeword = 16'b00; ac_codeword_length = 5'd2; end
			{4'd0, 4'd2} : begin ac_codeword = 16'b01; ac_codeword_length = 5'd2; end
			{4'd0, 4'd3} : begin ac_codeword = 16'b100; ac_codeword_length = 5'd3; end
			{4'd0, 4'd4} : begin ac_codeword = 16'b1011; ac_codeword_length = 5'd4; end
			{4'd0, 4'd5} : begin ac_codeword = 16'b11010; ac_codeword_length = 5'd5; end
			{4'd0, 4'd6} : begin ac_codeword = 16'b1111000; ac_codeword_length = 5'd7; end
			{4'd0, 4'd7} : begin ac_codeword = 16'b11111000; ac_codeword_length = 5'd8; end
			{4'd0, 4'd8} : begin ac_codeword = 16'b1111110110; ac_codeword_length = 5'd10; end
			{4'd0, 4'd9} : begin ac_codeword = 16'b1111111110000010; ac_codeword_length = 5'd16; end
			{4'd0, 4'd10} : begin ac_codeword = 16'b1111111110000011; ac_codeword_length = 5'd16; end
			{4'd1, 4'd1} : begin ac_codeword = 16'b1100; ac_codeword_length = 5'd4; end
			{4'd1, 4'd2} : begin ac_codeword = 16'b11011; ac_codeword_length = 5'd5; end
			{4'd1, 4'd3} : begin ac_codeword = 16'b1111001; ac_codeword_length = 5'd7; end
			{4'd1, 4'd4} : begin ac_codeword = 16'b111110110; ac_codeword_length = 5'd9; end
			{4'd1, 4'd5} : begin ac_codeword = 16'b11111110110; ac_codeword_length = 5'd11; end
			{4'd1, 4'd6} : begin ac_codeword = 16'b1111111110000100; ac_codeword_length = 5'd16; end
			{4'd1, 4'd7} : begin ac_codeword = 16'b1111111110000101; ac_codeword_length = 5'd16; end
			{4'd1, 4'd8} : begin ac_codeword = 16'b1111111110000110; ac_codeword_length = 5'd16; end
			{4'd1, 4'd9} : begin ac_codeword = 16'b1111111110000111; ac_codeword_length = 5'd16; end
			{4'd1, 4'd10} : begin ac_codeword = 16'b1111111110001000; ac_codeword_length = 5'd16; end
			{4'd2, 4'd1} : begin ac_codeword = 16'b11100; ac_codeword_length = 5'd5; end
			{4'd2, 4'd2} : begin ac_codeword = 16'b11111001; ac_codeword_length = 5'd8; end
			{4'd2, 4'd3} : begin ac_codeword = 16'b1111110111; ac_codeword_length = 5'd10; end
			{4'd2, 4'd4} : begin ac_codeword = 16'b111111110100; ac_codeword_length = 5'd12; end
			{4'd2, 4'd5} : begin ac_codeword = 16'b1111111110001001; ac_codeword_length = 5'd16; end
			{4'd2, 4'd6} : begin ac_codeword = 16'b1111111110001010; ac_codeword_length = 5'd16; end
			{4'd2, 4'd7} : begin ac_codeword = 16'b1111111110001011; ac_codeword_length = 5'd16; end
			{4'd2, 4'd8} : begin ac_codeword = 16'b1111111110001100; ac_codeword_length = 5'd16; end
			{4'd2, 4'd9} : begin ac_codeword = 16'b1111111110001101; ac_codeword_length = 5'd16; end
			{4'd2, 4'd10} : begin ac_codeword = 16'b1111111110001110; ac_codeword_length = 5'd16; end
			{4'd3, 4'd1} : begin ac_codeword = 16'b111010; ac_codeword_length = 5'd6; end
			{4'd3, 4'd2} : begin ac_codeword = 16'b111110111; ac_codeword_length = 5'd9; end
			{4'd3, 4'd3} : begin ac_codeword = 16'b111111110101; ac_codeword_length = 5'd12; end
			{4'd3, 4'd4} : begin ac_codeword = 16'b1111111110001111; ac_codeword_length = 5'd16; end
			{4'd3, 4'd5} : begin ac_codeword = 16'b1111111110010000; ac_codeword_length = 5'd16; end
			{4'd3, 4'd6} : begin ac_codeword = 16'b1111111110010001; ac_codeword_length = 5'd16; end
			{4'd3, 4'd7} : begin ac_codeword = 16'b1111111110010010; ac_codeword_length = 5'd16; end
			{4'd3, 4'd8} : begin ac_codeword = 16'b1111111110010011; ac_codeword_length = 5'd16; end
			{4'd3, 4'd9} : begin ac_codeword = 16'b1111111110010100; ac_codeword_length = 5'd16; end
			{4'd3, 4'd10} : begin ac_codeword = 16'b1111111110010101;ac_codeword_length = 5'd16; end
			{4'd4, 4'd1} : begin ac_codeword = 16'b111011;           ac_codeword_length = 5'd6; end
			{4'd4, 4'd2} : begin ac_codeword = 16'b1111111000;       ac_codeword_length = 5'd10; end
			{4'd4, 4'd3} : begin ac_codeword = 16'b1111111110010110; ac_codeword_length = 5'd16; end
			{4'd4, 4'd4} : begin ac_codeword = 16'b1111111110010111; ac_codeword_length = 5'd16; end
			{4'd4, 4'd5} : begin ac_codeword = 16'b1111111110011000; ac_codeword_length = 5'd16; end
			{4'd4, 4'd6} : begin ac_codeword = 16'b1111111110011001; ac_codeword_length = 5'd16; end
			{4'd4, 4'd7} : begin ac_codeword = 16'b1111111110011010; ac_codeword_length = 5'd16; end
			{4'd4, 4'd8} : begin ac_codeword = 16'b1111111110011011; ac_codeword_length = 5'd16; end
			{4'd4, 4'd9} : begin ac_codeword = 16'b1111111110011100; ac_codeword_length = 5'd16; end
			{4'd4, 4'd10}: begin ac_codeword = 16'b1111111110011101; ac_codeword_length = 5'd16; end
			{4'd5, 4'd1} : begin ac_codeword = 16'b1111010;          ac_codeword_length = 5'd7; end
			{4'd5, 4'd2} : begin ac_codeword = 16'b11111110111;      ac_codeword_length = 5'd11; end
			{4'd5, 4'd3} : begin ac_codeword = 16'b1111111110011110; ac_codeword_length = 5'd16; end
			{4'd5, 4'd4} : begin ac_codeword = 16'b1111111110011111; ac_codeword_length = 5'd16; end
			{4'd5, 4'd5} : begin ac_codeword = 16'b1111111110100000; ac_codeword_length = 5'd16; end
			{4'd5, 4'd6} : begin ac_codeword = 16'b1111111110100001; ac_codeword_length = 5'd16; end
			{4'd5, 4'd7} : begin ac_codeword = 16'b1111111110100010; ac_codeword_length = 5'd16; end
			{4'd5, 4'd8} : begin ac_codeword = 16'b1111111110100011; ac_codeword_length = 5'd16; end
			{4'd5, 4'd9} : begin ac_codeword = 16'b1111111110100100; ac_codeword_length = 5'd16; end
			{4'd5, 4'd10} : begin ac_codeword = 16'b1111111110100101;ac_codeword_length = 5'd16; end
			{4'd6, 4'd1} : begin ac_codeword = 16'b1111011;          ac_codeword_length = 5'd7; end
			{4'd6, 4'd2} : begin ac_codeword = 16'b111111110110;     ac_codeword_length = 5'd12; end
			{4'd6, 4'd3} : begin ac_codeword = 16'b1111111110100110; ac_codeword_length = 5'd16; end
			{4'd6, 4'd4} : begin ac_codeword = 16'b1111111110100111; ac_codeword_length = 5'd16; end
			{4'd6, 4'd5} : begin ac_codeword = 16'b1111111110101000; ac_codeword_length = 5'd16; end
			{4'd6, 4'd6} : begin ac_codeword = 16'b1111111110101001; ac_codeword_length = 5'd16; end
			{4'd6, 4'd7} : begin ac_codeword = 16'b1111111110101010; ac_codeword_length = 5'd16; end
			{4'd6, 4'd8} : begin ac_codeword = 16'b1111111110101011; ac_codeword_length = 5'd16; end
			{4'd6, 4'd9} : begin ac_codeword = 16'b1111111110101100; ac_codeword_length = 5'd16; end
			{4'd6, 4'd10} : begin ac_codeword = 16'b1111111110101101; ac_codeword_length = 5'd16; end
			{4'd7, 4'd1} : begin ac_codeword = 16'b11111010; ac_codeword_length = 5'd8; end
			{4'd7, 4'd2} : begin ac_codeword = 16'b111111110111; ac_codeword_length = 5'd12; end
			{4'd7, 4'd3} : begin ac_codeword = 16'b1111111110101110; ac_codeword_length = 5'd16; end
			{4'd7, 4'd4} : begin ac_codeword = 16'b1111111110101111; ac_codeword_length = 5'd16; end
			{4'd7, 4'd5} : begin ac_codeword = 16'b1111111110110000; ac_codeword_length = 5'd16; end
			{4'd7, 4'd6} : begin ac_codeword = 16'b1111111110110001; ac_codeword_length = 5'd16; end
			{4'd7, 4'd7} : begin ac_codeword = 16'b1111111110110010; ac_codeword_length = 5'd16; end
			{4'd7, 4'd8} : begin ac_codeword = 16'b1111111110110011; ac_codeword_length = 5'd16; end
			{4'd7, 4'd9} : begin ac_codeword = 16'b1111111110110100; ac_codeword_length = 5'd16; end
			{4'd7, 4'd10} : begin ac_codeword = 16'b1111111110110101; ac_codeword_length = 5'd16; end
			{4'd8, 4'd1} : begin ac_codeword = 16'b111111000; ac_codeword_length = 5'd9; end
			{4'd8, 4'd2} : begin ac_codeword = 16'b111111111000000; ac_codeword_length = 5'd15; end
			{4'd8, 4'd3} : begin ac_codeword = 16'b1111111110110110; ac_codeword_length = 5'd16; end
			{4'd8, 4'd4} : begin ac_codeword = 16'b1111111110110111; ac_codeword_length = 5'd16; end
			{4'd8, 4'd5} : begin ac_codeword = 16'b1111111110111000; ac_codeword_length = 5'd16; end
			{4'd8, 4'd6} : begin ac_codeword = 16'b1111111110111001; ac_codeword_length = 5'd16; end
			{4'd8, 4'd7} : begin ac_codeword = 16'b1111111110111010; ac_codeword_length = 5'd16; end
			{4'd8, 4'd8} : begin ac_codeword = 16'b1111111110111011; ac_codeword_length = 5'd16; end
			{4'd8, 4'd9} : begin ac_codeword = 16'b1111111110111100; ac_codeword_length = 5'd16; end
			{4'd8, 4'd10} : begin ac_codeword = 16'b1111111110111101; ac_codeword_length = 5'd16; end
			{4'd9, 4'd1} : begin ac_codeword = 16'b111111001; ac_codeword_length = 5'd9; end
			{4'd9, 4'd2} : begin ac_codeword = 16'b1111111110111110; ac_codeword_length = 5'd16; end
			{4'd9, 4'd3} : begin ac_codeword = 16'b1111111110111111; ac_codeword_length = 5'd16; end
			{4'd9, 4'd4} : begin ac_codeword = 16'b1111111111000000; ac_codeword_length = 5'd16; end
			{4'd9, 4'd5} : begin ac_codeword = 16'b1111111111000001; ac_codeword_length = 5'd16; end
			{4'd9, 4'd6} : begin ac_codeword = 16'b1111111111000010; ac_codeword_length = 5'd16; end
			{4'd9, 4'd7} : begin ac_codeword = 16'b1111111111000011; ac_codeword_length = 5'd16; end
			{4'd9, 4'd8} : begin ac_codeword = 16'b1111111111000100; ac_codeword_length = 5'd16; end
			{4'd9, 4'd9} : begin ac_codeword = 16'b1111111111000101; ac_codeword_length = 5'd16; end
			{4'd9, 4'd10} : begin ac_codeword = 16'b1111111111000110; ac_codeword_length = 5'd16; end
			{4'd10, 4'd1} : begin ac_codeword = 16'b111111010; ac_codeword_length = 5'd9; end
			{4'd10, 4'd2} : begin ac_codeword = 16'b1111111111000111; ac_codeword_length = 5'd16; end
			{4'd10, 4'd3} : begin ac_codeword = 16'b1111111111001000; ac_codeword_length = 5'd16; end
			{4'd10, 4'd4} : begin ac_codeword = 16'b1111111111001001; ac_codeword_length = 5'd16; end
			{4'd10, 4'd5} : begin ac_codeword = 16'b1111111111001010; ac_codeword_length = 5'd16; end
			{4'd10, 4'd6} : begin ac_codeword = 16'b1111111111001011; ac_codeword_length = 5'd16; end
			{4'd10, 4'd7} : begin ac_codeword = 16'b1111111111001100; ac_codeword_length = 5'd16; end
			{4'd10, 4'd8} : begin ac_codeword = 16'b1111111111001101; ac_codeword_length = 5'd16; end
			{4'd10, 4'd9} : begin ac_codeword = 16'b1111111111001110; ac_codeword_length = 5'd16; end
			{4'd10, 4'd10} : begin ac_codeword = 16'b1111111111001111; ac_codeword_length = 5'd16; end
			{4'd11, 4'd1} : begin ac_codeword = 16'b1111111001; ac_codeword_length = 5'd10; end
			{4'd11, 4'd2} : begin ac_codeword = 16'b1111111111010000; ac_codeword_length = 5'd16; end
			{4'd11, 4'd3} : begin ac_codeword = 16'b1111111111010001; ac_codeword_length = 5'd16; end
			{4'd11, 4'd4} : begin ac_codeword = 16'b1111111111010010; ac_codeword_length = 5'd16; end
			{4'd11, 4'd5} : begin ac_codeword = 16'b1111111111010011; ac_codeword_length = 5'd16; end
			{4'd11, 4'd6} : begin ac_codeword = 16'b1111111111010100; ac_codeword_length = 5'd16; end
			{4'd11, 4'd7} : begin ac_codeword = 16'b1111111111010101; ac_codeword_length = 5'd16; end
			{4'd11, 4'd8} : begin ac_codeword = 16'b1111111111010110; ac_codeword_length = 5'd16; end
			{4'd11, 4'd9} : begin ac_codeword = 16'b1111111111010111; ac_codeword_length = 5'd16; end
			{4'd11, 4'd10} : begin ac_codeword = 16'b1111111111011000; ac_codeword_length = 5'd16; end
			{4'd12, 4'd1} : begin ac_codeword = 16'b1111111010; ac_codeword_length = 5'd10; end
			{4'd12, 4'd2} : begin ac_codeword = 16'b1111111111011001; ac_codeword_length = 5'd16; end
			{4'd12, 4'd3} : begin ac_codeword = 16'b1111111111011010; ac_codeword_length = 5'd16; end
			{4'd12, 4'd4} : begin ac_codeword = 16'b1111111111011011; ac_codeword_length = 5'd16; end
			{4'd12, 4'd5} : begin ac_codeword = 16'b1111111111011100; ac_codeword_length = 5'd16; end
			{4'd12, 4'd6} : begin ac_codeword = 16'b1111111111011101; ac_codeword_length = 5'd16; end
			{4'd12, 4'd7} : begin ac_codeword = 16'b1111111111011110; ac_codeword_length = 5'd16; end
			{4'd12, 4'd8} : begin ac_codeword = 16'b1111111111011111; ac_codeword_length = 5'd16; end
			{4'd12, 4'd9} : begin ac_codeword = 16'b1111111111100000; ac_codeword_length = 5'd16; end
			{4'd12, 4'd10}: begin ac_codeword = 16'b1111111111100001; ac_codeword_length = 5'd16; end
			{4'd13, 4'd1} : begin ac_codeword = 16'b11111111000; ac_codeword_length = 5'd11; end
			{4'd13, 4'd2} : begin ac_codeword = 16'b1111111111100010; ac_codeword_length = 5'd16; end
			{4'd13, 4'd3} : begin ac_codeword = 16'b1111111111100011; ac_codeword_length = 5'd16; end
			{4'd13, 4'd4} : begin ac_codeword = 16'b1111111111100100; ac_codeword_length = 5'd16; end
			{4'd13, 4'd5} : begin ac_codeword = 16'b1111111111100101; ac_codeword_length = 5'd16; end
			{4'd13, 4'd6} : begin ac_codeword = 16'b1111111111100110; ac_codeword_length = 5'd16; end
			{4'd13, 4'd7} : begin ac_codeword = 16'b1111111111100111; ac_codeword_length = 5'd16; end
			{4'd13, 4'd8} : begin ac_codeword = 16'b1111111111101000; ac_codeword_length = 5'd16; end
			{4'd13, 4'd9} : begin ac_codeword = 16'b1111111111101001; ac_codeword_length = 5'd16; end
			{4'd13, 4'd10} : begin ac_codeword = 16'b1111111111101010; ac_codeword_length = 5'd16; end
			{4'd14, 4'd1} : begin ac_codeword = 16'b1111111111101011; ac_codeword_length = 5'd16; end
			{4'd14, 4'd2} : begin ac_codeword = 16'b1111111111101100; ac_codeword_length = 5'd16; end
			{4'd14, 4'd3} : begin ac_codeword = 16'b1111111111101101; ac_codeword_length = 5'd16; end
			{4'd14, 4'd4} : begin ac_codeword = 16'b1111111111101110; ac_codeword_length = 5'd16; end
			{4'd14, 4'd5} : begin ac_codeword = 16'b1111111111101111; ac_codeword_length = 5'd16; end
			{4'd14, 4'd6} : begin ac_codeword = 16'b1111111111110000; ac_codeword_length = 5'd16; end
			{4'd14, 4'd7} : begin ac_codeword = 16'b1111111111110001; ac_codeword_length = 5'd16; end
			{4'd14, 4'd8} : begin ac_codeword = 16'b1111111111110010; ac_codeword_length = 5'd16; end
			{4'd14, 4'd9} : begin ac_codeword = 16'b1111111111110011; ac_codeword_length = 5'd16; end
			{4'd14, 4'd10} : begin ac_codeword = 16'b1111111111110100; ac_codeword_length = 5'd16; end
			{4'd15, 4'd1} : begin ac_codeword = 16'b1111111111110101; ac_codeword_length = 5'd16; end
			{4'd15, 4'd2} : begin ac_codeword = 16'b1111111111110110; ac_codeword_length = 5'd16; end
			{4'd15, 4'd3} : begin ac_codeword = 16'b1111111111110111; ac_codeword_length = 5'd16; end
			{4'd15, 4'd4} : begin ac_codeword = 16'b1111111111111000; ac_codeword_length = 5'd16; end
			{4'd15, 4'd5} : begin ac_codeword = 16'b1111111111111001; ac_codeword_length = 5'd16; end
			{4'd15, 4'd6} : begin ac_codeword = 16'b1111111111111010; ac_codeword_length = 5'd16; end
			{4'd15, 4'd7} : begin ac_codeword = 16'b1111111111111011; ac_codeword_length = 5'd16; end
			{4'd15, 4'd8} : begin ac_codeword = 16'b1111111111111100; ac_codeword_length = 5'd16; end
			{4'd15, 4'd9} : begin ac_codeword = 16'b1111111111111101; ac_codeword_length = 5'd16; end
			{4'd15, 4'd10} : begin ac_codeword = 16'b1111111111111110; ac_codeword_length = 5'd16; end
			{4'd15, 4'd0} : begin ac_codeword = 16'b11111111001; ac_codeword_length = 5'd11; end
		endcase
		
	end
endmodule

