/*
 * Seven-segment LED emulator
 *
 * Stephen A. Edwards, Columbia University
 */

module VGA_LED_Emulator(
 input logic 	     clk50, reset,
 input logic [7:0]   map_crl, char_crl, filt_crl,
 input logic [23:0]  write_line_num, 
 input logic [11:0]  color, 
 input logic [15:0]  x_in, y_in,xv_in, yv_in,
 output logic [7:0]  VGA_R, VGA_G, VGA_B,
 output logic 	     VGA_CLK, VGA_HS, VGA_VS, VGA_BLANK_n, VGA_SYNC_n);

   parameter HACTIVE      = 11'd 1280,
             HFRONT_PORCH = 11'd 32,
             HSYNC        = 11'd 192,
             HBACK_PORCH  = 11'd 96,   
             HTOTAL       = HACTIVE + HFRONT_PORCH + HSYNC + HBACK_PORCH; // 1600
   
   // Parameters for vcount
   parameter VACTIVE      = 10'd 480,
             VFRONT_PORCH = 10'd 10,
             VSYNC        = 10'd 2,
             VBACK_PORCH  = 10'd 33,
             VTOTAL       = VACTIVE + VFRONT_PORCH + VSYNC + VBACK_PORCH; // 525

   logic [10:0]			     hcount; // Horizontal counter
                                             // Hcount[10:1] indicates pixel column (0-639)
   logic 			     endOfLine;
   logic [3:0]			R,B,G;			
   
   always_ff @(posedge clk50 or posedge reset)
     if (reset)          hcount <= 0;
     else if (endOfLine) hcount <= 0;
     else  	         hcount <= hcount + 32'd 1;

   assign endOfLine = hcount == HTOTAL - 1;

   logic [9:0] 			     vcount;
   logic 			     endOfField;
   
   always_ff @(posedge clk50 or posedge reset)
     if (reset)          vcount <= 0;
     else if (endOfLine)
       if (endOfField)   vcount <= 0;
       else              vcount <= vcount + 32'd 1;

   assign endOfField = vcount == VTOTAL - 1;

   assign VGA_HS = !( (hcount[10:8] == 3'b101) & !(hcount[7:5] == 3'b111));
   assign VGA_VS = !( vcount[9:1] == (VACTIVE + VFRONT_PORCH) / 2);

   assign VGA_SYNC_n = 1; // For adding sync to video signals; not used for VGA
   
   assign VGA_BLANK_n = !( hcount[10] & (hcount[9] | hcount[8]) ) &
			!( vcount[9] | (vcount[8:5] == 4'b1111) );   

   assign VGA_CLK = hcount[0]; // 25 MHz clock: pixel latched on rising edge


	logic [23:0] read_line_num;
	logic [11:0] color_out;
	logic we;
	logic flash;
	logic nuk;


	logic [10:0] adr_eric;
	logic [11:0] data_ericf1;
	logic [11:0] data_ericb1;
	logic [11:0] data_ericr1;
	logic [11:0] data_ericl1;

	logic [11:0] data_ericf2;
	logic [11:0] data_ericb2;
	logic [11:0] data_ericr2;
	logic [11:0] data_ericl2;

	logic [11:0] data_ericf3;
	logic [11:0] data_ericb3;
	logic [11:0] data_ericr3;
	logic [11:0] data_ericl3;


	logic ericf1_on;
	logic ericb1_on;
	logic ericl1_on;
	logic ericr1_on;

	logic ericf2_on;
	logic ericb2_on;
	logic ericl2_on;
	logic ericr2_on;

	logic ericf3_on;
	logic ericb3_on;
	logic ericl3_on;
	logic ericr3_on;
	logic ericv_on;

	pre_mem_process pre_mem_process(.hc(hc), .vc(vc), .read_line_num(read_line_num), .nuk(nuk), .map_crl(map_crl), .filt_crl(filt_crl), .we(we), .flash(flash));
	memory memory(.clk50(clk50), .write_line_num(write_line_num), .we(we), .read_line_num(read_line_num),.color(color), .color_out(color_out));
	ericf1   ericf1 (.clock(clk50), .address(adr_eric), .q(data_ericf1));
	ericf2   ericf2 (.clock(clk50), .address(adr_eric), .q(data_ericf2));
	ericf3   ericf3 (.clock(clk50), .address(adr_eric), .q(data_ericf3));


	ericb1   ericb1 (.clock(clk50), .address(adr_eric), .q(data_ericb1));
	ericb2   ericb2 (.clock(clk50), .address(adr_eric), .q(data_ericb2));
	ericb3   ericb3 (.clock(clk50), .address(adr_eric), .q(data_ericb3));


	ericl1   ericl1 (.clock(clk50), .address(adr_eric), .q(data_ericl1));
	ericl2   ericl2 (.clock(clk50), .address(adr_eric), .q(data_ericl2));
	ericl3   ericl3 (.clock(clk50), .address(adr_eric), .q(data_ericl3));


	ericr1   ericr1 (.clock(clk50), .address(adr_eric), .q(data_ericr1));
	ericr2   ericr2 (.clock(clk50), .address(adr_eric), .q(data_ericr2));
	ericr3   ericr3 (.clock(clk50), .address(adr_eric), .q(data_ericr3));

	integer hc, vc;

always_latch
	begin
	hc = hcount[10:1];
	vc = vcount;
	if( (hc >= (x_in)) && (hc < (16'd30 + x_in)) && (vc >= (y_in-16'd54) )&& (vc <= y_in))
			begin
			if (char_crl[3:0] == 4'b0001)
				begin
				ericf1_on = 1;
				end
			else if (char_crl[3:0] == 4'b0010)
				begin
				ericf2_on = 1;
				end
			else if (char_crl[3:0] == 4'b0011)
				begin
				ericf3_on = 1;
				end
			else if (char_crl[3:0] == 4'b0100)
				begin
				ericb1_on = 1;
				end
			else if (char_crl[3:0] == 4'b0101)
				begin
				ericb2_on = 1;
				end
			else if (char_crl[3:0] == 4'b0110)
				begin
				ericb3_on = 1;
				end
			else if (char_crl[3:0] == 4'b0111)
				begin
				ericl1_on = 1;
				end
			else if (char_crl[3:0] == 4'b1000)
				begin
				ericl2_on = 1;
				end
			else if (char_crl[3:0] == 4'b1001)
				begin
				ericl3_on = 1;
				end
			else if (char_crl[3:0] == 4'b1010)
				begin
				ericr1_on = 1;
				end
			else if (char_crl[3:0] == 4'b1011)
				begin
				ericr2_on = 1;
				end
			else if (char_crl[3:0] == 4'b1100)
				begin
				ericr3_on = 1;
				end
			else 
				begin
				ericf2_on = 1;
				end

			adr_eric <= ((hc-x_in) + (vc-(y_in-54))*30);
			end
	else 
			begin
			ericf1_on = 0;
			ericb1_on = 0;
			ericl1_on = 0;
			ericr1_on = 0;

			ericf2_on = 0;
			ericb2_on = 0;
			ericl2_on = 0;
			ericr2_on = 0;

			ericf3_on = 0;
			ericb3_on = 0;
			ericl3_on = 0;
			ericr3_on = 0;
			end


	if( (hc >= (xv_in)) && (hc < (16'd30 + xv_in)) && (vc >= (yv_in-16'd54) )&& (vc <= yv_in) && char_crl[7])
			begin
				ericv_on = 1;
				adr_eric <= ((hc-xv_in) + (vc-(yv_in-54))*30);
			end
	else 
			begin
				ericv_on = 0;
			end


	if( data_ericf1[11:0] == 12'd4095 || data_ericf1[11:0] == 12'd3822 || data_ericf1[11:0] == 12'd2730 | data_ericf1[11:0] == 12'd4078|| data_ericf1[11:0] == 12'd3276 || data_ericf1[11:0] == 12'd3003)
			begin
				ericf1_on = 0;
			end
	if( data_ericb1[11:0] == 12'd4095 || data_ericb1[11:0] == 12'd3822 || data_ericb1[11:0] == 12'd2730 || data_ericb1[11:0] == 12'd4078|| data_ericb1[11:0] == 12'd3276 || data_ericb1[11:0] == 12'd3003)
			begin
				ericb1_on = 0;
			end
	if( data_ericl1[11:0] == 12'd4095 || data_ericl1[11:0] == 12'd3822 || data_ericl1[11:0] == 12'd2730 || data_ericl1[11:0] == 12'd4078|| data_ericl1[11:0] == 12'd3276 || data_ericl1[11:0] == 12'd3003)
			begin
				ericl1_on = 0;
			end
	if( data_ericr1[11:0] == 12'd4095 || data_ericr1[11:0] == 12'd3822 || data_ericr1[11:0] == 12'd2730 || data_ericr1[11:0] == 12'd4078|| data_ericr1[11:0] == 12'd3276 || data_ericr1[11:0] == 12'd3003)
			begin
				ericr1_on = 0;
			end



	if( data_ericf2[11:0] == 12'd4095 || data_ericf2[11:0] == 12'd3822 || data_ericf2[11:0] == 12'd2730 || data_ericf2[11:0] == 12'd4078 || data_ericf2[11:0] == 12'd3276 || data_ericf2[11:0] == 12'd3003)
			begin
				ericf2_on = 0;
			end
	if( data_ericb2[11:0] == 12'd4095 || data_ericb2[11:0] == 12'd3822 || data_ericb2[11:0] == 12'd2730 || data_ericb2[11:0] == 12'd4078 || data_ericb2[11:0] == 12'd3276 || data_ericb2[11:0] == 12'd3003)
			begin
				ericb2_on = 0;
			end
	if( data_ericl2[11:0] == 12'd4095 || data_ericl2[11:0] == 12'd3822 || data_ericl2[11:0] == 12'd2730 || data_ericl2[11:0] == 12'd4078 || data_ericl2[11:0] == 12'd3276 || data_ericl2[11:0] == 12'd3003)
			begin
				ericl2_on = 0;
			end
	if( data_ericr2[11:0] == 12'd4095 || data_ericr2[11:0] == 12'd3822 || data_ericr2[11:0] == 12'd2730 || data_ericr2[11:0] == 12'd4078 || data_ericr2[11:0] == 12'd3276 || data_ericr2[11:0] == 12'd3003)
			begin
				ericr2_on = 0;
			end



	if( data_ericf3[11:0] == 12'd4095 || data_ericf3[11:0] == 12'd3822 || data_ericf3[11:0] == 12'd2730 || data_ericf3[11:0] == 12'd4078 || data_ericf3[11:0] == 12'd3276 || data_ericf3[11:0] == 12'd3003)
			begin
				ericf3_on = 0;
			end
	if( data_ericb3[11:0] == 12'd4095 || data_ericb3[11:0] == 12'd3822 || data_ericb3[11:0] == 12'd2730 || data_ericb3[11:0] == 12'd4078 || data_ericb3[11:0] == 12'd3276 || data_ericb3[11:0] == 12'd3003)
			begin
				ericb3_on = 0;
			end
	if( data_ericl3[11:0] == 12'd4095 || data_ericl3[11:0] == 12'd3822 || data_ericl3[11:0] == 12'd2730 || data_ericl3[11:0] == 12'd4078 || data_ericl3[11:0] == 12'd3276 || data_ericl3[11:0] == 12'd3003)
			begin
				ericl3_on = 0;
			end
	if( data_ericr3[11:0] == 12'd4095 || data_ericr3[11:0] == 12'd3822 || data_ericr3[11:0] == 12'd2730 || data_ericr3[11:0] == 12'd4078 || data_ericr3[11:0] == 12'd3276 || data_ericr3[11:0] == 12'd3003)
			begin
				ericr3_on = 0;
			end


	if(ericf1_on)
	begin
			VGA_R = {data_ericf1[11:8],data_ericf1[11:8]};
			VGA_G = {data_ericf1[7:4],data_ericf1[7:4]};
			VGA_B = {data_ericf1[3:0],data_ericf1[3:0]};
	end
	else if(ericf2_on)
	begin
			VGA_R = {data_ericf2[11:8],data_ericf2[11:8]};
			VGA_G = {data_ericf2[7:4],data_ericf2[7:4]};
			VGA_B = {data_ericf2[3:0],data_ericf2[3:0]};
	end
	else if(ericf3_on)
	begin
			VGA_R = {data_ericf3[11:8],data_ericf3[11:8]};
			VGA_G = {data_ericf3[7:4],data_ericf3[7:4]};
			VGA_B = {data_ericf3[3:0],data_ericf3[3:0]};
	end
	else if(ericb1_on)
	begin
			VGA_R = {data_ericb1[11:8],data_ericb1[11:8]};
			VGA_G = {data_ericb1[7:4],data_ericb1[7:4]};
			VGA_B = {data_ericb1[3:0],data_ericb1[3:0]};
	end
	else if(ericb2_on)
	begin
			VGA_R = {data_ericb2[11:8],data_ericb2[11:8]};
			VGA_G = {data_ericb2[7:4],data_ericb2[7:4]};
			VGA_B = {data_ericb2[3:0],data_ericb2[3:0]};
	end
	else if(ericb3_on)
	begin
			VGA_R = {data_ericb3[11:8],data_ericb3[11:8]};
			VGA_G = {data_ericb3[7:4],data_ericb3[7:4]};
			VGA_B = {data_ericb3[3:0],data_ericb3[3:0]};
	end
	else if(ericl1_on)
	begin
			VGA_R = {data_ericl1[11:8],data_ericl1[11:8]};
			VGA_G = {data_ericl1[7:4],data_ericl1[7:4]};
			VGA_B = {data_ericl1[3:0],data_ericl1[3:0]};
	end
	else if(ericl2_on)
	begin
			VGA_R = {data_ericl2[11:8],data_ericl2[11:8]};
			VGA_G = {data_ericl2[7:4],data_ericl2[7:4]};
			VGA_B = {data_ericl2[3:0],data_ericl2[3:0]};
	end
	else if(ericl3_on)
	begin
			VGA_R = {data_ericl3[11:8],data_ericl3[11:8]};
			VGA_G = {data_ericl3[7:4],data_ericl3[7:4]};
			VGA_B = {data_ericl3[3:0],data_ericl3[3:0]};
	end
	else if(ericr1_on)
	begin
			VGA_R = {data_ericr1[11:8],data_ericr1[11:8]};
			VGA_G = {data_ericr1[7:4],data_ericr1[7:4]};
			VGA_B = {data_ericr1[3:0],data_ericr1[3:0]};
	end

	else if(ericr2_on)
	begin
			VGA_R = {data_ericr2[11:8],data_ericr2[11:8]};
			VGA_G = {data_ericr2[7:4],data_ericr2[7:4]};
			VGA_B = {data_ericr2[3:0],data_ericr2[3:0]};
	end
	else if(ericr3_on)
	begin
			VGA_R = {data_ericr3[11:8],data_ericr3[11:8]};
			VGA_G = {data_ericr3[7:4],data_ericr3[7:4]};
			VGA_B = {data_ericr3[3:0],data_ericr3[3:0]};
	end
	else if(flash)
	begin
		VGA_R = {1'b0, color_out[11:9],1'b0, color_out[11:9]};
		VGA_G = {1'b0, color_out[11:9],1'b0, color_out[11:9]};
		VGA_B = {1'b0, color_out[11:9],1'b0, color_out[11:9]}; 
	end
	else if(nuk)
	begin
		VGA_R = {2'b0, color_out[11:10],2'b0, color_out[11:10]};
		VGA_G = {color_out[7:4],2'b0, color_out[7:4]};
		VGA_B = {color_out[3:0],2'b0, color_out[3:0]}; 
	end

	else
	begin
		VGA_R = {color_out[11:8],color_out[11:8]};
		VGA_G = {color_out[7:4],color_out[7:4]};
		VGA_B = {color_out[3:0],color_out[3:0]}; 
	end

	if(ericv_on)
	begin
			VGA_R = {data_ericf2[11:8],data_ericf2[11:8]};
			VGA_G = {data_ericf2[11:8],data_ericf2[11:8]};
			VGA_B = {data_ericf2[11:8],data_ericf2[11:8]};
	end
	end
endmodule


module pre_mem_process (

 input logic [32:0] hc, 
 input logic [32:0] vc,
 input logic [7:0] map_crl,
 input logic [7:0] filt_crl,
 output logic [23:0] read_line_num,
 output logic we,
 output logic flash,
 output logic nuk);

always_comb
begin
	we = map_crl[0];
	flash = filt_crl[0];
	nuk = filt_crl[1];
	read_line_num = hc+vc*640;
end
endmodule

module memory (
 input logic 	    clk50,
 input logic [11:0]  color, 
 input logic [23:0] write_line_num,
 input logic [23:0] read_line_num,
 input logic we,
 output logic [11:0] color_out);

logic [11:0] mem [307200:0];

always_ff @(posedge clk50)
begin
	if (we)
	begin
		mem[write_line_num] <= color;
	end
	color_out[11:0] <= mem[read_line_num];
end
endmodule











































