/*
 * Seven-segment LED emulator
 *
 * Original: Stephen A. Edwards, Columbia University
 * Modified: Blayne Kettlewell
 */

module VGA_SPRITE_EMULATOR(
 input logic 	    clk65, reset,
 output logic [10:0] hCount,
 output logic [9:0] vCount,
 output logic 	    vgaClk, vgaHS, vgaVS, vgaBlank_n, vgaSync_n, hActiveWindow, vActiveWindow, newFrameSync);

/*
 * 640 X 480 VGA timing for a 50 MHz clock: one pixel every other cycle
 *
 * hCountVal 1599 0             1279       1599 0
 *             _______________              ________
 * ___________|    Video      |____________|  Video
 *
 *
 * |SYNC| BP |<-- HACTIVE -->|FP|SYNC| BP |<-- HACTIVE
 *       _______________________      _____________
 * |____|       vgaHS          |____|
 */
   // Parameters for hCountVal
   parameter HACTIVE      = 11'd 1024,
             HFRONT_PORCH = 11'd 24,
             HSYNC        = 11'd 136,
             HBACK_PORCH  = 11'd 160, // XGA
				 //HBACK_PORCH  = 11'd 144, // VESA
             HTOTAL       = HACTIVE + HFRONT_PORCH + HSYNC + HBACK_PORCH; // 1344

   // Parameters for vCountVal
   parameter VACTIVE      = 10'd 768,
             VFRONT_PORCH = 10'd 3,
             VSYNC        = 10'd 6,
             VBACK_PORCH  = 10'd 29,
             VTOTAL       = VACTIVE + VFRONT_PORCH + VSYNC + VBACK_PORCH; // 806

   logic [10:0]			     hCountVal; // Horizontal counter
                                             // hCountVal[10:1] indicates pixel column (0-1023)
   logic 			     endOfLine;

	logic vgaHSBitSliceOne;
	logic vgaHSBitSliceTwo;
	logic vgaHSBitSliceThree;

	logic vgaVSBitSliceOne;
	logic vgaVSBitSliceTwo;
	logic vgaVSBitSliceThree;

   always_ff @(posedge clk65 or posedge reset)
     if (reset)          hCountVal <= 0;
     else if (endOfLine) hCountVal <= 0;
     else  	         hCountVal <= hCountVal + 11'd1;

   assign endOfLine = hCountVal == HTOTAL - 1;
   assign hActiveWindow = hCountVal < HACTIVE;

   // Vertical counter
   logic [9:0] 			     vCountVal;
   logic 			     endOfField;

   always_ff @(posedge clk65 or posedge reset)
     if (reset)          vCountVal <= 0;
     else if (endOfLine)
       if (endOfField)   vCountVal <= 0;
       else              vCountVal <= vCountVal + 10'd1;

  assign endOfField = vCountVal == VTOTAL - 1;
  assign vActiveWindow = vCountVal < VACTIVE;

	assign newFrameSync = endOfField & endOfLine;

   // Horizontal sync: from 0x520 to 0x5DF (0x57F)
   // 100 0001 1000 to 100 1010 0000
   // assign vgaHS = !( (hCountVal[10:8] == 3'b101) & !(hCountVal[7:5] == 3'b111));
   // assign vgaVS = !( vCountVal[9:1] == (VACTIVE + VFRONT_PORCH) / 2);

	//assign vgaHS = !((hCountVal < (HTOTAL - HBACK_PORCH)) & (hCountVal > (HACTIVE + HFRONT_PORCH - 1)));

	assign vgaHSBitSliceOne = (hCountVal[10:8] == 3'b100);
	assign vgaHSBitSliceTwo = !(hCountVal[7] & (hCountVal[6] | hCountVal[5]));
	assign vgaHSBitSliceThree =  ((hCountVal[7] | hCountVal[6] | hCountVal[5]) | (hCountVal[4] & hCountVal[3]));

	assign vgaHS = !(vgaHSBitSliceOne & vgaHSBitSliceTwo & vgaHSBitSliceThree);

	assign vgaVSBitSliceOne = vCountVal[9:4] == 6'b110000;
	assign vgaVSBitSliceTwo = !(vCountVal[3] & (vCountVal[2] | vCountVal[1] | vCountVal[0]));
	assign vgaVSBitSliceThree = (vCountVal[3] | vCountVal[2]) | (vCountVal[1] & vCountVal[0]);

	assign vgaVS = !(vgaVSBitSliceOne & vgaVSBitSliceTwo & vgaVSBitSliceThree);

   assign vgaSync_n = 1; // For adding sync to video signals; not used for VGA

   // Horizontal active: 0 to 1279     Vertical active: 0 to 479
   // 101 0000 0000  1280	       01 1110 0000  480
   // 110 0011 1111  1599	       10 0000 1100  524
   // assign vgaBlank_n = !( hCountVal[10] & (hCountVal[9] | hCountVal[8]) ) & !( vCountVal[9] | (vCountVal[8:5] == 4'b1111) );
	assign vgaBlank_n = !hCountVal[10] & !(vCountVal[9] & vCountVal[8]);
   /* vgaClk is 25 MHz
    *             __    __    __
    * clk65    __|  |__|  |__|
    *
    *             _____       __
    * hCountVal[0]__|     |_____|
    */
   //assign vgaClk = hCountVal[0]; // 25 MHz clock: pixel latched on rising edge
	assign vgaClk = clk65;      // No divide by two for XVGA, distributing 65MHz clock throughout design
	assign vCount = vCountVal;
	assign hCount = hCountVal;


endmodule // VGA_LED_Emulator
