/*
 * Seven-segment LED emulator
 *
 * Stephen A. Edwards, Columbia University
 */

module VGA_LED_Emulator(
 input logic 	    clk50, reset,
 
 /*
 input logic  [23:0] atk_left_pos,
 input logic  [23:0] atk_right_pos,
 */
 input logic  [2:0] opp_state,
 input logic  [1:0] spark_state,
 
 input logic  [1:0] my_state,
 input logic  isleft,
 input logic  [2:0] atk_state,
 input logic  [1:0] idle_state,
 input logic  def_state,
 input logic  [1:0] opp_fist_state,
 
 input logic  [3:0] myhealth_state,
 input logic  [3:0] opphealth_state,
 
 input logic  title_state,
 input logic  start_state,
 
 input logic  [1:0] num_state,
 
 output logic [7:0] VGA_R, VGA_G, VGA_B,
 output logic 	    VGA_CLK, VGA_HS, VGA_VS, VGA_BLANK_n, VGA_SYNC_n);

/*
 * 640 X 480 VGA timing for a 50 MHz clock: one pixel every other cycle
 * 
 * HCOUNT 1599 0             1279       1599 0
 *             _______________              ________
 * ___________|    Video      |____________|  Video
 * 
 * 
 * |SYNC| BP |<-- HACTIVE -->|FP|SYNC| BP |<-- HACTIVE
 *       _______________________      _____________
 * |____|       VGA_HS          |____|
 */
   // Parameters for hcount
   parameter HACTIVE      = 11'd 1280,
             HFRONT_PORCH = 11'd 32,
             HSYNC        = 11'd 192,
             HBACK_PORCH  = 11'd 96,   
             HTOTAL       = HACTIVE + HFRONT_PORCH + HSYNC + HBACK_PORCH; // 1600
   
   // Parameters for vcount
   parameter VACTIVE      = 10'd 480,
             VFRONT_PORCH = 10'd 10,
             VSYNC        = 10'd 2,
             VBACK_PORCH  = 10'd 33,
             VTOTAL       = VACTIVE + VFRONT_PORCH + VSYNC + VBACK_PORCH; // 525

   logic [10:0]			     hcount; // Horizontal counter
                                             // Hcount[10:1] indicates pixel column (0-639)
   logic 			     endOfLine;
   
   always_ff @(posedge clk50 or posedge reset)
     if (reset)          hcount <= 0;
     else if (endOfLine) hcount <= 0;
     else  	         hcount <= hcount + 11'd 1;

   assign endOfLine = hcount == HTOTAL - 1;

   // Vertical counter
   logic [9:0] 			     vcount;
   logic 			     endOfField;
   
   always_ff @(posedge clk50 or posedge reset)
     if (reset)          vcount <= 0;
     else if (endOfLine)
       if (endOfField)   vcount <= 0;
       else              vcount <= vcount + 10'd 1;

   assign endOfField = vcount == VTOTAL - 1;

   // Horizontal sync: from 0x520 to 0x5DF (0x57F)
   // 101 0010 0000 to 101 1101 1111
   assign VGA_HS = !( (hcount[10:8] == 3'b101) & !(hcount[7:5] == 3'b111));
   assign VGA_VS = !( vcount[9:1] == (VACTIVE + VFRONT_PORCH) / 2);

   assign VGA_SYNC_n = 1; // For adding sync to video signals; not used for VGA
   
   // Horizontal active: 0 to 1279     Vertical active: 0 to 479
   // 101 0000 0000  1280	       01 1110 0000  480
   // 110 0011 1111  1599	       10 0000 1100  524
   assign VGA_BLANK_n = !( hcount[10] & (hcount[9] | hcount[8]) ) &
			!( vcount[9] | (vcount[8:5] == 4'b1111) );   

   /* VGA_CLK is 25 MHz
    *             __    __    __
    * clk50    __|  |__|  |__|
    *        
    *             _____       __
    * hcount[0]__|     |_____|
    */
   assign VGA_CLK = hcount[0]; // 25 MHz clock: pixel latched on rising edge



	
	
	
	////////////////START HERE//////////////////////
	
	
	/*******************************data read in*******************************/
	
	//my health state --- 4bits
	logic [3:0] myhealth_state_temp;
	
	//opponent health state --- 4bits
	logic [3:0] opphealth_state_temp;
	
	//opponent position---3bits
	logic [2:0] opp_state_temp;	
	
	//spark---2bits  
	logic [1:0] spark_state_temp;
	
	//my states---2bits
	logic [1:0] my_state_temp;
	
	//left or right attack---1bits
	logic isleft_temp;
	
	//attack position states---3bits
	logic [2:0] atk_state_temp;
			
	//idle position states---2bits
	logic [1:0] idle_state_temp;
	
	//defend states---1bits
	logic def_state_temp;
	
	//opponent fist states---2bits
	logic [1:0] opp_fist_state_temp;
	
	//game title and start label
	logic title_state_temp;
	logic start_state_temp;
	
	//count down number state
	logic [1:0] num_state_temp;
	
	
	
		always_ff @(posedge clk50)
		begin
			if(vcount > 10'd480)
				begin
             opp_state_temp <= opp_state;
				 spark_state_temp <= spark_state;
				 my_state_temp <= my_state;
				 isleft_temp <= isleft;
				 atk_state_temp <= atk_state;
				 idle_state_temp <= idle_state;
				 def_state_temp <= def_state;
				 opp_fist_state_temp <= opp_fist_state;
				 myhealth_state_temp <= myhealth_state;
				 opphealth_state_temp <= opphealth_state;
				 start_state_temp <= start_state;
				 title_state_temp <= title_state;
				 num_state_temp <= num_state;
				end
			else
				begin
				  opp_state_temp <= opp_state_temp;
				  spark_state_temp <= spark_state_temp;
		  		  my_state_temp <= my_state_temp;
				  isleft_temp <= isleft_temp;
				  atk_state_temp <= atk_state_temp;
				  idle_state_temp <= idle_state_temp;
				  def_state_temp <= def_state_temp;
				  opp_fist_state_temp <= opp_fist_state_temp;
				  myhealth_state_temp <= myhealth_state_temp;
				  opphealth_state_temp <= opphealth_state_temp;
				  start_state_temp <= start_state_temp;
				  title_state_temp <= title_state_temp;
				  num_state_temp <= num_state_temp;
				end
		end
	
	
	
	/**********************************************************************************************/
	
	
	/*******************************ROM initialization*******************************/
	
	//background
	logic [23:0] data_background;
	logic [6:0] addr_background;
	bg_slice bg_slice(.address(addr_background), .clock(clk50), .q(data_background));

	
	//attack&idle fist
	logic [23:0] data_atk;
	logic [14:0] addr_atk;
	fist_atk fist_atk(.address(addr_atk), .clock(clk50), .q(data_atk));
	
	
	//defend fist
	logic [23:0] data_def;
	logic [14:0] addr_def;
	fist_def fist_def(.address(addr_def), .clock(clk50), .q(data_def));
	
	
	//opponent
	logic [23:0] data_opp;
	logic [15:0] addr_opp;
	opponent opponent(.address(addr_opp), .clock(clk50), .q(data_opp));
	
	
	//spark
	logic [23:0] data_spark;
	logic [14:0] addr_spark;
	spark spark(.address(addr_spark), .clock(clk50), .q(data_spark));
	
	
   //opponent fist
	logic [23:0] data_opp_fist;
	logic [13:0] addr_opp_fist;
	opp_fist opp_fist(.address(addr_opp_fist), .clock(clk50), .q(data_opp_fist));
	
	
	//game title
	logic data_title;
	logic [14:0] addr_title;
	title title(.address(addr_title), .clock(clk50), .q(data_title));
	
	//start label
	logic data_start;
	logic [12:0] addr_start;
	start start(.address(addr_start), .clock(clk50), .q(data_start));
	
	//count down number 1-3
	logic data_num1;
	logic [12:0] addr_num1;
	num1 num1(.address(addr_num1), .clock(clk50), .q(data_num1));
	
   logic data_num2;
	logic [12:0] addr_num2;
	num2 num2(.address(addr_num2), .clock(clk50), .q(data_num2));
	
	logic data_num3;
	logic [12:0] addr_num3;
	num3 num3(.address(addr_num3), .clock(clk50), .q(data_num3));
	
	
	/*****************************************************************************************/
	
	
	
	/***********************************sprite graphics**********************************/
	
	//count down number 1-3
	logic num1_on;
	logic num2_on;
	logic num3_on;
	
	logic num1_nblank;
	assign num1_nblank = !(data_num1==0);
	
	
	logic num2_nblank;
	assign num2_nblank = !(data_num2==0);
	
	
	logic num3_nblank;
	assign num3_nblank = !(data_num3==0);
	
	always_comb
	  begin
	    if(num_state_temp == 2'b01 & hcount[10:1]>=288 & hcount[10:1]<353 & vcount>=200 &vcount<296)
		    begin
			 num1_on <= 1;
			 num2_on <= 0;
			 num3_on <= 0;
			 addr_num1 <= (hcount[10:1]-288) + (vcount-200)*65; 
			 addr_num2 <= 0;
			 addr_num3 <= 0;
			 end
		 else if(num_state_temp == 2'b10 & hcount[10:1]>=288 & hcount[10:1]<353 & vcount>=200 &vcount<296)
		    begin
			 num1_on <= 0;
			 num2_on <= 1;
			 num3_on <= 0;
			 addr_num2 <= (hcount[10:1]-288) + (vcount-200)*65; 
			 addr_num1 <= 0;
			 addr_num3 <= 0;
			 end
	    else if(num_state_temp == 2'b11 & hcount[10:1]>=288 & hcount[10:1]<353 & vcount>=200 &vcount<296)
		    begin
			 num1_on <= 0;
			 num2_on <= 0;
			 num3_on <= 1;
			 addr_num3 <= (hcount[10:1]-288) + (vcount-200)*65; 
			 addr_num2 <= 0;
			 addr_num1 <= 0;
			 end
		 else
		 	 begin
			 num1_on <= 0;
			 num2_on <= 0;
			 num3_on <= 0;
			 addr_num1 <= 0; 
			 addr_num2 <= 0;
			 addr_num3 <= 0;
			 end
	  end
	
	
	
	//start label and game title
	logic start_on;
	logic title_on;

	always_comb
	  begin
	  	 if(title_state_temp & hcount[10:1]>=145 & hcount[10:1]<495 & vcount>=200 &vcount<265)
	    begin
		 title_on <= 1;
		 addr_title <= (hcount[10:1]-145) + (vcount-200)*350; 
		 end
		 else
	    begin
		 title_on <= 0;
		 addr_title <= 0; 
		 end	  	  
	  end
	
	always_comb
	  begin
	  	 if(start_state_temp & hcount[10:1]>=265 & hcount[10:1]<495 & vcount>=265 &vcount<290)
	    begin
		 start_on <= 1;
		 addr_start <= (hcount[10:1]-265) + (vcount-265)*230; 
		 end
		 else
	    begin
		 start_on <= 0;
		 addr_start <= 0; 
		 end	  	
	  end
	
	
	
	//health bar border
	logic border_on;
		
	always_comb
	  begin
	    if((hcount[10:1]>=0 & hcount[10:1]<=301 & vcount==30) | (hcount[10:1]>=0 & hcount[10:1]<=301 & vcount==51) |
		       (hcount[10:1]>=338 & hcount[10:1]<=639 & vcount==30) | (hcount[10:1]>=338 & hcount[10:1]<=639 & vcount==51) |
				 (hcount[10:1]==301 & vcount>30 & vcount<51) | (hcount[10:1]==338 & vcount>30 & vcount<51))
	       border_on <= 1;
		 else 
		    border_on <= 0;
	  end
	  
	  
	//health bar
	logic myhealth_on;
	logic opphealth_on;
	
	
	always_comb
	  begin
	   if(hcount[10:1]>=0 & hcount[10:1]<=(20*myhealth_state_temp) & vcount>30 & vcount<51)
		  myhealth_on <= 1;
		else
		  myhealth_on <= 0;	  
	  end
	
	always_comb
	  begin
	    if(hcount[10:1]<=639 & hcount[10:1]>=(639-20*opphealth_state_temp) & vcount>30 & vcount<51)
	      opphealth_on <= 1;
	    else
		   opphealth_on <= 0;
	  end
	
	
	
	
	//background
	logic bg_up_on;
	logic bg_mid_on;
	logic bg_btm_on;
	
	always_comb
	  begin
	    if(vcount<220)
		  begin
		  bg_up_on <= 1;
		  bg_mid_on <= 0;
		  bg_btm_on <= 0;
		  addr_background <= 0;
		  end
		 else if(vcount>=220 & vcount<345)
		  begin
		  bg_up_on <= 0;
		  bg_mid_on <= 1;
		  bg_btm_on <= 0;
		  addr_background <= (vcount-220);
		  end
		 else
		  begin
		  bg_up_on <= 0;
		  bg_mid_on <= 0;
		  bg_btm_on <= 1;
		  addr_background <= 0;
		  end
	  end


    

	 //opponent
	 logic opp_nblank;
	 assign opp_nblank = !(data_opp == 24'd16777215);
	 
	 logic opp_on;
	 
	 always_comb
	  begin
	   if(opp_state_temp == 3'b000 & hcount[10:1]>=220 & hcount[10:1]<420 & vcount>=180 &vcount<480)
	    begin
		 opp_on <= 1;
		 addr_opp <= (hcount[10:1]-220) + (vcount-180)*200; 
		 end
		else if(opp_state_temp == 3'b001 & hcount[10:1]>=215 & hcount[10:1]<415 & vcount>=180 &vcount<480)
		 begin
		 opp_on <= 1;
		 addr_opp <= (hcount[10:1]-215) + (vcount-180)*200;;
		 end
		else if(opp_state_temp == 3'b010 & hcount[10:1]>=225 & hcount[10:1]<425 & vcount>=180 &vcount<480)
		 begin
		 opp_on <= 1;
		 addr_opp <= (hcount[10:1]-225) + (vcount-180)*200;;
		 end
	  	else if(opp_state_temp == 3'b011 & hcount[10:1]>=180 & hcount[10:1]<380 & vcount>=180 &vcount<480)
		 begin
		 opp_on <= 1;
		 addr_opp <= (hcount[10:1]-180) + (vcount-180)*200;;
		 end
	  	else if(opp_state_temp == 3'b100 & hcount[10:1]>=260 & hcount[10:1]<460 & vcount>=180 &vcount<480)
		 begin
		 opp_on <= 1;
		 addr_opp <= (hcount[10:1]-260) + (vcount-180)*200;;
		 end
	  else if(opp_state_temp == 3'b101 & hcount[10:1]>=140 & hcount[10:1]<340 & vcount>=180 &vcount<480)
		 begin
		 opp_on <= 1;
		 addr_opp <= (hcount[10:1]-140) + (vcount-180)*200;;
		 end
	  else if(opp_state_temp == 3'b110 & hcount[10:1]>=300 & hcount[10:1]<500 & vcount>=180 &vcount<480)
		 begin
		 opp_on <= 1;
		 addr_opp <= (hcount[10:1]-300) + (vcount-180)*200;;
		 end		
	  else
		 begin
		 opp_on <= 0;
		 addr_opp <= 0;
		 end
	  end
	



	//opponent spark
   logic spark_nblank;
   assign spark_nblank = !(data_spark == 24'd16777215);
	
	logic spark_on;
	
	always_comb
	  begin
	   if(spark_state_temp == 2'b10 & hcount[10:1]>=275 & hcount[10:1]<395 & vcount>=170 &vcount<320)
		  begin
		  spark_on <= 1;
		  addr_spark <= (hcount[10:1]-275) + (vcount-170)*120;
		  end
		else if(spark_state_temp == 2'b11 & hcount[10:1]>=245 & hcount[10:1]<365 & vcount>=170 &vcount<320)
		  begin
		  spark_on <= 1;
		  addr_spark <= 119 - (hcount[10:1]-245) + (vcount-180)*120;
		  end
	   else
		  begin
		  spark_on <= 0;
		  addr_spark <= 0;
		  end
	  end
	
	
	
	//opponent fist
	logic opp_fist_nblank;
	assign opp_fist_nblank = !(data_opp_fist==24'd16777215);
   
   logic opp_fist_on;

	
	always_comb
	  begin
	    if(opp_fist_state_temp==2'b11 & hcount[10:1]>=205 & hcount[10:1]<435 & vcount>=245 & vcount<475)
         begin
		   addr_opp_fist <= 114-(hcount[10:2]-102) + (vcount[9:1]-122)*115;
			opp_fist_on <= 1;
	      end
		 else if(opp_fist_state_temp==2'b10 & hcount[10:1]>=263 & hcount[10:1]<378 & vcount>=302 & vcount<417)
		   begin
			addr_opp_fist <= 114-(hcount[10:1]-263) + (vcount-302)*115;
			opp_fist_on <= 1;
			end
		 else
		   begin
		   addr_opp_fist <= 0;
			opp_fist_on <= 0;
	      end		
	  end
   
   
	//my states
	logic atk_nblank;
	assign atk_nblank = !(data_atk == 24'd16777215);
	
	logic def_nblank;
	assign def_nblank = !(data_def == 24'd16777215);
	
	logic left_atk_on;
	logic right_atk_on;
	logic def_on;
	
   always_comb
      begin
	     if(my_state_temp == 2'b00)//attack
		    begin
			  if(!isleft_temp)//right fist attack
			    begin
				  if(atk_state_temp == 3'b000 & hcount[10:1]>=378 & hcount[10:1]<503 & vcount>=310 & vcount<480)
				    begin
					  def_on <= 0;
					  addr_def <= 0;
					  left_atk_on <= 0;
			        right_atk_on <= 1;
			        addr_atk <= (hcount[10:1]-378) + (vcount - 310)*125;
					 end
					else if(atk_state_temp == 3'b001 & hcount[10:1]>=366 & hcount[10:1]<491 & vcount>=300 & vcount<470)
				    begin
					  def_on <= 0;
					  addr_def <= 0;
					  left_atk_on <= 0;
			        right_atk_on <= 1;
			        addr_atk <= (hcount[10:1]-366) + (vcount - 300)*125;
					 end
					else if(atk_state_temp == 3'b010 & hcount[10:1]>=354 & hcount[10:1]<479 & vcount>=290 & vcount<460)
				    begin
					  def_on <= 0;
					  addr_def <= 0;
					  left_atk_on <= 0;
			        right_atk_on <= 1;
			        addr_atk <= (hcount[10:1]-354) + (vcount - 290)*125;
					 end
					else if(atk_state_temp == 3'b011 & hcount[10:1]>=342 & hcount[10:1]<467 & vcount>=280 & vcount<450)
				    begin
					  def_on <= 0;
					  addr_def <= 0;
					  left_atk_on <= 0;
			        right_atk_on <= 1;
			        addr_atk <= (hcount[10:1]-342) + (vcount - 280)*125;
					 end
					else if(atk_state_temp == 3'b100 & hcount[10:1]>=330 & hcount[10:1]<455 & vcount>=270 & vcount<440)
				    begin
					  def_on <= 0;
					  addr_def <= 0;
					  left_atk_on <= 0;
			        right_atk_on <= 1;
			        addr_atk <= (hcount[10:1]-330) + (vcount - 270)*125;
					 end
					else if(atk_state_temp == 3'b101 & hcount[10:1]>=318 & hcount[10:1]<443 & vcount>=260 & vcount<430)
				    begin
					  def_on <= 0;
					  addr_def <= 0;
					  left_atk_on <= 0;
			        right_atk_on <= 1;
			        addr_atk <= (hcount[10:1]-318) + (vcount - 260)*125;
					 end
					else if(atk_state_temp == 3'b110 & hcount[10:1]>=306 & hcount[10:1]<431 & vcount>=250 & vcount<420)
				    begin
					  def_on <= 0;
					  addr_def <= 0;
					  left_atk_on <= 0;
			        right_atk_on <= 1;
			        addr_atk <= (hcount[10:1]-306) + (vcount - 250)*125;
					 end	
				   else if(atk_state_temp == 3'b111 & hcount[10:1]>=294 & hcount[10:1]<419 & vcount>=240 & vcount<410)
				    begin
					  def_on <= 0;
					  addr_def <= 0;
					  left_atk_on <= 0;
			        right_atk_on <= 1;
			        addr_atk <= (hcount[10:1]-294) + (vcount - 240)*125;
					 end	
		         else if(hcount[10:1]>=125 & hcount[10:1]<250 & vcount>=320 & vcount<490)	
				    begin
					  def_on <= 0;
					  addr_def <= 0;
					  left_atk_on <= 1;
			        right_atk_on <= 0;
			        addr_atk <= (124-(hcount[10:1]-125)) + (vcount - 320)*125;
					 end
		         else 
		          begin
					  def_on <= 0;
					  addr_def <= 0;
					  left_atk_on <= 0;
			        right_atk_on <= 0;
			        addr_atk <= 0;		           
           		 end			 			 
				 end
			  else //left fist attack
			    begin
				   if(atk_state_temp == 3'b000 & hcount[10:1]>=137 & hcount[10:1]<262 & vcount>=310 & vcount<480)
				    begin
					  def_on <= 0;
					  addr_def <= 0;
					  left_atk_on <= 1;
			        right_atk_on <= 0;
			        addr_atk <= 124 - (hcount[10:1]-137) + (vcount - 310)*125;
					 end
					else if(atk_state_temp == 3'b001 & hcount[10:1]>=149 & hcount[10:1]<274 & vcount>=300 & vcount<470)
				    begin
					  def_on <= 0;
					  addr_def <= 0;
					  left_atk_on <= 1;
			        right_atk_on <= 0;
			        addr_atk <= 124 - (hcount[10:1]-149) + (vcount - 300)*125;
					 end
					else if(atk_state_temp == 3'b010 & hcount[10:1]>=161 & hcount[10:1]<286 & vcount>=290 & vcount<460)
				    begin
					  def_on <= 0;
					  addr_def <= 0;
					  left_atk_on <= 1;
			        right_atk_on <= 0;
			        addr_atk <= 124 - (hcount[10:1]-161) + (vcount - 290)*125;
					 end
					else if(atk_state_temp == 3'b011 & hcount[10:1]>=173 & hcount[10:1]<298 & vcount>=280 & vcount<450)
				    begin
					  def_on <= 0;
					  addr_def <= 0;
					  left_atk_on <= 1;
			        right_atk_on <= 0;
			        addr_atk <= 124 - (hcount[10:1]-173) + (vcount - 280)*125;
					 end
					else if(atk_state_temp == 3'b100 & hcount[10:1]>=185 & hcount[10:1]<310 & vcount>=270 & vcount<440)
				    begin
					  def_on <= 0;
					  addr_def <= 0;
					  left_atk_on <= 1;
			        right_atk_on <= 0;
			        addr_atk <= 124 - (hcount[10:1]-185) + (vcount - 270)*125;
					 end
					else if(atk_state_temp == 3'b101 & hcount[10:1]>=197 & hcount[10:1]<322 & vcount>=260 & vcount<430)
				    begin
					  def_on <= 0;
					  addr_def <= 0;
					  left_atk_on <= 1;
			        right_atk_on <= 0;
			        addr_atk <= 124 - (hcount[10:1]-197) + (vcount - 260)*125;
					 end
					else if(atk_state_temp == 3'b110 & hcount[10:1]>=209 & hcount[10:1]<334 & vcount>=250 & vcount<420)
				    begin
					  def_on <= 0;
					  addr_def <= 0;
					  left_atk_on <= 1;
			        right_atk_on <= 0;
			        addr_atk <= 124 - (hcount[10:1]-209) + (vcount - 250)*125;
					 end	
				   else if(atk_state_temp == 3'b111 & hcount[10:1]>=221 & hcount[10:1]<346 & vcount>=240 & vcount<410)
				    begin
					  def_on <= 0;
					  addr_def <= 0;
					  left_atk_on <= 1;
			        right_atk_on <= 0;
			        addr_atk <= 124 - (hcount[10:1]-221) + (vcount - 240)*125;
					 end	
		         else if(hcount[10:1]>=390 & hcount[10:1]<515 & vcount>=320 & vcount<490)	
				    begin
					  def_on <= 0;
					  addr_def <= 0;
					  left_atk_on <= 1;
			        right_atk_on <= 0;
			        addr_atk <= (hcount[10:1]-390) + (vcount - 320)*125;
					 end
		         else 
		          begin
					  def_on <= 0;
					  addr_def <= 0;
					  left_atk_on <= 0;
			        right_atk_on <= 0;
			        addr_atk <= 0;		           
           		 end			 
				 end

			 end

	  	  else if(my_state_temp == 2'b01)//defend
		    begin
		  	 if(def_state_temp==1)//not hurt
		  	 begin
			 if(hcount[10:1]>=190 & hcount[10:1]<315 & vcount>=310 & vcount<480)
		    begin
		      def_on <= 1;
			   addr_def <= (124-(hcount[10:1]-190)) + (vcount-310)*125;
			   left_atk_on <= 0;
			   right_atk_on <= 0;
			   addr_atk <= 0;	
			   end
		    else if(hcount[10:1]>=325 & hcount[10:1]<450 & vcount>=310 & vcount<480)
		    begin
		      def_on <= 1;
		      addr_def <= (hcount[10:1]-325) + (vcount-310)*125;
			   left_atk_on <= 0;
			   right_atk_on <= 0;
			   addr_atk <= 0;	
		      end
		    else
		    begin
		      def_on <= 0;
		      addr_def <= 0;
			   left_atk_on <= 0;
			   right_atk_on <= 0;
			   addr_atk <= 0;	
		      end
			 end
		  	 else//hurt
		  	 begin
			 if(hcount[10:1]>=190 & hcount[10:1]<315 & vcount>=320 & vcount<490)
		      begin
		      def_on <= 1;
			   addr_def <= (124-(hcount[10:1]-190)) + (vcount-320)*125;
			   left_atk_on <= 0;
			   right_atk_on <= 0;
			   addr_atk <= 0;	
			   end
		    else if(hcount[10:1]>=325 & hcount[10:1]<450 & vcount>=320 & vcount<490)
		      begin
		      def_on <= 1;
		      addr_def <= (hcount[10:1]-325) + (vcount-320)*125;
			   left_atk_on <= 0;
			   right_atk_on <= 0;
			   addr_atk <= 0;	
		      end
		    else
		      begin
		      def_on <= 0;
		      addr_def <= 0;
			   left_atk_on <= 0;
			   right_atk_on <= 0;
			   addr_atk <= 0;	
		      end
			 end			 
			 
		    end


    	  else if(my_state_temp == 2'b10)//idle
		    begin
             if(idle_state_temp == 2'b00)
				   begin
					if(hcount[10:1]>=390 & hcount[10:1]<515 & vcount>=320 & vcount<490)
				    begin
					  def_on <= 0;
					  addr_def <= 0;
					  left_atk_on <= 0;
			        right_atk_on <= 1;
			        addr_atk <= (hcount[10:1]-390) + (vcount - 320)*125;
					 end
			      else if(hcount[10:1]>=125 & hcount[10:1]<250 & vcount>=320 & vcount<490)	
				    begin
					  def_on <= 0;
					  addr_def <= 0;
					  left_atk_on <= 1;
			        right_atk_on <= 0;
			        addr_atk <= (124-(hcount[10:1]-125)) + (vcount - 320)*125;
					 end
					else
					 begin
				    def_on <= 0;
		          addr_def <= 0;
			       left_atk_on <= 0;
			       right_atk_on <= 0;
			       addr_atk <= 0;	
					 end
	            end
					
				 else if(idle_state_temp == 2'b01)
				   begin
					if(hcount[10:1]>=385 & hcount[10:1]<510 & vcount>=320 & vcount<490)
				    begin
					  def_on <= 0;
					  addr_def <= 0;
					  left_atk_on <= 0;
			        right_atk_on <= 1;
			        addr_atk <= (hcount[10:1]-385) + (vcount - 320)*125;
					 end
			      else if(hcount[10:1]>=120 & hcount[10:1]<245 & vcount>=320 & vcount<490)	
				    begin
					  def_on <= 0;
					  addr_def <= 0;
					  left_atk_on <= 1;
			        right_atk_on <= 0;
			        addr_atk <= (124-(hcount[10:1]-120)) + (vcount - 320)*125;
					 end
					else
					 begin
				    def_on <= 0;
		          addr_def <= 0;
			       left_atk_on <= 0;
			       right_atk_on <= 0;
			       addr_atk <= 0;	
					 end
	            end

				 else if(idle_state_temp == 2'b10)
				   begin
					if(hcount[10:1]>=395 & hcount[10:1]<520 & vcount>=320 & vcount<490)
				    begin
					  def_on <= 0;
					  addr_def <= 0;
					  left_atk_on <= 0;
			        right_atk_on <= 1;
			        addr_atk <= (hcount[10:1]-395) + (vcount - 320)*125;
					 end
			      else if(hcount[10:1]>=130 & hcount[10:1]<255 & vcount>=320 & vcount<490)	
				    begin
					  def_on <= 0;
					  addr_def <= 0;
					  left_atk_on <= 1;
			        right_atk_on <= 0;
			        addr_atk <= (124-(hcount[10:1]-130)) + (vcount - 320)*125;
					 end
					else
					 begin
				    def_on <= 0;
		          addr_def <= 0;
			       left_atk_on <= 0;
			       right_atk_on <= 0;
			       addr_atk <= 0;	
					 end
	            end
					
				 else
					 begin
				    def_on <= 0;
		          addr_def <= 0;
			       left_atk_on <= 0;
			       right_atk_on <= 0;
			       addr_atk <= 0;	
					 end		
		    end

		  else //hurt
		    begin
			   def_on <= 0;
			   addr_def <= 0;
			   left_atk_on <= 0;
			   right_atk_on <= 0;
			   addr_atk <= 0;	
	       end
	
		end	
	

	
	/*****************************************************************************************/
	
	
	/**********************************priority controller**********************************/
   always_comb
    begin
	   if(start_on)
		 {VGA_R, VGA_G, VGA_B} = data_start * 24'd16777215;
	   
      else if(title_on)
		 {VGA_R, VGA_G, VGA_B} = data_title * 24'd16777215;
		 
		else if(num1_on & num1_nblank)
		 {VGA_R, VGA_G, VGA_B} = data_num1 * 24'd8496056;
	
	   else if(num2_on & num2_nblank)
		 {VGA_R, VGA_G, VGA_B} = data_num2 * 24'd8496056;
		
	   else if(num3_on & num3_nblank)
		 {VGA_R, VGA_G, VGA_B} = data_num3 * 24'd8496056;
	
		
	   else if(my_state_temp == 2'b11 & opp_fist_on & opp_fist_nblank)
		 {VGA_R, VGA_G, VGA_B} = data_opp_fist;
		
		else if((left_atk_on | right_atk_on) & atk_nblank)
		 {VGA_R, VGA_G, VGA_B} = data_atk;

		else if(def_on & def_nblank)
		 {VGA_R, VGA_G, VGA_B} = data_def; 
		
		else if(opp_fist_on & opp_fist_nblank)
   	 {VGA_R, VGA_G, VGA_B} = data_opp_fist;
		
		else if(spark_on & spark_nblank)
		 {VGA_R, VGA_G, VGA_B} = data_spark;
		 
		else if(opp_on & opp_nblank)
		 {VGA_R, VGA_G, VGA_B} = data_opp; 
		
		else if(border_on & (!title_state_temp))
		  {VGA_R, VGA_G, VGA_B} = 24'd0;
		  
		else if(myhealth_on & (!title_state_temp))
		  {VGA_R, VGA_G, VGA_B} = {8'hff,8'h0,8'h0};
		
		else if(opphealth_on & (!title_state_temp))
		  {VGA_R, VGA_G, VGA_B} = {8'h0,8'h0,8'hff};
		
		else if(bg_up_on& (!title_state_temp))
		 {VGA_R, VGA_G, VGA_B} = 24'd16777215;
		
		else if(bg_mid_on& (!title_state_temp))
		  {VGA_R, VGA_G, VGA_B} = data_background;
		
		else if(bg_btm_on& (!title_state_temp))
		  {VGA_R, VGA_G, VGA_B} = 24'd8496056;
		  
		    		
		else
		  {VGA_R, VGA_G, VGA_B} = 24'd16777215;;
		
		
		  
	 end 
	
   /*****************************************************************************************/
endmodule // VGA_LED_Emulator
