/*
 * Bresenham's line drawing algorithm
 * 
 * Draw a straight line of pixels between (x0, y0) and (x1, y1) (inclusive)
 * Assert start when coordinates valid; done asserted when final pixel is
 * produced. When "plot" is asserted, (x,y) is the point to plot
 * 
 * Stephen A. Edwards
 */
module bresenham(input logic         clk, reset, start,
		 input logic [10:0]  x0, y0, x1, y1,
		 output logic [10:0] x, y,
		 output logic 	     done, plot);

   logic signed [11:0] 		     dx, dy, err, e2;
   logic 			     down, right;   
   
   typedef enum logic {IDLE, RUNNING} state_t;
   state_t state;

   always_ff @(posedge clk) begin
      done <= 0;
      plot <= 0;      
      if (reset)
	state <= IDLE;
      else
	case (state)
	  IDLE:
	    if (start) begin
	       dx = x1 - x0;
	       right = dx >= 0;
	       if (~right) dx = -dx;
	       dy = y1 - y0;
	       down = dy >= 0;
	       if (down) dy = -dy;
	       err = dx + dy;
	       x <= x0;
	       y <= y0;
	       plot <= 1;
	       state <= RUNNING;		
	    end
	  
	  RUNNING:
	    if (x == x1 && y == y1) begin
	       done <= 1;
	       state <= IDLE;		
	    end else begin
	       plot <= 1;
	       e2 = err << 1;
	       if (e2 > dy) begin
		  err += dy;		  
		  if (right) x <= x + 10'd 1;
		  else       x <= x - 10'd 1;
	       end
	       if (e2 < dx) begin
		  err += dx;
		  if (down) y <= y + 10'd 1;
		  else      y <= y - 10'd 1;
	       end
	    end
	  
	  default:
	    state <= IDLE;
	  
	endcase
   end     
endmodule
