/*
 * Device driver for the bouncing ball Emulator
 *
 * A Platform device implemented using the misc subsystem
 *
 * Jihua Li Wenbei Yu
 * jl4345		wy2228	 
 * Columbia University
 *
 * References:
 * Linux source: Documentation/driver-model/platform.txt
 *               drivers/misc/arm-charlcd.c
 * http://www.linuxforu.com/tag/linux-device-drivers/
 * http://free-electrons.com/docs/
 *
 * "make" to build
 * insmod vga_led.ko
 *
 * Check code style with
 */

#include <linux/module.h>
#include <linux/init.h>
#include <linux/errno.h>
#include <linux/version.h>
#include <linux/kernel.h>
#include <linux/platform_device.h>
#include <linux/miscdevice.h>
#include <linux/slab.h>
#include <linux/io.h>
#include <linux/of.h>
#include <linux/of_address.h>
#include <linux/fs.h>
#include <linux/uaccess.h>
#include "ball.h"

#define DRIVER_NAME "ball"

/*
 * Information about our device
 */
struct ball_dev {
	struct resource res; /* Resource: our registers */
	void __iomem *virtbase; /* Where registers can be accessed in memory */
	unsigned int data[BALL_REG_CNT];
} dev;

/*
 * Write segments of a single digit
 * Assumes digit is in range and the device information has been set up
 */
//x,y,b,r
//elliptical formular with a=1 b on nominator default 10
//each parameter counts for 2 add
static void write_reg(int addr,unsigned int data)
{
	iowrite32(data, dev.virtbase + addr*4);
	  dev.data[addr] = data;
}

/*
 * Handle ioctl() calls from userspace:
 * Read or write the segments on single digits.
 * Note extensive error checking of arguments
 */
static unsigned int read_reg(unsigned int addr)
{
	unsigned int data;
	data=ioread32(dev.virtbase+addr*4);
	return data;
}
static long ball_ioctl(struct file *f, unsigned int cmd, unsigned long arg)
{
	ball_arg_t balla;

	switch (cmd) {
	case BALL_WRITE_REG:
		if (copy_from_user(&balla, (ball_arg_t *) arg,
				   sizeof(ball_arg_t)))
			return -EACCES;
		if (balla.addr > BALL_REG_CNT)
			return -EINVAL;
////////////////////////////////////////////////////////////////////////////
		write_reg(balla.addr, balla.reg_data);
		break;

	case BALL_READ_REG:
		if (copy_from_user(&balla, (ball_arg_t *) arg,
				   sizeof(ball_arg_t)))
			return -EACCES;
		if (balla.addr > BALL_REG_CNT)
			return -EINVAL;
		balla.reg_data = read_reg(balla.addr);
		if (copy_to_user((ball_arg_t *) arg, &balla,
				 sizeof(ball_arg_t)))
			return -EACCES;
		break;

	default:
		return -EINVAL;
	}

	return 0;
}

/* The operations our device knows how to do */
static const struct file_operations ball_fops = {
	.owner		= THIS_MODULE,
	.unlocked_ioctl = ball_ioctl,
};

/* Information about our device for the "misc" framework -- like a char dev */
static struct miscdevice ball_misc_device = {
	.minor		= MISC_DYNAMIC_MINOR,
	.name		= DRIVER_NAME,
	.fops		= &ball_fops,
};

/*
 * Initialization code: get resources (registers) and display
 * a welcome message
 */
static int __init ball_probe(struct platform_device *pdev)
{
	int ret;

	ret = misc_register(&ball_misc_device);

	/* Get the address of our registers from the device tree */
	ret = of_address_to_resource(pdev->dev.of_node, 0, &dev.res);
	if (ret) {
		ret = -ENOENT;
		goto out_deregister;
	}

	/* Make sure we can use these registers */
	if (request_mem_region(dev.res.start, resource_size(&dev.res),
			       DRIVER_NAME) == NULL) {
		ret = -EBUSY;
		goto out_deregister;
	}

	/* Arrange access to our registers */
	dev.virtbase = of_iomap(pdev->dev.of_node, 0);
	if (dev.virtbase == NULL) {
		ret = -ENOMEM;
		goto out_release_mem_region;
	}
///////////////////////////////////////////////////////////
	/* Display a welcome message */

	return 0;

out_release_mem_region:
	release_mem_region(dev.res.start, resource_size(&dev.res));
out_deregister:
	misc_deregister(&ball_misc_device);
	return ret;
}

/* Clean-up code: release resources */
static int ball_remove(struct platform_device *pdev)
{
	iounmap(dev.virtbase);
	release_mem_region(dev.res.start, resource_size(&dev.res));
	misc_deregister(&ball_misc_device);
	return 0;
}

/* Which "compatible" string(s) to search for in the Device Tree */
#ifdef CONFIG_OF
static const struct of_device_id ball_of_match[] = {
	{ .compatible = "altr,ball" },
	{},
};
MODULE_DEVICE_TABLE(of, ball_of_match);
#endif

/* Information for registering ourselves as a "platform" driver */
static struct platform_driver ball_driver = {
	.driver	= {
		.name	= DRIVER_NAME,
		.owner	= THIS_MODULE,
		.of_match_table = of_match_ptr(ball_of_match),
	},
	.remove	= __exit_p(ball_remove),
};

/* Called when the module is loaded: set things up */
static int __init ball_init(void)
{
	pr_info(DRIVER_NAME ": init\n");
	return platform_driver_probe(&ball_driver, ball_probe);
}

/* Called when the module is unloaded: release resources */
static void __exit ball_exit(void)
{
	platform_driver_unregister(&ball_driver);
	pr_info(DRIVER_NAME ": exit\n");
}

module_init(ball_init);
module_exit(ball_exit);

MODULE_LICENSE("GPL");
MODULE_AUTHOR("Jihua Li Wenbei Yu");
MODULE_DESCRIPTION("BALL Emulator");
