/*
 * Seven-segment LED emulator
 *
 * Stephen A. Edwards, Columbia University
 */

module VGA_LED_Emulator(
 input logic 	    clk50, reset,
 //input logic [7:0]  hex0, hex1, hex2, hex3, hex4, hex5, hex6, hex7,
 input logic [9:0] x_coord,
 input logic [9:0] y_coord,
 input logic [9:0] radius,
 //----------------------------------------
 input logic [9:0] disp_stock_x [399:0],
 input logic [9:0] disp_stock_a1 [399:0],
 input logic [9:0] disp_stock_a2 [399:0],
 input logic [9:0] disp_stock_a3 [399:0],
 input logic send_to_vga,
 input logic [6:0] stock_name,
 input logic [1:0] bhs,
 //----------------------------------------
input logic [3:0] S_hundredthousand,
input logic [3:0] S_tenthousand,
input logic [3:0] S_thousand,
input logic [3:0] S_hundred,
input logic [3:0] S_ten,
input logic [3:0] S_one,

input logic [3:0] E4_hundredthousand, //low	
input logic [3:0] E4_tenthousand,
input logic [3:0] E4_thousand,
input logic [3:0] E4_hundred,
input logic [3:0] E4_ten,
input logic [3:0] E4_one,

input logic [3:0] E32_hundredthousand, //high
input logic [3:0] E32_tenthousand,
input logic [3:0] E32_thousand,
input logic [3:0] E32_hundred,
input logic [3:0] E32_ten,
input logic [3:0] E32_one,

input logic [3:0] E128_hundredthousand,
input logic [3:0] E128_tenthousand,
input logic [3:0] E128_thousand,
input logic [3:0] E128_hundred,
input logic [3:0] E128_ten,
input logic [3:0] E128_one,
//-----------------------------------------


 //----------------------------------------
 input logic [31:0] display,
 output logic [7:0] VGA_R, VGA_G, VGA_B,
 output logic 	    VGA_CLK, VGA_HS, VGA_VS, VGA_BLANK_n, VGA_SYNC_n);

/*
 * 640 X 480 VGA timing for a 50 MHz clock: one pixel every other cycle
 * 
 * HCOUNT 1599 0             1279       1599 0
 *             _______________              ________
 * ___________|    Video      |____________|  Video
 * 
 * 
 * |SYNC| BP |<-- HACTIVE -->|FP|SYNC| BP |<-- HACTIVE
 *       _______________________      _____________
 * |____|       VGA_HS          |____|
 */
   // Parameters for hcount
   parameter HACTIVE      = 11'd 1280,
             HFRONT_PORCH = 11'd 32,
             HSYNC        = 11'd 192,
             HBACK_PORCH  = 11'd 96,   
             HTOTAL       = HACTIVE + HFRONT_PORCH + HSYNC + HBACK_PORCH; // 1600
   
   // Parameters for vcount
   parameter VACTIVE      = 10'd 480,
             VFRONT_PORCH = 10'd 10,
             VSYNC        = 10'd 2,
             VBACK_PORCH  = 10'd 33,
             VTOTAL       = VACTIVE + VFRONT_PORCH + VSYNC + VBACK_PORCH; // 525

   logic [10:0]			     hcount; // Horizontal counter
                                             // Hcount[10:1] indicates pixel column (0-639)
   logic 			     endOfLine;
   
   always_ff @(posedge clk50 or posedge reset)
     if (reset)          hcount <= 0;
     else if (endOfLine) hcount <= 0;
     else  	         hcount <= hcount + 11'd 1;

   assign endOfLine = hcount == HTOTAL - 1;

   // Vertical counter
   logic [9:0] 			     vcount;
   logic 			     endOfField;
   
   always_ff @(posedge clk50 or posedge reset)
     if (reset)          vcount <= 0;
     else if (endOfLine)
       if (endOfField)   vcount <= 0;
       else              vcount <= vcount + 10'd 1;

   assign endOfField = vcount == VTOTAL - 1;

   // Horizontal sync: from 0x520 to 0x5DF (0x57F)
   // 101 0010 0000 to 101 1101 1111
   assign VGA_HS = !( (hcount[10:8] == 3'b101) & !(hcount[7:5] == 3'b111));
   assign VGA_VS = !( vcount[9:1] == (VACTIVE + VFRONT_PORCH) / 2);

   assign VGA_SYNC_n = 1; // For adding sync to video signals; not used for VGA
   
   // Horizontal active: 0 to 1279     Vertical active: 0 to 479
   // 101 0000 0000  1280	       01 1110 0000  480
   // 110 0011 1111  1599	       10 0000 1100  524
   assign VGA_BLANK_n = !( hcount[10] & (hcount[9] | hcount[8]) ) &
			!( vcount[9] | (vcount[8:5] == 4'b1111) );   

   /* VGA_CLK is 25 MHz
    *             __    __    __
    * clk50    __|  |__|  |__|
    *        
    *             _____       __
    * hcount[0]__|     |_____|
    */
   assign VGA_CLK = hcount[0]; // 25 MHz clock: pixel latched on rising edge

	logic write_okay_r; //whether we should draw this pixel
	logic write_okay_b;
	logic write_okay_g;
	logic write_okay_o;
	logic [31:0] full_x;
	logic [31:0] full_y;
	logic [31:0] full_r;
	logic [31:0] multi_x;
	logic [31:0] multi_y;
	logic [31:0] multi_tot;
	logic [31:0] multi_rad;
	assign full_x[31:10] = 22'b0;
	assign full_y[31:10] = 22'b0;
	assign full_r[31:10] = 22'b0;
	
	always_comb 
	begin
		if (x_coord[9:0] < hcount[10:1])
			full_x[9:0] = hcount[10:1] - x_coord[9:0];
		else	
			full_x[9:0] = x_coord[9:0] - hcount[10:1];
	end
	
	always_comb
	begin
		if (y_coord[9:0] < vcount[9:0])
			full_y[9:0] = vcount[9:0] - y_coord[9:0];
		else
			full_y[9:0] = y_coord[9:0] - vcount[9:0];
	end
	
	assign full_r[9:0] = radius[9:0];
	assign multi_x = full_x*full_x;
	assign multi_y = full_y*full_y;
	assign multi_rad = full_r*full_r;

	//assign write_okay = (okay_base || okay31 || okay30 || okay29 || okay28 || okay27 || okay26 || okay25 || okay24 || okay23 || okay22 || okay21 || okay20 || okay19 || okay18 || okay17 || okay16 || okay15 || okay14 || okay13 || okay12 || okay11 || okay10 || okay9 || okay8 || okay7 || okay6 || okay5 || okay4 || okay3 || okay2 || okay1 || okay0);
	//assign write_okay = (vcount[9:0] > x_coord[9:0] - 2'b11 & vcount[9:0] < x_coord[9:0] + 2'b11) & (hcount[10:1] > y_coord[9:0] - 2'b11 & hcount[10:1] < y_coord[9:0] + 2'b11);
	//assign write_okay = ((multi_x + multi_y) < multi_rad);
	//parameter radius_SQ = radius[4:0];
	
	//assign write_okay = (( (x_coord[9:0] - hcount[10:1])*(x_coord[9:0] - hcount[10:1]) + (y_coord[9:0] - vcount[9:0])*(y_coord[9:0] - vcount[9:0]) ) < radius[9:0]);
   
	logic write_okay_sn1;
	logic [9:0] x_letter1;
	logic [9:0] y_letter1;
 	assign x_letter1 = 500;
	assign y_letter1 = 100;
	
	logic write_okay_sn2;
	logic [9:0] x_letter2;
	logic [9:0] y_letter2;
 	assign x_letter2 = 525;
	assign y_letter2 = 100;
	
	logic write_okay_sn3;
	logic [9:0] x_letter3;
	logic [9:0] y_letter3;
 	assign x_letter3 = 550;
	assign y_letter3 = 100;
	
	logic write_okay_sn4;
	logic [9:0] x_letter4;
	logic [9:0] y_letter4;
 	assign x_letter4 = 575;
	assign y_letter4 = 100;
	
	logic sn_pos1;
	logic sn_pos2;
	logic sn_pos3;
	logic sn_pos4;
	
	/////////////////////
	// Draw Stock Name //
	/////////////////////
	
	// Write in location only ones 
	assign sn_pos1 = ((hcount[10:1] < 525)&&(hcount[10:1] > 500)&&(vcount[9:0] > 100)&&(vcount[9:0] < 140));
	assign sn_pos2 = ((hcount[10:1] < 550)&&(hcount[10:1] > 525)&&(vcount[9:0] > 100)&&(vcount[9:0] < 140));
	assign sn_pos3 = ((hcount[10:1] < 575)&&(hcount[10:1] > 550)&&(vcount[9:0] > 100)&&(vcount[9:0] < 140));
	assign sn_pos4 = ((hcount[10:1] < 600)&&(hcount[10:1] > 575)&&(vcount[9:0] > 100)&&(vcount[9:0] < 140));
	
	always_comb begin
	
	
   	if (stock_name == 7'b01) begin // TEVA 1
			write_okay_sn1 = ((S_T[hcount[10:1] - x_letter1][vcount[9:0] - y_letter1]));
			write_okay_sn2 = ((S_E[hcount[10:1] - x_letter2][vcount[9:0] - y_letter2]));
			write_okay_sn3 = ((S_V[hcount[10:1] - x_letter3][vcount[9:0] - y_letter3]));
			write_okay_sn4 = ((S_A[hcount[10:1] - x_letter4][vcount[9:0] - y_letter4]));
		end
		
		else if (stock_name == 7'b00) begin // IBM 0
			write_okay_sn1 = ((S_I[hcount[10:1] - x_letter1][vcount[9:0] - y_letter1]));
			write_okay_sn2 = ((S_B[hcount[10:1] - x_letter2][vcount[9:0] - y_letter2]));
			write_okay_sn3 = ((S_M[hcount[10:1] - x_letter3][vcount[9:0] - y_letter3]));
			write_okay_sn4 = ((S_blank[hcount[10:1] - x_letter3][vcount[9:0] - y_letter3]));
			
		end
		
		else if (stock_name == 7'b10) begin // INTC 2
			write_okay_sn1 = ((S_I[hcount[10:1] - x_letter1][vcount[9:0] - y_letter1]));
			write_okay_sn2 = ((S_N[hcount[10:1] - x_letter2][vcount[9:0] - y_letter2]));
			write_okay_sn3 = ((S_T[hcount[10:1] - x_letter3][vcount[9:0] - y_letter3]));
			write_okay_sn4 = ((S_C[hcount[10:1] - x_letter4][vcount[9:0] - y_letter4]));
		end
		
		else if (stock_name == 7'b11) begin // ED 3
			write_okay_sn1 = ((S_E[hcount[10:1] - x_letter1][vcount[9:0] - y_letter1]));
			write_okay_sn2 = ((S_D[hcount[10:1] - x_letter2][vcount[9:0] - y_letter2]));
			write_okay_sn3 = ((S_blank[hcount[10:1] - x_letter3][vcount[9:0] - y_letter3]));
			write_okay_sn4 = ((S_blank[hcount[10:1] - x_letter4][vcount[9:0] - y_letter4]));
		end
		
		else if (stock_name == 7'b100) begin // KO 4
			write_okay_sn1 = ((S_K[hcount[10:1] - x_letter1][vcount[9:0] - y_letter1]));
			write_okay_sn2 = ((S_O[hcount[10:1] - x_letter2][vcount[9:0] - y_letter2]));
			write_okay_sn3 = ((S_blank[hcount[10:1] - x_letter3][vcount[9:0] - y_letter3]));
			write_okay_sn4 = ((S_blank[hcount[10:1] - x_letter4][vcount[9:0] - y_letter4]));
		end
		
		else if (stock_name == 7'b101) begin // BDB 5
			write_okay_sn1 = ((S_D[hcount[10:1] - x_letter1][vcount[9:0] - y_letter1]));
			write_okay_sn2 = ((S_B[hcount[10:1] - x_letter2][vcount[9:0] - y_letter2]));
			write_okay_sn3 = ((S_D[hcount[10:1] - x_letter3][vcount[9:0] - y_letter3]));
			write_okay_sn4 = ((S_blank[hcount[10:1] - x_letter4][vcount[9:0] - y_letter4]));
		end
		
		else if (stock_name == 7'b110) begin // PG
			write_okay_sn1 = ((S_P[hcount[10:1] - x_letter1][vcount[9:0] - y_letter1]));
			write_okay_sn2 = ((S_G[hcount[10:1] - x_letter2][vcount[9:0] - y_letter2]));
			write_okay_sn3 = ((S_blank[hcount[10:1] - x_letter3][vcount[9:0] - y_letter3]));
			write_okay_sn4 = ((S_blank[hcount[10:1] - x_letter4][vcount[9:0] - y_letter4]));
		end
		
		else begin
			write_okay_sn1 = ((S_blank[hcount[10:1] - x_letter1][vcount[9:0] - y_letter1]));
			write_okay_sn2 = ((S_blank[hcount[10:1] - x_letter2][vcount[9:0] - y_letter2]));
			write_okay_sn3 = ((S_blank[hcount[10:1] - x_letter3][vcount[9:0] - y_letter3]));
			write_okay_sn4 = ((S_blank[hcount[10:1] - x_letter4][vcount[9:0] - y_letter4]));	
		end
		
		
	end
	//
	//
	//For prices on y axis
	logic write_okay_h1;
	logic write_okay_h2;
	logic write_okay_h3;
	logic write_okay_l1;
	logic write_okay_l2;
	logic write_okay_l3;
	
	logic left_h1;
	logic left_h2;
	logic left_h3;
	logic left_l1;
	logic left_l2;
	logic left_l3;
	
	assign left_h1 = ((hcount[10:1] <= 35)&&(hcount[10:1] >= 20)&&(vcount[9:0] >= 20)&&(vcount[9:0] < 40));
	assign left_h2 = ((hcount[10:1] <= 55)&&(hcount[10:1] >= 40)&&(vcount[9:0] >= 20)&&(vcount[9:0] < 40));
	assign left_h3 = ((hcount[10:1] <= 75)&&(hcount[10:1] >= 60)&&(vcount[9:0] >= 20)&&(vcount[9:0] < 40));
	assign left_l1 = ((hcount[10:1] <= 35)&&(hcount[10:1] >= 20)&&(vcount[9:0] >= 360)&&(vcount[9:0] < 380));
	assign left_l2 = ((hcount[10:1] <= 55)&&(hcount[10:1] >= 40)&&(vcount[9:0] >= 360)&&(vcount[9:0] < 380));
	assign left_l3 = ((hcount[10:1] <= 75)&&(hcount[10:1] >= 60)&&(vcount[9:0] >= 360)&&(vcount[9:0] < 380));
	
	logic [9:0] lx1;
	logic [9:0] ly1;
	logic [9:0] lx2;
	logic [9:0] ly2;
	logic [9:0] lx3;
	logic [9:0] ly3;
	
	logic [9:0] hx1;
	logic [9:0] hy1;
	logic [9:0] hx2;
	logic [9:0] hy2;
	logic [9:0] hx3;
	logic [9:0] hy3;
	
	assign lx1 = 20;
	assign lx2 = 40;
	assign lx3 = 60;
	assign hx1 = 20;
	assign hx2 = 40;
	assign hx3= 60;
	
	assign hy1 = 20;
	assign hy2 = 20;
	assign hy3 = 20;
	assign ly1 = 360;
	assign ly2 = 360;
	assign ly3 = 360;
	
	always_comb begin
		if (E32_tenthousand == 0) begin
			write_okay_h1 = ((s_O[hcount[10:1] - hx1][vcount[9:0] - hy1]));
		end
		else if (E32_tenthousand == 1) begin
			write_okay_h1 = ((s_1[hcount[10:1] - hx1][vcount[9:0] - hy1]));
		end
		else if (E32_tenthousand == 2) begin
			write_okay_h1 = ((s_2[hcount[10:1] - hx1][vcount[9:0] - hy1]));
		end
		else if (E32_tenthousand == 3) begin
			write_okay_h1 = ((s_3[hcount[10:1] - hx1][vcount[9:0] - hy1]));
		end
		else if (E32_tenthousand == 4) begin
			write_okay_h1 = ((s_4[hcount[10:1] - hx1][vcount[9:0] - hy1]));
		end
		else if (E32_tenthousand == 5) begin
			write_okay_h1 = ((s_5[hcount[10:1] - hx1][vcount[9:0] - hy1]));
		end
		else if (E32_tenthousand == 6) begin
			write_okay_h1 = ((s_6[hcount[10:1] - hx1][vcount[9:0] - hy1]));
		end
		else if (E32_tenthousand == 7) begin
			write_okay_h1 = ((s_7[hcount[10:1] - hx1][vcount[9:0] - hy1]));
		end
		else if (E32_tenthousand == 8) begin
			write_okay_h1 = ((s_8[hcount[10:1] - hx1][vcount[9:0] - hy1]));
		end		
		else if (E32_tenthousand == 9) begin
			write_okay_h1 = ((s_9[hcount[10:1] - hx1][vcount[9:0] - hy1]));
		end	
		else begin
			write_okay_h1 = ((s_blank[hcount[10:1] - hx1][vcount[9:0] - hy1]));
		end
	end
	
	always_comb begin
		if (E32_thousand == 0) begin
			write_okay_h2 = ((s_O[hcount[10:1] - hx2][vcount[9:0] - hy2]));
		end
		else if (E32_thousand == 1) begin
			write_okay_h2 = ((s_1[hcount[10:1] - hx2][vcount[9:0] - hy2]));
		end
		else if (E32_thousand == 2) begin
			write_okay_h2 = ((s_2[hcount[10:1] - hx2][vcount[9:0] - hy2]));
		end
		else if (E32_thousand == 3) begin
			write_okay_h2 = ((s_3[hcount[10:1] - hx2][vcount[9:0] - hy2]));
		end
		else if (E32_thousand == 4) begin
			write_okay_h2 = ((s_4[hcount[10:1] - hx2][vcount[9:0] - hy2]));
		end
		else if (E32_thousand == 5) begin
			write_okay_h2 = ((s_5[hcount[10:1] - hx2][vcount[9:0] - hy2]));
		end
		else if (E32_thousand == 6) begin
			write_okay_h2 = ((s_6[hcount[10:1] - hx2][vcount[9:0] - hy2]));
		end
		else if (E32_thousand == 7) begin
			write_okay_h2 = ((s_7[hcount[10:1] - hx2][vcount[9:0] - hy2]));
		end
		else if (E32_thousand == 8) begin
			write_okay_h2 = ((s_8[hcount[10:1] - hx2][vcount[9:0] - hy2]));
		end		
		else if (E32_thousand == 9) begin
			write_okay_h2 = ((s_9[hcount[10:1] - hx2][vcount[9:0] - hy2]));
		end	
		else begin
			write_okay_h2 = ((s_blank[hcount[10:1] - hx2][vcount[9:0] - hy2]));
		end
	end	

	always_comb begin
		if (E32_hundred == 0) begin
			write_okay_h3 = ((s_O[hcount[10:1] - hx3][vcount[9:0] - hy3]));
		end
		else if (E32_hundred == 1) begin
			write_okay_h3 = ((s_1[hcount[10:1] - hx3][vcount[9:0] - hy3]));
		end
		else if (E32_hundred == 2) begin
			write_okay_h3 = ((s_2[hcount[10:1] - hx3][vcount[9:0] - hy3]));
		end
		else if (E32_hundred == 3) begin
			write_okay_h3 = ((s_3[hcount[10:1] - hx3][vcount[9:0] - hy3]));
		end
		else if (E32_hundred == 4) begin
			write_okay_h3 = ((s_4[hcount[10:1] - hx3][vcount[9:0] - hy3]));
		end
		else if (E32_hundred == 5) begin
			write_okay_h3 = ((s_5[hcount[10:1] - hx3][vcount[9:0] - hy3]));
		end
		else if (E32_hundred == 6) begin
			write_okay_h3 = ((s_6[hcount[10:1] - hx3][vcount[9:0] - hy3]));
		end
		else if (E32_hundred == 7) begin
			write_okay_h3 = ((s_7[hcount[10:1] - hx3][vcount[9:0] - hy3]));
		end
		else if (E32_hundred == 8) begin
			write_okay_h3 = ((s_8[hcount[10:1] - hx3][vcount[9:0] - hy3]));
		end		
		else if (E32_hundred == 9) begin
			write_okay_h3 = ((s_9[hcount[10:1] - hx3][vcount[9:0] - hy3]));
		end	
		else begin
			write_okay_h3 = ((s_blank[hcount[10:1] - hx3][vcount[9:0] - hy3]));
		end
	end
	//
		always_comb begin
		if (E4_tenthousand == 0) begin
			write_okay_l1 = ((s_O[hcount[10:1] - lx1][vcount[9:0] - ly1]));
		end
		else if (E4_tenthousand == 1) begin
			write_okay_l1 = ((s_1[hcount[10:1] - lx1][vcount[9:0] - ly1]));
		end
		else if (E4_tenthousand == 2) begin
			write_okay_l1 = ((s_2[hcount[10:1] - lx1][vcount[9:0] - ly1]));
		end
		else if (E4_tenthousand == 3) begin
			write_okay_l1 = ((s_3[hcount[10:1] - lx1][vcount[9:0] - ly1]));
		end
		else if (E4_tenthousand == 4) begin
			write_okay_l1 = ((s_4[hcount[10:1] - lx1][vcount[9:0] - ly1]));
		end
		else if (E4_tenthousand == 5) begin
			write_okay_l1 = ((s_5[hcount[10:1] - lx1][vcount[9:0] - ly1]));
		end
		else if (E4_tenthousand == 6) begin
			write_okay_l1 = ((s_6[hcount[10:1] - lx1][vcount[9:0] - ly1]));
		end
		else if (E4_tenthousand == 7) begin
			write_okay_l1 = ((s_7[hcount[10:1] - lx1][vcount[9:0] - ly1]));
		end
		else if (E4_tenthousand == 8) begin
			write_okay_l1 = ((s_8[hcount[10:1] - lx1][vcount[9:0] - ly1]));
		end		
		else if (E4_tenthousand == 9) begin
			write_okay_l1 = ((s_9[hcount[10:1] - lx1][vcount[9:0] - ly1]));
		end	
		else begin
			write_okay_l1 = ((s_blank[hcount[10:1] - lx1][vcount[9:0] - ly1]));
		end
	end
	
	always_comb begin
		if (E4_thousand == 0) begin
			write_okay_l2 = ((s_O[hcount[10:1] - lx2][vcount[9:0] - ly2]));
		end
		else if (E4_thousand == 1) begin
			write_okay_l2 = ((s_1[hcount[10:1] - lx2][vcount[9:0] - ly2]));
		end
		else if (E4_thousand == 2) begin
			write_okay_l2 = ((s_2[hcount[10:1] - lx2][vcount[9:0] - ly2]));
		end
		else if (E4_thousand == 3) begin
			write_okay_l2 = ((s_3[hcount[10:1] - lx2][vcount[9:0] - ly2]));
		end
		else if (E4_thousand == 4) begin
			write_okay_l2 = ((s_4[hcount[10:1] - lx2][vcount[9:0] - ly2]));
		end
		else if (E4_thousand == 5) begin
			write_okay_l2 = ((s_5[hcount[10:1] - lx2][vcount[9:0] - ly2]));
		end
		else if (E4_thousand == 6) begin
			write_okay_l2 = ((s_6[hcount[10:1] - lx2][vcount[9:0] - ly2]));
		end
		else if (E4_thousand == 7) begin
			write_okay_l2 = ((s_7[hcount[10:1] - lx2][vcount[9:0] - ly2]));
		end
		else if (E4_thousand == 8) begin
			write_okay_l2 = ((s_8[hcount[10:1] - lx2][vcount[9:0] - ly2]));
		end		
		else if (E4_thousand == 9) begin
			write_okay_l2 = ((s_9[hcount[10:1] - lx2][vcount[9:0] - ly2]));
		end	
		else begin
			write_okay_l2 = ((s_blank[hcount[10:1] - lx2][vcount[9:0] - ly2]));
		end
	end	

	always_comb begin
		if (E4_hundred == 0) begin
			write_okay_l3 = ((s_O[hcount[10:1] - lx3][vcount[9:0] - ly3]));
		end
		else if (E4_hundred == 1) begin
			write_okay_l3 = ((s_1[hcount[10:1] - lx3][vcount[9:0] - ly3]));
		end
		else if (E4_hundred == 2) begin
			write_okay_l3 = ((s_2[hcount[10:1] - lx3][vcount[9:0] - ly3]));
		end
		else if (E4_hundred == 3) begin
			write_okay_l3 = ((s_3[hcount[10:1] - lx3][vcount[9:0] - ly3]));
		end
		else if (E4_hundred == 4) begin
			write_okay_l3 = ((s_4[hcount[10:1] - lx3][vcount[9:0] - ly3]));
		end
		else if (E4_hundred == 5) begin
			write_okay_l3 = ((s_5[hcount[10:1] - lx3][vcount[9:0] - ly3]));
		end
		else if (E4_hundred == 6) begin
			write_okay_l3 = ((s_6[hcount[10:1] - lx3][vcount[9:0] - ly3]));
		end
		else if (E4_hundred == 7) begin
			write_okay_l3 = ((s_7[hcount[10:1] - lx3][vcount[9:0] - ly3]));
		end
		else if (E4_hundred == 8) begin
			write_okay_l3 = ((s_8[hcount[10:1] - lx3][vcount[9:0] - ly3]));
		end		
		else if (E4_hundred == 9) begin
			write_okay_l3 = ((s_9[hcount[10:1] - lx3][vcount[9:0] - ly3]));
		end	
		else begin
			write_okay_l3 = ((s_blank[hcount[10:1] - lx3][vcount[9:0] - ly3]));
		end
	end
	
	
	
	//
	//
	//
	//For prices under graph
	logic write_okay_sp1;
	logic write_okay_sp2;
	logic write_okay_sp3;
	logic write_okay_sp4;
	logic write_okay_sp5;
	logic write_okay_sp6;
	logic write_okay_sp7;
	logic write_okay_sp8;
	logic write_okay_sp9;
	logic write_okay_sp10;
	logic write_okay_sp11;
	logic write_okay_sp12;
	logic write_okay_sp13;
	
	logic st_pos1;
	logic st_pos2;
	logic st_pos3;
	logic st_pos4;
	logic st_pos5;
	logic st_pos6;
	logic st_pos7;
	logic st_pos8;
	logic st_pos9;
	logic st_pos10;
	logic st_pos11;
	logic st_pos12;
	logic st_pos13;
	//
	logic [9:0] sx1;
	logic [9:0] sy1;
	logic [9:0] sx2;
	logic [9:0] sy2;
	logic [9:0] sx3;
	logic [9:0] sy3;
	logic [9:0] sx4;
	logic [9:0] sy4;
	logic [9:0] sx5;
	logic [9:0] sy5;
	logic [9:0] sx6;
	logic [9:0] sy6;
	logic [9:0] sx7;
	logic [9:0] sy7;
	logic [9:0] sx8;
	logic [9:0] sy8;
	logic [9:0] sx9;
	logic [9:0] sy9;
	logic [9:0] sx10;
	logic [9:0] sy10;
	logic [9:0] sx11;
	logic [9:0] sy11;
	logic [9:0] sx12;
	logic [9:0] sy12;
	logic [9:0] sx13;
	logic [9:0] sy13;	
	
	assign sx1 = 60;
	assign sy1 = 400;
	assign sx2 = 80;
	assign sy2 = 400;
	assign sx3 = 100;
	assign sy3 = 400;	
	assign sx4 = 120;
	assign sy4 = 400;	
	assign sx5 = 140;
	assign sy5 = 400;	
	assign sx6 = 160;
	assign sy6 = 400;
	assign sx7 = 180;
	assign sy7 = 400;
	assign sx8 = 200;
	assign sy8 = 400;
	assign sx9 = 220;
	assign sy9 = 400;
	assign sx10 = 240;
	assign sy10 = 400;
	assign sx11 = 260;
	assign sy11 = 400;
	assign sx12 = 280;
	assign sy12 = 400;
	assign sx13= 300;
	assign sy13 = 400;
	
	
	//
	assign st_pos1 = ((hcount[10:1] <= 75)&&(hcount[10:1] >= 60)&&(vcount[9:0] >= 400)&&(vcount[9:0] < 420));
	assign st_pos2 = ((hcount[10:1] <= 95)&&(hcount[10:1] >= 80)&&(vcount[9:0] >=400)&&(vcount[9:0] < 420));
	assign st_pos3 = ((hcount[10:1] <= 115)&&(hcount[10:1] >= 100)&&(vcount[9:0] >= 400)&&(vcount[9:0] < 420));
	assign st_pos4 = ((hcount[10:1] <= 135)&&(hcount[10:1] >= 120)&&(vcount[9:0] >= 400)&&(vcount[9:0] < 420));
	assign st_pos5 = ((hcount[10:1] <= 155)&&(hcount[10:1] >= 140)&&(vcount[9:0] >= 400)&&(vcount[9:0] < 420));
	assign st_pos6 = ((hcount[10:1] <= 175)&&(hcount[10:1] >= 160)&&(vcount[9:0] >= 400)&&(vcount[9:0] < 420));
	assign st_pos7 = ((hcount[10:1] <= 195)&&(hcount[10:1] >= 180)&&(vcount[9:0] >= 400)&&(vcount[9:0] < 420));
	assign st_pos8 = ((hcount[10:1] <= 215)&&(hcount[10:1] >= 200)&&(vcount[9:0] >= 400)&&(vcount[9:0] < 420));
	assign st_pos9 = ((hcount[10:1] <= 235)&&(hcount[10:1] >= 220)&&(vcount[9:0] >= 400)&&(vcount[9:0] < 420));
	assign st_pos10 = ((hcount[10:1] <= 255)&&(hcount[10:1] >= 240)&&(vcount[9:0] >= 400)&&(vcount[9:0] < 420));
	assign st_pos11 = ((hcount[10:1] <= 275)&&(hcount[10:1] >= 260)&&(vcount[9:0] >= 400)&&(vcount[9:0] < 420));
	assign st_pos12 = ((hcount[10:1] <= 295)&&(hcount[10:1] >= 280)&&(vcount[9:0] >= 400)&&(vcount[9:0] < 420));
	assign st_pos13 = ((hcount[10:1] <= 315)&&(hcount[10:1] >= 300)&&(vcount[9:0] >= 400)&&(vcount[9:0] < 420));
	
	
	//
	//
	assign write_okay_sp1 = ((s_S[hcount[10:1] - sx1][vcount[9:0] - sy1]));
	assign write_okay_sp2 = ((s_T[hcount[10:1] - sx2][vcount[9:0] - sy2]));
	assign write_okay_sp3 = ((s_O[hcount[10:1] - sx3][vcount[9:0] - sy3]));
	assign write_okay_sp4 = ((s_C[hcount[10:1] - sx4][vcount[9:0] - sy4]));
	assign write_okay_sp5 = ((s_K[hcount[10:1] - sx5][vcount[9:0] - sy5]));
	assign write_okay_sp6 = ((s_colon[hcount[10:1] - sx6][vcount[9:0] - sy6]));

	assign write_okay_sp11 = ((s_dot[hcount[10:1] - sx11][vcount[9:0] - sy11]));

	
	
	
	always_comb begin
		if (S_hundredthousand == 0) begin
			write_okay_sp7 = ((s_O[hcount[10:1] - sx7][vcount[9:0] - sy7]));
		end
		else if (S_hundredthousand == 1) begin
			write_okay_sp7 = ((s_1[hcount[10:1] - sx7][vcount[9:0] - sy7]));
		end
		else if (S_hundredthousand == 2) begin
			write_okay_sp7 = ((s_2[hcount[10:1] - sx7][vcount[9:0] - sy7]));
		end
		else if (S_hundredthousand == 3) begin
			write_okay_sp7 = ((s_3[hcount[10:1] - sx7][vcount[9:0] - sy7]));
		end
		else if (S_hundredthousand == 4) begin
			write_okay_sp7 = ((s_4[hcount[10:1] - sx7][vcount[9:0] - sy7]));
		end
		else if (S_hundredthousand == 5) begin
			write_okay_sp7 = ((s_5[hcount[10:1] - sx7][vcount[9:0] - sy7]));
		end
		else if (S_hundredthousand == 6) begin
			write_okay_sp7 = ((s_6[hcount[10:1] - sx7][vcount[9:0] - sy7]));
		end
		else if (S_hundredthousand == 7) begin
			write_okay_sp7 = ((s_7[hcount[10:1] - sx7][vcount[9:0] - sy7]));
		end
		else if (S_hundredthousand == 8) begin
			write_okay_sp7 = ((s_8[hcount[10:1] - sx7][vcount[9:0] - sy7]));
		end		
		else if (S_hundredthousand == 9) begin
			write_okay_sp7 = ((s_9[hcount[10:1] - sx7][vcount[9:0] - sy7]));
		end	
		else begin
			write_okay_sp7 = ((s_blank[hcount[10:1] - sx7][vcount[9:0] - sy7]));
		end
	end
	
	always_comb begin
		if (S_tenthousand == 0) begin
			write_okay_sp8 = ((s_O[hcount[10:1] - sx8][vcount[9:0] - sy8]));
		end
		else if (S_tenthousand == 1) begin
			write_okay_sp8 = ((s_1[hcount[10:1] - sx8][vcount[9:0] - sy8]));
		end
		else if (S_tenthousand == 2) begin
			write_okay_sp8 = ((s_2[hcount[10:1] - sx8][vcount[9:0] - sy8]));
		end
		else if (S_tenthousand == 3) begin
			write_okay_sp8 = ((s_3[hcount[10:1] - sx8][vcount[9:0] - sy8]));
		end
		else if (S_tenthousand == 4) begin
			write_okay_sp8 = ((s_4[hcount[10:1] - sx8][vcount[9:0] - sy8]));
		end
		else if (S_tenthousand == 5) begin
			write_okay_sp8 = ((s_5[hcount[10:1] - sx8][vcount[9:0] - sy8]));
		end
		else if (S_tenthousand == 6) begin
			write_okay_sp8 = ((s_6[hcount[10:1] - sx8][vcount[9:0] - sy8]));
		end
		else if (S_tenthousand == 7) begin
			write_okay_sp8 = ((s_7[hcount[10:1] - sx8][vcount[9:0] - sy8]));
		end
		else if (S_tenthousand == 8) begin
			write_okay_sp8 = ((s_8[hcount[10:1] - sx8][vcount[9:0] - sy8]));
		end		
		else if (S_tenthousand == 9) begin
			write_okay_sp8 = ((s_9[hcount[10:1] - sx8][vcount[9:0] - sy8]));
		end	
		else begin
			write_okay_sp8 = ((s_blank[hcount[10:1] - sx8][vcount[9:0] - sy8]));
		end
	end
	
	always_comb begin
		if (S_thousand == 0) begin
			write_okay_sp9 = ((s_O[hcount[10:1] - sx9][vcount[9:0] - sy9]));
		end
		else if (S_thousand == 1) begin
			write_okay_sp9 = ((s_1[hcount[10:1] - sx9][vcount[9:0] - sy9]));
		end
		else if (S_thousand == 2) begin
			write_okay_sp9 = ((s_2[hcount[10:1] - sx9][vcount[9:0] - sy9]));
		end
		else if (S_thousand == 3) begin
			write_okay_sp9 = ((s_3[hcount[10:1] - sx9][vcount[9:0] - sy9]));
		end
		else if (S_thousand == 4) begin
			write_okay_sp9 = ((s_4[hcount[10:1] - sx9][vcount[9:0] - sy9]));
		end
		else if (S_thousand == 5) begin
			write_okay_sp9 = ((s_5[hcount[10:1] - sx9][vcount[9:0] - sy9]));
		end
		else if (S_thousand == 6) begin
			write_okay_sp9 = ((s_6[hcount[10:1] - sx9][vcount[9:0] - sy9]));
		end
		else if (S_thousand == 7) begin
			write_okay_sp9 = ((s_7[hcount[10:1] - sx9][vcount[9:0] - sy9]));
		end
		else if (S_thousand == 8) begin
			write_okay_sp9 = ((s_8[hcount[10:1] - sx9][vcount[9:0] - sy9]));
		end		
		else if (S_thousand == 9) begin
			write_okay_sp9 = ((s_9[hcount[10:1] - sx9][vcount[9:0] - sy9]));
		end	
		else begin
			write_okay_sp9 = ((s_blank[hcount[10:1] - sx9][vcount[9:0] - sy9]));
		end
	end
	
	always_comb begin
		if (S_hundred == 0) begin
			write_okay_sp10 = ((s_O[hcount[10:1] - sx10][vcount[9:0] - sy10]));
		end
		else if (S_hundred == 1) begin
			write_okay_sp10 = ((s_1[hcount[10:1] - sx10][vcount[9:0] - sy10]));
		end
		else if (S_hundred == 2) begin
			write_okay_sp10 = ((s_2[hcount[10:1] - sx10][vcount[9:0] - sy10]));
		end
		else if (S_hundred == 3) begin
			write_okay_sp10 = ((s_3[hcount[10:1] - sx10][vcount[9:0] - sy10]));
		end
		else if (S_hundred == 4) begin
			write_okay_sp10 = ((s_4[hcount[10:1] - sx10][vcount[9:0] - sy10]));
		end
		else if (S_hundred == 5) begin
			write_okay_sp10 = ((s_5[hcount[10:1] - sx10][vcount[9:0] - sy10]));
		end
		else if (S_hundred == 6) begin
			write_okay_sp10 = ((s_6[hcount[10:1] - sx10][vcount[9:0] - sy10]));
		end
		else if (S_hundred == 7) begin
			write_okay_sp10 = ((s_7[hcount[10:1] - sx10][vcount[9:0] - sy10]));
		end
		else if (S_hundred == 8) begin
			write_okay_sp10 = ((s_8[hcount[10:1] - sx10][vcount[9:0] - sy10]));
		end		
		else if (S_hundred == 9) begin
			write_okay_sp10 = ((s_9[hcount[10:1] - sx10][vcount[9:0] - sy10]));
		end	
		else begin
			write_okay_sp10 = ((s_blank[hcount[10:1] - sx10][vcount[9:0] - sy10]));
		end
	end
	
always_comb begin
		if (S_ten == 0) begin
			write_okay_sp12 = ((s_O[hcount[10:1] - sx12][vcount[9:0] - sy12]));
		end
		else if (S_ten == 1) begin
			write_okay_sp12 = ((s_1[hcount[10:1] - sx12][vcount[9:0] - sy12]));
		end
		else if (S_ten == 2) begin
			write_okay_sp12 = ((s_2[hcount[10:1] - sx12][vcount[9:0] - sy12]));
		end
		else if (S_ten == 3) begin
			write_okay_sp12 = ((s_3[hcount[10:1] - sx12][vcount[9:0] - sy12]));
		end
		else if (S_ten == 4) begin
			write_okay_sp12 = ((s_4[hcount[10:1] - sx12][vcount[9:0] - sy12]));
		end
		else if (S_ten == 5) begin
			write_okay_sp12 = ((s_5[hcount[10:1] - sx12][vcount[9:0] - sy12]));
		end
		else if (S_ten == 6) begin
			write_okay_sp12 = ((s_6[hcount[10:1] - sx12][vcount[9:0] - sy12]));
		end
		else if (S_ten == 7) begin
			write_okay_sp12 = ((s_7[hcount[10:1] - sx12][vcount[9:0] - sy12]));
		end
		else if (S_ten == 8) begin
			write_okay_sp12 = ((s_8[hcount[10:1] - sx12][vcount[9:0] - sy12]));
		end		
		else if (S_ten == 9) begin
			write_okay_sp12 = ((s_9[hcount[10:1] - sx12][vcount[9:0] - sy12]));
		end	
		else begin
			write_okay_sp12 = ((s_blank[hcount[10:1] - sx12][vcount[9:0] - sy12]));
		end
	end

always_comb begin
		if (S_one == 0) begin
			write_okay_sp13 = ((s_O[hcount[10:1] - sx13][vcount[9:0] - sy13]));
		end
		else if (S_one == 1) begin
			write_okay_sp13 = ((s_1[hcount[10:1] - sx13][vcount[9:0] - sy13]));
		end
		else if (S_one == 2) begin
			write_okay_sp13 = ((s_2[hcount[10:1] - sx13][vcount[9:0] - sy13]));
		end
		else if (S_one == 3) begin
			write_okay_sp13 = ((s_3[hcount[10:1] - sx13][vcount[9:0] - sy13]));
		end
		else if (S_one == 4) begin
			write_okay_sp13 = ((s_4[hcount[10:1] - sx13][vcount[9:0] - sy13]));
		end
		else if (S_one == 5) begin
			write_okay_sp13 = ((s_5[hcount[10:1] - sx13][vcount[9:0] - sy13]));
		end
		else if (S_one == 6) begin
			write_okay_sp13 = ((s_6[hcount[10:1] - sx13][vcount[9:0] - sy13]));
		end
		else if (S_one == 7) begin
			write_okay_sp13 = ((s_7[hcount[10:1] - sx13][vcount[9:0] - sy13]));
		end
		else if (S_one == 8) begin
			write_okay_sp13 = ((s_8[hcount[10:1] - sx13][vcount[9:0] - sy13]));
		end		
		else if (S_one == 9) begin
			write_okay_sp13 = ((s_9[hcount[10:1] - sx13][vcount[9:0] - sy13]));
		end	
		else begin
			write_okay_sp13 = ((s_blank[hcount[10:1] - sx13][vcount[9:0] - sy13]));
		end
	end


	
	
	
	//
	//
	//

	//drawing plot box
	assign write_okay_xaxis1 = ((37 < vcount[9:0])&&(vcount[9:0] < 43)&&(37 < hcount[10:1])&&(hcount[10:1] < 449));
	assign write_okay_xaxis2 = ((337 < vcount[9:0])&&(vcount[9:0] < 343)&&(37 < hcount[10:1])&&(hcount[10:1] < 449));
	
	assign write_okay_yaxis1 = ((37 < vcount[9:0])&&(vcount[9:0] < 343)&&(37 < hcount[10:1])&&(hcount[10:1] < 43));
	assign write_okay_yaxis2 = ((37 < vcount[9:0])&&(vcount[9:0] < 343)&&(443 < hcount[10:1])&&(hcount[10:1] < 449));
	
	
	
	
	
	
	//write okay for drawing stock
	  //assign temp_address = hcount[10:1];
	  assign write_okay_r = (vcount[9:0] - 10 == disp_stock_x[hcount[10:1] - 42]); 
	  assign write_okay_g = (vcount[9:0] - 10 == disp_stock_a1[hcount[10:1] - 42]); 
	  assign write_okay_b = (vcount[9:0] - 10 == disp_stock_a2[hcount[10:1] - 42]); 
	  assign write_okay_o = (vcount[9:0] - 10 == disp_stock_a3[hcount[10:1] - 42]);
	  assign write_stock_inbounds = ((hcount[10:1] < 443)&&(hcount[10:1] > 43)&&(vcount[9:0] > 40)&&(vcount[9:0] < 340));
	
	  assign write_okay = ((multi_x + multi_y) < multi_rad);
	
	
	
	//Draw ball for light 
	/*
     logic Light_1; // condition whether (hc,vc) is inside ball
	  assign Light_1_Cx[9:0]= 10'b0100101100; //assing center x of the light1
     assign Light_1_Cy[9:0]= 10'b0100101100; //assing center y of the light1 
     parameter R2 = 1600; // radius of ball squared
     assign Light_1 = (( (Light_1_Cx - hcount[10:1])*(Light_1_Cx - hcount[10:1]) + (Light_1_Cy - vcount)*(Light_1_Cy - vcount) ) < R2);
	*/
	
	//Grey box
	assign write_trafficbox = ((hcount[10:1] <= 630)&&(hcount[10:1] >= 530)&&(vcount[9:0] >= 150)&&(vcount[9:0] <= 450));
	
	//Draw ball for light
    
	 parameter R = 1600; // radius of ball squared
    //parameter R_empty = 1600; // radius of empyt ball squared
	 logic [9:0] cx_1,cx_2,cx_3;
    logic [9:0] cy_1,cy_2,cy_3;
	 logic [9:0] R_dark_1,R_dark_2,R_dark_3;
	 logic [9:0] Dark_light1,Dark_light2,Dark_light3;
	 
	 //logic dark_light3;
	 
	 //logic [2:0] select_action;
	 //assign select_action = 2'b00; //
	 /*
	 always_comb begin
	 
	 if (bhs == 2'b11) begin // All Dark
	 R_dark_1 = R_empty;
	 R_dark_2 = R_empty;
	 R_dark_3 = R_empty;
//	      light_D_1 = (( (cx_g - hcount[10:1])*(cx_g - hcount[10:1]) + (cy_g - vcount)*(cy_g - vcount) ) < R_empty);
//		   light_D_2 = (( (cx_y - hcount[10:1])*(cx_y - hcount[10:1]) + (cy_y - vcount)*(cy_y - vcount) ) < R_empty);
//		   light_D_3 = (( (cx_r - hcount[10:1])*(cx_r - hcount[10:1]) + (cy_r - vcount)*(cy_r - vcount) ) < R_empty);
	 end
	 
	 else if (bhs == 2'b00) begin // Green On
	 R_dark_1 = 0;
	 R_dark_2 = R_empty;
	 R_dark_3 = R_empty;
//	      light_G_1 = (( (cx_g - hcount[10:1])*(cx_g - hcount[10:1]) + (cy_g - vcount)*(cy_g - vcount) ) < 0);
//		   light_G_2 = (( (cx_y - hcount[10:1])*(cx_y - hcount[10:1]) + (cy_y - vcount)*(cy_y - vcount) ) < R_empty);
//		   light_G_3 = (( (cx_r - hcount[10:1])*(cx_r - hcount[10:1]) + (cy_r - vcount)*(cy_r - vcount) ) < R_empty);
	 end

    else if (bhs == 2'b01) begin // Yellow On
	 R_dark_1 = R_empty;
	 R_dark_2 = 0;
	 R_dark_3 = R_empty;
//	      light_Y_1 = (( (cx_g - hcount[10:1])*(cx_g - hcount[10:1]) + (cy_g - vcount)*(cy_g - vcount) ) < R_empty);
//		   light_Y_2 = (( (cx_y - hcount[10:1])*(cx_y - hcount[10:1]) + (cy_y - vcount)*(cy_y - vcount) ) < 0);
//		   light_Y_3 = (( (cx_r - hcount[10:1])*(cx_r - hcount[10:1]) + (cy_r - vcount)*(cy_r - vcount) ) < R_empty);
	 end
	 
	 else begin // Red On
	 R_dark_1 = R_empty; 
	 R_dark_2 = R_empty;
	 R_dark_3 = 0;
//	      light_R_1 = (( (cx_g - hcount[10:1])*(cx_g - hcount[10:1]) + (cy_g - vcount)*(cy_g - vcount) ) < R_empty);
//		   light_R_2 = (( (cx_y - hcount[10:1])*(cx_y - hcount[10:1]) + (cy_y - vcount)*(cy_y - vcount) ) < R_empty);
//		   light_R_3 = (( (cx_r - hcount[10:1])*(cx_r - hcount[10:1]) + (cy_r - vcount)*(cy_r - vcount) ) < 0);
	 end

	 end
	 */
	 //assign Dark_light1 = (( (cx_g - hcount[10:1])*(cx_g - hcount[10:1]) + (cy_g - vcount)*(cy_g - vcount) ) < R_dark_1);
	 //assign Dark_light2 = (( (cx_y - hcount[10:1])*(cx_y - hcount[10:1]) + (cy_y - vcount)*(cy_y - vcount) ) < R_dark_2);
	 //assign Dark_light3 = (( (cx_r - hcount[10:1])*(cx_r - hcount[10:1]) + (cy_r - vcount)*(cy_r - vcount) ) < R_dark_3);
	 
    logic G_light; // condition whether (hc,vc) is inside ball
 	 parameter cx_g = 580;
	 parameter cy_g = 200;
    assign G_light = (( (cx_g - hcount[10:1])*(cx_g - hcount[10:1]) + (cy_g - vcount)*(cy_g - vcount) ) < R);
	
	 logic Y_light; // condition whether (hc,vc) is inside ball

	 parameter cx_y = 580;
	 parameter cy_y = 300;
    assign Y_light = (( (cx_y - hcount[10:1])*(cx_y - hcount[10:1]) + (cy_y - vcount)*(cy_y - vcount) ) < R);
	 
	 logic R_light; // condition whether (hc,vc) is inside ball

	 parameter cx_r = 580;
	 parameter cy_r = 400;
    assign R_light = (( (cx_r - hcount[10:1])*(cx_r - hcount[10:1]) + (cy_r - vcount)*(cy_r - vcount) ) < R);
	 
	 

	////////////////////////
	// show Sell Hold Buy //
	////////////////////////
	
	
	logic write_okay_os;
	logic [9:0] x_os_letter1;
	logic [9:0] y_os_letter1;
 	assign x_os_letter1 = 500;
	assign y_os_letter1 = 200;
	
	logic [9:0] x_os_letter2;
	logic [9:0] y_os_letter2;
	assign x_os_letter2 = 500;
	assign y_os_letter2 = 300;
	
	logic [9:0] x_os_letter3;
	logic [9:0] y_os_letter3;
	assign x_os_letter3 = 500;
	assign y_os_letter3 = 400;

	logic os_pos1;
	logic os_pos2;
	logic os_pos3;
	logic select_pos;
	//logic sn_pos4;
	
	// 500 100
	// Write in location only ones 
	assign os_pos1 = ((hcount[10:1] < 525)&&(hcount[10:1] > 500)&&(vcount[9:0] > 200)&&(vcount[9:0] < 240));
	assign os_pos2 = ((hcount[10:1] < 525)&&(hcount[10:1] > 500)&&(vcount[9:0] > 300)&&(vcount[9:0] < 340));
	assign os_pos3 = ((hcount[10:1] < 525)&&(hcount[10:1] > 500)&&(vcount[9:0] > 400)&&(vcount[9:0] < 440));
	//assign sn_pos4 = ((hcount[10:1] < 600)&&(hcount[10:1] > 575)&&(vcount[9:0] > 100)&&(vcount[9:0] < 140));
	
	
	//
	//
	//

	//logic [1:0] operation_select;	assign operation_select = 2'b00; //
	always_comb begin
	
		if (bhs == 2'b00) begin // Buy
			write_okay_os = ((S_B[hcount[10:1] - x_os_letter1][vcount[9:0] - y_os_letter1]));
			select_pos = os_pos1;

		end
		else if (bhs == 2'b01) begin // Hold
			write_okay_os = ((S_H[hcount[10:1] - x_os_letter2][vcount[9:0] - y_os_letter2]));
         select_pos = os_pos2;
			
		end
		else begin // Sell
			write_okay_os = ((S_S[hcount[10:1] - x_os_letter3][vcount[9:0] - y_os_letter3]));
         select_pos = os_pos3;
		end

	end

 
	always_comb begin
	{VGA_R, VGA_G, VGA_B} = {8'h0, 8'h0, 8'h0}; // Black
	if (write_okay_r && write_stock_inbounds) begin
		{VGA_R, VGA_G, VGA_B} = {8'hff, 8'h00, 8'h00}; // Red
	end
	if (write_okay_b && write_stock_inbounds) begin
		{VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'hff}; // Red
	end
	if (write_okay_g && write_stock_inbounds) begin
		{VGA_R, VGA_G, VGA_B} = {8'h00, 8'hff, 8'h00}; // Red
	end
	if (write_okay_o && write_stock_inbounds) begin
		{VGA_R, VGA_G, VGA_B} = {8'haa, 8'h00, 8'haa}; // Red
	end	
	//
	if ((write_okay_sp1 && st_pos1) || (write_okay_sp2 && st_pos2) || (write_okay_sp3 && st_pos3)) begin
		{VGA_R, VGA_G, VGA_B} = {8'hff, 8'hff, 8'hff}; // white
	end
	if ((write_okay_sp2 && st_pos2) || (write_okay_sp3 && st_pos3) || (write_okay_sp4 && st_pos4)) begin
		{VGA_R, VGA_G, VGA_B} = {8'hff, 8'hff, 8'hff}; // white
	end
	if ((write_okay_sp5 && st_pos5) || (write_okay_sp6 && st_pos6) || (write_okay_sp7 && st_pos7)) begin
		{VGA_R, VGA_G, VGA_B} = {8'hff, 8'hff, 8'hff}; // white
	end
	if ((write_okay_sp8 && st_pos8) || (write_okay_sp9 && st_pos9) || (write_okay_sp10 && st_pos10)) begin
		{VGA_R, VGA_G, VGA_B} = {8'hff, 8'hff, 8'hff}; // white
	end
	if ((write_okay_sp11 && st_pos11) || (write_okay_sp12 && st_pos12) || (write_okay_sp13 && st_pos13)) begin
		{VGA_R, VGA_G, VGA_B} = {8'hff, 8'hff, 8'hff}; // white
	end
	//
	
	if ((write_okay_h1 && left_h1)||(write_okay_h2 && left_h2)||(write_okay_h3 && left_h3)) begin
		{VGA_R, VGA_G, VGA_B} = {8'hff, 8'hff, 8'hff}; // white
	end
	
	if ((write_okay_l1 && left_l1)||(write_okay_l2 && left_l2)||(write_okay_l3 && left_l3)) begin
		{VGA_R, VGA_G, VGA_B} = {8'hff, 8'hff, 8'hff}; // white
	end
	//
	if ((write_okay_sn1 && sn_pos1)|| (write_okay_sn2 && sn_pos2) || (write_okay_sn3 && sn_pos3) || (write_okay_sn4 && sn_pos4) ) begin
		{VGA_R, VGA_G, VGA_B} = {8'hff, 8'hff, 8'hff}; // white
	end
	if (write_okay_xaxis1 || write_okay_xaxis2 || write_okay_yaxis1 || write_okay_yaxis2) begin
		{VGA_R, VGA_G, VGA_B} = {8'hbb, 8'hbb, 8'hbb}; // grey
	end
	if (write_trafficbox) begin
		{VGA_R, VGA_G, VGA_B} = {8'haa, 8'haa, 8'haa}; // grey
	end
	if ((G_light) && (bhs == 2'b00)) begin
		{VGA_R, VGA_G, VGA_B} = {8'h00, 8'hff, 8'h00}; // Green
	end
	if ((G_light) && (bhs != 2'b00)) begin
		{VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'h00}; // Black
	end	
	if ((Y_light) && (bhs == 2'b01)) begin
		{VGA_R, VGA_G, VGA_B} = {8'hff, 8'hff, 8'h00}; // Yellow
	end
	if ((Y_light) && (bhs != 2'b01)) begin
		{VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'h00}; // Black
	end
	if ((R_light) && (bhs == 2'b10)) begin
		{VGA_R, VGA_G, VGA_B} = {8'hff, 8'h00, 8'h00}; // Red
	end
	if ((R_light) && (bhs != 2'b10)) begin
		{VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'h00}; // Black
	end
	/*
		if (Dark_light1 || Dark_light2 || Dark_light3 ) begin
		{VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'h00}; // Black circles for the light
	end
	*/
		if ((write_okay_os && select_pos) ) begin
		{VGA_R, VGA_G, VGA_B} = {8'hff, 8'hff, 8'hff}; // Write decision letters
	end
	
	end
	
	
	logic[39:0] S_A[24:0];
	logic[39:0] S_B[24:0];
	logic[39:0] S_C[24:0];
	logic[39:0] S_D[24:0];
	logic[39:0] S_E[24:0];
	logic[39:0] S_F[24:0];
	logic[39:0] S_G[24:0];
	logic[39:0] S_H[24:0];
	logic[39:0] S_I[24:0];
	logic[39:0] S_J[24:0];
	logic[39:0] S_K[24:0];
	logic[39:0] S_L[24:0];
	logic[39:0] S_M[24:0];
	logic[39:0] S_N[24:0];
	logic[39:0] S_O[24:0];
	logic[39:0] S_P[24:0];
	logic[39:0] S_Q[24:0];
	logic[39:0] S_R[24:0];
	logic[39:0] S_S[24:0];
	logic[39:0] S_T[24:0];
	logic[39:0] S_U[24:0];
	logic[39:0] S_V[24:0];
	logic[39:0] S_W[24:0];
	logic[39:0] S_X[24:0];
	logic[39:0] S_Y[24:0];
	logic[39:0] S_Z[24:0];
	logic[39:0] S_colon[24:0];
	logic[39:0] S_1[24:0];
	logic[39:0] S_2[24:0];
	logic[39:0] S_3[24:0];
	logic[39:0] S_4[24:0];
	logic[39:0] S_5[24:0];
	logic[39:0] S_6[24:0];
	logic[39:0] S_7[24:0];
	logic[39:0] S_8[24:0];
	logic[39:0] S_9[24:0];
	logic[39:0] S_blank[24:0];
	
	
	assign S_blank [0] = 40'b0000000000000000000000000000000000000000;
	assign S_blank [1] = 40'b0000000000000000000000000000000000000000;
	assign S_blank [2] = 40'b0000000000000000000000000000000000000000;
	assign S_blank [3] = 40'b0000000000000000000000000000000000000000;
	assign S_blank [4] = 40'b0000000000000000000000000000000000000000;
	assign S_blank [5] = 40'b0000000000000000000000000000000000000000;
	assign S_blank [6] = 40'b0000000000000000000000000000000000000000;
	assign S_blank [7] = 40'b0000000000000000000000000000000000000000;
	assign S_blank [8] = 40'b0000000000000000000000000000000000000000;
	assign S_blank [9] = 40'b0000000000000000000000000000000000000000;
	assign S_blank [10] = 40'b0000000000000000000000000000000000000000;
	assign S_blank [11] = 40'b0000000000000000000000000000000000000000;
	assign S_blank [12] = 40'b0000000000000000000000000000000000000000;
	assign S_blank [13] = 40'b0000000000000000000000000000000000000000;
	assign S_blank [14] = 40'b0000000000000000000000000000000000000000;
	assign S_blank [15] = 40'b0000000000000000000000000000000000000000;
	assign S_blank [16] = 40'b0000000000000000000000000000000000000000;
	assign S_blank [17] = 40'b0000000000000000000000000000000000000000;
	assign S_blank [18] = 40'b0000000000000000000000000000000000000000;
	assign S_blank [19] = 40'b0000000000000000000000000000000000000000;
	assign S_blank [20] = 40'b0000000000000000000000000000000000000000;
	assign S_blank [21] = 40'b0000000000000000000000000000000000000000;
	assign S_blank [22] = 40'b0000000000000000000000000000000000000000;
	assign S_blank [23] = 40'b0000000000000000000000000000000000000000;
	assign S_blank [24] = 40'b0000000000000000000000000000000000000000;
	assign S_A[0] = 40'b0000000001100000000000000000000000000000;
	assign S_A[1] = 40'b0000000011111000000000000000000000000000;
	assign S_A[2] = 40'b0000000001111111000000000000000000000000;
	assign S_A[3] = 40'b0000000000011111111000000000000000000000;
	assign S_A[4] = 40'b0000000000000011111110000000000000000000;
	assign S_A[5] = 40'b0000000000000001111111110000000000000000;
	assign S_A[6] = 40'b0000000000000001110111111100000000000000;
	assign S_A[7] = 40'b0000000000000001110001111111100000000000;
	assign S_A[8] = 40'b0000000000000001110000001111111000000000;
	assign S_A[9] = 40'b0000000000000001110000000001111100000000;
	assign S_A[10] = 40'b0000000000000001110000001111111000000000;
	assign S_A[11] = 40'b0000000000000001110001111111100000000000;
	assign S_A[12] = 40'b0000000000000001110111111100000000000000;
	assign S_A[13] = 40'b0000000000000001111111110000000000000000;
	assign S_A[14] = 40'b0000000000000011111110000000000000000000;
	assign S_A[15] = 40'b0000000000011111111000000000000000000000;
	assign S_A[16] = 40'b0000000001111111000000000000000000000000;
	assign S_A[17] = 40'b0000000011111000000000000000000000000000;
	assign S_A[18] = 40'b0000000001100000000000000000000000000000;
	assign S_A[19] = 40'b0000000000000000000000000000000000000000;
	assign S_A[20] = 40'b0000000000000000000000000000000000000000;
	assign S_A[21] = 40'b0000000000000000000000000000000000000000;
	assign S_A[22] = 40'b0000000000000000000000000000000000000000;
	assign S_A[23] = 40'b0000000000000000000000000000000000000000;
	assign S_A[24] = 40'b0000000000000000000000000000000000000000;
	assign S_B[0] = 40'b0000000000000000000000000000000000000000;
	assign S_B[1] = 40'b0000000000000000000000000000000000000000;
	assign S_B[2] = 40'b0000000000000000000000000000000000000000;
	assign S_B[3] = 40'b0000000001111111111111111111111000000000;
	assign S_B[4] = 40'b0000000011111111111111111111111100000000;
	assign S_B[5] = 40'b0000000011111111111111111111111100000000;
	assign S_B[6] = 40'b0000000011100000000111000000011100000000;
	assign S_B[7] = 40'b0000000011100000000111000000011100000000;
	assign S_B[8] = 40'b0000000011100000000111000000011100000000;
	assign S_B[9] = 40'b0000000011100000000111000000011100000000;
	assign S_B[10] = 40'b0000000011100000000111000000011100000000;
	assign S_B[11] = 40'b0000000011100000000111000000011100000000;
	assign S_B[12] = 40'b0000000011100000000111000000011100000000;
	assign S_B[13] = 40'b0000000011100000000111000000011100000000;
	assign S_B[14] = 40'b0000000011100000000111000000011100000000;
	assign S_B[15] = 40'b0000000001110000001111100000111000000000;
	assign S_B[16] = 40'b0000000001111000011111110001111000000000;
	assign S_B[17] = 40'b0000000001111111111111111111111000000000;
	assign S_B[18] = 40'b0000000000111111111101111111110000000000;
	assign S_B[19] = 40'b0000000000000111100000001110000000000000;
	assign S_B[20] = 40'b0000000000000000000000000000000000000000;
	assign S_B[21] = 40'b0000000000000000000000000000000000000000;
	assign S_B[22] = 40'b0000000000000000000000000000000000000000;
	assign S_B[23] = 40'b0000000000000000000000000000000000000000;
	assign S_B[24] = 40'b0000000000000000000000000000000000000000;
	assign S_C[0] = 40'b0000000000000000000000000000000000000000;
	assign S_C[1] = 40'b0000000000000000000000000000000000000000;
	assign S_C[2] = 40'b0000000000000000111111110000000000000000;
	assign S_C[3] = 40'b0000000000000011111111111100000000000000;
	assign S_C[4] = 40'b0000000000011111111111111111100000000000;
	assign S_C[5] = 40'b0000000000111111000000001111110000000000;
	assign S_C[6] = 40'b0000000001111100000000000011111000000000;
	assign S_C[7] = 40'b0000000001111000000000000001111000000000;
	assign S_C[8] = 40'b0000000001110000000000000000111000000000;
	assign S_C[9] = 40'b0000000011100000000000000000011100000000;
	assign S_C[10] = 40'b0000000011100000000000000000011100000000;
	assign S_C[11] = 40'b0000000011100000000000000000011100000000;
	assign S_C[12] = 40'b0000000011100000000000000000011100000000;
	assign S_C[13] = 40'b0000000011100000000000000000011100000000;
	assign S_C[14] = 40'b0000000001110000000000000000111000000000;
	assign S_C[15] = 40'b0000000001111000000000000001111000000000;
	assign S_C[16] = 40'b0000000001111100000000000011111000000000;
	assign S_C[17] = 40'b0000000000111110000000000111110000000000;
	assign S_C[18] = 40'b0000000000011111000000001111100000000000;
	assign S_C[19] = 40'b0000000000000010000000000100000000000000;
	assign S_C[20] = 40'b0000000000000000000000000000000000000000;
	assign S_C[21] = 40'b0000000000000000000000000000000000000000;
	assign S_C[22] = 40'b0000000000000000000000000000000000000000;
	assign S_C[23] = 40'b0000000000000000000000000000000000000000;
	assign S_C[24] = 40'b0000000000000000000000000000000000000000;
	assign S_D[0] = 40'b0000000000000000000000000000000000000000;
	assign S_D[1] = 40'b0000000000000000000000000000000000000000;
	assign S_D[2] = 40'b0000000000000000000000000000000000000000;
	assign S_D[3] = 40'b0000000001111111111111111111111000000000;
	assign S_D[4] = 40'b0000000011111111111111111111111100000000;
	assign S_D[5] = 40'b0000000011111111111111111111111100000000;
	assign S_D[6] = 40'b0000000011100000000000000000011100000000;
	assign S_D[7] = 40'b0000000011100000000000000000011100000000;
	assign S_D[8] = 40'b0000000011100000000000000000011100000000;
	assign S_D[9] = 40'b0000000011100000000000000000011100000000;
	assign S_D[10] = 40'b0000000011100000000000000000011100000000;
	assign S_D[11] = 40'b0000000011100000000000000000011100000000;
	assign S_D[12] = 40'b0000000011100000000000000000011100000000;
	assign S_D[13] = 40'b0000000001110000000000000000111000000000;
	assign S_D[14] = 40'b0000000001111000000000000001111000000000;
	assign S_D[15] = 40'b0000000001111100000000000011111000000000;
	assign S_D[16] = 40'b0000000000111111000000001111110000000000;
	assign S_D[17] = 40'b0000000000011111111111111111100000000000;
	assign S_D[18] = 40'b0000000000000011111111111100000000000000;
	assign S_D[19] = 40'b0000000000000000111111110000000000000000;
	assign S_D[20] = 40'b0000000000000000000000000000000000000000;
	assign S_D[21] = 40'b0000000000000000000000000000000000000000;
	assign S_D[22] = 40'b0000000000000000000000000000000000000000;
	assign S_D[23] = 40'b0000000000000000000000000000000000000000;
	assign S_D[24] = 40'b0000000000000000000000000000000000000000;
	assign S_E[0] = 40'b0000000000000000000000000000000000000000;
	assign S_E[1] = 40'b0000000000000000000000000000000000000000;
	assign S_E[2] = 40'b0000000000000000000000000000000000000000;
	assign S_E[3] = 40'b0000000001111111111111111111111000000000;
	assign S_E[4] = 40'b0000000011111111111111111111111100000000;
	assign S_E[5] = 40'b0000000011111111111111111111111100000000;
	assign S_E[6] = 40'b0000000011100000000111000000011100000000;
	assign S_E[7] = 40'b0000000011100000000111000000011100000000;
	assign S_E[8] = 40'b0000000011100000000111000000011100000000;
	assign S_E[9] = 40'b0000000011100000000111000000011100000000;
	assign S_E[10] = 40'b0000000011100000000111000000011100000000;
	assign S_E[11] = 40'b0000000011100000000111000000011100000000;
	assign S_E[12] = 40'b0000000011100000000111000000011100000000;
	assign S_E[13] = 40'b0000000011100000000010000000011100000000;
	assign S_E[14] = 40'b0000000011100000000000000000011100000000;
	assign S_E[15] = 40'b0000000011100000000000000000011100000000;
	assign S_E[16] = 40'b0000000011100000000000000000011100000000;
	assign S_E[17] = 40'b0000000011100000000000000000011100000000;
	assign S_E[18] = 40'b0000000001000000000000000000001000000000;
	assign S_E[19] = 40'b0000000000000000000000000000000000000000;
	assign S_E[20] = 40'b0000000000000000000000000000000000000000;
	assign S_E[21] = 40'b0000000000000000000000000000000000000000;
	assign S_E[22] = 40'b0000000000000000000000000000000000000000;
	assign S_E[23] = 40'b0000000000000000000000000000000000000000;
	assign S_E[24] = 40'b0000000000000000000000000000000000000000;
	assign S_F[0] = 40'b0000000000000000000000000000000000000000;
	assign S_F[1] = 40'b0000000000000000000000000000000000000000;
	assign S_F[2] = 40'b0000000000000000000000000000000000000000;
	assign S_F[3] = 40'b0000000001111111111111111111111000000000;
	assign S_F[4] = 40'b0000000011111111111111111111111100000000;
	assign S_F[5] = 40'b0000000001111111111111111111111100000000;
	assign S_F[6] = 40'b0000000000000000000111000000011100000000;
	assign S_F[7] = 40'b0000000000000000000111000000011100000000;
	assign S_F[8] = 40'b0000000000000000000111000000011100000000;
	assign S_F[9] = 40'b0000000000000000000111000000011100000000;
	assign S_F[10] = 40'b0000000000000000000111000000011100000000;
	assign S_F[11] = 40'b0000000000000000000111000000011100000000;
	assign S_F[12] = 40'b0000000000000000000111000000011100000000;
	assign S_F[13] = 40'b0000000000000000000010000000011100000000;
	assign S_F[14] = 40'b0000000000000000000000000000011100000000;
	assign S_F[15] = 40'b0000000000000000000000000000011100000000;
	assign S_F[16] = 40'b0000000000000000000000000000011100000000;
	assign S_F[17] = 40'b0000000000000000000000000000011100000000;
	assign S_F[18] = 40'b0000000000000000000000000000001000000000;
	assign S_F[19] = 40'b0000000000000000000000000000000000000000;
	assign S_F[20] = 40'b0000000000000000000000000000000000000000;
	assign S_F[21] = 40'b0000000000000000000000000000000000000000;
	assign S_F[22] = 40'b0000000000000000000000000000000000000000;
	assign S_F[23] = 40'b0000000000000000000000000000000000000000;
	assign S_F[24] = 40'b0000000000000000000000000000000000000000;
	assign S_G[0] = 40'b0000000000000000000000000000000000000000;
	assign S_G[1] = 40'b0000000000000000000000000000000000000000;
	assign S_G[2] = 40'b0000000000000000111111110000000000000000;
	assign S_G[3] = 40'b0000000000000011111111111100000000000000;
	assign S_G[4] = 40'b0000000000011111111111111111100000000000;
	assign S_G[5] = 40'b0000000000111111000000001111110000000000;
	assign S_G[6] = 40'b0000000001111100000000000011111000000000;
	assign S_G[7] = 40'b0000000001111000000000000001111000000000;
	assign S_G[8] = 40'b0000000001110000000000000000111000000000;
	assign S_G[9] = 40'b0000000011100000000000000000011100000000;
	assign S_G[10] = 40'b0000000011100000000000000000011100000000;
	assign S_G[11] = 40'b0000000011100000000000000000011100000000;
	assign S_G[12] = 40'b0000000011100000010000000000011100000000;
	assign S_G[13] = 40'b0000000011100000111000000000011100000000;
	assign S_G[14] = 40'b0000000001110000111000000000111000000000;
	assign S_G[15] = 40'b0000000001111000111000000001111000000000;
	assign S_G[16] = 40'b0000000001111100111000000011111000000000;
	assign S_G[17] = 40'b0000000000111111111000000111110000000000;
	assign S_G[18] = 40'b0000000000011111111000001111100000000000;
	assign S_G[19] = 40'b0000000000000011110000000100000000000000;
	assign S_G[20] = 40'b0000000000000000000000000000000000000000;
	assign S_G[21] = 40'b0000000000000000000000000000000000000000;
	assign S_G[22] = 40'b0000000000000000000000000000000000000000;
	assign S_G[23] = 40'b0000000000000000000000000000000000000000;
	assign S_G[24] = 40'b0000000000000000000000000000000000000000;
	assign S_H[0] = 40'b0000000000000000000000000000000000000000;
	assign S_H[1] = 40'b0000000000000000000000000000000000000000;
	assign S_H[2] = 40'b0000000000000000000000000000000000000000;
	assign S_H[3] = 40'b0000000001111111111111111111111000000000;
	assign S_H[4] = 40'b0000000011111111111111111111111100000000;
	assign S_H[5] = 40'b0000000001111111111111111111111000000000;
	assign S_H[6] = 40'b0000000000000000000111000000000000000000;
	assign S_H[7] = 40'b0000000000000000000111000000000000000000;
	assign S_H[8] = 40'b0000000000000000000111000000000000000000;
	assign S_H[9] = 40'b0000000000000000000111000000000000000000;
	assign S_H[10] = 40'b0000000000000000000111000000000000000000;
	assign S_H[11] = 40'b0000000000000000000111000000000000000000;
	assign S_H[12] = 40'b0000000000000000000111000000000000000000;
	assign S_H[13] = 40'b0000000000000000000111000000000000000000;
	assign S_H[14] = 40'b0000000000000000000111000000000000000000;
	assign S_H[15] = 40'b0000000000000000000111000000000000000000;
	assign S_H[16] = 40'b0000000000000000000111000000000000000000;
	assign S_H[17] = 40'b0000000001111111111111111111111000000000;
	assign S_H[18] = 40'b0000000011111111111111111111111100000000;
	assign S_H[19] = 40'b0000000001111111111111111111111000000000;
	assign S_H[20] = 40'b0000000000000000000000000000000000000000;
	assign S_H[21] = 40'b0000000000000000000000000000000000000000;
	assign S_H[22] = 40'b0000000000000000000000000000000000000000;
	assign S_H[23] = 40'b0000000000000000000000000000000000000000;
	assign S_H[24] = 40'b0000000000000000000000000000000000000000;
	assign S_I[0] = 40'b0000000000000000000000000000000000000000;
	assign S_I[1] = 40'b0000000000000000000000000000000000000000;
	assign S_I[2] = 40'b0000000000000000000000000000000000000000;
	assign S_I[3] = 40'b0000000001111111111111111111111000000000;
	assign S_I[4] = 40'b0000000011111111111111111111111100000000;
	assign S_I[5] = 40'b0000000001111111111111111111111000000000;
	assign S_I[6] = 40'b0000000000000000000000000000000000000000;
	assign S_I[7] = 40'b0000000000000000000000000000000000000000;
	assign S_I[8] = 40'b0000000000000000000000000000000000000000;
	assign S_I[9] = 40'b0000000000000000000000000000000000000000;
	assign S_I[10] = 40'b0000000000000000000000000000000000000000;
	assign S_I[11] = 40'b0000000000000000000000000000000000000000;
	assign S_I[12] = 40'b0000000000000000000000000000000000000000;
	assign S_I[13] = 40'b0000000000000000000000000000000000000000;
	assign S_I[14] = 40'b0000000000000000000000000000000000000000;
	assign S_I[15] = 40'b0000000000000000000000000000000000000000;
	assign S_I[16] = 40'b0000000000000000000000000000000000000000;
	assign S_I[17] = 40'b0000000000000000000000000000000000000000;
	assign S_I[18] = 40'b0000000000000000000000000000000000000000;
	assign S_I[19] = 40'b0000000000000000000000000000000000000000;
	assign S_I[20] = 40'b0000000000000000000000000000000000000000;
	assign S_I[21] = 40'b0000000000000000000000000000000000000000;
	assign S_I[22] = 40'b0000000000000000000000000000000000000000;
	assign S_I[23] = 40'b0000000000000000000000000000000000000000;
	assign S_I[24] = 40'b0000000000000000000000000000000000000000;
	assign S_J[0] = 40'b0000000000000000000000000000000000000000;
	assign S_J[1] = 40'b0000000000000111100000000000000000000000;
	assign S_J[2] = 40'b0000000000111111110000000000000000000000;
	assign S_J[3] = 40'b0000000001111111100000000000000000000000;
	assign S_J[4] = 40'b0000000001111000000000000000000000000000;
	assign S_J[5] = 40'b0000000001110000000000000000000000000000;
	assign S_J[6] = 40'b0000000011100000000000000000000000000000;
	assign S_J[7] = 40'b0000000011100000000000000000000000000000;
	assign S_J[8] = 40'b0000000011100000000000000000000000000000;
	assign S_J[9] = 40'b0000000001110000000000000000000000000000;
	assign S_J[10] = 40'b0000000001111000000000000000000000000000;
	assign S_J[11] = 40'b0000000001111111111111111111111000000000;
	assign S_J[12] = 40'b0000000000111111111111111111111100000000;
	assign S_J[13] = 40'b0000000000000111111111111111111000000000;
	assign S_J[14] = 40'b0000000000000000000000000000000000000000;
	assign S_J[15] = 40'b0000000000000000000000000000000000000000;
	assign S_J[16] = 40'b0000000000000000000000000000000000000000;
	assign S_J[17] = 40'b0000000000000000000000000000000000000000;
	assign S_J[18] = 40'b0000000000000000000000000000000000000000;
	assign S_J[19] = 40'b0000000000000000000000000000000000000000;
	assign S_J[20] = 40'b0000000000000000000000000000000000000000;
	assign S_J[21] = 40'b0000000000000000000000000000000000000000;
	assign S_J[22] = 40'b0000000000000000000000000000000000000000;
	assign S_J[23] = 40'b0000000000000000000000000000000000000000;
	assign S_J[24] = 40'b0000000000000000000000000000000000000000;
	assign S_K[0] = 40'b0000000000000000000000000000000000000000;
	assign S_K[1] = 40'b0000000000000000000000000000000000000000;
	assign S_K[2] = 40'b0000000000000000000000000000000000000000;
	assign S_K[3] = 40'b0000000001111111111111111111111000000000;
	assign S_K[4] = 40'b0000000011111111111111111111111100000000;
	assign S_K[5] = 40'b0000000001111111111111111111111000000000;
	assign S_K[6] = 40'b0000000000000000111110000000000000000000;
	assign S_K[7] = 40'b0000000000000000011111000000000000000000;
	assign S_K[8] = 40'b0000000000000000001111100000000000000000;
	assign S_K[9] = 40'b0000000000000000000111110000000000000000;
	assign S_K[10] = 40'b0000000000000000011111111000000000000000;
	assign S_K[11] = 40'b0000000000000000111111111100000000000000;
	assign S_K[12] = 40'b0000000000000001111100111110000000000000;
	assign S_K[13] = 40'b0000000000000111111000011111000000000000;
	assign S_K[14] = 40'b0000000000001111110000001111100000000000;
	assign S_K[15] = 40'b0000000000011111000000000111110000000000;
	assign S_K[16] = 40'b0000000001111110000000000011111000000000;
	assign S_K[17] = 40'b0000000011111100000000000001111100000000;
	assign S_K[18] = 40'b0000000011110000000000000000111100000000;
	assign S_K[19] = 40'b0000000001100000000000000000011000000000;
	assign S_K[20] = 40'b0000000000000000000000000000000000000000;
	assign S_K[21] = 40'b0000000000000000000000000000000000000000;
	assign S_K[22] = 40'b0000000000000000000000000000000000000000;
	assign S_K[23] = 40'b0000000000000000000000000000000000000000;
	assign S_K[24] = 40'b0000000000000000000000000000000000000000;
	assign S_L[0] = 40'b0000000000000000000000000000000000000000;
	assign S_L[1] = 40'b0000000000000000000000000000000000000000;
	assign S_L[2] = 40'b0000000000000000000000000000000000000000;
	assign S_L[3] = 40'b0000000001111111111111111111111000000000;
	assign S_L[4] = 40'b0000000011111111111111111111111100000000;
	assign S_L[5] = 40'b0000000011111111111111111111111000000000;
	assign S_L[6] = 40'b0000000011100000000000000000000000000000;
	assign S_L[7] = 40'b0000000011100000000000000000000000000000;
	assign S_L[8] = 40'b0000000011100000000000000000000000000000;
	assign S_L[9] = 40'b0000000011100000000000000000000000000000;
	assign S_L[10] = 40'b0000000011100000000000000000000000000000;
	assign S_L[11] = 40'b0000000011100000000000000000000000000000;
	assign S_L[12] = 40'b0000000011100000000000000000000000000000;
	assign S_L[13] = 40'b0000000011100000000000000000000000000000;
	assign S_L[14] = 40'b0000000011100000000000000000000000000000;
	assign S_L[15] = 40'b0000000011100000000000000000000000000000;
	assign S_L[16] = 40'b0000000011100000000000000000000000000000;
	assign S_L[17] = 40'b0000000001000000000000000000000000000000;
	assign S_L[18] = 40'b0000000000000000000000000000000000000000;
	assign S_L[19] = 40'b0000000000000000000000000000000000000000;
	assign S_L[20] = 40'b0000000000000000000000000000000000000000;
	assign S_L[21] = 40'b0000000000000000000000000000000000000000;
	assign S_L[22] = 40'b0000000000000000000000000000000000000000;
	assign S_L[23] = 40'b0000000000000000000000000000000000000000;
	assign S_L[24] = 40'b0000000000000000000000000000000000000000;
	assign S_M[0] = 40'b0000000000000000000000000000000000000000;
	assign S_M[1] = 40'b0000000000000000000000000000000000000000;
	assign S_M[2] = 40'b0000000000000000000000000000000000000000;
	assign S_M[3] = 40'b0000000001111111111111111111111000000000;
	assign S_M[4] = 40'b0000000011111111111111111111111100000000;
	assign S_M[5] = 40'b0000000001111111111111111111111000000000;
	assign S_M[6] = 40'b0000000000000000000001111111100000000000;
	assign S_M[7] = 40'b0000000000000000000111111100000000000000;
	assign S_M[8] = 40'b0000000000000000111111110000000000000000;
	assign S_M[9] = 40'b0000000000000011111110000000000000000000;
	assign S_M[10] = 40'b0000000000011111111000000000000000000000;
	assign S_M[11] = 40'b0000000001111111000000000000000000000000;
	assign S_M[12] = 40'b0000000011111000000000000000000000000000;
	assign S_M[13] = 40'b0000000001111111000000000000000000000000;
	assign S_M[14] = 40'b0000000000011111111000000000000000000000;
	assign S_M[15] = 40'b0000000000000011111110000000000000000000;
	assign S_M[16] = 40'b0000000000000000111111110000000000000000;
	assign S_M[17] = 40'b0000000000000000000111111100000000000000;
	assign S_M[18] = 40'b0000000000000000000001111111100000000000;
	assign S_M[19] = 40'b0000000001111111111111111111111000000000;
	assign S_M[20] = 40'b0000000011111111111111111111111100000000;
	assign S_M[21] = 40'b0000000001111111111111111111111000000000;
	assign S_M[22] = 40'b0000000000000000000000000000000000000000;
	assign S_M[23] = 40'b0000000000000000000000000000000000000000;
	assign S_M[24] = 40'b0000000000000000000000000000000000000000;
	assign S_N[0] = 40'b0000000000000000000000000000000000000000;
	assign S_N[1] = 40'b0000000000000000000000000000000000000000;
	assign S_N[2] = 40'b0000000000000000000000000000000000000000;
	assign S_N[3] = 40'b0000000001111111111111111111111000000000;
	assign S_N[4] = 40'b0000000011111111111111111111111100000000;
	assign S_N[5] = 40'b0000000001111111111111111111111100000000;
	assign S_N[6] = 40'b0000000000000000000000000111110000000000;
	assign S_N[7] = 40'b0000000000000000000000011111100000000000;
	assign S_N[8] = 40'b0000000000000000000000111110000000000000;
	assign S_N[9] = 40'b0000000000000000000011111100000000000000;
	assign S_N[10] = 40'b0000000000000000000111110000000000000000;
	assign S_N[11] = 40'b0000000000000000011111100000000000000000;
	assign S_N[12] = 40'b0000000000000000111110000000000000000000;
	assign S_N[13] = 40'b0000000000000011111100000000000000000000;
	assign S_N[14] = 40'b0000000000000111110000000000000000000000;
	assign S_N[15] = 40'b0000000000011111100000000000000000000000;
	assign S_N[16] = 40'b0000000000111110000000000000000000000000;
	assign S_N[17] = 40'b0000000011111111111111111111111000000000;
	assign S_N[18] = 40'b0000000011111111111111111111111100000000;
	assign S_N[19] = 40'b0000000001111111111111111111111000000000;
	assign S_N[20] = 40'b0000000000000000000000000000000000000000;
	assign S_N[21] = 40'b0000000000000000000000000000000000000000;
	assign S_N[22] = 40'b0000000000000000000000000000000000000000;
	assign S_N[23] = 40'b0000000000000000000000000000000000000000;
	assign S_N[24] = 40'b0000000000000000000000000000000000000000;
	assign S_O[0] = 40'b0000000000000000000000000000000000000000;
	assign S_O[1] = 40'b0000000000000000000000000000000000000000;
	assign S_O[2] = 40'b0000000000000000111111110000000000000000;
	assign S_O[3] = 40'b0000000000000011111111111100000000000000;
	assign S_O[4] = 40'b0000000000011111111111111111100000000000;
	assign S_O[5] = 40'b0000000000111111000000001111110000000000;
	assign S_O[6] = 40'b0000000001111100000000000011111000000000;
	assign S_O[7] = 40'b0000000001111000000000000001111000000000;
	assign S_O[8] = 40'b0000000001110000000000000000111000000000;
	assign S_O[9] = 40'b0000000011100000000000000000011100000000;
	assign S_O[10] = 40'b0000000011100000000000000000011100000000;
	assign S_O[11] = 40'b0000000011100000000000000000011100000000;
	assign S_O[12] = 40'b0000000011100000000000000000011100000000;
	assign S_O[13] = 40'b0000000011100000000000000000011100000000;
	assign S_O[14] = 40'b0000000001110000000000000000111000000000;
	assign S_O[15] = 40'b0000000001111000000000000001111000000000;
	assign S_O[16] = 40'b0000000001111100000000000011111000000000;
	assign S_O[17] = 40'b0000000000111111000000001111110000000000;
	assign S_O[18] = 40'b0000000000011111111111111111100000000000;
	assign S_O[19] = 40'b0000000000000011111111111100000000000000;
	assign S_O[20] = 40'b0000000000000000111111110000000000000000;
	assign S_O[21] = 40'b0000000000000000000000000000000000000000;
	assign S_O[22] = 40'b0000000000000000000000000000000000000000;
	assign S_O[23] = 40'b0000000000000000000000000000000000000000;
	assign S_O[24] = 40'b0000000000000000000000000000000000000000;
	assign S_P[0] = 40'b0000000000000000000000000000000000000000;
	assign S_P[1] = 40'b0000000000000000000000000000000000000000;
	assign S_P[2] = 40'b0000000000000000000000000000000000000000;
	assign S_P[3] = 40'b0000000001111111111111111111111000000000;
	assign S_P[4] = 40'b0000000011111111111111111111111100000000;
	assign S_P[5] = 40'b0000000001111111111111111111111100000000;
	assign S_P[6] = 40'b0000000000000000001110000000011100000000;
	assign S_P[7] = 40'b0000000000000000001110000000011100000000;
	assign S_P[8] = 40'b0000000000000000001110000000011100000000;
	assign S_P[9] = 40'b0000000000000000001110000000011100000000;
	assign S_P[10] = 40'b0000000000000000001110000000011100000000;
	assign S_P[11] = 40'b0000000000000000001110000000011100000000;
	assign S_P[12] = 40'b0000000000000000001110000000011100000000;
	assign S_P[13] = 40'b0000000000000000001110000000011100000000;
	assign S_P[14] = 40'b0000000000000000001110000000011100000000;
	assign S_P[15] = 40'b0000000000000000000111000000111000000000;
	assign S_P[16] = 40'b0000000000000000000111100001111000000000;
	assign S_P[17] = 40'b0000000000000000000111111111111000000000;
	assign S_P[18] = 40'b0000000000000000000011111111110000000000;
	assign S_P[19] = 40'b0000000000000000000000011110000000000000;
	assign S_P[20] = 40'b0000000000000000000000000000000000000000;
	assign S_P[21] = 40'b0000000000000000000000000000000000000000;
	assign S_P[22] = 40'b0000000000000000000000000000000000000000;
	assign S_P[23] = 40'b0000000000000000000000000000000000000000;
	assign S_P[24] = 40'b0000000000000000000000000000000000000000;
	assign S_Q[0] = 40'b0000000000000000000000000000000000000000;
	assign S_Q[1] = 40'b0000000000000000000000000000000000000000;
	assign S_Q[2] = 40'b0000000000000000111111110000000000000000;
	assign S_Q[3] = 40'b0000000000000011111111111100000000000000;
	assign S_Q[4] = 40'b0000000000011111111111111111100000000000;
	assign S_Q[5] = 40'b0000000000111111000000001111110000000000;
	assign S_Q[6] = 40'b0000000001111100000000000011111000000000;
	assign S_Q[7] = 40'b0000000001111000000000000001111000000000;
	assign S_Q[8] = 40'b0000000001110000000000000000111000000000;
	assign S_Q[9] = 40'b0000000011100000000000000000011100000000;
	assign S_Q[10] = 40'b0000000011100000000000000000011100000000;
	assign S_Q[11] = 40'b0000000011101100000000000000011100000000;
	assign S_Q[12] = 40'b0000000011111110000000000000011100000000;
	assign S_Q[13] = 40'b0000000011111110000000000000011100000000;
	assign S_Q[14] = 40'b0000000001111100000000000000111000000000;
	assign S_Q[15] = 40'b0000000011111000000000000001111000000000;
	assign S_Q[16] = 40'b0000000111111100000000000011111000000000;
	assign S_Q[17] = 40'b0000001111111111000000001111110000000000;
	assign S_Q[18] = 40'b0000001111011111111111111111100000000000;
	assign S_Q[19] = 40'b0000000110000011111111111100000000000000;
	assign S_Q[20] = 40'b0000000000000000111111110000000000000000;
	assign S_Q[21] = 40'b0000000000000000000000000000000000000000;
	assign S_Q[22] = 40'b0000000000000000000000000000000000000000;
	assign S_Q[23] = 40'b0000000000000000000000000000000000000000;
	assign S_Q[24] = 40'b0000000000000000000000000000000000000000;
	assign S_R[0] = 40'b0000000000000000000000000000000000000000;
	assign S_R[1] = 40'b0000000000000000000000000000000000000000;
	assign S_R[2] = 40'b0000000000000000000000000000000000000000;
	assign S_R[3] = 40'b0000000001111111111111111111111000000000;
	assign S_R[4] = 40'b0000000011111111111111111111111100000000;
	assign S_R[5] = 40'b0000000001111111111111111111111100000000;
	assign S_R[6] = 40'b0000000000000000000111000000011100000000;
	assign S_R[7] = 40'b0000000000000000000111000000011100000000;
	assign S_R[8] = 40'b0000000000000000000111000000011100000000;
	assign S_R[9] = 40'b0000000000000000000111000000011100000000;
	assign S_R[10] = 40'b0000000000000000000111000000011100000000;
	assign S_R[11] = 40'b0000000000000000011111000000011100000000;
	assign S_R[12] = 40'b0000000000000000111111000000011100000000;
	assign S_R[13] = 40'b0000000000000011111111000000011100000000;
	assign S_R[14] = 40'b0000000000000111111111000000011100000000;
	assign S_R[15] = 40'b0000000000011111100011100000111000000000;
	assign S_R[16] = 40'b0000000000111111000011110001111000000000;
	assign S_R[17] = 40'b0000000011111100000011111111111000000000;
	assign S_R[18] = 40'b0000000011111000000001111111110000000000;
	assign S_R[19] = 40'b0000000001100000000000001110000000000000;
	assign S_R[20] = 40'b0000000000000000000000000000000000000000;
	assign S_R[21] = 40'b0000000000000000000000000000000000000000;
	assign S_R[22] = 40'b0000000000000000000000000000000000000000;
	assign S_R[23] = 40'b0000000000000000000000000000000000000000;
	assign S_R[24] = 40'b0000000000000000000000000000000000000000;
	assign S_S[0] = 40'b0000000000000000000000000000000000000000;
	assign S_S[1] = 40'b0000000000000000000000000000000000000000;
	assign S_S[2] = 40'b0000000000011000000000000111100000000000;
	assign S_S[3] = 40'b0000000000111100000000111111110000000000;
	assign S_S[4] = 40'b0000000001111100000001111111111000000000;
	assign S_S[5] = 40'b0000000001111000000001111001111000000000;
	assign S_S[6] = 40'b0000000001110000000001110000111000000000;
	assign S_S[7] = 40'b0000000011100000000011100000011100000000;
	assign S_S[8] = 40'b0000000011100000000011100000011100000000;
	assign S_S[9] = 40'b0000000011100000000111000000011100000000;
	assign S_S[10] = 40'b0000000011100000000111000000011100000000;
	assign S_S[11] = 40'b0000000011100000000111000000011100000000;
	assign S_S[12] = 40'b0000000011100000001110000000011100000000;
	assign S_S[13] = 40'b0000000011100000001110000000011100000000;
	assign S_S[14] = 40'b0000000001110000011100000000111000000000;
	assign S_S[15] = 40'b0000000001111000111100000001111000000000;
	assign S_S[16] = 40'b0000000001111111111100000011111000000000;
	assign S_S[17] = 40'b0000000000111111111000000011110000000000;
	assign S_S[18] = 40'b0000000000011111000000000001100000000000;
	assign S_S[19] = 40'b0000000000000000000000000000000000000000;
	assign S_S[20] = 40'b0000000000000000000000000000000000000000;
	assign S_S[21] = 40'b0000000000000000000000000000000000000000;
	assign S_S[22] = 40'b0000000000000000000000000000000000000000;
	assign S_S[23] = 40'b0000000000000000000000000000000000000000;
	assign S_S[24] = 40'b0000000000000000000000000000000000000000;
	assign S_T[0] = 40'b0000000000000000000000000000001000000000;
	assign S_T[1] = 40'b0000000000000000000000000000011100000000;
	assign S_T[2] = 40'b0000000000000000000000000000011100000000;
	assign S_T[3] = 40'b0000000000000000000000000000011100000000;
	assign S_T[4] = 40'b0000000000000000000000000000011100000000;
	assign S_T[5] = 40'b0000000000000000000000000000011100000000;
	assign S_T[6] = 40'b0000000000000000000000000000011100000000;
	assign S_T[7] = 40'b0000000001111111111111111111111100000000;
	assign S_T[8] = 40'b0000000011111111111111111111111100000000;
	assign S_T[9] = 40'b0000000001111111111111111111111100000000;
	assign S_T[10] = 40'b0000000000000000000000000000011100000000;
	assign S_T[11] = 40'b0000000000000000000000000000011100000000;
	assign S_T[12] = 40'b0000000000000000000000000000011100000000;
	assign S_T[13] = 40'b0000000000000000000000000000011100000000;
	assign S_T[14] = 40'b0000000000000000000000000000011100000000;
	assign S_T[15] = 40'b0000000000000000000000000000011100000000;
	assign S_T[16] = 40'b0000000000000000000000000000001000000000;
	assign S_T[17] = 40'b0000000000000000000000000000000000000000;
	assign S_T[18] = 40'b0000000000000000000000000000000000000000;
	assign S_T[19] = 40'b0000000000000000000000000000000000000000;
	assign S_T[20] = 40'b0000000000000000000000000000000000000000;
	assign S_T[21] = 40'b0000000000000000000000000000000000000000;
	assign S_T[22] = 40'b0000000000000000000000000000000000000000;
	assign S_T[23] = 40'b0000000000000000000000000000000000000000;
	assign S_T[24] = 40'b0000000000000000000000000000000000000000;
	assign S_U[0] = 40'b0000000000000000000000000000000000000000;
	assign S_U[1] = 40'b0000000000000000000000000000000000000000;
	assign S_U[2] = 40'b0000000000000000000000000000000000000000;
	assign S_U[3] = 40'b0000000000000011111111111111111000000000;
	assign S_U[4] = 40'b0000000000011111111111111111111100000000;
	assign S_U[5] = 40'b0000000000111111111111111111111000000000;
	assign S_U[6] = 40'b0000000001111100000000000000000000000000;
	assign S_U[7] = 40'b0000000001111000000000000000000000000000;
	assign S_U[8] = 40'b0000000001110000000000000000000000000000;
	assign S_U[9] = 40'b0000000011100000000000000000000000000000;
	assign S_U[10] = 40'b0000000011100000000000000000000000000000;
	assign S_U[11] = 40'b0000000011100000000000000000000000000000;
	assign S_U[12] = 40'b0000000011100000000000000000000000000000;
	assign S_U[13] = 40'b0000000011100000000000000000000000000000;
	assign S_U[14] = 40'b0000000001110000000000000000000000000000;
	assign S_U[15] = 40'b0000000001111000000000000000000000000000;
	assign S_U[16] = 40'b0000000001111100000000000000000000000000;
	assign S_U[17] = 40'b0000000000111111111111111111111000000000;
	assign S_U[18] = 40'b0000000000011111111111111111111100000000;
	assign S_U[19] = 40'b0000000000000011111111111111111000000000;
	assign S_U[20] = 40'b0000000000000000000000000000000000000000;
	assign S_U[21] = 40'b0000000000000000000000000000000000000000;
	assign S_U[22] = 40'b0000000000000000000000000000000000000000;
	assign S_U[23] = 40'b0000000000000000000000000000000000000000;
	assign S_U[24] = 40'b0000000000000000000000000000000000000000;
	assign S_V[0] = 40'b0000000000000000000000000000011000000000;
	assign S_V[1] = 40'b0000000000000000000000000001111100000000;
	assign S_V[2] = 40'b0000000000000000000000001111111000000000;
	assign S_V[3] = 40'b0000000000000000000001111111100000000000;
	assign S_V[4] = 40'b0000000000000000000111111100000000000000;
	assign S_V[5] = 40'b0000000000000000111111110000000000000000;
	assign S_V[6] = 40'b0000000000000011111110000000000000000000;
	assign S_V[7] = 40'b0000000000011111111000000000000000000000;
	assign S_V[8] = 40'b0000000001111111000000000000000000000000;
	assign S_V[9] = 40'b0000000011111000000000000000000000000000;
	assign S_V[10] = 40'b0000000001111111000000000000000000000000;
	assign S_V[11] = 40'b0000000000011111111000000000000000000000;
	assign S_V[12] = 40'b0000000000000011111110000000000000000000;
	assign S_V[13] = 40'b0000000000000000111111110000000000000000;
	assign S_V[14] = 40'b0000000000000000000111111100000000000000;
	assign S_V[15] = 40'b0000000000000000000001111111100000000000;
	assign S_V[16] = 40'b0000000000000000000000001111111000000000;
	assign S_V[17] = 40'b0000000000000000000000000001111100000000;
	assign S_V[18] = 40'b0000000000000000000000000000011000000000;
	assign S_V[19] = 40'b0000000000000000000000000000000000000000;
	assign S_V[20] = 40'b0000000000000000000000000000000000000000;
	assign S_V[21] = 40'b0000000000000000000000000000000000000000;
	assign S_V[22] = 40'b0000000000000000000000000000000000000000;
	assign S_V[23] = 40'b0000000000000000000000000000000000000000;
	assign S_V[24] = 40'b0000000000000000000000000000000000000000;
	assign S_W[0] = 40'b0000000000000000000000000000000000000000;
	assign S_W[1] = 40'b0000000000000000000000000000011000000000;
	assign S_W[2] = 40'b0000000000000000000000001111111100000000;
	assign S_W[3] = 40'b0000000000000000000011111111111000000000;
	assign S_W[4] = 40'b0000000000000000111111111111000000000000;
	assign S_W[5] = 40'b0000000000001111111111110000000000000000;
	assign S_W[6] = 40'b0000000001111111111100000000000000000000;
	assign S_W[7] = 40'b0000000011111111000000000000000000000000;
	assign S_W[8] = 40'b0000000001111111111100000000000000000000;
	assign S_W[9] = 40'b0000000000001111111111110000000000000000;
	assign S_W[10] = 40'b0000000000000000111111111111000000000000;
	assign S_W[11] = 40'b0000000000000000000011111111111000000000;
	assign S_W[12] = 40'b0000000000000000000000001111111100000000;
	assign S_W[13] = 40'b0000000000000000000011111111111000000000;
	assign S_W[14] = 40'b0000000000000000111111111111000000000000;
	assign S_W[15] = 40'b0000000000001111111111110000000000000000;
	assign S_W[16] = 40'b0000000001111111111100000000000000000000;
	assign S_W[17] = 40'b0000000011111111000000000000000000000000;
	assign S_W[18] = 40'b0000000001111111111100000000000000000000;
	assign S_W[19] = 40'b0000000000001111111111110000000000000000;
	assign S_W[20] = 40'b0000000000000000111111111111000000000000;
	assign S_W[21] = 40'b0000000000000000000011111111111000000000;
	assign S_W[22] = 40'b0000000000000000000000001111111100000000;
	assign S_W[23] = 40'b0000000000000000000000000000011000000000;
	assign S_W[24] = 40'b0000000000000000000000000000000000000000;
	assign S_X[0] = 40'b0000000000000000000000000000000000000000;
	assign S_X[1] = 40'b0000000000000000000000000000000000000000;
	assign S_X[2] = 40'b0000000001100000000000000000011000000000;
	assign S_X[3] = 40'b0000000011110000000000000000111100000000;
	assign S_X[4] = 40'b0000000011111100000000000011111100000000;
	assign S_X[5] = 40'b0000000000111110000000000111110000000000;
	assign S_X[6] = 40'b0000000000011111100000011111100000000000;
	assign S_X[7] = 40'b0000000000000111110000111110000000000000;
	assign S_X[8] = 40'b0000000000000011111111111100000000000000;
	assign S_X[9] = 40'b0000000000000000111111110000000000000000;
	assign S_X[10] = 40'b0000000000000000011111100000000000000000;
	assign S_X[11] = 40'b0000000000000000111111110000000000000000;
	assign S_X[12] = 40'b0000000000000011111111111100000000000000;
	assign S_X[13] = 40'b0000000000000111110000111110000000000000;
	assign S_X[14] = 40'b0000000000011111100000011111100000000000;
	assign S_X[15] = 40'b0000000000111110000000000111110000000000;
	assign S_X[16] = 40'b0000000011111100000000000011111100000000;
	assign S_X[17] = 40'b0000000011110000000000000000111100000000;
	assign S_X[18] = 40'b0000000001100000000000000000011000000000;
	assign S_X[19] = 40'b0000000000000000000000000000000000000000;
	assign S_X[20] = 40'b0000000000000000000000000000000000000000;
	assign S_X[21] = 40'b0000000000000000000000000000000000000000;
	assign S_X[22] = 40'b0000000000000000000000000000000000000000;
	assign S_X[23] = 40'b0000000000000000000000000000000000000000;
	assign S_X[24] = 40'b0000000000000000000000000000000000000000;
	assign S_Y[0] = 40'b0000000000000000000000000000011000000000;
	assign S_Y[1] = 40'b0000000000000000000000000000111100000000;
	assign S_Y[2] = 40'b0000000000000000000000000001111100000000;
	assign S_Y[3] = 40'b0000000000000000000000000111111000000000;
	assign S_Y[4] = 40'b0000000000000000000000001111100000000000;
	assign S_Y[5] = 40'b0000000000000000000000011111000000000000;
	assign S_Y[6] = 40'b0000000000000000000000111110000000000000;
	assign S_Y[7] = 40'b0000000000000000000011111100000000000000;
	assign S_Y[8] = 40'b0000000001111111111111110000000000000000;
	assign S_Y[9] = 40'b0000000011111111111111100000000000000000;
	assign S_Y[10] = 40'b0000000001111111111111110000000000000000;
	assign S_Y[11] = 40'b0000000000000000000011111100000000000000;
	assign S_Y[12] = 40'b0000000000000000000000111110000000000000;
	assign S_Y[13] = 40'b0000000000000000000000011111000000000000;
	assign S_Y[14] = 40'b0000000000000000000000001111100000000000;
	assign S_Y[15] = 40'b0000000000000000000000000111111000000000;
	assign S_Y[16] = 40'b0000000000000000000000000001111100000000;
	assign S_Y[17] = 40'b0000000000000000000000000000111100000000;
	assign S_Y[18] = 40'b0000000000000000000000000000011000000000;
	assign S_Y[19] = 40'b0000000000000000000000000000000000000000;
	assign S_Y[20] = 40'b0000000000000000000000000000000000000000;
	assign S_Y[21] = 40'b0000000000000000000000000000000000000000;
	assign S_Y[22] = 40'b0000000000000000000000000000000000000000;
	assign S_Y[23] = 40'b0000000000000000000000000000000000000000;
	assign S_Y[24] = 40'b0000000000000000000000000000000000000000;
	assign S_Z[0] = 40'b0000000000000000000000000000000000000000;
	assign S_Z[1] = 40'b0000000000000000000000000000000000000000;
	assign S_Z[2] = 40'b0000000001100000000000000000001000000000;
	assign S_Z[3] = 40'b0000000011110000000000000000011100000000;
	assign S_Z[4] = 40'b0000000011111100000000000000011100000000;
	assign S_Z[5] = 40'b0000000011111110000000000000011100000000;
	assign S_Z[6] = 40'b0000000011111111100000000000011100000000;
	assign S_Z[7] = 40'b0000000011100111110000000000011100000000;
	assign S_Z[8] = 40'b0000000011100011111100000000011100000000;
	assign S_Z[9] = 40'b0000000011100000111110000000011100000000;
	assign S_Z[10] = 40'b0000000011100000011111100000011100000000;
	assign S_Z[11] = 40'b0000000011100000000111110000011100000000;
	assign S_Z[12] = 40'b0000000011100000000011111100011100000000;
	assign S_Z[13] = 40'b0000000011100000000000111110011100000000;
	assign S_Z[14] = 40'b0000000011100000000000011111111100000000;
	assign S_Z[15] = 40'b0000000011100000000000000111111100000000;
	assign S_Z[16] = 40'b0000000011100000000000000011111100000000;
	assign S_Z[17] = 40'b0000000011100000000000000000111100000000;
	assign S_Z[18] = 40'b0000000001000000000000000000011000000000;
	assign S_Z[19] = 40'b0000000000000000000000000000000000000000;
	assign S_Z[20] = 40'b0000000000000000000000000000000000000000;
	assign S_Z[21] = 40'b0000000000000000000000000000000000000000;
	assign S_Z[22] = 40'b0000000000000000000000000000000000000000;
	assign S_Z[23] = 40'b0000000000000000000000000000000000000000;
	assign S_Z[24] = 40'b0000000000000000000000000000000000000000;
	assign S_colon[0] = 40'b0000000000000000000000000000000000000000;
	assign S_colon[1] = 40'b0000000000000000000000000000000000000000;
	assign S_colon[2] = 40'b0000000000000000000000000000000000000000;
	assign S_colon[3] = 40'b0000000001110000000001110000000000000000;
	assign S_colon[4] = 40'b0000000011111000000011111000000000000000;
	assign S_colon[5] = 40'b0000000011111000000011111000000000000000;
	assign S_colon[6] = 40'b0000000011111000000011111000000000000000;
	assign S_colon[7] = 40'b0000000001110000000001110000000000000000;
	assign S_colon[8] = 40'b0000000000000000000000000000000000000000;
	assign S_colon[9] = 40'b0000000000000000000000000000000000000000;
	assign S_colon[10] = 40'b0000000000000000000000000000000000000000;
	assign S_colon[11] = 40'b0000000000000000000000000000000000000000;
	assign S_colon[12] = 40'b0000000000000000000000000000000000000000;
	assign S_colon[13] = 40'b0000000000000000000000000000000000000000;
	assign S_colon[14] = 40'b0000000000000000000000000000000000000000;
	assign S_colon[15] = 40'b0000000000000000000000000000000000000000;
	assign S_colon[16] = 40'b0000000000000000000000000000000000000000;
	assign S_colon[17] = 40'b0000000000000000000000000000000000000000;
	assign S_colon[18] = 40'b0000000000000000000000000000000000000000;
	assign S_colon[19] = 40'b0000000000000000000000000000000000000000;
	assign S_colon[20] = 40'b0000000000000000000000000000000000000000;
	assign S_colon[21] = 40'b0000000000000000000000000000000000000000;
	assign S_colon[22] = 40'b0000000000000000000000000000000000000000;
	assign S_colon[23] = 40'b0000000000000000000000000000000000000000;
	assign S_colon[24] = 40'b0000000000000000000000000000000000000000;
	assign S_1[0] = 40'b0000000000000000000000000000000000000000;
	assign S_1[1] = 40'b0000000000000000000000000000000000000000;
	assign S_1[2] = 40'b0000000000000000000000000000000000000000;
	assign S_1[3] = 40'b0000000000000000000000000000000000000000;
	assign S_1[4] = 40'b0000000000000000000000000000000000000000;
	assign S_1[5] = 40'b0000000000000000000000000010000000000000;
	assign S_1[6] = 40'b0000000000000000000000000111000000000000;
	assign S_1[7] = 40'b0000000000000000000000000011100000000000;
	assign S_1[8] = 40'b0000000000000000000000000011110000000000;
	assign S_1[9] = 40'b0000000000000000000000000011111000000000;
	assign S_1[10] = 40'b0000000001111111111111111111111100000000;
	assign S_1[11] = 40'b0000000011111111111111111111111100000000;
	assign S_1[12] = 40'b0000000001111111111111111111111000000000;
	assign S_1[13] = 40'b0000000000000000000000000000000000000000;
	assign S_1[14] = 40'b0000000000000000000000000000000000000000;
	assign S_1[15] = 40'b0000000000000000000000000000000000000000;
	assign S_1[16] = 40'b0000000000000000000000000000000000000000;
	assign S_1[17] = 40'b0000000000000000000000000000000000000000;
	assign S_1[18] = 40'b0000000000000000000000000000000000000000;
	assign S_1[19] = 40'b0000000000000000000000000000000000000000;
	assign S_1[20] = 40'b0000000000000000000000000000000000000000;
	assign S_1[21] = 40'b0000000000000000000000000000000000000000;
	assign S_1[22] = 40'b0000000000000000000000000000000000000000;
	assign S_1[23] = 40'b0000000000000000000000000000000000000000;
	assign S_1[24] = 40'b0000000000000000000000000000000000000000;
	assign S_2[0] = 40'b0000000000000000000000000000000000000000;
	assign S_2[1] = 40'b0000000000000000000000000000000000000000;
	assign S_2[2] = 40'b0000000001100000000000000000000000000000;
	assign S_2[3] = 40'b0000000011110000000000000110000000000000;
	assign S_2[4] = 40'b0000000011111000000000001111110000000000;
	assign S_2[5] = 40'b0000000011111100000000000111111000000000;
	assign S_2[6] = 40'b0000000011111110000000000001111000000000;
	assign S_2[7] = 40'b0000000011111111000000000000111000000000;
	assign S_2[8] = 40'b0000000011101111100000000000011100000000;
	assign S_2[9] = 40'b0000000011100111110000000000011100000000;
	assign S_2[10] = 40'b0000000011100011111000000000011100000000;
	assign S_2[11] = 40'b0000000011100001111100000000011100000000;
	assign S_2[12] = 40'b0000000011100000111110000000011100000000;
	assign S_2[13] = 40'b0000000011100000011111000000111000000000;
	assign S_2[14] = 40'b0000000011100000001111110001111000000000;
	assign S_2[15] = 40'b0000000011100000000011111111111000000000;
	assign S_2[16] = 40'b0000000011100000000001111111110000000000;
	assign S_2[17] = 40'b0000000011100000000000001110000000000000;
	assign S_2[18] = 40'b0000000001000000000000000000000000000000;
	assign S_2[19] = 40'b0000000000000000000000000000000000000000;
	assign S_2[20] = 40'b0000000000000000000000000000000000000000;
	assign S_2[21] = 40'b0000000000000000000000000000000000000000;
	assign S_2[22] = 40'b0000000000000000000000000000000000000000;
	assign S_2[23] = 40'b0000000000000000000000000000000000000000;
	assign S_2[24] = 40'b0000000000000000000000000000000000000000;
	assign S_3[0] = 40'b0000000000000000000000000000000000000000;
	assign S_3[1] = 40'b0000000000000000000000000000000000000000;
	assign S_3[2] = 40'b0000000000000100000000000000000000000000;
	assign S_3[3] = 40'b0000000000111110000000000000000000000000;
	assign S_3[4] = 40'b0000000001111100000000000000001000000000;
	assign S_3[5] = 40'b0000000001111000000000000000011100000000;
	assign S_3[6] = 40'b0000000001110000000000000000011100000000;
	assign S_3[7] = 40'b0000000011100000000000000000011100000000;
	assign S_3[8] = 40'b0000000011100000000000000000011100000000;
	assign S_3[9] = 40'b0000000011100000000000110000011100000000;
	assign S_3[10] = 40'b0000000011100000000001111000011100000000;
	assign S_3[11] = 40'b0000000011100000000001111110011100000000;
	assign S_3[12] = 40'b0000000011100000000001111111011100000000;
	assign S_3[13] = 40'b0000000001110000000001111111111100000000;
	assign S_3[14] = 40'b0000000001111000000011100111111100000000;
	assign S_3[15] = 40'b0000000001111100000111100011111100000000;
	assign S_3[16] = 40'b0000000000111111111111100000111100000000;
	assign S_3[17] = 40'b0000000000011111111111000000011000000000;
	assign S_3[18] = 40'b0000000000000011111000000000000000000000;
	assign S_3[19] = 40'b0000000000000000000000000000000000000000;
	assign S_3[20] = 40'b0000000000000000000000000000000000000000;
	assign S_3[21] = 40'b0000000000000000000000000000000000000000;
	assign S_3[22] = 40'b0000000000000000000000000000000000000000;
	assign S_3[23] = 40'b0000000000000000000000000000000000000000;
	assign S_3[24] = 40'b0000000000000000000000000000000000000000;
	assign S_4[0] = 40'b0000000000000000000000000000000000000000;
	assign S_4[1] = 40'b0000000000000000000000000000000000000000;
	assign S_4[2] = 40'b0000000000000000110000000000000000000000;
	assign S_4[3] = 40'b0000000000000001111000000000000000000000;
	assign S_4[4] = 40'b0000000000000001111110000000000000000000;
	assign S_4[5] = 40'b0000000000000001111111000000000000000000;
	assign S_4[6] = 40'b0000000000000001111111110000000000000000;
	assign S_4[7] = 40'b0000000000000001110111111000000000000000;
	assign S_4[8] = 40'b0000000000000001110001111100000000000000;
	assign S_4[9] = 40'b0000000000000001110000111111000000000000;
	assign S_4[10] = 40'b0000000000000001110000011111100000000000;
	assign S_4[11] = 40'b0000000000000001110000000111111000000000;
	assign S_4[12] = 40'b0000000001111111111111111111111100000000;
	assign S_4[13] = 40'b0000000011111111111111111111111100000000;
	assign S_4[14] = 40'b0000000001111111111111111111111000000000;
	assign S_4[15] = 40'b0000000000000001110000000000000000000000;
	assign S_4[16] = 40'b0000000000000001110000000000000000000000;
	assign S_4[17] = 40'b0000000000000001110000000000000000000000;
	assign S_4[18] = 40'b0000000000000001110000000000000000000000;
	assign S_4[19] = 40'b0000000000000000100000000000000000000000;
	assign S_4[20] = 40'b0000000000000000000000000000000000000000;
	assign S_4[21] = 40'b0000000000000000000000000000000000000000;
	assign S_4[22] = 40'b0000000000000000000000000000000000000000;
	assign S_4[23] = 40'b0000000000000000000000000000000000000000;
	assign S_4[24] = 40'b0000000000000000000000000000000000000000;
	assign S_5[0] = 40'b0000000000000000000000000000000000000000;
	assign S_5[1] = 40'b0000000000000000000000000000000000000000;
	assign S_5[2] = 40'b0000000000000100000000000000000000000000;
	assign S_5[3] = 40'b0000000000111110000001111100000000000000;
	assign S_5[4] = 40'b0000000001111100000011111111111000000000;
	assign S_5[5] = 40'b0000000001111000000011111111111100000000;
	assign S_5[6] = 40'b0000000001110000000001110011111100000000;
	assign S_5[7] = 40'b0000000011100000000000111000011100000000;
	assign S_5[8] = 40'b0000000011100000000000111000011100000000;
	assign S_5[9] = 40'b0000000011100000000000111000011100000000;
	assign S_5[10] = 40'b0000000011100000000000111000011100000000;
	assign S_5[11] = 40'b0000000011100000000000111000011100000000;
	assign S_5[12] = 40'b0000000011100000000000111000011100000000;
	assign S_5[13] = 40'b0000000001110000000001110000011100000000;
	assign S_5[14] = 40'b0000000001111000000011110000011100000000;
	assign S_5[15] = 40'b0000000001111100000111110000011100000000;
	assign S_5[16] = 40'b0000000000111111111111100000001000000000;
	assign S_5[17] = 40'b0000000000011111111111000000000000000000;
	assign S_5[18] = 40'b0000000000000011111000000000000000000000;
	assign S_5[19] = 40'b0000000000000000000000000000000000000000;
	assign S_5[20] = 40'b0000000000000000000000000000000000000000;
	assign S_5[21] = 40'b0000000000000000000000000000000000000000;
	assign S_5[22] = 40'b0000000000000000000000000000000000000000;
	assign S_5[23] = 40'b0000000000000000000000000000000000000000;
	assign S_5[24] = 40'b0000000000000000000000000000000000000000;
	assign S_6[0] = 40'b0000000000000000000000000000000000000000;
	assign S_6[1] = 40'b0000000000000000000000000000000000000000;
	assign S_6[2] = 40'b0000000000000000000000000000000000000000;
	assign S_6[3] = 40'b0000000000000001111111110000000000000000;
	assign S_6[4] = 40'b0000000000011111111111111111000000000000;
	assign S_6[5] = 40'b0000000000111111111111111111100000000000;
	assign S_6[6] = 40'b0000000001111100001111101111111000000000;
	assign S_6[7] = 40'b0000000001111000000111100001111000000000;
	assign S_6[8] = 40'b0000000001110000000011100000111000000000;
	assign S_6[9] = 40'b0000000011100000000001110000011100000000;
	assign S_6[10] = 40'b0000000011100000000001110000011100000000;
	assign S_6[11] = 40'b0000000011100000000001110000011100000000;
	assign S_6[12] = 40'b0000000011100000000001110000011100000000;
	assign S_6[13] = 40'b0000000001110000000011100000011100000000;
	assign S_6[14] = 40'b0000000001111000000111100000111000000000;
	assign S_6[15] = 40'b0000000001111100001111100001111000000000;
	assign S_6[16] = 40'b0000000000111111111111000011110000000000;
	assign S_6[17] = 40'b0000000000011111111110000001000000000000;
	assign S_6[18] = 40'b0000000000000011110000000000000000000000;
	assign S_6[19] = 40'b0000000000000000000000000000000000000000;
	assign S_6[20] = 40'b0000000000000000000000000000000000000000;
	assign S_6[21] = 40'b0000000000000000000000000000000000000000;
	assign S_6[22] = 40'b0000000000000000000000000000000000000000;
	assign S_6[23] = 40'b0000000000000000000000000000000000000000;
	assign S_6[24] = 40'b0000000000000000000000000000000000000000;
	assign S_7[0] = 40'b0000000000000000000000000000000000000000;
	assign S_7[1] = 40'b0000000000000000000000000000000000000000;
	assign S_7[2] = 40'b0000000000000000000000000000001000000000;
	assign S_7[3] = 40'b0000000000000000000000000000011100000000;
	assign S_7[4] = 40'b0000000000000000000000000000011100000000;
	assign S_7[5] = 40'b0000000000000000000000000000011100000000;
	assign S_7[6] = 40'b0000000001000000000000000000011100000000;
	assign S_7[7] = 40'b0000000011110000000000000000011100000000;
	assign S_7[8] = 40'b0000000001111110000000000000011100000000;
	assign S_7[9] = 40'b0000000000011111100000000000011100000000;
	assign S_7[10] = 40'b0000000000000111111000000000011100000000;
	assign S_7[11] = 40'b0000000000000001111110000000011100000000;
	assign S_7[12] = 40'b0000000000000000011111100000011100000000;
	assign S_7[13] = 40'b0000000000000000000111111000011100000000;
	assign S_7[14] = 40'b0000000000000000000001111110011100000000;
	assign S_7[15] = 40'b0000000000000000000000011111111100000000;
	assign S_7[16] = 40'b0000000000000000000000000111111100000000;
	assign S_7[17] = 40'b0000000000000000000000000000111100000000;
	assign S_7[18] = 40'b0000000000000000000000000000001000000000;
	assign S_7[19] = 40'b0000000000000000000000000000000000000000;
	assign S_7[20] = 40'b0000000000000000000000000000000000000000;
	assign S_7[21] = 40'b0000000000000000000000000000000000000000;
	assign S_7[22] = 40'b0000000000000000000000000000000000000000;
	assign S_7[23] = 40'b0000000000000000000000000000000000000000;
	assign S_7[24] = 40'b0000000000000000000000000000000000000000;
	assign S_8[0] = 40'b0000000000000000000000000000000000000000;
	assign S_8[1] = 40'b0000000000000000000000000000000000000000;
	assign S_8[2] = 40'b0000000000000111100000000000000000000000;
	assign S_8[3] = 40'b0000000000111111111100000111000000000000;
	assign S_8[4] = 40'b0000000001111111111110011111110000000000;
	assign S_8[5] = 40'b0000000001111000011111111111111000000000;
	assign S_8[6] = 40'b0000000001110000001111110000111000000000;
	assign S_8[7] = 40'b0000000011100000000111110000011100000000;
	assign S_8[8] = 40'b0000000011100000000011110000011100000000;
	assign S_8[9] = 40'b0000000011100000000011100000011100000000;
	assign S_8[10] = 40'b0000000011100000000011100000011100000000;
	assign S_8[11] = 40'b0000000011100000000011100000011100000000;
	assign S_8[12] = 40'b0000000011100000000011110000011100000000;
	assign S_8[13] = 40'b0000000011100000000111110000011100000000;
	assign S_8[14] = 40'b0000000001110000001111110000111000000000;
	assign S_8[15] = 40'b0000000001111000011111111111111000000000;
	assign S_8[16] = 40'b0000000001111111111110011111110000000000;
	assign S_8[17] = 40'b0000000000111111111100000111000000000000;
	assign S_8[18] = 40'b0000000000000111100000000000000000000000;
	assign S_8[19] = 40'b0000000000000000000000000000000000000000;
	assign S_8[20] = 40'b0000000000000000000000000000000000000000;
	assign S_8[21] = 40'b0000000000000000000000000000000000000000;
	assign S_8[22] = 40'b0000000000000000000000000000000000000000;
	assign S_8[23] = 40'b0000000000000000000000000000000000000000;
	assign S_8[24] = 40'b0000000000000000000000000000000000000000;
	assign S_9[0] = 40'b0000000000000000000000000000000000000000;
	assign S_9[1] = 40'b0000000000000000000000000000000000000000;
	assign S_9[2] = 40'b0000000000000000000000111100000000000000;
	assign S_9[3] = 40'b0000000000001000000111111111100000000000;
	assign S_9[4] = 40'b0000000000111100001111111111110000000000;
	assign S_9[5] = 40'b0000000001111000011111000011111000000000;
	assign S_9[6] = 40'b0000000001110000011110000001111000000000;
	assign S_9[7] = 40'b0000000011100000011100000000111000000000;
	assign S_9[8] = 40'b0000000011100000111000000000011100000000;
	assign S_9[9] = 40'b0000000011100000111000000000011100000000;
	assign S_9[10] = 40'b0000000011100000111000000000011100000000;
	assign S_9[11] = 40'b0000000011100000111000000000011100000000;
	assign S_9[12] = 40'b0000000001110000011100000000111000000000;
	assign S_9[13] = 40'b0000000001111000011110000001111000000000;
	assign S_9[14] = 40'b0000000001111111011111000011111000000000;
	assign S_9[15] = 40'b0000000000011111111111111111110000000000;
	assign S_9[16] = 40'b0000000000001111111111111111100000000000;
	assign S_9[17] = 40'b0000000000000000111111111000000000000000;
	assign S_9[18] = 40'b0000000000000000000000000000000000000000;
	assign S_9[19] = 40'b0000000000000000000000000000000000000000;
	assign S_9[20] = 40'b0000000000000000000000000000000000000000;
	assign S_9[21] = 40'b0000000000000000000000000000000000000000;
	assign S_9[22] = 40'b0000000000000000000000000000000000000000;
	assign S_9[23] = 40'b0000000000000000000000000000000000000000;
	assign S_9[24] = 40'b0000000000000000000000000000000000000000;
	
	logic[19:0] s_A[14:0];
logic[19:0] s_B[14:0];
logic[19:0] s_C[14:0];
logic[19:0] s_D[14:0];
logic[19:0] s_E[14:0];
logic[19:0] s_F[14:0];
logic[19:0] s_G[14:0];
logic[19:0] s_H[14:0];
logic[19:0] s_I[14:0];
logic[19:0] s_J[14:0];
logic[19:0] s_K[14:0];
logic[19:0] s_L[14:0];
logic[19:0] s_M[14:0];
logic[19:0] s_N[14:0];
logic[19:0] s_O[14:0];
logic[19:0] s_P[14:0];
logic[19:0] s_Q[14:0];
logic[19:0] s_R[14:0];
logic[19:0] s_S[14:0];
logic[19:0] s_T[14:0];
logic[19:0] s_U[14:0];
logic[19:0] s_V[14:0];
logic[19:0] s_W[14:0];
logic[19:0] s_X[14:0];
logic[19:0] s_Y[14:0];
logic[19:0] s_Z[14:0];
logic[19:0] s_dot[14:0];
logic[19:0] s_colon[14:0];
logic[19:0] s_blank [14:0];
logic[19:0] s_1[14:0];
logic[19:0] s_2[14:0];
logic[19:0] s_3[14:0];
logic[19:0] s_4[14:0];
logic[19:0] s_5[14:0];
logic[19:0] s_6[14:0];
logic[19:0] s_7[14:0];
logic[19:0] s_8[14:0];
logic[19:0] s_9[14:0];
assign s_A[0] = 20'b00000000000000000000;
assign s_A[1] = 20'b00001110000000000000;
assign s_A[2] = 20'b00011111110000000000;
assign s_A[3] = 20'b00001111111100000000;
assign s_A[4] = 20'b00000011111111100000;
assign s_A[5] = 20'b00000001111111111100;
assign s_A[6] = 20'b00000001110011111110;
assign s_A[7] = 20'b00000001110111111100;
assign s_A[8] = 20'b00000001111111111000;
assign s_A[9] = 20'b00000011111111000000;
assign s_A[10] = 20'b00001111111100000000;
assign s_A[11] = 20'b00011111100000000000;
assign s_A[12] = 20'b00000000000000000000;
assign s_A[13] = 20'b00000000000000000000;
assign s_A[14] = 20'b00000000000000000000;
assign s_B[0] = 20'b00000000000000000000;
assign s_B[1] = 20'b00000000000000000000;
assign s_B[2] = 20'b00001111111111111100;
assign s_B[3] = 20'b00011111111111111110;
assign s_B[4] = 20'b00011111111111111110;
assign s_B[5] = 20'b00011100001110001110;
assign s_B[6] = 20'b00011100001110001110;
assign s_B[7] = 20'b00011100001110001110;
assign s_B[8] = 20'b00011100001110001110;
assign s_B[9] = 20'b00011100011110011110;
assign s_B[10] = 20'b00011110011111011100;
assign s_B[11] = 20'b00001111111111111100;
assign s_B[12] = 20'b00000000000000000000;
assign s_B[13] = 20'b00000000000000000000;
assign s_B[14] = 20'b00000000000000000000;
assign s_C[0] = 20'b00000000000000000000;
assign s_C[1] = 20'b00000000000000000000;
assign s_C[2] = 20'b00000001111111100000;
assign s_C[3] = 20'b00000111111111111000;
assign s_C[4] = 20'b00001111111111111100;
assign s_C[5] = 20'b00001111000000111100;
assign s_C[6] = 20'b00011110000000011110;
assign s_C[7] = 20'b00011100000000001110;
assign s_C[8] = 20'b00011100000000001110;
assign s_C[9] = 20'b00011110000000011110;
assign s_C[10] = 20'b00001111000000111100;
assign s_C[11] = 20'b00001111000001111100;
assign s_C[12] = 20'b00000000000000000000;
assign s_C[13] = 20'b00000000000000000000;
assign s_C[14] = 20'b00000000000000000000;
assign s_D[0] = 20'b00000000000000000000;
assign s_D[1] = 20'b00000000000000000000;
assign s_D[2] = 20'b00001111111111111100;
assign s_D[3] = 20'b00011111111111111110;
assign s_D[4] = 20'b00011111111111111110;
assign s_D[5] = 20'b00011100000000001110;
assign s_D[6] = 20'b00011100000000001110;
assign s_D[7] = 20'b00011100000000001110;
assign s_D[8] = 20'b00011100000000011110;
assign s_D[9] = 20'b00011110000000011100;
assign s_D[10] = 20'b00001111000001111100;
assign s_D[11] = 20'b00000111111111111000;
assign s_D[12] = 20'b00000000000000000000;
assign s_D[13] = 20'b00000000000000000000;
assign s_D[14] = 20'b00000000000000000000;
assign s_E[0] = 20'b00000000000000000000;
assign s_E[1] = 20'b00000000000000000000;
assign s_E[2] = 20'b00001111111111111100;
assign s_E[3] = 20'b00011111111111111110;
assign s_E[4] = 20'b00011111111111111110;
assign s_E[5] = 20'b00011100001110001110;
assign s_E[6] = 20'b00011100001110001110;
assign s_E[7] = 20'b00011100001110001110;
assign s_E[8] = 20'b00011100001110001110;
assign s_E[9] = 20'b00011100000100001110;
assign s_E[10] = 20'b00011100000000001110;
assign s_E[11] = 20'b00011100000000001110;
assign s_E[12] = 20'b00000000000000000000;
assign s_E[13] = 20'b00000000000000000000;
assign s_E[14] = 20'b00000000000000000000;
assign s_F[0] = 20'b00000000000000000000;
assign s_F[1] = 20'b00000000000000000000;
assign s_F[2] = 20'b00001111111111111100;
assign s_F[3] = 20'b00011111111111111110;
assign s_F[4] = 20'b00001111111111111110;
assign s_F[5] = 20'b00000000001110001110;
assign s_F[6] = 20'b00000000001110001110;
assign s_F[7] = 20'b00000000001110001110;
assign s_F[8] = 20'b00000000001110001110;
assign s_F[9] = 20'b00000000000100001110;
assign s_F[10] = 20'b00000000000000001110;
assign s_F[11] = 20'b00000000000000001110;
assign s_F[12] = 20'b00000000000000000000;
assign s_F[13] = 20'b00000000000000000000;
assign s_F[14] = 20'b00000000000000000000;
assign s_G[0] = 20'b00000000000000000000;
assign s_G[1] = 20'b00000000000000000000;
assign s_G[2] = 20'b00000001111111100000;
assign s_G[3] = 20'b00000111111111111000;
assign s_G[4] = 20'b00001111111111111100;
assign s_G[5] = 20'b00001111000000111100;
assign s_G[6] = 20'b00011110000000011110;
assign s_G[7] = 20'b00011100000000001110;
assign s_G[8] = 20'b00011100010000001110;
assign s_G[9] = 20'b00011110111000011110;
assign s_G[10] = 20'b00001111111000111100;
assign s_G[11] = 20'b00001111111001111100;
assign s_G[12] = 20'b00000000000000000000;
assign s_G[13] = 20'b00000000000000000000;
assign s_G[14] = 20'b00000000000000000000;
assign s_H[0] = 20'b00000000000000000000;
assign s_H[1] = 20'b00000000000000000000;
assign s_H[2] = 20'b00001111111111111100;
assign s_H[3] = 20'b00011111111111111110;
assign s_H[4] = 20'b00001111111111111100;
assign s_H[5] = 20'b00000000001110000000;
assign s_H[6] = 20'b00000000001110000000;
assign s_H[7] = 20'b00000000001110000000;
assign s_H[8] = 20'b00000000001110000000;
assign s_H[9] = 20'b00000000001110000000;
assign s_H[10] = 20'b00000000001110000000;
assign s_H[11] = 20'b00001111111111111100;
assign s_H[12] = 20'b00000000000000000000;
assign s_H[13] = 20'b00000000000000000000;
assign s_H[14] = 20'b00000000000000000000;
assign s_I[0] = 20'b00000000000000000000;
assign s_I[1] = 20'b00000000000000000000;
assign s_I[2] = 20'b00001111111111111100;
assign s_I[3] = 20'b00011111111111111110;
assign s_I[4] = 20'b00001111111111111100;
assign s_I[5] = 20'b00000000000000000000;
assign s_I[6] = 20'b00000000000000000000;
assign s_I[7] = 20'b00000000000000000000;
assign s_I[8] = 20'b00000000000000000000;
assign s_I[9] = 20'b00000000000000000000;
assign s_I[10] = 20'b00000000000000000000;
assign s_I[11] = 20'b00000000000000000000;
assign s_I[12] = 20'b00000000000000000000;
assign s_I[13] = 20'b00000000000000000000;
assign s_I[14] = 20'b00000000000000000000;
assign s_J[0] = 20'b00000000000000000000;
assign s_J[1] = 20'b00000001100000000000;
assign s_J[2] = 20'b00001111110000000000;
assign s_J[3] = 20'b00001111100000000000;
assign s_J[4] = 20'b00011110000000000000;
assign s_J[5] = 20'b00011100000000000000;
assign s_J[6] = 20'b00011100000000000000;
assign s_J[7] = 20'b00001111111111111100;
assign s_J[8] = 20'b00001111111111111110;
assign s_J[9] = 20'b00000111111111111100;
assign s_J[10] = 20'b00000000000000000000;
assign s_J[11] = 20'b00000000000000000000;
assign s_J[12] = 20'b00000000000000000000;
assign s_J[13] = 20'b00000000000000000000;
assign s_J[14] = 20'b00000000000000000000;
assign s_K[0] = 20'b00000000000000000000;
assign s_K[1] = 20'b00000000000000000000;
assign s_K[2] = 20'b00001111111111111100;
assign s_K[3] = 20'b00011111111111111110;
assign s_K[4] = 20'b00001111111111111100;
assign s_K[5] = 20'b00000000111100000000;
assign s_K[6] = 20'b00000000011110000000;
assign s_K[7] = 20'b00000000011111000000;
assign s_K[8] = 20'b00000001111111100000;
assign s_K[9] = 20'b00000011111011110000;
assign s_K[10] = 20'b00001111110001111000;
assign s_K[11] = 20'b00011111000000111100;
assign s_K[12] = 20'b00000000000000000000;
assign s_K[13] = 20'b00000000000000000000;
assign s_K[14] = 20'b00000000000000000000;
assign s_L[0] = 20'b00000000000000000000;
assign s_L[1] = 20'b00000000000000000000;
assign s_L[2] = 20'b00001111111111111100;
assign s_L[3] = 20'b00011111111111111110;
assign s_L[4] = 20'b00011111111111111100;
assign s_L[5] = 20'b00011100000000000000;
assign s_L[6] = 20'b00011100000000000000;
assign s_L[7] = 20'b00011100000000000000;
assign s_L[8] = 20'b00011100000000000000;
assign s_L[9] = 20'b00011100000000000000;
assign s_L[10] = 20'b00011100000000000000;
assign s_L[11] = 20'b00011100000000000000;
assign s_L[12] = 20'b00000000000000000000;
assign s_L[13] = 20'b00000000000000000000;
assign s_L[14] = 20'b00000000000000000000;
assign s_M[0] = 20'b00000000000000000000;
assign s_M[1] = 20'b00000000000000000000;
assign s_M[2] = 20'b00001111111111111100;
assign s_M[3] = 20'b00011111111111111110;
assign s_M[4] = 20'b00001111111111111100;
assign s_M[5] = 20'b00000000011111111000;
assign s_M[6] = 20'b00000011111111000000;
assign s_M[7] = 20'b00001111111100000000;
assign s_M[8] = 20'b00011111100000000000;
assign s_M[9] = 20'b00001111111000000000;
assign s_M[10] = 20'b00000111111110000000;
assign s_M[11] = 20'b00000001111111110000;
assign s_M[12] = 20'b00000000000000000000;
assign s_M[13] = 20'b00000000000000000000;
assign s_M[14] = 20'b00000000000000000000;
assign s_N[0] = 20'b00000000000000000000;
assign s_N[1] = 20'b00000000000000000000;
assign s_N[2] = 20'b00001111111111111100;
assign s_N[3] = 20'b00011111111111111110;
assign s_N[4] = 20'b00001111111111111100;
assign s_N[5] = 20'b00000000000011111000;
assign s_N[6] = 20'b00000000001111110000;
assign s_N[7] = 20'b00000000011111000000;
assign s_N[8] = 20'b00000001111110000000;
assign s_N[9] = 20'b00000011111000000000;
assign s_N[10] = 20'b00001111110000000000;
assign s_N[11] = 20'b00011111111111111100;
assign s_N[12] = 20'b00000000000000000000;
assign s_N[13] = 20'b00000000000000000000;
assign s_N[14] = 20'b00000000000000000000;
assign s_O[0] = 20'b00000000000000000000;
assign s_O[1] = 20'b00000000000000000000;
assign s_O[2] = 20'b00000001111111100000;
assign s_O[3] = 20'b00000111111111111000;
assign s_O[4] = 20'b00001111111111111100;
assign s_O[5] = 20'b00001111000000111100;
assign s_O[6] = 20'b00011110000000011110;
assign s_O[7] = 20'b00011100000000001110;
assign s_O[8] = 20'b00011100000000001110;
assign s_O[9] = 20'b00011110000000011110;
assign s_O[10] = 20'b00001111000000111100;
assign s_O[11] = 20'b00001111111111111100;
assign s_O[12] = 20'b00000011111111100000;
assign s_O[13] = 20'b00000000000000000000;
assign s_O[14] = 20'b00000000000000000000;
assign s_P[0] = 20'b00000000000000000000;
assign s_P[1] = 20'b00000000000000000000;
assign s_P[2] = 20'b00001111111111111100;
assign s_P[3] = 20'b00011111111111111110;
assign s_P[4] = 20'b00001111111111111110;
assign s_P[5] = 20'b00000000011100001110;
assign s_P[6] = 20'b00000000011100001110;
assign s_P[7] = 20'b00000000011100001110;
assign s_P[8] = 20'b00000000011100001110;
assign s_P[9] = 20'b00000000011100011110;
assign s_P[10] = 20'b00000000011110011100;
assign s_P[11] = 20'b00000000001111111100;
assign s_P[12] = 20'b00000000000000000000;
assign s_P[13] = 20'b00000000000000000000;
assign s_P[14] = 20'b00000000000000000000;
assign s_Q[0] = 20'b00000000000000000000;
assign s_Q[1] = 20'b00000000000000000000;
assign s_Q[2] = 20'b00000001111111100000;
assign s_Q[3] = 20'b00000111111111111000;
assign s_Q[4] = 20'b00001111111111111100;
assign s_Q[5] = 20'b00001111000000111100;
assign s_Q[6] = 20'b00011110000000011110;
assign s_Q[7] = 20'b00011110000000001110;
assign s_Q[8] = 20'b00011111000000001110;
assign s_Q[9] = 20'b00011111000000011110;
assign s_Q[10] = 20'b00011111000000111100;
assign s_Q[11] = 20'b00111111111111111100;
assign s_Q[12] = 20'b00000000000000000000;
assign s_Q[13] = 20'b00000000000000000000;
assign s_Q[14] = 20'b00000000000000000000;
assign s_R[0] = 20'b00000000000000000000;
assign s_R[1] = 20'b00000000000000000000;
assign s_R[2] = 20'b00001111111111111100;
assign s_R[3] = 20'b00011111111111111110;
assign s_R[4] = 20'b00001111111111111110;
assign s_R[5] = 20'b00000000001110001110;
assign s_R[6] = 20'b00000000001110001110;
assign s_R[7] = 20'b00000000011110001110;
assign s_R[8] = 20'b00000000111110001110;
assign s_R[9] = 20'b00000011111110011110;
assign s_R[10] = 20'b00001111111111011100;
assign s_R[11] = 20'b00011111101111111100;
assign s_R[12] = 20'b00000000000000000000;
assign s_R[13] = 20'b00000000000000000000;
assign s_R[14] = 20'b00000000000000000000;
assign s_S[0] = 20'b00000000000000000000;
assign s_S[1] = 20'b00000000000000000000;
assign s_S[2] = 20'b00000110000011111000;
assign s_S[3] = 20'b00001111000111111100;
assign s_S[4] = 20'b00001111000111111100;
assign s_S[5] = 20'b00011110001111011110;
assign s_S[6] = 20'b00011100001110001110;
assign s_S[7] = 20'b00011100001110001110;
assign s_S[8] = 20'b00011100011100001110;
assign s_S[9] = 20'b00011110011100011110;
assign s_S[10] = 20'b00011111111000111100;
assign s_S[11] = 20'b00001111111000111100;
assign s_S[12] = 20'b00000011110000000000;
assign s_S[13] = 20'b00000000000000000000;
assign s_S[14] = 20'b00000000000000000000;
assign s_T[0] = 20'b00000000000000000000;
assign s_T[1] = 20'b00000000000000000100;
assign s_T[2] = 20'b00000000000000001110;
assign s_T[3] = 20'b00000000000000001110;
assign s_T[4] = 20'b00000000000000001110;
assign s_T[5] = 20'b00001111111111111110;
assign s_T[6] = 20'b00011111111111111110;
assign s_T[7] = 20'b00001111111111111110;
assign s_T[8] = 20'b00000000000000001110;
assign s_T[9] = 20'b00000000000000001110;
assign s_T[10] = 20'b00000000000000001110;
assign s_T[11] = 20'b00000000000000000100;
assign s_T[12] = 20'b00000000000000000000;
assign s_T[13] = 20'b00000000000000000000;
assign s_T[14] = 20'b00000000000000000000;
assign s_U[0] = 20'b00000000000000000000;
assign s_U[1] = 20'b00000000000000000000;
assign s_U[2] = 20'b00000001111111111100;
assign s_U[3] = 20'b00000111111111111110;
assign s_U[4] = 20'b00001111111111111100;
assign s_U[5] = 20'b00001111000000000000;
assign s_U[6] = 20'b00011110000000000000;
assign s_U[7] = 20'b00011100000000000000;
assign s_U[8] = 20'b00011100000000000000;
assign s_U[9] = 20'b00011110000000000000;
assign s_U[10] = 20'b00011111000000000000;
assign s_U[11] = 20'b00001111111111111100;
assign s_U[12] = 20'b00000000000000000000;
assign s_U[13] = 20'b00000000000000000000;
assign s_U[14] = 20'b00000000000000000000;
assign s_V[0] = 20'b00000000000000000000;
assign s_V[1] = 20'b00000000000000111100;
assign s_V[2] = 20'b00000000000011111110;
assign s_V[3] = 20'b00000000011111111100;
assign s_V[4] = 20'b00000011111111100000;
assign s_V[5] = 20'b00001111111100000000;
assign s_V[6] = 20'b00011111110000000000;
assign s_V[7] = 20'b00001111110000000000;
assign s_V[8] = 20'b00001111111110000000;
assign s_V[9] = 20'b00000001111111100000;
assign s_V[10] = 20'b00000000001111111100;
assign s_V[11] = 20'b00000000000011111110;
assign s_V[12] = 20'b00000000000000000000;
assign s_V[13] = 20'b00000000000000000000;
assign s_V[14] = 20'b00000000000000000000;
assign s_W[0] = 20'b00000000000000000000;
assign s_W[1] = 20'b00000000000000011100;
assign s_W[2] = 20'b00000000000111111110;
assign s_W[3] = 20'b00000001111111111100;
assign s_W[4] = 20'b00001111111111110000;
assign s_W[5] = 20'b00011111111100000000;
assign s_W[6] = 20'b00001111111111000000;
assign s_W[7] = 20'b00000111111111111100;
assign s_W[8] = 20'b00000000011111111110;
assign s_W[9] = 20'b00000001111111111100;
assign s_W[10] = 20'b00001111111111110000;
assign s_W[11] = 20'b00011111111100000000;
assign s_W[12] = 20'b00000000000000000000;
assign s_W[13] = 20'b00000000000000000000;
assign s_W[14] = 20'b00000000000000000000;
assign s_X[0] = 20'b00000000000000000000;
assign s_X[1] = 20'b00000000000000000000;
assign s_X[2] = 20'b00001100000000001100;
assign s_X[3] = 20'b00011111000000111110;
assign s_X[4] = 20'b00011111100001111100;
assign s_X[5] = 20'b00000111111111110000;
assign s_X[6] = 20'b00000011111111100000;
assign s_X[7] = 20'b00000000111110000000;
assign s_X[8] = 20'b00000011111111100000;
assign s_X[9] = 20'b00000111111111110000;
assign s_X[10] = 20'b00011111100001111100;
assign s_X[11] = 20'b00011111000000111110;
assign s_X[12] = 20'b00000000000000000000;
assign s_X[13] = 20'b00000000000000000000;
assign s_X[14] = 20'b00000000000000000000;
assign s_Y[0] = 20'b00000000000000000000;
assign s_Y[1] = 20'b00000000000000001100;
assign s_Y[2] = 20'b00000000000000111110;
assign s_Y[3] = 20'b00000000000001111100;
assign s_Y[4] = 20'b00000000000011110000;
assign s_Y[5] = 20'b00001111111111100000;
assign s_Y[6] = 20'b00011111111111000000;
assign s_Y[7] = 20'b00001111111111000000;
assign s_Y[8] = 20'b00000000000111100000;
assign s_Y[9] = 20'b00000000000011110000;
assign s_Y[10] = 20'b00000000000001111100;
assign s_Y[11] = 20'b00000000000000011110;
assign s_Y[12] = 20'b00000000000000000000;
assign s_Y[13] = 20'b00000000000000000000;
assign s_Y[14] = 20'b00000000000000000000;
assign s_Z[0] = 20'b00000000000000000000;
assign s_Z[1] = 20'b00000000000000000000;
assign s_Z[2] = 20'b00001100000000000100;
assign s_Z[3] = 20'b00011111000000001110;
assign s_Z[4] = 20'b00011111100000001110;
assign s_Z[5] = 20'b00011111111000001110;
assign s_Z[6] = 20'b00011111111100001110;
assign s_Z[7] = 20'b00011100111110001110;
assign s_Z[8] = 20'b00011100011111101110;
assign s_Z[9] = 20'b00011100000111111110;
assign s_Z[10] = 20'b00011100000001111110;
assign s_Z[11] = 20'b00011100000000111110;
assign s_Z[12] = 20'b00000000000000000000;
assign s_Z[13] = 20'b00000000000000000000;
assign s_Z[14] = 20'b00000000000000000000;
assign s_dot[0] = 20'b00000000000000000000;
assign s_dot[1] = 20'b00000000000000000000;
assign s_dot[2] = 20'b00000100000000000000;
assign s_dot[3] = 20'b00011110000000000000;
assign s_dot[4] = 20'b00011110000000000000;
assign s_dot[5] = 20'b00011110000000000000;
assign s_dot[6] = 20'b00000100000000000000;
assign s_dot[7] = 20'b00000000000000000000;
assign s_dot[8] = 20'b00000000000000000000;
assign s_dot[9] = 20'b00000000000000000000;
assign s_dot[10] = 20'b00000000000000000000;
assign s_dot[11] = 20'b00000000000000000000;
assign s_dot[12] = 20'b00000000000000000000;
assign s_dot[13] = 20'b00000000000000000000;
assign s_dot[14] = 20'b00000000000000000000;
assign s_colon[0] = 20'b00000000000000000000;
assign s_colon[1] = 20'b00000000000000000000;
assign s_colon[2] = 20'b00000100000010000000;
assign s_colon[3] = 20'b00011110001111000000;
assign s_colon[4] = 20'b00011110001111000000;
assign s_colon[5] = 20'b00011110001111000000;
assign s_colon[6] = 20'b00000100000010000000;
assign s_colon[7] = 20'b00000000000000000000;
assign s_colon[8] = 20'b00000000000000000000;
assign s_colon[9] = 20'b00000000000000000000;
assign s_colon[10] = 20'b00000000000000000000;
assign s_colon[11] = 20'b00000000000000000000;
assign s_colon[12] = 20'b00000000000000000000;
assign s_colon[13] = 20'b00000000000000000000;
assign s_colon[14] = 20'b00000000000000000000;
assign s_blank [0] = 20'b00000000000000000000;
assign s_blank [1] = 20'b00000000000000000000;
assign s_blank [2] = 20'b00000000000000000000;
assign s_blank [3] = 20'b00000000000000000000;
assign s_blank [4] = 20'b00000000000000000000;
assign s_blank [5] = 20'b00000000000000000000;
assign s_blank [6] = 20'b00000000000000000000;
assign s_blank [7] = 20'b00000000000000000000;
assign s_blank [8] = 20'b00000000000000000000;
assign s_blank [9] = 20'b00000000000000000000;
assign s_blank [10] = 20'b00000000000000000000;
assign s_blank [11] = 20'b00000000000000000000;
assign s_blank [12] = 20'b00000000000000000000;
assign s_blank [13] = 20'b00000000000000000000;
assign s_blank [14] = 20'b00000000000000000000;
assign s_1[0] = 20'b00000000000000000000;
assign s_1[1] = 20'b00000000000000000000;
assign s_1[2] = 20'b00000000000000000000;
assign s_1[3] = 20'b00000000000000000000;
assign s_1[4] = 20'b00000000000000110000;
assign s_1[5] = 20'b00000000000001111000;
assign s_1[6] = 20'b00000000000000111000;
assign s_1[7] = 20'b00001111111111111100;
assign s_1[8] = 20'b00011111111111111110;
assign s_1[9] = 20'b00001111111111111100;
assign s_1[10] = 20'b00000000000000000000;
assign s_1[11] = 20'b00000000000000000000;
assign s_1[12] = 20'b00000000000000000000;
assign s_1[13] = 20'b00000000000000000000;
assign s_1[14] = 20'b00000000000000000000;
assign s_2[0] = 20'b00000000000000000000;
assign s_2[1] = 20'b00000000000000000000;
assign s_2[2] = 20'b00001100000000100000;
assign s_2[3] = 20'b00011110000001111000;
assign s_2[4] = 20'b00011111000000111100;
assign s_2[5] = 20'b00011111100000011100;
assign s_2[6] = 20'b00011111110000001110;
assign s_2[7] = 20'b00011111111000001110;
assign s_2[8] = 20'b00011101111100001110;
assign s_2[9] = 20'b00011100111110011100;
assign s_2[10] = 20'b00011100011111111100;
assign s_2[11] = 20'b00011100000111111000;
assign s_2[12] = 20'b00000000000000000000;
assign s_2[13] = 20'b00000000000000000000;
assign s_2[14] = 20'b00000000000000000000;
assign s_3[0] = 20'b00000000000000000000;
assign s_3[1] = 20'b00000000000000000000;
assign s_3[2] = 20'b00000110000000000000;
assign s_3[3] = 20'b00001111000000000100;
assign s_3[4] = 20'b00001111000000001110;
assign s_3[5] = 20'b00011100000000001110;
assign s_3[6] = 20'b00011100000010001110;
assign s_3[7] = 20'b00011100000111001110;
assign s_3[8] = 20'b00011100000111111110;
assign s_3[9] = 20'b00011110001111111110;
assign s_3[10] = 20'b00001111111110111110;
assign s_3[11] = 20'b00000111111110011110;
assign s_3[12] = 20'b00000000000000000000;
assign s_3[13] = 20'b00000000000000000000;
assign s_3[14] = 20'b00000000000000000000;
assign s_4[0] = 20'b00000000000000000000;
assign s_4[1] = 20'b00000000000000000000;
assign s_4[2] = 20'b00000000110000000000;
assign s_4[3] = 20'b00000001111100000000;
assign s_4[4] = 20'b00000001111110000000;
assign s_4[5] = 20'b00000001111111000000;
assign s_4[6] = 20'b00000001110111110000;
assign s_4[7] = 20'b00000001110011111000;
assign s_4[8] = 20'b00001111111111111100;
assign s_4[9] = 20'b00011111111111111110;
assign s_4[10] = 20'b00001111111111111100;
assign s_4[11] = 20'b00000001110000000000;
assign s_4[12] = 20'b00000000000000000000;
assign s_4[13] = 20'b00000000000000000000;
assign s_4[14] = 20'b00000000000000000000;
assign s_5[0] = 20'b00000000000000000000;
assign s_5[1] = 20'b00000000000000000000;
assign s_5[2] = 20'b00000110000110000000;
assign s_5[3] = 20'b00001111001111111100;
assign s_5[4] = 20'b00001111001111111110;
assign s_5[5] = 20'b00011100000111111110;
assign s_5[6] = 20'b00011100000111001110;
assign s_5[7] = 20'b00011100000111001110;
assign s_5[8] = 20'b00011100000111001110;
assign s_5[9] = 20'b00011110000111001110;
assign s_5[10] = 20'b00001111111111001110;
assign s_5[11] = 20'b00000111111110000100;
assign s_5[12] = 20'b00000000000000000000;
assign s_5[13] = 20'b00000000000000000000;
assign s_5[14] = 20'b00000000000000000000;
assign s_6[0] = 20'b00000000000000000000;
assign s_6[1] = 20'b00000000000000000000;
assign s_6[2] = 20'b00000000111100000000;
assign s_6[3] = 20'b00000111111111110000;
assign s_6[4] = 20'b00001111111111111100;
assign s_6[5] = 20'b00001111011111111100;
assign s_6[6] = 20'b00011110001110011100;
assign s_6[7] = 20'b00011100000111011110;
assign s_6[8] = 20'b00011100000111001110;
assign s_6[9] = 20'b00011110001110011110;
assign s_6[10] = 20'b00001111111110111100;
assign s_6[11] = 20'b00000111111100111000;
assign s_6[12] = 20'b00000000000000000000;
assign s_6[13] = 20'b00000000000000000000;
assign s_6[14] = 20'b00000000000000000000;
assign s_7[0] = 20'b00000000000000000000;
assign s_7[1] = 20'b00000000000000000000;
assign s_7[2] = 20'b00000000000000000100;
assign s_7[3] = 20'b00000000000000001110;
assign s_7[4] = 20'b00001000000000001110;
assign s_7[5] = 20'b00011110000000001110;
assign s_7[6] = 20'b00001111110000001110;
assign s_7[7] = 20'b00000111111100001110;
assign s_7[8] = 20'b00000001111111001110;
assign s_7[9] = 20'b00000000011111111110;
assign s_7[10] = 20'b00000000000111111110;
assign s_7[11] = 20'b00000000000001111110;
assign s_7[12] = 20'b00000000000000000000;
assign s_7[13] = 20'b00000000000000000000;
assign s_7[14] = 20'b00000000000000000000;
assign s_8[0] = 20'b00000000000000000000;
assign s_8[1] = 20'b00000000000000000000;
assign s_8[2] = 20'b00000111110000110000;
assign s_8[3] = 20'b00001111111011111000;
assign s_8[4] = 20'b00001111111111111100;
assign s_8[5] = 20'b00011100001111011110;
assign s_8[6] = 20'b00011100001111001110;
assign s_8[7] = 20'b00011100001110001110;
assign s_8[8] = 20'b00011100001111001110;
assign s_8[9] = 20'b00011100001111011110;
assign s_8[10] = 20'b00011111111111111100;
assign s_8[11] = 20'b00001111111011111000;
assign s_8[12] = 20'b00000000000000000000;
assign s_8[13] = 20'b00000000000000000000;
assign s_8[14] = 20'b00000000000000000000;
assign s_9[0] = 20'b00000000000000000000;
assign s_9[1] = 20'b00000000000000000000;
assign s_9[2] = 20'b00000010001111110000;
assign s_9[3] = 20'b00001111001111111000;
assign s_9[4] = 20'b00001110111111111100;
assign s_9[5] = 20'b00011100111000011100;
assign s_9[6] = 20'b00011100111000011110;
assign s_9[7] = 20'b00011100111000001110;
assign s_9[8] = 20'b00011110111000011100;
assign s_9[9] = 20'b00001111111111111100;
assign s_9[10] = 20'b00001111111111111100;
assign s_9[11] = 20'b00000011111111111000;
assign s_9[12] = 20'b00000000000000000000;
assign s_9[13] = 20'b00000000000000000000;
assign s_9[14] = 20'b00000000000000000000;

	
endmodule // VGA_LED_Emulator


	
	//
	
	/* 
    *    0 1 2 3 4 5 6 7
    * 
    * 0  afa a a a ab
    * 1  f         b
    * 2  f         b
    * 3  f         b
    * 4  f         b
    * 5  f         b
    * 6 efgggggggggbc 
    * 7  e         c
    * 8  e         c
    * 9  e         c
    * 10 e         c
    * 11 e         c
    * 12 edddddddddc
    * 13
    * 14             h
    * 15
    *
    * 640 x 480
    * 
    * Each seven-segment "pixel" is 8x8: 64 pixels across, 512 pixels for
    * 8 characters being displayed
    * 64 + 512 + 64 = 640  Start in column 64, end in column 576
    * 
    * 128 pixels high: start at row 128, end at row 256
    * 128 + 128 + 224 = 480  Start in row 128
    */

 /*  logic 			     inChar; // In any character

   assign inChar = (vcount[9:7] == 3'd1) &
		   (hcount[10:7] != 4'd0 & hcount[10:7] != 4'd9);
   
   logic [2:0] 			     charx; // Coordinate within the 8x16 char
   logic [3:0] 			     chary;

   assign charx = hcount[6:4];
   assign chary = vcount[6:3];

   logic horizBar, leftCol, rightCol, topCol, botCol; // Parts of the disp.

   assign horizBar = !(charx[2:1] == 2'b11);  // When in any horizontal bar
   assign leftCol  = (charx == 3'd0);         // When in left column
   assign rightCol = (charx == 3'd5);         // When in right column
   assign topCol   = !chary[3] & !(chary[2:0] == 3'd7); // Top columns
   assign botCol   = (chary >= 4'd6) & (chary <= 4'd12); // Bottom columns

   logic [7:0] segment; // True when in each segment
   assign segment[0] = horizBar & (chary == 4'd 0);
   assign segment[1] = rightCol & topCol;
   assign segment[2] = rightCol & botCol;
   assign segment[3] = horizBar & (chary == 4'd 12);
   assign segment[4] = leftCol & botCol;
   assign segment[5] = leftCol & topCol;
   assign segment[6] = horizBar & (chary == 4'd 6);
   assign segment[7] = (charx == 3'd6) & (chary == 4'd14);

   logic [2:0] column; // Being displayed
   assign column = hcount[9:7];
   
   logic [7:0] curSegs;
   assign curSegs = column == 3'd1 ? hex0 :
		    column == 3'd2 ? hex1 :
		    column == 3'd3 ? hex2 :
		    column == 3'd4 ? hex3 :
		    column == 3'd5 ? hex4 :
		    column == 3'd6 ? hex5 :
		    column == 3'd7 ? hex6 :
		    hex7;
   
   always_comb begin
      {VGA_R, VGA_G, VGA_B} = {8'h0, 8'h0, 8'h0}; // Black
      if (inChar)
	if ( |(curSegs & segment) ) // In any active segment?
	  {VGA_R, VGA_G, VGA_B} = {8'hff, 8'h00, 8'h00}; // Red
	else if ( |segment )        // In any (inactive) segment?
	  {VGA_R, VGA_G, VGA_B} = {8'h20, 8'h20, 8'h20}; // Dark Gray
   end  
 */  
 //
 //
 /*
 assign em4_pos1 = ((hcount[10:1] < 60)&&(hcount[10:1] > 50)&&(vcount[9:0] > 390)&&(vcount[9:0] < 410));
	assign em4_pos2 = ((hcount[10:1] < 70)&&(hcount[10:1] > 60)&&(vcount[9:0] > 390)&&(vcount[9:0] < 410));
	assign em4_pos3 = ((hcount[10:1] < 80)&&(hcount[10:1] > 70)&&(vcount[9:0] > 390)&&(vcount[9:0] < 410));
	assign em4_pos4 = ((hcount[10:1] < 90)&&(hcount[10:1] > 80)&&(vcount[9:0] > 390)&&(vcount[9:0] < 410));
	assign em4_pos5 = ((hcount[10:1] < 100)&&(hcount[10:1] > 100)&&(vcount[9:0] > 390)&&(vcount[9:0] < 410));
	assign em4_pos6 = ((hcount[10:1] < 110)&&(hcount[10:1] > 110)&&(vcount[9:0] > 390)&&(vcount[9:0] < 410));
	assign em4_pos7 = ((hcount[10:1] < 120)&&(hcount[10:1] > 120)&&(vcount[9:0] > 390)&&(vcount[9:0] < 410));
	assign em4_pos8 = ((hcount[10:1] < 130)&&(hcount[10:1] > 130)&&(vcount[9:0] > 390)&&(vcount[9:0] < 410));
	assign em4_pos9 = ((hcount[10:1] < 140)&&(hcount[10:1] > 140)&&(vcount[9:0] > 390)&&(vcount[9:0] < 410));
	assign em4_pos10 = ((hcount[10:1] < 150)&&(hcount[10:1] > 150)&&(vcount[9:0] > 390)&&(vcount[9:0] < 410));
	assign em4_pos11 = ((hcount[10:1] < 160)&&(hcount[10:1] > 160)&&(vcount[9:0] > 390)&&(vcount[9:0] < 410));
	assign em4_pos12 = ((hcount[10:1] < 170)&&(hcount[10:1] > 170)&&(vcount[9:0] > 390)&&(vcount[9:0] < 410));
	
	assign em32_pos1 = ((hcount[10:1] < 220)&&(hcount[10:1] > 210)&&(vcount[9:0] > 360)&&(vcount[9:0] < 380));
	assign em32_pos2 = ((hcount[10:1] < 230)&&(hcount[10:1] > 220)&&(vcount[9:0] > 360)&&(vcount[9:0] < 380));
	assign em32_pos3 = ((hcount[10:1] < 240)&&(hcount[10:1] > 230)&&(vcount[9:0] > 360)&&(vcount[9:0] < 380));
	assign em32_pos4 = ((hcount[10:1] < 250)&&(hcount[10:1] > 240)&&(vcount[9:0] > 360)&&(vcount[9:0] < 380));
	assign em32_pos5 = ((hcount[10:1] < 260)&&(hcount[10:1] > 250)&&(vcount[9:0] > 360)&&(vcount[9:0] < 380));
	assign em32_pos6 = ((hcount[10:1] < 270)&&(hcount[10:1] > 260)&&(vcount[9:0] > 360)&&(vcount[9:0] < 380));
	assign em32_pos7 = ((hcount[10:1] < 280)&&(hcount[10:1] > 270)&&(vcount[9:0] > 360)&&(vcount[9:0] < 380));
	assign em32_pos8 = ((hcount[10:1] < 290)&&(hcount[10:1] > 280)&&(vcount[9:0] > 360)&&(vcount[9:0] < 380));
	assign em32_pos9 = ((hcount[10:1] < 300)&&(hcount[10:1] > 290)&&(vcount[9:0] > 360)&&(vcount[9:0] < 380));
	assign em32_pos10 = ((hcount[10:1] < 310)&&(hcount[10:1] > 300)&&(vcount[9:0] > 360)&&(vcount[9:0] < 380));
	assign em32_pos11 = ((hcount[10:1] < 320)&&(hcount[10:1] > 310)&&(vcount[9:0] > 360)&&(vcount[9:0] < 380));
	assign em32_pos12 = ((hcount[10:1] < 330)&&(hcount[10:1] > 320)&&(vcount[9:0] > 360)&&(vcount[9:0] < 380));
	assign em32_pos13 = ((hcount[10:1] < 340)&&(hcount[10:1] > 330)&&(vcount[9:0] > 360)&&(vcount[9:0] < 380));
	
	assign em128_pos1 = ((hcount[10:1] < 220)&&(hcount[10:1] > 210)&&(vcount[9:0] > 390)&&(vcount[9:0] < 410));
	assign em128_pos2 = ((hcount[10:1] < 230)&&(hcount[10:1] > 220)&&(vcount[9:0] > 390)&&(vcount[9:0] < 410));
	assign em128_pos3 = ((hcount[10:1] < 240)&&(hcount[10:1] > 230)&&(vcount[9:0] > 390)&&(vcount[9:0] < 410));
	assign em128_pos4 = ((hcount[10:1] < 250)&&(hcount[10:1] > 240)&&(vcount[9:0] > 390)&&(vcount[9:0] < 410));
	assign em128_pos5 = ((hcount[10:1] < 260)&&(hcount[10:1] > 250)&&(vcount[9:0] > 390)&&(vcount[9:0] < 410));
	assign em128_pos6 = ((hcount[10:1] < 270)&&(hcount[10:1] > 260)&&(vcount[9:0] > 390)&&(vcount[9:0] < 410));
	assign em128_pos7 = ((hcount[10:1] < 280)&&(hcount[10:1] > 270)&&(vcount[9:0] > 390)&&(vcount[9:0] < 410));
	assign em128_pos8 = ((hcount[10:1] < 290)&&(hcount[10:1] > 280)&&(vcount[9:0] > 390)&&(vcount[9:0] < 410));
	assign em128_pos9 = ((hcount[10:1] < 300)&&(hcount[10:1] > 290)&&(vcount[9:0] > 390)&&(vcount[9:0] < 410));
	assign em128_pos10 = ((hcount[10:1] < 310)&&(hcount[10:1] > 300)&&(vcount[9:0] > 390)&&(vcount[9:0] < 410));
	assign em128_pos11 = ((hcount[10:1] < 320)&&(hcount[10:1] > 310)&&(vcount[9:0] > 390)&&(vcount[9:0] < 410));
	assign em128_pos12 = ((hcount[10:1] < 330)&&(hcount[10:1] > 320)&&(vcount[9:0] > 390)&&(vcount[9:0] < 410));
	assign em128_pos13 = ((hcount[10:1] < 340)&&(hcount[10:1] > 330)&&(vcount[9:0] > 390)&&(vcount[9:0] < 410));
	assign em128_pos14 = ((hcount[10:1] < 350)&&(hcount[10:1] > 340)&&(vcount[9:0] > 390)&&(vcount[9:0] < 410));
	
		logic write_okay_e4p1;
	logic write_okay_e4p2;
	logic write_okay_e4p3;
	logic write_okay_e4p4;
	logic write_okay_e4p5;
	logic write_okay_e4p6;
	logic write_okay_e4p7;
	logic write_okay_e4p8;
	logic write_okay_e4p9;
	logic write_okay_e4p10;
	logic write_okay_e4p11;
	logic write_okay_e4p12;
	
	logic write_okay_e32p1;
	logic write_okay_e32p2;
	logic write_okay_e32p3;
	logic write_okay_e32p4;
	logic write_okay_e32p5;
	logic write_okay_e32p6;
	logic write_okay_e32p7;
	logic write_okay_e32p8;
	logic write_okay_e32p9;
	logic write_okay_e32p10;
	logic write_okay_e32p11;
	logic write_okay_e32p12;
	logic write_okay_e32p13;

	logic write_okay_e128p1;
	logic write_okay_e128p2;
	logic write_okay_e128p3;
	logic write_okay_e128p4;
	logic write_okay_e128p5;
	logic write_okay_e128p6;
	logic write_okay_e128p7;
	logic write_okay_e128p8;
	logic write_okay_e128p9;
	logic write_okay_e128p10;
	logic write_okay_e128p11;
	logic write_okay_e128p12;
	logic write_okay_e128p13;
	logic write_okay_e128p14;
	
	assign a4x1 = 50;
	assign a4y1 = 390;
	assign a4x2 = 75;
	assign a4y2 = 390;
	assign a4x3 = 90;
	assign a4y3 = 390;	
	assign a4x4 = 105;
	assign a4y4 = 390;	
	assign a4x5 = 120;
	assign a4y5 = 390;	
	assign a4x6 = 135;
	assign a4y6 = 390;
	assign a4x7 = 150;
	assign a4y7 = 390;
	assign a4x8 = 165;
	assign a4y8 = 390;
	assign a4x9 = 180;
	assign a4y9 = 390;
	assign a4x10 = 195;
	assign a4y10 = 390;
	assign a4x11 = 210;
	assign a4y11 = 390;
	assign a4x12 = 225;
	assign a4y12 = 390;
	
	assign a32x1 = 210;
	assign a32y1 = 360;
	assign a32x2 = 220;
	assign a32y2 = 360;
	assign a32x3 = 230;
	assign a32y3 = 360;	
	assign a32x4 = 240;
	assign a32y4 = 360;	
	assign a32x5 = 250;
	assign a32y5 = 360;	
	assign a32x6 = 260;
	assign a32y6 = 360;
	assign a32x7 = 270;
	assign a32y7 = 360;
	assign a32x8 = 280;
	assign a32y8 = 360;
	assign a32x9 = 290;
	assign a32y9 = 360;
	assign a32x10 = 300;
	assign a32y10 = 360;
	assign a32x11 = 310;
	assign a32y11 = 360;
	assign a32x12 = 320;
	assign a32y12 = 360;
	assign a32x13= 330;
	assign a32y13 = 360;
	
	assign a128x1 = 210;
	assign a128y1 = 390;
	assign a128x2 = 220;
	assign a128y2 = 390;
	assign a128x3 = 230;
	assign a128y3 = 390;	
	assign a128x4 = 240;
	assign a128y4 = 390;	
	assign a128x5 = 250;
	assign a128y5 = 390;	
	assign a128x6 = 260;
	assign a128y6 = 390;
	assign a128x7 = 270;
	assign a128y7 = 390;
	assign a128x8 = 280;
	assign a128y8 = 390;
	assign a128x9 = 290;
	assign a128y9 = 390;
	assign a128x10 = 300;
	assign a128y10 = 390;
	assign a128x11 = 310;
	assign a128y11 = 390;
	assign a128x12 = 320;
	assign a128y12 = 390;
	assign a128x13= 330;
	assign a128y13 = 390;
	assign a128x13= 330;
	assign a128y13 = 390;

assign write_okay_e4p1 = ((S_E[hcount[10:1] - a4x1][vcount[9:0] - a4y1]));
	assign write_okay_e4p2 = ((S_M[hcount[10:1] - a4x2][vcount[9:0] - a4y2]));
	assign write_okay_e4p3 = ((S_A[hcount[10:1] - a4x3][vcount[9:0] - a4y3]));
	assign write_okay_e4p4 = ((S_4[hcount[10:1] - a4x4][vcount[9:0] - a4y4]));
	assign write_okay_e4p5 = ((S_colon[hcount[10:1] - a4x5][vcount[9:0] - a4y5]));

	//assign write_okay_e4p10;

	
	assign write_okay_e32p1  = ((S_E[hcount[10:1] - a32x1][vcount[9:0] - a32y1]));
	assign write_okay_e32p2 = ((S_M[hcount[10:1] - a32x2][vcount[9:0] - a32y2]));
	assign write_okay_e32p3 = ((S_A[hcount[10:1] - a32x3][vcount[9:0] - a32y3]));
	assign write_okay_e32p4 = ((S_3[hcount[10:1] - a32x4][vcount[9:0] - a32y4]));
	assign write_okay_e32p5 = ((S_2[hcount[10:1] - a32x5][vcount[9:0] - a32y5]));
	assign write_okay_e32p6 = ((S_colon[hcount[10:1] - a32x6][vcount[9:0] - a32y6]));

	//assign write_okay_e32p11;


	assign write_okay_e128p1 = ((S_E[hcount[10:1] - a128x1][vcount[9:0] - a128y1]));
	assign write_okay_e128p2 = ((S_M[hcount[10:1] - a128x2][vcount[9:0] - a128y2]));
	assign write_okay_e128p3 = ((S_A[hcount[10:1] - a128x3][vcount[9:0] - a128y3]));
	assign write_okay_e128p4 = ((S_1[hcount[10:1] - a128x4][vcount[9:0] - a128y4]));
	assign write_okay_e128p5 = ((S_2[hcount[10:1] - a128x5][vcount[9:0] - a128y5]));
	assign write_okay_e128p6 = ((S_8[hcount[10:1] - a128x6][vcount[9:0] - a128y6]));
	assign write_okay_e128p7 = ((S_colon[hcount[10:1] - a128x7][vcount[9:0] - a128y7]));

	//assign write_okay_e128p12 
*/



// For time
/*
	logic write_okay_t1;
	logic write_okay_t2;
	logic write_okay_t3;
	logic write_okay_t4;
	logic write_okay_t5;
	
	assign write_okay_t3 = ((s_colon[hcount[10:1] - tx3][vcount[9:0] - ty3]));
		
	always_comb begin
		if (time_a == 0) begin
			write_okay_t1 = ((s_O[hcount[10:1] - tx1][vcount[9:0] - ty1]));
		end
		else if (time_a == 1) begin
			write_okay_t1 = ((s_1[hcount[10:1] - tx1][vcount[9:0] - ty1]));
		end	
		else if (time_a == 2) begin
			write_okay_t1 = ((s_2[hcount[10:1] - tx1][vcount[9:0] - ty1]));
		end
		else begin
			write_okay_t1 = ((s_blank[hcount[10:1] - tx1][vcount[9:0] - ty1]));
		end
	end		
		
	always_comb begin
		if (time_c == 0) begin
			write_okay_t4 = ((s_O[hcount[10:1] - tx4][vcount[9:0] - ty4]));
		end
		else if (time_c == 1) begin
			write_okay_t4 = ((s_1[hcount[10:1] - tx4][vcount[9:0] - ty4]));
		end
		else if (time_c == 2) begin
			write_okay_t4 = ((s_2[hcount[10:1] - tx4][vcount[9:0] - ty4]));
		end
		else if (time_c == 3) begin
			write_okay_t4 = ((s_3[hcount[10:1] - tx4][vcount[9:0] - ty4]));
		end
		else if (time_c == 4) begin
			write_okay_t4 = ((s_4[hcount[10:1] - tx4][vcount[9:0] - ty4]));
		end
		else if (time_c == 5) begin
			write_okay_t4 = ((s_5[hcount[10:1] - tx4][vcount[9:0] - ty4]));
		end	
		else begin
			write_okay_t4 = ((s_blank[hcount[10:1] - tx4][vcount[9:0] - ty4]));
		end
	end	
		
	always_comb begin
		if (time_d == 0) begin
			write_okay_t5 = ((s_O[hcount[10:1] - tx5][vcount[9:0] - ty5]));
		end
		else if (time_d == 1) begin
			write_okay_t5 = ((s_1[hcount[10:1] - tx5][vcount[9:0] - ty5]));
		end
		else if (time_d == 2) begin
			write_okay_t5 = ((s_2[hcount[10:1] - tx5][vcount[9:0] - ty5]));
		end
		else if (time_d == 3) begin
			write_okay_t5 = ((s_3[hcount[10:1] - tx5][vcount[9:0] - ty5]));
		end
		else if (time_d == 4) begin
			write_okay_t5 = ((s_4[hcount[10:1] - tx5][vcount[9:0] - ty5]));
		end
		else if (time_d == 5) begin
			write_okay_t5 = ((s_5[hcount[10:1] - tx5][vcount[9:0] - ty5]));
		end
		else if (time_d == 6) begin
			write_okay_t5 = ((s_6[hcount[10:1] - tx5][vcount[9:0] - ty5]));
		end
		else if (time_d == 7) begin
			write_okay_t5 = ((s_7[hcount[10:1] - tx5][vcount[9:0] - ty5]));
		end
		else if (time_d == 8) begin
			write_okay_t5 = ((s_8[hcount[10:1] - tx5][vcount[9:0] - ty5]));
		end		
		else if (time_d == 9) begin
			write_okay_t5 = ((s_9[hcount[10:1] - tx5][vcount[9:0] - ty5]));
		end	
		else begin
			write_okay_t5 = ((s_blank[hcount[10:1] - tx5][vcount[9:0] - ty5]));
		end
	end
	
	always_comb begin
		if (time_b == 0) begin
			write_okay_t2 = ((s_O[hcount[10:1] - tx2][vcount[9:0] - ty2]));
		end
		else if (time_b == 1) begin
			write_okay_t2 = ((s_1[hcount[10:1] - tx2][vcount[9:0] - ty2]));
		end
		else if (time_b == 2) begin
			write_okay_t2 = ((s_2[hcount[10:1] - tx2][vcount[9:0] - ty2]));
		end
		else if (time_b == 3) begin
			write_okay_t2 = ((s_3[hcount[10:1] - tx2][vcount[9:0] - ty2]));
		end
		else if (time_b == 4) begin
			write_okay_t2 = ((s_4[hcount[10:1] - tx2][vcount[9:0] - ty2]));
		end
		else if (time_b == 5) begin
			write_okay_t2 = ((s_5[hcount[10:1] - tx2][vcount[9:0] - ty2]));
		end
		else if (time_b == 6) begin
			write_okay_t2 = ((s_6[hcount[10:1] - tx2][vcount[9:0] - ty2]));
		end
		else if (time_b == 7) begin
			write_okay_t2 = ((s_7[hcount[10:1] - tx2][vcount[9:0] - ty2]));
		end
		else if (time_b == 8) begin
			write_okay_t2 = ((s_8[hcount[10:1] - tx2][vcount[9:0] - ty2]));
		end		
		else if (time_b == 9) begin
			write_okay_t2 = ((s_9[hcount[10:1] - tx2][vcount[9:0] - ty2]));
		end	
		else begin
			write_okay_t2 = ((s_blank[hcount[10:1] - tx2][vcount[9:0] - ty2]));
		end
	end
	
	logic time_pos1;
	logic time_pos2;
	logic time_pos3;
	logic time_pos4;
	logic time_pos5;
	
	assign time_pos1 = ((hcount[10:1] <= 335)&&(hcount[10:1] >= 320)&&(vcount[9:0] >= 360)&&(vcount[9:0] < 380));
	assign time_pos2 = ((hcount[10:1] <= 355)&&(hcount[10:1] >= 340)&&(vcount[9:0] >= 360)&&(vcount[9:0] < 380));
	assign time_pos3 = ((hcount[10:1] <= 375)&&(hcount[10:1] >= 360)&&(vcount[9:0] >= 360)&&(vcount[9:0] < 380));
	assign time_pos4 = ((hcount[10:1] <= 395)&&(hcount[10:1] >= 380)&&(vcount[9:0] >= 360)&&(vcount[9:0] < 380));
	assign time_pos5 = ((hcount[10:1] <= 415)&&(hcount[10:1] >= 400)&&(vcount[9:0] >= 360)&&(vcount[9:0] < 380));
	
	logic [9:0] tx1;
	logic [9:0] ty1;
	logic [9:0] tx2;
	logic [9:0] ty2;
	logic [9:0] tx3;
	logic [9:0] ty3;
	logic [9:0] tx4;
	logic [9:0] ty4;
	logic [9:0] tx5;
	logic [9:0] ty5;
	
	assign tx1 = 320;
	assign tx2 = 340;
	assign tx3 = 360;
	assign tx4 = 380;
	assign tx5 = 400;
	assign ty1 = 360;
	assign ty2 = 360;
	assign ty3 = 360;
	assign ty4 = 360;
	assign ty5 = 360;
	*/
