/*
 * Userspace program that communicates with the led_vga device driver
 * primarily through ioctls
 *
 */

#include <stdio.h>
#include "vga_led.h"
#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <string.h>
#include <unistd.h>
#include <stdlib.h>

#define LINES 14822
#define LINES 10
#define CMD_KERNEL_DENSITY_ESTIMATION 1
#define CMD_LAPLACE 2
#define CMD_TRAIN_NORMAL 4
#define CMD_TRAIN_LEAKAGE 12
#define CMD_SORT 0

int vga_led_fd;

/* Read and print the maxmin values */
unsigned short read_data() {
    vga_led_arg_t vla;
    if (ioctl(vga_led_fd, VGA_LED_READ, &vla)) {
        perror("ioctl(VGA_LED_READ) failed\n");
        return;
    }
    return vla.data;
}

/* Write max and min values to vga */
void write_maxmin(const unsigned short maxmin, unsigned char command)
{
    vga_led_arg_t vla;
    vla.data = maxmin;
    vla.addr = command;
    if (ioctl(vga_led_fd, VGA_LED_WRITE, &vla)) {
        perror("ioctl(VGA_LED_WRITE) failed");
        return;
    }
    //printf("fpga comm success\n");
}

/* parse file f and load vals into minimum and maximum arrays */
void load_csv(char *f, unsigned short maxminarr[]) {
    // open file
    char buf[1024];
    FILE *file = fopen(f,"r");
    if (!file) {
      fprintf(stderr, "Can't open file.\n");
      exit(EXIT_FAILURE);
    }

    // loop through file line by line
    int i = 0;
    while (fgets(buf, sizeof buf, file) != NULL) {
        int max;
        int min;

        // scan integers into max and min
        if (sscanf(buf, "%d,%d", &max, &min) != 2) {
            // check for failure
            fprintf(stderr, "sscanf failed.");
            exit(EXIT_FAILURE);
        }

        //printf("%i and %i\n", max,min);
        
        // represent max and min in binary by concatenating
        unsigned short maxmin = (unsigned short) max;
        maxmin = maxmin << 8;
        maxmin = maxmin + min;

        maxminarr[i] = maxmin;
        i++;
    }

    fclose(file);
}

void print_as_binary(unsigned short x)
{
    static char b[9];
    b[0] = '\0';

    while (x) {
        if (x & 1)
            strcat(b, "1");
        else
            strcat(b, "0");

        x >>= 1;
    }
    printf("%s\n",b);
}

const char * byte_to_binary(unsigned short x)
{
    static char b[9];
    b[0] = '\0';

    int z;
    for (z = 128; z > 0; z >>= 1)
    {
        strcat(b, ((x & z) == z) ? "1" : "0");
    }

    printf("%s\n",b);
}

int main()
{
    vga_led_arg_t vla;
    int i;
    static const char device[] = "/dev/vga_led";

    //printf("VGA LED Userspace program started\n");

    if ( (vga_led_fd = open(device, O_RDWR)) == -1) {
        fprintf(stderr, "could not open %s\n", device);
        return -1;
    }

    // extract minimum and maximum values from csv
    unsigned short maxminarr[LINES];
    // TODO change back to features.csv
    // load_csv("./features.csv", maxminarr); 
    load_csv("./features.csv", maxminarr); 

    unsigned short rdata = 0;

    // kernel density estimation
    printf("====================\n");
    printf("execute kernel density estimation\n");
    for (i = 0; i < LINES; i++) {
        //printf("\n");

        // write max and min to fpga
        write_maxmin(maxminarr[i], CMD_KERNEL_DENSITY_ESTIMATION);

        // read data from fpga until non-zero
        //printf("processing ");
        while (rdata == 0) {
            //printf(".");
            rdata = read_data();
        }
        //printf("\n");
        //printf("result received = %d\n", rdata);
    }

    // start laplace on fpga
    printf("====================\n");
    printf("execute laplace\n");
    write_maxmin(42, CMD_LAPLACE);
    rdata = 0;
    while (rdata < 8) {
        rdata = read_data();
    }

    // training
    printf("====================\n");
    printf("execute training\n");
    for (i = 0; i < LINES; i++) {
        if (i%128 == 0) {
            // update with leakage
            write_maxmin(maxminarr[i], CMD_TRAIN_LEAKAGE);
        } else {
            // update normally
            write_maxmin(maxminarr[i], CMD_TRAIN_NORMAL);
        }
        // continue training when sample is done processing
        rdata = 0;
        while (rdata < 8) {
            rdata = read_data();
        }
        print_as_binary(rdata);
        //printf("trained return val: %d\n", (rdata));
    }

    // sorting
    printf("====================\n");
    printf("execute sorting\n");
    for (i = 0; i < LINES; i++) {
        // send sort signal
        write_maxmin(maxminarr[i], CMD_SORT);
        // continue training when sample is done processing
        //rdata = 0;
        rdata = read_data();
        /*
        while (rdata < 8) {
            rdata = read_data();
        }
        */
        //printf("sorted value: %d\n", (rdata - 8));
    }

    printf("Spike Sorting userspace program terminating\n");
    return 0;
}
