/*
 * C prototype of Bresenham's line algorithm
 *
 * Requires SDL libraries
 *
 * Stephen A. Edwards
 */

#include <stdio.h>
#include <SDL/SDL.h>

#define WIDTH 640
#define HEIGHT 480
#define DEPTH 32
#define BPP 4

SDL_Surface *screen;
Uint32 color;

void plot(int x, int y) {
  if (x < 0 || x >= WIDTH || y < 0 || y >= HEIGHT) {
    fprintf(stderr,"Bad coordinate %d %d\n", x, y);
    exit(1);
  }
  *((Uint32*) screen->pixels + x + y * (screen->pitch)/BPP) = color;
}

/*

Pseudocode from Wikipedia page:

http://en.wikipedia.org/wiki/Bresenham%27s_line_algorithm

 function line(x0, y0, x1, y1)
   dx := abs(x1-x0)
   dy := abs(y1-y0) 
   if x0 < x1 then sx := 1 else sx := -1
   if y0 < y1 then sy := 1 else sy := -1
   err := dx-dy
 
   loop
     setPixel(x0,y0)
     if x0 = x1 and y0 = y1 exit loop
     e2 := 2*err
     if e2 > -dy then 
       err := err - dy
       x0 := x0 + sx
     end if
     if e2 <  dx then 
       err := err + dx
       y0 := y0 + sy 
     end if
   end loop

Probably the guy who entered this in Wikipedia

http://free.pages.at/easyfilter/bresenham.html

void plotLine(int x0, int y0, int x1, int y1)
{
  int dx =  abs(x1-x0), sx = x0<x1 ? 1 : -1;
  int dy = -abs(y1-y0), sy = y0<y1 ? 1 : -1; 
  int err = dx+dy, e2; // error value e_xy
 
  for(;;){  // loop
    setPixel(x0,y0);
    if (x0==x1 && y0==y1) break;
    e2 = 2*err;
    if (e2 >= dy) { err += dy; x0 += sx; } // e_xy+e_x > 0
    if (e2 <= dx) { err += dx; y0 += sy; } // e_xy+e_y < 0
  }
}

Trick is that the sign of e2 is either always non-negative or non-positive.

Assume |dy| < |dx|

dy is always negative; dx is always positive

err is always positive; rather than comparing error against 0,
it actually compares it against dy.

Thus, one of the "if" statements at the bottom will always fire
and effectively cancel out the extra.

 */

void line(Uint16 x0, Uint16 y0, Uint16 x1, Uint16 y1)
{
  Sint16 dx, dy;// width and height of bounding box
  Uint16 x, y; // Current point
  Sint16 err;   // Loop-carried value
  Sint16 e2;    // Temporary variable
  int right, down; // Boolean

  dx = x1 - x0;
  right = dx >= 0;
  if (!right) dx = -dx;

  dy = y1 - y0;
  down = dy >= 0;
  if (down) dy = -dy;

  err = dx + dy;
  x = x0;
  y = y0;
  
  for (;;) {
    plot(x, y);
    e2 = err << 1;
    /*    printf("dx:%d dy:%d (%d,%d) %d\n", dx, dy, x, y, e2); */
    if (x == x1 && y == y1) break;
    if (e2 > dy) {
      err += dy;
      if (right) x++; else x--;
    }
    if (e2 < dx) {
      err += dx;
      if (down) y++; else y--;
    }
  }

}

int main()
{
  SDL_Event event;

  int quit = 0;

  Uint16 x, y;

  if (SDL_Init(SDL_INIT_VIDEO) < 0) {
    fprintf(stderr, "SDL_Init failed\n");
    return 1;
  }

  if (!(screen = SDL_SetVideoMode(WIDTH, HEIGHT, DEPTH, SDL_HWSURFACE))) {
    fprintf(stderr, "SDL_SetVideoMode failed\n");
    SDL_Quit();
    return 1;
  }

  color = SDL_MapRGB(screen->format, 255, 255, 255);

  if (SDL_MUSTLOCK(screen)) SDL_LockSurface(screen);

  /* line(0,0,7,4); */

  for (x = 0 ; x < WIDTH ; x += 10) {
    line (x, 0, 100, 130);
    line (x, HEIGHT - 1, 100, 130);
  }
  for (y = 0 ; y < HEIGHT ; y += 10) {
    line (0, y, 100, 130);
    line (WIDTH - 1, y, 100, 130);
  }

  if (SDL_MUSTLOCK(screen)) SDL_UnlockSurface(screen);

  SDL_Flip(screen);

  do {

    while (SDL_PollEvent(&event))
      switch (event.type) {
      case SDL_QUIT:
      case SDL_KEYDOWN:
	quit = 1;
	break;
      }
  } while (!quit);

  SDL_Quit();
  return 0;
}
