/*
 * Device driver for the Altera FIFO
 *
 * A Platform device implemented using the misc subsystem
 *
 * Andrea Lottarini
 * Columbia University
 *
 * References:
 * Linux source: Documentation/driver-model/platform.txt
 *               drivers/misc/arm-charlcd.c
 * http://www.linuxforu.com/tag/linux-device-drivers/
 * http://free-electrons.com/docs/
 *
 * "make" to build
 * insmod fifo0.ko
 *
 * Check code style with
 * checkpatch.pl --file --no-tree fifo_data0.c
 */

#include <linux/module.h>
#include <linux/init.h>
#include <linux/errno.h>
#include <linux/version.h>
#include <linux/kernel.h>
#include <linux/platform_device.h>
#include <linux/miscdevice.h>
#include <linux/slab.h>
#include <linux/io.h>
#include <linux/of.h>
#include <linux/of_address.h>
#include <linux/fs.h>
#include <linux/uaccess.h>
#include "fifo.h"

#define DRIVER_NAME "fifo0"

/*
 * Information about our device
 */
struct fifo_dev {
	struct resource res; /* Resource: our registers */
	void __iomem *virtbase; /* Where registers can be accessed in memory */

	struct resource status_res; /* register where I can read the status */
	void __iomem *status_virtbase; /* Where this register can be accessed in memory */

} dev;


/*
 * Handle ioctl() calls from userspace:
 * This believes whatever the user passes without checking it
 */
static long fifo_ioctl(struct file *f, unsigned int cmd, unsigned long arg)
{
	int i, specs, fill, to_write, to_read;
	opcode* op = (opcode*) arg;

	switch (cmd) {
	case FIFO_WRITE_DATA:
		// first check that the fifo is not full
		fill = ioread32(dev.status_virtbase);
		
		to_write = MIN(FIFO_SIZE - fill , op->length);
		
		printk("Writer Driver 0 - I received an order for %d writes and I can do %d\n",op->length, to_write);
		if ( to_write > 1 ){
			printk("Writer Driver 0 - writing specs %d\n",START_PACKET_CHANNEL0);
			iowrite32( START_PACKET_CHANNEL0, dev.virtbase+4 );
			/* trusting the user buffer to avoid coping that too */
			for ( i = 0 ; i < to_write ; i++ ){
				if ( i == (to_write - 1) ){ /* write the end packet flag before wrting the last int*/
					
					if (op->done && to_write == op->length){ 		/* that was ALSO the last transfer for this stream ad I wrote it all down*/
						printk("Writer Driver 0 - writing specs %d\n",DONE_END_PACKET_CHANNEL0);
						iowrite32(DONE_END_PACKET_CHANNEL0, dev.virtbase+4);
					}else{
						printk("Writer Driver 0 - writing specs %d\n",END_PACKET_CHANNEL0);
						iowrite32(END_PACKET_CHANNEL0, dev.virtbase+4);
					}
				}
				printk("Writer Driver 0 - writing %d\n",op->buf[i]);
				iowrite32( op->buf[i], dev.virtbase);
			}
		}else{
			if ( to_write > 0){
				if (op->done && to_write == op->length){
					iowrite32(DONE_SINGLE_PACKET_CHANNEL0,dev.virtbase+4);
				}else{
					iowrite32(SINGLE_PACKET_CHANNEL0,dev.virtbase+4);
				}
				iowrite32(op->buf[0], dev.virtbase);
			}
		}

		/* write back in the op struct how many int were actually sent */
		op->length = to_write;

		break;

	case FIFO_READ_DATA:

		fill = ioread32(dev.status_virtbase);
		to_read = MIN(fill, op->length);

		printk("Reader Driver 0 - I received an order for %d reads but there are %d in the fifo\n",op->length,fill);
		if (fill > 0 ){
			/* trusting the user buffer to avoid coping that too */
			for ( i = 0 ; i < to_read ; i++ ){
				op->buf[i] = ioread32( dev.virtbase );
			}
			/* write back in the op struct how many int were actually read */				
			op->length = to_read;

			/* check if it was the last one */
			specs = ioread32(dev.virtbase+4);
			printk("Reader Driver 0 - these are the specs: %d\n",specs);
			if (specs & DONE_MASK){
				op->done = 1;
			}else{
				op->done = 0;
			}
		}else{
			op->length = 0;
			op->done = 0;
		}
		break;

	case FIFO_READ_STATUS:
		fill = ioread32(dev.status_virtbase);
		if (copy_to_user( (int *) arg, &fill,
						  sizeof(int)) )
			return -EACCES;
		break;

	default:
		return -EINVAL;
	}

	return 0;
}

/* The operations our device knows how to do */
static const struct file_operations fifo_fops = {
	.owner		= THIS_MODULE,
	.unlocked_ioctl = fifo_ioctl,
};

/* Information about our device for the "misc" framework -- like a char dev */
static struct miscdevice fifo_misc_device = {
	.minor		= MISC_DYNAMIC_MINOR,
	.name		= DRIVER_NAME,
	.fops		= &fifo_fops,
};

/*
 * Initialization code: get resources (registers) and display
 * a welcome message
 */
static int __init fifo_probe(struct platform_device *pdev)
{

	int ret;

	pr_info(DRIVER_NAME ": probe\n");

	/* Register ourselves as a misc device: creates /dev/fifo */
	ret = misc_register(&fifo_misc_device);

	/* Get the address of data registers from the device tree */
	ret = of_address_to_resource(pdev->dev.of_node, 0, &dev.res);
	if (ret) {
		ret = -ENOENT;
		goto out_deregister;
	}

	printk("DEVICE DATA START %x END %x \n",dev.res.start,dev.res.end);

	/* Make sure we can use these registers */
	if (request_mem_region(dev.res.start, resource_size(&dev.res),
			       DRIVER_NAME) == NULL) {
		ret = -EBUSY;
		goto out_deregister;
	}

	/* Get the address of status registers from the device tree */
	ret = of_address_to_resource(pdev->dev.of_node, 1, &dev.status_res);
	if (ret) {
		ret = -ENOENT;
		goto out_deregister;
	}

	printk("DEVICE STATUS START %x END %x \n",dev.status_res.start,dev.status_res.end);

	/* Make sure we can use these registers */
	if (request_mem_region(dev.status_res.start, resource_size(&dev.status_res),
			       DRIVER_NAME) == NULL) {
		ret = -EBUSY;
		goto out_deregister;
	}

	/* Arrange access to data registers */
	dev.virtbase = of_iomap(pdev->dev.of_node, 0);
	if (dev.virtbase == NULL) {
		ret = -ENOMEM;
		goto out_release_mem_region2;
	}

	printk("VIRTUAL ADDRESS OF DATA FIFO0: %x \n",(unsigned int) dev.virtbase);

	/* Arrange access to status registers */
	dev.status_virtbase = of_iomap(pdev->dev.of_node, 1);
	if (dev.status_virtbase == NULL) {
		ret = -ENOMEM;
		goto out_release_mem_region1;
	}

	printk("VIRTUAL ADDRESS OF STATUS FIFO0: %x \n",(unsigned int) dev.status_virtbase);

	return 0;

out_release_mem_region1:
	release_mem_region(dev.res.start, resource_size(&dev.res));
out_release_mem_region2:
	release_mem_region(dev.status_res.start, resource_size(&dev.status_res));
out_deregister:
	misc_deregister(&fifo_misc_device);
	return ret;
}

/* Clean-up code: release resources */
static int fifo_remove(struct platform_device *pdev)
{
	iounmap(dev.virtbase);
	iounmap(dev.status_virtbase);
	release_mem_region(dev.res.start, resource_size(&dev.res));
	release_mem_region(dev.status_res.start, resource_size(&dev.status_res));
	misc_deregister(&fifo_misc_device);
	return 0;
}

/* Which "compatible" string(s) to search for in the Device Tree */
#ifdef CONFIG_OF
static const struct of_device_id fifo_of_match[] = {
	{ .compatible = "altr,fifo0" },
	{},
};
MODULE_DEVICE_TABLE(of, fifo_of_match);
#endif

/* Information for registering ourselves as a "platform" driver */
static struct platform_driver fifo_driver = {
	.driver	= {
		.name	= DRIVER_NAME,
		.owner	= THIS_MODULE,
		.of_match_table = of_match_ptr(fifo_of_match),
	},
	.remove	= __exit_p(fifo_remove),
};

/* Called when the module is loaded: set things up */
static int __init fifo_init(void)
{
	pr_info(DRIVER_NAME ": init\n");
	return platform_driver_probe(&fifo_driver, fifo_probe);
}

/* Called when the module is unloaded: release resources */
static void __exit fifo_exit(void)
{
	platform_driver_unregister(&fifo_driver);
	pr_info(DRIVER_NAME ": exit\n");
}

module_init(fifo_init);
module_exit(fifo_exit);

MODULE_LICENSE("GPL");
MODULE_AUTHOR("Andrea Lottarini, Columbia University");
MODULE_DESCRIPTION("Altera FIFO driver");
