/* sorter base tile*/

//`include "interfaces.sv"

/* Working */
module sort_tile #(
				   parameter COLUMN_WIDTH = 32,
				   parameter NSWAPS = 2,
				   parameter TOP_ROW = 1,
				   parameter BOTTOM_ROW = 0,
				   parameter CTRL_WIDTH = 2
				   )

   (
	//data in AvalonST
	AvalonLink.Sink ava_in,
	// input logic [COLUMN_WIDTH-1:0] 	col_1_i,
	// input logic 					valid_1_i,
	// output logic 					ready_1_o,
	// input logic 					done_1_i,
	// input logic 					srt_pck_1_i,
	// input logic 					end_pck_1_i,
	
	//ctrl in
	CtrLink.Sink ctrl_in,
	// input logic [CTRL_WIDTH-1:0] 	col_2_i,
	// input logic 					valid_2_i,
	// output logic 					ready_2_o,
	   
	//data out AvalonST
	AvalonLink.Source ava_out,
	// output logic [COLUMN_WIDTH-1:0] col_1_o, 
	// output logic 					valid_1_o, 
	// input logic 					ready_1_i, 
	// output logic 					done_1_o, 
	// output logic 					srt_pck_1_o,
	// output logic 					end_pck_1_o,

	//ctrl out AvalonST
	CtrLink.Source ctrl_out,
	// output logic [CTRL_WIDTH-1:0] 	col_2_o, 
	// output logic 					valid_2_o,
	// input logic 					ready_2_i,

	// output logic 					done_2_o, 
	// output logic 					srt_pck_2_o,
	// output logic 					end_pck_2_o,

	
	/* std signals */
	input logic 					clk,
	input logic 					reset_i // reset everything
	);
   
   logic 							ready_data; //ready input to preceding tile in the conveyor
   logic 							ready_ctrl; //ready input to the slave tile
   
   logic 							done1;   

   logic 							all_rdy,conv_rdy;
   logic 							inputs_valid, inputs_done, swap, ready_i, flush_buffer;
   
   /**
	state can be:
	EMPTY,RUNNING,FLUSH,DONE
	EMPTY		: no data has arrived yet
	RUNNING		: I am receiving data and swapping it as needed
	FLUSH		: a sorted run has ended    
	DONE		: a DONE bit has been received 
	*/   
   logic [1:0] 						state;
   //used as regs in the tile
   logic [COLUMN_WIDTH-1:0] 		buffer,count;
   //wires connected to the input buffers
   logic [COLUMN_WIDTH-1:0] 		data;
   logic [CTRL_WIDTH-1:0] 			ctrl;
   logic 							data_valid,ctrl_valid;

   //partial results going out
   logic 							valid_data, valid_ctrl, ctrl_out_dk, done_o, ready_o;
   
   // buffer in1(.col_i(col_1_i), 
   // 			  .valid_i(valid_1_i), 
   // 			  .ready_i(ready_data),
   //            .col_o(data),
   // 			  .valid_o(data_valid),  
   // 			  .ready_o(ready_1_o),
   // 			  .clk(clk),
   // 			  .reset_i(reset_i),
   // 			  .start_pck_in(srt_pck_1_i),
   // 			  .end_pck_in(end_pck_1_i),
   // 			  .done_i(done_1_i),
   // 			  .done_o(done1)
   // 			  );
   buffer in1(.col_i(ava_in.data), 
			  .valid_i(ava_in.valid), 
			  .ready_i(ready_data),
              .col_o(data),
			  .valid_o(data_valid),  
			  .ready_o(ava_in.ready),
			  .clk(clk),
			  .reset_i(reset_i),
			  .start_pck_in(ava_in.srt_pck),
			  .end_pck_in(ava_in.end_pck),
			  .done_i(ava_in.done),
			  .done_o(done1)
			  );

//sorter in the top row does not need a buffer for ctrl signals
generate
   if (TOP_ROW) begin
	     	  //still connecting these outside of the module - 
   	  // just in case I can think of something to use
   	  // assign ctrl       = ctrl_in.ctrl;
   	  // assign ctrl_valid = ctrl_in.valid;
	  assign ctrl = 2'b00;
	  assign ctrl_valid = 1'b1;
	  
   	  assign ctrl_in.ready  = 1'b0;  //this is ignored in the top row   

   end
   else begin
	 buffer_ctrl in2(
					 .col_i(ctrl_in.ctrl), 
					 .valid_i(ctrl_in.valid), 
					 .ready_i(ready_ctrl),
					 .col_o(ctrl),
					 .valid_o(ctrl_valid),  
					 .ready_o(ctrl_in.ready),
					 .clk(clk),
					 .reset_i(reset_i)
					 );	  
   end
endgenerate

   initial begin
	  $display("%m TOP ROW: %d BOTTOM ROW: %d",TOP_ROW,BOTTOM_ROW);
	  
   end
   
   //out of band data in the conveyor for the done bit
   // assign srt_pck_1_o				= 1;
   // assign end_pck_1_o				= 1;
   // assign done_o				    = ( state == 2'b11 ) ? 1'b1 : 1'b0;
   assign ava_out.srt_pck				= 1;
   assign ava_out.end_pck				= 1;
   assign ava_out.done				    = ( state == 2'b11 ) ? 1'b1 : 1'b0;


   //assign valid_1_o                 = valid_data;
   assign ava_out.valid             = valid_data;
   assign ctrl_out.valid                 = valid_ctrl;
   
   //only the conveyor data matters
   assign inputs_done			= done1; 
   assign inputs_valid			= data_valid & ctrl_valid;


   //if this bit is 1 then we are just flushing the conveyor not doing any swapping nor sending commands
   assign flush_buffer          = state[1];
   // indicates if the buffer and incoming data should be swapped [ ignored when flushing ]
   assign swap		            = ctrl[0] ?  ctrl[1] : ( data > buffer );
   // indicates whether ctrl_out is a command or not [ master does not know what the slave should do ]
   assign ctrl_out_dk           = ( ctrl[0] | ( data != buffer ) );
   assign ava_out.data          = (flush_buffer) ? buffer: ( swap ? buffer : data );
   //assign col_1_o               = (flush_buffer) ? buffer: ( swap ? buffer : data );
   assign ctrl_out.ctrl          = {swap,ctrl_out_dk};

   //assign done_1_o              = done_o;
   
   //logic and of the ready bits coming in
   //assign ready_i               = ready_1_i & ready_2_i;
   	 
   //assignment of valid and ready bits
   always_comb begin
	  ready_data		= 1'b0;
	  ready_ctrl		= 1'b0;
	  valid_data		= 1'b0;
	  valid_ctrl		= 1'b0;
	  
	  case(state)
		//EMPTY   - only shift the conveyor and do not output anything
		//NOTICE that no ctrl input is requested 
		2'b00: begin
		   ready_data	= data_valid;
		end
		//RUNNING - shift and output both vertically and horizontally
		2'b01: begin
		   ready_data	= all_rdy && inputs_valid;
		   ready_ctrl	= all_rdy && inputs_valid;
		   valid_data	= all_rdy && inputs_valid;
		   valid_ctrl   = all_rdy && inputs_valid;
		end
		//FLUSH  - shift conveyor
		//I am losing one clock cycle here but I make the circuit simpler
		//IF there was input I could have switched to state 1
		2'b10: begin
		   //ready_o	= 1'b0;   
		   valid_data	= conv_rdy;
		end
		//DONE  - shift conveyor and output done bit
		2'b11: begin
		   //ready_o	= 1'b0;
		   valid_data	= conv_rdy;
		end
		
		default:
		  $display ( "%m PROBLEM IN SORT FSM - ALWAYS COMB" );
		
	  endcase // case (state)
   end // always_comb

   initial begin
	  state = 2'b00;
   end
   
   always_ff @(posedge clk) begin

      if(reset_i) begin
		 state    <= 2'b0;
		 //op       <= 3'b000;
		 all_rdy  <= 1'b0;
		 count    <= 0;
		 buffer   <= 0;
      end
	  else begin
		 // if(chipselect && write) begin
		 // 	op <= conf[2:0];
		 // end
		 if ( BOTTOM_ROW ) 
		   all_rdy  <= ava_out.ready ;
		 else
		   all_rdy  <= ava_out.ready && ctrl_out.ready;
		 
		 //all_rdy  <= ready_1_i && ready_2_i;
		 //conv_rdy <= ready_1_i;
		 conv_rdy <= ava_out.ready;
		 
		 case(state)  //probably I can avoid using a state variable and just use the count reg
		   
		   //EMPTY
		   2'b00: begin
			  if ( data_valid ) begin
				 state       <= 2'b01;
				 buffer      <= data;
				 count       <= 1;
			  end 
		   end
		   //RUNNING
		   2'b01: begin
			  if ( inputs_valid && all_rdy ) begin //this could be more efficient - consume data from the same group
				 if ( inputs_done ) begin
					state       <= 2'b11;
				 end
				 else begin
					if ( swap ) begin
					   buffer <= data;
					end 
					
					if ( count == NSWAPS )
					  state <= 2'b10;
					else
					  count <= count + 1;			
				 end
			  end // if ( inputs_valid && all_rdy )
		   end
		   //FLUSH
		   2'b10: begin
			  if ( conv_rdy ) begin //I only care about the conveyor when flushing
				 $display("Time: %t - flushing %d",$time,buffer);				 
				 state <= 2'b00;
				 count <= 0;
			  end
		   end
		   //DONE
		   2'b11: begin
			  if ( conv_rdy ) begin
				 state <= 2'b00;
				 count <= 0;
			  end
		   end
		   //ERROR		   
		   default: 
			 $display("Error in AGGR FSM");
		   
		 endcase
	  end
   end

endmodule
