/* aggregator */

/* Working */
module aggregator #(
					parameter COLUMN_WIDTH = 32,
					parameter AGGR_OP = 0
					)

   (
	//data in AvalonST
	input logic [COLUMN_WIDTH-1:0] 	col_1_i,
	input logic 					valid_1_i,
	output logic 					ready_1_o,
	input logic 					done_1_i,
	input logic 					srt_pck_1_i,
	input logic 					end_pck_1_i,
	
	//data in AvalosST
	input logic [COLUMN_WIDTH-1:0] 	col_2_i,
	input logic 					valid_2_i,
	output logic 					ready_2_o,
	input logic 					done_2_i,
	input logic 					srt_pck_2_i,
	input logic 					end_pck_2_i,
   
	//control in AvalonMM
	input logic [COLUMN_WIDTH-1:0] 	conf,
	input logic 					write,
	input logic 					chipselect,
	//input logic 					address,

	//data out AvalonST
	output logic [COLUMN_WIDTH-1:0] col_o, 
	output logic 					valid_o,
	input logic 					ready_i,
	output logic 					done_o, 
	output logic 					srt_pck_o,
	output logic 					end_pck_o,

	/* std signals */
	input logic 					clk,
	input logic 					reset_i // reset everything
	);

   //logic [COLUMN_WIDTH-1:0] 		col_ob; //Module out

   logic [COLUMN_WIDTH-1:0] 		group; //this
   logic [COLUMN_WIDTH-1:0] 		data; // out from RS2
   
   logic 							ready1; //ready input to first RS
   logic 							ready2; //ready input to second RS

   logic 							group_valid; //out from RS1
   logic 							data_valid; //out from RS2

   logic 							done1, done2;   

   logic 							was_rdy;
   logic 							inputs_valid, inputs_done;
   logic 							change_of_group;
   
   /**
	op can be:
	MIN,MAX,SUM,AVG,COUNT,COUNT-DISTINCT,NO-OP
	we could use the boolgen to detect groups
	*/
   logic [2:0] 						op;

   /**
	state can be:
	EMPTY,RUNNING,DONE
	EMPTY: no data has arrived yet
	RUNNING: 
	DONE: a DONE bit has been received - send a done packet in the next clock cycle
	*/   
   logic [1:0] 						state;
   logic [COLUMN_WIDTH-1:0] 		old_group, accumulator; 
   logic [COLUMN_WIDTH-1:0] 		updated_accumulator,new_accumulator;
   logic [COLUMN_WIDTH-1:0] 		count,group_count,average;
   logic 							smaller_input;
   
   buffer in1(.col_i(col_1_i), 
			  .valid_i(valid_1_i), 
			  .ready_i(ready1),
              .col_o(group),
			  .valid_o(group_valid),  
			  .ready_o(ready_1_o),
			  .clk(clk),
			  .reset_i(reset_i),
			  .start_pck_in(srt_pck_1_i),
			  .end_pck_in(end_pck_1_i),
			  .done_i(done_1_i),
			  .done_o(done1)
			  );
   
   buffer in2(.col_i(col_2_i), 
			  .valid_i(valid_2_i), 
			  .ready_i(ready2),
              .col_o(data),
			  .valid_o(data_valid),  
			  .ready_o(ready_2_o),
			  .clk(clk),
			  .reset_i(reset_i),
			  .start_pck_in(srt_pck_2_i),
			  .end_pck_in(end_pck_2_i),
			  .done_i(done_2_i),
			  .done_o(done2)
			  );

   assign srt_pck_o				= 1;
   assign end_pck_o				= 1;

   assign inputs_done			= done1 & done2; 
   assign inputs_valid			= group_valid & data_valid;
   assign change_of_group		= (group != old_group);

   assign average               = $signed(accumulator) / $signed(count);
   
   assign col_o					= (op == 3) ? average : accumulator; //3 is avg
   assign done_o				= ( state == 2'b10 ) ? 1'b1 : 1'b0;

   assign smaller_input			= ( $signed(data) < $signed(accumulator) );
   
   //assignment of valid and ready bits
   always_comb begin
	  ready1	= 1'b0;
	  ready2	= 1'b0;
	  valid_o	= 1'b0;
	  
	  case(state)
		//EMPTY
		2'b00: begin
		   //BUG HERE? 
		   // ready1  = 1'b0;
		   // ready2   = 1'b0;
		   
		   ready1   = inputs_valid;
		   ready2   = inputs_valid;
		   
		end
		//RUNNING
		2'b01: begin
		   if ( was_rdy && inputs_valid ) begin
			  ready1 = 1'b1;
			  ready2 = 1'b1;
			  if ( change_of_group | inputs_done )
				valid_o = 1'b1;
		   end
		end
		//DONE
		2'b10: begin
		   if ( was_rdy ) begin
			  valid_o = 1'b1; //output the done packet
		   end
		end
		
	  endcase // case (state)
   end // always_comb

   //this is the accumulator when within a group
   always_comb begin
	  case ( op ) 
		3'b000	: updated_accumulator = smaller_input ? data : accumulator;
		3'b001	: updated_accumulator = smaller_input ? accumulator : data;
		3'b010	: updated_accumulator = accumulator + data;
		3'b011	: updated_accumulator = accumulator + data;
		3'b100	: updated_accumulator = accumulator + 1;
		3'b101	: updated_accumulator = accumulator + 1;
		3'b110	: updated_accumulator = accumulator;
		default	: updated_accumulator = 0;
	  endcase
   end

   //this is the accumulator function when changing group
   always_comb begin
	  case ( op ) 
		3'b000	: new_accumulator = data;
		3'b001	: new_accumulator = data;
		3'b010	: new_accumulator = data;
		3'b011	: new_accumulator = data;
		3'b100	: new_accumulator = 1;
		3'b101	: new_accumulator = 1;
		3'b110	: new_accumulator = data;
		default	: new_accumulator = 0;
	  endcase
   end

   always @(posedge clk) begin
   	  $display("Time %t group is %d and data is %d - inputs are %b - ready are %b %b",$time,group,$signed(data),inputs_valid,ready1,ready2);
	  
   end
   
   always_ff @(posedge clk) begin

      if(reset_i) begin
		 state    <= 2'b00;
		 op       <= AGGR_OP;
		 was_rdy  <= 1'b0;
		 count    <= 0;    //used for avg
		 group_count <= 0; //used for count distinct
      end
	  else begin
		 if(chipselect && write) begin
			op <= conf[2:0];
		 end
		 was_rdy <= ready_i;

		 case(state)
		   
		   //EMPTY
		   2'b00: begin
			  if ( inputs_valid ) begin
				 if ( inputs_done ) begin
					state    <= 2'b10;
					accumulator <= 32'b0; 
				 end
				 else begin
					state       <= 2'b01;
					old_group   <= group;
					accumulator <= new_accumulator;
					count <= 1;
				 end				 
			  end 
		   end
		   //RUNNING
		   2'b01: begin
			  if ( inputs_valid && was_rdy ) begin //this could be more efficient - consume data from the same group
				 if ( inputs_done ) begin
					state       <= 2'b10;
					// put accumulator to zero - this will be outputted in the done packet.
					// 
					accumulator <= 32'b0; 
				 end
				 else begin
					count <= count + 1;				 
					if ( change_of_group ) begin
					   accumulator <= new_accumulator;
					   old_group   <= group;
					   count <= 1;
					end
					else begin
					   count <= count + 1;
					   accumulator <= updated_accumulator;  
					end
				 end
			  end // if ( inputs_valid && was_rdy )
		   end
		   //DONE
		   2'b10: begin
			  if ( was_rdy )
				state <= 2'b00;
		   end
		   //ERROR		   
		   default: 
			 $display("Error in AGGR FSM");
		   
		 endcase
	  end
   end

endmodule
