/*
 * Seven-segment LED emulator
 *
 * Stephen A. Edwards, Columbia University
 *
 */

module VGA_LED_Emulator(
  input logic         clk50, reset,
 input logic [15:0]  hex0,			//V of grenade		hex0[15:10]:the number of crater hex0[9:0]: cracter_location
							hex1,			//hex1[9:0]: H of grenade hex1[15:12] explosion figure selec		
							hex2,			//V 0f stand1		hex2[15]: 	0:face to right, 1:face to left 
											//						hex2[14:12]	show the pic of stand1 from #0 to #6
							hex3,			//H of stamd1		hex3[15:10] blood of player1
							hex4,			//V 0f stand2		hex4[15]: 	0:face to right, 1:face to left 
											//						hex4[14:12]	show the pic of stand2 from #0 to #6
							hex5,			//H 0f stand2		hex5[15:10] blood of player2
							hex6,			//hex6[5:0]: V of pointer; [11:6]: V of pointer; hex[14]: en
							hex7,			//hex7[14:10] time_count 30 secs hex7[15] time of player selec
											//hex7[9:4] strength
							hex8,			// sound       
							hex9,			
							hex10,
							hex11,
							hex12,
							hex13,
							hex14,
							hex15,
 output logic [7:0]	VGA_R, VGA_G, VGA_B,
 output logic        VGA_CLK, VGA_HS, VGA_VS, VGA_BLANK_n, VGA_SYNC_n,
 
 input logic[7:0] 	in_R,in_G,in_B,							//soil
							in_R2,in_G2,in_B2,						//stone
							in_R3,in_G3,in_B3,						//grass	
							in_RG,in_GG,in_BG,						//grenade	
							in_RL01,in_GL01,in_BL01,				//background
							in_R_stand1,in_G_stand1,in_B_stand1,//stand1
							in_R_stand2,in_G_stand2,in_B_stand2,//stand2
							in_R_pointer,in_G_pointer,in_B_pointer,//pointer
							in_R_explosion,in_G_explosion,in_B_explosion,//explosion
							in_R_head1, in_G_head1, in_B_head1,  //head1
							in_R_head2, in_G_head2, in_B_head2,  //head2
							
 output logic[10:0]	hcount,
 output logic[9:0]	vcount );
/*
 * 640 X 480 VGA timing for a 50 MHz clock: one pixel every other cycle
 * 
 *HCOUNT 1599 0             1279       1599 0
 *            _______________              ________
 * __________|    Video      |____________|  Video
 * 
 * 
 * |SYNC| BP |<-- HACTIVE -->|FP|SYNC| BP |<-- HACTIVE
 *       _______________________      _____________
 * |____|       VGA_HS          |____|
 */

   parameter HACTIVE      = 11'd 1280,
             HFRONT_PORCH = 11'd 32,
             HSYNC        = 11'd 192,
             HBACK_PORCH  = 11'd 96,   
             HTOTAL       = HACTIVE + HFRONT_PORCH + HSYNC + HBACK_PORCH; //1600

   parameter VACTIVE      = 10'd 480,
             VFRONT_PORCH = 10'd 10,
             VSYNC        = 10'd 2,
             VBACK_PORCH  = 10'd 33,
             VTOTAL       = VACTIVE + VFRONT_PORCH + VSYNC + VBACK_PORCH; //525


   always_ff @(posedge clk50 or posedge reset)
     if (reset)          hcount <= 0;
     else if (endOfLine) hcount <= 0;
     else  	         hcount <= hcount + 11'd 1;

   assign endOfLine = hcount == HTOTAL - 1;


   logic 			     endOfField;
   
   always_ff @(posedge clk50 or posedge reset)
     if (reset)          vcount <= 0;
     else if (endOfLine)
       if (endOfField)   vcount <= 0;
       else              vcount <= vcount + 10'd 1;

   assign endOfField = vcount == VTOTAL - 1;

   // Horizontal sync: from 0x520 to 0x57F
   // 101 0010 0000 to 101 0111 1111
   assign VGA_HS = !( (hcount[10:7] == 4'b1010) & (hcount[6] | hcount[5]));
   assign VGA_VS = !( vcount[9:1] == (VACTIVE + VFRONT_PORCH) / 2);

   assign VGA_SYNC_n = 1; // For adding sync to video signals; not used for VGA
   
   // Horizontal active: 0 to 1279     Vertical active: 0 to 479
   // 101 0000 0000  1280	       01 1110 0000  480	       
   // 110 0011 1111  1599	       10 0000 1100  524        
   assign VGA_BLANK_n = !( hcount[10] & (hcount[9] | hcount[8]) ) &
			!( vcount[9] | (vcount[8:5] == 4'b1111) );   

			
   assign VGA_CLK = hcount[0]; // 25 MHz clock: pixel latched on rising edge
	
	
	
	
  
  
  
//////////////////////////////////////////////////////////////////////////////
//**************************************************************************//
//*																								*//
//*								Here we start our design								*//
//*																								*//
//**************************************************************************//
//////////////////////////////////////////////////////////////////////////////
						
//============================================================================
//2d array for Ground_block_eabi
//============================================================================
/*
logic [143:0][639:0] Ground_block_eabi;
logic [31:0]			Ground_block_ctrl;
logic [7:0]				V_Ground_block_eabi;
logic [9:0]				H_Ground_block_eabi;


assign 	Ground_block_ctrl={hex7, hex6};
assign 	V_Ground_block_eabi=Ground_block_ctrl[10:3];
assign 	H_Ground_block_eabi=Ground_block_ctrl[20:11];

   always_ff @(posedge clk50 or posedge reset)		 
     if (reset)          Ground_block_eabi <= 0;      //Initialize Ground_block_eabi
     else if 	(Ground_block_ctrl[0])  
					Ground_block_eabi[V_Ground_block_eabi][H_Ground_block_eabi]<= Ground_block_ctrl[1];
     else		Ground_block_eabi  <=  Ground_block_eabi;

*/

//============================================================================
//Crater
//============================================================================	
logic[19:0] 	n_of_crater,crater_flag;
logic[19:0][19:0] 	cracter_location;

assign n_of_crater=hex0[15:10];


generate
		genvar i;
		for (i=0; i<20; i=i+1) begin : crater_eabi
			assign crater_flag[i] = 
				((hcount[10:1]-cracter_location[i][19:10])*(hcount[10:1]-cracter_location[i][19:10])+
				(vcount[9:0]-cracter_location[i][9:0])*(vcount[9:0]-cracter_location[i][9:0]) <= 1024);
		end
endgenerate
	
		
always_ff @(posedge clk50 or posedge reset)		 
     if (reset)          cracter_location = 20'b0;      //Initialize 
     else if 	(!(|(n_of_crater)==0))  begin		
	  
					cracter_location[n_of_crater]<={hex1[9:0], hex0[9:0]};
							
					end

//============================================================================
//Indicate the range of moving element and ground_block
//============================================================================

logic inpointer, inGrenade, inStand1, inSoil, inGrass, inExplosion, 
		inHead1, inHead2, inBlood1, inBlood2, inBloodBlank1, inBloodBlank2, 
		inStrengh1, inStrengh2, inStrenghBlank1, inStrenghBlank2, inTime1, 
		inTime2, inTimeBlank1, inTimeBlank2;
logic[7:0] display_select;
logic[7:0] blood_red1, blood_green1, blood_blue1, blood_red2, blood_green2, blood_blue2,
			  time_red, time_green, time_blue, strength_red1, strength_green1, strength_blue1,
			  strength_red2, strength_green2, strength_blue2;
logic[5:0]  hexh, hexv;
logic[15:0] hexv_stand, hexh_stand;
logic[9:0] differ_h, differ_v;



///////////////////////////////////////pointer
assign hexv = hex6[5:0];
assign hexh = hex6[11:6];
assign hexv_stand = hex6[13]? // 1: stand_2, 0: stand_1;
							hex4[15:0]:
							hex2[15:0];

assign hexh_stand = hex6[13]?
							hex5[15:0]:
							hex3[15:0];
							
assign differ_v = hex6[12]?  // 0: down, 1: up;
						(hexv_stand[9:0]-hexv):
						(hexv_stand[9:0]+hexv);
						
assign differ_h = hexv_stand[15]? //0:face to right, 1:face to left 
						(hexh_stand[9:0]-hexh):
						(hexh_stand[9:0]+hexh);
////////////////////////////////////////////
assign inBlood1 = ((hcount/2 > 10'd80) && (hcount/2 <= 10'd80 + 2*hex9[5:0]) && 
						 (vcount >= 10'd50) && (vcount <= 10'd57));
assign inBlood2 = ((hcount/2 > 10'd560 - 2*hex9[11:6]) && (hcount/2 <  10'd560) && 
						 (vcount >= 10'd50) && (vcount <= 10'd57));
assign inTime1 = ((hcount/2 > 10'd80) && (hcount/2 <= 10'd80 + 3*hex7[14:10]) && 
					   (vcount >= 10'd36) && (vcount <= 10'd43) && !hex7[15]);//time left 0-90
assign inTime2 = ((hcount/2 > 10'd560 - 3*hex7[14:10]) && (hcount/2 < 10'd560) && 
						(vcount >= 10'd36) && (vcount <= 10'd43) && hex7[15]);//time right
assign inStrengh1 = ((hcount/2 > 10'd80) && (hcount/2 <= 10'd80 + 3*hex7[9:4]) && 
					   (vcount >= 10'd22) && (vcount <= 10'd29) && !hex7[15]);//strength left 0-189
assign inStrengh2 = ((hcount/2 > 10'd560 - 3*hex7[9:4]) && (hcount/2 < 10'd560) && 
					   (vcount >= 10'd22) && (vcount <= 10'd29) && hex7[15]);
assign inBloodBlank1 = ((hcount/2 > 10'd80) && (hcount/2 <= 10'd206) && 
								(vcount >= 10'd50) && (vcount <= 10'd57));//left blood 0-126
assign inBloodBlank2	= ((hcount/2 > 10'd434) && (hcount/2 < 10'd560) && 
								(vcount >= 10'd50) && (vcount <= 10'd57));//right	blood
assign inTimeBlank1  = ((hcount/2 > 10'd80) && (hcount/2 <= 10'd170) && 	
								(vcount >= 10'd36) && (vcount <= 10'd43));//left time
assign inTimeBlank2  = ((hcount/2 > 10'd470) && (hcount/2 < 10'd560) && 
								(vcount >= 10'd36) && (vcount <= 10'd43));//right time
assign inStrenghBlank1 = ((hcount/2 > 10'd80) && (hcount/2 <= 10'd269) && 
								  (vcount >= 10'd22) && (vcount <= 10'd29));//left strength
assign inStrenghBlank2 = ((hcount/2 > 10'd371) && (hcount/2 < 10'd560) && 
								  (vcount >= 10'd22) && (vcount <= 10'd29));//right stength
assign inpointer =(!((in_R_pointer== 8'd0) && (in_G_pointer == 8'd0) && (in_B_pointer == 8'd0))&&
						(hcount/2 > (differ_h[9:0]-6)) && (hcount/2 <= (differ_h[9:0]+6)) &&
						(vcount > (differ_v[9:0]-6)) && vcount <= (differ_v[9:0]+6)&& hex6[14]);
//logic in_screen = (hex[9:0] > 31);

//logic[9:0] explosion_h;

//assign explosion_h = in_screen?
//							hex[9:0] - 31 : 1;
 
						
assign inExplosion =(!((in_R_explosion== 8'd0) && (in_G_explosion == 8'd0) && (in_B_explosion == 8'd0))&&
						    (hcount/2 > hex1[9:0]-32) && (hcount/2 <= hex1[9:0]+32) &&
						     (vcount > hex0[9:0]-31) && (vcount <= hex0[9:0]+32));
						
assign inGrenade =(!((in_RG == 8'd0) && (in_GG == 8'd0) && (in_BG == 8'd0))&&
						(hcount/2 > (hex1[9:0]-7)) && (hcount/2 <= (hex1[9:0]+7)) &&
						(vcount > (hex0[9:0]-7)) && (vcount <=(hex0[9:0]+7)));

assign inStand1 = (!((in_R_stand1 == 8'd255) && (in_G_stand1 == 8'd255) && (in_B_stand1 == 8'd255))&&
						(hcount/2 > (hex3[9:0]-21) && hcount/2 <= (hex3[9:0]+22) )&& 
						(vcount > (hex2[9:0]-21) && vcount <= (hex2[9:0]+22)));

assign inStand2 = (!((in_R_stand2 == 8'd255) && (in_G_stand2 == 8'd255) && (in_B_stand2 == 8'd255))&&
						(hcount/2 >(hex5[9:0]-21) && hcount/2 <= (hex5[9:0]+22) )&&
						(vcount > (hex4[9:0]-21) && vcount <= (hex4[9:0]+22)));
						
assign inSoil = 	((hcount/2 >= 0 && hcount/2 < 640 && vcount >= 384 && vcount < 448)||
						(hcount/2 >= 96 && hcount/2 < 560 && vcount >= 368 && vcount < 384)|| 
						(hcount/2 >= 132 && hcount/2 < 528 && vcount >= 352 && vcount < 368))&&
						(!(|crater_flag));
						//(!Ground_block_eabi[vcount-335][hcount/2]);//whether this block has been distroied or not
						
assign inGrass =	((hcount/2 >= 0 && hcount/2 < 96 && vcount >= 368 && vcount < 384)||
						(hcount/2 >= 96 && hcount/2 < 132 && vcount >= 352 && vcount < 368)|| 
						(hcount/2 >= 132 && hcount/2 < 528 && vcount >= 336 && vcount < 352)||
						(hcount/2 >= 528 && hcount/2 < 560 && vcount >= 352 && vcount < 368)||
						(hcount/2 >= 560 && hcount/2 < 640 && vcount >= 368 && vcount < 384))&&
						(!(|crater_flag));
				//	(!Ground_block_eabi[vcount-335][hcount/2]);//whether this block has been distroied or not
	
assign inHead1 =	(hcount/2 > 20 && hcount/2 < 76 && vcount >= 20 && vcount < 58);
assign inHead2 =	(hcount/2 > 564 && hcount/2 < 620 && vcount >= 20 && vcount < 58);


//============================================================================
//Indicate the segment of blood in diff color
//============================================================================


	
	
	
//============================================================================
//Indicate which element to display
//============================================================================



 always_comb begin      
display_select <=

			inpointer?//pointer
	8'b00000001:	
			inExplosion?
	8'b00000010:	
			inSoil?//soil
	8'b00000011:
			inGrass?//grass
	8'b00000100:
			(hcount/2 >= 0 && hcount/2 < 640 && vcount >= 448 && vcount < 480)?	//stone
	8'b00000101:
			inBlood1?
	8'b00000110:
			inBlood2?
	8'b00000111:		
			inTime1?
	8'b00001000:		
			inTime2?
	8'b00001001:
			inStrengh1?
	8'b00001010:
			inStrengh2?
	8'b00001011:		
			inBloodBlank1?
	8'b00001110:
			inTimeBlank1?
	8'b00001111:
			inStrenghBlank1?
	8'b00010000:
			inBloodBlank2?
	8'b00010001:
			inTimeBlank2?
	8'b00010010:
			inStrenghBlank2?
	8'b00010011:
			inGrenade? 	//grenade
	8'b00010100:
			inStand1?
	8'b00010101:
			inStand2?		
	8'b00010110:
			inHead1?
	8'b00010111:
			inHead2?
	8'b00011000:
	      (hcount/2 >= 0 && hcount/2 < 640 && vcount >= 0 && vcount < 480)?    //Background_left01
	8'b00011001:     	
	8'b00000000;
	
{blood_red1, blood_green1, blood_blue1} <=
	((2*hex9[5:0] >= 7'd0) && (2*hex9[5:0] <= 7'd41))? {8'd200, 8'd0, 8'd0}:
	((2*hex9[5:0] > 7'd42) && (2*hex9[5:0] <= 7'd83))? {8'd200, 8'd180, 8'd0}:
	{8'd0, 8'd180, 8'd0};
	
{blood_red2, blood_green2, blood_blue2} <=
	((2*hex9[11:6] >= 7'd0) && (2*hex9[11:6] <= 7'd41))? {8'd200, 8'd0, 8'd0}:
	((2*hex9[11:6] > 7'd42) && (2*hex9[11:6] <= 7'd83))? {8'd200, 8'd180, 8'd0}:
	{8'd0, 8'd180, 8'd0};
	
{time_red, time_green, time_blue} <=
	(3*hex7[14:10] >= 6'd15)? {8'd120, 8'd187, 8'd254}:{8'd180, 8'd0, 8'd0};
	
{strength_red1,strength_green1,strength_blue1} <= 
	((hcount/2 - 10'd80>= 8'd0) && (hcount/2 - 10'd80 <= 8'd90))? {2*(hcount/2 - 10'd80) + 8'd61, 8'd0, 8'd0}:
	{8'd240, 8'd0, 8'd0 };//red 0-89:151-240 green blue 90-189:0-99
	
{strength_red2,strength_green2,strength_blue2} <= 
	((10'd560 - hcount/2>= 8'd0) && (10'd560 - hcount/2 <= 8'd90))? {2*(10'd560 - hcount/2) + 8'd61, 8'd0, 8'd0}:
	{8'd240, 8'd0, 8'd0};//red 0-89:151-240 green blue 90-189:0-99
	
	case (display_select) 
	8'b00000001:{VGA_R, VGA_G, VGA_B} = {in_R_pointer, in_G_pointer, in_B_pointer };		//pointer
	8'b00000010:{VGA_R, VGA_G, VGA_B} = {in_R_explosion, in_G_explosion, in_B_explosion };//explosion
	8'b00000011:{VGA_R, VGA_G, VGA_B} = {in_R, in_G, in_B };								//soil
	8'b00000100:{VGA_R, VGA_G, VGA_B} = {in_R3, in_G3, in_B3 };							//grass
	8'b00000101:{VGA_R, VGA_G, VGA_B} = {in_R2, in_G2, in_B2 };							//stone
	8'b00000110:{VGA_R, VGA_G, VGA_B} = {blood_red1, blood_green1, blood_blue1};
	8'b00000111:{VGA_R, VGA_G, VGA_B} = {blood_red2, blood_green2, blood_blue2};
	8'b00001000:{VGA_R, VGA_G, VGA_B} = {time_red, time_green, time_blue };//time1
	8'b00001001:{VGA_R, VGA_G, VGA_B} = {time_red, time_green, time_blue };//time2
	8'b00001010:{VGA_R, VGA_G, VGA_B} = {strength_red1, strength_green1, strength_blue1 };//strength1
	8'b00001011:{VGA_R, VGA_G, VGA_B} = {strength_red2, strength_green2, strength_blue2 };//strength2
	8'b00001110:{VGA_R, VGA_G, VGA_B} = {8'h20, 8'h20, 8'h20 };//bloodblank1
	8'b00001111:{VGA_R, VGA_G, VGA_B} = {8'h20, 8'h20, 8'h20 };//timeblank1
	8'b00010000:{VGA_R, VGA_G, VGA_B} = {8'h20, 8'h20, 8'h20 };//strengthblank1
	8'b00010001:{VGA_R, VGA_G, VGA_B} = {8'h20, 8'h20, 8'h20 };//bloodblank2
	8'b00010010:{VGA_R, VGA_G, VGA_B} = {8'h20, 8'h20, 8'h20 };//timeblank2	
   8'b00010011:{VGA_R, VGA_G, VGA_B} = {8'h20, 8'h20, 8'h20 };//strengthblank2
	8'b00010100:{VGA_R, VGA_G, VGA_B} = {in_RG, in_GG, in_BG };							//grenade
	8'b00010101:{VGA_R, VGA_G, VGA_B} = {in_R_stand1, in_G_stand1, in_B_stand1};	//stand1
   8'b00010110:{VGA_R, VGA_G, VGA_B} = {in_R_stand2, in_G_stand2, in_B_stand2};	//stand2	
	8'b00010111:{VGA_R, VGA_G, VGA_B} = {in_R_head1, in_G_head1, in_B_head1};	//head1
   8'b00011000:{VGA_R, VGA_G, VGA_B} = {in_R_head2, in_G_head2, in_B_head2};	//head2	
	8'b00011001:{VGA_R, VGA_G, VGA_B} = {in_RL01, in_GL01, in_BL01 };//Background_left01
	8'b00000000:{VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'h00};//Black
	endcase

   end 
  
   
endmodule // VGA_LED_Emulator
