/*
 * Seven-segment LED emulator
 *
 * Stephen A. Edwards, Columbia University
 *
 */

module VGA_BALL_Emulator(
    	input logic       clk50, reset,
        input logic enable_ram,
        input logic ram_select,
		  input logic [1:0] life,
			input logic [1:0] bomb_value,
		input logic [9:0] score,
		input logic [3:0] tens,
		input logic [3:0] digits,
		input logic game_over,
		   input logic en_emulator,//ff
    	input logic update_done,//game logic says you can work
        input logic [31:0] game_logic_to_emulator,
		output logic draw_done,//tell game logic you finish
		output logic [7:0] VGA_R, VGA_G, VGA_B,
		output [31:0]A,B,C,
		output [31:0]mem_out_u,mem_out,A_in,B_in,C_in,ram_out,
		output logic      VGA_CLK, VGA_HS, VGA_VS, VGA_BLANK_n, VGA_SYNC_n);

        logic [23:0] color_spaceship;
        logic [23:0] color_background;
        logic [23:0] color_bullet;
		  logic [23:0] color_spawn0;
		  logic [23:0] color_spawn45;
		  logic [23:0] color_enemy400;
		  logic [23:0] color_enemy500;
		  logic [23:0] color_enemy600;
		  logic [23:0] color_enemy700;
		  logic [23:0] color_explosion_bullet;
		  logic [23:0] color_explosion_bullet180;
		  logic [23:0] color_explosion_spaceship;
		  logic [23:0] color_explosion_spaceship1;
		  logic [23:0] color_lifeword;
		  logic [23:0] color_number_zero;
		  logic [23:0] color_number_one;
		  logic [23:0] color_number_two;
		  logic [23:0] color_number_three;
		  logic [23:0] color_number_four;
		  logic [23:0] color_number_five;
		  logic [23:0] color_number_six;
		  logic [23:0] color_number_seven;
		  logic [23:0] color_number_eight;
		  logic [23:0] color_number_nine;
		  logic [23:0] color_bomb;
		  logic [23:0] color_score;
		  logic [23:0] color_start;
		  logic [23:0] color_starwars;

        logic [10:0]    hcount; // Horizontal counter
        logic           endOfLine;

      // Vertical counter
        logic [9:0]     vcount;
        logic           endOfField;

      	logic  [9:0]    xcoor;
      	logic  [9:0]    ycoor;

        logic  [9:0]  space_address;
        logic  [11:0]  background_address;
        logic  [9:0]  bullet_rom_address; //ff bullet is 16*16 pixels
		  logic  [9:0]	 spawn_address;
		  logic  [9:0]	 enemy4_address;
		  logic  [9:0]	 enemy5_address;
		  logic  [9:0]	 enemy6_address;
		  logic  [9:0]	 enemy7_address;
		  logic  [9:0]  explosion_bullet_address;
		  logic  [9:0]  explosion_spaceship_address;
		  logic  [11:0] lifeword_address;
		  logic  [9:0]  number_address;
		  logic  [12:0]  bombword_address;
		  logic  [11:0]  scoreword_address; 
		  logic  [14:0]  startword_address;
		  logic  [14:0] startwarsword_address;
		  //logic  		 draw_con_lifeword;
     //   logic           draw_spaceship;
       // logic           draw_bullet;
		  //logic 				draw_spawn;


  parameter HACTIVE      = 11'd 1280,
            HFRONT_PORCH = 11'd 32,
            HSYNC        = 11'd 192,
            HBACK_PORCH  = 11'd 96,
            HTOTAL       = HACTIVE + HFRONT_PORCH + HSYNC + HBACK_PORCH; //1600

  parameter VACTIVE      = 10'd 480,
            VFRONT_PORCH = 10'd 10,
            VSYNC        = 10'd 2,
            VBACK_PORCH  = 10'd 33,
            VTOTAL       = VACTIVE + VFRONT_PORCH + VSYNC + VBACK_PORCH; //525

parameter 	NO_ID 		=8'd0,
				SPACESHIP	=8'd1,
		      BULLET		=8'd2,
				SPAWN 		=8'd3,
				ENEMY4		=8'd4,
				ENEMY5		=8'd5,
				ENEMY6		=8'd6,
				ENEMY7		=8'd7,
				EXPLOSION_BULLET = 8'd11,
				EXPLOSION_SPACESHIP = 8'd12;



enum logic [2:0] {waiting, processing} states;
enum logic [1:0] {wait_receive_finish,receive_data} states_process;
enum logic [3:0] {line_buffer_setting, line_buffer_ram,line_buffer_waiting, line_buffer_working} line_buffer_states;
enum logic [4:0] {loop_waiting, loop_tmp_waiting, loop_tmp_setting,loop_writing} loop_states;
enum logic [1:0] {draw_waiting, draw_processing} draw_states;
//uA_cB_dC : update A, Clear B, draw C
//uB_cC_dA : update B, clear C, draw A
//uC_cA_dB : update C, clear A, draw B
enum logic [2:0] {uA_cB_dC, uB_cC_dA,uC_cA_dB} line_buffer_select;


always_ff @(posedge clk50 or posedge reset) begin
    if (reset) begin
        hcount <= 0;
        vcount <= 0;
        states <= waiting;
        draw_done <= 0;
    end
    else if (states == waiting) begin
        hcount <= 0;
        vcount <= 0;
        draw_done <= 0;

		  //states_process <= receive_data;
        if (update_done == 1'd1) begin //ffff
            states <= processing;
        end
    end

    else if (states == processing) begin
        if(endOfLine) begin
            hcount <= 0;
            if (endOfField) begin
                vcount <= 0;
                draw_done <= 1;
                states <= waiting;
            end
            else begin
                vcount <= vcount + 10'd1;
            end
        end
        else begin
            hcount <= hcount +11'd1;
        end
    end
end

  assign endOfLine = hcount == HTOTAL - 1;
  assign endOfField = vcount == VTOTAL - 1;

  /*
  * 640 X 480 VGA timing for a 50 MHz clock: one pixel every other cycle
  *
  *HCOUNT 1599 0             1279       1599 0
  *            _______________              ________
  * __________|    Video      |____________|  Video
  *
  *
  * |SYNC| BP |<-- HACTIVE -->|FP|SYNC| BP |<-- HACTIVE
  *       _______________________      _____________
  * |____|       VGA_HS          |____|
  */


  // Horizontal sync: from 0x520 to 0x57F
  // 101 0010 0000 to 101 0111 1111
  assign VGA_HS = !( (hcount[10:7] == 4'b1010) & (hcount[6] | hcount[5]));
  assign VGA_VS = !( vcount[9:1] == (VACTIVE + VFRONT_PORCH) / 2);

  assign VGA_SYNC_n = 1; // For adding sync to video signals; not used for VGA

  // Horizontal active: 0 to 1279     Vertical active: 0 to 479
  // 101 0000 0000  1280	       01 1110 0000  480
  // 110 0011 1111  1599	       10 0000 1100  524
  assign VGA_BLANK_n = !( hcount[10] & (hcount[9] | hcount[8]) ) & !( vcount[9] | (vcount[8:5] == 4'b1111) );



    ////fffff

    //logic [31:0] ram_a_tmp[0:59];
    //logic [31:0] ram_b_tmp[0:59];
    logic [7:0] ram_addr;
	 logic [7:0] ram_counter;
	 logic ram_a_we,ram_b_we;
	 logic ram_we;
	 logic [31:0] ram_a_in,ram_b_in,ram_in;
	 logic [31:0] ram_a_out,ram_b_out;
	 logic [7:0] ram_a_address,ram_b_address;

	 myram ram_a_tmp(.clk(clk50),.we(ram_a_we),.data_in(ram_a_in),.data_out(ram_a_out),.address(ram_a_address));
	 myram ram_b_tmp(.clk(clk50),.we(ram_b_we),.data_in(ram_b_in),.data_out(ram_b_out),.address(ram_b_address));
	 assign ram_a_we = ram_select ? ram_we:0;//ram_selct = 1, ram_a receives data from vga_BALL.
	 assign ram_b_we = ram_select ? 0:ram_we; //ram_selct = 0, ram_b receives data from vga_BALL.
	 assign ram_a_in = ram_select ? ram_in:0;//ram_selct = 1, ram_a receives data from vga_BALL.
	 assign ram_b_in = ram_select ? 0:ram_in;//ram_selct = 0, ram_b receives data from vga_BALL.
	 assign ram_a_address = ram_select ? ram_addr:ram_counter;//ram_selct = 1, ram_a receives data from vga_BALL.
	 assign ram_b_address = ram_select ? ram_counter:ram_addr;//ram_selct = 1, used to draw graph
	 assign ram_out = ram_select ? ram_b_out:ram_a_out;//ram_selct = 0, ram_a used to draw graph

	 always_ff @(posedge clk50) begin
	     if (states == waiting) begin
				states_process <= receive_data;
				ram_addr <= 0;
		  end
        else if(enable_ram == 1'd1) begin //start to update data from ram
				if (states_process == receive_data) begin
					 if (en_emulator) begin
						ram_in <= game_logic_to_emulator;
						states_process <= wait_receive_finish;
						ram_we <= 1;
						end
				end
				else if (states_process == wait_receive_finish) begin
						ram_addr <= ram_addr + 8'd1;
						ram_we <= 0;
						states_process <= receive_data;
				end
        end
        else begin
		      ram_addr <= 8'd0; //next update start from ram_addr = 0;
		  end
    end

  logic we;
  logic A_we;
  logic B_we;
  logic C_we;
  logic update_we;
  logic clean_we;
  logic [9:0] buffer_A_address;
  logic [9:0] buffer_B_address;
  logic [9:0] buffer_C_address;
  logic [9:0] buffer_update_address;
  logic [9:0] buffer_clean_address;
  logic [9:0] buffer_draw_address;
  logic [31:0]mem_in_update;
  logic [31:0]mem_in_clean;
  logic [31:0]mem_in_A;
  logic [31:0]mem_in_B;
  logic [31:0]mem_in_C;
  //logic [31:0]mem_out;
  //logic [31:0]mem_out_u;
  logic [31:0]A_out;
  logic [31:0]B_out;
  logic [31:0]C_out;
  //logic [31:0]A_in;
  //logic [31:0]B_in;
  //logic [31:0]C_in;
  assign A = A_out;
  assign B = B_out;
  assign C = C_out;
  assign A_in = mem_in_A;
  assign B_in = mem_in_B;
  assign C_in = mem_in_C;
  
  

  line_buffer line_buffer_A(.clk(clk50),.we(A_we),.data_in(mem_in_A),.data_out(A_out),.address(buffer_A_address));
  line_buffer line_buffer_B(.clk(clk50),.we(B_we),.data_in(mem_in_B),.data_out(B_out),.address(buffer_B_address));
  line_buffer line_buffer_C(.clk(clk50),.we(C_we),.data_in(mem_in_C),.data_out(C_out),.address(buffer_C_address));

  always_comb begin
		A_we = 0;
      B_we = 0;
		C_we = 0;
      mem_in_A = 32'd0;
      mem_in_B = 32'd0;
		mem_in_C = 32'd0;
      buffer_A_address = 9'd0;
		buffer_B_address = 9'd0;
		buffer_C_address = 9'd0;
    if (line_buffer_select == uA_cB_dC) begin
      A_we = update_we;
      B_we = clean_we;
      mem_in_A = mem_in_update ;
      mem_in_B = mem_in_clean;
      buffer_A_address = buffer_update_address;
      buffer_B_address = buffer_clean_address;
		buffer_C_address = buffer_draw_address;
    end
    else if (line_buffer_select == uB_cC_dA) begin
      B_we = update_we;
      C_we = clean_we;
      mem_in_B = mem_in_update ;
      mem_in_C = mem_in_clean;
      buffer_B_address = buffer_update_address;
      buffer_C_address = buffer_clean_address;
		buffer_A_address = buffer_draw_address;
    end
    else  if (line_buffer_select == uC_cA_dB) begin
      C_we = update_we;
      A_we = clean_we;
      mem_in_C = mem_in_update ;
      mem_in_A = mem_in_clean;
      buffer_C_address = buffer_update_address;
      buffer_A_address = buffer_clean_address;
		buffer_B_address = buffer_draw_address;
    end

  end

  assign VGA_CLK = hcount[0]; // 25 MHz clock: pixel latched on rising edge
  assign xcoor = hcount[10:1];
  assign ycoor = vcount;
  assign background_address = xcoor+ycoor*64;
  logic [4:0] store_counter;
  logic jump_flag1;
  logic [31:0] object_info;
  assign object_info = ram_out;

  always_ff @(posedge clk50) begin
    if (reset) begin
		  line_buffer_states <= line_buffer_setting;
		  draw_states <= draw_waiting;
			loop_states <= loop_writing;
			store_counter <= 0;
			line_buffer_select <= uA_cB_dC;
			jump_flag1 <= 0;
			space_address <= 10'd0;
			bullet_rom_address <= 10'd0;
			spawn_address <= 10'd0;

	  end
    else if (states == processing) begin
      if(line_buffer_states == line_buffer_setting) begin
        line_buffer_states <= line_buffer_working;
      end
  		if (line_buffer_states == line_buffer_working) begin
          if (vcount - object_info[13:4] >= 0 && vcount - object_info[13:4] <= 10'd31 ) begin
            // write first pixel in buffer line
            if (loop_states == loop_writing) begin
              update_we <= 0;
              //if fisrt pixel of the line in a image, use current x aixs, indicate where the data at the line buffer.
              if (store_counter == 5'd0) begin
                buffer_update_address <= object_info[23:14];
                if (jump_flag1 != 1) loop_states <= loop_tmp_waiting;
              end
              //if the lastpixel of the line in a image, finish, go to fetch next value in ram
              else if(store_counter == 5'd31) begin
                loop_states <= loop_writing;
                jump_flag1 <= 1;
                store_counter <= 0;
              end
              else begin
					 buffer_update_address <= buffer_update_address + 1;
				    loop_states <= loop_tmp_waiting;
				  end

            end
				else if(loop_states == loop_tmp_waiting)begin
					loop_states <= loop_waiting;
					//update_we <= 1;
						if (object_info[31:24] == SPACESHIP) begin
							space_address <= (buffer_update_address-object_info[23:14])+(vcount-object_info[13:4])*32;
							if (color_spaceship != 24'd0) update_we <= 1;
							mem_in_update <= {object_info[31:24],color_spaceship[23:16], color_spaceship[15:8], color_spaceship[7:0]};
						end
						else if (object_info[31:24] == BULLET) begin
							bullet_rom_address <= (buffer_update_address-object_info[23:14])+(vcount-object_info[13:4])*32;
							if (color_bullet != 24'd0) update_we <= 1;
							mem_in_update <= {object_info[31:24],color_bullet[23:16], color_bullet[15:8], color_bullet[7:0]};
						end
						else if (object_info[31:24] == SPAWN) begin
							spawn_address <= (buffer_update_address-object_info[23:14])+(vcount-object_info[13:4])*32;
							if (object_info[3:0] == 4'd0 || object_info[3:0] == 4'd2) begin
								if (color_spawn0 != 24'd0) update_we <= 1;
								mem_in_update <= {object_info[31:24],color_spawn0[23:16], color_spawn0[15:8], color_spawn0[7:0]};
							end
							else if (object_info[3:0] == 4'd1 || object_info[3:0] == 4'd3) begin
								if (color_spawn45 != 24'd0) update_we <= 1;
								mem_in_update <= {object_info[31:24],color_spawn45[23:16], color_spawn45[15:8], color_spawn45[7:0]};
							end
						end
						else if (object_info[31:24] ==ENEMY4) begin
							enemy4_address <= (buffer_update_address-object_info[23:14])+(vcount-object_info[13:4])*32;
							if (color_enemy400 != 24'd0) update_we <= 1;
							mem_in_update <= {object_info[31:24],color_enemy400[23:16], color_enemy400[15:8], color_enemy400[7:0]};
						end
						else if (object_info[31:24] ==ENEMY5) begin
							enemy5_address <= (buffer_update_address-object_info[23:14])+(vcount-object_info[13:4])*32;
							if (color_enemy500 != 24'd0) update_we <= 1;
							mem_in_update <= {object_info[31:24],color_enemy500[23:16], color_enemy500[15:8], color_enemy500[7:0]};
						end
						else if (object_info[31:24] ==ENEMY6) begin
							enemy6_address <= (buffer_update_address-object_info[23:14])+(vcount-object_info[13:4])*32;
							if (color_enemy600 != 24'd0) update_we <= 1;
							mem_in_update <= {object_info[31:24],color_enemy600[23:16], color_enemy600[15:8], color_enemy600[7:0]};
						end
						else if (object_info[31:24] ==ENEMY7) begin
							enemy7_address <= (buffer_update_address-object_info[23:14])+(vcount-object_info[13:4])*32;
							if (color_enemy700 != 24'd0) update_we <= 1;
							mem_in_update <= {object_info[31:24],color_enemy700[23:16], color_enemy700[15:8], color_enemy700[7:0]};
						end
						else if (object_info[31:24] ==EXPLOSION_BULLET) begin
							explosion_bullet_address <= (buffer_update_address-object_info[23:14])+(vcount-object_info[13:4])*32;
							if (object_info[3:0] == 4'b0000) begin
								if (color_explosion_bullet != 24'd0) update_we <= 1;
								mem_in_update <= {object_info[31:24],color_explosion_bullet[23:16], color_explosion_bullet[15:8], color_explosion_bullet[7:0]};
							end
							else if (object_info[3:0] == 4'b0001) begin
								if (color_explosion_bullet180 != 24'd0) update_we <= 1;
								mem_in_update <= {object_info[31:24],color_explosion_bullet180[23:16], color_explosion_bullet180[15:8], color_explosion_bullet180[7:0]};
							end
						end
				end
            //preapre for next pixel in buffer line
            else if(loop_states == loop_waiting) begin
					update_we <= 0;
             // if (store_counter == 5'd0) buffer_update_address <= buffer_update_address;
            //  else buffer_update_address <= buffer_update_address + 1;
              store_counter <= store_counter + 1;
              loop_states <= loop_writing;
            end
          end
			 else jump_flag1 <= 1;


        if(jump_flag1) begin
          line_buffer_states <= line_buffer_waiting;
          jump_flag1 <=0;
        end
      end
      if (line_buffer_states == line_buffer_waiting) begin
        if(ram_counter == 8'd59) begin
		    if (endOfLine) begin
				 ram_counter <=0;
				 if(line_buffer_select == uA_cB_dC) begin
					buffer_update_address <= 10'd0;
						buffer_clean_address <= 10'd0;
					line_buffer_select <= uB_cC_dA;
				 end
				 else if(line_buffer_select == uB_cC_dA) begin
					buffer_update_address <= 10'd0;
						buffer_clean_address <= 10'd0;
					line_buffer_select <= uC_cA_dB;
				 end
				 else if (line_buffer_select == uC_cA_dB) begin
					buffer_update_address <= 10'd0;
						buffer_clean_address <= 10'd0;
					line_buffer_select <= uA_cB_dC;
				 end
				line_buffer_states <= line_buffer_setting;
				clean_we <= 0;
			end
			else line_buffer_states <= line_buffer_waiting;
        end
        else begin
				ram_counter <= ram_counter + 1;
				line_buffer_states <= line_buffer_setting;
			end
		   update_we <= 0;
      end
    end
		//===========================================================================================================
		//drawing
		//===========================================================================================================
		if (draw_states == draw_waiting) begin
		   buffer_draw_address <= xcoor;
			buffer_clean_address <= xcoor;
			draw_states <= draw_processing;
			clean_we <= 1;
			mem_in_clean <= 32'd0;
		end
		if (draw_states == draw_processing) begin
		   if (xcoor >= 10'd0 && (xcoor <= 10'd96) && (vcount >= 0) && (vcount <= 10'd29)) begin
				lifeword_address <= (xcoor)+(vcount)*97;
				{VGA_R, VGA_G, VGA_B} <= {color_lifeword[23:16], color_lifeword[15:8], color_lifeword[7:0]};
			end
			if (xcoor >= 10'd97 && (xcoor <= 10'd97+10'd31) && (vcount >= 0) && (vcount <= 10'd31)) begin
				number_address <= (xcoor-10'd97)+(vcount)*32;
				if (life == 3'd0) {VGA_R, VGA_G, VGA_B} <= {color_number_zero[23:16], color_number_zero[15:8], color_number_zero[7:0]};
				else if (life == 3'd1) {VGA_R, VGA_G, VGA_B} <= {color_number_one[23:16], color_number_one[15:8], color_number_one[7:0]};
				else if (life == 3'd2) {VGA_R, VGA_G, VGA_B} <= {color_number_two[23:16], color_number_two[15:8], color_number_two[7:0]};
				else if (life == 3'd3) {VGA_R, VGA_G, VGA_B} <= {color_number_three[23:16], color_number_three[15:8], color_number_three[7:0]};
			end
			if (xcoor >= 10'd200 && (xcoor <= 10'd200 + 10'd121) && (vcount >= 0) && (vcount <= 10'd35)) begin
				bombword_address <= (xcoor-10'd200)+(vcount)*121;
				{VGA_R, VGA_G, VGA_B} <= {color_bomb[23:16], color_bomb[15:8], color_bomb[7:0]};
			end
			if (xcoor >= 10'd322 && (xcoor <= 10'd322 + 10'd31) && (vcount >= 0) && (vcount <= 10'd31)) begin
				number_address <= (xcoor-10'd322)+(vcount)*32;
				if (bomb_value == 2'd0) {VGA_R, VGA_G, VGA_B} <= {color_number_zero[23:16], color_number_zero[15:8], color_number_zero[7:0]};
				else if (bomb_value == 2'd1) {VGA_R, VGA_G, VGA_B} <= {color_number_one[23:16], color_number_one[15:8], color_number_one[7:0]};
			end
			
			if (xcoor >= 10'd400 && (xcoor <= 10'd400 + 10'd126) && (vcount >= 0) && (vcount <= 10'd31)) begin
				scoreword_address <= (xcoor-10'd400)+(vcount)*127;
				{VGA_R, VGA_G, VGA_B} <= {color_score[23:16], color_score[15:8], color_score[7:0]};
			end
			
			if (xcoor >= 10'd527 && (xcoor <= 10'd527 + 10'd31) && (vcount >= 0) && (vcount <= 10'd31)) begin
				number_address <= (xcoor-10'd527)+(vcount)*32;
				if (tens == 4'd0) {VGA_R, VGA_G, VGA_B} <= {color_number_zero[23:16], color_number_zero[15:8], color_number_zero[7:0]};
				if (tens == 4'd1) {VGA_R, VGA_G, VGA_B} <= {color_number_one[23:16], color_number_one[15:8], color_number_one[7:0]};
				if (tens == 4'd2) {VGA_R, VGA_G, VGA_B} <= {color_number_two[23:16], color_number_two[15:8], color_number_two[7:0]};
				if (tens == 4'd3) {VGA_R, VGA_G, VGA_B} <= {color_number_three[23:16], color_number_three[15:8], color_number_three[7:0]};
				if (tens == 4'd4) {VGA_R, VGA_G, VGA_B} <= {color_number_four[23:16], color_number_four[15:8], color_number_four[7:0]};
				if (tens == 4'd5) {VGA_R, VGA_G, VGA_B} <= {color_number_five[23:16], color_number_five[15:8], color_number_five[7:0]};
				if (tens == 4'd6) {VGA_R, VGA_G, VGA_B} <= {color_number_six[23:16], color_number_six[15:8], color_number_six[7:0]};
				if (tens == 4'd7) {VGA_R, VGA_G, VGA_B} <= {color_number_seven[23:16], color_number_seven[15:8], color_number_seven[7:0]};
				if (tens == 4'd8) {VGA_R, VGA_G, VGA_B} <= {color_number_eight[23:16], color_number_eight[15:8], color_number_eight[7:0]};
				if (tens == 4'd9) {VGA_R, VGA_G, VGA_B} <= {color_number_nine[23:16], color_number_nine[15:8], color_number_nine[7:0]};
			end
			
			if (xcoor >= 10'd559 && (xcoor <= 10'd559 + 10'd31) && (vcount >= 0) && (vcount <= 10'd31)) begin
				number_address <= (xcoor-10'd559)+(vcount)*32;
				if (digits == 4'd0) {VGA_R, VGA_G, VGA_B} <= {color_number_zero[23:16], color_number_zero[15:8], color_number_zero[7:0]};
				if (digits == 4'd1) {VGA_R, VGA_G, VGA_B} <= {color_number_one[23:16], color_number_one[15:8], color_number_one[7:0]};
				if (digits == 4'd2) {VGA_R, VGA_G, VGA_B} <= {color_number_two[23:16], color_number_two[15:8], color_number_two[7:0]};
				if (digits == 4'd3) {VGA_R, VGA_G, VGA_B} <= {color_number_three[23:16], color_number_three[15:8], color_number_three[7:0]};
				if (digits == 4'd4) {VGA_R, VGA_G, VGA_B} <= {color_number_four[23:16], color_number_four[15:8], color_number_four[7:0]};
				if (digits == 4'd5) {VGA_R, VGA_G, VGA_B} <= {color_number_five[23:16], color_number_five[15:8], color_number_five[7:0]};
				if (digits == 4'd6) {VGA_R, VGA_G, VGA_B} <= {color_number_six[23:16], color_number_six[15:8], color_number_six[7:0]};
				if (digits == 4'd7) {VGA_R, VGA_G, VGA_B} <= {color_number_seven[23:16], color_number_seven[15:8], color_number_seven[7:0]};
				if (digits == 4'd8) {VGA_R, VGA_G, VGA_B} <= {color_number_eight[23:16], color_number_eight[15:8], color_number_eight[7:0]};
				if (digits == 4'd9) {VGA_R, VGA_G, VGA_B} <= {color_number_nine[23:16], color_number_nine[15:8], color_number_nine[7:0]};
			end
			
			if (xcoor >= 10'd0 && (xcoor <= 10'd31) && (vcount >= 80) && (vcount <= 10'd80+10'd259)) begin
				startwarsword_address <= (xcoor)+(vcount-10'd80)*32;
				{VGA_R, VGA_G, VGA_B} <= {color_starwars[23:16], color_starwars[15:8], color_starwars[7:0]};
			end
			if (xcoor >= 10'd32 && (xcoor <= 10'd32+10'd555) && (vcount >= 340) && (vcount <= 10'd340+10'd47)&&game_over) begin
					startword_address <= (xcoor-10'd32)+(vcount-10'd340)*556;
					{VGA_R, VGA_G, VGA_B} <= {color_start[23:16], color_start[15:8], color_start[7:0]};
			end
			
			if ( xcoor >= 10'd32 && xcoor <= 10'd639 && vcount >=32 && !game_over) {VGA_R, VGA_G, VGA_B} <= {mem_out[23:16], mem_out[15:8], mem_out[7:0]};
				draw_states <= draw_waiting;
		end
	 end


  always_comb begin
    mem_out = 32'd0;
    if (line_buffer_select == uA_cB_dC) begin
		mem_out = C_out;
		mem_out_u = A_out;
	 end
    else if (line_buffer_select == uB_cC_dA) begin
		mem_out = A_out;
		mem_out_u = B_out;
	 end
    else begin
		mem_out = B_out;
		mem_out_u = C_out;
	 end
	end
	
	rom_starwarsword starwars (.address(startwarsword_address),.clock(clk50),.q(color_starwars));
   rom_startgameword startword (.address(startword_address),.clock(clk50),.q(color_start));
	rom_scoreword scoreword (.address(scoreword_address),.clock(clk50),.q(color_score));
	rom_bombword bombword(.address(bombword_address),.clock(clk50),.q(color_bomb));
	rom_0 number_zero(.address(number_address),.clock(clk50),.q(color_number_zero));
	rom_1 number_one(.address(number_address),.clock(clk50),.q(color_number_one));
	rom_2 number_two(.address(number_address),.clock(clk50),.q(color_number_two));
	rom_3 number_three(.address(number_address),.clock(clk50),.q(color_number_three));
	rom_4 number_four(.address(number_address),.clock(clk50),.q(color_number_four));
	rom_5 number_five(.address(number_address),.clock(clk50),.q(color_number_five));
	rom_6 number_six(.address(number_address),.clock(clk50),.q(color_number_six));
	rom_7 number_seven(.address(number_address),.clock(clk50),.q(color_number_seven));
	rom_8 number_eight(.address(number_address),.clock(clk50),.q(color_number_eight));
	rom_9 number_nine(.address(number_address),.clock(clk50),.q(color_number_nine));
   rom_lifeword lifeword(.address(lifeword_address),.clock(clk50),.q(color_lifeword));
	rom_spaceship spaceship(.address(space_address),.clock(clk50),.q(color_spaceship));
	rom_background background(.address(background_address),.clock(clk50),.q(color_background));
	rom_bullet1 bullet(.address(bullet_rom_address),.clock(clk50),.q(color_bullet));
	rom_spawn0 spawn0(.address(spawn_address),.clock(clk50),.q(color_spawn0));
	rom_spawn45 spawn45(.address(spawn_address),.clock(clk50),.q(color_spawn45));
	rom_enemy4 enemy400(.address(enemy4_address),.clock(clk50),.q(color_enemy400));
	rom_enemy200 enemy500(.address(enemy5_address),.clock(clk50),.q(color_enemy500));
	rom_enemy6 enemy600(.address(enemy6_address),.clock(clk50),.q(color_enemy600));
	rom_enemy700 enemy700(.address(enemy7_address),.clock(clk50),.q(color_enemy700));
		//////////////////////////////////////////////////fffff
	rom_explosion_bullet explosion_bullet(.address(explosion_bullet_address),.clock(clk50),.q(color_explosion_bullet));
	rom_explosion_bullet180 explosion_bullet180(.address(explosion_bullet_address),.clock(clk50),.q(color_explosion_bullet180));
	rom_explosion_spaceship explosion_spaceship(.address(explosion_spaceship_address),.clock(clk50),.q(color_explosion_spaceship));
	rom_explosion_spaceship1 explosion_spaceship1(.address(explosion_spaceship_address),.clock(clk50),.q(color_explosion_spaceship1));

endmodule // VGA_LED_Emulator
