/*
 * Seven-segment LED emulator
 *
 * Stephen A. Edwards, Columbia University
 *
 */

module VGA_BALL_Emulator(
    	input logic       clk50, reset,
        input logic enable_ram,
        input logic ram_select,
		  input logic en_emulator,//ff
		  input logic [1:0] life,
		  input logic [1:0] bomb_value,
		  input logic [9:0] score,
    	input logic update_done,//game logic says you can work
        input logic [31:0] game_logic_to_emulator,
		output logic draw_done,//tell game logic you finish
		output logic [7:0] VGA_R, VGA_G, VGA_B,
		output [31:0]A,B,C,
		output [31:0]mem_out_u,
		output logic      VGA_CLK, VGA_HS, VGA_VS, VGA_BLANK_n, VGA_SYNC_n);

        logic [23:0] color_spaceship;
        logic [23:0] color_background;
        logic [23:0] color_bullet;
		  logic [23:0] color_spawn0;
		  logic [23:0] color_spawn45;
		  logic [23:0] color_enemy400;
		  logic [23:0] color_enemy500;
		  logic [23:0] color_enemy600;
		  logic [23:0] color_enemy700;
		  logic [23:0] color_explosion_bullet;
		  logic [23:0] color_explosion_bullet180;
		  logic [23:0] color_explosion_spaceship;
		  logic [23:0] color_explosion_spaceship1;

        logic [10:0]    hcount; // Horizontal counter
        logic           endOfLine;

      // Vertical counter
        logic [9:0]     vcount;
        logic           endOfField;

      	logic  [9:0]    xcoor;
      	logic  [9:0]    ycoor;

        logic  [9:0]  space_address;
        logic  [11:0]  background_address;
        logic  [9:0]  bullet_rom_address; //ff bullet is 16*16 pixels
		  logic  [9:0]	 spawn_address;
		  logic  [9:0]	 enemy4_address;
		  logic  [9:0]	 enemy5_address;
		  logic  [9:0]	 enemy6_address;
		  logic  [9:0]	 enemy7_address;
		  logic  [9:0]  explosion_bullet_address;
		  logic  [9:0]  explosion_spaceship_address;
		  logic  		 draw_con;
     //   logic           draw_spaceship;
       // logic           draw_bullet;
		  //logic 				draw_spawn;


  parameter HACTIVE      = 11'd 1280,
            HFRONT_PORCH = 11'd 32,
            HSYNC        = 11'd 192,
            HBACK_PORCH  = 11'd 96,
            HTOTAL       = HACTIVE + HFRONT_PORCH + HSYNC + HBACK_PORCH; //1600

  parameter VACTIVE      = 10'd 480,
            VFRONT_PORCH = 10'd 10,
            VSYNC        = 10'd 2,
            VBACK_PORCH  = 10'd 33,
            VTOTAL       = VACTIVE + VFRONT_PORCH + VSYNC + VBACK_PORCH; //525

parameter 	NO_ID 		=8'd0,
				SPACESHIP	=8'd1,
		      BULLET		=8'd2,
				SPAWN 		=8'd3,
				ENEMY4		=8'd4,
				ENEMY5		=8'd5,
				ENEMY6		=8'd6,
				ENEMY7		=8'd7,
				EXPLOSION_BULLET = 8'd11,
				EXPLOSION_SPACESHIP = 8'd12;



enum logic [2:0] {waiting, processing} states;
enum logic [1:0] {wait_receive_finish,receive_data} states_process;
enum logic [1:0] {line_buffer_waiting, line_buffer_working} line_buffer_states;
enum logic [1:0] {loop_waiting, loop_tmp_waiting, loop_writing} loop_states;
enum logic [1:0] {draw_waiting, draw_processing} draw_states;
//uA_cB_dC : update A, Clear B, draw C
//uB_cC_dA : update B, clear C, draw A
//uC_cA_dB : update C, clear A, draw B
enum logic [2:0] {uA_cB_dC, uB_cC_dA,uC_cA_dB} line_buffer_select;


always_ff @(posedge clk50 or posedge reset) begin
    if (reset) begin
        hcount <= 0;
        vcount <= 0;
        states <= waiting;
        draw_done <= 0;
    end
    else if (states == waiting) begin
        hcount <= 0;
        vcount <= 0;
        draw_done <= 0;
		  //states_process <= receive_data;
        if (update_done == 1'd1) begin //ffff
            states <= processing;
        end
    end

    else if (states == processing) begin
        if(endOfLine) begin
            hcount <= 0;
            if (endOfField) begin
                vcount <= 0;
                draw_done <= 1;
                states <= waiting;
            end
            else begin
                vcount <= vcount + 10'd1;
            end
        end
        else begin
            hcount <= hcount +11'd1;
        end
    end
end

  assign endOfLine = hcount == HTOTAL - 1;
  assign endOfField = vcount == VTOTAL - 1;

  /*
  * 640 X 480 VGA timing for a 50 MHz clock: one pixel every other cycle
  *
  *HCOUNT 1599 0             1279       1599 0
  *            _______________              ________
  * __________|    Video      |____________|  Video
  *
  *
  * |SYNC| BP |<-- HACTIVE -->|FP|SYNC| BP |<-- HACTIVE
  *       _______________________      _____________
  * |____|       VGA_HS          |____|
  */


  // Horizontal sync: from 0x520 to 0x57F
  // 101 0010 0000 to 101 0111 1111
  assign VGA_HS = !( (hcount[10:7] == 4'b1010) & (hcount[6] | hcount[5]));
  assign VGA_VS = !( vcount[9:1] == (VACTIVE + VFRONT_PORCH) / 2);

  assign VGA_SYNC_n = 1; // For adding sync to video signals; not used for VGA

  // Horizontal active: 0 to 1279     Vertical active: 0 to 479
  // 101 0000 0000  1280	       01 1110 0000  480
  // 110 0011 1111  1599	       10 0000 1100  524
  assign VGA_BLANK_n = !( hcount[10] & (hcount[9] | hcount[8]) ) & !( vcount[9] | (vcount[8:5] == 4'b1111) );



    ////fffff
    logic [31:0] ram_a_tmp[0:63];
    logic [31:0] ram_b_tmp[0:63];
    logic [7:0] ram_addr;

    always_ff @(posedge clk50) begin
	     if (states == waiting) begin
				states_process <= receive_data;
		  end
        else if(enable_ram == 1'd1) begin //start to update data from ram
				
					if (states_process == receive_data) begin
						if(en_emulator)begin
								if(ram_select == 1'd0) begin       //select==0, pick ram A, else pick ram B
									 ram_b_tmp[ram_addr] <= game_logic_to_emulator;
								end
								else begin
									 ram_a_tmp[ram_addr] <= game_logic_to_emulator;
								end
								states_process <= wait_receive_finish;
						end
					end
					else if (states_process == wait_receive_finish) begin
							ram_addr <= ram_addr + 8'd1;
							states_process <= receive_data;
					end
        end
        else begin
		      ram_addr <= 8'd0; //next update start from ram_addr = 0;
		  end
    end

  logic we;
  logic A_we;
  logic B_we;
  logic C_we;
  logic update_we;
  logic clean_we;
  logic [8:0] buffer_A_address;
  logic [8:0] buffer_B_address;
  logic [8:0] buffer_C_address;
  logic [8:0] buffer_update_address;
  logic [8:0] buffer_clean_address;
  logic [8:0] buffer_draw_address;
  logic [31:0]mem_in_update;
  logic [31:0]mem_in_clean;
  logic [31:0]mem_in_A;
  logic [31:0]mem_in_B;
  logic [31:0]mem_in_C;
  logic [31:0]mem_out;
  //logic [31:0]mem_out_u;
  logic [31:0]A_out;
  logic [31:0]B_out;
  logic [31:0]C_out;
  logic [7:0] ram_counter;
  assign A = A_out;
  assign B = B_out;
  assign C = C_out;

  line_buffer line_buffer_A(.clk(clk50),.we(A_we),.data_in(mem_in_A),.data_out(A_out),.address(buffer_A_address));
  line_buffer line_buffer_B(.clk(clk50),.we(B_we),.data_in(mem_in_B),.data_out(B_out),.address(buffer_B_address));
  line_buffer line_buffer_C(.clk(clk50),.we(C_we),.data_in(mem_in_C),.data_out(C_out),.address(buffer_C_address));

  always_comb begin
		A_we = 0;
      B_we = 0;
		C_we = 0;
      mem_in_A = 32'd0;
      mem_in_B = 32'd0;
		mem_in_C = 32'd0;
      buffer_A_address = 8'd0;
		buffer_B_address = 8'd0;
		buffer_C_address = 8'd0;
    if (line_buffer_select == uA_cB_dC) begin
      A_we = update_we;
      B_we = clean_we;
      mem_in_A = mem_in_update ;
      mem_in_B = mem_in_clean;
      buffer_A_address = buffer_update_address;
      buffer_B_address = buffer_clean_address;
		buffer_C_address = buffer_draw_address;
    end
    else if (line_buffer_select == uB_cC_dA) begin
      B_we = update_we;
      C_we = clean_we;
      mem_in_B = mem_in_update ;
      mem_in_C = mem_in_clean;
      buffer_B_address = buffer_update_address;
      buffer_C_address = buffer_clean_address;
		buffer_A_address = buffer_draw_address;
    end
    else  if (line_buffer_select == uC_cA_dB) begin
      C_we = update_we;
      A_we = clean_we;
      mem_in_C = mem_in_update ;
      mem_in_A = mem_in_clean;
      buffer_C_address = buffer_update_address;
      buffer_A_address = buffer_clean_address;
		buffer_B_address = buffer_draw_address;
    end

  end
  //line_buffer_a line_buffer_even(.clock(clk50),.address(buffer_even_address),.data(mem_in),.wren(even_we),.q(even_out));
  //line_buffer_a line_buffer_odd(.clock(clk50),.address(buffer_odd_address),.data(mem_in),.wren(odd_we),.q(odd_out));

  assign VGA_CLK = hcount[0]; // 25 MHz clock: pixel latched on rising edge
  assign xcoor = hcount[10:1];
  assign ycoor = vcount;
  assign background_address = xcoor+ycoor*64;
  logic [4:0] store_counter;
  logic jump_flag1;
  //logic jump_flag2;

  always_ff @(posedge clk50) begin
    if (reset) begin
		  
		  
		  ram_counter <= 0;
		  line_buffer_states <= line_buffer_working;
		  draw_states <= draw_waiting;
			loop_states <= loop_writing;
			store_counter <= 0;
			line_buffer_select <= uA_cB_dC;
			//jump_flag2 <= 0;
			jump_flag1 <= 0;
	  end
    else if (states == processing) begin
  		if (line_buffer_states == line_buffer_working) begin
        if(ram_select == 1'd0) begin
          if (vcount - ram_a_tmp[ram_counter][13:4] >= 0 && vcount - ram_a_tmp[ram_counter][13:4] <= 10'd31 ) begin
            // write first pixel in buffer line
            if (loop_states == loop_writing) begin
              mem_in_update <= ram_a_tmp[ram_counter];
              update_we <= 0;
              //if fisrt pixel of the line in a image, use current x aixs.
              if (store_counter == 5'd0) begin
                buffer_update_address <= ram_a_tmp[ram_counter][22:14];
                if (jump_flag1 != 1) loop_states <= loop_tmp_waiting;
              end
              //if the lastpixel of the line in a image, finish, go to fetch next value in ram
              else if(store_counter == 5'd31) begin
                loop_states <= loop_writing;
                jump_flag1 <= 1;
                store_counter <= 0;
              end
              else begin
					 buffer_update_address <= buffer_update_address + 1;
				    loop_states <= loop_tmp_waiting;
				  end

            end
				else if(loop_states == loop_tmp_waiting)begin
					loop_states <= loop_waiting;
				end
            //preapre for next pixel in buffer line
            else if(loop_states == loop_waiting) begin
             // update_we <= 1;
				  if(mem_out_u[31:24]==SPACESHIP || mem_out_u[31:24] == BULLET)begin
						update_we <= 0;
				  end
				  else begin
						update_we <= 1;
				  end
             // if (store_counter == 5'd0) buffer_update_address <= buffer_update_address;
            //  else buffer_update_address <= buffer_update_address + 1;
              store_counter <= store_counter + 1;
              loop_states <= loop_writing;
            end
          end
			 else jump_flag1 <= 1;
		  end
		  
        if(ram_select == 1'd1) begin
          if (vcount - ram_b_tmp[ram_counter][13:4] >= 0 && vcount - ram_b_tmp[ram_counter][13:4] <= 10'd31) begin
            // write first pixel in buffer line
            if (loop_states == loop_writing) begin
              mem_in_update <= ram_b_tmp[ram_counter];
              update_we <= 0;
              //if fisrt pixel of the line in a image, use current x aixs.
              if (store_counter == 5'd0) begin
                buffer_update_address <= ram_b_tmp[ram_counter][22:14];
                if (jump_flag1 != 1) loop_states <= loop_tmp_waiting;
              end
              //if the lastpixel of the line in a image, finish, go to fetch next value in ram
              else if(store_counter == 5'd31) begin
                loop_states <= loop_writing;
                jump_flag1 <= 1;
                store_counter <= 0;
              end
              else begin
						 buffer_update_address <= buffer_update_address + 1;
						 loop_states <= loop_tmp_waiting;
				  end

            end
				else if(loop_states == loop_tmp_waiting)begin
					loop_states <= loop_waiting;
				end
				
            //preapre for next pixel in buffer line
           else if(loop_states == loop_waiting) begin
             // update_we <= 1;
				  if(mem_out_u[31:24] == SPACESHIP||mem_out_u[31:24] == BULLET)begin	//-------debug  jS&xx
						update_we <= 0;
				  end
				  else begin
						update_we <= 1;
				  end
             // if (store_counter == 5'd0) buffer_update_address <= buffer_update_address;
            //  else buffer_update_address <= buffer_update_address + 1;
              store_counter <= store_counter + 1;
              loop_states <= loop_writing;
            end
				
          end
			 else jump_flag1 <= 1;
		  end
        /*if (buffer_clean_address != 9'd511) begin
            clean_we <= 1;
            buffer_clean_address <= buffer_clean_address + 1;
            mem_in_clean <= 32'd0;
        end
        else jump_flag2 <= 1;*/
        if(jump_flag1) begin
          line_buffer_states <= line_buffer_waiting;
          jump_flag1 <=0;
          //jump_flag2 <=0;
        end
      end
      if (line_buffer_states == line_buffer_waiting) begin
        if(ram_counter == 8'd59) begin           //fffff
		    if (endOfLine) begin
				 ram_counter <=0;
				 if(line_buffer_select == uA_cB_dC) begin
					buffer_update_address <= 9'd0;
						buffer_clean_address <= 9'd0;
					line_buffer_select <= uB_cC_dA;
				 end
				 else if(line_buffer_select == uB_cC_dA) begin
					buffer_update_address <= 9'd0;
						buffer_clean_address <= 9'd0;
					line_buffer_select <= uC_cA_dB;
				 end
				 else if (line_buffer_select == uC_cA_dB) begin
					buffer_update_address <= 9'd0;
						buffer_clean_address <= 9'd0;
					line_buffer_select <= uA_cB_dC;
				 end
				line_buffer_states <= line_buffer_working;
				clean_we <= 0;
			end
			else line_buffer_states <= line_buffer_waiting;
        end
        else begin
				ram_counter <= ram_counter + 1;
				line_buffer_states <= line_buffer_working;
			end
		   update_we <= 0;
      end
    end
		//===========================================================================================================
		//drawing
		//===========================================================================================================
		if (draw_states == draw_waiting) begin
		   buffer_draw_address <= xcoor;
			buffer_clean_address <= xcoor;
			draw_states <= draw_processing;
			clean_we <= 1;
		end
		if (draw_states == draw_processing) begin
		{VGA_R, VGA_G, VGA_B} <= {color_background[23:16], color_background[15:8], color_background[7:0]}; // Black	
		if(draw_con)begin
			if (mem_out[31:24] == SPACESHIP && xcoor <= 10'd512) begin
			  space_address <= (xcoor-mem_out[23:14])+(ycoor-mem_out[13:4])*32;
			  if (color_spaceship != 24'd0)begin
				 {VGA_R, VGA_G, VGA_B} <= {color_spaceship[23:16], color_spaceship[15:8], color_spaceship[7:0]};
			  end
			end
			else if (mem_out[31:24] == BULLET && xcoor <= 10'd512 ) begin
			  bullet_rom_address <= (xcoor-mem_out[23:14])+(ycoor-mem_out[13:4])*32;
			  if (color_bullet != 24'd0)begin
				 {VGA_R, VGA_G, VGA_B} <= {color_bullet[23:16], color_bullet[15:8], color_bullet[7:0]};
			  end
			end
			else if (mem_out[31:24] == SPAWN && xcoor <= 10'd512 ) begin
			  spawn_address <= (xcoor-mem_out[23:14]-1)+(ycoor-mem_out[13:4])*32;
			  if (mem_out[3:0] == 4'd0 || mem_out[3:0] == 4'd2) begin
				  if (color_spawn0 != 24'd0)begin
					 {VGA_R, VGA_G, VGA_B} <= {color_spawn0[23:16], color_spawn0[15:8], color_spawn0[7:0]};
				  end
			  end
			  else if (mem_out[3:0] == 4'd1 || mem_out[3:0] == 4'd3) begin 
				  if (color_spawn45 != 24'd0)begin
					 {VGA_R, VGA_G, VGA_B} <= {color_spawn45[23:16], color_spawn45[15:8], color_spawn45[7:0]};
				  end
			  end
			end
			//xxxxxxxxxxxxxxxxxxxxxxx
			else if (mem_out[31:24] == ENEMY4 && xcoor <= 10'd512) begin
				enemy4_address <= (xcoor-mem_out[23:14])+(ycoor-mem_out[13:4])*32;
				if (color_enemy400 != 24'd0)begin
					{VGA_R, VGA_G, VGA_B} <= {color_enemy400[23:16], color_enemy400[15:8], color_enemy400[7:0]};
			  end
			end
			else if (mem_out[31:24] == ENEMY5 && xcoor <= 10'd512) begin
				enemy5_address <= (xcoor-mem_out[23:14])+(ycoor-mem_out[13:4])*32;
				if (color_enemy500 != 24'd0)begin
					{VGA_R, VGA_G, VGA_B} <= {color_enemy500[23:16], color_enemy500[15:8], color_enemy500[7:0]};
			  end
			end
			else if (mem_out[31:24] == ENEMY6 && xcoor <= 10'd512) begin
				enemy6_address <= (xcoor-mem_out[23:14])+(ycoor-mem_out[13:4])*32;
				if (color_enemy600 != 24'd0)begin
					{VGA_R, VGA_G, VGA_B} <= {color_enemy600[23:16], color_enemy600[15:8], color_enemy600[7:0]};
			  end
			end
			else if (mem_out[31:24] == ENEMY7 && xcoor <= 10'd512) begin
				enemy7_address <= (xcoor-mem_out[23:14])+(ycoor-mem_out[13:4])*32;
				if (color_enemy700 != 24'd0)begin
					{VGA_R, VGA_G, VGA_B} <= {color_enemy700[23:16], color_enemy700[15:8], color_enemy700[7:0]};
			  end
			end
			
			else if (mem_out[31:24] == EXPLOSION_BULLET && xcoor <= 10'd512) begin
				explosion_bullet_address <= (xcoor-mem_out[23:14])+(ycoor-mem_out[13:4])*32;
				if (mem_out[3:0] == 4'b0000) begin
					if (color_explosion_bullet != 24'd0)begin
						{VGA_R, VGA_G, VGA_B} <= {color_explosion_bullet[23:16], color_explosion_bullet[15:8], color_explosion_bullet[7:0]};
					end
				end
				else if (mem_out[3:0] == 4'b0001) begin
					if (color_explosion_bullet180 != 24'd0)begin
						 {VGA_R, VGA_G, VGA_B} <= {color_explosion_bullet180[23:16], color_explosion_bullet180[15:8], color_explosion_bullet180[7:0]};
					end
				end
			end
			
			//xxxxxxxxxxxxxxxxxxxxxxx
		end
		  draw_states <= draw_waiting;
		end
	 end

	 
  always_comb begin
    mem_out = 32'd0;
    if (line_buffer_select == uA_cB_dC) begin 
		mem_out = C_out;
		mem_out_u = A_out;
	 end
    else if (line_buffer_select == uB_cC_dA) begin
		mem_out = A_out;
		mem_out_u = B_out;
	 end
    else begin
		mem_out = B_out;
		mem_out_u = C_out;
	 end
	 // Only satisfying the following condition may begin to draw the objects
	 		//ff
		 if(xcoor>=mem_out[23:14] && (xcoor <= mem_out[23:14]+10'd31) && (ycoor >= mem_out[13:4])&& (ycoor<=mem_out[13:4]+10'd31))begin
			draw_con = 1;
			end
		 else draw_con = 0;
	
	 end
  



   rom_spaceship spaceship(.address(space_address),.clock(clk50),.q(color_spaceship));
	rom_background background(.address(background_address),.clock(clk50),.q(color_background));
	rom_bullet1 bullet(.address(bullet_rom_address),.clock(clk50),.q(color_bullet));
	rom_spawn0 spawn0(.address(spawn_address),.clock(clk50),.q(color_spawn0));
	rom_spawn45 spawn45(.address(spawn_address),.clock(clk50),.q(color_spawn45));
	rom_enemy4 enemy400(.address(enemy4_address),.clock(clk50),.q(color_enemy400));
	rom_enemy200 enemy500(.address(enemy5_address),.clock(clk50),.q(color_enemy500));
	rom_enemy6 enemy600(.address(enemy6_address),.clock(clk50),.q(color_enemy600));
	rom_enemy700 enemy700(.address(enemy7_address),.clock(clk50),.q(color_enemy700));
		//////////////////////////////////////////////////fffff
	rom_explosion_bullet explosion_bullet(.address(explosion_bullet_address),.clock(clk50),.q(color_explosion_bullet));
	rom_explosion_bullet180 explosion_bullet180(.address(explosion_bullet_address),.clock(clk50),.q(color_explosion_bullet180));
	rom_explosion_spaceship explosion_spaceship(.address(explosion_spaceship_address),.clock(clk50),.q(color_explosion_spaceship));
	rom_explosion_spaceship1 explosion_spaceship1(.address(explosion_spaceship_address),.clock(clk50),.q(color_explosion_spaceship1));

endmodule // VGA_LED_Emulator
