/*
 * Parking Robot Team
 * Columbia University
 */

#include <linux/module.h>
#include <linux/init.h>
#include <linux/errno.h>
#include <linux/version.h>
#include <linux/kernel.h>
#include <linux/platform_device.h>
#include <linux/miscdevice.h>
#include <linux/slab.h>
#include <linux/io.h>
#include <linux/of.h>
#include <linux/of_address.h>
#include <linux/fs.h>
#include <linux/uaccess.h>
#include "ultrasonic_sensor.h"
#include <linux/types.h>

#define DRIVER_NAME "ultrasonic_sensor"

/*
 * Information about our device
 */
struct ultrasonic_sensor_dev {
	struct resource res; /* Resource: our registers */
	void __iomem *virtbase; /* Where registers can be accessed in memory */
	//s_motor_t s_motor_val;
} dev;

/*
 * Write position
 * Assumes position is in range and the device information has been set up
 */
static __u32 read_distance(us_position_t pos)
{
	return ioread32(dev.virtbase + pos*4);
}

/*
 * Handle ioctl() calls from userspace:
 * Write position to peripheral.
 * Note extensive error checking of arguments
 */
static long ultrasonic_sensor_ioctl(struct file *f, unsigned int cmd, unsigned long arg)
{
	ultrasonic_t us;
	

	switch (cmd) {
	case ULTRASONIC_GET_DIST_WITH_POS:
		if (copy_from_user(&us, (ultrasonic_t *) arg, sizeof(ultrasonic_t)))
			return -EACCES;
		//check whether in range. Idealy it's already checked by compiler, but
		//in case compiler has different behavior for enum default values or 
		//someone try to assign number for enum manually in .h file
		if(us.pos < 0 || us.pos >= NUM_OF_US)
			return -EINVAL;

		us.distance = read_distance(us.pos);
		if (copy_to_user((ultrasonic_t *) arg, &us, sizeof(ultrasonic_t)))
			return -EACCES;
		
		break;

	default:
		return -EINVAL;
	}

	return 0;
}

/* The operations our device knows how to do */
static const struct file_operations ultrasonic_sensor_fops = {
	.owner		= THIS_MODULE,
	.unlocked_ioctl = ultrasonic_sensor_ioctl,
};

/* Information about our device for the "misc" framework -- like a char dev */
static struct miscdevice ultrasonic_sensor_misc_device = {
	.minor		= MISC_DYNAMIC_MINOR,
	.name		= DRIVER_NAME,
	.fops		= &ultrasonic_sensor_fops,
};

/*
 * Initialization code: get resources (registers) and display
 * a welcome message
 */
static int __init ultrasonic_sensor_probe(struct platform_device *pdev)
{
	int ret;

	/* Register ourselves as a misc device: creates /dev/ultrasonic_sensor */
	ret = misc_register(&ultrasonic_sensor_misc_device);

	/* Get the address of our registers from the device tree */
	ret = of_address_to_resource(pdev->dev.of_node, 0, &dev.res);
	if (ret) {
		ret = -ENOENT;
		goto out_deregister;
	}

	/* Make sure we can use these registers */
	if (request_mem_region(dev.res.start, resource_size(&dev.res),
			       DRIVER_NAME) == NULL) {
		ret = -EBUSY;
		goto out_deregister;
	}

	/* Arrange access to our registers */
	dev.virtbase = of_iomap(pdev->dev.of_node, 0);
	if (dev.virtbase == NULL) {
		ret = -ENOMEM;
		goto out_release_mem_region;
	}

	
	return 0;

out_release_mem_region:
	release_mem_region(dev.res.start, resource_size(&dev.res));
out_deregister:
	misc_deregister(&ultrasonic_sensor_misc_device);
	return ret;
}

/* Clean-up code: release resources */
static int ultrasonic_sensor_remove(struct platform_device *pdev)
{
	iounmap(dev.virtbase);
	release_mem_region(dev.res.start, resource_size(&dev.res));
	misc_deregister(&ultrasonic_sensor_misc_device);
	return 0;
}

/* Which "compatible" string(s) to search for in the Device Tree */
#ifdef CONFIG_OF
static const struct of_device_id ultrasonic_sensor_of_match[] = {
	{ .compatible = "altr,ultrasonic_sensor" },
	{},
};
MODULE_DEVICE_TABLE(of, ultrasonic_sensor_of_match);
#endif

/* Information for registering ourselves as a "platform" driver */
static struct platform_driver ultrasonic_sensor_driver = {
	.driver	= {
		.name	= DRIVER_NAME,
		.owner	= THIS_MODULE,
		.of_match_table = of_match_ptr(ultrasonic_sensor_of_match),
	},
	.remove	= __exit_p(ultrasonic_sensor_remove),
};

/* Calball when the module is loaded: set things up */
static int __init ultrasonic_sensor_init(void)
{
	pr_info(DRIVER_NAME ": init\n");
	return platform_driver_probe(&ultrasonic_sensor_driver, ultrasonic_sensor_probe);
}

/* Calball when the module is unloaded: release resources */
static void __exit ultrasonic_sensor_exit(void)
{
	platform_driver_unregister(&ultrasonic_sensor_driver);
	pr_info(DRIVER_NAME ": exit\n");
}

module_init(ultrasonic_sensor_init);
module_exit(ultrasonic_sensor_exit);

MODULE_LICENSE("GPL");
MODULE_AUTHOR("Stephen A. Edwards, Columbia University");
MODULE_DESCRIPTION("VGA 7-segment ball Emulator");
