/*
 * Parking Robot Team
 * Columbia University
 */

#include <linux/module.h>
#include <linux/init.h>
#include <linux/errno.h>
#include <linux/version.h>
#include <linux/kernel.h>
#include <linux/platform_device.h>
#include <linux/miscdevice.h>
#include <linux/slab.h>
#include <linux/io.h>
#include <linux/of.h>
#include <linux/of_address.h>
#include <linux/fs.h>
#include <linux/uaccess.h>
#include "steering_motor.h"
#include <linux/types.h>

#define DRIVER_NAME "steering_motor"

/*
 * Information about our device
 */
struct steering_motor_dev {
	struct resource res; /* Resource: our registers */
	void __iomem *virtbase; /* Where registers can be accessed in memory */
	s_motor_t s_motor_val; /* Current position of the ball, is useless unless we want to read it */
} dev;

/*
 * Write position
 * Assumes position is in range and the device information has been set up
 */
static void write_dir(s_direction_t s_dir)
{
	u32 writedata;
	if(s_dir == RIGHT)
		writedata = 0;
	else
		writedata = 1;

	iowrite32(writedata, dev.virtbase);
	dev.s_motor_val.s_dir = s_dir;
}

static void write_angle(__u32 angle)
{
	iowrite32(angle, dev.virtbase + 4);
	dev.s_motor_val.angle = angle;
}

/*
 * Handle ioctl() calls from userspace:
 * Write position to peripheral.
 * Note extensive error checking of arguments
 */
static long steering_motor_ioctl(struct file *f, unsigned int cmd, unsigned long arg)
{
	s_direction_t s_dir;
	__u32 angle;
	

	switch (cmd) {
	case S_MOTOR_SET_DIR:
		if (copy_from_user(&s_dir, (s_direction_t *) arg, sizeof(s_direction_t)))
			return -EACCES;
		//check whether it's in valid range
		if (s_dir != RIGHT && s_dir != LEFT) //shouldn't happen actually, but in case
			return -EINVAL;
		write_dir(s_dir);
		break;

	case S_MOTOR_SET_ANGLE:
		if (copy_from_user(&angle, (__u32 *) arg, sizeof(__u32)))
			return -EACCES;
		//check whether it's in valid range
		if (angle < MIN_ANGLE_LV || angle > MAX_ANGLE_LV) //shouldn't happen actually, but in case
			return -EINVAL;
		write_angle(angle);
		break;

	case S_MOTOR_GET_DIR:
		if (copy_to_user((s_direction_t *) arg, &dev.s_motor_val.s_dir, sizeof(s_direction_t)))
			return -EACCES;
		break;

	case S_MOTOR_GET_ANGLE:
		if (copy_to_user((__u32 *) arg, &dev.s_motor_val.angle, sizeof(__u32)))
			return -EACCES;
		break;

	default:
		return -EINVAL;
	}

	return 0;
}

/* The operations our device knows how to do */
static const struct file_operations steering_motor_fops = {
	.owner		= THIS_MODULE,
	.unlocked_ioctl = steering_motor_ioctl,
};

/* Information about our device for the "misc" framework -- like a char dev */
static struct miscdevice steering_motor_misc_device = {
	.minor		= MISC_DYNAMIC_MINOR,
	.name		= DRIVER_NAME,
	.fops		= &steering_motor_fops,
};

/*
 * Initialization code: get resources (registers) and display
 * a welcome message
 */
static int __init steering_motor_probe(struct platform_device *pdev)
{
	int ret;

	/* Register ourselves as a misc device: creates /dev/steering_motor */
	ret = misc_register(&steering_motor_misc_device);

	/* Get the address of our registers from the device tree */
	ret = of_address_to_resource(pdev->dev.of_node, 0, &dev.res);
	if (ret) {
		ret = -ENOENT;
		goto out_deregister;
	}

	/* Make sure we can use these registers */
	if (request_mem_region(dev.res.start, resource_size(&dev.res),
			       DRIVER_NAME) == NULL) {
		ret = -EBUSY;
		goto out_deregister;
	}

	/* Arrange access to our registers */
	dev.virtbase = of_iomap(pdev->dev.of_node, 0);
	if (dev.virtbase == NULL) {
		ret = -ENOMEM;
		goto out_release_mem_region;
	}

	
	return 0;

out_release_mem_region:
	release_mem_region(dev.res.start, resource_size(&dev.res));
out_deregister:
	misc_deregister(&steering_motor_misc_device);
	return ret;
}

/* Clean-up code: release resources */
static int steering_motor_remove(struct platform_device *pdev)
{
	iounmap(dev.virtbase);
	release_mem_region(dev.res.start, resource_size(&dev.res));
	misc_deregister(&steering_motor_misc_device);
	return 0;
}

/* Which "compatible" string(s) to search for in the Device Tree */
#ifdef CONFIG_OF
static const struct of_device_id steering_motor_of_match[] = {
	{ .compatible = "altr,steering_motor" },
	{},
};
MODULE_DEVICE_TABLE(of, steering_motor_of_match);
#endif

/* Information for registering ourselves as a "platform" driver */
static struct platform_driver steering_motor_driver = {
	.driver	= {
		.name	= DRIVER_NAME,
		.owner	= THIS_MODULE,
		.of_match_table = of_match_ptr(steering_motor_of_match),
	},
	.remove	= __exit_p(steering_motor_remove),
};

/* Calball when the module is loaded: set things up */
static int __init steering_motor_init(void)
{
	pr_info(DRIVER_NAME ": init\n");
	return platform_driver_probe(&steering_motor_driver, steering_motor_probe);
}

/* Calball when the module is unloaded: release resources */
static void __exit steering_motor_exit(void)
{
	platform_driver_unregister(&steering_motor_driver);
	pr_info(DRIVER_NAME ": exit\n");
}

module_init(steering_motor_init);
module_exit(steering_motor_exit);

MODULE_LICENSE("GPL");
MODULE_AUTHOR("Stephen A. Edwards, Columbia University");
MODULE_DESCRIPTION("VGA 7-segment ball Emulator");
