/*
 * Parking Robot Team
 * Columbia University
 */

#include <linux/module.h>
#include <linux/init.h>
#include <linux/errno.h>
#include <linux/version.h>
#include <linux/kernel.h>
#include <linux/platform_device.h>
#include <linux/miscdevice.h>
#include <linux/slab.h>
#include <linux/io.h>
#include <linux/of.h>
#include <linux/of_address.h>
#include <linux/fs.h>
#include <linux/uaccess.h>
#include "regular_motor.h"
#include <linux/types.h>

#define DRIVER_NAME "regular_motor"

/*
 * Information about our device
 */
struct regular_motor_dev {
	struct resource res; /* Resource: our registers */
	void __iomem *virtbase; /* Where registers can be accessed in memory */
	r_motor_t r_motor_val; /* Current position of the ball, is useless unless we want to read it */
} dev;

/*
 * Write position
 * Assumes position is in range and the device information has been set up
 */
static void write_dir(r_direction_t r_dir)
{
	u32 writedata;
	if(r_dir == FORWARD)
		writedata = 0;
	else
		writedata = 1;

	iowrite32(writedata, dev.virtbase);
	dev.r_motor_val.r_dir = r_dir;
}

static void write_speed(__u32 speed)
{
	iowrite32(speed, dev.virtbase + 4);
	dev.r_motor_val.speed = speed;
}

/*
 * Handle ioctl() calls from userspace:
 * Write position to peripheral.
 * Note extensive error checking of arguments
 */
static long regular_motor_ioctl(struct file *f, unsigned int cmd, unsigned long arg)
{
	r_direction_t r_dir;
	__u32 speed;
	

	switch (cmd) {
	case R_MOTOR_SET_DIR:
		if (copy_from_user(&r_dir, (r_direction_t *) arg, sizeof(r_direction_t)))
			return -EACCES;
		//check whether it's in valid range
		if (r_dir != FORWARD && r_dir != BACKWARD) //shouldn't happen actually, but in case
			return -EINVAL;
		write_dir(r_dir);
		break;

	case R_MOTOR_SET_SPEED:
		if (copy_from_user(&speed, (__u32 *) arg, sizeof(__u32)))
			return -EACCES;
		//check whether it's in valid range
		if (speed < MIN_SPEED_LV || speed > MAX_SPEED_LV) //shouldn't happen actually, but in case
			return -EINVAL;
		write_speed(speed);
		break;
	case R_MOTOR_GET_DIR:
		if (copy_to_user((r_direction_t *) arg, &dev.r_motor_val.r_dir, sizeof(r_direction_t)))
			return -EACCES;
		break;

	case R_MOTOR_GET_SPEED:
		if (copy_to_user((__u32 *) arg, &dev.r_motor_val.speed, sizeof(__u32)))
			return -EACCES;
		break;

	default:
		return -EINVAL;
	}

	return 0;
}

/* The operations our device knows how to do */
static const struct file_operations regular_motor_fops = {
	.owner		= THIS_MODULE,
	.unlocked_ioctl = regular_motor_ioctl,
};

/* Information about our device for the "misc" framework -- like a char dev */
static struct miscdevice regular_motor_misc_device = {
	.minor		= MISC_DYNAMIC_MINOR,
	.name		= DRIVER_NAME,
	.fops		= &regular_motor_fops,
};

/*
 * Initialization code: get resources (registers) and display
 * a welcome message
 */
static int __init regular_motor_probe(struct platform_device *pdev)
{
	int ret;

	/* Register ourselves as a misc device: creates /dev/regular_motor */
	ret = misc_register(&regular_motor_misc_device);

	/* Get the address of our registers from the device tree */
	ret = of_address_to_resource(pdev->dev.of_node, 0, &dev.res);
	if (ret) {
		ret = -ENOENT;
		goto out_deregister;
	}

	/* Make sure we can use these registers */
	if (request_mem_region(dev.res.start, resource_size(&dev.res),
			       DRIVER_NAME) == NULL) {
		ret = -EBUSY;
		goto out_deregister;
	}

	/* Arrange access to our registers */
	dev.virtbase = of_iomap(pdev->dev.of_node, 0);
	if (dev.virtbase == NULL) {
		ret = -ENOMEM;
		goto out_release_mem_region;
	}

	
	return 0;

out_release_mem_region:
	release_mem_region(dev.res.start, resource_size(&dev.res));
out_deregister:
	misc_deregister(&regular_motor_misc_device);
	return ret;
}

/* Clean-up code: release resources */
static int regular_motor_remove(struct platform_device *pdev)
{
	iounmap(dev.virtbase);
	release_mem_region(dev.res.start, resource_size(&dev.res));
	misc_deregister(&regular_motor_misc_device);
	return 0;
}

/* Which "compatible" string(s) to search for in the Device Tree */
#ifdef CONFIG_OF
static const struct of_device_id regular_motor_of_match[] = {
	{ .compatible = "altr,regular_motor" },
	{},
};
MODULE_DEVICE_TABLE(of, regular_motor_of_match);
#endif

/* Information for registering ourselves as a "platform" driver */
static struct platform_driver regular_motor_driver = {
	.driver	= {
		.name	= DRIVER_NAME,
		.owner	= THIS_MODULE,
		.of_match_table = of_match_ptr(regular_motor_of_match),
	},
	.remove	= __exit_p(regular_motor_remove),
};

/* Calball when the module is loaded: set things up */
static int __init regular_motor_init(void)
{
	pr_info(DRIVER_NAME ": init\n");
	return platform_driver_probe(&regular_motor_driver, regular_motor_probe);
}

/* Calball when the module is unloaded: release resources */
static void __exit regular_motor_exit(void)
{
	platform_driver_unregister(&regular_motor_driver);
	pr_info(DRIVER_NAME ": exit\n");
}

module_init(regular_motor_init);
module_exit(regular_motor_exit);

MODULE_LICENSE("GPL");
MODULE_AUTHOR("Stephen A. Edwards, Columbia University");
MODULE_DESCRIPTION("VGA 7-segment ball Emulator");
