/*
 * Parking Robot Team
 * Columbia University
 */


#include "parameters.h"
#include "driver_functions.h"
#include "parallel_parking.h"
#include "garage_parking.h"
#include "independent_parking.h"

#include <stdio.h>
#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <string.h>
#include <unistd.h>
#include <time.h>
#include <pthread.h>
#include <stdlib.h>
#include <errno.h>



//for data collect thread
typedef struct{
	char isRecording;
	int timeToRecord;
} data_collector_t;





//============= Function declaration ===================
//Check whether is in specified distance, and adjust servo motor if not
void keep_in_dist(const float keep_dist, const float keep_dist_max, const float keep_dist_min,
				  const float tolerable_delta_f_b, 
				  const float us_R_dist, const float us_R_reverse);


//====== thread functions ======
void *data_collect_thread_f(void *);
void *auto_parking_thread_f(void *);
//==============================

int main()
{
  //=========== threads ===========
  /* Used to collect data */
  pthread_t data_collect_thread ;
  pthread_t auto_parking_thread;
  //===============================

  

  r_direction_t r_dir;
  s_direction_t s_dir;
  __u32 speed, angle, distance;
  char cmd;
  int tmpDir;
  int timeToRecord;
  int parking_mode;
  data_collector_t data_collector_arg;

  int thread_condition;

  char had_auto_parking_created = FALSE;
  char quit = FALSE;

  printf("Parking Robot program started\n");

  if(!open_drivers()){
  	puts("Open drivers failed! Program stopped.");
  	return -1;
  }

  //initlize(Maybe should put some in driver in init)
  //ball_pos.x = X_RANGE/2;
  //ball_pos.y = Y_RANGE/2;
  r_dir = FORWARD;
  speed = 0;
  write_r_motor_dir(r_dir);  
  write_r_motor_speed(speed);
  s_dir = RIGHT;
  angle = 0;
  write_s_motor_dir(s_dir);  
  write_s_motor_angle(angle);
  data_collector_arg.isRecording = FALSE;

  //start to cmd loop
  while(!quit) {

    //usleep(20000); //controll how fast the ball runs, lower value -> run faster
    puts("");
    puts(SEPERATE_BAR_1);
    printf("What to do?\tr: r_motor\ts: s_motor\tu: ultrasonic\td: record_data\na: auto_parking\tp: parallel_parking\tg: garage_parking\ni:status\te: emergency_stop\tq: quit \n");
    scanf ("%c", &cmd);
	switch (cmd){
		//Emergemcy stop (set speed to 0, but all other states remains)
		case 'e':
			write_r_motor_speed(0);
			break;

		//start a thread to colloect data for n sec, at most 30sec
		case 'd':
			puts("");
			printf("Please enter how many seconds you want to record data(at most 100sec):\n");
			scanf("%d", &timeToRecord);
			if(!data_collector_arg.isRecording){
				if(timeToRecord <= 0 || timeToRecord > 100){
					puts("Invalid value. It should be 0~100.");
					continue;
				}
				data_collector_arg.isRecording = TRUE;
				data_collector_arg.timeToRecord =  timeToRecord;
				pthread_create(&data_collect_thread, NULL, data_collect_thread_f, (void *)&data_collector_arg);
			}else{
				printf("It's already recording, please try again later.\n");
			}
			break;
		case 'r':
			puts("");
			printf("Please enter direction(0: FORWARD, 1: BACKWARD) and speed(0~8).\n");
			scanf("%d %d", &tmpDir, &speed);
			r_dir = (tmpDir == 0) ? FORWARD : BACKWARD;
			write_r_motor_dir(r_dir);  
            write_r_motor_speed(speed);
			break;
		case 's':
			puts("");
			printf("Please enter direction(0: RIGHT, 1: LEFT) and angle(0~7).\n");
			scanf("%d %d", &tmpDir, &angle);
			s_dir = (tmpDir == 0) ? RIGHT : LEFT;
			write_s_motor_dir(s_dir);  
            write_s_motor_angle(angle);
			break;

		//fire up/cancel auto parking mechanism
		case 'a':
			puts("");
			
			if(!had_auto_parking_created){
				printf("Please enter the parking mode (0: go_along_wall, 1: parallel_parking):\n");
				scanf("%d", &parking_mode);
				if(parking_mode < 0 || parking_mode > 1){
					puts("Invalid value. It should be 0~1.");
					continue;
				}

				puts("Starting auto parking...");
				pthread_create(&auto_parking_thread, NULL, auto_parking_thread_f, (void *)&parking_mode);
				had_auto_parking_created = TRUE;				

			}else{
				//used to check wheter the thread is running
				thread_condition = pthread_kill(auto_parking_thread, 0);
				if(thread_condition == EINVAL){ //shouldn't occur indeed
					
					puts("Invalid signal for pthread_kill");

				}else if(thread_condition == ESRCH){
					printf("Please enter the parking mode (0: go_along_wall, 1: parallel_parking):\n");
					scanf("%d", &parking_mode);
					if(parking_mode < 0 || parking_mode > 1){
						puts("Invalid value. It should be 0~1.");
						continue;
					}

					puts("Starting auto parking...");
					pthread_create(&auto_parking_thread, NULL, auto_parking_thread_f, (void *)&parking_mode);

				}else{

					puts("Trying to stop auto parking ...");
					/* Terminate the auto_parking_thread */
	 			 	pthread_cancel(auto_parking_thread);
	  				/* Wait for the auto_parking_thread to finish */
	  				pthread_join(auto_parking_thread, NULL);
					puts("Stopped");
				}
            }
			break;

		case 'g':
			garage_parking();
			break;

		case 'p':
			parallel_parking_independent();
			break;

		//print out all info
		case 'i':
			puts("");
			if(!had_auto_parking_created){

				printf("Auto parking status: off\n");

			}else{
				
				thread_condition = pthread_kill(auto_parking_thread, 0);
			
				if(thread_condition == ESRCH){

					printf("Auto parking status: off\n");

				}else if(thread_condition == EINVAL){

					printf("Auto parking status: unknown\n");

				}else{

					printf("Auto parking status: on, mode: %d\n", parking_mode);
					
				}


			}
			

			printf("Data recording status: %s\n", data_collector_arg.isRecording ? "on" : "off" );

			printf("Regular motor: dir is %s, speed is %d\n", (read_r_motor_dir()==FORWARD)?"FORWARD":"BACKWARD",
																read_r_motor_speed());

			printf("Steering motor: dir is %s, angle is %d\n", (read_s_motor_dir()==RIGHT)?"RIGHT":"LEFT",
																read_s_motor_angle());
		case 'u':
			puts("");
			printf("Ultrasonic FR distance: %.2f cm\n", cycles_to_cm(read_ultrasonic_distance(FR)));
			printf("Ultrasonic RF distance: %.2f cm\n", cycles_to_cm(read_ultrasonic_distance(RF)));
			printf("Ultrasonic RB distance: %.2f cm\n", cycles_to_cm(read_ultrasonic_distance(RB)));
			printf("Ultrasonic BR distance: %.2f cm\n", cycles_to_cm(read_ultrasonic_distance(BR)));
			
			break;
		
		case 'q':
			quit = TRUE;
			break;

		default:;


	}
    

  }
  
  if(had_auto_parking_created){
			
	thread_condition = pthread_kill(auto_parking_thread, 0);

	if(thread_condition != ESRCH && thread_condition != EINVAL){
		puts("Trying to stop auto parking ...");
		/* Terminate the auto_parking_thread */
	 	pthread_cancel(auto_parking_thread);
		/* Wait for the auto_parking_thread to finish */
		pthread_join(auto_parking_thread, NULL);
		puts("Stopped");
	}
  }
  //Todo: close data collect thread
  write_r_motor_speed(0);
  write_s_motor_angle(0);
  printf("Parking robot terminated\n");
  return 0;
}

//data collect thread function
//Todo : add pthread_cleanup_push and pop when have time
void *data_collect_thread_f(void *arg)
{

  time_t start_time, end_time, tmp_time;
  data_collector_t *data_collector_arg;
  FILE *us_FR, *us_RF, *us_RB, *us_BR;  

  data_collector_arg = (data_collector_t *)arg;

  //Get current time
  start_time = time(0);
  
  puts(SEPERATE_BAR_2);
  printf("Start to record data at ");
  puts(SEPERATE_BAR_2);
  printf(ctime(&start_time));
 
  //open files
  us_FR = fopen("us_FR.txt","w+");
  us_RF = fopen("us_RF.txt","w+");
  us_RB = fopen("us_RB.txt","w+");
  us_BR = fopen("us_BR.txt","w+");
  if(!us_FR){
	puts("Can't open us_FR.txt");
	goto end_thread;
  }
  if(!us_RF){
	puts("Can't open us_RF.txt");
	goto end_thread;
  }
  if(!us_RB){
	puts("Can't open us_RB.txt");
	goto end_thread;
  }
  if(!us_BR){
	puts("Can't open us_BR.txt");
	goto end_thread;
  }


  while(difftime(time(NULL), start_time) <= data_collector_arg->timeToRecord){
	tmp_time = time(0);

    fprintf(us_FR, "%.2f\t%s", cycles_to_cm(read_ultrasonic_distance(FR)), ctime(&tmp_time));
	fprintf(us_RF, "%.2f\t%s", cycles_to_cm(read_ultrasonic_distance(RF)), ctime(&tmp_time));
	fprintf(us_RB, "%.2f\t%s", cycles_to_cm(read_ultrasonic_distance(RB)), ctime(&tmp_time));
	fprintf(us_BR, "%.2f\t%s", cycles_to_cm(read_ultrasonic_distance(BR)), ctime(&tmp_time));
	
	//fprintf(us_FR, "%.2f\n", (float)read_ultrasonic_distance(FR) * 340 * 100 /2 / 50000000);
	//fprintf(us_RF, "%.2f\n", (float)read_ultrasonic_distance(RF) * 340 * 100 /2 / 50000000);
	//fprintf(us_RB, "%.2f\n", (float)read_ultrasonic_distance(RB) * 340 * 100 /2 / 50000000);
	//fprintf(us_BR, "%.2f\n", (float)read_ultrasonic_distance(BR) * 340 * 100 /2 / 50000000);

    //sleep for specified time
    usleep(ULTRASONICS_READ_INTERVAL);
  }


  end_time = time(0);
  puts(SEPERATE_BAR_2);
  printf("Finished recording at ");
  puts(SEPERATE_BAR_2);
  printf(ctime(&end_time));

end_thread:
  if(us_FR)
  	fclose(us_FR);
  if(us_RF)
  	fclose(us_RF);
  if(us_RB)  
	fclose(us_RB);
  if(us_BR)
  	fclose(us_BR);
  data_collector_arg->isRecording = FALSE;
  return NULL;
}

//used to auto parking
void *auto_parking_thread_f(void *arg)
{
	puts(SEPERATE_BAR_2);
	puts("Auto parking started");
	puts(SEPERATE_BAR_2);

	//keep the certain distance from wall
	static const float short_dist = 10;
	//need a tolerate range since ultrasonic value is not always exactly the same 
	static const float tolerable_range = 0.25;
	//max and min	(need to be modify according to above values)
	static const float short_dist_max = 10.25;
	static const float short_dist_min = 9.75;

	//define what value can be said it's a obstacle in long 
	//area.
	static const float gap_leng = 3;
	//define after how many cycle we say we see a gap
	static const char gap_cons = 3;

	//indicate the longest distance that the us can receive
	static const float largest_dist = 100;

    //define what is large distance that should change area;
	static const float long_dist = 30;
	//determin after how many consecutive long dist detection
	//will we say that is in long dist area
	static const char long_dist_cons = 3;
	//same for short distance
	static const char short_dist_cons = 3;	

	//tolerable us_RF, RB difference
	//(should be < tolerable_range*2 , in general I think)
	static const float tolerable_delta_f_b = 0.2;

	//define whether us_RF and us_RB are both in short or long
	//static const float in_same_area_diff = 5;

	//determine when to turn left
	static const float turn_left_dist = 25;

	//current long dist value that should keep
	float cur_long_dist, cur_long_dist_max, cur_long_dist_min;

	//indicate how long want to keep from wall currently, could be short or long 
	float keep_dist, keep_dist_max, keep_dist_min;
	
	//indicate in what times of the detection of the issue that when 
	//ultrasonic sensor is too close the value will become the maximum
	//when short distant -> long distant, this will start counting.
	//This machenism can also avoid small hole/glitch
	char long_dist_counter = 0;
	char short_dist_counter = 0;

	//the 0 will be the newest value, and 1 is one time prior to 0, and so on
	//float us_RF[val_count] = {0};
    //float us_RB[val_count] = {0};
	float us_RF, us_RB, us_FR, us_BR;
	//could be us_RF or us_RB, depend on which direction is going	
	float us_R_dist, us_R_reverse;

	//last value before seeing a gap in long area
	float last_long_dist_bef_gap;
	float delta_cur_last_long_dist;
	char gap_counter = 0;

	//
	float last_short_dist_bef_in_long_area[3] = {short_dist};

	//could be FR or BR
	float us_FB_dist;

	//indicate whehter really going into long distnat area
	char is_in_long_dist_area = FALSE;

	//remember previous regular motor direction
	r_direction_t prev_r_dir = FORWARD;
	r_direction_t cur_r_dir;

	//remember previous servo motor direction
	//s_direction_t prev_s_dir = RIGHT;
	//s_direction_t cur_s_dir;

	//depth of lot in long area(minimum)
	float lot_depth;

	//decide whether the space is long enough to park
	long parallel_leng_counter = 0;
	long garage_leng_counter = 0;

	//determine whether has turn left and need turn right now
	char has_turned_left = FALSE;

	//parking mode
	char parallel_mode, garage_mode;

	int i;

	//convert use choice
	switch(*(int *)arg){
		case 0: //along the wall
			parallel_mode = FALSE;
			garage_mode = FALSE;
			break;
		case 1:
			parallel_mode = TRUE;
			garage_mode = FALSE;
			break;

		case 2:
			parallel_mode = FALSE;
			garage_mode = TRUE;
			break;

		case 3: //auto choose
			parallel_mode = TRUE;
			garage_mode = TRUE;
			break;

		default:
			puts("Invalid choice, auto parking aborted");
	}

	write_r_motor_speed(1);

	while(TRUE){

		//sleep here so that when we use continue will sleep
		usleep(ULTRASONICS_READ_INTERVAL);

		//if not moving, continue to next loop
		if(read_r_motor_speed() == 0){
			continue;
		}

		//======== for protection =====================
	
		us_FR = cycles_to_cm(read_ultrasonic_distance(FR));
		us_BR = cycles_to_cm(read_ultrasonic_distance(BR));
		if(us_FR < 6 || us_BR < 6){
			write_r_motor_speed(0);
			continue;
		}		
		//=============================================

		//shift the previous values		
		//for(int i=val_count-1; i > 0; i--){
		//	us_RF[i] = us_RF[i-1];
		//	us_RB[i] = us_RB[i-1];
		//}

		//fetch new value
		//us_RF[0] = cycles_to_cm(read_ultrasonic_distance(RF));
		//us_RB[0] = cycles_to_cm(read_ultrasonic_distance(RB));
		us_RF = cycles_to_cm(read_ultrasonic_distance(RF));
		us_RB = cycles_to_cm(read_ultrasonic_distance(RB));


		//clear some memorized states if direction changed
		cur_r_dir = read_r_motor_dir();
		if(cur_r_dir != prev_r_dir){
			long_dist_counter = 0;
			short_dist_counter = 0;
		}
		prev_r_dir = cur_r_dir; //cannot put at the end of while loop, or will be skipped if use continue
		

		// //Determine whether need to turn left
		// us_FB_dist = (cur_r_dir == FORWARD)? 
		// 	cycles_to_cm(read_ultrasonic_distance(FR)):cycles_to_cm(read_ultrasonic_distance(BR)); 
		
		// if(us_FB_dist < turn_left_dist){ //start to turn left to avoid bump the wall

		// 	write_s_motor_dir(LEFT);
		// 	write_s_motor_angle(MAX_ANGLE_LV);
		// 	has_turned_left = TRUE;
		// 	continue;
		// }

		// if(has_turned_left){

		// 	is_in_long_dist_area = FALSE;

		// 	has_turned_left = FALSE;
		// 	continue;
		// }

		//Determine whether it's really a long/short dist or just some accident occurred
		us_R_dist = (cur_r_dir == FORWARD)? us_RF : us_RB;
		us_R_reverse = (cur_r_dir == FORWARD)? us_RB : us_RF;
        
        printf("RF:%.2f, RB:%.2f\n", us_RF, us_RB);

		if(us_R_dist > long_dist){
			short_dist_counter = 0;
			if(!is_in_long_dist_area){
				
				//turn straight first so that it won't go bias
				write_s_motor_angle(0);
				long_dist_counter ++ ;
				if(long_dist_counter >= long_dist_cons){
					is_in_long_dist_area = TRUE;
					cur_long_dist = us_R_dist;
					cur_long_dist_max = cur_long_dist + tolerable_range;
					cur_long_dist_min = cur_long_dist - tolerable_range;
					long_dist_counter = 0;

					lot_depth = us_R_dist;
					last_long_dist_bef_gap = us_R_dist;
					parallel_leng_counter = 0;
					garage_leng_counter = 0;
					gap_counter = 0;
					//prev_was_in_range = FALSE;
					//steering_feedback_counter = 0;
				}else{ // in uncertain state, we don't modify track, so go another run
					continue;
				}
			}else{ //is in long distant area

				delta_cur_last_long_dist = us_R_dist - last_long_dist_bef_gap;
				if(abs(delta_cur_last_long_dist) > gap_leng){
					write_s_motor_angle(0);
					gap_counter++;
					if(gap_counter >= gap_cons){
						lot_depth = (us_R_dist > lot_depth)? us_R_dist : lot_depth;
						last_long_dist_bef_gap = us_R_dist;

						cur_long_dist = us_R_dist;
						cur_long_dist_max = cur_long_dist + tolerable_range;
						cur_long_dist_min = cur_long_dist - tolerable_range;


					}else{
						continue;
					}
				}

				gap_counter = 0;

				lot_depth = (us_R_dist > lot_depth)? us_R_dist : lot_depth;

				//use the reverse one to determin
				if(us_R_reverse > last_short_dist_bef_in_long_area[2] + CAR_WIDTH + CAR_LENGTH_WIDTH_SAFE_MARGIN)
					parallel_leng_counter++;
				else
					parallel_leng_counter = 0;

				if(us_R_reverse > last_short_dist_bef_in_long_area[2] + CAR_LENGTH + CAR_LENGTH_WIDTH_SAFE_MARGIN)
					garage_leng_counter++;
				else
					garage_leng_counter = 0;

				if(parallel_leng_counter >= PARALLEL_ENOUGH_COUNT && parallel_mode){
					
					parallel_parking(last_short_dist_bef_in_long_area[2], lot_depth - last_short_dist_bef_in_long_area[2]);
					puts("Done parking");
					return NULL;
				}
				
				
				
			}							
		}else{

			long_dist_counter = 0;
			if(is_in_long_dist_area){

				write_s_motor_angle(0);				
				short_dist_counter ++;	
				if(short_dist_counter >= short_dist_cons){
					is_in_long_dist_area = FALSE;
					short_dist_counter = 0;
					//prev_was_in_range = FALSE;
					//steering_feedback_counter = 0;
					last_short_dist_bef_in_long_area[0] = last_short_dist_bef_in_long_area[2];
					last_short_dist_bef_in_long_area[1] = last_short_dist_bef_in_long_area[2];
				}else{
					continue;					
				}				
			}else{
				

				if(abs(us_R_reverse - us_R_dist) > 8){ //reverse side still in long area

					//use the reverse one to determin
					if(us_R_reverse > last_short_dist_bef_in_long_area[2] + CAR_WIDTH + CAR_LENGTH_WIDTH_SAFE_MARGIN)
						parallel_leng_counter++;
					else
						parallel_leng_counter = 0;

					if(us_R_reverse > last_short_dist_bef_in_long_area[2] + CAR_LENGTH + CAR_LENGTH_WIDTH_SAFE_MARGIN)
						garage_leng_counter++;
					else
						garage_leng_counter = 0;

					if(parallel_leng_counter >= PARALLEL_ENOUGH_COUNT && parallel_mode){
						
						parallel_parking(us_R_dist, lot_depth - us_R_dist);
						puts("Done parking");
						return NULL;
					}

				}else{
					//shitft
					for(i=2; i > 0; i--){
						last_short_dist_bef_in_long_area[i] = last_short_dist_bef_in_long_area[i-1];
					}
					last_short_dist_bef_in_long_area[0] = us_R_dist;
				}
			}
		}
		
		if(is_in_long_dist_area){

			keep_dist = cur_long_dist;
			keep_dist_max = cur_long_dist_max;
			keep_dist_min = cur_long_dist_min;
		
		}else{

			keep_dist = short_dist;
			keep_dist_max = short_dist_max;
			keep_dist_min = short_dist_min;
		}

		//printf("Current is%s in long_dist_area. keep_dist:%.2f, max: %.2f, min: %.2f\n", 
		//	(is_in_long_dist_area)? "":" not", keep_dist, keep_dist_max,keep_dist_min);

		//cur_s_dir = read_s_motor_dir();
		

		keep_in_dist(keep_dist, keep_dist_max, keep_dist_min, tolerable_delta_f_b, us_R_dist, us_R_reverse);

		//prev_s_dir = cur_s_dir;
		
	}


	return NULL;
}



void keep_in_dist(const float keep_dist, const float keep_dist_max, const float keep_dist_min,
				  const float tolerable_delta_f_b, 
				  const float us_R_dist, const float us_R_reverse){

	//float for tmperary calculation
	float delta_cur_keep;
	float delta_f_b_us;

	//start to modify track
	//check whether in tolerable range (most common case, so detect first)
	if(us_R_dist < keep_dist_max && us_R_dist > keep_dist_min){
		

		
		delta_f_b_us = us_R_dist - us_R_reverse;
		if(delta_f_b_us > tolerable_delta_f_b){
			write_s_motor_dir(RIGHT);
			write_s_motor_angle(1);
		}else if(delta_f_b_us < (-tolerable_delta_f_b)){
			write_s_motor_dir(LEFT);
			write_s_motor_angle(1);
		}else{
			write_s_motor_angle(0);
		}

		
	}else{
		

		delta_cur_keep = us_R_dist - keep_dist;
		
		if(delta_cur_keep > 0){ //bias outside

			write_s_motor_dir(RIGHT);
			
			if(delta_cur_keep < 2){
				write_s_motor_angle(1);
			}else if(delta_cur_keep < 5){
				write_s_motor_angle(2);
			}else if(delta_cur_keep < 10){
				write_s_motor_angle(3);
			}else if(delta_cur_keep < 20){
				write_s_motor_angle(4);
			}else if(delta_cur_keep < 30){		
				write_s_motor_angle(5);
			}else if(delta_cur_keep < 45){
				write_s_motor_angle(6);
			}else{
				write_s_motor_angle(7);	
			}
		}else{ //bias toward wall
			write_s_motor_dir(LEFT);
			if(delta_cur_keep > -1){
				write_s_motor_angle(1);
			}else if(delta_cur_keep > -3){
				write_s_motor_angle(4);
			}else{
				write_s_motor_angle(7);
			}

		}
				
	}

}


