#include "driver_functions.h"
#include "parameters.h"

#include <stdio.h>
#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <string.h>
#include <unistd.h>
#include <stdlib.h>
#include <errno.h>



static const char u_filename[] = "/dev/ultrasonic_sensor"; 
static const char r_filename[] = "/dev/regular_motor";
static const char s_filename[] = "/dev/steering_motor";


//resources for fpga peripherals
int regular_motor_fd, steering_motor_fd, ultrasonic_sensor_fd;


char open_drivers(){

  if ( (ultrasonic_sensor_fd = open(u_filename, O_RDWR)) == -1) {
    fprintf(stderr, "could not open %s\n", u_filename);
    return FALSE;
  }

  if ( (regular_motor_fd = open(r_filename, O_RDWR)) == -1) {
    fprintf(stderr, "could not open %s\n", r_filename);
    return FALSE;
  }

  if ( (steering_motor_fd = open(s_filename, O_RDWR)) == -1) {
    fprintf(stderr, "could not open %s\n", s_filename);
    return FALSE;
  }

  return TRUE;

}

//=== functions for regular motor control ===
void write_r_motor_dir(const r_direction_t r_dir)
{
  if (ioctl(regular_motor_fd, R_MOTOR_SET_DIR, &r_dir)) {
     printf("ioctl(R_MOTOR_SET_DIR) faiball");
     return;
  }
}

void write_r_motor_speed(const __u32 speed)
{
  if (ioctl(regular_motor_fd, R_MOTOR_SET_SPEED, &speed)) {
     printf("ioctl(R_MOTOR_SET_SPEED) faiball");
     return;
  }
}

r_direction_t read_r_motor_dir()
{
  r_direction_t r_dir;
  if (ioctl(regular_motor_fd, R_MOTOR_GET_DIR, &r_dir)) {
     printf("ioctl(R_MOTOR_GET_DIR) faiball");
     return FORWARD; //randomly return one
  }
  
  return r_dir;
}

__u32 read_r_motor_speed()
{
  __u32 speed;
  if (ioctl(regular_motor_fd, R_MOTOR_GET_SPEED, &speed)) {
     printf("ioctl(R_MOTOR_GET_SPEED) faiball");
     return MAX_SPEED_LV + 1;//impossible value in real, so used as error val
  }
  
  return speed;
}

//=== functions for steering motor control ===
void write_s_motor_dir(const s_direction_t s_dir)
{
  if (ioctl(steering_motor_fd, S_MOTOR_SET_DIR, &s_dir)) {
     printf("ioctl(S_MOTOR_SET_DIR) faiball");
     return;
  }
}

void write_s_motor_angle(const __u32 angle)
{
  if (ioctl(steering_motor_fd, S_MOTOR_SET_ANGLE, &angle)) {
     printf("ioctl(S_MOTOR_SET_ANGLE) faiball");
     return;
  }
}

s_direction_t read_s_motor_dir()
{
  s_direction_t s_dir;
  if (ioctl(steering_motor_fd, S_MOTOR_GET_DIR, &s_dir)) {
     printf("ioctl(S_MOTOR_GET_DIR) faiball");
     return RIGHT; //randomly return one
  }
  
  return s_dir;
}

__u32 read_s_motor_angle()
{
  __u32 angle;
  if (ioctl(steering_motor_fd, S_MOTOR_GET_ANGLE, &angle)) {
     printf("ioctl(S_MOTOR_GET_ANGLE) faiball");
     return MAX_ANGLE_LV + 1;//impossible value in real, so used as error val
  }
  
  return angle;
}

//=== functions for ultrasonic sensor control ===
__u32 read_ultrasonic_distance(us_position_t pos)
{
  ultrasonic_t us;
  us.pos = pos;

  if (ioctl(ultrasonic_sensor_fd, ULTRASONIC_GET_DIST_WITH_POS, &us)) {
     printf("ioctl(ULTRASONIC_GET_DIST_WITH_POS) faiball");
     return 0;//impossible value in real, so used as error val
  }

  //since the position of us_RF and us_RB is not exactly the same
  if(us.pos == RF)
    us.distance = us.distance - US_RF_OFFSET;
  
  return us.distance;
}

//conver cycles from read_ultrasonic_distance to cm value
float cycles_to_cm(__u32 cycles){
  return ((float)cycles) * 340 * 100 / 2 / 50000000;
}