/*
 * Device driver for the VGA Piano Emulator
 *
 * A Platform device implemented using the misc subsystem
 *
 * Base code courtesy of:
 * Stephen A. Edwards
 * Columbia University
 *
 * References:
 * Linux source: Documentation/driver-model/platform.txt
 *               drivers/misc/arm-charlcd.c
 * http://www.linuxforu.com/tag/linux-device-drivers/
 * http://free-electrons.com/docs/
 *
 * "make" to build
 * insmod vga_piano.ko
 *
 * Check code style with
 * checkpatch.pl --file --no-tree vga_piano.c
 */

#include <linux/module.h>
#include <linux/init.h>
#include <linux/errno.h>
#include <linux/version.h>
#include <linux/kernel.h>
#include <linux/platform_device.h>
#include <linux/miscdevice.h>
#include <linux/slab.h>
#include <linux/io.h>
#include <linux/of.h>
#include <linux/of_address.h>
#include <linux/fs.h>
#include <linux/uaccess.h>
#include "vga_piano.h"

#define DRIVER_NAME "vga_piano"
#define MAX_MEM_ADDRESS 4095

/*
 * Information about our device
 */
struct vga_piano_dev {
	struct resource res; /* Resource: our registers */
	void __iomem *virtbase; /* Where registers can be accessed in memory */
} dev;

//writes coordinates to the VGA hardware
static void write_coord(u16 xcoord, u16 ycoord)
{
  	unsigned short coords[2];
	coords[0] = xcoord;
	coords[1] = ycoord;
	iowrite32(*(int *)coords, dev.virtbase+MAX_MEM_ADDRESS + 1);
}

//writes data to the VGA hardware
static void write_data(u16 address, u32 data) {
	iowrite32(data, dev.virtbase+address);
}

//writes the key that was pressed to the VGA hardware
static void write_keypress(u32 keypress) {
	iowrite32( keypress, dev.virtbase + MAX_MEM_ADDRESS + 5);
}

/*
 * Handle ioctl() calls from userspace:
 * Read or write the segments on single digits.
 * Note extensive error checking of arguments
 */
static long vga_piano_ioctl(struct file *f, unsigned int cmd, unsigned long arg)
{
	vga_piano_arg_t vla;

	if (copy_from_user(&vla, (vga_piano_arg_t *) arg,
			   sizeof(vga_piano_arg_t)))
		return -EACCES;
	switch (cmd) {
	case VGA_PIANO_WRITE_DATA:
		if (vla.address > MAX_MEM_ADDRESS - 3) {
			return -EINVAL;
		}
		write_data(vla.address, vla.data);
		break;
	case VGA_PIANO_WRITE_COORD:
		if (vla.xcoord >= 640 || vla.ycoord >= 480)
			return -EINVAL;
		write_coord(vla.xcoord, vla.ycoord);
		break;
	case VGA_PIANO_WRITE_KEYPRESS:
		if (vla.keypress > 15) {
			return -EINVAL;
		}
		write_keypress(vla.keypress);
		break;
	default:
		return -EINVAL;
	}
	return 0;
}

/* The operations our device knows how to do */
static const struct file_operations vga_piano_fops = {
	.owner		= THIS_MODULE,
	.unlocked_ioctl = vga_piano_ioctl,
};

/* Information about our device for the "misc" framework -- like a char dev */
static struct miscdevice vga_piano_misc_device = {
	.minor		= MISC_DYNAMIC_MINOR,
	.name		= DRIVER_NAME,
	.fops		= &vga_piano_fops,
};

/*
 * Initialization code: get resources (registers) and display
 * a welcome message
 */
static int __init vga_piano_probe(struct platform_device *pdev)
{
	int ret;

	/* Register ourselves as a misc device: creates /dev/vga_piano */
	ret = misc_register(&vga_piano_misc_device);

	/* Get the address of our registers from the device tree */
	ret = of_address_to_resource(pdev->dev.of_node, 0, &dev.res);
	if (ret) {
		ret = -ENOENT;
		goto out_deregister;
	}

	/* Make sure we can use these registers */
	if (request_mem_region(dev.res.start, resource_size(&dev.res),
			       DRIVER_NAME) == NULL) {
		ret = -EBUSY;
		goto out_deregister;
	}

	/* Arrange access to our registers */
	dev.virtbase = of_iomap(pdev->dev.of_node, 0);
	if (dev.virtbase == NULL) {
		ret = -ENOMEM;
		goto out_release_mem_region;
	}

	return 0;

out_release_mem_region:
	release_mem_region(dev.res.start, resource_size(&dev.res));
out_deregister:
	misc_deregister(&vga_piano_misc_device);
	return ret;
}

/* Clean-up code: release resources */
static int vga_piano_remove(struct platform_device *pdev)
{
	iounmap(dev.virtbase);
	release_mem_region(dev.res.start, resource_size(&dev.res));
	misc_deregister(&vga_piano_misc_device);
	return 0;
}

/* Which "compatible" string(s) to search for in the Device Tree */
#ifdef CONFIG_OF
static const struct of_device_id vga_piano_of_match[] = {
	{ .compatible = "altr,vga_piano" },
	{},
};
MODULE_DEVICE_TABLE(of, vga_piano_of_match);
#endif

/* Information for registering ourselves as a "platform" driver */
static struct platform_driver vga_piano_driver = {
	.driver	= {
		.name	= DRIVER_NAME,
		.owner	= THIS_MODULE,
		.of_match_table = of_match_ptr(vga_piano_of_match),
	},
	.remove	= __exit_p(vga_piano_remove),
};

/* Called when the module is loaded: set things up */
static int __init vga_piano_init(void)
{
	pr_info(DRIVER_NAME ": init\n");
	return platform_driver_probe(&vga_piano_driver, vga_piano_probe);
}

/* Called when the module is unloaded: release resources */
static void __exit vga_piano_exit(void)
{
	platform_driver_unregister(&vga_piano_driver);
	pr_info(DRIVER_NAME ": exit\n");
}

module_init(vga_piano_init);
module_exit(vga_piano_exit);

MODULE_LICENSE("GPL");
MODULE_AUTHOR("Team Leap-Motion: Patrice Liang, Matthew Patey, Vanshil Shah, Kevin Walters; Columbia University");
MODULE_DESCRIPTION("VGA Piano Emulator");
