/**
 * 
 * Our userspace program that communicates with the Leap and with the VGA and audio peripherals
 *	Receives and converts coordinates from the Leap device, uses keypress logic to tell the hardware peripherals what to display and play
 */


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/socket.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <arpa/inet.h>
#include <fcntl.h>
#include <unistd.h>
#include <pthread.h>
#include "vga_piano.h"

#define PORT htons(4000)
#define BUFFER_SIZE 4096
#define KEYPRESS_VEL -10
#define WHITE_KEY_WIDTH 91
#define BLACK_KEY_WIDTH 30
#define WHITE_KEY_HEIGHT 400
#define WHITE_KEY_HEIGHT_PRESSED 430
#define BLACK_KEY_HEIGHT 380
#define BLACK_KEY_HEIGHT_PRESSED 395
#define SCREEN_WIDTH_MM 340
#define SCREEN_WIDTH_PIXELS 640
#define SCREEN_HEIGHT_MM 270
#define SCREEN_HEIGHT_PIXELS 480
#define SCREEN_OFFSET 100 //distance in millimeters from bottom of screen to Leap
#define CURSOR_SIZE 32

void write_data(unsigned short address, unsigned int data);
void write_keypress(unsigned short keypress);
void write_coords(const unsigned short xcoord, const unsigned short ycoord);

extern void audio_thread_fn(void *);
extern send_to_audio_thread(unsigned int key);

int vga_piano_fd;

/* Variables to deal with audio */
int key;
int pressed_key;
pthread_t audio_thread;
int play = 0;

int main() 
{
	init();
	//variables to deal with receiving data packets
	int sockfd;
	struct sockaddr_in saddr;
	char recvBuf[BUFFER_SIZE+1];
	int n;
	
	//open a new UDP socket
	if ((sockfd = socket(AF_INET, SOCK_DGRAM, 0)) < 0 ) 
	{
		fprintf(stderr, "Error: Could not create socket\n");
		exit (1);
	}

	saddr.sin_family = AF_INET;
	saddr.sin_port = PORT;
	saddr.sin_addr.s_addr = htonl(INADDR_ANY);
	
	if (bind (sockfd, (struct sockaddr *)&saddr, sizeof(saddr)) < 0)
	{
		fprintf(stderr, "Error: Could not bind to a port");
		exit (1);
	}

	printf("%s","Beginning to read packets\n");

	//write initial values to the hardware
	write_coords(100, 100);
	write_keypress(0);
	pressed_key = 0;
	key = 0; 
	//audio thread for concurrent playback and VGA updates
	pthread_create(&audio_thread, NULL, &audio_thread_fn, NULL);

	while (1)
	{
		memset(recvBuf, 0, BUFFER_SIZE+1);
		//receive packet from UDP socket
		n = recvfrom(sockfd, recvBuf, BUFFER_SIZE - 1, 0, NULL, NULL);

		//get data from the packet
		int *coord = (int *)recvBuf;
		int x = ntohl(*coord++);
		int y = ntohl(*coord++);
		int y_vel = ntohl(*coord);

		//translate from millimeters to pixels
		get_coords(&x,&y,&y_vel);

		//keep the cursor within the screen borders
		if (x < 0)
			x = 0;
		else if (x > SCREEN_WIDTH_PIXELS-CURSOR_SIZE)
			x = SCREEN_WIDTH_PIXELS-CURSOR_SIZE;
		
		if (y < 0)
			y = 0;
		else if (y > SCREEN_HEIGHT_PIXELS-CURSOR_SIZE)
			y = SCREEN_HEIGHT_PIXELS-CURSOR_SIZE;
		write_coords((unsigned short)x, (unsigned short)y);

		//if a different key has been pressed, tell the audio and VGA drivers
		int old_key = key;
		key = isKeypress(x,y, y_vel);
		if (key != old_key) {
			write_keypress((unsigned short) key);
			if (key > 0) {
				send_to_audio_thread(key);
			}
		}
	}
	
	return 0;
}

/*Use the coordinates and velocity to determine if a key has been pressed
  Returns the number of the key pressed*/
int isKeypress(int x, int y, int y_vel) {
	int x_mid = x + CURSOR_SIZE/2;//for a 32x32 sprite
	int y_bot = y + CURSOR_SIZE;
	int keypress = 0;

	//Determine if a black key has been pressed
	if (x_mid > (WHITE_KEY_WIDTH - BLACK_KEY_WIDTH/2) && x_mid < (WHITE_KEY_WIDTH + BLACK_KEY_WIDTH/2) && y_bot >= BLACK_KEY_HEIGHT && (y_vel < KEYPRESS_VEL || key == 8))
		keypress = 8;
	else if (x_mid > (2*WHITE_KEY_WIDTH - BLACK_KEY_WIDTH/2) && x_mid < (2*WHITE_KEY_WIDTH + BLACK_KEY_WIDTH/2) && y_bot >= BLACK_KEY_HEIGHT && (y_vel < KEYPRESS_VEL || key == 9))
		keypress = 9;
	else if (x_mid > (4*WHITE_KEY_WIDTH - BLACK_KEY_WIDTH/2) && x_mid < (4*WHITE_KEY_WIDTH + BLACK_KEY_WIDTH/2) && y_bot >= BLACK_KEY_HEIGHT && (y_vel < KEYPRESS_VEL || key == 10))
		keypress = 10;
	else if (x_mid > (5*WHITE_KEY_WIDTH - BLACK_KEY_WIDTH/2) && x_mid < (5*WHITE_KEY_WIDTH + BLACK_KEY_WIDTH/2) && y_bot >= BLACK_KEY_HEIGHT && (y_vel < KEYPRESS_VEL || key == 11))
		keypress = 11;
	else if (x_mid > (6*WHITE_KEY_WIDTH - BLACK_KEY_WIDTH/2) && x_mid < (6*WHITE_KEY_WIDTH + BLACK_KEY_WIDTH/2) && y_bot >= BLACK_KEY_HEIGHT && (y_vel < KEYPRESS_VEL || key == 12))
		keypress = 12;
	//Determine if a white key has been pressed
	else if (x_mid < WHITE_KEY_WIDTH && y_bot >= WHITE_KEY_HEIGHT && (y_vel < KEYPRESS_VEL || key == 1))
		keypress = 1;
	else if (x_mid > WHITE_KEY_WIDTH && x_mid < 2*WHITE_KEY_WIDTH && y_bot >= WHITE_KEY_HEIGHT && (y_vel < KEYPRESS_VEL || key == 2))
		keypress = 2;
	else if (x_mid > 2*WHITE_KEY_WIDTH && x_mid < 3*WHITE_KEY_WIDTH && y_bot >= WHITE_KEY_HEIGHT && (y_vel < KEYPRESS_VEL || key == 3))
		keypress = 3;
	else if (x_mid > 3*WHITE_KEY_WIDTH && x_mid < 4*WHITE_KEY_WIDTH && y_bot >= WHITE_KEY_HEIGHT && (y_vel < KEYPRESS_VEL || key == 4))
		keypress = 4;
	else if (x_mid > 4*WHITE_KEY_WIDTH && x_mid < 5*WHITE_KEY_WIDTH && y_bot >= WHITE_KEY_HEIGHT && (y_vel < KEYPRESS_VEL || key == 5))
		keypress = 5;
	else if (x_mid > 5*WHITE_KEY_WIDTH && x_mid < 6*WHITE_KEY_WIDTH && y_bot >= WHITE_KEY_HEIGHT && (y_vel < KEYPRESS_VEL || key == 6))
		keypress = 6;
	else if (x_mid > 6*WHITE_KEY_WIDTH && x_mid < 7*WHITE_KEY_WIDTH && y_bot >= WHITE_KEY_HEIGHT && (y_vel < KEYPRESS_VEL || key == 7))
		keypress = 7;
	else
		keypress = 0;
	return keypress;
}

/* Converts from Leap coordinates (millimeters) to pixels*/
int get_coords(int *x, int *y, int *y_vel)
{
	//middle of screen is 170 mm = 320 pixels
	int tempx, tempy;
	tempx = *x;
	tempy = *y;
	//handle Leap coordinate system offset
	tempx = tempx + SCREEN_WIDTH_MM/2;
	//convert from millimeters to pixels
	*x = tempx  * SCREEN_WIDTH_PIXELS/SCREEN_WIDTH_MM;
	
	//subtract distance from the Leap to the bottom of the monitor used
	tempy -= SCREEN_OFFSET;
	tempy = tempy * SCREEN_HEIGHT_PIXELS/SCREEN_HEIGHT_MM;
	//invert to a top-down coordinate system
	*y = SCREEN_HEIGHT_PIXELS - tempy;
	
	return 0;
}

/* Initializes the file descriptor used to communicate with the vga_piano peripheral */
int init() {
  vga_piano_arg_t vla;
  int i;
  static const char filename[] = "/dev/vga_piano";
  static const char imageFileName[] = "greenCircle.dat";
  FILE *fp;

  printf("Userspace program initialized\n");

  if ( (vga_piano_fd = open(filename, O_RDWR)) == -1) {
    fprintf(stderr, "could not open %s\n", filename);
    return -1;
  }

  fp = fopen(imageFileName, "rb");
  if(!fp) {
    fprintf(stdout, "could not open %s\n", imageFileName);
    return -1;
  }

  //writes cursor data to the driver
	for(i = 0; i < CURSOR_SIZE*CURSOR_SIZE; i++) {
    	unsigned int temp;
		fread((void *)&temp, sizeof(int), 1, fp);
		write_data(i*4, temp);
	}
  
  init_audio();
	
  return 0;
}


/* Write the coordinates of the cursor to the VGA hardware through the driver */
void write_coords(const unsigned short xcoord, const unsigned short ycoord) {
  vga_piano_arg_t vla;
  vla.xcoord = xcoord;
  vla.ycoord = ycoord;
  if (ioctl(vga_piano_fd, VGA_PIANO_WRITE_COORD, &vla)) {
    perror("ioctl(VGA_PIANO_WRITE_COORD) failed");
    return;
  }
}

/* Write the number of a pressed key to the VGA hardware through the driver*/
void write_keypress(unsigned short keypress) {
  vga_piano_arg_t vla;
  vla.keypress = keypress;
  if (ioctl(vga_piano_fd, VGA_PIANO_WRITE_KEYPRESS, &vla)) {
    perror("ioctl(VGA_PIANO_WRITE_KEYPRESS) failed");
    return;
  }
}

/* Write data to the VGA hardware through the driver*/
void write_data(unsigned short address, unsigned int data) {
	vga_piano_arg_t vla;
	vla.address = address;
	vla.data = data;
	
	if (ioctl(vga_piano_fd, VGA_PIANO_WRITE_DATA, &vla)) {
		perror("ioctl(VGA_PIANO_WRITE_DATA) failed");
		return;
	}
}

