/*
 * playback portion based on code by Howard Mao
 * http://zhehaomao.com/blog/fpga/2014/01/15/sockit-8.html
 */

module audio_interface (
	 input audio_clk,
    input  sample_req,
    output [15:0] audio_output,
	 
	 //Inputs from the HPS
	 input hps_clk,
	 input reset,
	 input chipselect,
	 input [11:0] address,
	 input [15:0] data,
	 input write,
	 input read,
	 
	 output [15:0] readdata,
	 output [3:0] LEDS
);

//2 buffer audio memory
reg [15:0] wavData0 [1023:0];
reg [15:0] wavData1 [1023:0];

logic writeBuffer; // the buffer currently being written to
logic playBuffer; // the buffer currently being played from

logic ready0; // flag high when buffer 0 ready to be played
logic ready1; // flag high when buffer 1 ready to be played

// represent events of buffer becoming ready or no longer being ready, used to update ready flags
logic enterReady0;
logic enterReady1;
logic exitReady0;
logic exitReady1;

reg [10:0] index = 10'd0; // current position in playback
reg [15:0] dat; // assigned to audio data out

assign audio_output = dat;
assign LEDS[0] = ready0;
assign LEDS[1] = ready1;
assign LEDS[2] = index[10];
assign LEDS[3] = '1;

always_ff @(posedge hps_clk or posedge reset) begin
	if (reset) begin
		ready0 <= 0;
		ready1 <= 0;
	end
	else begin
		// ready flag logic: on event flag, update value accordingly
		// otherwise, value is unchanged
		if (enterReady0)
			ready0 <= 1;
		else if (exitReady0)
			ready0 <= 0;
		
		if (enterReady1)
			ready1 <= 1;
		else if (exitReady1)
			ready1 <= 0;
	end
end

// hps reads and writes
always_ff @(posedge hps_clk or posedge reset) begin
	if (reset) begin
		writeBuffer <= 0;
		enterReady0 <= 0;
		enterReady1 <= 0;
	end
	else if (chipselect && read) begin
		if (address == 11'd1025) begin // only one valid read address, used for buffer status
			// return bitwise or of ready flags
			if(!ready1 && !ready0)
				readdata <= 16'd0;
			else if (ready0 && !ready1)
				readdata <= 16'd1;
			else if(!ready0 && ready1)
				readdata <= 16'd2;
			else if(ready1 && ready0)
				readdata <= 16'd3;
			//readdata <= !ready0 || !ready1;
		end
		enterReady0 <= 0;
		enterReady1 <= 0;
	end
	else begin
		if(writeBuffer && write && chipselect) begin // currently writing to buffer 1
			if(address == 11'd1024) begin // play signal, sets ready flag for currently writing buffer
				enterReady1 <= 1; // set event flag to enter ready 1 state
				writeBuffer <= !writeBuffer; // switch write buffer to be buffer 0
			end else begin
				wavData1[address] <= data; // write data into buffer 1
				enterReady1 <= 0;
			end
			enterReady0 <= 0;
		end else if (!writeBuffer && write && chipselect) begin // writing to buffer 0
			if(address == 11'd1024) begin // play signal, sets ready flag for currently writing buffer
				enterReady0 <= 1; // set event flag to enter ready 0 state
				writeBuffer <= !writeBuffer; // switch write buffer to be buffer 1
			end else begin
				wavData0[address] <= data;
				enterReady0 <= 0;
			end
			enterReady1 <= 0;
		end else begin
			enterReady0 <= 0;
			enterReady1 <= 0;
		end
	end
end

// audio playback
always_ff @(posedge audio_clk or posedge reset) begin
	if (reset) begin
		playBuffer <= 0;
		exitReady0 <= 0;
		exitReady1 <= 0;
	end else begin
		if (sample_req && playBuffer && ready1) begin // playing from buffer 1 and it is ready
			dat <= wavData1[index];
			if (index == 10'd1023) begin // reached end of buffer
				exitReady1 <= 1; // set flag to exit ready 1 state
				playBuffer <= 0; // set playbuffer to buffer 0
				index <= 10'h0;
			end else begin
				exitReady1 <= 0;
				index <= index + 1'b1;
			end
			exitReady0 <= 0;
		end else if (sample_req && !playBuffer && ready0) begin // playing from buffer 0 and it is ready
			dat <= wavData0[index];
			if (index == 10'd1023) begin // reached end of buffer
				exitReady0 <= 1; // set flag to exit ready 0 state
				playBuffer <= 1; // set playbuffer to buffer 1
				index <= 10'h0;
			end else begin
				exitReady0 <= 0;
				index <= index + 1'b1;
			end
			exitReady1 <= 0;
		end else if(sample_req) begin // if the next buffer to play is not ready send 0's
			dat <= 16'h0;
			exitReady0 <= 0;
			exitReady1 <= 0;
		end else begin // don't send anything if the next sample has not been requested
			exitReady0 <= 0;
			exitReady1 <= 0;
		end
	end
end


endmodule
