/*
 * Piano
 */

module VGA_Piano_Emulator(
 input logic 	    clk50, reset,
 input logic [7:0] keypress,
 input logic [15:0] xcoord, ycoord, // the bottom center point of the cursor
 input logic [7:0] sR, sG, sB, sA,
 output logic [4:0] spixelx, spixely, // address to pixel memory top left = 0, 0
 output logic [7:0] VGA_R, VGA_G, VGA_B,
 output logic 	    VGA_CLK, VGA_HS, VGA_VS, VGA_BLANK_n, VGA_SYNC_n);
 
 logic [15:0] xcount, ycount; // pixel location on screen of raster scan

/*
 * 640 X 480 VGA timing for a 50 MHz clock: one pixel every other cycle
 * 
 *HCOUNT 1599 0             1279       1599 0
 *            _______________              ________
 * __________|    Video      |____________|  Video
 * 
 * 
 * |SYNC| BP |<-- HACTIVE -->|FP|SYNC| BP |<-- HACTIVE
 *       _______________________      _____________
 * |____|       VGA_HS          |____|
 */

	/*Parameters for the VGA screen*/
   parameter HACTIVE      = 11'd 1280,
             HFRONT_PORCH = 11'd 32,
             HSYNC        = 11'd 192,
             HBACK_PORCH  = 11'd 96,   
             HTOTAL       = HACTIVE + HFRONT_PORCH + HSYNC + HBACK_PORCH; //1600

   parameter VACTIVE      = 10'd 480,
             VFRONT_PORCH = 10'd 10,
             VSYNC        = 10'd 2,
             VBACK_PORCH  = 10'd 33,
             VTOTAL       = VACTIVE + VFRONT_PORCH + VSYNC + VBACK_PORCH; //525
				 
	/*Parameters for the key display*/
	parameter WHITE_KEY_WIDTH	= 10'd 91,
				 BLACK_KEY_WIDTH	= 10'd 30,
				 WHITE_KEY_HEIGHT	= 10'd 400,
				 WHITE_KEY_HEIGHT_PRESSED	= 10'd 430,
				 BLACK_KEY_HEIGHT	= 10'd 380,
				 BLACK_KEY_HEIGHT_PRESSED	= 10'd 395;
				 
	parameter CURSOR_SIZE	= 16'd 32;

   logic [10:0]				hcount; // Horizontal counter
   logic 						endOfLine;
   
   always_ff @(posedge clk50 or posedge reset)
     if (reset)          	hcount <= 0;
     else if (endOfLine) 	hcount <= 0;
     else  	         		hcount <= hcount + 11'd 1;

   assign endOfLine = hcount == HTOTAL - 1;

   // Vertical counter
   logic [9:0] 				vcount;
   logic 			   		endOfField;
   
   always_ff @(posedge clk50 or posedge reset)
     if (reset)          	vcount <= 0;
     else if (endOfLine)
       if (endOfField)   	vcount <= 0;
       else              	vcount <= vcount + 10'd 1;

   assign endOfField = vcount == VTOTAL - 1;

   // Horizontal sync: from 0x520 to 0x57F
   // 101 0010 0000 to 101 0111 1111
   assign VGA_HS = !( (hcount[10:7] == 4'b1010) & (hcount[6] | hcount[5]));
   assign VGA_VS = !( vcount[9:1] == (VACTIVE + VFRONT_PORCH) / 2);

   assign VGA_SYNC_n = 1; // For adding sync to video signals; not used for VGA
   
   // Horizontal active: 0 to 1279     Vertical active: 0 to 479
   // 101 0000 0000  1280	       01 1110 0000  480	       
   // 110 0011 1111  1599	       10 0000 1100  524        
   assign VGA_BLANK_n = !( hcount[10] & (hcount[9] | hcount[8]) ) &
			!( vcount[9] | (vcount[8:5] == 4'b1111) );

   assign VGA_CLK = hcount[0]; // 25 MHz clock: pixel latched on rising edge
	
	assign xcount = hcount >> 1;
	assign ycount = vcount;
   
	logic on_white_key; 			// raster point is on a white key
	logic [7:0]key_horizontal; // index of the white key that the horizontal position corresponds to, if any
	logic on_border;				// set to 1 if on border between white keys
	logic on_black_key;
	logic [7:0]black_key_horizontal; // index of the black key that the horizontal position corresponds to, if any
	logic [15:0] tempx, tempy; 		// used to pick off lower bits of calculated memory address
	assign spixelx = tempx[4:0]; 
	assign spixely = tempy[4:0];
   
	always_comb begin
		key_horizontal = 0;
		on_white_key = 0;
		on_border = 0;
		black_key_horizontal = 0;
		on_black_key = 0;
		
		/*
		 * raster scan painting logic
		 */
		
		// determine which white key the horizontal position corresponds to
		if (xcount < WHITE_KEY_WIDTH)
			key_horizontal = 1;
		else if (xcount > WHITE_KEY_WIDTH && xcount < WHITE_KEY_WIDTH*2)
			key_horizontal = 2;
		else if (xcount > WHITE_KEY_WIDTH*2 && xcount < WHITE_KEY_WIDTH*3)
			key_horizontal = 3;
		else if (xcount > WHITE_KEY_WIDTH*3 && xcount < WHITE_KEY_WIDTH*4)
			key_horizontal = 4;
		else if (xcount > WHITE_KEY_WIDTH*4 && xcount < WHITE_KEY_WIDTH*5)
			key_horizontal = 5;
		else if (xcount > WHITE_KEY_WIDTH*5 && xcount < WHITE_KEY_WIDTH*6)
			key_horizontal = 6;
		else if (xcount > WHITE_KEY_WIDTH*6)
			key_horizontal = 7;
		else
			on_border = ycount > WHITE_KEY_HEIGHT; // if not horizontally aligned with a white key, must be on a border if vertically within white keys
			
		// determine if vertically on a white key
		if (key_horizontal == keypress) // if key pressed, check for pressed white key height
			on_white_key = ycount > WHITE_KEY_HEIGHT_PRESSED;
		else
			on_white_key = ycount > WHITE_KEY_HEIGHT;
		
		// determine which black key the horizontal position corresponds to
		if(xcount > (WHITE_KEY_WIDTH - BLACK_KEY_WIDTH/2) && xcount < (WHITE_KEY_WIDTH + BLACK_KEY_WIDTH/2))
			black_key_horizontal = 8;
		else if(xcount > (2*WHITE_KEY_WIDTH - BLACK_KEY_WIDTH/2) && xcount < (2*WHITE_KEY_WIDTH + BLACK_KEY_WIDTH/2))
			black_key_horizontal = 9;
		else if(xcount > (4*WHITE_KEY_WIDTH - BLACK_KEY_WIDTH/2) && xcount < (4*WHITE_KEY_WIDTH + BLACK_KEY_WIDTH/2))
			black_key_horizontal = 10;
		else if(xcount > (5*WHITE_KEY_WIDTH - BLACK_KEY_WIDTH/2) && xcount < (5*WHITE_KEY_WIDTH + BLACK_KEY_WIDTH/2))
			black_key_horizontal = 11;
		else if(xcount > (6*WHITE_KEY_WIDTH - BLACK_KEY_WIDTH/2) && xcount < (6*WHITE_KEY_WIDTH + BLACK_KEY_WIDTH/2))
			black_key_horizontal = 12;

		// determine if vertically and horizontally on a black key
		if(black_key_horizontal == keypress && black_key_horizontal != 0) // if current black key is pressed check for pressed black key height
			on_black_key = ycount > BLACK_KEY_HEIGHT_PRESSED;
		else if(black_key_horizontal != 0)
			on_black_key = ycount > BLACK_KEY_HEIGHT;
	
		/*
		 * Do the actual painting
		 */
	
		// draw background color, lowest priority
      {VGA_R, VGA_G, VGA_B} = {8'ha8, 8'hba, 8'hea};
		
		tempx = 0;
		tempy = 0;
		
		// if in the range of the cursor, calculate pixel address and send to memory
		if (xcount >= xcoord && xcount < (xcoord + CURSOR_SIZE) && ycount >= ycoord && ycount < (ycoord + CURSOR_SIZE)) begin
			tempx = (xcount - xcoord);
			tempy = (ycount - ycoord);
		end
		
		if(on_black_key) // paint black key, low priority
			{VGA_R, VGA_G, VGA_B} = {8'h0, 8'h0, 8'h0}; 	//draw black key
		if( on_white_key ) // paint white key, medium priority
			{VGA_R, VGA_G, VGA_B} = {8'hff, 8'hff, 8'hff}; //draw white key
		if (on_border) // paint border, high priority
			{VGA_R, VGA_G, VGA_B} = {8'h0, 8'h0, 8'h0}; 	//draw black border
		if (xcount > xcoord && xcount <= (xcoord + CURSOR_SIZE) && ycount >= ycoord && ycount < (ycoord + CURSOR_SIZE) && sA != 0) //draw cursor
			{VGA_R, VGA_G, VGA_B} = {sR, sG, sB}; // paint cursor, highest priority
   end  
   
endmodule // VGA_Piano_Emulator
