/*
 * VGA Piano
 */

module VGA_Piano(input logic        clk,
		input logic 	  reset,
		input logic [31:0]  writedata,
		input logic 	  write,
		input 		  chipselect,
		input logic [15:0]  address,

		output logic [7:0] VGA_R, VGA_G, VGA_B,
		output logic 	  VGA_CLK, VGA_HS, VGA_VS, VGA_BLANK_n,
		output logic 	  VGA_SYNC_n);

// RGBA read from sprite memory, sent to Emulator
logic [7:0] R;
logic [7:0] G;
logic [7:0] B;
logic [7:0] A;

logic [31:0] sprite_data; // data written to sprite memory
logic [4:0] pixelx, pixely; // sprite input x and y pixel address selector
logic [4:0] vga_pixelx, vga_pixely; // output from emulator, requested coordinates to read from sprite mem

// current position of cursor on the screen
logic [15:0] cursorx;
logic [15:0] cursory;
logic [7:0] keypress; // determines which key is being pressed

initial begin
	cursorx = '0;
	cursory = '0;
	keypress = '0;
end

logic swrite; // input to sprite mem, telling it whether to write

always_comb begin
	sprite_data = writedata;
	
	// if avalon not writing to device, these are the values sent to sprite inputs
	swrite = 0;
	pixelx = vga_pixelx;
	pixely = vga_pixely;
	
	if (write && chipselect) begin
		if(address <= 16'h03ff) begin // if address in memory's range, write data to sprite mem
			swrite = 1;
			
			// split 1D address into 2D sprite address
			pixelx = address[4:0];
			pixely = address[9:5];
		end
	end
end

always_ff @(posedge clk) begin
	if (write && chipselect) begin
		if (address == 16'h0400) begin // cursor position write address
				cursorx = writedata[15:0];
				cursory = writedata[31:16];
		end else if (address[15:0] == 16'h0401) begin // keypress write address
				keypress = writedata[7:0];
		end
	end
end
	
	
	
	
sprite_memory mem(.clk(clk), .reset(reset), .writedata(sprite_data), .x(pixelx), .y(pixely),
		.write(swrite), .R(R), .G(G), .B(B), .A(A)
		);

VGA_Piano_Emulator piano_emulator(.clk50(clk), .reset(reset), .keypress(keypress),  .xcoord(cursorx), .ycoord(cursory),
		.sR(R), .sG(G), .sB(B), .sA(A), .spixelx(vga_pixelx), .spixely(vga_pixely),
		.VGA_R(VGA_R), .VGA_G(VGA_G), .VGA_B(VGA_B), .VGA_CLK(VGA_CLK), .VGA_HS(VGA_HS), .VGA_VS(VGA_VS),
		.VGA_BLANK_n(VGA_BLANK_n), .VGA_SYNC_n(VGA_SYNC_n));

endmodule

/*Handles all memory for the sprite
  Stores the sprite data in memory, and outputs the appropriate pixel if requested*/
module sprite_memory(input logic clk,
		input logic reset,
		input logic [31:0] writedata,
		input logic [4:0] x,
		input logic [4:0] y,
		input logic write,

		output logic [7:0] R,
		output logic [7:0] G,
		output logic [7:0] B,
		output logic [7:0] A
		);


reg [31:0] sprite1 [31:0][31:0]; // 2D array of 32 bit words, each stores ones pixel
reg [31:0] out;

initial begin
	for (int i =0; i<32;i++) begin
		for (int j=0; j<32; j++)
			sprite1[i][j] = '0;
	end
end

// split up output into 4 values
assign R = out[7:0];
assign G = out[15:8];
assign B = out[23:16];
assign A = out[31:24];

always_ff @(posedge clk) begin
	if(write) begin
		sprite1[x][y] <= writedata;
	end	
	out <= sprite1[x][y];
end	
endmodule

