#include <stdio.h>
#include <stdint.h>
#include <string.h>
#include <sys/ioctl.h>
#include <curl/curl.h>
#include <jansson.h>
#include <pthread.h>
#include <fcntl.h>
#include "vga_led.h"
#include <unistd.h>
#include <sys/time.h>

#define BUFFER_SIZE  (256 * 1024)  /* 256 KB */

#define HEADER_BUFFER_SIZE 96
#define MIDSTATE_SIZE 64
#define DATA_SIZE 32

#define CHARS_TO_DATA 188
#define DATA_LENGTH 256

#define IP_OFFSET 7
#define IP_LENGTH 14

#define SIGNAL_OFFSET 7
#define NONCE_CHECK_START 152
#define NONCE_CHECK_FINISH 188
#define DATA_START 147
#define DATA_END 167
#define INIT_WRITE 102

// Example json response from network
// curl --user halffast.worker1:WyhZfpFS --data-binary '{ "id":"curltest", "method":"getwork", "params":[] }' -H 'content-type: text/plain;' http://localhost:8332/lp -i
//{"error": null, 
//  "id": "curltest",
//  "result":
//  {"hash1": "00000000000000000000000000000000000000000000000000000000000000000000008000000000000000000000000000000000000000000000000000010000",
//    "data": "000000029b39574cb8a5c25e94cdd844f7d3b942384af65fb264d5a000000000000000008adb353c754fe07c40341728c5efe6e3782d994e0e3ebcb4879106de94dd1db0533b2c461900db9900000000000000800000000000000000000000000000000000000000000000000000000000000000000000000000000080020000",
//    "target": "0000000000000000000000000000000000000000000000000000ffff00000000", 
//    "midstate": "49bc9ada38ba6b43814a3275edfa09b796be70080d5c641d32b80776aef13f78"}
//}

//defaults
const char *req = "{ \"id\":\"1\", \"method\":\"getwork\", \"params\":[] }";
const char *usrpwd = "halffast.worker1:WyhZfpFS";
const char *header = "content-type: text/plain;";
const char *pool_url = "http://207.10.143.139:8332/";
const char *lp_pool_url = "http://207.10.143:8332/lp";

struct timeval stop, start;

//initial data values
char current_data[DATA_LENGTH] = "deadbee14cc2c57c7905fd399965282c87fe259e7da366e035dc087a0000141f000000006427b6492f2b052578fb4bc23655ca4e8b9e2b9b69c88041b2ac8c771571d1be4de695931a2694217a00110e000000800000000000000000000000000000000000000000000000000000000000000000000000000000000080020000\0";

int vga_led_fd;
unsigned char nonce[4];

void die(const char *message)
{
    perror(message);
    exit(1);
}

void json_die(const int error_line, const char *error_text)
{
    fprintf(stderr, "error: on line %d: %s\n", error_line, error_text);
    exit(1);
}

void
json_data_error(json_int_t json_data_error_num){
    if(json_data_error_num != 0){
        switch(json_data_error_num){
            case 20 : printf("Other/Unknown\n");            
            case 21 : printf("Job not found (=stale)\n");            
            case 22 : printf("Duplicate share\n");            
            case 23 : printf("Low difficulty share\n");            
            case 24 : printf("Unauthorized worker\n");            
            case 25 : printf("Not subscribed\n");
            default : printf("strange error google resuslts\n");            
        }
        die("error in result from json response");
    }
}

/* Read and print the segment values */
void print_segment_info() {
  vga_led_arg_t vla;
  int i;

  for (i = VGA_LED_DIGITS+SIGNAL_OFFSET; i > -1; i--){
    vla.digit = i;
    if (ioctl(vga_led_fd, VGA_LED_READ_DIGIT, &vla)) {
      perror("ioctl(VGA_LED_READ_DIGIT) failed");
      return;
    }
    printf("%02x ", vla.segments);
  }
  printf("\n");
}

//Returns 1 if the if there is a solved nonce value
int checkTicket() {
  int i;
  vga_led_arg_t vla;
  for(i=NONCE_CHECK_START; i < NONCE_CHECK_FINISH; i+=5) {
    vla.digit = i;
    if (ioctl(vga_led_fd, VGA_LED_READ_DIGIT, &vla)) {
      perror("ioctl(VGA_LED_READ_DIGIT) failed");
      return 0;
    }
    if(vla.segments == 1) {
      int j, idx;
      idx = 3;
      for(j=i-1; j > i-5; j--) {
        vla.digit = j;
        if (ioctl(vga_led_fd, VGA_LED_READ_DIGIT, &vla)) {
            perror("ioctl(VGA_LED_READ_DIGIT) failed");
            return 0;
        }
        *(nonce+idx) = (unsigned char) vla.segments;
        idx--;
      }
      return 1;
    }
      
  }
  return 0;
}

//print what was written to the kernel module
void print_result() {
  vga_led_arg_t vla;
  int i;
  int j=0;
  for (i = DATA_END; i > DATA_START; i--){
    vla.digit = i;
    if (ioctl(vga_led_fd, VGA_LED_READ_DIGIT, &vla)) {
      perror("ioctl(VGA_LED_READ_DIGIT) failed");
      return;
    }
    printf("%02x ", vla.segments);
    printf("%d ", i);
    if( (j++) % 5 == 0)
      printf("\n");
  }
  printf("\n");
}

void write_segments(unsigned char *segs)
{
  vga_led_arg_t vla;
  int i;
  for (i = 0 ; i < VGA_LED_DIGITS; i++){
    vla.digit = i;
    vla.segments = segs[i];
    if (ioctl(vga_led_fd, VGA_LED_WRITE_DIGIT, &vla)) {
      perror("ioctl(VGA_LED_WRITE_DIGIT) failed");
      return;
    }
  }
}

void init_write(){
    vga_led_arg_t vla;
    vla.digit = INIT_WRITE;
    vla.segments = 1;
    if (ioctl(vga_led_fd, VGA_LED_WRITE_DIGIT, &vla)) {
      perror("ioctl(VGA_LED_WRITE_DIGIT) failed");
      return;
    }
}

// write_result, write_response and request came from the jansson tutorial
// http://jansson.readthedocs.org/en/2.3/tutorial.html
struct write_result
{
    char *data;
    int pos;
};

size_t write_response(void *ptr, size_t size, size_t nmemb, void *stream)
{
    struct write_result *result = (struct write_result *)stream;

    if(result->pos + size * nmemb >= BUFFER_SIZE - 1)
    {
        fprintf(stderr, "error: too small buffer\n");
        return 0;
    }

    memcpy(result->data + result->pos, ptr, size * nmemb);
    result->pos += size * nmemb;

    return size * nmemb;
}

//send http request with easy_curl
char *request(const char *url, const char *bin_data)
{
    CURL *curl = NULL;
    CURLcode status;
    struct curl_slist *headers = NULL;
    char *data = NULL;
    long code;

    curl_global_init(CURL_GLOBAL_ALL);
    curl = curl_easy_init();
    if(!curl)
        goto error;

    data = malloc(BUFFER_SIZE);
    if(!data)
        goto error;

    struct write_result write_result = {
        .data = data,
        .pos = 0
    };

    //set up url
    curl_easy_setopt(curl, CURLOPT_URL, url);
    
    //set up bin_data the crux of the getwork request
    curl_easy_setopt(curl, CURLOPT_POSTFIELDS, ((void *) bin_data));
    curl_easy_setopt(curl, CURLOPT_POSTFIELDSIZE, ((long)-1) );

    //set up usrpwd
    curl_easy_setopt(curl, CURLOPT_USERPWD, usrpwd);

    //set up headers from init response
    headers = curl_slist_append(headers, header);
    curl_easy_setopt(curl, CURLOPT_HTTPHEADER, headers);

    //set up writing repsonse
    curl_easy_setopt(curl, CURLOPT_WRITEFUNCTION, write_response);
    curl_easy_setopt(curl, CURLOPT_WRITEDATA, &write_result);

    //preform the curl
    status = curl_easy_perform(curl);

    //error handle
    if(status != 0)
    {
        fprintf(stderr, "error: unable to request data from %s:\n", url);
        fprintf(stderr, "%s\n", curl_easy_strerror(status));
        goto error;
    }

    curl_easy_getinfo(curl, CURLINFO_RESPONSE_CODE, &code);
    if(code != 200)
    {
        fprintf(stderr, "error: server responded with code %ld\n", code);
        goto error;
    }

    curl_easy_cleanup(curl);
    curl_slist_free_all(headers);
    curl_global_cleanup();

    /* zero-terminate the result */
    data[write_result.pos] = '\0';

    return data;

error:
    if(data)
        free(data);
    if(curl)
        curl_easy_cleanup(curl);
    if(headers)
        curl_slist_free_all(headers);
    curl_global_cleanup();
    return NULL;
}

unsigned char nibbleFromChar(char c)
{
    if(c >= '0' && c <= '9') return c - '0';
    if(c >= 'a' && c <= 'f') return c - 'a' + 10;
    if(c >= 'A' && c <= 'F') return c - 'A' + 10;
    return 255;
}

/* Convert a string of characters representing a hex buffer into a series of bytes of that real value */
unsigned char *hexStringToBytes(unsigned char *inhex)
{
    unsigned char *retval;
    unsigned char *p;
    int len, i;
    
    len = strlen(inhex) / 2;
    retval = malloc(len+1);
    for(i=0, p = (unsigned char *) inhex; i<len; i++) {
        retval[i] = (nibbleFromChar(*p) << 4) | nibbleFromChar(*(p+1));
        p += 2;
    }
    retval[len] = 0;
    return retval;
}


void *proof_of_work(void *arg){
    char *proof_resp;
    char nonce_hex[9];
    json_t *json_setup, *json_resp;
    json_error_t json_error;
    json_t *result = NULL;
    while(1){
        //if(1){ testing
        if(checkTicket() == 1){
	    gettimeofday(&stop, NULL);
	    printf("%lu\n", (stop.tv_usec - start.tv_usec));
	    printf("Miner solved a block\n");
	    printf("Sending proof of work to Mining pool\n");
            
            //updating the data with solved nonce
            sprintf(nonce_hex, "%02x%02x%02x%02x", nonce[0],nonce[1],nonce[2],nonce[3]);
            printf("%s\n",nonce_hex);
            strncpy((char *)arg+152, nonce_hex, 8);
	       
            printf("current data in proof %s", (char *)arg );	      
            
	    json_setup = json_pack("{ss,s[s]si}","method", "getwork", "params", (char *)arg, "id",1);
    
            if(!json_setup){
                json_die(json_error.line, json_error.text);
            }

            proof_resp = request(pool_url, json_string_value(json_setup));
            if(!proof_resp){
                die("proof_of_work repsonse failed check args");
            }

            printf("resp %s\n", proof_resp);
            json_resp = json_loads(proof_resp, 0, &json_error);
  
            if(!json_resp){
                json_die(json_error.line, json_error.text);
            }
            result = json_object_get(json_resp, "result");

            json_decref(json_resp);
            json_decref(json_setup);
            printf("result %s\n", json_string_value(result));
            // break; testing         
	}        
    }
}

void
request_and_write_work(char * url){
    char *init_resp;
    json_t *json_resp;
    json_error_t json_error;
    json_t *data_error = NULL, *result = NULL, *data = NULL, *target = NULL, *midstate = NULL;

    init_resp = request(url, req);
    if(!init_resp){
        die("initial repsonse failed check args");
    }

    json_resp = json_loads(init_resp, 0, &json_error);

    if(!json_resp){
        json_die(json_error.line, json_error.text);
    }

    data_error = json_object_get(json_resp, "error");

    json_data_error(json_integer_value(data_error));

    result = json_object_get(json_resp, "result");
    data = json_object_get(result, "data");
    target = json_object_get(result, "target");
    midstate = json_object_get(result, "midstate");

    if(!result || !data || !target || !midstate){
        fprintf(stderr, "%s\n", init_resp);
        die("initial repsonse format error");
    }

    memcpy(current_data, init_resp+CHARS_TO_DATA, DATA_LENGTH);
 
    unsigned char *data_bytes = hexStringToBytes(json_string_value(data));
    unsigned char *midstate_bytes = hexStringToBytes(json_string_value(midstate));
    unsigned char *data_bytes = hexStringToBytes(data_test);
    unsigned char *midstate_bytes = hexStringToBytes(midstate_test);

    unsigned char header_buffer[HEADER_BUFFER_SIZE];
    memcpy(header_buffer, midstate_bytes, MIDSTATE_SIZE);
    memcpy(header_buffer+MIDSTATE_SIZE, data_bytes+DATA_SIZE, DATA_SIZE); 

    init_write();

    sleep(1);
    gettimeofday(&start, NULL);
    write_segments(header_buffer);
    printf("Json response from bitcoin network:\n %s\n", init_resp); //for testing
    printf("Writing Data to hardware:\n");
    print_segment_info();

    json_decref(json_resp);
    free(init_resp);
}

int
main(int argc, char**argv){
    static const char filename[] = "/dev/vga_led";
    if((vga_led_fd = open(filename, O_RDWR)) == -1){
        fprintf(stderr, "could not open %s\n", filename);
        return -1;
    }

    pthread_t ack_thread;
    pthread_create(&ack_thread, NULL, proof_of_work, current_data);

    init_write();

    sleep(1);
    print_segment_info();
    request_and_write_work(pool_url);

    //long polling hangs until there is new work to operate on
    while(1){
        request_and_write_work(lp_pool_url);
    }

    return 0;

}

// testing
    
/*       static unsigned char correct[96] = {
    0x90, 0xf7, 0x41, 0xaf, 0xb3, 0xab, 0x06, 0xf1, 0xa5, 0x82,
    0xc5, 0xc8, 0x5e, 0xe7, 0xa5, 0x61, 0x91, 0x2b, 0x25, 0xa7,
    0xcd, 0x09, 0xc0, 0x60, 0xa8, 0x9b, 0x3c, 0x2a, 0x73, 0xa4, 
    0x8e, 0x22, 0x15, 0x71, 0xd1, 0xbe, 0x4d, 0xe6, 0x95, 0x93, 
    0x1a, 0x26, 0x94, 0x21, 0x7a, 0x22, 0x22, 0x0e, 0x00, 0x00, 
    0x00, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
    0x00, 0x00, 0x80, 0x02, 0x00, 0x0};
*/
  //  unsigned char data_test[256] = "000000029b39574cb8a5c25e94cdd844f7d3b942384af65fb264d5a000000000000000008adb353c754fe07c40341728c5efe6e3782d994e0e3ebcb4879106de94dd1db0533b2c461900db9900000000000000800000000000000000000000000000000000000000000000000000000000000000000000000000000080020000\0" ;
//    unsigned char data_test[256] = "000000014cc2c57c7905fd399965282c87fe259e7da366e035dc087a0000141f000000006427b6492f2b052578fb4bc23655ca4e8b9e2b9b69c88041b2ac8c771571d1be4de695931a2694217a00110e000000800000000000000000000000000000000000000000000000000000000000000000000000000000000080020000\0";
//    unsigned char midstate_test[64] = "49bc9ada38ba6b43814a3275edfa09b796be70080d5c641d32b80776aef13f78\0";
//real test    unsigned char midstate_test[64] = "90f741afb3ab06f1a582c5c85ee7a561912b25a7cd09c060a89b3c2a73a48e22";
  //   unsigned char *data_bytes = hexStringToBytes(endian_flip_32_bit_chunks(data_test));
 //    unsigned char *midstate_bytes = hexStringToBytes(endian_flip_32_bit_chunks(midstate_test));

/*
    unsigned char *data_bytes = hexStringToBytes(data_test);
    unsigned char *midstate_bytes = hexStringToBytes(midstate_test);

    unsigned char header_buffer[96];
    memcpy(header_buffer, midstate_bytes, 32);
    memcpy(header_buffer+32, data_bytes+64, 64); 
*/
   //unsigned char test_thing[8] = "abcd1234\0";
   //unsigned char *new_thing = hexStringToBytes(endian_flip_32_bit_chunks(test_thing));
  /* 
    init_write();
    sleep(1);
    write_segments(header_buffer);

    sleep(1);
    printf("example Data\n");
    print_segment_info();
 */
   //printf("Result Ram:\n");
   // print_result();

/*
    init_write();
    sleep(1);
    write_segments(correct);
    sleep(1);
    printf("Correct Header Data\n");
    print_segment_info();
*/

//    printf("Result Ram:\n");
//    print_result();
    
   //request_work(pool_url);
   
//    sleep(5);
//    printf("check ticket = %d",checkTicket());


// static unsigned char incorrect[96] = {
// 0x90, 0xf7, 0x41, 0xaf, 0xb3, 0xab, 0x06, 0xf1, 0xa5, 0x82, 
// 0xc5, 0xc8, 0x5e, 0xe7, 0xa5, 0x61, 0x91, 0x2b, 0x25, 0xa7, 
// 0xcd, 0x09, 0xc0, 0x60, 0xa8, 0x9b, 0x3c, 0x2a, 0x73, 0xa4, 
// 0xff, 0xff, 0xff, 0xff, 0xff, 0xbe, 0x4d, 0xe6, 0x95, 0x93, 
// 0x1a, 0x26, 0x94, 0x21, 0x7a, 0x22, 0x22, 0x0e, 0x00, 0x00, 
// 0x00, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
// 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
// 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
// 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
// 0x00, 0x00, 0x80, 0x02, 0x00, 0x00};


//    printf("     // printf("\n");
     // int i;
     // for(i=3; i>-1; i--) {
     //  printf("%02x", nonce[i]);
     // }writing segments\n");


    // init_write();
    // sleep(1);
    // write_segments(incorrect);
    // sleep(1);
    // printf("Incorrect Header Data\n");
    // print_segment_info();
    // printf("Result Ram:\n");
    // print_result();

    // sleep(1);
    // checkTicket(1);
    // sleep(1);

    // while((!checkTicket());
     // printf("\n");
     // int i;
     // for(i=3; i>-1; i--) {
     //  printf("%02x", nonce[i]);
     // }
   
    //getwork from network
	//request_work(url);    
   //loop for long polling
    //while(1){
    
    //request_work(lp_pool_url);        
 
//    return 0;
//}


//Used for testing not in final implementation
/*
void swap(char *a, char *b){ 
  char temp;
  temp = *a;
  *a = *b;
  *b = temp;
}

//TESTS
// char test_thing[17] = "0123456701234567\0";
// printf("%s\n", endian_flip_32_bit_chunks(test_thing));
unsigned char *endian_flip_32_bit_chunks(unsigned char *input)
{
    //32 bits = 4*4 bytes = 4*4 chars
    for (int i = 0; i < strlen(input); i += 8){   
        swap(&input[i], &input[i+6]);
        swap(&input[i+1], &input[i+7]);
        swap(&input[i+2], &input[i+4]);
        swap(&input[i+3], &input[i+5]);
    }
    return input;        
}

unsigned char *bytesToStringHex(unsigned char *bin)
{
    unsigned int binsz = sizeof(bin);
    unsigned char          hex_str[]= "0123456789abcdef";
    unsigned int  i;
    unsigned char** result = NULL;

    *result = (unsigned char *)malloc(binsz * 2 + 1);
    (*result)[binsz * 2] = 0;

    for (i = 0; i < binsz; i++)
    {
        (*result)[i * 2 + 0] = hex_str[bin[i] >> 4  ];
        (*result)[i * 2 + 1] = hex_str[bin[i] & 0x0F];
    }  
    return *result;
}
*/
