/*
 * Seven-segment LED emulator
 *
 * Stephen A. Edwards, Columbia University
 */

module PIANO_DISPLAY(input logic 	     clk50, reset,
                     input logic [7:0]   pixel [0:1057],
							input logic [1:0]   pixel_RGB [0:351],
							input logic [1:0]   fix_pixel_RGB [0:351],
							input logic [1:0]   fix_pixel_RGB_denise [0:351],
							input logic [4:0]   fix_keys_position [0:351],
						   input logic [8:0]   row_num,
							input logic [3:0]   SW,
							input logic [3:0]   KEY,
							input logic [4:0]   key_num,
							input logic [7:0]   mode_select,
							input logic [7:0]   audio_demo,
							input logic [16:0]  press,
							input logic [7:0]   in_R1, in_G1, in_B1,
							input logic [7:0]   in_R2, in_G2, in_B2,
							input logic [7:0]   in_R3, in_G3, in_B3,
							input logic [7:0]   in_R4, in_G4, in_B4,			
							input logic [7:0]   in_R5, in_G5, in_B5,
                     output logic [7:0]  VGA_R, VGA_G, VGA_B,
                     output logic 	     VGA_CLK, VGA_HS, VGA_VS, VGA_BLANK_n, VGA_SYNC_n,
							output logic        clk_vga,
                     output logic [9:0]  h,v);
/*
 * 640 X 480 VGA timing for a 50 MHz clock: one pixel every other cycle
 * 
 * HCOUNT 1599 0             1279       1599 0
 *             _______________              ________
 * ___________|    Video      |____________|  Video
 * 
 * 
 * |SYNC| BP |<-- HACTIVE -->|FP|SYNC| BP |<-- HACTIVE
 *       _______________________      _____________
 * |____|       VGA_HS          |____|
 */
   // Parameters for hcount
   parameter HACTIVE      = 11'd 1280,
             HFRONT_PORCH = 11'd 32,
             HSYNC        = 11'd 192,
             HBACK_PORCH  = 11'd 96,   
             HTOTAL       = HACTIVE + HFRONT_PORCH + HSYNC + HBACK_PORCH; // 1600
   
   // Parameters for vcount
   parameter VACTIVE      = 10'd 480,
             VFRONT_PORCH = 10'd 10,
             VSYNC        = 10'd 2,
             VBACK_PORCH  = 10'd 33,
             VTOTAL       = VACTIVE + VFRONT_PORCH + VSYNC + VBACK_PORCH; // 525

   logic [10:0]			     hcount; // Horizontal counter
                                             // Hcount[10:1] indicates pixel column (0-639)
   logic 			     endOfLine;
    
   logic [7:0] real_VGA_R, real_VGA_G, real_VGA_B;
			
	integer byecounter;
	logic byechange;
	
	always_ff @(posedge clk50) begin
		if (byecounter == 30000000) begin
			byecounter <= 0;
			byechange <= byechange + 1;
		end else byecounter <= byecounter + 1;
	end
	
   always_ff @(posedge clk50 or posedge reset)
     if (reset)          hcount <= 0;
     else if (endOfLine) hcount <= 0;
     else  	         
	    begin
		   hcount <= hcount + 11'd 1;
			real_VGA_R <= pixel[(hcount[10:1] - 10'd146) * 3];
	      real_VGA_G <= pixel[(hcount[10:1] - 10'd146) * 3 + 1];
	      real_VGA_B <= pixel[(hcount[10:1] - 10'd146) * 3 + 2];
		 end

   assign endOfLine = hcount == HTOTAL - 1;

   // Vertical counter
   logic [9:0] 			     vcount;
   logic 			     endOfField;
	
   always_ff @(posedge clk50 or posedge reset)
     if (reset)          vcount <= 0;
     else if (endOfLine)
       if (endOfField)   vcount <= 0;
       else              vcount <= vcount + 10'd 1;

   assign endOfField = vcount == VTOTAL - 1;

   // Horizontal sync: from 0x520 to 0x5DF (0x57F)
   // 101 0010 0000 to 101 1101 1111
   assign VGA_HS = !( (hcount[10:8] == 3'b101) & !(hcount[7:5] == 3'b111));
   assign VGA_VS = !( vcount[9:1] == (VACTIVE + VFRONT_PORCH) / 2);

   assign VGA_SYNC_n = 1; // For adding sync to video signals; not used for VGA
   
   // Horizontal active: 0 to 1279     Vertical active: 0 to 479
   // 101 0000 0000  1280	       01 1110 0000  480
   // 110 0011 1111  1599	       10 0000 1100  524
   assign VGA_BLANK_n = !( hcount[10] & (hcount[9] | hcount[8]) ) &
			!( vcount[9] | (vcount[8:5] == 4'b1111) );   

   /* VGA_CLK is 25 MHz
    *             __    __    __
    * clk50    __|  |__|  |__|
    *        
    *             _____       __
    * hcount[0]__|     |_____|
    */
   assign VGA_CLK = hcount[0]; // 25 MHz clock: pixel latched on rising edge
	
	//word
	reg [43:0]  bit_next [0:13];
	assign bit_next[0]  = 44'b00000000000000000000000000000000011000011110;
	assign bit_next[1]  = 44'b00001100000000000000000000000000011000011110;
	assign bit_next[2]  = 44'b00001100000000000000000000000000011000110110;
	assign bit_next[3]  = 44'b00001100000000000000000000000000011000110110;
	assign bit_next[4]  = 44'b00111111000000000000000000000000011000110110;
	assign bit_next[5]  = 44'b00111111000110000110000011111000011001100110;
	assign bit_next[6]  = 44'b00001100000110000110001111111110011001100110;
	assign bit_next[7]  = 44'b00001100000011001100001100000110011001100110;
	assign bit_next[8]  = 44'b00001100000011111100001111111110011011000110;
	assign bit_next[9]  = 44'b00001100000000110000001111111110011011000110;
	assign bit_next[10]  = 44'b00001100000011001100000000000110011011000110;
	assign bit_next[11]  = 44'b00110000000111001110000000000110011110000110;
	assign bit_next[12]  = 44'b00110000000110000110001111111100011110000110;
	assign bit_next[13]  = 44'b00000000000000000000000000000000000000000000;

	
   reg [50:0]  bit_color [0:13];
	assign bit_color[0]  = 51'b000000000000000000000000000000000000000000000000000;
	assign bit_color[1]  = 51'b000000000000000000000000011000000000000000000000000;
	assign bit_color[2]  = 51'b000000000000000000000000011000000000000000000000000;
	assign bit_color[3]  = 51'b000000000000000000000000011000000000000000000000000;
	assign bit_color[4]  = 51'b000000000000000000000000011000000000000000000000000;
	assign bit_color[5]  = 51'b000111011000001111000000011000001111000000011111000;
	assign bit_color[6]  = 51'b001111111000001111000000011000001111000001111111110;
	assign bit_color[7]  = 51'b001100111000110000110000011000110000110001100000110;
	assign bit_color[8]  = 51'b000000011000110000110000011000110000110000000000110;
	assign bit_color[9]  = 51'b000000011000110000110000011000110000110000000000110;
	assign bit_color[10]  = 51'b000000011000110000110000011000110000110001100000110;
	assign bit_color[11]  = 51'b000000011000001111000001110000001111000001111111110;
	assign bit_color[12]  = 51'b000000011000001111000001100000001111000000011111000;
	assign bit_color[13]  = 51'b000000000000000000000000000000000000000000000000000;

	
	reg [96:0]  bit_calibration [0:13];
	assign bit_calibration[0]  = 97'b0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000;
	assign bit_calibration[1]  = 97'b0000000000000000000000000000011000000000000000000000000000000011000000001100000000000000000000000;
	assign bit_calibration[2]  = 97'b0000000000000000000000110000011000000000000000000000000000000011001100001100000000000000000000000;
	assign bit_calibration[3]  = 97'b0000000000000000000000110000011000000000000000000000000000000011001100001100000000000000000000000;
	assign bit_calibration[4]  = 97'b0000000000000000000000000001111110000000000000000000000000000011000000001100000000000000000000000;
	assign bit_calibration[5]  = 97'b0001111111000011110000110001111110000111110000011101100001110011001100001100001111100000011111000;
	assign bit_calibration[6]  = 97'b0001111111000011110000110000011000011000000000111111100001111111001100001100110000000001111111110;
	assign bit_calibration[7]  = 97'b0110000011001100001100110000011000011000000000110011100110001111001100001100110000000001100000110;
	assign bit_calibration[8]  = 97'b0110000011001100001100110000011000011111110000000001100110000011001100001100111111100000000000110;
	assign bit_calibration[9]  = 97'b0110000011001100001100110000011000011000001100000001100110000011001100001100110000011000000000110;
	assign bit_calibration[10]  = 97'b0110000011001100001100110000011000011000001100000001100110000011001100001100110000011001100000110;
	assign bit_calibration[11] = 97'b0110000011000011110000110001100000011111110000000001100001111111001100111000111111100001111111110;
	assign bit_calibration[12]  = 97'b0110000011000011110000110001100000011111110000000001100001111111001100110000111111100000011111000;
	assign bit_calibration[13]  = 97'b0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000;
                                                                                                                                  
	
	reg [72:0]  bit_confirm [0:13];
	assign bit_confirm[0]  = 73'b0000000000000000000000000000000000000000000000000000000000000000000000000;
	assign bit_confirm[1]  = 73'b0000000000000000000000000000001110000000000000000000000000000000000000000;
	assign bit_confirm[2]  = 73'b0000000000000000000000000110001111100000000000000000000000000000000000000;
	assign bit_confirm[3]  = 73'b0000000000000000000000000110000001100000000000000000000000000000000000000;
	assign bit_confirm[4]  = 73'b0000000000000000000000000000000001100000000000000000000000000000000000000;
	assign bit_confirm[5]  = 73'b0000110011000000111011000110001111111000001111111000001111000000011111000;
	assign bit_confirm[6]  = 73'b0000111111110001111111000110001111111000001111111000001111000001111111110;
	assign bit_confirm[7]  = 73'b0011001100110001100111000110000001100000110000011000110000110001100000110;
	assign bit_confirm[8]  = 73'b0011001100110000000011000110000001100000110000011000110000110000000000110;
	assign bit_confirm[9]  = 73'b0011001100110000000011000110000001100000110000011000110000110000000000110;
	assign bit_confirm[10]  = 73'b0011001100110000000011000110000001100000110000011000110000110001100000110;
	assign bit_confirm[11]  = 73'b0011001100110000000011000110000001100000110000011000001111000001111111110;
	assign bit_confirm[12]  = 73'b0011001100110000000011000110000001100000110000011000001111000000011111000;
	assign bit_confirm[13]  = 73'b0000000000000000000000000000000000000000000000000000000000000000000000000;

	reg [79:0]  bit_no_color [0:13];
	assign bit_no_color[0]  = 80'b00000000000000000000000000000000000000000000000000000000000000000000000000000000;
	assign bit_no_color[1]  = 80'b00000000000000000000000001100000000000000000000000000000000000000000000000000000;
	assign bit_no_color[2]  = 80'b00000000000000000000000001100000000000000000000000000000000000000000000000000000;
	assign bit_no_color[3]  = 80'b00000000000000000000000001100000000000000000000000000000000000000000000000000000;
	assign bit_no_color[4]  = 80'b00000000000000000000000001100000000000000000000000000000000000000000000000000000;
	assign bit_no_color[5]  = 80'b00011101100000111100000001100000111100000001111100000000000001111000000011111110;
	assign bit_no_color[6]  = 80'b00111111100000111100000001100000111100000111111111000000000001111000000011111110;
	assign bit_no_color[7]  = 80'b00110011100011000011000001100011000011000110000011000000000110000110001100000110;
	assign bit_no_color[8]  = 80'b00000001100011000011000001100011000011000000000011000000000110000110001100000110;
	assign bit_no_color[9]  = 80'b00000001100011000011000001100011000011000000000011000000000110000110001100000110;
	assign bit_no_color[10]  = 80'b00000001100011000011000001100011000011000110000011000000000110000110001100000110;
	assign bit_no_color[11]  = 80'b00000001100000111100000111000000111100000111111111000000000001111000001100000110;
	assign bit_no_color[12]  = 80'b00000001100000111100000110000000111100000001111100000000000001111000001100000110;
	assign bit_no_color[13]  = 80'b00000000000000000000000000000000000000000000000000000000000000000000000000000000;

	
   reg [94:0]  bit_recognize [0:13];
	assign bit_recognize[0]  = 95'b00000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000;
	assign bit_recognize[1]  = 95'b00000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000;
	assign bit_recognize[2]  = 95'b00000000000000000000000001100000000000000000000000000000000000000000000000000000000000000000000;
	assign bit_recognize[3]  = 95'b00000000000000000000000001100000000000000000000000000000000000000000000000000000000000000000000;
	assign bit_recognize[4]  = 95'b00000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000;
	assign bit_recognize[5]  = 95'b00001111100000111111110001100000111111000011111000000011110000000111110000000111110000001110110;
	assign bit_recognize[6]  = 95'b00111111111000111111110001100001111111100011111110000011110000011111111100011111111100011111110;
	assign bit_recognize[7]  = 95'b00110000011000011000000001100011000001100011000110001100001100011000001100011000001100011001110;
	assign bit_recognize[8]  = 95'b00111111111000001100000001100011000001100011000110001100001100000000001100011111111100000000110;
	assign bit_recognize[9]  = 95'b00111111111000000110000001100011000001100011111110001100001100000000001100011111111100000000110;
	assign bit_recognize[10]  = 95'b00000000011000000001100001100011000001100011111000001100001100011000001100000000001100000000110;
	assign bit_recognize[11]  = 95'b00000000011000111111110001100011000001100011000000000011110000011111111100000000001100000000110;
	assign bit_recognize[12]  = 95'b00111111110000111111110001100011000001100011000000000011110000000111110000011111111000000000110;
	assign bit_recognize[13]  = 95'b00000000000000000000000000000000000000000001111100000000000000000000000000000000000000000000000;

	
	reg [42:0]  bit_keys [0:13];
	assign bit_keys[0]  = 43'b0000000000000000000000000000000000000000000;
	assign bit_keys[1]  = 43'b0000000000000000000000000000000000000000110;
	assign bit_keys[2]  = 43'b0000000000000000000000000000000000000000110;
	assign bit_keys[3]  = 43'b0000000000000000000000000000000000000000110;
	assign bit_keys[4]  = 43'b0000000000000000000000000000000000110000110;
	assign bit_keys[5]  = 43'b0011111000001100011000001111100000110000110;
	assign bit_keys[6]  = 43'b0011111110001100011000111111111000011100110;
	assign bit_keys[7]  = 43'b0000000110001100011000110000011000001110110;
	assign bit_keys[8]  = 43'b0000000110001111111000111111111000000011110;
	assign bit_keys[9]  = 43'b0001111100001111100000111111111000000011110;
	assign bit_keys[10]  = 43'b0011111100001100000000000000011000001100110;
	assign bit_keys[11]  = 43'b0011000000001100000000000000011000111000110;
	assign bit_keys[12]  = 43'b0011111110001111111000111111110000110000110;
	assign bit_keys[13]  = 43'b0000000000000111111000000000000000000000000;

	
	reg [43:0]  bit_free [0:13];
	assign bit_free[0]  = 44'b00000000000000000000000000000000000000000000;
	assign bit_free[1]  = 44'b00000000000000000000000000000000000011100000;
	assign bit_free[2]  = 44'b00000000000000000000000000000000000011111000;
	assign bit_free[3]  = 44'b00000000000000000000000000000000000000011000;
	assign bit_free[4]  = 44'b00000000000000000000000000000000000000011000;
	assign bit_free[5]  = 44'b00001111100000001111100000011101100011111110;
	assign bit_free[6]  = 44'b00111111111000111111111000111111100011111110;
	assign bit_free[7]  = 44'b00110000011000110000011000110011100000011000;
	assign bit_free[8]  = 44'b00111111111000111111111000000001100000011000;
	assign bit_free[9]  = 44'b00111111111000111111111000000001100000011000;
	assign bit_free[10]  = 44'b00000000011000000000011000000001100000011000;
	assign bit_free[11]  = 44'b00000000011000000000011000000001100000011000;
	assign bit_free[12]  = 44'b00111111110000111111110000000001100000011000;
	assign bit_free[13]  = 44'b00000000000000000000000000000000000000000000;
	

	reg [47:0]  bit_mode [0:13];
	assign bit_mode[0]  = 48'b000000000000000000000000000000000000000000000000;
	assign bit_mode[1]  = 48'b000000000000001100000000000000000000000000000000;
	assign bit_mode[2]  = 48'b000000000000001100000000000000000000000000000000;
	assign bit_mode[3]  = 48'b000000000000001100000000000000000000000000000000;
	assign bit_mode[4]  = 48'b000000000000001100000000000000000000000000000000;
	assign bit_mode[5]  = 48'b000011111000001100111000000011110000001110011100;
	assign bit_mode[6]  = 48'b001111111110001111111000000011110000011111111110;
	assign bit_mode[7]  = 48'b001100000110001111000110001100001100011001100110;
	assign bit_mode[8]  = 48'b001111111110001100000110001100001100011001100110;
	assign bit_mode[9]  = 48'b001111111110001100000110001100001100011001100110;
	assign bit_mode[10]  = 48'b000000000110001100000110001100001100011001100110;
	assign bit_mode[11]  = 48'b000000000110001111111000000011110000011001100110;
	assign bit_mode[12]  = 48'b001111111100001111111000000011110000011001100110;
	assign bit_mode[13]  = 48'b000000000000000000000000000000000000000000000000;

	
	reg [81:0]  bit_auto_play [0:13];
	assign bit_auto_play[0]  = 82'b0000000000000000000000000000000000000000000000000000000000000000000000000000000000;
	assign bit_auto_play[1]  = 82'b0000000000000000000000000011000000000000000000000000001100000000000000000000000000;
	assign bit_auto_play[2]  = 82'b0000000000000000000000000011000000000000000000000000001100000000000000000000000000;
	assign bit_auto_play[3]  = 82'b0000000000000000000000000011000000000000000000000000001100000000000000000000000000;
	assign bit_auto_play[4]  = 82'b0000000000000000000000000011000000000000000000000000111111000000000000000000000000;
	assign bit_auto_play[5]  = 82'b0011000110000011111000000011000011101100000111100000111111000110000110000011111000;
	assign bit_auto_play[6]  = 82'b0011000110001100000000000011000011111100000111100000001100000110000110001100000000;
	assign bit_auto_play[7]  = 82'b0011000110001100000000000011000110011100011000011000001100000110000110001100000000;
	assign bit_auto_play[8]  = 82'b0011111110001111111000000011000110001100011000011000001100000110000110001111111000;
	assign bit_auto_play[9]  = 82'b0011111000001100000110000011000011111100011000011000001100000110000110001100000110;
	assign bit_auto_play[10]  = 82'b0011000000001100000110000011000011111100011000011000001100000110000110001100000110;
	assign bit_auto_play[11]  = 82'b0011000000001111111000001110000000001100000111100000110000000111111000001111111000;
	assign bit_auto_play[12]  = 82'b0011111110001111111000001100000000001100000111100000110000000111111000001111111000;
	assign bit_auto_play[13]  = 82'b0001111110000000000000000000000000001100000000000000000000000000000000000000000000;

	
	reg [46:0]  bit_game [0:13];
	assign bit_game[0]   = 47'b00000000000000000000000000000000000000000000000;
	assign bit_game[1]   = 47'b00000000000000000000000000000000000000000000000;
	assign bit_game[2]   = 47'b00000000000000000000000000000000000000000000000;
	assign bit_game[3]   = 47'b00000000000000000000000000000000000000000000000;
	assign bit_game[4]   = 47'b00000000000000000000000000000000000000000000000;
	assign bit_game[5]   = 47'b00001111100000011100111000000111110000011111000;
	assign bit_game[6]   = 47'b00111111111000111111111100011000000000011111110;
	assign bit_game[7]   = 47'b00110000011000110011001100011000000000011000110;
	assign bit_game[8]   = 47'b00111111111000110011001100011111110000011000110;
	assign bit_game[9]   = 47'b00111111111000110011001100011000001100011111110;
	assign bit_game[10]   = 47'b00000000011000110011001100011000001100011111000;
	assign bit_game[11]   = 47'b00000000011000110011001100011111110000011000000;
	assign bit_game[12]   = 47'b00111111110000110011001100011111110000011000000;
	assign bit_game[13]   = 47'b00000000000000000000000000000000000000001111100;
	
	reg [36:0] bit_exit [0:13];
	assign bit_exit[0] = 37'b0000000000000000000000000000000000000;
	assign bit_exit[1] = 37'b0000110000000000000000000000000000000;
	assign bit_exit[2] = 37'b0000110000011000000000000000000000000;
	assign bit_exit[3] = 37'b0000110000011000000000000000000000000;
	assign bit_exit[4] = 37'b0011111100000000000000000000000000000;
	assign bit_exit[5] = 37'b0011111100011000110000110000011111000;
	assign bit_exit[6] = 37'b0000110000011000110000110001111111110;
	assign bit_exit[7] = 37'b0000110000011000011001100001100000110;
	assign bit_exit[8] = 37'b0000110000011000011111100001111111110;
	assign bit_exit[9] = 37'b0000110000011000000110000001111111110;
	assign bit_exit[10] = 37'b0000110000011000011001100000000000110;
	assign bit_exit[11] = 37'b0011000000011000111001110000000000110;
	assign bit_exit[12] = 37'b0011000000011000110000110001111111100;
	assign bit_exit[13] = 37'b0000000000000000000000000000000000000;

  
	
	reg [69:0]  bit_back_to [0:13];
	assign bit_back_to[0]  = 70'b0000000000000000000000000000000000000000000000000000000000000000000000;
	assign bit_back_to[1]  = 70'b0000000000000001100000000000000110000000000000000000000000000000000110;
	assign bit_back_to[2]  = 70'b0000000000000001100000000000000110000000000000000000000000000000000110;
	assign bit_back_to[3]  = 70'b0000000000000001100000000000000110000000000000000000000000000000000110;
	assign bit_back_to[4]  = 70'b0000000000000111111000000110000110000000000000000000000000000000000110;
	assign bit_back_to[5]  = 70'b0000111100000111111000000110000110000011111000000011111000000011100110;
	assign bit_back_to[6]  = 70'b0000111100000001100000000011100110001111111110001100000000000011111110;
	assign bit_back_to[7]  = 70'b0011000011000001100000000001110110001100000110001100000000001100011110;
	assign bit_back_to[8]  = 70'b0011000011000001100000000000011110000000000110001111111000001100000110;
	assign bit_back_to[9]  = 70'b0011000011000001100000000000011110000000000110001100000110001100000110;
	assign bit_back_to[10]  = 70'b0011000011000001100000000001100110001100000110001100000110001100000110;
	assign bit_back_to[11]  = 70'b0000111100000110000000000111000110001111111110001111111000000011111110;
	assign bit_back_to[12]  = 70'b0000111100000110000000000110000110000011111000001111111000000011111110;
	assign bit_back_to[13]  = 70'b0000000000000000000000000000000000000000000000000000000000000000000000;

	
	reg [67:0]  bit_screen [0:13];
	assign bit_screen[0]  = 68'b00000000000000000000000000000000000000000000000000000000000000000000;
	assign bit_screen[1]  = 68'b00000000000000000000000000000000000000000000000000000000000000000000;
	assign bit_screen[2]  = 68'b00000000000000000000000000000000000000000000000000000000000000000000;
	assign bit_screen[3]  = 68'b00000000000000000000000000000000000000000000000000000000000000000000;
	assign bit_screen[4]  = 68'b00000000000000000000000000000000000000000000000000000000000000000000;
	assign bit_screen[5]  = 68'b00001111110000001111100000001111100000011101100000111110000011111000;
	assign bit_screen[6]  = 68'b00011111111000111111111000111111111000111111100011111111100011111110;
	assign bit_screen[7]  = 68'b00110000011000110000011000110000011000110011100011000001100000000110;
	assign bit_screen[8]  = 68'b00110000011000111111111000111111111000000001100000000001100000000110;
	assign bit_screen[9]  = 68'b00110000011000111111111000111111111000000001100000000001100001111100;
	assign bit_screen[10]  = 68'b00110000011000000000011000000000011000000001100011000001100011111100;
	assign bit_screen[11]  = 68'b00110000011000000000011000000000011000000001100011111111100011000000;
	assign bit_screen[12]  = 68'b00110000011000111111110000111111110000000001100000111110000011111110;
	assign bit_screen[13]  = 68'b00000000000000000000000000000000000000000000000000000000000000000000;
                                                                                                 
	
	reg [174:0] bit_screen_calibration [0:13];
	assign bit_screen_calibration[0]  = 175'b0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000001111100000000000000000000000000000000000000000000000000000000000000011111000;
	assign bit_screen_calibration[1]  = 175'b0000000000000000000000000000000011000000000000000000000000000000000011000000000011000000000000000111111110000000000000000000000000000000000000000000000000000000000000011111110;
	assign bit_screen_calibration[2]  = 175'b0000000000000000000000000110000011000000000000000000000000000000000011000110000011000000000000000110000110000000000000000000000000000000000000000000000000000000000000000000110;
	assign bit_screen_calibration[3]  = 175'b0000000000000000000000000110000011000000000000000000000000000000000011000110000011000000000000000000000011000000000000000000000000000000000000000000000000000000000000000000110;
	assign bit_screen_calibration[4]  = 175'b0000000000000000000000000000001111110000000000000000000000000000000011000000000011000000000000000000000011000000000000000000000000000000000000000000000000000000000000000000110;
	assign bit_screen_calibration[5]  = 175'b0000111111000000111100000110001111110000011111000000111011000001110011000110000011000001111100000000000011000001111110000001111100000001111100000011101100000111110000000000110;
	assign bit_screen_calibration[6]  = 175'b0001111111100000111100000110000011000001100000000001111111000001111111000110000011000110000000000000000011000011111111000111111111000111111111000111111100011111111100001111100;
	assign bit_screen_calibration[7]  = 175'b0011000001100011000011000110000011000001100000000001100111000110001111000110000011000110000000000000000011000110000011000110000011000110000011000110011100011000001100011111100;
	assign bit_screen_calibration[8]  = 175'b0011000001100011000011000110000011000001111111000000000011000110000011000110000011000111111100000000000011000110000011000111111111000111111111000000001100000000001100011000000;
	assign bit_screen_calibration[9]  = 175'b0011000001100011000011000110000011000001100000110000000011000110000011000110000011000110000011000000000011000110000011000111111111000111111111000000001100000000001100011000000;
	assign bit_screen_calibration[10]  = 175'b0011000001100011000011000110000011000001100000110000000011000110000011000110000011000110000011000110000010000110000011000000000011000000000011000000001100011000001100011000000;
	assign bit_screen_calibration[11]  = 175'b0011000001100000111100000110001100000001111111000000000011000001111111000110001110000111111100000111111110000110000011000000000011000000000011000000001100011111111100011111110;
	assign bit_screen_calibration[12]  = 175'b0011000001100000111100000110001100000001111111000000000011000001111111000110001100000111111100000001111100000110000011000111111110000111111110000000001100000111110000001111110;
	assign bit_screen_calibration[13]  = 175'b0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000;

	
	reg [160:0] bit_color_calibration [0:13];
	assign bit_color_calibration[0]  = 161'b00000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000011111000000000000000000000000000000000000000000000000011111000;
	assign bit_color_calibration[1]  = 161'b00000000000000000000000000000000110000000000000000000000000000000000110000000000110000000000000001111111100000000000000000000000000000011000000000000001111111100;
	assign bit_color_calibration[2]  = 161'b00000000000000000000000001100000110000000000000000000000000000000000110001100000110000000000000001100001100000000000000000000000000000011000000000000001100001100;
	assign bit_color_calibration[3]  = 161'b00000000000000000000000001100000110000000000000000000000000000000000110001100000110000000000000000000000110000000000000000000000000000011000000000000000000000110;
	assign bit_color_calibration[4]  = 161'b00000000000000000000000000000011111100000000000000000000000000000000110000000000110000000000000000000000110000000000000000000000000000011000000000000000000000110;
	assign bit_color_calibration[5]  = 161'b00001111111000001111000001100011111100000111110000001110110000011100110001100000110000011111000000000000110000000111011000001111000000011000001111000000000000110;
	assign bit_color_calibration[6]  = 161'b00001111111000001111000001100000110000011000000000011111110000011111110001100000110001100000000000000000110000001111111000001111000000011000001111000000000000110;
	assign bit_color_calibration[7]  = 161'b00110000011000110000110001100000110000011000000000011001110001100011110001100000110001100000000000000000110000001100111000110000110000011000110000110000000000110;
	assign bit_color_calibration[8]  = 161'b00110000011000110000110001100000110000011111110000000000110001100000110001100000110001111111000000000000110000000000011000110000110000011000110000110000000000110;
	assign bit_color_calibration[9]  = 161'b00110000011000110000110001100000110000011000001100000000110001100000110001100000110001100000110000000000110000000000011000110000110000011000110000110000000000110;
	assign bit_color_calibration[10]  = 161'b00110000011000110000110001100000110000011000001100000000110001100000110001100000110001100000110001100000100000000000011000110000110000011000110000110001100000100;
	assign bit_color_calibration[11]  = 161'b00110000011000001111000001100011000000011111110000000000110000011111110001100011100001111111000001111111100000000000011000001111000001110000001111000001111111100;
	assign bit_color_calibration[12]  = 161'b00110000011000001111000001100011000000011111110000000000110000011111110001100011000001111111000000011111000000000000011000001111000001100000001111000000011111000;
	assign bit_color_calibration[13]  = 161'b00000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000;                                                                                                                                                                                                              
	
	reg [334:0] bit_adjust1 [0:13];
	assign bit_adjust1[0]   = 335'b00000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000111100000;
	assign bit_adjust1[1]   = 335'b00000000000000000000000000000000011000000000000000000000000000000000000000000000000000000000000000000000000000000000011000000000011000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000011000001100000000001100000000000000000000000000000000000011000000000000111100000;
	assign bit_adjust1[2]   = 335'b00000000000000000000000000110000011000000000000000000000000000000000000000000000000000000000000000000000000000000000011000110000011000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000011000001100000000001100000000000000000000000000110000000011000000000001100110000;
	assign bit_adjust1[3]   = 335'b00000000000000000000000000110000011000000000000000000000000000000000000000000000000000000000000000000000000000000000011000110000011000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000011000001100000000001100000000000000000000000000110000000011000000000001100110000;
	assign bit_adjust1[4]   = 335'b00000000000000000000000000000000011000000000000000000000000000000000000000000000000000000000000000000000000000000000011000000001111110000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000011000111111000000111111000000000000000000000000000000000011000000000001100110000;
	assign bit_adjust1[5]   = 335'b00001111100000001111110000110000011000000001111110000001111100000001111100000011101100011111000000000011111000000000011000110001111110000011111100001100001100000000111110000001110110000011111000000111001110000001111100000001111100000000001111100000001110011000111111000000111111000111110000011000011000000000000011001110000011000011000;
	assign bit_adjust1[6]   = 335'b00111111111000011111111000110000011000000011111111000111111111000111111111000111111100011111110000001100000000000000011000110000011000000111111110001100001100000011000000000011111110001111111110001111111111000110000000000111111111000000111111111000001111111000001100000000001100000111111100011000011000110000000011111110000011000011000;
	assign bit_adjust1[7]   = 335'b00110000011000110000011000110000011000000110000011000110000011000110000011000110011100011000110000001100000000000000011000110000011000001100000110001100001100000011000000000011001110001100000110001100110011000110000000000110000011000000110000011000110001111000001100000000001100000000001100011000011000110000000011110001100011111111000;
	assign bit_adjust1[8]   = 335'b00111111111000110000011000110000011000000110000011000111111111000111111111000000001100011000110000001111111000000000011000110000011000001100000110001100001100000011111110000000000110001111111110001100110011000111111100000000000011000000111111111000110000011000001100000000001100000000001100011000011000110000000011000001100111111111100;
	assign bit_adjust1[9]   = 335'b00111111111000110000011000110000011000000110000011000111111111000111111111000000001100011111110000001100000110000000011000110000011000001100000110001100001100000011000001100000000110001111111110001100110011000110000011000000000011000000111111111000110000011000001100000000001100000011111000011000011000110000000011000001100110000001100;
	assign bit_adjust1[10]   = 335'b00000000011000110000011000110000011000000110000011000000000011000000000011000000001100011111000000001100000110000000011000110000011000001100000110001100001100000011000001100000000110000000000110001100110011000110000011000110000011000000000000011000110000011000001100000000001100000111111000011000011000110000000011000001100110000001100;
	assign bit_adjust1[11]   = 335'b00000000011000110000011000110001110000000110000011000000000011000000000011000000001100011000000000001111111000000001110000110001100000001100000110001111110000000011111110000000000110000000000110001100110011000111111100000111111111000000000000011000110000011000110000000000110000000110000000011111100000110001100011111110001100000000110;
	assign bit_adjust1[12]   = 335'b00111111110000110000011000110001100000000110000011000111111110000111111110000000001100011000000000001111111000000001100000110001100000001100000110001111110000000011111110000000000110001111111100001100110011000111111100000001111100000000111111110000110000011000110000000000110000000111111100011111100000111111100011111110001100000000110;
	assign bit_adjust1[13]   = 335'b00000000000000000000000000000000000000000000000000000000000000000000000000000000000000001111100000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000011110000000000000000000000000000;

	
	reg [367:0] bit_adjust2 [0:13];
	assign bit_adjust2[0]   = 368'b00000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000;
	assign bit_adjust2[1]   = 368'b00000000000000000000000000000000000000000000000000000000000000000000000000000110000011000000000000000000000000000000000000000000000001100000110000000011100000000000000000000000000000000000110001100000000001100000000000000000000000000000000000000000000000000110000011000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000;
	assign bit_adjust2[2]   = 368'b00000000000000000000000000000000000000000000000000000000000000000000000000000110000011000000000000000000000000000000000000000000000001100000110000000011111000000000000000000000000000000000110001100000000001100000000001100000000000000000000000000000000000000110000011000000000000000000001100000000000000000000000000000000000000000000000000000000000000000000000000000000;
	assign bit_adjust2[3]   = 368'b00000000000000000000000000000000000000000000000000000000000000000000000000000110000011000000000000000000000000000000000000000000000001100000110000000000011000000000000000000000000000000000110001100000000001100000000001100000000000000000000000000000000000000110000011000000000000000000001100000000000000000000000000000000000000000000000000000000000000000000000000000000;
	assign bit_adjust2[4]   = 368'b00000000000000000000000000000000000000000000000000000000000000000000000000000110000011000000000000000000000000000000000000000000000001100011111100000000011000000000000000000000000000000000110001100000000001100000000000000000000000000000000000000000000000000110001111110000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000;
	assign bit_adjust2[5]   = 368'b00001111100000001111100000011101100000111110000000011000000110000011110000000110000011000001111100000110001100000000111110000000111001100011111100000011111110000011110000000000111110000000110001100111000001100111000001100001110011100000000011111000000011100110001111110000000011111100001100000011111000000111011000001111100000001111100000011101100001110110000011111000;
	assign bit_adjust2[6]   = 368'b00110000000000111111111000111111100011000000000000011000000110000011110000000110000011000111111111000110001100000011111111100000111111100000110000000011111110000011110000000011111111100000110001111111000001111111000001100011111111110000001111111110000011111110000011000000000111111110001100000011111110001111111000110000000000111111111000011111100001111110001100000000;
	assign bit_adjust2[7]   = 368'b00110000000000110000011000110011100011000000000000011001100110001100001100000110000011000110000011000110001100000011000001100011000111100000110000000000011000001100001100000011000001100000110001111000110001111000110001100011001100110000001100000110001100011110000011000000001100000110001100000000000110001100111000110000000000110000011000110011100011001110001100000000;
	assign bit_adjust2[8]   = 368'b00111111100000111111111000000001100011111110000000011001100110001100001100000110000011000111111111000111111100000011111111100011000001100000110000000000011000001100001100000011111111100000110001100000110001100000110001100011001100110000001111111110001100000110000011000000001100000110001100000000000110000000011000111111100000111111111000110001100011000110001111111000;
	assign bit_adjust2[9]   = 368'b00110000011000111111111000000001100011000001100000011001100110001100001100000110000011000111111111000111110000000011111111100011000001100000110000000000011000001100001100000011111111100000110001100000110001100000110001100011001100110000001111111110001100000110000011000000001100000110001100000001111100000000011000110000011000111111111000011111100001111110001100000110;
	assign bit_adjust2[10]   = 368'b00110000011000000000011000000001100011000001100000011001100110001100001100000110000011000000000011000110000000000000000001100011000001100000110000000000011000001100001100000000000001100000110001100000110001100000110001100011001100110000000000000110001100000110000011000000001100000110001100000011111100000000011000110000011000000000011000011111100001111110001100000110;
	assign bit_adjust2[11]   = 368'b00111111100000000000011000000001100011111110000000001111111100000011110000011100001110000000000011000110000000000000000001100011000001100011000000000000011000000011110000000000000001100011100001111111000001111111000001100011001100110000000000000110001100000110001100000000001100000110001100000011000000000000011000111111100000000000011000000001100000000110001111111000;
	assign bit_adjust2[12]   = 368'b00111111100000111111110000000001100011111110000000000111111000000011110000011000001100000111111110000111111100000011111111000011000001100011000000000000011000000011110000000011111111000011000001111111000001111111000001100011001100110000001111111100001100000110001100000000001100000110001100000011111110000000011000111111100000111111110000000001100000000110001111111000;
	assign bit_adjust2[13]   = 368'b00000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000011111100000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000001100000000110000000000000;

  
	reg [334:0] bit_autoplay [0:13];
	assign bit_autoplay[0] = 335'b00000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000;
	assign bit_autoplay[1] = 335'b00001100000000000000000000000000000000000000000000000000000000000000000000011000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000011000000001110000000000000000000000000000011000001100000000000000000000000000000000000000000011000000000000000000000000000000000000000000000000000000001100000000000;
	assign bit_autoplay[2] = 335'b00001100000000000000000000000000011000000000000000000000000000000000000000011000000000000000000001100000000000000000000000000000000000000000000000000000000000000000000000011000110001111100000000000000000000000000011000001100000000000000000000000000000000000000000011000000000000000000000000000000000000000000000000000000001100000000000;
	assign bit_autoplay[3] = 335'b00001100000000000000000000000000011000000000000000000000000000000000000000011000000000000000000001100000000000000000000000000000000000000000000000000000000000000000000000011000110000001100000000000000000000000000011000001100000000000000000000000000000000000000000011000000000000000000000000000000000000000000000000000000001100000000000;
	assign bit_autoplay[4] = 335'b00001100000000000000000000000000000000000000000000000000000000000000000001111110000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000011000000000001100000000000000000000000000011000111111000000000000000000000000000000000000001111110000000000000000000000000000000000000000000000000000001100000000000;
	assign bit_autoplay[5] = 335'b00001100000111110000000111111000011000011100111000001110110000011111000001111110000000011111100001100000000111110000001110011100000011111000000011111100000000011111000000011000110001111111000000001111100000001110011000111111000000001111100000011101100011000110001111110000000011111000001111100000001111100000001111100000001100001110110;
	assign bit_autoplay[6] = 335'b00001100011000000000001111111100011000111111111100011111110001111111110000011000000000111111110001100000011111111100011111111110001100000000000111111110000001111111110000011000110001111111000000111111111000001111111000001100000000111111111000011111100011000110000011000000001111111110001111111000110000000000111111111000001100001111110;
	assign bit_autoplay[7] = 335'b00001100011000000000011000001100011000110011001100011001110001100000110000011000000001100000110001100000011000001100011001100110001100000000001100000110000001100000110000011000110000001100000000110000011000110001111000001100000000110000011000110011100011000110000011000000001100000110000000011000110000000000110000011000001100011001110;
	assign bit_autoplay[8] = 335'b00001100011111110000011000001100011000110011001100000000110001111111110000011000000001100000110001100000011111111100011001100110001111111000001100000110000001111111110000011000110000001100000000111111111000110000011000001100000000111111111000110001100011111110000011000000001111111110000111110000111111100000111111111000001100011000110;
	assign bit_autoplay[9] = 335'b00001100011000001100011000001100011000110011001100000000110001111111110000011000000001100000110001100000011111111100011001100110001100000110001100000110000001111111110000011000110000001100000000111111111000110000011000001100000000111111111000011111100011111000000011000000001111111110001111110000110000011000111111111000001100001111110;
	assign bit_autoplay[10] = 335'b00001100011000001100011000001100011000110011001100000000110000000000110000011000000001100000110001100000000000001100011001100110001100000110001100000110000000000000110000011000110000001100000000000000011000110000011000001100000000000000011000011111100011000000000011000000000000000110001100000000110000011000000000011000001100001111110;
	assign bit_autoplay[11] = 335'b00111000011111110000011000001100011000110011001100000000110000000000110001100000000001100000110001100000000000001100011001100110001111111000001100000110000000000000110001110000110000001100000000000000011000110000011000110000000000000000011000000001100011000000001100000000000000000110001111111000111111100000000000011000111000000000110;
	assign bit_autoplay[12] = 335'b00110000011111110000011000001100011000110011001100000000110001111111100001100000000001100000110001100000011111111000011001100110001111111000001100000110000001111111100001100000110000001100000000111111110000110000011000110000000000111111110000000001100011111110001100000000001111111100000111111000111111100000111111110000110000000000110;
	assign bit_autoplay[13] = 335'b00000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000001100001111110000000000000000000000000000000000000000000000000000000000000000000000000110;
 
	
	reg [205:0] bit_freemode [0:13];
	assign bit_freemode[0] = 206'b00000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000;
	assign bit_freemode[1] = 206'b00000000000000000000110000000000110000000000000000000000000000000000000000000000000000000000000000000000000000000000000000110000000000000000000000001100000000000000000000000000000000000000000001100000000000;
	assign bit_freemode[2] = 206'b00000000000000000000110001100000110000000000000000000000000000000000000000000000000000000000000000000000000000000000000000110000000000000000000000001100000000000000000000000000000000000000000001100000000000;
	assign bit_freemode[3] = 206'b00000000000000000000110001100000110000000000000000000000000000000000000000000000000000000000000000000000000000000000000000110000000000000000000000001100000000000000000000000000000000000000000001100000000000;
	assign bit_freemode[4] = 206'b0000000000000110000110000000000110000000000000000000000000000000000000000000000000000000000000000000000000000000000000011111100000000000000000000001100000000000000000000000000000000000000000001100000000000;
	assign bit_freemode[5] = 206'b00001111100000110000110001100000110000001100001100000111100000110001100000001110110000011111000001100001100000111110000011111100000111110000000111001100011000000110000001100011000001111100000001100001110110;
	assign bit_freemode[6] = 206'b00111111111000011100110001100000110000001100001100000111100000110001100000011111110001111111110001100001100011111111100000110000011000000000000111111100011000000110000001100011000110000000000001100001111110;
	assign bit_freemode[7] = 206'b00110000011000001110110001100000110000001100001100011000011000110001100000011001110001100000110001100001100011000001100000110000011000000000011000111100011001100110000001100011000110000000000001100011001110;
	assign bit_freemode[8] = 206'b00111111111000000011110001100000110000001100001100011000011000111111100000000000110001111111110001100001100011111111100000110000011111110000011000001100011001100110000001111111000111111100000001100011000110;
	assign bit_freemode[9] = 206'b00111111111000000011110001100000110000001100001100011000011000111110000000000000110001111111110001100001100011111111100000110000011000001100011000001100011001100110000001111100000110000011000001100001111110;
	assign bit_freemode[10] = 206'b00000000011000001100110001100000110000001100001100011000011000110000000000000000110000000000110000110011000000000001100000110000011000001100011000001100011001100110000001100000000110000011000001100001111110;
	assign bit_freemode[11] = 206'b00000000011000111000110001100011100000001111110000000111100000110000000000000000110000000000110000011110000000000001100011000000011111110000011000001100001111111100000001100000000111111100000111000000000110;
	assign bit_freemode[12] = 206'b00111111110000110000110001100011000000001111110000000111100000111111100000000000110001111111100000001100000011111111000011000000011111110000011000001100000111111000000001111111000111111100000110000000000110;
	assign bit_freemode[13] = 206'b00000000000000000000000000000000000000000000000000000000000000011111100000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000111111000000000000000000000000000110;
                                                                                                                                                                                                                                             
	
	reg [195:0] bit_gamemode [0:13];
	assign bit_gamemode[0] = 196'b0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000;
	assign bit_gamemode[1] = 196'b0000000000000000000000000000000000001100000000000000000000000000000000000000000000000000000000000000000000000000000000000000110000000000000000000000000000000000000001100000110000000000000011100000;
	assign bit_gamemode[2] = 196'b0000000000000000000000000000000000001100000000000000000000000000000000000000001100000000000000000000000000000000000000000000110000000000000000000000000000000000000001100000110000000000000011111000;
	assign bit_gamemode[3] = 196'b0000000000000000000000000000000000001100000000000000000000000000000000000000001100000000000000000000000000000000000000000000110000000000000000000000000000000000000001100000110000000000000000011000;
	assign bit_gamemode[4] = 196'b0000000000000000000000000000000000111111000000000000000000000000000000000000000000000000000000000000000000000000000000000000110000000000000000000000000000000000000001100000110000000000000000011000;
	assign bit_gamemode[5] = 196'b0011111000000111011000001111100000111111000111110000000011111000000011111100001100001110110000111011000001111000000111011000110011100000000110000001100000111100000001100000110000011110000011111110;
	assign bit_gamemode[6] = 196'b0011111110001111111000110000000000001100000111111100000011111110000111111110001100001111110000111111000001111000001111111000111111100000000110000001100000111100000001100000110000011110000011111110;
	assign bit_gamemode[7] = 196'b0000000110001100111000110000000000001100000000001100000011000110001100000110001100011001110001100111000110000110001100111000111100011000000110011001100011000011000001100000110001100001100000011000;
	assign bit_gamemode[8] = 196'b0001111100000000011000111111100000001100000011111000000011000110001100000110001100011000110001100011000110000110000000011000110000011000000110011001100011000011000001100000110001100001100000011000;
	assign bit_gamemode[9] = 196'b0011111100000000011000110000011000001100000111111000000011111110001100000110001100001111110000111111000110000110000000011000110000011000000110011001100011000011000001100000110001100001100000011000;
	assign bit_gamemode[10] = 196'b0011000000000000011000110000011000001100000110000000000011111000001100000110001100001111110000111111000110000110000000011000110000011000000110011001100011000011000001100000110001100001100000011000;
	assign bit_gamemode[11] = 196'b0011111110000000011000111111100000110000000111111100000011000000001100000110001100000000110000000011000001111000000000011000111111100000000011111111000000111100000111000011100000011110000000011000;
	assign bit_gamemode[12] = 196'b0001111110000000011000111111100000110000000011111100000011000000001100000110001100000000110000000011000001111000000000011000111111100000000001111110000000111100000110000011000000011110000000011000;
	assign bit_gamemode[13] = 196'b0000000000000000000000000000000000000000000000000000000001111100000000000000000000000000110000000011000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000;

	
	reg[22:0] bit_dao [0:13];
	assign bit_dao[0] = 23'b00000000000000000000000;
	assign bit_dao[1] = 23'b00000000000001100000000;
	assign bit_dao[2] = 23'b00000000000001100000000;
	assign bit_dao[3] = 23'b00000000000001100000000;
	assign bit_dao[4] = 23'b00000000000001100000000;
	assign bit_dao[5] = 23'b00001111000001100111000;
	assign bit_dao[6] = 23'b00001111000001111111000;
	assign bit_dao[7] = 23'b00110000110001111000110;
	assign bit_dao[8] = 23'b00110000110001100000110;
	assign bit_dao[9] = 23'b00110000110001100000110;
	assign bit_dao[10] = 23'b00110000110001100000110;
	assign bit_dao[11] = 23'b00001111000001111111000;
	assign bit_dao[12] = 23'b00001111000001111111000;
	assign bit_dao[13] = 23'b00000000000000000000000;

	
	reg[21:0] bit_re [0:13];
	assign bit_re[0] = 22'b0000000000000000000000;
	assign bit_re[1] = 22'b0000000000000000000000;
	assign bit_re[2] = 22'b0000000000000000000000;
	assign bit_re[3] = 22'b0000000000000000000000;
	assign bit_re[4] = 22'b0000000000000000000000;
	assign bit_re[5] = 22'b0000111110000001110110;
	assign bit_re[6] = 22'b0011111111100011111110;
	assign bit_re[7] = 22'b0011000001100011001110;
	assign bit_re[8] = 22'b0011111111100000000110;
	assign bit_re[9] = 22'b0011111111100000000110;
	assign bit_re[10] = 22'b0000000001100000000110;
	assign bit_re[11] = 22'b0000000001100000000110;
	assign bit_re[12] = 22'b0011111111000000000110;
	assign bit_re[13] = 22'b0000000000000000000000;
	

	reg[17:0] bit_mi [0:13];
	assign bit_mi[0] = 18'b000000000000000000;
	assign bit_mi[1] = 18'b000000000000000000;
	assign bit_mi[2] = 18'b001100000000000000;
	assign bit_mi[3] = 18'b001100000000000000;
	assign bit_mi[4] = 18'b000000000000000000;
	assign bit_mi[5] = 18'b001100001110011100;
	assign bit_mi[6] = 18'b001100011111111110;
	assign bit_mi[7] = 18'b001100011001100110;
	assign bit_mi[8] = 18'b001100011001100110;
	assign bit_mi[9] = 18'b001100011001100110;
	assign bit_mi[10] = 18'b001100011001100110;
	assign bit_mi[11] = 18'b001100011001100110;
	assign bit_mi[12] = 18'b001100011001100110;
	assign bit_mi[13] = 18'b000000000000000000;

	
	reg[21:0] bit_fa [0:13];
	assign bit_fa[0] = 22'b0000000000000000000000;
	assign bit_fa[1] = 22'b0000000000000011100000;
	assign bit_fa[2] = 22'b0000000000000011111000;
	assign bit_fa[3] = 22'b0000000000000000011000;
	assign bit_fa[4] = 22'b0000000000000000011000;
	assign bit_fa[5] = 22'b0000111110000011111110;
	assign bit_fa[6] = 22'b0011000000000011111110;
	assign bit_fa[7] = 22'b0011000000000000011000;
	assign bit_fa[8] = 22'b0011111110000000011000;
	assign bit_fa[9] = 22'b0011000001100000011000;
	assign bit_fa[10] = 22'b0011000001100000011000;
	assign bit_fa[11] = 22'b0011111110000000011000;
	assign bit_fa[12] = 22'b0011111110000000011000;
	assign bit_fa[13] = 22'b0000000000000000000000;

	
	reg[20:0] bit_so [0:13];
	assign bit_so[0] = 21'b000000000000000000000;
	assign bit_so[1] = 21'b000000000000000000000;
	assign bit_so[2] = 21'b000000000000000000000;
	assign bit_so[3] = 21'b000000000000000000000;
	assign bit_so[4] = 21'b000000000000000000000;
	assign bit_so[5] = 21'b000011110000011111000;
	assign bit_so[6] = 21'b000011110000011111110;
	assign bit_so[7] = 21'b001100001100000000110;
	assign bit_so[8] = 21'b001100001100001111100;
	assign bit_so[9] = 21'b001100001100011111100;
	assign bit_so[10] = 21'b001100001100011000000;
	assign bit_so[11] = 21'b000011110000011111110;
	assign bit_so[12] = 21'b000011110000001111110;
	assign bit_so[13] = 21'b000000000000000000000;

	
	reg[18:0] bit_la [0:13];
	assign bit_la[0] = 19'b0000000000000000000;
	assign bit_la[1] = 19'b0000000000000000110;
	assign bit_la[2] = 19'b0000000000000000110;
	assign bit_la[3] = 19'b0000000000000000110;
	assign bit_la[4] = 19'b0000000000000000110;
	assign bit_la[5] = 19'b0000111110000000110;
	assign bit_la[6] = 19'b0011000000000000110;
	assign bit_la[7] = 19'b0011000000000000110;
	assign bit_la[8] = 19'b0011111110000000110;
	assign bit_la[9] = 19'b0011000001100000110;
	assign bit_la[10] = 19'b0011000001100000110;
	assign bit_la[11] = 19'b0011111110000011100;
	assign bit_la[12] = 19'b0011111110000011000;
	assign bit_la[13] = 19'b0000000000000000000;

	
	reg[15:0] bit_xi [0:13];
	assign bit_xi[0] = 16'b0000000000000000;
	assign bit_xi[1] = 16'b0000000000000000;
	assign bit_xi[2] = 16'b0011000000000000;
	assign bit_xi[3] = 16'b0011000000000000;
	assign bit_xi[4] = 16'b0000000000000000;
	assign bit_xi[5] = 16'b0011000110000110;
	assign bit_xi[6] = 16'b0011000110000110;
	assign bit_xi[7] = 16'b0011000011001100;
	assign bit_xi[8] = 16'b0011000011111100;
	assign bit_xi[9] = 16'b0011000000110000;
	assign bit_xi[10] = 16'b0011000011001100;
	assign bit_xi[11] = 16'b0011000111001110;
	assign bit_xi[12] = 16'b0011000110000110;
	assign bit_xi[13] = 16'b0000000000000000;

	
   //Background & Animation
   assign h = hcount[10:1];
   assign v = vcount[9:0];	
	
	logic animation_1, animation_2, animation_3, animation_4, animation_5, animation_6, animation_7, animation_8, animation_9, animation_10, animation_11, animation_12, animation_13, animation_14, animation_15, animation_16;
   
	assign animation_1 = (vcount <= 10'd295) & (hcount > 11'd80 & hcount <= 11'd150);
   assign animation_2 = (vcount <= 10'd295) & (hcount > 11'd150 & hcount <= 11'd220);
   assign animation_3 = (vcount <= 10'd295) & (hcount > 11'd220 & hcount <= 11'd290);
   assign animation_4 = (vcount <= 10'd295) & (hcount > 11'd290 & hcount <= 11'd360);
   assign animation_5 = (vcount <= 10'd295) & (hcount > 11'd360 & hcount <= 11'd430);
   assign animation_6 = (vcount <= 10'd295) & (hcount > 11'd430 & hcount <= 11'd500);
   assign animation_7 = (vcount <= 10'd295) & (hcount > 11'd500 & hcount <= 11'd570);
   assign animation_8 = (vcount <= 10'd295) & (hcount > 11'd570 & hcount <= 11'd640);
   assign animation_9 = (vcount <= 10'd295) & (hcount > 11'd640 & hcount <= 11'd710);
   assign animation_10 = (vcount <= 10'd295) & (hcount > 11'd710 & hcount <= 11'd780);
   assign animation_11 = (vcount <= 10'd295) & (hcount > 11'd780 & hcount <= 11'd850);
   assign animation_12 = (vcount <= 10'd295) & (hcount > 11'd850 & hcount <= 11'd920);
   assign animation_13 = (vcount <= 10'd295) & (hcount > 11'd920 & hcount <= 11'd990);
   assign animation_14 = (vcount <= 10'd295) & (hcount > 11'd990 & hcount <= 11'd1060);
   assign animation_15 = (vcount <= 10'd295) & (hcount > 11'd1060 & hcount <= 11'd1130);
   assign animation_16 = (vcount <= 10'd295) & (hcount > 11'd1130 & hcount <= 11'd1200);
	
	logic keychar1, keychar2, keychar3, keychar4, keychar5, keychar6, keychar7, keychar8, keychar9, keychar10, keychar11, keychar12, keychar13, keychar14, keychar15, keychar16;
 
   assign keychar1 = (vcount >= 10'd464) & (hcount[10:1] >= 10'd55 & hcount[10:1] < 11'd63);
   assign keychar2 = (vcount >= 10'd464) & (hcount[10:1] >= 10'd90 & hcount[10:1] < 11'd98);
   assign keychar3 = (vcount >= 10'd464) & (hcount[10:1] >= 10'd125 & hcount[10:1] < 11'd133);
   assign keychar4 = (vcount >= 10'd464) & (hcount[10:1] >= 10'd160 & hcount[10:1] < 11'd168);
   assign keychar5 = (vcount >= 10'd464) & (hcount[10:1] >= 10'd195 & hcount[10:1] < 11'd203);
   assign keychar6 = (vcount >= 10'd464) & (hcount[10:1] >= 10'd230 & hcount[10:1] < 11'd238);
   assign keychar7 = (vcount >= 10'd464) & (hcount[10:1] >= 10'd265 & hcount[10:1] < 11'd273);
   assign keychar8 = (vcount >= 10'd464) & (hcount[10:1] >= 10'd300 & hcount[10:1] < 11'd308);
   assign keychar9 = (vcount >= 10'd464) & (hcount[10:1] >= 10'd335 & hcount[10:1] < 11'd343);
   assign keychar10 = (vcount >= 10'd464) & (hcount[10:1] >= 10'd370 & hcount[10:1] < 11'd378);
   assign keychar11 = (vcount >= 10'd464) & (hcount[10:1] >= 10'd405 & hcount[10:1] < 11'd413);
   assign keychar12 = (vcount >= 10'd464) & (hcount[10:1] >= 10'd440 & hcount[10:1] < 11'd448);
   assign keychar13 = (vcount >= 10'd464) & (hcount[10:1] >= 10'd475 & hcount[10:1] < 11'd483);
   assign keychar14 = (vcount >= 10'd464) & (hcount[10:1] >= 10'd510 & hcount[10:1] < 11'd518);
   assign keychar15 = (vcount >= 10'd464) & (hcount[10:1] >= 10'd545 & hcount[10:1] < 11'd553);
   assign keychar16 = (vcount >= 10'd464) & (hcount[10:1] >= 10'd580 & hcount[10:1] < 11'd588);
	
	logic [2:0] key_digit, key_char_offset;
	
	always_comb begin
		if (key1) begin key_digit = 3'd1; key_char_offset = 3'd7;end
		else if (key2) begin key_digit = 3'd2; key_char_offset = 3'd2;end
		else if (key3) begin key_digit = 3'd3; key_char_offset = 3'd5;end
		else if (key4) begin key_digit = 3'd4; key_char_offset = 3'd0;end
		else if (key5) begin key_digit = 3'd5; key_char_offset = 3'd3;end
		else if (key6) begin key_digit = 3'd6; key_char_offset = 3'd6;end
		else if (key7) begin key_digit = 3'd7; key_char_offset = 3'd1;end
		else if (key8) begin key_digit = 3'd1; key_char_offset = 3'd4;end
		else if (key9) begin key_digit = 3'd2; key_char_offset = 3'd7;end
		else if (key10) begin key_digit = 3'd3; key_char_offset = 3'd2;end
		else if (key11) begin key_digit = 3'd4; key_char_offset = 3'd5;end
		else if (key12) begin key_digit = 3'd5; key_char_offset = 3'd0;end
		else if (key13) begin key_digit = 3'd6; key_char_offset = 3'd3;end
		else if (key14) begin key_digit = 3'd7; key_char_offset = 3'd6;end
		else if (key15) begin key_digit = 3'd1; key_char_offset = 3'd1;end
		else if (key16) begin key_digit = 3'd2; key_char_offset = 3'd4;end
		else begin key_digit = 3'd0; key_char_offset = 3'd0;end
	end
	
	logic [7:0] keysegment;
   logic [7:0] keycurSegs;
	KEYCHAR_DISP key_disp(.digit(key_digit), .offset(key_char_offset), .*);
	
	//Test char
   HEX_CONVERT hex(.*);
	
   logic inChar;
	assign inChar = (vcount[9:4] == 6'd1) & (hcount[10:1] < 10'd64); 	
	
	logic [7:0] segment;
   logic [7:0] curSegs;
	logic [8:0] digit_1, digit_2;
	
	assign digit_1 = {3'b0, row_num};
	//assign digit_1 = {8'b0, starfall1};
	assign digit_2 = {1'b0, audio_demo};
	//assign digit_2 = {3'b0, insert1};
		
	//Test pixel line
	logic real_line_area, seg_line_area, denoise_line_area, fix_keys_line_area; 
	
	assign real_line_area = (vcount[9:3] == 7'b0001101) & (hcount[10:1] > 10'd145) &(hcount[10:1] < 10'd496);
	assign seg_line_area = (vcount[9:3] == 7'b0001111) & (hcount[10:1] > 10'd145) & (hcount[10:1] < 10'd496);
	assign denoise_line_area = (vcount[9:3] == 7'b0010001) & (hcount[10:1] > 10'd145) & (hcount[10:1] < 10'd496);
	assign fix_keys_line_area = (vcount[9:3] == 7'b0010011) & (hcount[10:1] > 10'd145) & (hcount[10:1] < 10'd496);
	
	logic real_line_area1, seg_line_area1;
	
	assign real_line_area1 = (vcount[9:1] == 9'b000000001) & (hcount[10:1] > 10'd145) &(hcount[10:1] < 10'd496);
	assign seg_line_area1 = (vcount[9:1] == 9'b000000011) & (hcount[10:1] > 10'd145) & (hcount[10:1] < 10'd496);
	
	//Calibration												  
	logic sprite1on, sprite2on, sprite3on, sprite4on, feedback, feedbackframe, keyboardarea;
	assign sprite1on = (vcount >= 10'd0 & vcount < 10'd340);
	assign sprite2on = (vcount >= 10'd100 & vcount < 10'd182 & hcount[10:1] >= 10'd20 & hcount[10:1] < 10'd620);
	assign sprite3on = (vcount >= 10'd240 & vcount < 10'd340);
	assign sprite4on = (vcount >= 10'd240 & vcount < 10'd360 & hcount[10:1] >= 10'd275 & hcount[10:1] < 10'd365);
	
	
	assign feedback = (vcount >= 10'd199 & vcount <= 10'd295) & (hcount[10:1] >= 10'd144) & (hcount[10:1] <= 10'd495);
	assign feedbackframe = (vcount >= 10'd194 & vcount <= 10'd300) & (hcount[10:1] >= 10'd139) & (hcount[10:1] <= 10'd500);
	assign keyboardarea = (vcount >= 10'd340);
	
	logic [9:0] feedback_x, feedback_y;
	logic [2:0] feedback_xx;
	assign feedback_x = (hcount[10:1] - 10'd144) >> 4;
	assign feedback_xx = hcount[4:2] - 3'b010;
	assign feedback_y = (vcount[9:0] - 10'd199) >> 1;
	
	logic [10:0] feedback_index;
	assign feedback_index = feedback_y * 22 + feedback_x;
	
	logic whitearea, blackarea, lineup, linedown;
	assign whitearea = (vcount >= 10'd360 & vcount < 10'd385);
	assign blackarea = (vcount >= 10'd415);
   assign lineup = (vcount >= 10'd340 & vcount <= 10'd360);
	assign linedown = (vcount >= 10'd450);
	
	logic fb1, fb2, fb3, fb4, fb5, fb6, fb7, fb8, fb9, fb10, fb11, fb12, fb13, fb14, fb15, fb16;
		
	assign fb1 = (vcount >= 10'd320 & vcount < 10'd340) & (hcount > 11'd80 & hcount <= 11'd150);
   assign fb2 = (vcount >= 10'd320 & vcount < 10'd340) & (hcount > 11'd150 & hcount <= 11'd220);
   assign fb3 = (vcount >= 10'd320 & vcount < 10'd340) & (hcount > 11'd220 & hcount <= 11'd290);
   assign fb4 = (vcount >= 10'd320 & vcount < 10'd340) & (hcount > 11'd290 & hcount <= 11'd360);
   assign fb5 = (vcount >= 10'd320 & vcount < 10'd340) & (hcount > 11'd360 & hcount <= 11'd430);
   assign fb6 = (vcount >= 10'd320 & vcount < 10'd340) & (hcount > 11'd430 & hcount <= 11'd500);
   assign fb7 = (vcount >= 10'd320 & vcount < 10'd340) & (hcount > 11'd500 & hcount <= 11'd570);
   assign fb8 = (vcount >= 10'd320 & vcount < 10'd340) & (hcount > 11'd570 & hcount <= 11'd640);
   assign fb9 = (vcount >= 10'd320 & vcount < 10'd340) & (hcount > 11'd640 & hcount <= 11'd710);
   assign fb10 = (vcount >= 10'd320 & vcount < 10'd340) & (hcount > 11'd710 & hcount <= 11'd780);
   assign fb11 = (vcount >= 10'd320 & vcount < 10'd340) & (hcount > 11'd780 & hcount <= 11'd850);
   assign fb12 = (vcount >= 10'd320 & vcount < 10'd340) & (hcount > 11'd850 & hcount <= 11'd920);
   assign fb13 = (vcount >= 10'd320 & vcount < 10'd340) & (hcount > 11'd920 & hcount <= 11'd990);
   assign fb14 = (vcount >= 10'd320 & vcount < 10'd340) & (hcount > 11'd990 & hcount <= 11'd1060);
   assign fb15 = (vcount >= 10'd320 & vcount < 10'd340) & (hcount > 11'd1060 & hcount <= 11'd1130);
   assign fb16 = (vcount >= 10'd320 & vcount < 10'd340) & (hcount > 11'd1130 & hcount <= 11'd1200);
	
	logic screen_button1, screen_button2, screen_button3, screen_button4, screen_button5;

	assign screen_button1 = (vcount >= 10'd350 & vcount <= 10'd415) & (hcount[10:1] >= 10'd71 & hcount[10:1] <= 10'd169);
	assign screen_button2 = (vcount >= 10'd350 & vcount <= 10'd415) & (hcount[10:1] >= 10'd171 & hcount[10:1] <= 10'd269);
	assign screen_button3 = (vcount >= 10'd350 & vcount <= 10'd415) & (hcount[10:1] >= 10'd271 & hcount[10:1] <= 10'd369);
	assign screen_button4 = (vcount >= 10'd350 & vcount <= 10'd415) & (hcount[10:1] >= 10'd371 & hcount[10:1] <= 10'd469);
	assign screen_button5 = (vcount >= 10'd350 & vcount <= 10'd415) & (hcount[10:1] >= 10'd471 & hcount[10:1] <= 10'd569);
	
	//Display
	logic line1, line2;
	logic key1, key2, key3, key4, key5, key6, key7, key8, key9, key10, key11, key12, key13, key14, key15, key16;

   assign line1 = (vcount >= 10'd340) & (hcount >= 11'd0 & hcount <= 11'd80);
	assign line2 = (vcount >= 10'd340) & (hcount > 11'd1200 & hcount <= 11'd1280);
   assign key1 = (vcount >= 10'd340) & (hcount > 11'd80 & hcount <= 11'd150);
   assign key2 = (vcount >= 10'd340) & (hcount > 11'd150 & hcount <= 11'd220);
   assign key3 = (vcount >= 10'd340) & (hcount > 11'd220 & hcount <= 11'd290);
   assign key4 = (vcount >= 10'd340) & (hcount > 11'd290 & hcount <= 11'd360);
   assign key5 = (vcount >= 10'd340) & (hcount > 11'd360 & hcount <= 11'd430);
   assign key6 = (vcount >= 10'd340) & (hcount > 11'd430 & hcount <= 11'd500);
   assign key7 = (vcount >= 10'd340) & (hcount > 11'd500 & hcount <= 11'd570);
   assign key8 = (vcount >= 10'd340) & (hcount > 11'd570 & hcount <= 11'd640);
   assign key9 = (vcount >= 10'd340) & (hcount > 11'd640 & hcount <= 11'd710);
   assign key10 = (vcount >= 10'd340) & (hcount > 11'd710 & hcount <= 11'd780);
   assign key11 = (vcount >= 10'd340) & (hcount > 11'd780 & hcount <= 11'd850);
   assign key12 = (vcount >= 10'd340) & (hcount > 11'd850 & hcount <= 11'd920);
   assign key13 = (vcount >= 10'd340) & (hcount > 11'd920 & hcount <= 11'd990);
   assign key14 = (vcount >= 10'd340) & (hcount > 11'd990 & hcount <= 11'd1060);
   assign key15 = (vcount >= 10'd340) & (hcount > 11'd1060 & hcount <= 11'd1130);
   assign key16 = (vcount >= 10'd340) & (hcount > 11'd1130 & hcount <= 11'd1200);
	
	logic autoplay, freemode; //gamemode//;
	assign autoplay = (vcount > 10'd210 & vcount <= 10'd224) & (hcount[10:1] > 11'd165 & hcount[10:1] <= 11'd500);
	assign freemode = (vcount > 10'd210 & vcount <= 10'd224) & (hcount[10:1] > 11'd217 & hcount[10:1] <= 11'd423);
	//assign gamemode = (vcount > 10'd210 & vcount <= 10'd224) & (hcount[10:1] > 11'd222 & hcount[10:1] <= 11'd418);
	
	
	logic explaintext;
	
	//assign explainframe = ((vcount > 10'd204 & vcount <= 10'd207) | (vcount > 10'd227 & vcount <= 10'd230));
	//assign explaintext = (vcount > 10'd210 & vcount <= 10'd224) & (hcount[10:1] > 11'd75 & hcount[10:1] <= 11'd563);
	
	logic w_next1, w_color1, w_calibration1, w_confirm, w_no_color, w_calibration2, w_recognize, w_keys;
	logic w_next2, w_free, w_mode1, w_auto_play, w_mode2, w_exit, w_back_to1;
	logic w_screen, w_calibration3, w_back_to2, w_color3, w_calibration4;
	
	
	
	assign w_next1 = (vcount > 10'd367 & vcount <= 10'd395) & (hcount[10:1] > 11'd275 & hcount[10:1] <= 11'd363);
	
	assign w_color1 = (vcount > 10'd366 & vcount <= 10'd380) & (hcount[10:1] > 11'd96 & hcount[10:1] <= 11'd147);
	assign w_calibration1 = (vcount > 10'd385 & vcount <= 10'd399) & (hcount[10:1] > 11'd72 & hcount[10:1] <= 11'd169);
	assign w_no_color = (vcount > 10'd366 & vcount <= 10'd380) & (hcount[10:1] > 11'd180 & hcount[10:1] <= 11'd269);
	assign w_calibration2 = (vcount > 10'd385 & vcount <= 10'd399) & (hcount[10:1] > 11'd172 & hcount[10:1] <= 11'd269);
	assign w_confirm = (vcount > 10'd375 & vcount <= 10'd389) & (hcount[10:1] > 11'd285 & hcount[10:1] <= 11'd358);
	assign w_recognize = (vcount > 10'd366 & vcount <= 10'd380) & (hcount[10:1] > 11'd372 & hcount[10:1] <= 11'd467);
	assign w_keys = (vcount > 10'd385 & vcount <= 10'd399) & (hcount[10:1] > 11'd399 & hcount[10:1] <= 11'd442);
	assign w_next2 = (vcount > 10'd375 & vcount <= 10'd389) & (hcount[10:1] > 11'd499 & hcount[10:1] <= 11'd543);
	
	assign w_free = (vcount > 10'd366 & vcount <= 10'd380) & (hcount[10:1] > 11'd99 & hcount[10:1] <= 11'd143);
	assign w_mode1 = (vcount > 10'd385 & vcount <= 10'd399) & (hcount[10:1] > 11'd98 & hcount[10:1] <= 11'd146);
	assign w_auto_play = (vcount > 10'd366 & vcount <= 10'd380) & (hcount[10:1] > 11'd179 & hcount[10:1] <= 11'd261);
	assign w_mode2 = (vcount > 10'd385 & vcount <= 10'd399) & (hcount[10:1] > 11'd198 & hcount[10:1] <= 11'd246);
	//assign w_game = (vcount > 10'd366 & vcount <= 10'd380) & (hcount[10:1] > 11'd297 & hcount[10:1] <= 11'd344);
	//assign w_mode3 = (vcount > 10'd385 & vcount <= 10'd399) & (hcount[10:1] > 11'd298 & hcount[10:1] <= 11'd346);
	assign w_exit = (vcount > 10'd375 & vcount <= 10'd389) & (hcount[10:1] > 11'd300 & hcount[10:1] <= 11'd337);
	assign w_back_to1 = (vcount > 10'd360 & vcount <= 10'd374) & (hcount[10:1] > 11'd386 & hcount[10:1] <= 11'd456);
	assign w_screen = (vcount > 10'd375 & vcount <= 10'd389) & (hcount[10:1] > 11'd387 & hcount[10:1] <= 11'd455);
	assign w_calibration3 = (vcount > 10'd390 & vcount <= 10'd404) & (hcount[10:1] > 11'd372 & hcount[10:1] <= 11'd469);
   assign w_back_to2 = (vcount > 10'd360 & vcount <= 10'd374) & (hcount[10:1] > 11'd486 & hcount[10:1] <= 11'd556);
	assign w_color2 = (vcount > 10'd375 & vcount <= 10'd389) & (hcount[10:1] > 11'd496 & hcount[10:1] <= 11'd547);
	assign w_calibration4 = (vcount > 10'd390 & vcount <= 10'd404) & (hcount[10:1] > 11'd472 & hcount[10:1] <= 11'd569);

	logic b_screen_calibration, b_color_calibration, b_adjust1, b_adjust2;
	
	assign b_screen_calibration = (vcount >= 10'd50 & vcount < 10'd78) & (hcount[10:1] > 11'd145 & hcount[10:1] <= 11'd495);
	assign b_color_calibration = (vcount >= 10'd200 & vcount < 10'd228) & (hcount[10:1] > 11'd159 & hcount[10:1] <= 11'd481);
	assign b_adjust1 = (vcount >= 10'd140 & vcount < 10'd154) & (hcount[10:1] > 11'd153 & hcount[10:1] <= 11'd488);
	assign b_adjust2 = (vcount >= 10'd160 & vcount < 10'd174) & (hcount[10:1] > 11'd136 & hcount[10:1] <= 11'd504);
	
	logic backbutton;
	assign backbutton = (vcount >= 10'd330 & vcount < 10'd340) & (hcount[10:1] >= 10'd40 & hcount[10:1] <= 10'd600);

	logic note_display;
	
	assign note_display = (vcount >= 10'd180 & vcount < 10'd240) & (hcount[10:1] > 10'd290 & hcount[10:1] <= 10'd350);
	
	logic dao,re,mi,fa,so,la,xi;
	
	assign dao = (vcount >= 10'd196 & vcount < 10'd224) & (hcount[10:1] >= 10'd296 & hcount[10:1] < 10'd342);
	assign re = (vcount >= 10'd196 & vcount < 10'd224) & (hcount[10:1] >= 10'd298 & hcount[10:1] < 10'd342);
	assign mi = (vcount >= 10'd196 & vcount < 10'd224) & (hcount[10:1] >= 10'd302 & hcount[10:1] < 10'd338);
	assign fa = (vcount >= 10'd196 & vcount < 10'd224) & (hcount[10:1] >= 10'd298 & hcount[10:1] < 10'd342);
	assign so = (vcount >= 10'd196 & vcount < 10'd224) & (hcount[10:1] >= 10'd300 & hcount[10:1] < 10'd342);
	assign la = (vcount >= 10'd196 & vcount < 10'd224) & (hcount[10:1] >= 10'd300 & hcount[10:1] < 10'd338);
	assign xi = (vcount >= 10'd196 & vcount < 10'd224) & (hcount[10:1] >= 10'd304 & hcount[10:1] < 10'd336);
	
	/*
	logic starfall1, starfall2, starfall3, starfall4, starfall5, starfall6, starfall7, starfall8, starfall9, starfall10, starfall11, starfall12, starfall13, starfall14, starfall15, starfall16;
	
	assign starfall1 = (drop_position == 1) ? 1'b1:1'b0;
	assign starfall2 = (drop_position == 2) ? 1'b1:1'b0;
	assign starfall3 = (drop_position == 3) ? 1'b1:1'b0;
	assign starfall4 = (drop_position == 4) ? 1'b1:1'b0;
	assign starfall5 = (drop_position == 5) ? 1'b1:1'b0;
	assign starfall6 = (drop_position == 6) ? 1'b1:1'b0;
	assign starfall7 = (drop_position == 7) ? 1'b1:1'b0;
	assign starfall8 = (drop_position == 8) ? 1'b1:1'b0;
	assign starfall9 = (drop_position == 9) ? 1'b1:1'b0;
	assign starfall10 = (drop_position == 10) ? 1'b1:1'b0;
	assign starfall11 = (drop_position == 11) ? 1'b1:1'b0;
	assign starfall12 = (drop_position == 12) ? 1'b1:1'b0;
	assign starfall13 = (drop_position == 13) ? 1'b1:1'b0;
	assign starfall14 = (drop_position == 14) ? 1'b1:1'b0;
	assign starfall15 = (drop_position == 15) ? 1'b1:1'b0;
	assign starfall16 = (drop_position == 16) ? 1'b1:1'b0;
	
	logic [5:0] star_row1 [0:330];
	logic [5:0] star_row2 [0:330];
	logic [5:0] star_row3 [0:330];
	logic [5:0] star_row4 [0:330];
	logic [5:0] star_row5 [0:330];
	logic [5:0] star_row6 [0:330];
	logic [5:0] star_row7 [0:330];
	logic [5:0] star_row8 [0:330];
	logic [5:0] star_row9 [0:330];
	logic [5:0] star_row10 [0:330];
	logic [5:0] star_row11 [0:330];
	logic [5:0] star_row12 [0:330];
	logic [5:0] star_row13 [0:330];
	logic [5:0] star_row14 [0:330];
	logic [5:0] star_row15 [0:330];
	logic [5:0] star_row16 [0:330];
	
	logic [5:0] insert1, insert2, insert3, insert4, insert5, insert6, insert7, insert8, insert9, insert10, insert11, insert12, insert13, insert14, insert15, insert16;
	
	STAR_BUFFER star1 (.clk50(clk), .fall(starfall1), .speed(speed), .star_row(star_row1), .insert_temp(insert1));
	STAR_BUFFER star2 (.clk50(clk), .fall(starfall2), .speed(speed), .star_row(star_row2), .insert_temp(insert2));
	STAR_BUFFER star3 (.clk50(clk), .fall(starfall3), .speed(speed), .star_row(star_row3), .insert_temp(insert3));
	STAR_BUFFER star4 (.clk50(clk), .fall(starfall4), .speed(speed), .star_row(star_row4), .insert_temp(insert4));
	STAR_BUFFER star5 (.clk50(clk), .fall(starfall5), .speed(speed), .star_row(star_row5), .insert_temp(insert5));
	STAR_BUFFER star6 (.clk50(clk), .fall(starfall6), .speed(speed), .star_row(star_row6), .insert_temp(insert6));
	STAR_BUFFER star7 (.clk50(clk), .fall(starfall7), .speed(speed), .star_row(star_row7), .insert_temp(insert7));
	STAR_BUFFER star8 (.clk50(clk), .fall(starfall8), .speed(speed), .star_row(star_row8), .insert_temp(insert8));
	STAR_BUFFER star9 (.clk50(clk), .fall(starfall9), .speed(speed), .star_row(star_row9), .insert_temp(insert9));
	STAR_BUFFER star10 (.clk50(clk), .fall(starfall10), .speed(speed), .star_row(star_row10), .insert_temp(insert10));
	STAR_BUFFER star11 (.clk50(clk), .fall(starfall11), .speed(speed), .star_row(star_row11), .insert_temp(insert11));
	STAR_BUFFER star12 (.clk50(clk), .fall(starfall12), .speed(speed), .star_row(star_row12), .insert_temp(insert12));
	STAR_BUFFER star13 (.clk50(clk), .fall(starfall13), .speed(speed), .star_row(star_row13), .insert_temp(insert13));
	STAR_BUFFER star14 (.clk50(clk), .fall(starfall14), .speed(speed), .star_row(star_row14), .insert_temp(insert14));
	STAR_BUFFER star15 (.clk50(clk), .fall(starfall15), .speed(speed), .star_row(star_row15), .insert_temp(insert15));
	STAR_BUFFER star16 (.clk50(clk), .fall(starfall16), .speed(speed), .star_row(star_row16), .insert_temp(insert16));
	
	logic star_area1, star_area2, star_area3, star_area4, star_area5, star_area6, star_area7, star_area8, star_area9, star_area10, star_area11, star_area12, star_area13, star_area14, star_area15, star_area16; 
	
	assign star_area1 = (vcount <= 10'd330) & (hcount >= 11'd80 & hcount < 11'd150);
   assign star_area2 = (vcount <= 10'd330) & (hcount >= 11'd150 & hcount < 11'd220);
   assign star_area3 = (vcount <= 10'd330) & (hcount >= 11'd220 & hcount < 11'd290);
   assign star_area4 = (vcount <= 10'd330) & (hcount >= 11'd290 & hcount < 11'd360);
   assign star_area5 = (vcount <= 10'd330) & (hcount >= 11'd360 & hcount < 11'd430);
   assign star_area6 = (vcount <= 10'd330) & (hcount >= 11'd430 & hcount < 11'd500);
   assign star_area7 = (vcount <= 10'd330) & (hcount >= 11'd500 & hcount < 11'd570);
   assign star_area8 = (vcount <= 10'd330) & (hcount >= 11'd570 & hcount < 11'd640);
   assign star_area9 = (vcount <= 10'd330) & (hcount >= 11'd640 & hcount < 11'd710);
   assign star_area10 = (vcount <= 10'd330) & (hcount >= 11'd710 & hcount < 11'd780);
   assign star_area11 = (vcount <= 10'd330) & (hcount >= 11'd780 & hcount < 11'd850);
   assign star_area12 = (vcount <= 10'd330) & (hcount >= 11'd850 & hcount < 11'd920);
   assign star_area13 = (vcount <= 10'd330) & (hcount >= 11'd920 & hcount < 11'd990);
   assign star_area14 = (vcount <= 10'd330) & (hcount >= 11'd990 & hcount < 11'd1060);
   assign star_area15 = (vcount <= 10'd330) & (hcount >= 11'd1060 & hcount < 11'd1130);
   assign star_area16 = (vcount <= 10'd330) & (hcount >= 11'd1130 & hcount < 11'd1200);
	
	always_comb begin
		if (star_area1) begin
			star_offset = 40;
			star_row_current = star_row1[vcount];
		end else if (star_area2) begin
			star_offset = 75;
			star_row_current = star_row2[vcount];
		end else if (star_area3) begin
			star_offset = 110;
			star_row_current = star_row3[vcount];
		end else if (star_area4) begin
			star_offset = 145;
			star_row_current = star_row4[vcount];
		end else if (star_area5) begin
			star_offset = 180;
			star_row_current = star_row5[vcount];
		end else if (star_area6) begin
			star_offset = 215;
			star_row_current = star_row6[vcount];
		end else if (star_area7) begin
			star_offset = 250;
			star_row_current = star_row7[vcount];
		end else if (star_area8) begin
			star_offset = 285;
			star_row_current = star_row8[vcount];
		end else if (star_area9) begin
			star_offset = 320;
			star_row_current = star_row9[vcount];
		end else if (star_area10) begin
			star_offset = 355;
			star_row_current = star_row10[vcount];
		end else if (star_area11) begin
			star_offset = 390;
			star_row_current = star_row11[vcount];
		end else if (star_area12) begin
			star_offset = 425;
			star_row_current = star_row12[vcount];
		end else if (star_area13) begin
			star_offset = 460;
			star_row_current = star_row13[vcount];
		end else if (star_area14) begin
			star_offset = 495;
			star_row_current = star_row14[vcount];
		end else if (star_area15) begin
			star_offset = 530;
			star_row_current = star_row15[vcount];
		end else begin
			star_offset = 565;
			star_row_current = star_row16[vcount];
		end
	end
	*/
	
	always_comb begin
		{VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'h00}; // red

		//Screen Calibration mode
		if ((mode_select == 0) | (mode_select == 1) | (mode_select == 2)) begin
			if (feedback) begin
				if ((vcount - 199) == (row_num - 114) | (vcount - 199) == (row_num - 98) | (vcount - 199) == (row_num - 130)) {VGA_R, VGA_G, VGA_B} = {8'h00, 8'hff, 8'h00};//green
				else if (pixel[feedback_index][feedback_xx] == 1) {VGA_R, VGA_G, VGA_B} = {8'hff, 8'hff, 8'h00};//yellow
				else {VGA_R, VGA_G, VGA_B} = {8'hff, 8'h00, 8'h00};//red
			end else if (feedbackframe) {VGA_R, VGA_G, VGA_B} = {8'h00, 8'hff, 8'h00};//green
			
			else if (b_screen_calibration) begin
				if (bit_screen_calibration[vcount[9:1] - 5'b11001][hcount[10:2] - 7'b1000111] == 1) {VGA_R, VGA_G, VGA_B} = {8'hff, 8'hff, 8'h00};//yellow
				else {VGA_R, VGA_G, VGA_B} = {8'hff, 8'h00, 8'h00};//red
			end else if (b_adjust1) begin
				if (bit_adjust1[vcount - 140][hcount[10:1] - 154] == 1) {VGA_R, VGA_G, VGA_B} = {8'h00, 8'hff, 8'h00};//green
				else {VGA_R, VGA_G, VGA_B} = {8'hff, 8'h00, 8'h00};//red
			end else if (b_adjust2) begin
				if (bit_adjust2[vcount - 160][hcount[10:1] - 137] == 1) {VGA_R, VGA_G, VGA_B} = {8'h00, 8'hff, 8'h00};//green
				else {VGA_R, VGA_G, VGA_B} = {8'hff, 8'h00, 8'h00};//red
	      
			end else if (sprite1on) {VGA_R, VGA_G, VGA_B} = {8'hff, 8'h00, 8'h00};//red
			else if ((mode_select == 1) & screen_button3) begin
				
				if (w_next1) begin
					if (bit_next[vcount[9:1] - 8'b10111000][hcount[10:2] - 8'b10001010] == 1) {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'hff};//blue
					else {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'h8b};//dark blue
				
				end else {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'h8b};//dark blue
			end else if ((mode_select == 2) & screen_button3)  begin
				
				if (w_next1) begin
					if (bit_next[vcount[9:1] - 8'b10111000][hcount[10:2] - 8'b10001010] == 1) {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'hff};//blue
					else {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'h8b};//dark blue
				
				end else {VGA_R, VGA_G, VGA_B} = {8'h00, 8'hff, 8'h00};//green
			end else {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'hff};//blue	
		
		
		//Keynoard Calibration mode
		end else if (mode_select == 3) begin
			if (real_line_area) {VGA_R, VGA_G, VGA_B} = {real_VGA_R, real_VGA_G, real_VGA_B};
			else if (seg_line_area) begin
				if (pixel_RGB[hcount[10:1] - 10'd146] == 2'b01) {VGA_R, VGA_G, VGA_B} = {8'hff, 8'h00, 8'h00};
				else if (pixel_RGB[hcount[10:1] - 10'd146] == 2'b10) {VGA_R, VGA_G, VGA_B} = {8'h00, 8'hff, 8'h00};
				else if (pixel_RGB[hcount[10:1] - 10'd146] == 2'b11) {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'hff};
				else {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'h00};			
			end else if (denoise_line_area) begin
				if (fix_pixel_RGB_denise[hcount[10:1] - 10'd146] == 2'b01) {VGA_R, VGA_G, VGA_B} = {8'hff, 8'h00, 8'h00};
				else if (fix_pixel_RGB_denise[hcount[10:1] - 10'd146] == 2'b10) {VGA_R, VGA_G, VGA_B} = {8'h00, 8'hff, 8'h0};
				else if (fix_pixel_RGB_denise[hcount[10:1] - 10'd146] == 2'b11) {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'hff};
				else {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'h00};
			end else if (fix_keys_line_area) {VGA_R, VGA_G, VGA_B} = {fix_keys_position[hcount[10:1] - 10'd146], 3'b111, fix_keys_position[hcount[10:1] - 10'd146], 3'b111, fix_keys_position[hcount[10:1] - 10'd146], 3'b111};		
			else if (inChar) begin
				if ( |(curSegs & segment) ) {VGA_R, VGA_G, VGA_B} = {8'hff, 8'h00, 8'h00}; // Red
				else if ( |segment ) {VGA_R, VGA_G, VGA_B} = {8'h30, 8'h30, 8'h30}; // Dark Gray	
			end else if (lineup | whitearea) {VGA_R, VGA_G, VGA_B} = {8'hff, 8'hff, 8'hff};//white
			else if (blackarea) {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'h00};//black	
			else if (line1|line2) {VGA_R, VGA_G, VGA_B} = {8'hff, 8'h00, 8'h00};
			else if (key2 | key4 | key6 | key8 | key10 | key12 | key14 | key16) {VGA_R, VGA_G, VGA_B} = {8'h00, 8'hff, 8'h00};
			else if (key1 | key3 | key5 | key7 | key9 | key11 | key13 | key15) {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'hff};
			
			else if (b_color_calibration) begin
				if (bit_color_calibration[vcount[9:1] - 7'b1100100][hcount[10:2] - 7'b1010000] == 1) {VGA_R, VGA_G, VGA_B} = {8'hff, 8'hff, 8'h00};//yellow
				else {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'h00};//black
			
			end else {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'h00};//black
			
		//Keyboard Calibration select mode
		end else if (mode_select == 4 | mode_select == 5 | mode_select == 6 | mode_select == 7 | mode_select == 8 | mode_select == 9) begin
			if (real_line_area) {VGA_R, VGA_G, VGA_B} = {real_VGA_R, real_VGA_G, real_VGA_B};
			else if (seg_line_area) begin
				if (pixel_RGB[hcount[10:1] - 10'd146] == 2'b01) {VGA_R, VGA_G, VGA_B} = {8'hff, 8'h00, 8'h00};
				else if (pixel_RGB[hcount[10:1] - 10'd146] == 2'b10) {VGA_R, VGA_G, VGA_B} = {8'h00, 8'hff, 8'h00};
				else if (pixel_RGB[hcount[10:1] - 10'd146] == 2'b11) {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'hff};
				else {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'h00};			
			end else if (denoise_line_area) begin
				if (fix_pixel_RGB_denise[hcount[10:1] - 10'd146] == 2'b01) {VGA_R, VGA_G, VGA_B} = {8'hff, 8'h00, 8'h00};
				else if (fix_pixel_RGB_denise[hcount[10:1] - 10'd146] == 2'b10) {VGA_R, VGA_G, VGA_B} = {8'h00, 8'hff, 8'h0};
				else if (fix_pixel_RGB_denise[hcount[10:1] - 10'd146] == 2'b11) {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'hff};
				else {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'h00};
			end else if (fix_keys_line_area) {VGA_R, VGA_G, VGA_B} = {fix_keys_position[hcount[10:1] - 10'd146], 3'b111, fix_keys_position[hcount[10:1] - 10'd146], 3'b111, fix_keys_position[hcount[10:1] - 10'd146], 3'b111};		
			else if (inChar) begin
				if ( |(curSegs & segment) ) {VGA_R, VGA_G, VGA_B} = {8'hff, 8'h00, 8'h00}; // Red
				else if ( |segment ) {VGA_R, VGA_G, VGA_B} = {8'h30, 8'h30, 8'h30}; // Dark Gray	
			
			end else if (w_color1) begin
				if (bit_color[vcount - 367][hcount[10:1] - 97] == 1) {VGA_R, VGA_G, VGA_B} = {8'h00, 8'hff, 8'h00};//green
				else {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'h8b};//dark blue
			end else if (w_calibration1) begin
				if (bit_calibration[vcount - 386][hcount[10:1] - 73] == 1) {VGA_R, VGA_G, VGA_B} = {8'h00, 8'hff, 8'h00};//green
				else {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'h8b};//dark blue
			end else if (w_no_color) begin
				if (bit_no_color[vcount - 367][hcount[10:1] - 181] == 1) {VGA_R, VGA_G, VGA_B} = {8'h00, 8'hff, 8'h00};//green
				else {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'h8b};//dark blue
			end else if (w_calibration2) begin
				if (bit_calibration[vcount - 386][hcount[10:1] - 173] == 1) {VGA_R, VGA_G, VGA_B} = {8'h00, 8'hff, 8'h00};//green
				else {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'h8b};//dark blue
			end else if (w_confirm) begin
				if (bit_confirm[vcount - 376][hcount[10:1] - 286] == 1) {VGA_R, VGA_G, VGA_B} = {8'h00, 8'hff, 8'h00};//green
				else {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'h8b};//dark blue
			end else if (w_recognize) begin
				if (bit_recognize[vcount - 367][hcount[10:1] - 373] == 1) {VGA_R, VGA_G, VGA_B} = {8'h00, 8'hff, 8'h00};//green
				else {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'h8b};//dark blue
			end else if (w_keys) begin
				if (bit_keys[vcount - 386][hcount[10:1] - 400] == 1) {VGA_R, VGA_G, VGA_B} = {8'h00, 8'hff, 8'h00};//green
				else {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'h8b};//dark blue
			end else if (w_next2) begin
				if (bit_next[vcount - 376][hcount[10:1] - 500] == 1) {VGA_R, VGA_G, VGA_B} = {8'h00, 8'hff, 8'h00};//green
				else {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'h8b};//dark blue
					
			end else if (screen_button1) begin
				if (mode_select == 5) {VGA_R, VGA_G, VGA_B} = {8'h00, 8'hff, 8'h00};//green
				else {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'h8b};//dark blue
			end else if (screen_button2) begin
				if (mode_select == 6) {VGA_R, VGA_G, VGA_B} = {8'h00, 8'hff, 8'h00};//green
				else {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'h8b};//dark blue
			end else if (screen_button3) begin
				if (mode_select == 7) {VGA_R, VGA_G, VGA_B} = {8'h00, 8'hff, 8'h00};//green
				else {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'h8b};//dark blue
			end else if (screen_button4) begin
				if (mode_select == 8) {VGA_R, VGA_G, VGA_B} = {8'h00, 8'hff, 8'h00};//green
				else {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'h8b};//dark blue
			end else if (screen_button5) begin
				if (mode_select == 9) {VGA_R, VGA_G, VGA_B} = {8'h00, 8'hff, 8'h00};//green
				else {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'h8b};//dark blue
			end else if (lineup | whitearea) {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'hff};//blue	
			else if (blackarea) {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'hff};//blue		
			else if (line1|line2) {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'hff};//blue
			else if (key2 | key4 | key6 | key8 | key10 | key12 | key14 | key16) {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'hff};
			else if (key1 | key3 | key5 | key7 | key9 | key11 | key13 | key15) {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'hff};
						
			else if (b_color_calibration) begin
				if (bit_color_calibration[vcount[9:1] - 7'b1100100][hcount[10:2] - 7'b1010000] == 1) {VGA_R, VGA_G, VGA_B} = {8'hff, 8'hff, 8'h00};//yellow
				else {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'h00};//black
			
			end else {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'h00};//black
		
		//Welcome page
		end else if (mode_select == 10 | mode_select == 11 | mode_select == 12 | mode_select == 13 | mode_select == 14 | mode_select == 15) begin
			if (real_line_area1) {VGA_R, VGA_G, VGA_B} = {real_VGA_R, real_VGA_G, real_VGA_B};
			else if (seg_line_area1) begin
				if (pixel_RGB[hcount[10:1] - 10'd146] == 2'b01) {VGA_R, VGA_G, VGA_B} = {8'hff, 8'h00, 8'h00};
				else if (pixel_RGB[hcount[10:1] - 10'd146] == 2'b10) {VGA_R, VGA_G, VGA_B} = {8'h00, 8'hff, 8'h00};
				else if (pixel_RGB[hcount[10:1] - 10'd146] == 2'b11) {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'hff};
				else {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'h00};	
		   //end 
			
			end else if (w_free) begin
				if (bit_free[vcount - 367][hcount[10:1] - 100] == 1) {VGA_R, VGA_G, VGA_B} = {8'h00, 8'hff, 8'h00};//green
				else {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'h8b};//dark blue
			end else if (w_mode1) begin
				if (bit_mode[vcount - 386][hcount[10:1] - 99] == 1) {VGA_R, VGA_G, VGA_B} = {8'h00, 8'hff, 8'h00};//green
				else {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'h8b};//dark blue
			end else if (w_auto_play) begin
				if (bit_auto_play[vcount - 367][hcount[10:1] - 180] == 1) {VGA_R, VGA_G, VGA_B} = {8'h00, 8'hff, 8'h00};//green
				else {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'h8b};//dark blue
			end else if (w_mode2) begin
				if (bit_mode[vcount - 386][hcount[10:1] - 199] == 1) {VGA_R, VGA_G, VGA_B} = {8'h00, 8'hff, 8'h00};//green
				else {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'h8b};//dark blue
			end else if (w_exit) begin
				if (bit_exit[vcount - 376][hcount[10:1] - 301] == 1) {VGA_R, VGA_G, VGA_B} = {8'h00, 8'hff, 8'h00};//green
				else {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'h8b};//dark blue
			//end else if (w_mode3) begin
			//	if (bit_mode[vcount - 386][hcount[10:1] - 299] == 1) {VGA_R, VGA_G, VGA_B} = {8'h00, 8'hff, 8'h00};//green
			//	else {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'h8b};//dark blue
			end else if (w_back_to1) begin
				if (bit_back_to[vcount - 360][hcount[10:1] - 387] == 1) {VGA_R, VGA_G, VGA_B} = {8'h00, 8'hff, 8'h00};//green
				else {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'h8b};//dark blue
			end else if (w_screen) begin
				if (bit_screen[vcount - 376][hcount[10:1] - 388] == 1) {VGA_R, VGA_G, VGA_B} = {8'h00, 8'hff, 8'h00};//green
				else {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'h8b};//dark blue
			end else if (w_calibration3) begin
				if (bit_calibration[vcount - 391][hcount[10:1] - 373] == 1) {VGA_R, VGA_G, VGA_B} = {8'h00, 8'hff, 8'h00};//green
				else {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'h8b};//dark blue
			end else if (w_back_to2) begin
				if (bit_back_to[vcount - 360][hcount[10:1] - 487] == 1) {VGA_R, VGA_G, VGA_B} = {8'h00, 8'hff, 8'h00};//green
				else {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'h8b};//dark blue
			end else if (w_color2) begin
				if (bit_color[vcount - 376][hcount[10:1] - 497] == 1) {VGA_R, VGA_G, VGA_B} = {8'h00, 8'hff, 8'h00};//green
				else {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'h8b};//dark blue
			end else if (w_calibration4) begin
				if (bit_calibration[vcount - 391][hcount[10:1] - 473] == 1) {VGA_R, VGA_G, VGA_B} = {8'h00, 8'hff, 8'h00};//green
				else {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'h8b};//dark blue
			/*
		   end else if (explainframe) begin
				if ((mode_select == 11) | (mode_select == 12) | (mode_select == 13)) {VGA_R, VGA_G, VGA_B} = {8'hff, 8'h00, 8'hff};//pink
				else {VGA_R, VGA_G, VGA_B} = {8'hf1, 8'hf1, 8'hbb};//background	
			*/
			end else if (autoplay && mode_select == 12) begin
			
					if (bit_autoplay[vcount - 211][hcount[10:1] - 166] == 1) {VGA_R, VGA_G, VGA_B} = {8'hff, 8'h00, 8'hff};//pink
					else {VGA_R, VGA_G, VGA_B} = {8'hf1, 8'hf1, 8'hbb};//background
			
			end else if (freemode && mode_select == 11) begin
		
					if (bit_freemode[vcount - 211][hcount[10:1] - 218] == 1) {VGA_R, VGA_G, VGA_B} = {8'hff, 8'h00, 8'hff};//pink
					else {VGA_R, VGA_G, VGA_B} = {8'hf1, 8'hf1, 8'hbb};//background
		
		//	end else if (gamemode && mode_select == 13) begin
		
		//			if (bit_gamemode[vcount - 211][hcount[10:1] - 223] == 1) {VGA_R, VGA_G, VGA_B} = {8'hff, 8'h00, 8'hff};//pink
		//			else {VGA_R, VGA_G, VGA_B} = {8'hf1, 8'hf1, 8'hbb};//background
				
			end else if (screen_button1) begin
				if (mode_select == 11) {VGA_R, VGA_G, VGA_B} = {8'h00, 8'hff, 8'h00};//green
				else {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'h8b};//dark blue
			end else if (screen_button2) begin
				if (mode_select == 12) {VGA_R, VGA_G, VGA_B} = {8'h00, 8'hff, 8'h00};//green
				else {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'h8b};//dark blue
			end else if (screen_button3) begin
				if (mode_select == 13) {VGA_R, VGA_G, VGA_B} = {8'h00, 8'hff, 8'h00};//green
				else {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'h8b};//dark blue
			end else if (screen_button4) begin
				if (mode_select == 14) {VGA_R, VGA_G, VGA_B} = {8'h00, 8'hff, 8'h00};//green
				else {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'h8b};//dark blue
			end else if (screen_button5) begin
				if (mode_select == 15) {VGA_R, VGA_G, VGA_B} = {8'h00, 8'hff, 8'h00};//green
				else {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'h8b};//dark blue
			end else if (sprite2on) {VGA_R, VGA_G, VGA_B} = {in_R2,in_G2,in_B2};
			else if (sprite3on) {VGA_R, VGA_G, VGA_B} = {in_R3,in_G3,in_B3};
			else if (keyboardarea) {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'hff};//blue
			else {VGA_R, VGA_G, VGA_B} = {8'hf1, 8'hf1, 8'hbb};
		
		//Freedom mode / Song player mode / Game mode
		end else if ((mode_select == 16) | (mode_select == 17) | (mode_select == 18) | (mode_select == 19)) begin
			if (real_line_area1) {VGA_R, VGA_G, VGA_B} = {real_VGA_R, real_VGA_G, real_VGA_B};
		   else if (seg_line_area1) begin
				if (pixel_RGB[hcount[10:1] - 10'd146] == 2'b01) {VGA_R, VGA_G, VGA_B} = {8'hff, 8'h0, 8'h0};
				else if (pixel_RGB[hcount[10:1] - 10'd146] == 2'b10) {VGA_R, VGA_G, VGA_B} = {8'h0, 8'hff, 8'h0};
				else if (pixel_RGB[hcount[10:1] - 10'd146] == 2'b11) {VGA_R, VGA_G, VGA_B} = {8'h0, 8'h0, 8'hff};
				else {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'h00};
			end else if (line1|line2) {VGA_R, VGA_G, VGA_B} = {8'hff, 8'hff, 8'h00};
			else if (keychar1|keychar3|keychar5|keychar7|keychar9|keychar11|keychar13|keychar15) begin
				if ( |(keycurSegs & keysegment) ) begin
					if (keychar1|keychar3|keychar5|keychar7) {VGA_R, VGA_G, VGA_B} = {8'hff, 8'hff, 8'h00}; // yellow
					else if (keychar9|keychar11|keychar13) {VGA_R, VGA_G, VGA_B} = {8'hff, 8'h8c, 8'h00}; // yellow_2
					else {VGA_R, VGA_G, VGA_B} = {8'hff, 8'h69, 8'hb4}; // yellow_3
	         end else {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h99, 8'hff};
			end else if (keychar2|keychar4|keychar6|keychar8|keychar10|keychar12|keychar14|keychar16) begin
				if ( |(keycurSegs & keysegment) ) begin
					if (keychar2|keychar4|keychar6) {VGA_R, VGA_G, VGA_B} = {8'hff, 8'hff, 8'h00}; // yellow
					else if (keychar8|keychar10|keychar12|keychar14) {VGA_R, VGA_G, VGA_B} = {8'hff, 8'h8c, 8'h00}; // yellow_2
					else {VGA_R, VGA_G, VGA_B} = {8'hff, 8'h69, 8'hb4}; // yellow_3
	         end else {VGA_R, VGA_G, VGA_B} = {8'h00, 8'hcc, 8'hff};
         end else if (key1) begin
				if (press[1] == 0) {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h99, 8'hff};
			   else begin
					if (vcount >= 343 & vcount <= 477 & hcount > 83 & hcount <= 147) {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'hee};
					else {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'h00};
			   end
			end else if (key2) begin
			   if (press[2] == 0) {VGA_R, VGA_G, VGA_B} = {8'h00, 8'hcc, 8'hff};
			   else begin
					if (vcount >= 343 & vcount <= 477 & hcount > 153 & hcount <= 217) {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'hee};
					else {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'h00};
				end
			end else if (key3) begin
			   if (press[3] == 0) {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h99, 8'hff};
			   else begin
					if (vcount >= 343 & vcount <= 477 & hcount > 223 & hcount <= 287){VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'hee};
					else {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'h00};
				end
			end else if (key4) begin
			   if (press[4] == 0) {VGA_R, VGA_G, VGA_B} = {8'h00, 8'hcc, 8'hff};
			   else begin
					if (vcount >= 343 & vcount <= 477 & hcount > 293 & hcount <= 357) {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'hee};
					else {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'h00};
				end
			end else if (key5) begin
			   if (press[5] == 0) {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h99, 8'hff};
			   else begin
					if (vcount >= 343 & vcount <= 477 & hcount > 363 & hcount <= 427) {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'hee};
					else {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'h00};
				end
			end else if (key6) begin
			   if (press[6] == 0) {VGA_R, VGA_G, VGA_B} = {8'h00, 8'hcc, 8'hff};
			   else begin
					if (vcount >= 343 & vcount <= 477 & hcount > 433 & hcount <= 497) {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'hee};
					else {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'h00};
				end
			end else if (key7) begin
			   if (press[7] == 0) {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h99, 8'hff};
			   else begin
					if (vcount >= 343 & vcount <= 477 & hcount > 503 & hcount <= 567) {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'hee};
					else {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'h00};
				end
			end else if (key8) begin
			   if (press[8] == 0) {VGA_R, VGA_G, VGA_B} = {8'h00, 8'hcc, 8'hff};
			   else begin
					if (vcount >= 343 & vcount <= 477 & hcount > 573 & hcount <= 637) {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'hee};
					else {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'h00};
				end
			end else if (key9) begin
			   if (press[9] == 0) {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h99, 8'hff};
			   else begin
					if (vcount >= 343 & vcount <= 477 & hcount > 643 & hcount <= 707) {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'hee};
				   else {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'h00};
				end
			end else if (key10) begin
				if (press[10] == 0) {VGA_R, VGA_G, VGA_B} = {8'h00, 8'hcc, 8'hff};
			   else begin
					if (vcount >= 343 & vcount <= 477 & hcount > 713 & hcount <= 777) {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'hee};
					else {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'h00};
				end
			end else if (key11) begin
			   if (press[11] == 0) {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h99, 8'hff};
			   else begin
					if (vcount >= 343 & vcount <= 477 & hcount > 783 & hcount <= 847) {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'hee};
					else {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'h00};
				end
			end else if (key12) begin
			   if (press[12] == 0) {VGA_R, VGA_G, VGA_B} = {8'h00, 8'hcc, 8'hff};
			   else begin
					if (vcount >= 343 & vcount <= 477 & hcount > 853 & hcount <= 917) {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'hee};
					else {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'h00};
				end
			end else if (key13) begin
			   if (press[13] == 0) {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h99, 8'hff};
			   else begin
					if (vcount >= 343 & vcount <= 477 & hcount > 923 & hcount <= 987) {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'hee};
					else {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'h00};
				end
			end else if (key14) begin
			   if (press[14] == 0) {VGA_R, VGA_G, VGA_B} = {8'h00, 8'hcc, 8'hff};
			   else begin
					if (vcount >= 343 & vcount <= 477 & hcount > 993 & hcount <= 1057) {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'hee};
					else {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'h00};
				end
			end else if (key15) begin
			   if (press[15] == 0) {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h99, 8'hff};
			   else begin
					if (vcount >= 343 & vcount <= 477 & hcount > 1063 & hcount <= 1127) {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'hee};
					else {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'h00};
				end
			end else if (key16) begin
			   if (press[16] == 0) {VGA_R, VGA_G, VGA_B} = {8'h00, 8'hcc, 8'hff};
			   else begin
					if (vcount >= 343 & vcount <= 477 & hcount > 1133 & hcount <= 1197) {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'hee};
					else {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'h00};
				end
			end else if (backbutton) begin
				if ((mode_select == 17) | (mode_select == 19)) {VGA_R, VGA_G, VGA_B} = {8'h00, 8'hff, 8'h00};
				else {VGA_R, VGA_G, VGA_B} = {8'hff, 8'hff, 8'h00};
			
			end else if (note_display) begin 
				{VGA_R, VGA_G, VGA_B} = {in_R1,in_G1,in_B1};
				if (mode_select == 18) begin
					if (press[1] == 1 | press[8] == 1 | press[15] == 1) begin
						if (dao) begin 
							if (bit_dao[vcount[9:1] - 7'b1100010][hcount[10:2] - 8'b10010100] == 1){VGA_R, VGA_G, VGA_B} = {8'hff, 8'h00, 8'hff};
							else {VGA_R, VGA_G, VGA_B} = {in_R1,in_G1,in_B1}; // background
			         end
					end else if (press[2] == 1 | press[9] == 1 | press[16] == 1) begin
						if (re) begin 
							if (bit_re[vcount[9:1] - 7'b1100010][hcount[10:2] - 8'b10010101] == 1){VGA_R, VGA_G, VGA_B} = {8'hff, 8'h00, 8'hff};
							else {VGA_R, VGA_G, VGA_B} = {in_R1,in_G1,in_B1}; // background
					   end 
					end else if (press[3] == 1 | press[10] == 1) begin
						if (mi) begin 
							if (bit_mi[vcount[9:1] - 7'b1100010][hcount[10:2] - 8'b10010111] == 1){VGA_R, VGA_G, VGA_B} = {8'hff, 8'h00, 8'hff};
							else {VGA_R, VGA_G, VGA_B} = {in_R1,in_G1,in_B1}; // background
					   end 
					end else if (press[4] == 1 | press[11] == 1) begin
						if (fa) begin 
							if (bit_fa[vcount[9:1] - 7'b1100010][hcount[10:2] - 8'b10010101] == 1){VGA_R, VGA_G, VGA_B} = {8'hff, 8'h00, 8'hff};
							else {VGA_R, VGA_G, VGA_B} = {in_R1,in_G1,in_B1}; // background
						end
					end else if (press[5] == 1 | press[12] == 1) begin
						if (so) begin 
							if (bit_so[vcount[9:1] - 7'b1100010][hcount[10:2] - 8'b10010110] == 1){VGA_R, VGA_G, VGA_B} = {8'hff, 8'h00, 8'hff};
							else {VGA_R, VGA_G, VGA_B} = {in_R1,in_G1,in_B1}; // background
				      end
					end else if (press[6] == 1 | press[13] == 1) begin
						if (la) begin 
							if (bit_la[vcount[9:1] - 7'b1100010][hcount[10:2] - 8'b10010110] == 1){VGA_R, VGA_G, VGA_B} = {8'hff, 8'h00, 8'hff};
							else {VGA_R, VGA_G, VGA_B} = {in_R1,in_G1,in_B1}; // background
						end 
					end else if (press[7] == 1 | press[14] == 1) begin
						if (xi) begin 
							if (bit_xi[vcount[9:1] - 7'b1100010][hcount[10:2] - 8'b10011000] == 1){VGA_R, VGA_G, VGA_B} = {8'hff, 8'h00, 8'hff};
							else {VGA_R, VGA_G, VGA_B} = {in_R1,in_G1,in_B1}; // background
				      end 
               end else {VGA_R, VGA_G, VGA_B} = {in_R1,in_G1,in_B1};
				end
			/*end else if (inChar) begin
				if ( |(curSegs & segment) ) {VGA_R, VGA_G, VGA_B} = {8'hff, 8'h00, 8'h00}; // Red
				else if ( |segment ) {VGA_R, VGA_G, VGA_B} = {8'h30, 8'h30, 8'h30}; // Dark Gray	
			end else if (star_area1) begin
				//if ((in_R5 == 0) & (in_G5 == 0) & (in_B5 == 0)) {VGA_R, VGA_G, VGA_B} = {in_R1,in_G1,in_B1};
				//else 
				if (star_row_current != 0) {VGA_R, VGA_G, VGA_B} = {in_R4,in_G4,in_B4};
				else {VGA_R, VGA_G, VGA_B} = {in_R1,in_G1,in_B1};
			*/		
		   end else if (sprite1on) {VGA_R, VGA_G, VGA_B} = {in_R1,in_G1,in_B1};
			
		//Exit page
		end else if ((mode_select == 20)) begin
			if (sprite4on) begin
				if (byechange == 0) {VGA_R, VGA_G, VGA_B} = {in_R4,in_G4,in_B4};
				else {VGA_R, VGA_G, VGA_B} = {in_R5,in_G5,in_B5};
			end else {VGA_R, VGA_G, VGA_B} = {8'hff, 8'hff, 8'hff};
		end
	end
endmodule // VGA_LED_Emulator
