/*
 * Avalon memory-mapped peripheral for the VGA LED Emulator
 *
 * Stephen A. Edwards
 * Columbia University
 */

module PIANO_CONTROL(input logic         clk,
							input logic 	     reset,
							input logic [7:0]   writedata,
							input logic 	     write,
							input 		        chipselect,
							input logic [10:0]  address,
							input logic [3:0]   SW,
							input logic [3:0]   KEY,
			
				
							output logic [7:0]  VGA_R, VGA_G, VGA_B,
							output logic 	     VGA_CLK, VGA_HS, VGA_VS, VGA_BLANK_n,
							output logic 	     VGA_SYNC_n,
							
//AUD/////////////////////////////////////////////////////////////////////////////////////////
                     inout AUD_ADCLRCK,
							input AUD_ADCDAT,
							inout AUD_DACLRCK,
							output AUD_DACDAT,
							output AUD_XCK,
							inout AUD_BCLK,
							output AUD_I2C_SCLK,
							inout AUD_I2C_SDAT,
							output AUD_MUTE
							);
		
	//pixel: 0~1055 
	//base line: 1056~1057
	//mode select: 1058
	//audio demo: 1059
	//game control: 1060
	//audio player: 1061~1062
   logic [7:0] pixel [0:1057]; 
	logic [7:0] mode_select;	
	//mode  0: Screen Calibration running
	//mode  1: Screen Calibration done
	//mode  2: Screen Calibration button3 pressed to Color Calibration mode
	//mode  3: Color Calibration running
	//mode  4: Color Calibration function select
	//mode  5: Color Calibration button1 pressed to redo Color Calibration
	//mode  6: Color Calibration button2 pressed to display Original pixel
	//mode  7: Color Calibration button3 pressed to denoise pixel
	//mode  8: Color Calibration button4 pressed to recognize keys
	//mode  9: Color Calibration button5 pressed to Welcome page
	//mode 10: Welcome page
	//mode 11: Welcome page button1 pressed to Freedom mode
	//mode 12: Welcome page button2 pressed to Song player mode
	//mode 13: Welcome page button3 pressed to Game mode 
	//mode 14: Welcome page button4 pressed to Screen Calibration mode
	//mode 15: Welcome page button5 pressed to Color Calibration mode
	//mode 16: Freedom mode
	//mode 17: Freedom mode all button pressed to back to Welcome page
	//mode 18: Song player mode
	//mode 19: Song player mode all button pressed to back to Welcome page
	//mode 20: Exit page
	logic [7:0] audio_demo;
	logic [7:0] drop_position;	
	logic [7:0] speed;
	
	initial begin
		mode_select = 0;
	end
	
	logic [1:0] pixel_RGB [0:351];
	logic [4:0] fix_keys_position [0:351];
	logic [1:0] fix_pixel_RGB [0:351];
	logic [1:0] fix_pixel_RGB_denise [0:351];

	logic [4:0] key_num;
	logic [16:0] press;
	logic [16:0] press1;
	logic [16:0] press2;
	logic [7:0] R1, G1, B1;
	logic [7:0] R2, G2, B2;
	logic [7:0] R3, G3, B3;
	logic [7:0] R4, G4, B4;
	logic [7:0] R5, G5, B5;
	logic [9:0] star_offset;
	logic [5:0] star_row_current;
	integer i;
   logic [7:0] row_num_1, row_num_2;
	logic [8:0] row_num;
	logic [15:0]  addr1, addr2, addr3, addr4, addr5;
   logic [23:0] pixel1, pixel2, pixel3, pixel4, pixel5;
   logic [9:0] hcount, vcount;
	logic       clk_vga;
	
	assign row_num = row_num_1 + row_num_2;
   
	rom newrom1(.clock(clk), .address(addr1), .q(pixel1));
	ROM1 newrom2(.clock(clk), .address(addr2), .q(pixel2));
	rom2 newrom3(.clock(clk), .address(addr3), .q(pixel3));
	ROM_bye1 newrom4(.clock(clk), .address(addr4), .q(pixel4));
	ROM_bye2 newrom5(.clock(clk), .address(addr5), .q(pixel5));
	rgb_controller controller_1 (.clk(clk), .h(hcount), .c(vcount),.addr_bg1(addr1), .M_bg1(pixel1), .addr_bg2(addr2), .M_bg2(pixel2), .addr_bg3(addr3), .M_bg3(pixel3), .addr_bg4(addr4), .M_bg4(pixel4), .addr_bg5(addr5), .M_bg5(pixel5), .*);
   	
   PIANO_DISPLAY display(.clk50(clk), .in_R1(R1), .in_G1(G1), .in_B1(B1), .in_R2(R2), .in_G2(G2), .in_B2(B2),.in_R3(R3), .in_G3(G3), .in_B3(B3), .in_R4(R4), .in_G4(G4), .in_B4(B4), .in_R5(R5), .in_G5(G5), .in_B5(B5), .h(hcount), .v(vcount), .*);
	SEGMENT seg(.clk50(clk), .*);
	RECOGNIZE_KEY recog(.clk50(clk), .*);
	DENOISE deno(.clk50(clk), .*);
	MULTI_KEY multikey (.clk50(clk), .press(press), .press1(press1), .press2(press2));
	
   
	logic [16:0]  press_real;
	FINGER_DETECTION finger(.clk50(clk), .press(press_real), .*);
	PRESS pre(.clk50(clk), .*);

   always_ff @(posedge clk)
     if (reset) 
		  begin
			 for (i = 0; i < 1056; i = i + 1) 
				begin
				  pixel[i] <= 8'd255;
				end
		  end 
	  else if (chipselect && write)
	    begin
		   if (address == 11'd1056) 
			  begin 
			    row_num_1 <= writedata;
		     end 
			else if (address == 11'd1057) 
			  begin 
			    row_num_2 <= writedata;
		     end
			else if (address == 11'd1058) 
			  begin 
			    mode_select <= writedata;
		     end  
			else if (address == 11'd1059) 
			  begin 
			    audio_demo <= writedata;
		     end  
			else 
			  begin 
			    pixel[address] <= writedata;
		     end 
	    end
   
	always_ff @(posedge clk) begin
		if (mode_select == 3) fix_pixel_RGB <= pixel_RGB;
	end
	      
//AUD/////////////////////////////////////////////////////////////////////////////////////////
   wire AUD_reset = !KEY[2];
	wire main_clk;
	wire audio_clk;
	wire clk27;
	wire [1:0] sample_end;
	wire [1:0] sample_req;
	wire [15:0] audio_output;
	wire [15:0] audio_input;
	wire [15:0] audio_output1;
	wire [15:0] audio_output2;
	
	clock_pll pll (
			.refclk (clk),
			.rst (AUD_reset),
			.outclk_0 (main_clk),
			.outclk_1 (audio_clk),
			.outclk_2 (clk27)
	);
	
	i2c_av_config av_config (
			.clk (main_clk),
			.reset (AUD_reset),
			.i2c_sclk (AUD_I2C_SCLK),
			.i2c_sdat (AUD_I2C_SDAT),
	);
	
	assign AUD_XCK = audio_clk;
	assign AUD_MUTE = 1;	
	
	assign audio_output = (audio_output1 + audio_output2)/2;
	
	audio_codec ac (
				.clk (audio_clk),
				.reset (AUD_reset),
				.sample_end (sample_end),
				.sample_req (sample_req),
				.audio_output (audio_output),
				.audio_input (audio_input),
				.channel_sel (2'b10),
				.AUD_ADCLRCK (AUD_ADCLRCK),
				.AUD_ADCDAT (AUD_ADCDAT),
				.AUD_DACLRCK (AUD_DACLRCK),
				.AUD_DACDAT (AUD_DACDAT),
				.AUD_BCLK (AUD_BCLK)
	);
	karplus_note kar1 (
				.clock50 (clk27),
				.audiolrclk (AUD_DACLRCK),
				.reset (AUD_reset),
				.audio_output (audio_output1),
				.audio_input (audio_input),
				.control (SW),
				.press (press1)
	);
	karplus_note kar2 (
				.clock50 (clk27),
				.audiolrclk (AUD_DACLRCK),
				.reset (AUD_reset),
				.audio_output (audio_output2),
				.audio_input (audio_input),
				.control (SW),
				.press (press2)
	);
   //AUD/////////////////////////////////////////////////////////////////////////////////////////
	
		
endmodule
