library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;

entity de2_vga_raster is
  
  port (
    clk50     : in std_logic;
    reset : in std_logic;
    chipselect : in std_logic;
    write, read : in std_logic;
    address  :  in std_logic_vector(15 downto 0);
    writedata : in std_logic_vector(15 downto 0);
    readdata : out std_logic_vector(15 downto 0);
	VGA_IRQ : OUT std_logic ;
    VGA_CLK,                         -- Clock
    VGA_HS,                          -- H_SYNC
    VGA_VS,                          -- V_SYNC
    VGA_BLANK,                       -- BLANK
    VGA_SYNC : out std_logic;        -- SYNC
    VGA_R,                           -- Red[9:0]
    VGA_G,                           -- Green[9:0]
    VGA_B : out unsigned(9 downto 0) -- Blue[9:0]
    );

end de2_vga_raster;

architecture rtl of de2_vga_raster is
  -- Address Assignment:
  -- a(15:14) == 00  ->  Set Tile
  --     a(5:0)      ->  X
  --     a(10:6)     ->  Y
  -- a(15:14) == 01  ->  Set Man Property
  --     a(2:0)      ->  Property
  --       000       ->  X
  --       001       ->  Y
  --       010       ->  Direction
  -- a(15:14) == 10  ->  Settings
  --     a(2:0)      ->  Setting
  --       000       ->  Speed


component man_rom
	PORT
	(
		address		: IN STD_LOGIC_VECTOR (10 DOWNTO 0);
		clock		: IN STD_LOGIC ;
		q		: OUT STD_LOGIC_VECTOR (24 DOWNTO 0)
	);
end component;


  component man_register is
    port
    (
      clk   : in std_logic;
      reset_n : in std_logic;
      write  : in std_logic;
      property : in std_logic_vector(2 downto 0);
      value : in std_logic_vector(15 downto 0);

      x : out integer;
      y : out integer;
      dir : out integer
    );
  end component;

component tile_rom
	PORT
	(
		address		: IN STD_LOGIC_VECTOR (13 DOWNTO 0);
		clock		: IN STD_LOGIC ;
		q		: OUT STD_LOGIC_VECTOR (15 DOWNTO 0)
	);
end component;

component tile_ram_mega
	PORT
	(
		clock		: IN STD_LOGIC ;
		data		: IN STD_LOGIC_VECTOR (5 DOWNTO 0);
		rdaddress_a		: IN STD_LOGIC_VECTOR (10 DOWNTO 0);
		rdaddress_b		: IN STD_LOGIC_VECTOR (10 DOWNTO 0);
		wraddress		: IN STD_LOGIC_VECTOR (10 DOWNTO 0);
		wren		: IN STD_LOGIC  := '1';
		qa		: OUT STD_LOGIC_VECTOR (5 DOWNTO 0);
		qb		: OUT STD_LOGIC_VECTOR (5 DOWNTO 0)
	);
end component;



  -- Video parameters

  constant HTOTAL       : integer := 800;
  constant HSYNC        : integer := 96;
  constant HBACK_PORCH  : integer := 48;
  constant HACTIVE      : integer := 640;
  constant HFRONT_PORCH : integer := 16;
  
  constant VTOTAL       : integer := 525;
  constant VSYNC        : integer := 2;
  constant VBACK_PORCH  : integer := 33;
  constant VACTIVE      : integer := 480;
  constant VFRONT_PORCH : integer := 10;

  constant MAN_W    : integer := 20;
  constant MAN_H    : integer := 28;

  constant tileWidth : integer := 16;
  constant tileHeight : integer := 16;

  constant statusTileHeight : integer := 2;
  -- Signals for the video controller
  signal Hcount : unsigned(9 downto 0);  -- Horizontal position (0-800)
  signal Vcount : unsigned(9 downto 0);  -- Vertical position (0-524)
  signal EndOfLine, EndOfField : std_logic;
  signal offset, tileOffset : integer;
  signal counter : unsigned (15 downto 0);
  signal counter_max : unsigned (15 downto 0) := to_unsigned(512, 16);
  signal vga_hblank, vga_hsync,
    vga_vblank, vga_vsync, blank : std_logic;  -- Sync. signals

  signal addrX, addrY: integer;  -- the pixel to read from each tile rom
  signal RGB: STD_LOGIC_VECTOR (15 DOWNTO 0);
  signal tileNumberX: integer; -- the horizontal index of the tile
  signal tileNumberY : integer; -- vertical one
  signal tileNumber: integer ;  -- tile name, now it is "000" to "111"

  signal tileWrite : std_logic;
  signal tileNumberA, tileNumberB : std_logic_vector(5 downto 0);

  signal romAddr, alloAddrY : integer;
  signal clk25 : std_logic := '0';

  signal vga_active : std_logic;
  signal x_pos, y_pos : unsigned(9 downto 0);

  signal write_man, man_on : std_logic;
  signal man_x, man_y, man_dir : integer;
  signal man_address_base, man_address : integer;
  signal man_data     : std_logic_vector(24 downto 0);
  signal man_pixel : integer;
begin
  vga_active <= not vga_hblank and not vga_vblank;
  x_pos <= Hcount-(HSYNC + HBACK_PORCH);
  y_pos <= Vcount-(VSYNC + VBACK_PORCH);

  process (clk50)
  begin
    if rising_edge(clk50) then
      clk25 <= not clk25;

		if chipselect = '1' and write = '1' and address(15 downto 14) = "10" and address(2 downto 0) = "000" then
			counter_max <= unsigned(writedata);
		else 
			counter_max <= counter_max;
		end if;
    end if;
  end process;

-- generate the IRQ when the vcount reach the bottom of the screen
 process (clk25)
 begin
	if rising_edge (clk25) then
		if reset = '1' then
			VGA_IRQ <= '0';
		else
			if chipselect = '1' and write = '1' and address(15 downto 14) = "11" then
				VGA_IRQ <= '0';
			else
				if vga_vsync = '1' then
					VGA_IRQ <= '1' ;
				else
					VGA_IRQ <= '0';
				end if;
			end if;
		end if;
	end if;
end process;



  write_man <= chipselect and write and '1' when (address(15 downto 14) = "01") else '0';
  MAN_R: man_register port map (
    clk         => clk50,
    reset_n     => not reset,
    write       => write_man,
    property    => address(2 downto 0),
    value       => writedata,

    x           => man_x,
    y           => man_y,
    dir         => man_dir
  );

  man_address_base <= 0 when man_dir = 0 else
                      MAN_W*MAN_H*1 when man_dir = 1 else
                      MAN_W*MAN_H*2 when man_dir = 2 else
                      0;
  man_address <= man_address_base + man_pixel + 1;

  man_rom_inst : man_rom PORT MAP (
    address	 => std_logic_vector(to_unsigned(man_address, 11)),
    clock	 => clk25,
    q	 => man_data
  );

  ManGen : process (clk25)
  begin
    if rising_edge(clk25) then 
      if reset = '1' then
        man_on <= '0';
      elsif(vga_active = '1') then
        if (y_pos >= man_y and y_pos < man_y + MAN_H) and
           (x_pos >= man_x and x_pos < man_x + MAN_W) and
           (man_data(24) = '1') then
          man_on <= '1';
        else 
          man_on <= '0';
        end if;
      else
        man_on <= '0';
      end if;
    end if;
  end process ManGen;

  ManPixelGen : process (clk25)
  begin
    if rising_edge(clk25) then 
      if reset = '1' then
        man_pixel <= -1;
      elsif(vga_active = '1') then 
        if(x_pos + 1 = man_x-1 and y_pos + 1 = man_y-1) then
          man_pixel <= -1;
        elsif	(y_pos + 1 >= man_y and y_pos+ 1 < man_y + MAN_H) and
               (x_pos + 1>= man_x and x_pos + 1< man_x + MAN_W) then
          man_pixel <= man_pixel + 1;
        end if;
      end if;
    end if;
  end process ManPixelGen;

tile_rom_inst : tile_rom PORT MAP (
		address	 => std_logic_vector(to_unsigned(romAddr, 14)),
		clock	 => not clk25,
		q	 => RGB
	);	
	
	
tileWrite <= chipselect and write and '1' when address(15 downto 14) = "00" else '0';
tile_ram_mega_inst : tile_ram_mega PORT MAP (
		clock	 => clk50,
		data	 => writedata(5 downto 0),
		rdaddress_a	 => std_logic_vector(to_unsigned(alloAddrY, 5)) & std_logic_vector(to_unsigned(tileNumberX, 6)),
		rdaddress_b	 => address(10 downto 0),
		wraddress	 => address(10 downto 0),
		wren	 => tileWrite,
		qa	 => tileNumberA,
		qb	 => tileNumberB
	);

tileNumber <= to_integer(unsigned(tileNumberA));
readdata(15 downto 11) <= std_logic_vector (to_unsigned (tileOffset, 5));
readdata (10 downto 7) <= std_logic_vector (to_unsigned (offset, 4));
readdata(5 downto 0) <= tileNumberB;


count: process (clk25) 
begin
	if rising_edge (clk25) then

	--------------------------
		if reset = '1' or counter = counter_max or counter_max = 0 then
			counter <= (others => '0');
		elsif vga_vblank = '1' then
			counter <= counter + 1;	
		end if;
		
		if tileNumberY>=statusTileHeight then 
			romAddr <= ((tileNumber)*256 + (addrY-1)*(tileWidth) + addrX);
		else
			romAddr <= ((tileNumber)*256 + (addrY-1-offset)*(tileWidth) + addrX);
		end if;
		addrY <= to_integer (Vcount - VSYNC - VBACK_PORCH - tileHeight * tileNumberY + offset + 1); -- plus the offset

	end if;
end process count;


offsetPro: process (clk25)
begin
	if rising_edge(clk25) then
		if reset = '1' then
			offset <= 0;
			tileOffset <= statusTileHeight-1;
-- only when the vcount reach the bottom, do we change the offset
-- so there will never the flashing on the tile
-- the sprites flashing issue is solved by the IRQ in software level
		elsif  counter = counter_max and counter_max > 0 then
			if offset = tileHeight - 1 then
				offset <=  0;
				
				if tileOffset = 31 then
					tileOffset <= statusTileHeight;
				else
					tileOffset <= tileOffset + 1;
				end if;
			else
				offset <= offset + 1;
			end if;
		end if;
		
		if tileNumberY>=statusTileHeight then
			if tileNumberY+tileOffset-1 > 31 then
				alloAddrY <= tileNumberY+tileOffset-1+statusTileHeight;
			else
				alloAddrY <= tileNumberY+tileOffset-1;
			end if;
		else
			alloAddrY <= tileNumberY; 
		end if;
	end if;
end process offsetPro;


tileX: process (clk25)
begin 
	if falling_edge (clk25) then
		if reset = '1' then
			tileNumberX <= 0;
		else 
			if Hcount = "0000000000" then
				tileNumberX<= 0;
			elsif Hcount  > (tileNumberX+1)* tileWidth +  HSYNC + HBACK_PORCH - 2 - 1 then
				tileNumberX <= tileNumberX+1;
			end if;
		end if;
		
		addrX <= to_integer (Hcount - HSYNC - HBACK_PORCH - tileWidth * tileNumberX + 1);
	end if;
end process tileX;

 tileY: process (clk25)
begin 
	if falling_edge (clk25) then
		if reset = '1' then
			tileNumberY <= 0;
		else 
			if Vcount = "0000000000" then
				tileNumberY<= 0;
			elsif tileNumberY >= statusTileHeight and Vcount > (tileNumberY +1)* tileHeight + VSYNC + VBACK_PORCH - offset -1 then
				tileNumberY <= tileNumberY+1;
			elsif Vcount > (tileNumberY +1)* tileHeight + VSYNC + VBACK_PORCH -1 then
				tileNumberY <= tileNumberY+1;
			end if;
		end if;
	end if;
end process tileY;
   

  HCounter : process (clk25)
  begin
    if rising_edge(clk25) then      
      if reset = '1' then
        Hcount <= (others => '0');
      elsif EndOfLine = '1' then
        Hcount <= (others => '0');
      else
        Hcount <= Hcount + 1;
      end if;      
    end if;
  end process HCounter;

  EndOfLine <= '1' when Hcount = HTOTAL - 1 else '0';
  
  VCounter: process (clk25)
  begin
    if rising_edge(clk25) then      
      if reset = '1' then
        Vcount <= (others => '0');
      elsif EndOfLine = '1' then
        if EndOfField = '1' then
          Vcount <= (others => '0');
        else
          Vcount <= Vcount + 1;
        end if;
      end if;
    end if;
  end process VCounter;

  EndOfField <= '1' when Vcount = VTOTAL - 1 else '0';

  -- State machines to generate HSYNC, VSYNC, HBLANK, and VBLANK

  HSyncGen : process (clk25)
  begin
    if rising_edge(clk25) then     
      if reset = '1' or EndOfLine = '1' then
        vga_hsync <= '1';
      elsif Hcount = HSYNC - 1 then
        vga_hsync <= '0';
      end if;
    end if;
  end process HSyncGen;
  
  HBlankGen : process (clk25)
  begin
    if rising_edge(clk25) then
      if reset = '1' then
        vga_hblank <= '1';
      elsif Hcount = HSYNC + HBACK_PORCH then
        vga_hblank <= '0';
      elsif Hcount = HSYNC + HBACK_PORCH + HACTIVE then
        vga_hblank <= '1';
      end if;      
    end if;
  end process HBlankGen;

  VSyncGen : process (clk25)
  begin
    if rising_edge(clk25) then
      if reset = '1' then
        vga_vsync <= '1';
      elsif EndOfLine ='1' then
        if EndOfField = '1' then
          vga_vsync <= '1';
        elsif Vcount = VSYNC - 1 then
          vga_vsync <= '0';
        end if;
      end if;      
    end if;
  end process VSyncGen;

  VBlankGen : process (clk25)
  begin
    if rising_edge(clk25) then    
      if reset = '1' then
        vga_vblank <= '1';
      elsif EndOfLine = '1' then
        if Vcount = VSYNC + VBACK_PORCH - 1 then
          vga_vblank <= '0';
        elsif Vcount = VSYNC + VBACK_PORCH + VACTIVE - 1 then
          vga_vblank <= '1';
        end if;
      end if;
    end if;
  end process VBlankGen;

--process (clk25)
--  begin
--    if rising_edge(clk25) then    
--      if reset = '1' then
--        blank <= '0';
--      elsif EndOfLine = '1' then
--        if Vcount = VSYNC + VBACK_PORCH + VACTIVE then
--         blank <= '0';
--        elsif Vcount = VSYNC + VBACK_PORCH + VACTIVE - 1 then
--          blank <= '1';
--        end if;
--      end if;
--    end if;
--  end process;
  -- Registered video signals going to the video DAC

  VideoOut: process (clk25, reset)
  begin
    if reset = '1' then
      VGA_R <= "0000000000";
      VGA_G <= "0000000000";
      VGA_B <= "0000000000";
    elsif clk25'event and clk25 = '1' then

	  if vga_active = '1' then
		if man_on = '1' then 
			VGA_R <= unsigned(man_data(23 downto 16) & "00");
			VGA_G <= unsigned(man_data(15 downto 8) & "00");
			VGA_B <= unsigned(man_data(7 downto 0) & "00"); 
		else 
			VGA_R <= unsigned (RGB(15 downto 11) & "00000");
			VGA_G <= unsigned(RGB(10 downto 5) & "0000");
			VGA_B <= unsigned(RGB(4 downto 0) & "00000");
		end if;
      else
        VGA_R <= "0000000000";
        VGA_G <= "0000000000";
        VGA_B <= "0000000000";    
      end if;
    end if;
  end process VideoOut;

  VGA_CLK <= clk25;
  VGA_HS <= not vga_hsync;
  VGA_VS <= not vga_vsync;
  VGA_SYNC <= '0';
  VGA_BLANK <= not (vga_hsync or vga_vsync);

end rtl;
