-------------------------------------------------------------------------------
--
-- Simple VGA raster display
--
-- Stephen A. Edwards
-- sedwards@cs.columbia.edu
--
-------------------------------------------------------------------------------
library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;

entity vga_circle is
  
  port (
    VGA_CLK,                         -- Clock
    VGA_HS,                          -- H_SYNC
    VGA_VS,                          -- V_SYNC
    VGA_BLANK,                       -- BLANK
    VGA_SYNC : out std_logic;        -- SYNC
    VGA_R,                           -- Red[9:0]
    VGA_G,                           -- Green[9:0]
    VGA_B : out unsigned(9 downto 0); -- Blue[9:0]
	clkin      : in  std_logic;
    reset_n    : in  std_logic;
    read       : in  std_logic;
    write      : in  std_logic;
    chipselect : in  std_logic;
    address    : in  unsigned(15 downto 0);
    readdata   : out unsigned(15 downto 0);
    writedata  : in  unsigned(15 downto 0)
    );






end vga_circle;

architecture rtl of vga_circle is
  
  -- Video parameters
  
  constant HTOTAL       : integer := 800;
  constant HSYNC        : integer := 96;
  constant HBACK_PORCH  : integer := 48;
  constant HACTIVE      : integer := 640;
  constant HFRONT_PORCH : integer := 16;
  
  constant VTOTAL       : integer := 525;
  constant VSYNC        : integer := 2;
  constant VBACK_PORCH  : integer := 33;
  constant VACTIVE      : integer := 480;
  constant VFRONT_PORCH : integer := 10;

  
  signal clk: std_logic :='0'; 
  signal Hcount : unsigned(9 downto 0);  -- Horizontal position (0-800)
  signal Vcount : unsigned(9 downto 0);  -- Vertical position (0-524)
  signal EndOfLine, EndOfField : std_logic;

  signal vga_hblank, vga_hsync,
    vga_vblank, vga_vsync : std_logic;  -- Sync. signals
  signal reset:std_logic;
  
  
 signal we,re,we_in : std_logic;
 signal vadd : unsigned (7 downto 0);
 signal do : unsigned ( 7 downto 0 );	
signal do2 : unsigned ( 7 downto 0 );
signal X : unsigned(9 downto 0);
signal Y : unsigned(9 downto 0);
signal shortad : unsigned(9 downto 0);
signal Xr : unsigned (3 downto 0);
signal Yr : unsigned (3 downto 0);
signal win : unsigned (7 downto 0);
type ram_type is array(225 downto 0) of unsigned(7 downto 0);
  signal RAM : ram_type;
  signal ram_address, display_address : unsigned(7 downto 0);
signal dra:std_logic;
signal col:unsigned(2 downto 0);
signal wri: unsigned ( 7 downto 0 );
begin
  reset <= not reset_n;
  ram_address <= address (7 downto 0);
wri <= writedata ( 7 downto 0);
process (clkin)
  begin
    if rising_edge(clkin) then
      if reset_n = '0' then
        readdata <= (others => '0');
      else
        if chipselect = '1' then
            if read = '1' then
              readdata <= x"00" & RAM(to_integer(ram_address));
            elsif write = '1' then
              RAM(to_integer(ram_address)) <= wri;
            end if;
          end if;
		
end if;
end if;
end process;





 process (clkin)
  begin
    if rising_edge(clkin) then
      clk <= not clk;
    end if;
  end process;

do <= RAM(to_integer(vadd));

--Address 
X    	<= (Hcount - HSYNC - HBACK_PORCH) ;
Y    	<= (Vcount - VSYNC - VBACK_PORCH) ;
Xr 		<= X (8 downto 5);
Yr 		<= Y (8 downto 5);
vadd 	<= Xr + Yr*15;
dra 	<= do(0);
col 	<= do(3 downto 1);






  HCounter : process (clk)
  begin
    if rising_edge(clk) then      
      if reset = '1' then
        Hcount <= (others => '0');
      elsif EndOfLine = '1' then
        Hcount <= (others => '0');
      else
        Hcount <= Hcount + 1;
      end if;      
    end if;
  end process HCounter;

  EndOfLine <= '1' when Hcount = HTOTAL - 1 else '0';
  
  VCounter: process (clk)
  begin
    if rising_edge(clk) then      
      if reset = '1' then
        Vcount <= (others => '0');
      elsif EndOfLine = '1' then
        if EndOfField = '1' then
          Vcount <= (others => '0');
        else
          Vcount <= Vcount + 1;
        end if;
      end if;
    end if;
  end process VCounter;

  EndOfField <= '1' when Vcount = VTOTAL - 1 else '0';

  -- State machines to generate HSYNC, VSYNC, HBLANK, and VBLANK

  HSyncGen : process (clk)
  begin
    if rising_edge(clk) then     
      if reset = '1' or EndOfLine = '1' then
        vga_hsync <= '1';
      elsif Hcount = HSYNC - 1 then
        vga_hsync <= '0';
      end if;
    end if;
  end process HSyncGen;
  
  HBlankGen : process (clk)
  begin
    if rising_edge(clk) then
      if reset = '1' then
        vga_hblank <= '1';
      elsif Hcount = HSYNC + HBACK_PORCH then
        vga_hblank <= '0';
      elsif Hcount = HSYNC + HBACK_PORCH + HACTIVE then
        vga_hblank <= '1';
      end if;      
    end if;
  end process HBlankGen;

  VSyncGen : process (clk)
  begin
    if rising_edge(clk) then
      if reset = '1' then
        vga_vsync <= '1';
      elsif EndOfLine ='1' then
        if EndOfField = '1' then
          vga_vsync <= '1';
        elsif Vcount = VSYNC - 1 then
          vga_vsync <= '0';
        end if;
      end if;      
    end if;
  end process VSyncGen;

  VBlankGen : process (clk)
  begin
    if rising_edge(clk) then    
      if reset = '1' then
        vga_vblank <= '1';
      elsif EndOfLine = '1' then
        if Vcount = VSYNC + VBACK_PORCH - 1 then
          vga_vblank <= '0';
        elsif Vcount = VSYNC + VBACK_PORCH + VACTIVE - 1 then
          vga_vblank <= '1';
        end if;
      end if;
    end if;
  end process VBlankGen;

  

  -- Registered video signals going to the video DAC

  VideoOut: process (clk, reset)
  begin

    if reset = '1' then
      VGA_R <= "0000000000";
      VGA_G <= "0000000000";
      VGA_B <= "0000000000";
    elsif clk'event and clk = '1' then
      if dra = '1' and col = "000" and  vga_hblank <= '0' and vga_vblank <= '0' and X<480 then
		VGA_R <= "1111111111";
        VGA_G <= "0000000000";
        VGA_B <= "0000000000";
        elsif dra = '1' and col = "001" and  vga_hblank <= '0' and vga_vblank <= '0' and X<480 then 
		VGA_R <= "0000000000";
        VGA_G <= "1111111111";
        VGA_B <= "0000000000";
		elsif dra = '1' and col = "010" and  vga_hblank <= '0' and vga_vblank <= '0' and X<480 then 
		VGA_R <= "0000000000";
        VGA_G <= "0000000000";
        VGA_B <= "1111111111";
		elsif dra = '1' and col = "011" and  vga_hblank <= '0' and vga_vblank <= '0' and X<480 then 
		VGA_R <= "1111111111";
        VGA_G <= "1111111111";
        VGA_B <= "0000000000";
		elsif dra = '1' and col = "100" and  vga_hblank <= '0' and vga_vblank <= '0' and X<480 then 
		VGA_R <= "1111111111";
        VGA_G <= "0000000000";
        VGA_B <= "1111111111";
		
	    
      elsif vga_hblank = '0' and vga_vblank ='0' then
        VGA_R <= "1000000000";
        VGA_G <= "0111000000";
        VGA_B <= "1111111111";
      else
        VGA_R <= "0000000000";
        VGA_G <= "0000000000";
        VGA_B <= "0000000000";    
      end if;
 end if;
  end process VideoOut;

  VGA_CLK <= clk;
  VGA_HS <= not vga_hsync;
  VGA_VS <= not vga_vsync;
  VGA_SYNC <= '0';
  VGA_BLANK <= not (vga_hsync or vga_vsync);

end rtl;
