-- This file is part of The Awesome Guitar Game.
--
--    The Awesome Guitar Game is a free software: you can redistribute it and/or modify
--    it under the terms of the GNU General Public License as published by
--    the Free Software Foundation, either version 3 of the License, or
--    (at your option) any later version.
--
--    The Awesome Guitar Game is distributed in the hope that it will be useful,
--    but WITHOUT ANY WARRANTY; without even the implied warranty of
--    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--    GNU General Public License for more details.
--
--    You should have received a copy of the GNU General Public License
--    along with The Awesome Guitar Game.  If not, see <http://www.gnu.org/licenses/>.
--
--
--###########################################################
-- The awesome guitar game (A clone of Guitar Hero for FPGA #
--###########################################################
-- EMBEDDED SYSTEM PROJECT
-- Columbia Unviersity Spring 2012
--
-- Avijit Singh Wasu -- asw2156@columbia.edu
-- Laurent Charignon -- lc2817@columbia.edu
-- 
-- Licensed under the GPL license
-- Have a look at the license file in the root of 
-- the project to have more details about the license
--
--##########################################################
-------------------------------------------------------------------------------
--
-- Modified version of the Simple VGA raster display
-- Initial design by:
-- Stephen A. Edwards
-- sedwards@cs.columbia.edu
--
-------------------------------------------------------------------------------
library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;

entity vga_controller is
  
  port (
    VGA_CLK,                         -- Clock
    VGA_HS,                          -- H_SYNC
    VGA_VS,                          -- V_SYNC
    VGA_BLANK,                       -- BLANK
    VGA_SYNC : out std_logic;        -- SYNC
    VGA_R,                           -- Red[9:0]
    VGA_G,                           -- Green[9:0]
    VGA_B : out unsigned(9 downto 0); -- Blue[9:0]
	clkin      : in  std_logic;
    reset_n    : in  std_logic;
    read       : in  std_logic;
    write      : in  std_logic;
    chipselect : in  std_logic;
    address    : in  unsigned(15 downto 0);
    readdata   : out unsigned(15 downto 0);
    writedata  : in  unsigned(15 downto 0)
    );



end vga_controller;

architecture rtl of vga_controller is
  
  -- Video parameters
  
  constant HTOTAL       : integer := 800;
  constant HSYNC        : integer := 96;
  constant HBACK_PORCH  : integer := 48;
  constant HACTIVE      : integer := 640;
  constant HFRONT_PORCH : integer := 16;
  
  constant VTOTAL       : integer := 525;
  constant VSYNC        : integer := 2;
  constant VBACK_PORCH  : integer := 33;
  constant VACTIVE      : integer := 480;
  constant VFRONT_PORCH : integer := 10;

  
  signal clk                    : std_logic :='0'; 
  signal Hcount                 : unsigned(9 downto 0);  -- Horizontal position (0-800)
  signal Vcount                 : unsigned(9 downto 0);  -- Vertical position (0-524)
  signal EndOfLine, EndOfField  : std_logic;
  signal vga_hblank, vga_hsync,
  vga_vblank, vga_vsync         : std_logic;  -- Sync. signals
  signal reset                  : std_logic;
  signal X                      : unsigned(9 downto 0);
  signal Y                      : unsigned(9 downto 0);
type ram_type is array(31 downto 0) of unsigned(11 downto 0);

  signal RAM                    : ram_type;
  signal ram_address            : unsigned(4 downto 0);
 
  signal RCUM 					: unsigned(9 downto 0); --R CUMULATIVE
  signal GCUM 					: unsigned(9 downto 0); --G CUMULATIVE
  signal BCUM 					: unsigned(9 downto 0); --B CUMULATIVE




component cell is port(
	clk 	: in std_logic;
	X  		: in unsigned(9 downto 0);
	Y  		: in  unsigned(9 downto 0);
	REGVAL	: in  unsigned(11 downto 0);
	RGB 	: out  unsigned(30 downto 0); -- First bit is transparency 0 = transparent
	ADD_cell: out unsigned(9 downto 0 );
	RGB_CELL: in unsigned (24 downto 0 );
	en 		: out std_logic
	
);
end component cell; 
  component rombut is port(
	clk , en : in std_logic;
	data : out unsigned (24 downto 0);
	addr : in unsigned (9 downto 0)
);
end component rombut;

signal A0,A1,A2,A3,A4,A5,A6,A7,A8,A9,A10,A11,A12,A13,A14,A15,Ad,Ad1,Ad2,Ad3,Ad4: unsigned (9 downto 0 );
signal E0,E1,E2,E3,E4,E5,E6,E7,E8,E9,E10,E11,E12,E13,E14,E15: std_logic;
signal RG0,RG1,RG2,RG3,RG4 : unsigned (24 downto 0 );
signal RGB0,RGB1,RGB2,RGB3,RGB4,RGB5,RGB6,RGB7,RGB8,RGB9,RGB10,RGB11,RGB12,RGB13,RGB14,RGB15: unsigned (30 downto 0);
signal en:std_logic;
begin

  reset       <= not reset_n;
  ram_address <= address (4 downto 0);
  X    	      <= (Hcount - HSYNC - HBACK_PORCH) ;
  Y    	      <= (Vcount - VSYNC - VBACK_PORCH) ;

R0 : rombut port map (clk, en, RG0, Ad); 
R1 : rombut port map (clk, en, RG1, Ad1); 
R2 : rombut port map (clk, en, RG2, Ad2); 
R3 : rombut port map (clk, en, RG3, Ad3); 
R4 : rombut port map (clk, en, RG4, Ad4); 


C0: cell port map (clk,X,Y,RAM(0),RGB0,A0,RG0,E0);
C1: cell port map (clk,X,Y,RAM(1),RGB1,A1,RG0,E1);
C2: cell port map (clk,X,Y,RAM(2),RGB2,A2,RG0,E2);
C3: cell port map (clk,X,Y,RAM(3),RGB3,A3,RG0,E3);
C4: cell port map (clk,X,Y,RAM(4),RGB4,A4,RG0,E4);
C5: cell port map (clk,X,Y,RAM(5),RGB5,A5,RG0,E5);
C6: cell port map (clk,X,Y,RAM(6),RGB6,A6,RG0,E6);
C7: cell port map (clk,X,Y,RAM(7),RGB7,A7,RG0,E7);
C8: cell port map (clk,X,Y,RAM(8),RGB8,A8,RG0,E8);
C9: cell port map (clk,X,Y,RAM(9),RGB9,A9,RG0,E9);
C10: cell port map (clk,X,Y,RAM(10),RGB10,A10,RG0,E10);
C11: cell port map (clk,X,Y,RAM(11),RGB11,A11,RG0,E11);
C12: cell port map (clk,X,Y,RAM(12),RGB12,A12,RG0,E12);
C13: cell port map (clk,X,Y,RAM(13),RGB13,A13,RG0,E13);
C14: cell port map (clk,X,Y,RAM(14),RGB14,A14,RG0,E14);
C15: cell port map (clk,X,Y,RAM(15),RGB15,A15,RG0,E15);
en <= E0 or E1 or E2 or E3 or E4 or E5 or E6 or E7 or E8 or E9 or E10 or E11 or E12 or E13 or E14 or E15;
ad <= A0 when E0 = '1' else 
	  A1 when E1 = '1' else
	  A2 when E2 = '1' else
	  A3 when E3 = '1' else
	  A4 when E4 = '1' else
	  A5 when E5 = '1' else 
	  A6 when E6 = '1' else
	  A7 when E7 = '1' else
	  A8 when E8 = '1' else
	  A9 when E9 = '1' else
	  A10 when E10 = '1' else 
	  A11 when E11 = '1' else
	  A12 when E12 = '1' else
	  A13 when E13 = '1' else
	  A14 when E14 = '1' else
		A15;
	

RCUM <= RGB0(29 downto 20)+
RGB1(29 downto 20)+
RGB2(29 downto 20)+
RGB3(29 downto 20)+
RGB4(29 downto 20)+
RGB5(29 downto 20)+
RGB6(29 downto 20)+
RGB7(29 downto 20)+
RGB8(29 downto 20)+
RGB9(29 downto 20)+
RGB10(29 downto 20)+
RGB11(29 downto 20)+
RGB12(29 downto 20)+
RGB13(29 downto 20)+
RGB14(29 downto 20)+
RGB15(29 downto 20);



GCUM <= RGB0(19 downto 10)+
RGB1(19 downto 10)+
RGB2(19 downto 10)+
RGB3(19 downto 10)+
RGB4(19 downto 10)+
RGB5(19 downto 10)+
RGB6(19 downto 10)+
RGB7(19 downto 10)+
RGB8(19 downto 10)+
RGB9(19 downto 10)+
RGB10(19 downto 10)+
RGB11(19 downto 10)+
RGB12(19 downto 10)+
RGB13(19 downto 10)+
RGB14(19 downto 10)+
RGB15(19 downto 10);


BCUM <= RGB0(9 downto 0)+
RGB1(9 downto 0)+
RGB2(9 downto 0)+
RGB3(9 downto 0)+
RGB4(9 downto 0)+
RGB5(9 downto 0)+
RGB6(9 downto 0)+
RGB7(9 downto 0)+
RGB8(9 downto 0)+
RGB9(9 downto 0)+
RGB10(9 downto 0)+
RGB11(9 downto 0)+
RGB12(9 downto 0)+
RGB13(9 downto 0)+
RGB14(9 downto 0)+
RGB15(9 downto 0);





 -- CLK GENERATION
 process (clkin)
  begin
    if rising_edge(clkin) then
      clk <= not clk;
    end if;
  end process;
  
  
  -- MEMORY MAPPED INTERFACE
  --------------------------
  -- The memory is made of 32 CELLS of 12 bits, the 3 MSB are the sprites
  -- number to be displayed and the 9 LSB tell the Y position of the sprite 
  --
process (clkin)
  begin
    if rising_edge(clkin) then
      if reset_n = '0' then
        readdata <= (others => '0');
      else
        if chipselect = '1' then
          if read = '1' then
            readdata <= x"0" & RAM(to_integer(ram_address));
          elsif write = '1' then
            RAM(to_integer(ram_address)) <= writedata(11 downto 0);
          end if;
        end if;
		end if;
  end if;
end process;



  HCounter : process (clk)
  begin
    if rising_edge(clk) then      
      if reset = '1' then
        Hcount <= (others => '0');
      elsif EndOfLine = '1' then
        Hcount <= (others => '0');
      else
        Hcount <= Hcount + 1;
      end if;      
    end if;
  end process HCounter;

  EndOfLine <= '1' when Hcount = HTOTAL - 1 else '0';
  
  VCounter: process (clk)
  begin
    if rising_edge(clk) then      
      if reset = '1' then
        Vcount <= (others => '0');
      elsif EndOfLine = '1' then
        if EndOfField = '1' then
          Vcount <= (others => '0');
        else
          Vcount <= Vcount + 1;
        end if;
      end if;
    end if;
  end process VCounter;

  EndOfField <= '1' when Vcount = VTOTAL - 1 else '0';

  -- State machines to generate HSYNC, VSYNC, HBLANK, and VBLANK

  HSyncGen : process (clk)
  begin
    if rising_edge(clk) then     
      if reset = '1' or EndOfLine = '1' then
        vga_hsync <= '1';
      elsif Hcount = HSYNC - 1 then
        vga_hsync <= '0';
      end if;
    end if;
  end process HSyncGen;
  
  HBlankGen : process (clk)
  begin
    if rising_edge(clk) then
      if reset = '1' then
        vga_hblank <= '1';
      elsif Hcount = HSYNC + HBACK_PORCH then
        vga_hblank <= '0';
      elsif Hcount = HSYNC + HBACK_PORCH + HACTIVE then
        vga_hblank <= '1';
      end if;      
    end if;
  end process HBlankGen;

  VSyncGen : process (clk)
  begin
    if rising_edge(clk) then
      if reset = '1' then
        vga_vsync <= '1';
      elsif EndOfLine ='1' then
        if EndOfField = '1' then
          vga_vsync <= '1';
        elsif Vcount = VSYNC - 1 then
          vga_vsync <= '0';
        end if;
      end if;      
    end if;
  end process VSyncGen;

  VBlankGen : process (clk)
  begin
    if rising_edge(clk) then    
      if reset = '1' then
        vga_vblank <= '1';
      elsif EndOfLine = '1' then
        if Vcount = VSYNC + VBACK_PORCH - 1 then
          vga_vblank <= '0';
        elsif Vcount = VSYNC + VBACK_PORCH + VACTIVE - 1 then
          vga_vblank <= '1';
        end if;
      end if;
    end if;
  end process VBlankGen;

  

  -- Registered video signals going to the video DAC

  VideoOut: process (clk, reset)
  begin

    if reset = '1' then
      VGA_R <= "0000000000";
      VGA_G <= "0000000000";
      VGA_B <= "0000000000";
	elsif clk'event and clk = '1' then
     if vga_hblank = '0' and vga_vblank ='0' then
        VGA_R <= RCUM;
        VGA_G <= GCUM;
        VGA_B <= BCUM;
     else
        VGA_R <= "0000000000";
        VGA_G <= "0000000000";
        VGA_B <= "0000000000";    
      end if;
 end if;
  end process VideoOut;

  VGA_CLK <= clk;
  VGA_HS <= not vga_hsync;
  VGA_VS <= not vga_vsync;
  VGA_SYNC <= '0';
  VGA_BLANK <= not (vga_hsync or vga_vsync);

end rtl;
