library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;

entity fmOperator is
port (
    clk : in std_logic;  
    reset_n : in std_logic;
	mute    : in std_logic;
	-- Oscillator select:
	--	0 = sine
	--  1 = saw
	--  2 = square
	--  3 = white noise (random)
	osc_select   : in std_logic_vector(1 downto 0);

	-- Oscillator period
	osc_freq   : in unsigned(15 downto 0);
	
	-- Modulator
	modulator  : in unsigned(23 downto 0);
	mod_factor : in unsigned(4 downto 0);
  
	-- Sample out
	fm_out   : out std_logic_vector(23 downto 0)
  );
end fmOperator;

architecture rtl of fmOperator is     

    signal sin_out     : unsigned(23 downto 0);
	signal saw_out     : unsigned(23 downto 0);
	signal random_out  : unsigned(23 downto 0);
	signal square_out  : unsigned(23 downto 0);
	
	signal pre_out     : unsigned(23 downto 0);
	
    signal sin_counter : unsigned(8 downto 0);    
	signal square_sel  : unsigned(0 downto 0);
	
	signal freq_counter: unsigned(19 downto 0);
	signal conversion  : unsigned(19 downto 0) := x"17D78"; 
	--signal conversion  : unsigned(19 downto 0) := x"0BEBC"; 
	signal temp		   : unsigned(19 downto 0);
begin
  
	-- Generate wave form
	with osc_select select pre_out <=
		sin_out when "00",
		saw_out when "01",
		square_out when "10",
		random_out when "11",
		sin_out when others;
    
	fm_out <= std_logic_vector(pre_out) when mute = '0' else x"000000";
	
	saw_out <= "000" & sin_counter & x"000";
	
	square_out <= x"011111" when square_sel = "1" else x"000000";
	
	temp <= conversion / (osc_freq+(shift_right(modulator, TO_INTEGER(mod_factor))));

	process(clk) 
	  begin
		if rising_edge(clk) then
			if reset_n = '0' then
				sin_counter <= (others => '0');
				freq_counter <= (others => '0');
			else
				if freq_counter = temp then
					freq_counter <= (others => '0');
					sin_counter <= sin_counter + 1;
					
					square_sel <= not square_sel;
				else
					freq_counter <= freq_counter + 1;
				end if;	
				
			end if;
		end if;
	end process;

--256 sin samples, one complete oscillation
with sin_counter select sin_out <=
X"000000" when "000000000",
X"003243" when "000000001",
X"006485" when "000000010",
X"0096c3" when "000000011",
X"00c8fb" when "000000100",
X"00fb2b" when "000000101",
X"012d51" when "000000110",
X"015f6c" when "000000111",
X"01917a" when "000001000",
X"01c378" when "000001001",
X"01f564" when "000001010",
X"02273d" when "000001011",
X"025901" when "000001100",
X"028aae" when "000001101",
X"02bc42" when "000001110",
X"02edbb" when "000001111",
X"031f16" when "000010000",
X"035053" when "000010001",
X"038170" when "000010010",
X"03b269" when "000010011",
X"03e33e" when "000010100",
X"0413ed" when "000010101",
X"044474" when "000010110",
X"0474d0" when "000010111",
X"04a501" when "000011000",
X"04d503" when "000011001",
X"0504d6" when "000011010",
X"053478" when "000011011",
X"0563e6" when "000011100",
X"05931f" when "000011101",
X"05c220" when "000011110",
X"05f0e9" when "000011111",
X"061f78" when "000100000",
X"064dca" when "000100001",
X"067bdd" when "000100010",
X"06a9b1" when "000100011",
X"06d743" when "000100100",
X"070492" when "000100101",
X"07319b" when "000100110",
X"075e5d" when "000100111",
X"078ad6" when "000101000",
X"07b705" when "000101001",
X"07e2e8" when "000101010",
X"080e7d" when "000101011",
X"0839c3" when "000101100",
X"0864b7" when "000101101",
X"088f59" when "000101110",
X"08b9a6" when "000101111",
X"08e39d" when "000110000",
X"090d3c" when "000110001",
X"093682" when "000110010",
X"095f6c" when "000110011",
X"0987fb" when "000110100",
X"09b02b" when "000110101",
X"09d7fc" when "000110110",
X"09ff6c" when "000110111",
X"0a2678" when "000111000",
X"0a4d21" when "000111001",
X"0a7364" when "000111010",
X"0a9940" when "000111011",
X"0abeb3" when "000111100",
X"0ae3bd" when "000111101",
X"0b085a" when "000111110",
X"0b2c8b" when "000111111",
X"0b504e" when "001000000",
X"0b73a1" when "001000001",
X"0b9683" when "001000010",
X"0bb8f2" when "001000011",
X"0bdaee" when "001000100",
X"0bfc75" when "001000101",
X"0c1d86" when "001000110",
X"0c3e1f" when "001000111",
X"0c5e3f" when "001001000",
X"0c7de5" when "001001001",
X"0c9d10" when "001001010",
X"0cbbbe" when "001001011",
X"0cd9ef" when "001001100",
X"0cf7a1" when "001001101",
X"0d14d2" when "001001110",
X"0d3183" when "001001111",
X"0d4db2" when "001010000",
X"0d695d" when "001010001",
X"0d8484" when "001010010",
X"0d9f25" when "001010011",
X"0db940" when "001010100",
X"0dd2d4" when "001010101",
X"0debdf" when "001010110",
X"0e0461" when "001010111",
X"0e1c58" when "001011000",
X"0e33c4" when "001011001",
X"0e4aa4" when "001011010",
X"0e60f7" when "001011011",
X"0e76bc" when "001011100",
X"0e8bf2" when "001011101",
X"0ea099" when "001011110",
X"0eb4af" when "001011111",
X"0ec834" when "001100000",
X"0edb28" when "001100001",
X"0eed88" when "001100010",
X"0eff56" when "001100011",
X"0f108f" when "001100100",
X"0f2134" when "001100101",
X"0f3143" when "001100110",
X"0f40bc" when "001100111",
X"0f4f9f" when "001101000",
X"0f5deb" when "001101001",
X"0f6b9f" when "001101010",
X"0f78bb" when "001101011",
X"0f853e" when "001101100",
X"0f9128" when "001101101",
X"0f9c78" when "001101110",
X"0fa72f" when "001101111",
X"0fb14a" when "001110000",
X"0fbacb" when "001110001",
X"0fc3b1" when "001110010",
X"0fcbfb" when "001110011",
X"0fd3a9" when "001110100",
X"0fdabb" when "001110101",
X"0fe131" when "001110110",
X"0fe70a" when "001110111",
X"0fec45" when "001111000",
X"0ff0e4" when "001111001",
X"0ff4e5" when "001111010",
X"0ff849" when "001111011",
X"0ffb0f" when "001111100",
X"0ffd38" when "001111101",
X"0ffec3" when "001111110",
X"0fffb0" when "001111111",
X"0fffff" when "010000000",
X"0fffb0" when "010000001",
X"0ffec3" when "010000010",
X"0ffd38" when "010000011",
X"0ffb0f" when "010000100",
X"0ff849" when "010000101",
X"0ff4e5" when "010000110",
X"0ff0e4" when "010000111",
X"0fec45" when "010001000",
X"0fe70a" when "010001001",
X"0fe131" when "010001010",
X"0fdabb" when "010001011",
X"0fd3a9" when "010001100",
X"0fcbfb" when "010001101",
X"0fc3b1" when "010001110",
X"0fbacb" when "010001111",
X"0fb14a" when "010010000",
X"0fa72f" when "010010001",
X"0f9c78" when "010010010",
X"0f9128" when "010010011",
X"0f853e" when "010010100",
X"0f78bb" when "010010101",
X"0f6b9f" when "010010110",
X"0f5deb" when "010010111",
X"0f4f9f" when "010011000",
X"0f40bc" when "010011001",
X"0f3143" when "010011010",
X"0f2134" when "010011011",
X"0f108f" when "010011100",
X"0eff56" when "010011101",
X"0eed88" when "010011110",
X"0edb28" when "010011111",
X"0ec834" when "010100000",
X"0eb4af" when "010100001",
X"0ea099" when "010100010",
X"0e8bf2" when "010100011",
X"0e76bc" when "010100100",
X"0e60f7" when "010100101",
X"0e4aa4" when "010100110",
X"0e33c4" when "010100111",
X"0e1c58" when "010101000",
X"0e0461" when "010101001",
X"0debdf" when "010101010",
X"0dd2d4" when "010101011",
X"0db940" when "010101100",
X"0d9f25" when "010101101",
X"0d8484" when "010101110",
X"0d695d" when "010101111",
X"0d4db2" when "010110000",
X"0d3183" when "010110001",
X"0d14d2" when "010110010",
X"0cf7a1" when "010110011",
X"0cd9ef" when "010110100",
X"0cbbbe" when "010110101",
X"0c9d10" when "010110110",
X"0c7de5" when "010110111",
X"0c5e3f" when "010111000",
X"0c3e1f" when "010111001",
X"0c1d86" when "010111010",
X"0bfc75" when "010111011",
X"0bdaee" when "010111100",
X"0bb8f2" when "010111101",
X"0b9683" when "010111110",
X"0b73a1" when "010111111",
X"0b504e" when "011000000",
X"0b2c8b" when "011000001",
X"0b085a" when "011000010",
X"0ae3bd" when "011000011",
X"0abeb3" when "011000100",
X"0a9940" when "011000101",
X"0a7364" when "011000110",
X"0a4d21" when "011000111",
X"0a2678" when "011001000",
X"09ff6c" when "011001001",
X"09d7fc" when "011001010",
X"09b02b" when "011001011",
X"0987fb" when "011001100",
X"095f6c" when "011001101",
X"093682" when "011001110",
X"090d3c" when "011001111",
X"08e39d" when "011010000",
X"08b9a6" when "011010001",
X"088f59" when "011010010",
X"0864b7" when "011010011",
X"0839c3" when "011010100",
X"080e7d" when "011010101",
X"07e2e8" when "011010110",
X"07b705" when "011010111",
X"078ad6" when "011011000",
X"075e5d" when "011011001",
X"07319b" when "011011010",
X"070492" when "011011011",
X"06d743" when "011011100",
X"06a9b1" when "011011101",
X"067bdd" when "011011110",
X"064dca" when "011011111",
X"061f78" when "011100000",
X"05f0e9" when "011100001",
X"05c220" when "011100010",
X"05931f" when "011100011",
X"0563e6" when "011100100",
X"053478" when "011100101",
X"0504d6" when "011100110",
X"04d503" when "011100111",
X"04a501" when "011101000",
X"0474d0" when "011101001",
X"044474" when "011101010",
X"0413ed" when "011101011",
X"03e33e" when "011101100",
X"03b269" when "011101101",
X"038170" when "011101110",
X"035053" when "011101111",
X"031f16" when "011110000",
X"02edbb" when "011110001",
X"02bc42" when "011110010",
X"028aae" when "011110011",
X"025901" when "011110100",
X"02273d" when "011110101",
X"01f564" when "011110110",
X"01c378" when "011110111",
X"01917a" when "011111000",
X"015f6c" when "011111001",
X"012d51" when "011111010",
X"00fb2b" when "011111011",
X"00c8fb" when "011111100",
X"0096c3" when "011111101",
X"006485" when "011111110",
X"003243" when "011111111",
X"000000" when "100000000",
X"ffcdbd" when "100000001",
X"ff9b7b" when "100000010",
X"ff693d" when "100000011",
X"ff3705" when "100000100",
X"ff04d5" when "100000101",
X"fed2af" when "100000110",
X"fea094" when "100000111",
X"fe6e86" when "100001000",
X"fe3c88" when "100001001",
X"fe0a9c" when "100001010",
X"fdd8c3" when "100001011",
X"fda6ff" when "100001100",
X"fd7552" when "100001101",
X"fd43be" when "100001110",
X"fd1245" when "100001111",
X"fce0ea" when "100010000",
X"fcafad" when "100010001",
X"fc7e90" when "100010010",
X"fc4d97" when "100010011",
X"fc1cc2" when "100010100",
X"fbec13" when "100010101",
X"fbbb8c" when "100010110",
X"fb8b30" when "100010111",
X"fb5aff" when "100011000",
X"fb2afd" when "100011001",
X"fafb2a" when "100011010",
X"facb88" when "100011011",
X"fa9c1a" when "100011100",
X"fa6ce1" when "100011101",
X"fa3de0" when "100011110",
X"fa0f17" when "100011111",
X"f9e088" when "100100000",
X"f9b236" when "100100001",
X"f98423" when "100100010",
X"f9564f" when "100100011",
X"f928bd" when "100100100",
X"f8fb6e" when "100100101",
X"f8ce65" when "100100110",
X"f8a1a3" when "100100111",
X"f8752a" when "100101000",
X"f848fb" when "100101001",
X"f81d18" when "100101010",
X"f7f183" when "100101011",
X"f7c63d" when "100101100",
X"f79b49" when "100101101",
X"f770a7" when "100101110",
X"f7465a" when "100101111",
X"f71c63" when "100110000",
X"f6f2c4" when "100110001",
X"f6c97e" when "100110010",
X"f6a094" when "100110011",
X"f67805" when "100110100",
X"f64fd5" when "100110101",
X"f62804" when "100110110",
X"f60094" when "100110111",
X"f5d988" when "100111000",
X"f5b2df" when "100111001",
X"f58c9c" when "100111010",
X"f566c0" when "100111011",
X"f5414d" when "100111100",
X"f51c43" when "100111101",
X"f4f7a6" when "100111110",
X"f4d375" when "100111111",
X"f4afb2" when "101000000",
X"f48c5f" when "101000001",
X"f4697d" when "101000010",
X"f4470e" when "101000011",
X"f42512" when "101000100",
X"f4038b" when "101000101",
X"f3e27a" when "101000110",
X"f3c1e1" when "101000111",
X"f3a1c1" when "101001000",
X"f3821b" when "101001001",
X"f362f0" when "101001010",
X"f34442" when "101001011",
X"f32611" when "101001100",
X"f3085f" when "101001101",
X"f2eb2e" when "101001110",
X"f2ce7d" when "101001111",
X"f2b24e" when "101010000",
X"f296a3" when "101010001",
X"f27b7c" when "101010010",
X"f260db" when "101010011",
X"f246c0" when "101010100",
X"f22d2c" when "101010101",
X"f21421" when "101010110",
X"f1fb9f" when "101010111",
X"f1e3a8" when "101011000",
X"f1cc3c" when "101011001",
X"f1b55c" when "101011010",
X"f19f09" when "101011011",
X"f18944" when "101011100",
X"f1740e" when "101011101",
X"f15f67" when "101011110",
X"f14b51" when "101011111",
X"f137cc" when "101100000",
X"f124d8" when "101100001",
X"f11278" when "101100010",
X"f100aa" when "101100011",
X"f0ef71" when "101100100",
X"f0decc" when "101100101",
X"f0cebd" when "101100110",
X"f0bf44" when "101100111",
X"f0b061" when "101101000",
X"f0a215" when "101101001",
X"f09461" when "101101010",
X"f08745" when "101101011",
X"f07ac2" when "101101100",
X"f06ed8" when "101101101",
X"f06388" when "101101110",
X"f058d1" when "101101111",
X"f04eb6" when "101110000",
X"f04535" when "101110001",
X"f03c4f" when "101110010",
X"f03405" when "101110011",
X"f02c57" when "101110100",
X"f02545" when "101110101",
X"f01ecf" when "101110110",
X"f018f6" when "101110111",
X"f013bb" when "101111000",
X"f00f1c" when "101111001",
X"f00b1b" when "101111010",
X"f007b7" when "101111011",
X"f004f1" when "101111100",
X"f002c8" when "101111101",
X"f0013d" when "101111110",
X"f00050" when "101111111",
X"f00001" when "110000000",
X"f00050" when "110000001",
X"f0013d" when "110000010",
X"f002c8" when "110000011",
X"f004f1" when "110000100",
X"f007b7" when "110000101",
X"f00b1b" when "110000110",
X"f00f1c" when "110000111",
X"f013bb" when "110001000",
X"f018f6" when "110001001",
X"f01ecf" when "110001010",
X"f02545" when "110001011",
X"f02c57" when "110001100",
X"f03405" when "110001101",
X"f03c4f" when "110001110",
X"f04535" when "110001111",
X"f04eb6" when "110010000",
X"f058d1" when "110010001",
X"f06388" when "110010010",
X"f06ed8" when "110010011",
X"f07ac2" when "110010100",
X"f08745" when "110010101",
X"f09461" when "110010110",
X"f0a215" when "110010111",
X"f0b061" when "110011000",
X"f0bf44" when "110011001",
X"f0cebd" when "110011010",
X"f0decc" when "110011011",
X"f0ef71" when "110011100",
X"f100aa" when "110011101",
X"f11278" when "110011110",
X"f124d8" when "110011111",
X"f137cc" when "110100000",
X"f14b51" when "110100001",
X"f15f67" when "110100010",
X"f1740e" when "110100011",
X"f18944" when "110100100",
X"f19f09" when "110100101",
X"f1b55c" when "110100110",
X"f1cc3c" when "110100111",
X"f1e3a8" when "110101000",
X"f1fb9f" when "110101001",
X"f21421" when "110101010",
X"f22d2c" when "110101011",
X"f246c0" when "110101100",
X"f260db" when "110101101",
X"f27b7c" when "110101110",
X"f296a3" when "110101111",
X"f2b24e" when "110110000",
X"f2ce7d" when "110110001",
X"f2eb2e" when "110110010",
X"f3085f" when "110110011",
X"f32611" when "110110100",
X"f34442" when "110110101",
X"f362f0" when "110110110",
X"f3821b" when "110110111",
X"f3a1c1" when "110111000",
X"f3c1e1" when "110111001",
X"f3e27a" when "110111010",
X"f4038b" when "110111011",
X"f42512" when "110111100",
X"f4470e" when "110111101",
X"f4697d" when "110111110",
X"f48c5f" when "110111111",
X"f4afb2" when "111000000",
X"f4d375" when "111000001",
X"f4f7a6" when "111000010",
X"f51c43" when "111000011",
X"f5414d" when "111000100",
X"f566c0" when "111000101",
X"f58c9c" when "111000110",
X"f5b2df" when "111000111",
X"f5d988" when "111001000",
X"f60094" when "111001001",
X"f62804" when "111001010",
X"f64fd5" when "111001011",
X"f67805" when "111001100",
X"f6a094" when "111001101",
X"f6c97e" when "111001110",
X"f6f2c4" when "111001111",
X"f71c63" when "111010000",
X"f7465a" when "111010001",
X"f770a7" when "111010010",
X"f79b49" when "111010011",
X"f7c63d" when "111010100",
X"f7f183" when "111010101",
X"f81d18" when "111010110",
X"f848fb" when "111010111",
X"f8752a" when "111011000",
X"f8a1a3" when "111011001",
X"f8ce65" when "111011010",
X"f8fb6e" when "111011011",
X"f928bd" when "111011100",
X"f9564f" when "111011101",
X"f98423" when "111011110",
X"f9b236" when "111011111",
X"f9e088" when "111100000",
X"fa0f17" when "111100001",
X"fa3de0" when "111100010",
X"fa6ce1" when "111100011",
X"fa9c1a" when "111100100",
X"facb88" when "111100101",
X"fafb2a" when "111100110",
X"fb2afd" when "111100111",
X"fb5aff" when "111101000",
X"fb8b30" when "111101001",
X"fbbb8c" when "111101010",
X"fbec13" when "111101011",
X"fc1cc2" when "111101100",
X"fc4d97" when "111101101",
X"fc7e90" when "111101110",
X"fcafad" when "111101111",
X"fce0ea" when "111110000",
X"fd1245" when "111110001",
X"fd43be" when "111110010",
X"fd7552" when "111110011",
X"fda6ff" when "111110100",
X"fdd8c3" when "111110101",
X"fe0a9c" when "111110110",
X"fe3c88" when "111110111",
X"fe6e86" when "111111000",
X"fea094" when "111111001",
X"fed2af" when "111111010",
X"ff04d5" when "111111011",
X"ff3705" when "111111100",
X"ff693d" when "111111101",
X"ff9b7b" when "111111110",
X"ffcdbd" when "111111111",
X"000000" when others;

with sin_counter(7 downto 0) select random_out <=
X"06b62d" when "00000000",
X"0904e0" when "00000001",
X"08c3ee" when "00000010",
X"03fcbc" when "00000011",
X"089923" when "00000100",
X"0bcfbd" when "00000101",
X"08e090" when "00000110",
X"01856f" when "00000111",
X"0dbae2" when "00001000",
X"031556" when "00001001",
X"0aa0e6" when "00001010",
X"058dce" when "00001011",
X"0fa46a" when "00001100",
X"06b14f" when "00001101",
X"09e375" when "00001110",
X"059a82" when "00001111",
X"0570c5" when "00010000",
X"004ec7" when "00010001",
X"08c0b0" when "00010010",
X"043d34" when "00010011",
X"0b6731" when "00010100",
X"0631cd" when "00010101",
X"0976b8" when "00010110",
X"0f35bc" when "00010111",
X"092b5d" when "00011000",
X"0e6163" when "00011001",
X"0e177f" when "00011010",
X"0240b1" when "00011011",
X"095157" when "00011100",
X"08c8cb" when "00011101",
X"0c00ed" when "00011110",
X"0bf40f" when "00011111",
X"0536da" when "00100000",
X"0f1aad" when "00100001",
X"0e7ab4" when "00100010",
X"017e85" when "00100011",
X"069781" when "00100100",
X"0bf4b4" when "00100101",
X"07a95c" when "00100110",
X"081214" when "00100111",
X"0d65a2" when "00101000",
X"0ff654" when "00101001",
X"095a3e" when "00101010",
X"0f3694" when "00101011",
X"04054a" when "00101100",
X"033140" when "00101101",
X"000000" when "00101110",
X"0634d0" when "00101111",
X"061b1f" when "00110000",
X"0b8e48" when "00110001",
X"018b33" when "00110010",
X"003ec7" when "00110011",
X"06b4ae" when "00110100",
X"0d752b" when "00110101",
X"0f9ec1" when "00110110",
X"0eedbb" when "00110111",
X"09a3cd" when "00111000",
X"0b3a0c" when "00111001",
X"00f3b0" when "00111010",
X"0922b3" when "00111011",
X"097aba" when "00111100",
X"098724" when "00111101",
X"0093f7" when "00111110",
X"0d4a01" when "00111111",
X"07aa84" when "01000000",
X"032d4d" when "01000001",
X"02e7f3" when "01000010",
X"01767e" when "01000011",
X"06d974" when "01000100",
X"0d7a0b" when "01000101",
X"086621" when "01000110",
X"01bd0b" when "01000111",
X"009b05" when "01001000",
X"0765b7" when "01001001",
X"0408bc" when "01001010",
X"05ad06" when "01001011",
X"07470f" when "01001100",
X"09a05f" when "01001101",
X"0080e5" when "01001110",
X"08b643" when "01001111",
X"02c1cd" when "01010000",
X"0bb80a" when "01010001",
X"0a72e7" when "01010010",
X"0ec141" when "01010011",
X"039a52" when "01010100",
X"0346f5" when "01010101",
X"07db4b" when "01010110",
X"063218" when "01010111",
X"02f919" when "01011000",
X"054c6e" when "01011001",
X"0d1049" when "01011010",
X"0b02e6" when "01011011",
X"0f6c46" when "01011100",
X"0786ed" when "01011101",
X"0ea73e" when "01011110",
X"0b9641" when "01011111",
X"0eb33b" when "01100000",
X"07da09" when "01100001",
X"04d9cf" when "01100010",
X"075653" when "01100011",
X"033f26" when "01100100",
X"0a7ee8" when "01100101",
X"00da44" when "01100110",
X"012949" when "01100111",
X"093222" when "01101000",
X"0cf1d4" when "01101001",
X"06dad9" when "01101010",
X"0c37ac" when "01101011",
X"023a8a" when "01101100",
X"0b437d" when "01101101",
X"085f7f" when "01101110",
X"00754b" when "01101111",
X"09bc28" when "01110000",
X"07a8af" when "01110001",
X"09c5d8" when "01110010",
X"05d148" when "01110011",
X"0b4cd2" when "01110100",
X"08d6ec" when "01110101",
X"03c2e7" when "01110110",
X"00e5ff" when "01110111",
X"049e20" when "01111000",
X"0af0b7" when "01111001",
X"0dea2c" when "01111010",
X"02754f" when "01111011",
X"0981d7" when "01111100",
X"087813" when "01111101",
X"0225c8" when "01111110",
X"0bf1f9" when "01111111",
X"0ccaa5" when "10000000",
X"0cf1dc" when "10000001",
X"025b3e" when "10000010",
X"06caa4" when "10000011",
X"0b5ed6" when "10000100",
X"0198a6" when "10000101",
X"094c81" when "10000110",
X"021f33" when "10000111",
X"0fe6a0" when "10001000",
X"05b1b2" when "10001001",
X"0618e0" when "10001010",
X"0afd06" when "10001011",
X"0d139d" when "10001100",
X"07caa3" when "10001101",
X"0e957e" when "10001110",
X"0e32cc" when "10001111",
X"02931e" when "10010000",
X"0b5a8c" when "10010001",
X"0808e9" when "10010010",
X"084078" when "10010011",
X"0550f7" when "10010100",
X"0ef503" when "10010101",
X"0fa978" when "10010110",
X"07dc13" when "10010111",
X"06c456" when "10011000",
X"0271ca" when "10011001",
X"0c9210" when "10011010",
X"01f7e6" when "10011011",
X"08945b" when "10011100",
X"0a1fb6" when "10011101",
X"0bb4ed" when "10011110",
X"0b2ac8" when "10011111",
X"09f5f9" when "10100000",
X"0c4d75" when "10100001",
X"03a80d" when "10100010",
X"09cf07" when "10100011",
X"0ab8d7" when "10100100",
X"0926d2" when "10100101",
X"01fae5" when "10100110",
X"0e20a5" when "10100111",
X"0d0cbf" when "10101000",
X"021910" when "10101001",
X"0e18f4" when "10101010",
X"021785" when "10101011",
X"0a9d41" when "10101100",
X"0da50a" when "10101101",
X"0e90e5" when "10101110",
X"01ae1d" when "10101111",
X"0bb533" when "10110000",
X"0eb9ab" when "10110001",
X"04ebcf" when "10110010",
X"05b960" when "10110011",
X"07af9e" when "10110100",
X"036564" when "10110101",
X"021437" when "10110110",
X"0f6a68" when "10110111",
X"0d0fc2" when "10111000",
X"0de13e" when "10111001",
X"09a080" when "10111010",
X"010696" when "10111011",
X"052dd2" when "10111100",
X"0cac2d" when "10111101",
X"05285c" when "10111110",
X"0956c0" when "10111111",
X"0b7a04" when "11000000",
X"0aa13a" when "11000001",
X"0a20dc" when "11000010",
X"0ac204" when "11000011",
X"02eb90" when "11000100",
X"06a72e" when "11000101",
X"0c8ed2" when "11000110",
X"04fcef" when "11000111",
X"05454e" when "11001000",
X"0bcc5c" when "11001001",
X"073308" when "11001010",
X"00a98e" when "11001011",
X"076d82" when "11001100",
X"0e5522" when "11001101",
X"06a80b" when "11001110",
X"0f390d" when "11001111",
X"0c2a7c" when "11010000",
X"0077ca" when "11010001",
X"003d14" when "11010010",
X"0ed382" when "11010011",
X"0b29b7" when "11010100",
X"023469" when "11010101",
X"0f2daf" when "11010110",
X"09fe87" when "11010111",
X"0d4aff" when "11011000",
X"02eb5d" when "11011001",
X"0b29da" when "11011010",
X"0d2f3f" when "11011011",
X"08bd54" when "11011100",
X"010900" when "11011101",
X"09f614" when "11011110",
X"096867" when "11011111",
X"0f8f26" when "11100000",
X"0c513f" when "11100001",
X"033f1f" when "11100010",
X"027919" when "11100011",
X"09e77c" when "11100100",
X"0a09b1" when "11100101",
X"0990fe" when "11100110",
X"0964d4" when "11100111",
X"0a31b5" when "11101000",
X"0774bd" when "11101001",
X"0db4e5" when "11101010",
X"0ff17c" when "11101011",
X"0d197f" when "11101100",
X"06c48f" when "11101101",
X"0a8d9c" when "11101110",
X"094bda" when "11101111",
X"0948a5" when "11110000",
X"04e05a" when "11110001",
X"06de15" when "11110010",
X"0d2755" when "11110011",
X"036c49" when "11110100",
X"0b0aa4" when "11110101",
X"0039c3" when "11110110",
X"0eb16f" when "11110111",
X"0a1b3c" when "11111000",
X"06ab26" when "11111001",
X"0eb252" when "11111010",
X"0f25b5" when "11111011",
X"0acb0a" when "11111100",
X"0b32eb" when "11111101",
X"03aa6c" when "11111110",
X"04c2b8" when "11111111",
X"000000" when others;



end architecture;


