library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;

entity de2_wm8731_audio is
port (
    clk : in std_logic;       --  Audio CODEC Chip Clock AUD_XCK (18.43 MHz)
    reset_n : in std_logic;
    test_mode : in std_logic;       --    Audio CODEC controller test mode
    audio_request : out std_logic;  --    Audio controller request new data
    data : in unsigned(15 downto 0);
    signal_start : in std_logic;
  
    -- Audio interface signals
    AUD_ADCLRCK  : out  std_logic;   --    Audio CODEC ADC LR Clock
    AUD_ADCDAT   : in   std_logic;   --    Audio CODEC ADC Data
    AUD_DACLRCK  : out  std_logic;   --    Audio CODEC DAC LR Clock
    AUD_DACDAT   : out  std_logic;   --    Audio CODEC DAC Data
    AUD_BCLK     : inout std_logic  --    Audio CODEC Bit-Stream Clock
  );
end  de2_wm8731_audio;

architecture rtl of de2_wm8731_audio is     

    signal lrck : std_logic;
    signal bclk : std_logic;
    signal xck  : std_logic;
    
    signal lrck_divider : unsigned(10 downto 0); 
    signal bclk_divider : unsigned(3 downto 0);
    
    signal set_bclk : std_logic;
    signal set_lrck : std_logic;
    signal clr_bclk : std_logic;
    signal lrck_lat : std_logic;
    
    signal shift_out : unsigned(15 downto 0);

    signal sin_out     : unsigned(15 downto 0);
    signal sin_counter : unsigned(11 downto 0);    

begin
  
    -- LRCK divider 
    -- Audio chip main clock is 25MHz / Sample rate 12KHz
    -- Divider is 25 MHz / 12KHz = 2083 ("10000010000")
    -- Left justify mode set by I2C controller


   
  process (clk)
  begin
    if rising_edge(clk) then
      if reset_n = '0' then 
        lrck_divider <= (others => '0');
      elsif lrck_divider = "01000010000"  then        -- "C0" minus 1
        lrck_divider <= "00000000000";
      else 
        lrck_divider <= lrck_divider + 1;
      end if;
    end if;   
  end process;

  process (clk)
  begin
    if rising_edge(clk) then      
      if reset_n = '0' then 
        bclk_divider <= (others => '0');
      elsif bclk_divider = X"B" or set_lrck = '1'  then  
        bclk_divider <= X"0";
      else 
        bclk_divider <= bclk_divider + 1;
      end if;
    end if;
  end process;

  set_lrck <= '1' when lrck_divider = "01000010000" else '0';
    
  process (clk)
  begin
    if rising_edge(clk) then
      if reset_n = '0' then
        lrck <= '0';
      elsif set_lrck = '1' then 
        lrck <= not lrck;
      end if;
    end if;
  end process;
    
  -- BCLK divider
  set_bclk <= '1' when bclk_divider(3 downto 0) = "0101" else '0';
  clr_bclk <= '1' when bclk_divider(3 downto 0) = "1011" else '0';
  
  process (clk)
  begin
    if rising_edge(clk) then
      if reset_n = '0' then
        bclk <= '0';
      elsif set_lrck = '1' or clr_bclk = '1' then
        bclk <= '0';
      elsif set_bclk = '1' then 
        bclk <= '1';
      end if;
    end if;
  end process;

  -- Audio data shift output
  process (clk)
  begin
    if rising_edge(clk) then
      if reset_n = '0' then
        shift_out <= (others => '0');
      elsif set_lrck = '1' then
        if test_mode = '1' then 
          shift_out <= sin_out;
        else 
          shift_out <= data;
        end if;
      elsif clr_bclk = '1' then 
        shift_out <= shift_out (14 downto 0) & '0';
      end if;
    end if;   
  end process;

    -- Audio outputs
    
    AUD_ADCLRCK  <= lrck;          
    AUD_DACLRCK  <= lrck;          
    AUD_DACDAT   <= shift_out(15); 
    AUD_BCLK     <= bclk;          

    -- Self test with Sin wave
    
    process(clk)      
    begin
      if rising_edge(clk) then
        if reset_n = '0' then 
            sin_counter <= (others => '0');
        elsif lrck_lat = '1' and lrck = '0'  then  
          if signal_start = '1' then 
             sin_counter <= X"000";	     
          elsif (sin_counter < X"494") then
             sin_counter <= sin_counter + 1;
          end if;
        end if;
                 
      end if;
    end process;

    process(clk)
    begin
      if rising_edge(clk) then
        lrck_lat <= lrck;
      end if;
    end process;

    process (clk) 
    begin
      if rising_edge(clk) then 
        if lrck_lat = '1' and lrck = '0' then
          audio_request <= '1';
        else 
          audio_request <= '0';
        end if;
      end if;
    end process;

  with sin_counter select sin_out <=
    X"f665" when X"000",
    X"ff3d" when X"001",
    X"ff79" when X"002",
    X"fb98" when X"003",
    X"f9bf" when X"004",
    X"fad4" when X"005",
    X"fb0e" when X"006",
    X"fef8" when X"007",
    X"f94f" when X"008",
    X"fe3e" when X"009",
    X"f9f3" when X"00a",
    X"ed68" when X"00b",
    X"078b" when X"00c",
    X"fa06" when X"00d",
    X"f762" when X"00e",
    X"f94c" when X"00f",
    X"f97c" when X"010",
    X"fcb9" when X"011",
    X"f611" when X"012",
    X"fae7" when X"013",
    X"feb5" when X"014",
    X"fa75" when X"015",
    X"fa4f" when X"016",
    X"fdd6" when X"017",
    X"fbcd" when X"018",
    X"fa54" when X"019",
    X"ff45" when X"01a",
    X"027a" when X"01b",
    X"02d9" when X"01c",
    X"fec1" when X"01d",
    X"ef44" when X"01e",
    X"fe5d" when X"01f",
    X"02a6" when X"020",
    X"f7e7" when X"021",
    X"07ed" when X"022",
    X"f8a1" when X"023",
    X"faff" when X"024",
    X"0e71" when X"025",
    X"fd75" when X"026",
    X"0590" when X"027",
    X"025e" when X"028",
    X"0265" when X"029",
    X"01ee" when X"02a",
    X"fdf9" when X"02b",
    X"0519" when X"02c",
    X"0041" when X"02d",
    X"0a7e" when X"02e",
    X"0cfb" when X"02f",
    X"0483" when X"030",
    X"09bd" when X"031",
    X"0ab0" when X"032",
    X"0596" when X"033",
    X"02b3" when X"034",
    X"0626" when X"035",
    X"0414" when X"036",
    X"0550" when X"037",
    X"0826" when X"038",
    X"0466" when X"039",
    X"04f9" when X"03a",
    X"0108" when X"03b",
    X"fdc2" when X"03c",
    X"044d" when X"03d",
    X"ff0b" when X"03e",
    X"f876" when X"03f",
    X"0e81" when X"040",
    X"07d4" when X"041",
    X"fb2a" when X"042",
    X"fe76" when X"043",
    X"03cb" when X"044",
    X"00aa" when X"045",
    X"fc77" when X"046",
    X"019d" when X"047",
    X"04f8" when X"048",
    X"04cf" when X"049",
    X"f80b" when X"04a",
    X"ff43" when X"04b",
    X"fda2" when X"04c",
    X"0299" when X"04d",
    X"06f5" when X"04e",
    X"f6fe" when X"04f",
    X"03c3" when X"050",
    X"0c53" when X"051",
    X"00ce" when X"052",
    X"04e7" when X"053",
    X"00ad" when X"054",
    X"ffea" when X"055",
    X"024e" when X"056",
    X"fbc8" when X"057",
    X"0302" when X"058",
    X"fe17" when X"059",
    X"0310" when X"05a",
    X"007d" when X"05b",
    X"febe" when X"05c",
    X"0758" when X"05d",
    X"00ea" when X"05e",
    X"fe29" when X"05f",
    X"feb9" when X"060",
    X"fee6" when X"061",
    X"fbfb" when X"062",
    X"fe4f" when X"063",
    X"fd3f" when X"064",
    X"fa1c" when X"065",
    X"fab2" when X"066",
    X"0111" when X"067",
    X"fe36" when X"068",
    X"0037" when X"069",
    X"f594" when X"06a",
    X"f344" when X"06b",
    X"0a86" when X"06c",
    X"03fe" when X"06d",
    X"fcd2" when X"06e",
    X"fb4f" when X"06f",
    X"f9b7" when X"070",
    X"ffb4" when X"071",
    X"f8e9" when X"072",
    X"fb96" when X"073",
    X"fd9a" when X"074",
    X"ffd9" when X"075",
    X"fe5d" when X"076",
    X"fc79" when X"077",
    X"0006" when X"078",
    X"f4cf" when X"079",
    X"f9ad" when X"07a",
    X"f64e" when X"07b",
    X"f2e7" when X"07c",
    X"fd85" when X"07d",
    X"ec8a" when X"07e",
    X"f806" when X"07f",
    X"ffd1" when X"080",
    X"f8a6" when X"081",
    X"ffcd" when X"082",
    X"f509" when X"083",
    X"fe64" when X"084",
    X"03ab" when X"085",
    X"f69e" when X"086",
    X"fc2d" when X"087",
    X"fa86" when X"088",
    X"fd1c" when X"089",
    X"fc79" when X"08a",
    X"fd95" when X"08b",
    X"fdd7" when X"08c",
    X"ff0a" when X"08d",
    X"fcb7" when X"08e",
    X"fb6b" when X"08f",
    X"feb2" when X"090",
    X"002d" when X"091",
    X"00dd" when X"092",
    X"fde3" when X"093",
    X"0407" when X"094",
    X"ffc0" when X"095",
    X"05df" when X"096",
    X"fcf9" when X"097",
    X"ffc6" when X"098",
    X"0091" when X"099",
    X"02e4" when X"09a",
    X"0323" when X"09b",
    X"f960" when X"09c",
    X"0187" when X"09d",
    X"f91c" when X"09e",
    X"fbb8" when X"09f",
    X"0c01" when X"0a0",
    X"0ca5" when X"0a1",
    X"0455" when X"0a2",
    X"0216" when X"0a3",
    X"073a" when X"0a4",
    X"0649" when X"0a5",
    X"05fc" when X"0a6",
    X"0a92" when X"0a7",
    X"004f" when X"0a8",
    X"0572" when X"0a9",
    X"03df" when X"0aa",
    X"f7b8" when X"0ab",
    X"076d" when X"0ac",
    X"ffdd" when X"0ad",
    X"ffc0" when X"0ae",
    X"0488" when X"0af",
    X"091f" when X"0b0",
    X"004d" when X"0b1",
    X"0096" when X"0b2",
    X"00cf" when X"0b3",
    X"ffb1" when X"0b4",
    X"057e" when X"0b5",
    X"ff76" when X"0b6",
    X"0370" when X"0b7",
    X"fc4c" when X"0b8",
    X"fa42" when X"0b9",
    X"ff61" when X"0ba",
    X"0330" when X"0bb",
    X"00d0" when X"0bc",
    X"003f" when X"0bd",
    X"0054" when X"0be",
    X"fea6" when X"0bf",
    X"ffa8" when X"0c0",
    X"fe7f" when X"0c1",
    X"fdfd" when X"0c2",
    X"0390" when X"0c3",
    X"0545" when X"0c4",
    X"fe70" when X"0c5",
    X"fd60" when X"0c6",
    X"ffbe" when X"0c7",
    X"003f" when X"0c8",
    X"fedc" when X"0c9",
    X"fe91" when X"0ca",
    X"ff52" when X"0cb",
    X"fe64" when X"0cc",
    X"0130" when X"0cd",
    X"f3cd" when X"0ce",
    X"f98c" when X"0cf",
    X"0ad3" when X"0d0",
    X"f88e" when X"0d1",
    X"fdbf" when X"0d2",
    X"fd02" when X"0d3",
    X"fc36" when X"0d4",
    X"f974" when X"0d5",
    X"f84a" when X"0d6",
    X"fec2" when X"0d7",
    X"f990" when X"0d8",
    X"f922" when X"0d9",
    X"fdae" when X"0da",
    X"f8aa" when X"0db",
    X"f91e" when X"0dc",
    X"00dd" when X"0dd",
    X"fcec" when X"0de",
    X"fe24" when X"0df",
    X"fdb2" when X"0e0",
    X"0064" when X"0e1",
    X"ffff" when X"0e2",
    X"f929" when X"0e3",
    X"fc3f" when X"0e4",
    X"fc23" when X"0e5",
    X"f542" when X"0e6",
    X"f6f1" when X"0e7",
    X"fa69" when X"0e8",
    X"f6f1" when X"0e9",
    X"f729" when X"0ea",
    X"f8dc" when X"0eb",
    X"fa30" when X"0ec",
    X"fda8" when X"0ed",
    X"e06d" when X"0ee",
    X"a82f" when X"0ef",
    X"9621" when X"0f0",
    X"9b20" when X"0f1",
    X"98c8" when X"0f2",
    X"9a00" when X"0f3",
    X"9944" when X"0f4",
    X"99ac" when X"0f5",
    X"9986" when X"0f6",
    X"9965" when X"0f7",
    X"99dc" when X"0f8",
    X"98c9" when X"0f9",
    X"9ac4" when X"0fa",
    X"975f" when X"0fb",
    X"9d59" when X"0fc",
    X"91b8" when X"0fd",
    X"b574" when X"0fe",
    X"2e1c" when X"0ff",
    X"69ea" when X"100",
    X"64cf" when X"101",
    X"65ce" when X"102",
    X"65bf" when X"103",
    X"6572" when X"104",
    X"6628" when X"105",
    X"6500" when X"106",
    X"66ff" when X"107",
    X"63cb" when X"108",
    X"68b0" when X"109",
    X"61a2" when X"10a",
    X"6ae4" when X"10b",
    X"5f29" when X"10c",
    X"6d24" when X"10d",
    X"5dad" when X"10e",
    X"b2a3" when X"10f",
    X"95be" when X"110",
    X"9dff" when X"111",
    X"958e" when X"112",
    X"9cce" when X"113",
    X"96bb" when X"114",
    X"9bb5" when X"115",
    X"9754" when X"116",
    X"9ba8" when X"117",
    X"9710" when X"118",
    X"9bee" when X"119",
    X"97a3" when X"11a",
    X"99c2" when X"11b",
    X"9dba" when X"11c",
    X"87df" when X"11d",
    X"3b98" when X"11e",
    X"7977" when X"11f",
    X"5903" when X"120",
    X"6e1b" when X"121",
    X"6027" when X"122",
    X"692a" when X"123",
    X"63d9" when X"124",
    X"669b" when X"125",
    X"6566" when X"126",
    X"65b4" when X"127",
    X"65aa" when X"128",
    X"65dd" when X"129",
    X"654b" when X"12a",
    X"662d" when X"12b",
    X"6664" when X"12c",
    X"5093" when X"12d",
    X"d815" when X"12e",
    X"8c28" when X"12f",
    X"9f23" when X"130",
    X"96dd" when X"131",
    X"9a96" when X"132",
    X"993e" when X"133",
    X"9979" when X"134",
    X"998a" when X"135",
    X"99ce" when X"136",
    X"9877" when X"137",
    X"9b77" when X"138",
    X"95f6" when X"139",
    X"9f77" when X"13a",
    X"8e9c" when X"13b",
    X"b9a5" when X"13c",
    X"3b93" when X"13d",
    X"66d1" when X"13e",
    X"65c0" when X"13f",
    X"65a8" when X"140",
    X"658a" when X"141",
    X"65c1" when X"142",
    X"65ed" when X"143",
    X"651c" when X"144",
    X"6715" when X"145",
    X"6384" when X"146",
    X"6922" when X"147",
    X"6112" when X"148",
    X"6b60" when X"149",
    X"5f07" when X"14a",
    X"6c44" when X"14b",
    X"6209" when X"14c",
    X"a1fa" when X"14d",
    X"90e9" when X"14e",
    X"a13c" when X"14f",
    X"9343" when X"150",
    X"9e66" when X"151",
    X"95b2" when X"152",
    X"9c4a" when X"153",
    X"971e" when X"154",
    X"9b91" when X"155",
    X"9772" when X"156",
    X"9b4d" when X"157",
    X"988a" when X"158",
    X"9891" when X"159",
    X"9f30" when X"15a",
    X"863b" when X"15b",
    X"2be0" when X"15c",
    X"778d" when X"15d",
    X"5a06" when X"15e",
    X"6d47" when X"15f",
    X"60d6" when X"160",
    X"6894" when X"161",
    X"6452" when X"162",
    X"6635" when X"163",
    X"65a4" when X"164",
    X"65ac" when X"165",
    X"6545" when X"166",
    X"6707" when X"167",
    X"62e4" when X"168",
    X"6aba" when X"169",
    X"5d51" when X"16a",
    X"6eff" when X"16b",
    X"fd1d" when X"16c",
    X"83b0" when X"16d",
    X"a2b8" when X"16e",
    X"955f" when X"16f",
    X"9b16" when X"170",
    X"9953" when X"171",
    X"98fe" when X"172",
    X"9a82" when X"173",
    X"97ec" when X"174",
    X"9bd6" when X"175",
    X"95c0" when X"176",
    X"9f0f" when X"177",
    X"914f" when X"178",
    X"a587" when X"179",
    X"8781" when X"17a",
    X"bd31" when X"17b",
    X"747e" when X"17c",
    X"6327" when X"17d",
    X"645a" when X"17e",
    X"6841" when X"17f",
    X"62cf" when X"180",
    X"684b" when X"181",
    X"63b6" when X"182",
    X"674b" when X"183",
    X"649e" when X"184",
    X"66ae" when X"185",
    X"64e4" when X"186",
    X"6682" when X"187",
    X"64a8" when X"188",
    X"668b" when X"189",
    X"6627" when X"18a",
    X"0d21" when X"18b",
    X"a2d3" when X"18c",
    X"95db" when X"18d",
    X"9bbf" when X"18e",
    X"981b" when X"18f",
    X"9a69" when X"190",
    X"98c3" when X"191",
    X"99f1" when X"192",
    X"9914" when X"193",
    X"99ce" when X"194",
    X"9948" when X"195",
    X"9996" when X"196",
    X"99d7" when X"197",
    X"98af" when X"198",
    X"9a24" when X"199",
    X"b2f6" when X"19a",
    X"3719" when X"19b",
    X"7402" when X"19c",
    X"5e55" when X"19d",
    X"69da" when X"19e",
    X"6357" when X"19f",
    X"6710" when X"1a0",
    X"6502" when X"1a1",
    X"65e4" when X"1a2",
    X"65fa" when X"1a3",
    X"6493" when X"1a4",
    X"683c" when X"1a5",
    X"60f7" when X"1a6",
    X"6dfd" when X"1a7",
    X"576e" when X"1a8",
    X"8000" when X"1a9",
    X"1a3b" when X"1aa",
    X"7f13" when X"1ab",
    X"a42b" when X"1ac",
    X"951d" when X"1ad",
    X"9ad3" when X"1ae",
    X"99cf" when X"1af",
    X"987a" when X"1b0",
    X"9aea" when X"1b1",
    X"97a7" when X"1b2",
    X"9be2" when X"1b3",
    X"95f9" when X"1b4",
    X"9e70" when X"1b5",
    X"9279" when X"1b6",
    X"a37e" when X"1b7",
    X"8b48" when X"1b8",
    X"b35d" when X"1b9",
    X"63fb" when X"1ba",
    X"655e" when X"1bb",
    X"6379" when X"1bc",
    X"6884" when X"1bd",
    X"62d9" when X"1be",
    X"67f3" when X"1bf",
    X"645d" when X"1c0",
    X"6641" when X"1c1",
    X"662e" when X"1c2",
    X"6480" when X"1c3",
    X"67a8" when X"1c4",
    X"635c" when X"1c5",
    X"67b0" when X"1c6",
    X"64c5" when X"1c7",
    X"6371" when X"1c8",
    X"67de" when X"1c9",
    X"bb19" when X"1ca",
    X"890a" when X"1cb",
    X"a45d" when X"1cc",
    X"922b" when X"1cd",
    X"9e53" when X"1ce",
    X"9660" when X"1cf",
    X"9b3d" when X"1d0",
    X"9887" when X"1d1",
    X"99f3" when X"1d2",
    X"995c" when X"1d3",
    X"9975" when X"1d4",
    X"99e3" when X"1d5",
    X"98ca" when X"1d6",
    X"9997" when X"1d7",
    X"b2fd" when X"1d8",
    X"3394" when X"1d9",
    X"73bb" when X"1da",
    X"5ed4" when X"1db",
    X"696b" when X"1dc",
    X"63af" when X"1dd",
    X"66b6" when X"1de",
    X"654f" when X"1df",
    X"65a8" when X"1e0",
    X"65f3" when X"1e1",
    X"652f" when X"1e2",
    X"66a1" when X"1e3",
    X"643f" when X"1e4",
    X"681d" when X"1e5",
    X"616e" when X"1e6",
    X"6e12" when X"1e7",
    X"494c" when X"1e8",
    X"cf14" when X"1e9",
    X"9604" when X"1ea",
    X"9a23" when X"1eb",
    X"9978" when X"1ec",
    X"9978" when X"1ed",
    X"99aa" when X"1ee",
    X"9932" when X"1ef",
    X"99ed" when X"1f0",
    X"98aa" when X"1f1",
    X"9aab" when X"1f2",
    X"979f" when X"1f3",
    X"9bff" when X"1f4",
    X"967c" when X"1f5",
    X"9d00" when X"1f6",
    X"96fb" when X"1f7",
    X"c80d" when X"1f8",
    X"6256" when X"1f9",
    X"6a8f" when X"1fa",
    X"60f0" when X"1fb",
    X"69a4" when X"1fc",
    X"6279" when X"1fd",
    X"6834" when X"1fe",
    X"63fc" when X"1ff",
    X"670c" when X"200",
    X"64c4" when X"201",
    X"6688" when X"202",
    X"6506" when X"203",
    X"6616" when X"204",
    X"65b8" when X"205",
    X"643a" when X"206",
    X"6c49" when X"207",
    X"178e" when X"208",
    X"a9d6" when X"209",
    X"95ee" when X"20a",
    X"9afe" when X"20b",
    X"98c9" when X"20c",
    X"99c9" when X"20d",
    X"995a" when X"20e",
    X"9973" when X"20f",
    X"999b" when X"210",
    X"9980" when X"211",
    X"9953" when X"212",
    X"9a2f" when X"213",
    X"97dd" when X"214",
    X"9d1b" when X"215",
    X"906e" when X"216",
    X"c927" when X"217",
    X"41a0" when X"218",
    X"6aa9" when X"219",
    X"6426" when X"21a",
    X"664a" when X"21b",
    X"6574" when X"21c",
    X"65bd" when X"21d",
    X"65b9" when X"21e",
    X"6587" when X"21f",
    X"660e" when X"220",
    X"652b" when X"221",
    X"66ad" when X"222",
    X"6450" when X"223",
    X"67a1" when X"224",
    X"630f" when X"225",
    X"66f5" when X"226",
    X"2fd5" when X"227",
    X"baf1" when X"228",
    X"997f" when X"229",
    X"9998" when X"22a",
    X"9942" when X"22b",
    X"99f6" when X"22c",
    X"990d" when X"22d",
    X"99d8" when X"22e",
    X"9910" when X"22f",
    X"99d7" when X"230",
    X"98f9" when X"231",
    X"9a0a" when X"232",
    X"98da" when X"233",
    X"9a6d" when X"234",
    X"989c" when X"235",
    X"99f2" when X"236",
    X"e489" when X"237",
    X"479c" when X"238",
    X"6597" when X"239",
    X"64fe" when X"23a",
    X"6633" when X"23b",
    X"654f" when X"23c",
    X"660b" when X"23d",
    X"6598" when X"23e",
    X"65cd" when X"23f",
    X"65bd" when X"240",
    X"65a5" when X"241",
    X"65b2" when X"242",
    X"65c0" when X"243",
    X"6544" when X"244",
    X"6767" when X"245",
    X"502e" when X"246",
    X"09b8" when X"247",
    X"b173" when X"248",
    X"9424" when X"249",
    X"9be0" when X"24a",
    X"9846" when X"24b",
    X"9a1a" when X"24c",
    X"9935" when X"24d",
    X"999c" when X"24e",
    X"998e" when X"24f",
    X"996e" when X"250",
    X"999b" when X"251",
    X"9947" when X"252",
    X"99e3" when X"253",
    X"9861" when X"254",
    X"a1ee" when X"255",
    X"d2da" when X"256",
    X"2170" when X"257",
    X"58de" when X"258",
    X"662a" when X"259",
    X"65a2" when X"25a",
    X"65a9" when X"25b",
    X"65ae" when X"25c",
    X"65ab" when X"25d",
    X"65d7" when X"25e",
    X"6583" when X"25f",
    X"662e" when X"260",
    X"6511" when X"261",
    X"66b2" when X"262",
    X"6451" when X"263",
    X"66bb" when X"264",
    X"5155" when X"265",
    X"1538" when X"266",
    X"d752" when X"267",
    X"9dbd" when X"268",
    X"9814" when X"269",
    X"9aee" when X"26a",
    X"9863" when X"26b",
    X"9a62" when X"26c",
    X"98c9" when X"26d",
    X"99e8" when X"26e",
    X"9920" when X"26f",
    X"99af" when X"270",
    X"9957" when X"271",
    X"99b5" when X"272",
    X"996d" when X"273",
    X"9954" when X"274",
    X"b15c" when X"275",
    X"e315" when X"276",
    X"1529" when X"277",
    X"46d9" when X"278",
    X"62f6" when X"279",
    X"65bc" when X"27a",
    X"65e0" when X"27b",
    X"659d" when X"27c",
    X"65d4" when X"27d",
    X"65a4" when X"27e",
    X"65bb" when X"27f",
    X"65a8" when X"280",
    X"659c" when X"281",
    X"65d9" when X"282",
    X"65e3" when X"283",
    X"5ac0" when X"284",
    X"2476" when X"285",
    X"ed94" when X"286",
    X"c5a3" when X"287",
    X"a1ab" when X"288",
    X"992d" when X"289",
    X"9975" when X"28a",
    X"998c" when X"28b",
    X"9987" when X"28c",
    X"9987" when X"28d",
    X"997f" when X"28e",
    X"997f" when X"28f",
    X"9968" when X"290",
    X"999f" when X"291",
    X"9905" when X"292",
    X"9d43" when X"293",
    X"b519" when X"294",
    X"d967" when X"295",
    X"fe33" when X"296",
    X"218c" when X"297",
    X"4010" when X"298",
    X"5a92" when X"299",
    X"6562" when X"29a",
    X"65bc" when X"29b",
    X"65af" when X"29c",
    X"65ca" when X"29d",
    X"65a8" when X"29e",
    X"65ed" when X"29f",
    X"6580" when X"2a0",
    X"659f" when X"2a1",
    X"5cf5" when X"2a2",
    X"4622" when X"2a3",
    X"2c34" when X"2a4",
    X"0f69" when X"2a5",
    X"f0a4" when X"2a6",
    X"d1d9" when X"2a7",
    X"b843" when X"2a8",
    X"a140" when X"2a9",
    X"98d4" when X"2aa",
    X"9a0d" when X"2ab",
    X"991c" when X"2ac",
    X"99b2" when X"2ad",
    X"995a" when X"2ae",
    X"9998" when X"2af",
    X"9979" when X"2b0",
    X"9995" when X"2b1",
    X"a09b" when X"2b2",
    X"b2e3" when X"2b3",
    X"c9ee" when X"2b4",
    X"e353" when X"2b5",
    X"fd27" when X"2b6",
    X"16a4" when X"2b7",
    X"2c62" when X"2b8",
    X"3e3e" when X"2b9",
    X"4cbc" when X"2ba",
    X"54bd" when X"2bb",
    X"5a52" when X"2bc",
    X"5c0c" when X"2bd",
    X"5c07" when X"2be",
    X"5a92" when X"2bf",
    X"4fbb" when X"2c0",
    X"4421" when X"2c1",
    X"379c" when X"2c2",
    X"25b8" when X"2c3",
    X"12e1" when X"2c4",
    X"fc53" when X"2c5",
    X"e6cc" when X"2c6",
    X"d3e5" when X"2c7",
    X"c12a" when X"2c8",
    X"b3e9" when X"2c9",
    X"a9ae" when X"2ca",
    X"a2c1" when X"2cb",
    X"a032" when X"2cc",
    X"9f21" when X"2cd",
    X"a037" when X"2ce",
    X"a48b" when X"2cf",
    X"aaa7" when X"2d0",
    X"b3c1" when X"2d1",
    X"bfc3" when X"2d2",
    X"cdd8" when X"2d3",
    X"dfa0" when X"2d4",
    X"f2b8" when X"2d5",
    X"050e" when X"2d6",
    X"16c8" when X"2d7",
    X"253b" when X"2d8",
    X"309b" when X"2d9",
    X"3a4e" when X"2da",
    X"4059" when X"2db",
    X"42fb" when X"2dc",
    X"44ef" when X"2dd",
    X"4410" when X"2de",
    X"4069" when X"2df",
    X"3a81" when X"2e0",
    X"3321" when X"2e1",
    X"2ac8" when X"2e2",
    X"1d82" when X"2e3",
    X"0ef4" when X"2e4",
    X"fe3f" when X"2e5",
    X"edd2" when X"2e6",
    X"ddcd" when X"2e7",
    X"d0dc" when X"2e8",
    X"c6f4" when X"2e9",
    X"be40" when X"2ea",
    X"b89a" when X"2eb",
    X"b3af" when X"2ec",
    X"b27a" when X"2ed",
    X"b409" when X"2ee",
    X"b580" when X"2ef",
    X"b9c0" when X"2f0",
    X"bffa" when X"2f1",
    X"c97e" when X"2f2",
    X"d56a" when X"2f3",
    X"e36e" when X"2f4",
    X"f256" when X"2f5",
    X"0118" when X"2f6",
    X"0f6d" when X"2f7",
    X"1c19" when X"2f8",
    X"2571" when X"2f9",
    X"2cd5" when X"2fa",
    X"34df" when X"2fb",
    X"3742" when X"2fc",
    X"3731" when X"2fd",
    X"36f5" when X"2fe",
    X"3407" when X"2ff",
    X"2eeb" when X"300",
    X"27c7" when X"301",
    X"1ea8" when X"302",
    X"1368" when X"303",
    X"0657" when X"304",
    X"f833" when X"305",
    X"eb5c" when X"306",
    X"e066" when X"307",
    X"d553" when X"308",
    X"cd8f" when X"309",
    X"c864" when X"30a",
    X"c368" when X"30b",
    X"c170" when X"30c",
    X"c1fa" when X"30d",
    X"c2a6" when X"30e",
    X"c3dd" when X"30f",
    X"c839" when X"310",
    X"cf5e" when X"311",
    X"d7ca" when X"312",
    X"e242" when X"313",
    X"ed48" when X"314",
    X"f903" when X"315",
    X"04d5" when X"316",
    X"0fc1" when X"317",
    X"17cc" when X"318",
    X"1eb3" when X"319",
    X"244e" when X"31a",
    X"27f2" when X"31b",
    X"2a33" when X"31c",
    X"2a5d" when X"31d",
    X"2967" when X"31e",
    X"27e1" when X"31f",
    X"24d2" when X"320",
    X"1ef9" when X"321",
    X"1802" when X"322",
    X"0e64" when X"323",
    X"05e5" when X"324",
    X"fb04" when X"325",
    X"f1de" when X"326",
    X"e8f9" when X"327",
    X"e0bb" when X"328",
    X"dc04" when X"329",
    X"d630" when X"32a",
    X"d3c0" when X"32b",
    X"d187" when X"32c",
    X"d181" when X"32d",
    X"d24a" when X"32e",
    X"d368" when X"32f",
    X"d687" when X"330",
    X"db56" when X"331",
    X"e121" when X"332",
    X"e8a2" when X"333",
    X"ef90" when X"334",
    X"f954" when X"335",
    X"049e" when X"336",
    X"0b40" when X"337",
    X"111b" when X"338",
    X"157a" when X"339",
    X"1941" when X"33a",
    X"1b10" when X"33b",
    X"1bd9" when X"33c",
    X"1c21" when X"33d",
    X"1bc8" when X"33e",
    X"19fd" when X"33f",
    X"166e" when X"340",
    X"1286" when X"341",
    X"0cd8" when X"342",
    X"082f" when X"343",
    X"0372" when X"344",
    X"fb41" when X"345",
    X"f53f" when X"346",
    X"f034" when X"347",
    X"eb51" when X"348",
    X"e8a6" when X"349",
    X"e6ad" when X"34a",
    X"e466" when X"34b",
    X"e414" when X"34c",
    X"e396" when X"34d",
    X"e36c" when X"34e",
    X"e51e" when X"34f",
    X"e793" when X"350",
    X"eb40" when X"351",
    X"ee1b" when X"352",
    X"f348" when X"353",
    X"f67c" when X"354",
    X"fad7" when X"355",
    X"ffc4" when X"356",
    X"0233" when X"357",
    X"05df" when X"358",
    X"0781" when X"359",
    X"0886" when X"35a",
    X"09af" when X"35b",
    X"09b1" when X"35c",
    X"0a65" when X"35d",
    X"08df" when X"35e",
    X"080e" when X"35f",
    X"0766" when X"360",
    X"0649" when X"361",
    X"0378" when X"362",
    X"017a" when X"363",
    X"ff78" when X"364",
    X"fc97" when X"365",
    X"fbc3" when X"366",
    X"f9b8" when X"367",
    X"f7de" when X"368",
    X"f801" when X"369",
    X"f737" when X"36a",
    X"f716" when X"36b",
    X"f780" when X"36c",
    X"f599" when X"36d",
    X"f789" when X"36e",
    X"f9c7" when X"36f",
    X"f95c" when X"370",
    X"f9ff" when X"371",
    X"fa81" when X"372",
    X"fa50" when X"373",
    X"fa8c" when X"374",
    X"fb39" when X"375",
    X"fb34" when X"376",
    X"fb3e" when X"377",
    X"fa70" when X"378",
    X"fafb" when X"379",
    X"fb53" when X"37a",
    X"fb21" when X"37b",
    X"fb3b" when X"37c",
    X"fb65" when X"37d",
    X"fc52" when X"37e",
    X"fb9d" when X"37f",
    X"fb4e" when X"380",
    X"fb08" when X"381",
    X"fa72" when X"382",
    X"fbfe" when X"383",
    X"fb37" when X"384",
    X"fbfa" when X"385",
    X"fb37" when X"386",
    X"fc62" when X"387",
    X"fc04" when X"388",
    X"fb5f" when X"389",
    X"fbdf" when X"38a",
    X"fb45" when X"38b",
    X"fc00" when X"38c",
    X"fb29" when X"38d",
    X"fc0c" when X"38e",
    X"fb3a" when X"38f",
    X"fb14" when X"390",
    X"fc01" when X"391",
    X"fb71" when X"392",
    X"fb04" when X"393",
    X"fbb8" when X"394",
    X"fc15" when X"395",
    X"fbf6" when X"396",
    X"fc09" when X"397",
    X"fbf9" when X"398",
    X"fc08" when X"399",
    X"fbfa" when X"39a",
    X"fc01" when X"39b",
    X"fcc7" when X"39c",
    X"fc37" when X"39d",
    X"fb39" when X"39e",
    X"fc13" when X"39f",
    X"fd79" when X"3a0",
    X"fb61" when X"3a1",
    X"fc59" when X"3a2",
    X"fd57" when X"3a3",
    X"fbe2" when X"3a4",
    X"fceb" when X"3a5",
    X"fbdb" when X"3a6",
    X"fc08" when X"3a7",
    X"fcc6" when X"3a8",
    X"fce2" when X"3a9",
    X"fc99" when X"3aa",
    X"fd46" when X"3ab",
    X"fb39" when X"3ac",
    X"fcbb" when X"3ad",
    X"fcfe" when X"3ae",
    X"fc4f" when X"3af",
    X"fdb0" when X"3b0",
    X"fccd" when X"3b1",
    X"fea0" when X"3b2",
    X"fd75" when X"3b3",
    X"fcc5" when X"3b4",
    X"fda5" when X"3b5",
    X"fcd9" when X"3b6",
    X"fd62" when X"3b7",
    X"fd4d" when X"3b8",
    X"fc6c" when X"3b9",
    X"fd74" when X"3ba",
    X"fd46" when X"3bb",
    X"fcc3" when X"3bc",
    X"fcc1" when X"3bd",
    X"fcee" when X"3be",
    X"fd9a" when X"3bf",
    X"fccd" when X"3c0",
    X"fcc4" when X"3c1",
    X"fcd7" when X"3c2",
    X"fcca" when X"3c3",
    X"fc69" when X"3c4",
    X"fb9a" when X"3c5",
    X"fd04" when X"3c6",
    X"fd6c" when X"3c7",
    X"fca5" when X"3c8",
    X"fcdd" when X"3c9",
    X"fcbe" when X"3ca",
    X"fbdc" when X"3cb",
    X"fc0f" when X"3cc",
    X"fe5d" when X"3cd",
    X"fd13" when X"3ce",
    X"fc9b" when X"3cf",
    X"fcf6" when X"3d0",
    X"fcad" when X"3d1",
    X"fce7" when X"3d2",
    X"fcb9" when X"3d3",
    X"fce3" when X"3d4",
    X"fca0" when X"3d5",
    X"fbdd" when X"3d6",
    X"fc15" when X"3d7",
    X"fbf5" when X"3d8",
    X"fbf5" when X"3d9",
    X"fce1" when X"3da",
    X"fbe3" when X"3db",
    X"fdd3" when X"3dc",
    X"fe2d" when X"3dd",
    X"fd5a" when X"3de",
    X"fd80" when X"3df",
    X"fc6e" when X"3e0",
    X"fd3d" when X"3e1",
    X"fd77" when X"3e2",
    X"fcbb" when X"3e3",
    X"fcd1" when X"3e4",
    X"fccd" when X"3e5",
    X"fccd" when X"3e6",
    X"fccb" when X"3e7",
    X"fcd6" when X"3e8",
    X"fcba" when X"3e9",
    X"fcf9" when X"3ea",
    X"fc47" when X"3eb",
    X"fc40" when X"3ec",
    X"fcec" when X"3ed",
    X"fcc8" when X"3ee",
    X"fcc6" when X"3ef",
    X"fcd8" when X"3f0",
    X"fcc8" when X"3f1",
    X"fbfb" when X"3f2",
    X"fcc9" when X"3f3",
    X"fc21" when X"3f4",
    X"fc1e" when X"3f5",
    X"fb48" when X"3f6",
    X"fc12" when X"3f7",
    X"fca5" when X"3f8",
    X"fc0c" when X"3f9",
    X"fce5" when X"3fa",
    X"fbbc" when X"3fb",
    X"fc2d" when X"3fc",
    X"fcc9" when X"3fd",
    X"fc07" when X"3fe",
    X"fbef" when X"3ff",
    X"fca8" when X"400",
    X"fcf4" when X"401",
    X"fc39" when X"402",
    X"fc3b" when X"403",
    X"fd33" when X"404",
    X"fc30" when X"405",
    X"fc77" when X"406",
    X"fdba" when X"407",
    X"fd94" when X"408",
    X"fcaa" when X"409",
    X"fbfb" when X"40a",
    X"fda6" when X"40b",
    X"fc06" when X"40c",
    X"fddd" when X"40d",
    X"fdff" when X"40e",
    X"fe07" when X"40f",
    X"fdc7" when X"410",
    X"fcca" when X"411",
    X"fdd0" when X"412",
    X"fd99" when X"413",
    X"fd74" when X"414",
    X"fcd9" when X"415",
    X"fd99" when X"416",
    X"fcc9" when X"417",
    X"fe37" when X"418",
    X"fde9" when X"419",
    X"fd76" when X"41a",
    X"fd9a" when X"41b",
    X"fdff" when X"41c",
    X"fe73" when X"41d",
    X"fea9" when X"41e",
    X"ff14" when X"41f",
    X"fd2a" when X"420",
    X"fcf3" when X"421",
    X"fe70" when X"422",
    X"fdaf" when X"423",
    X"fd5b" when X"424",
    X"feae" when X"425",
    X"fccd" when X"426",
    X"fe30" when X"427",
    X"fcb4" when X"428",
    X"fbb7" when X"429",
    X"ffe0" when X"42a",
    X"fe0c" when X"42b",
    X"fd13" when X"42c",
    X"fe6f" when X"42d",
    X"fd8e" when X"42e",
    X"fd99" when X"42f",
    X"fd93" when X"430",
    X"fdba" when X"431",
    X"fd44" when X"432",
    X"feb4" when X"433",
    X"fc99" when X"434",
    X"fd21" when X"435",
    X"fe17" when X"436",
    X"fcda" when X"437",
    X"feb2" when X"438",
    X"fd4d" when X"439",
    X"fdbf" when X"43a",
    X"fe66" when X"43b",
    X"fda7" when X"43c",
    X"fd88" when X"43d",
    X"fdc0" when X"43e",
    X"fced" when X"43f",
    X"fd31" when X"440",
    X"fdcc" when X"441",
    X"fd72" when X"442",
    X"fdc7" when X"443",
    X"fd52" when X"444",
    X"fcce" when X"445",
    X"fcc1" when X"446",
    X"fcd5" when X"447",
    X"fd9f" when X"448",
    X"fcce" when X"449",
    X"fd87" when X"44a",
    X"fcfb" when X"44b",
    X"fd4c" when X"44c",
    X"fd44" when X"44d",
    X"fcf3" when X"44e",
    X"fdd1" when X"44f",
    X"fd6d" when X"450",
    X"fc94" when X"451",
    X"fd03" when X"452",
    X"fca7" when X"453",
    X"fbea" when X"454",
    X"ff43" when X"455",
    X"fc46" when X"456",
    X"fde4" when X"457",
    X"fce9" when X"458",
    X"fcf5" when X"459",
    X"fe59" when X"45a",
    X"fc2f" when X"45b",
    X"fd1e" when X"45c",
    X"fcab" when X"45d",
    X"fcd5" when X"45e",
    X"fcd5" when X"45f",
    X"fd9a" when X"460",
    X"fd9b" when X"461",
    X"fda7" when X"462",
    X"fcf9" when X"463",
    X"fcc5" when X"464",
    X"fc73" when X"465",
    X"fbbd" when X"466",
    X"fc42" when X"467",
    X"fbb3" when X"468",
    X"fc73" when X"469",
    X"fc8b" when X"46a",
    X"fc30" when X"46b",
    X"fe6b" when X"46c",
    X"fce9" when X"46d",
    X"fcb0" when X"46e",
    X"fcd7" when X"46f",
    X"fd50" when X"470",
    X"fd61" when X"471",
    X"fc63" when X"472",
    X"fd73" when X"473",
    X"fd42" when X"474",
    X"fcea" when X"475",
    X"fdb8" when X"476",
    X"fd92" when X"477",
    X"fd9b" when X"478",
    X"fd99" when X"479",
    X"fda2" when X"47a",
    X"fd7b" when X"47b",
    X"fe55" when X"47c",
    X"fddf" when X"47d",
    X"fde1" when X"47e",
    X"fea9" when X"47f",
    X"fe0e" when X"480",
    X"fd74" when X"481",
    X"fdd0" when X"482",
    X"fe70" when X"483",
    X"fe66" when X"484",
    X"fd9d" when X"485",
    X"fcb1" when X"486",
    X"feeb" when X"487",
    X"feca" when X"488",
    X"fe29" when X"489",
    X"fe97" when X"48a",
    X"fe3d" when X"48b",
    X"fe90" when X"48c",
    X"fe3c" when X"48d",
    X"fe95" when X"48e",
    X"fe36" when X"48f",
    X"fe98" when X"490",
    X"ff2a" when X"491",
    X"fccd" when X"492",
    X"ff82" when X"493",
    X"0056" when X"494",


    X"0000" when others;      

end architecture;


