library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;

entity sprite_controller is
port(
	-- interface with CPU
	address   : in std_logic_vector(15 downto 0);-- higher 4 bit is sprite index, lower 6 bit is ROM number
	writedata : in unsigned(31 downto 0);-- bit 31 is IRQ switch, bit 30~16 is y value, bit 15~0 is x value
	write     : in std_logic;
	reset     : in std_logic;
	clk_50    : in std_logic;
	irq       : out std_logic;
	
	-- interface with VGA controller
    pixel : out unsigned(3 downto 0); -- current pixel code
    table : out unsigned(2 downto 0); -- color table index to use
	x, y  :  in unsigned(9 downto 0);
	y_blank: in std_logic;
	x_count: in unsigned(11 downto 0);
	
	-- interface to SRAM
	sram_addr   : out std_logic_vector(17 downto 0);
	sram_datain : in  std_logic_vector(15 downto 0)
	);
end sprite_controller;

architecture decode of sprite_controller is

----input buffers
signal irq_mask : std_logic := '0';
signal x_buf, y_buf : unsigned(9 downto 0);
signal sprite_index_buf : integer range 0 to 21;
signal rom_buf : integer range 0 to 127;

--store the sprite status
type sprite_pos_type is array(integer range 0 to 21) of unsigned(9 downto 0);
signal 	sprite_width,
		sprite_x, 
		sprite_y,
		sprite_cut,
		sprite_narrow, 
		sprite_y_end : sprite_pos_type;
signal sprite_show: std_logic_vector(21 downto 0) := (others => '0');
type sprite_color_type is array(integer range 0 to 21) of unsigned(2 downto 0);
signal sprite_color : sprite_color_type := (others => "100");
signal sprite_flip : std_logic_vector(21 downto 0);

----------rom size data
type rom_size_type is array(integer range 0 to 67) of integer range 1 to 256;
constant rom_width : rom_size_type := (
  64, 132, 132, 132,  84,  76,  76,  76, 
 112, 112, 112, 112, 112, 112, 112,  60,
  68,  68,  68,  68,  68,  68,  72, 100,
 100, 100, 136, 136,  52,  76,  76,  68, 
  68,  68, 152,  44, 152, 112, 104, 104,
 104, 104, 116, 116,  48,  48,  48,  48,  
  48,  48,  48,  48,  48,  48, 248, 248,  
  48, 248, 112, 112,  32,  68,  68, 240,
  80,  80,  68,  32
  );
constant rom_height : rom_size_type := (
150, 150, 150, 150, 150, 150, 110, 110, 
110, 110, 110, 110, 110, 110, 110, 150,
150, 150, 150, 150, 150, 150, 150, 162,
162, 162, 150, 150, 162, 120,  80, 162, 
162, 162,  37, 150,  88, 150, 150, 150, 
150, 150, 150, 150,  52,  52,  52,  52, 
 52,  52,  52,  52,  52,  52,  55,  55,  
 26,  72,  72,  72,  40,  61,  61,  48, 
311, 311,  28,  72 
);

--sram information store
type sram_offset_type is array(integer range 0 to 68) of integer range 0 to 262143;
constant sram_offset : sram_offset_type := ( 
76800,  79200,  84150,  89100,  94050,  97200,  100050, 102140,
104230, 107310, 110390, 113470, 116550, 119630, 122710, 125790,
128040, 130590, 133140, 135690, 138240, 140790, 143340, 146040,
150090, 154140, 158190, 163290, 168390, 170496, 172776, 174296,
177050, 179804, 182558, 183964, 185614, 188958, 193158, 197058,
200958, 204858, 208758, 213108, 217458, 218082, 218706, 219330,
219954, 220578, 221202, 221826, 222450, 223074, 223698, 227108,
230518, 230830, 235294, 237310, 239326, 239646, 240683, 241720,
244600, 250820, 257040, 257516, 258092
);

type sprite_addr_offset_type is array(integer range 0 to 21) of unsigned(17 downto 0);
signal sprite_addr_offset : sprite_addr_offset_type;

--buffer refresh pipeline stage1-----
signal current_sprite  	: integer range 0 to 63;
signal next_x		 	: unsigned(9 downto 0);
signal next_y			: unsigned(9 downto 0);
signal next_y_end 		: unsigned(9 downto 0);
signal next_width	 	: unsigned(9 downto 0);
signal next_narrow		: unsigned(9 downto 0);
signal next_cut			: unsigned(9 downto 0);
signal next_offset	 	: unsigned(7 downto 0);
signal next_color	 	: unsigned(2 downto 0);
signal next_flip	 	: std_logic;
signal next_show	 	: std_logic;
signal next_addr_offset : unsigned(17 downto 0);

--buffer refresh pipeline stage2-----
signal next_position		: unsigned(9 downto 0);
signal next_position_plus	: unsigned(9 downto 0);
signal next_next_color 		: unsigned(2 downto 0);
signal next_next_show	 	: std_logic;
signal next_next_flip		: std_logic;
signal sram_addr_buf		: unsigned(17 downto 0);

--buffer refresh pipeline stage3-----
signal sram_in_buf		: std_logic_vector(15 downto 0);
signal write_addr_buf	: unsigned(31 downto 0);
signal next3_show		: std_logic;
signal next3_color		: unsigned(2 downto 0);
signal next3_flip		: std_logic;
signal position_roll	: unsigned(1 downto 0);

--buffer refresh pipeline stage4-----
signal next4_buf		: unsigned(15 downto 0);
signal next4_show		: std_logic;
signal next4_color		: unsigned(2 downto 0);
signal write4_addr_buf	: unsigned(31 downto 0);

--buffer refresh pipeline stage5-----
signal we_buf			: std_logic_vector(3 downto 0);
signal addr_0, addr_1 	: unsigned(31 downto 0);
signal di		 		: unsigned(15 downto 0);
signal color_table		: unsigned(2 downto 0);
signal we1, we0 		: std_logic_vector(3 downto 0);
signal dataout_buf 		: unsigned(6 downto 0);
type do_type is array(integer range 0 to 3) of unsigned(6 downto 0);
signal do_1, do_0 		: do_type;
-------------------------------
--interface signal:
--writedata:
--bit 0 ~9  : x position
--bit 10~19 : y position
--bit 20~22 : color table	
--bit 28~29	: cut set
--bit 30    : flip direction
--bit 31    : irq release
-------------------------------
--sprite 0~20 can be used
--sprite 21 is used for background color setting
-------------------------------
begin
	
	irq <= irq_mask and y_blank;	
	x_buf <= writedata(9 downto 0);
	y_buf <= writedata(19 downto 10);
	sprite_index_buf <= to_integer( unsigned( address(11 downto 7) ) );
	rom_buf <= to_integer( unsigned( address(6 downto 0) ) );
	sram_addr <= std_logic_vector(sram_addr_buf);

	UPDATE: process(clk_50)
	begin
		if rising_edge(clk_50) then
			if y_blank = '0' then
				irq_mask <= '1';
			end if;
			if write = '1' then
				if y_blank = '1' then
					if writedata(31) = '1' then
						irq_mask <= '0';	
					end if;
									
					if x_buf = "1111111111" and y_buf = "1111111111" then
						sprite_show( sprite_index_buf ) <= '0';	
					elsif writedata(28) = '1' then
						sprite_narrow( sprite_index_buf ) <= sprite_width( sprite_index_buf ) - x_buf;
					elsif writedata(29) = '1' then
						sprite_narrow( sprite_index_buf ) <= sprite_width( sprite_index_buf ) - x_buf;
						sprite_cut( sprite_index_buf ) <= x_buf;
					else
						sprite_flip( sprite_index_buf ) <= writedata(30);
						sprite_show( sprite_index_buf ) <= '1';				
						sprite_x( sprite_index_buf ) <= x_buf;
						sprite_y( sprite_index_buf ) <= y_buf;
						sprite_y_end( sprite_index_buf ) <= y_buf + to_unsigned( rom_height(rom_buf), 10 );
						sprite_width( sprite_index_buf ) <= to_unsigned( rom_width(rom_buf), 10);
						sprite_addr_offset( sprite_index_buf ) <= to_unsigned( sram_offset(rom_buf), 18 );
						sprite_color( sprite_index_buf ) <= writedata(22 downto 20);
						sprite_cut( sprite_index_buf ) <= (others => '0');
						sprite_narrow( sprite_index_buf ) <= to_unsigned( rom_width(rom_buf), 10);
			end if; end if; end if;
				   --update the new sprite location only when the vga refreshing is blank between two frames 
		end if;
	end process UPDATE;

-------------------------------------------------------------------------	
	current_sprite <= to_integer( x_count(11 downto 6) ) - 4;
	BUF_REFRESH_1 : process(clk_50)
	begin
		if rising_edge(clk_50) then
			if  x_count < 160 then
				next_offset	<= x_count(7 downto 0);
				next_color	<= sprite_color(21);
				next_width	<= to_unsigned(640, 10);
				next_narrow	<= to_unsigned(640, 10);
				next_cut	<= to_unsigned(  0, 10);
				next_x 		<= to_unsigned(  0, 10);
				next_y 		<= to_unsigned(  0, 10);
				next_y_end 	<= to_unsigned(479, 10);
				next_flip	<= '0';
				next_show	<= '1';
				next_addr_offset <= to_unsigned(0, 18);				
			elsif current_sprite <= 20 and current_sprite >= 0 then
				next_offset	<= "00" & x_count(5 downto 0);
				next_x 		<= sprite_x(current_sprite);
				next_y 		<= sprite_y(current_sprite);
				next_y_end 	<= sprite_y_end(current_sprite);
				next_width 	<= sprite_width(current_sprite);
				next_flip 	<= sprite_flip(current_sprite);
				next_show 	<= sprite_show(current_sprite);
				next_color 	<= sprite_color(current_sprite);
				next_cut	<= sprite_cut(current_sprite);
				next_narrow <= sprite_narrow(current_sprite);
				next_addr_offset <= sprite_addr_offset(current_sprite);	
			else
				next_show	<= '0';
			end if;						
		end if;
	end process BUF_REFRESH_1;
	
	------------------------------------------------------------------
	BUF_REFRESH_2 : process(clk_50)
	begin
		if rising_edge(clk_50) then			
			if y >= next_y and y < next_y_end
			and next_offset < next_narrow(9 downto 2) then
					if next_flip = '1' then
						sram_addr_buf <= next_addr_offset - next_offset - next_cut(9 downto 2) - 1
									+ (1 + y - next_y) * next_width(9 downto 2);
					else
						sram_addr_buf <= next_addr_offset + next_offset + next_cut(9 downto 2)
									+ (y - next_y) * next_width(9 downto 2);	
					end if;
					next_position <= next_cut + next_x + (next_offset&"00");
					next_position_plus <= next_cut + next_x + (next_offset&"00") + 4;
					next_next_show <= next_show;
					next_next_color <= next_color;
					next_next_flip <= next_flip;
			else
					next_next_show <= '0';
			end if;
		end if;
	end process BUF_REFRESH_2;	
	
	------------------------------------------------------------------
	BUF_REFRESH_3 : process(clk_50)
	begin
		if rising_edge(clk_50) then
			next3_flip <= next_next_flip;
			next3_show <= next_next_show;
			next3_color <= next_next_color;
			position_roll <= next_position(1 downto 0);
			sram_in_buf <= sram_datain;		
			case next_position(1 downto 0) is
				when "00" =>write_addr_buf <= next_position(9 downto 2) & next_position(9 downto 2)
											& next_position(9 downto 2) & next_position(9 downto 2);							
				when "01" =>write_addr_buf <= next_position_plus(9 downto 2) & next_position(9 downto 2)
											& next_position(9 downto 2) & next_position(9 downto 2);							
				when "10" =>write_addr_buf <= next_position_plus(9 downto 2) & next_position_plus(9 downto 2)
											& next_position(9 downto 2) & next_position(9 downto 2);							
				when "11" =>write_addr_buf <= next_position_plus(9 downto 2) & next_position_plus(9 downto 2)
											& next_position_plus(9 downto 2) & next_position(9 downto 2);
			end case;
		end if;
	end process BUF_REFRESH_3;		

-----------------------------------------------------------------------------------------------------------
	BUF_REFRESH_4 : process(clk_50)
	begin
		if rising_edge(clk_50) then
		
			if next3_flip = '0' then
				case position_roll is
					when "00" => next4_buf <= unsigned(sram_in_buf);
					when "01" => next4_buf <= unsigned(sram_in_buf(3 downto 0)& sram_in_buf(15 downto 4));
					when "10" => next4_buf <= unsigned(sram_in_buf(7 downto 0)& sram_in_buf(15 downto 8));
					when "11" => next4_buf <= unsigned(sram_in_buf(11 downto 0)& sram_in_buf(15 downto 12));
				end case;
			else
				case position_roll is
					when "00" => next4_buf <= unsigned(sram_in_buf(3 downto 0)& sram_in_buf(7 downto 4)&sram_in_buf(11 downto 8)&sram_in_buf(15 downto 12) );
					when "01" => next4_buf <= unsigned(sram_in_buf(15 downto 12)&sram_in_buf(3 downto 0)& sram_in_buf(7 downto 4)&sram_in_buf(11 downto 8) );
					when "10" => next4_buf <= unsigned(sram_in_buf(11 downto 8)&sram_in_buf(15 downto 12)&sram_in_buf(3 downto 0)& sram_in_buf(7 downto 4) );
					when "11" => next4_buf <= unsigned(sram_in_buf(7 downto 4)&sram_in_buf(11 downto 8)&sram_in_buf(15 downto 12)&sram_in_buf(3 downto 0) );
				end case;
			end if;
			write4_addr_buf <= write_addr_buf;
			next4_color <= next3_color;
			next4_show <= next3_show;
		end if;
	end process BUF_REFRESH_4;	
		
-----------------------------------------------------------------------------------------------------------
	BUF_REFRESH_5 : process(clk_50)
	begin
		if rising_edge(clk_50) then	
		di <= next4_buf;
		color_table <= next4_color;
			if y(0) = '1' then			
				if next4_show = '1' then
					we1(0) <= not (next4_buf(15) and next4_buf(14) and next4_buf(13) and next4_buf(12));
					we1(1) <= not (next4_buf(11) and next4_buf(10) and next4_buf( 9) and next4_buf( 8));
					we1(2) <= not (next4_buf( 7) and next4_buf( 6) and next4_buf( 5) and next4_buf( 4));
					we1(3) <= not (next4_buf( 3) and next4_buf( 2) and next4_buf( 1) and next4_buf( 0));
				else
					we1 <= (others => '0');
				end if;
				we0 <= (others => '0');
				addr_1 <= write4_addr_buf;
				addr_0 <= x(9 downto 2) & x(9 downto 2)
						& x(9 downto 2) & x(9 downto 2);	
			else		
				if next4_show = '1' then
					we0(0) <= not (next4_buf(15) and next4_buf(14) and next4_buf(13) and next4_buf(12));
					we0(1) <= not (next4_buf(11) and next4_buf(10) and next4_buf( 9) and next4_buf( 8));
					we0(2) <= not (next4_buf( 7) and next4_buf( 6) and next4_buf( 5) and next4_buf( 4));
					we0(3) <= not (next4_buf( 3) and next4_buf( 2) and next4_buf( 1) and next4_buf( 0));
				else
					we0 <= (others => '0');
				end if;			
				we1 <= (others => '0');
				addr_0 <= write4_addr_buf;
				addr_1 <= x(9 downto 2) & x(9 downto 2)
						& x(9 downto 2) & x(9 downto 2);
			end if;
		end if;
	end process BUF_REFRESH_5;	
		
	dataout_buf <=  do_0(0) when x(1 downto 0)="01" and y(0) = '1' else
					do_0(1) when x(1 downto 0)="10" and y(0) = '1' else
					do_0(2) when x(1 downto 0)="11" and y(0) = '1' else
					do_0(3) when x(1 downto 0)="00" and y(0) = '1' else
					do_1(0) when x(1 downto 0)="01" and y(0) = '0' else
					do_1(1) when x(1 downto 0)="10" and y(0) = '0' else
					do_1(2) when x(1 downto 0)="11" and y(0) = '0' else
					do_1(3);
	pixel <= dataout_buf(3 downto 0);
	table <= dataout_buf(6 downto 4);

--------------------------------------------------------
	LINE_BUF_0m0 : entity work.line_buf port map(
		clk	=> clk_50, we=> we0(0),	 
		a=> addr_0(31 downto 24), do=>do_0(0),		
		di(3 downto 0)=> di(15 downto 12), 
		di(6 downto 4) => color_table );
	LINE_BUF_0m1 : entity work.line_buf port map(
		clk	=> clk_50, we=> we0(1),	do=>do_0(1), 
		a=> addr_0(23 downto 16),		
		di(3 downto 0)=> di(11 downto 8), 
		di(6 downto 4) => color_table );
	LINE_BUF_0m2 : entity work.line_buf port map(
		clk	=> clk_50, we=> we0(2),	do=>do_0(2), 
		a=> addr_0(15 downto 8), 		
		di(3 downto 0)=> di(7 downto 4), 
		di(6 downto 4) => color_table );
	LINE_BUF_0m3 : entity work.line_buf port map(
		clk	=> clk_50, we=> we0(3),	do=>do_0(3), 
		a=> addr_0(7 downto 0), 		
		di(3 downto 0)=> di(3 downto 0), 
		di(6 downto 4) => color_table );	
--------------------------------------------------------
	LINE_BUF_1m0 : entity work.line_buf port map(
		clk	=> clk_50, we=> we1(0), do=>do_1(0), 
		a=> addr_1(31 downto 24), 		
		di(3 downto 0)=> di(15 downto 12), 
		di(6 downto 4) => color_table );
	LINE_BUF_1m1 : entity work.line_buf port map(
		clk	=> clk_50, we=> we1(1),	do=>do_1(1), 
		a=> addr_1(23 downto 16),		
		di(3 downto 0)=> di(11 downto 8), 
		di(6 downto 4) => color_table );
	LINE_BUF_1m2 : entity work.line_buf port map(
		clk	=> clk_50, we=> we1(2),	do=>do_1(2), 
		a=> addr_1(15 downto 8),		
		di(3 downto 0)=> di(7 downto 4), 
		di(6 downto 4) => color_table );
	LINE_BUF_1m3 : entity work.line_buf port map(
		clk	=> clk_50, we=> we1(3),	do=>do_1(3), 
		a=> addr_1(7 downto 0),		
		di(3 downto 0)=> di(3 downto 0), 
		di(6 downto 4) => color_table );		
end decode;
