library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;

entity de2_vga_controller is
  
  port (
    reset : in std_logic;
    clk_50: in std_logic;
-- port with sprite controller
    pixel : in unsigned(3 downto 0); -- current pixel code
    table : in unsigned(2 downto 0); -- color table index to use
	x_index: out unsigned(9 downto 0);
	y_index: out unsigned(9 downto 0);
	x_count: out unsigned(11 downto 0);
	y_blank: out std_logic;

-- port with DE2 board
    VGA_CLK,                         -- Clock
    VGA_HS,                          -- H_SYNC
    VGA_VS,                          -- V_SYNC
    VGA_BLANK,                       -- BLANK
    VGA_SYNC : out std_logic;        -- SYNC
    VGA_R,                           -- Red[9:0]
    VGA_G,                           -- Green[9:0]
    VGA_B : out unsigned(9 downto 0) -- Blue[9:0]
    );

end de2_vga_controller;

architecture rtl of de2_vga_controller is
  
  -- Video parameters
  constant HTOTAL       : integer := 800;
  constant HSYNC        : integer := 96;
  constant HBACK_PORCH  : integer := 48;
  constant HACTIVE      : integer := 640;
  constant HFRONT_PORCH : integer := 16;
  
  constant VTOTAL       : integer := 525;
  constant VSYNC        : integer := 2;
  constant VBACK_PORCH  : integer := 33;
  constant VACTIVE      : integer := 480;
  constant VFRONT_PORCH : integer := 10;

  -- Signals for the video controller
  signal clk    : std_logic;
  signal x_cnt  : unsigned(11 downto 0);

  signal Hcount : unsigned(9 downto 0);  -- Horizontal position (0-800)
  signal Vcount : unsigned(9 downto 0);  -- Vertical position (0-524)
  signal EndOfLine, EndOfField : std_logic;
  signal vga_hblank, vga_hsync,
         vga_vblank, vga_vsync : std_logic;  -- Sync. signals
  -- Indexed color look up tables
  type color_table is array (integer range 0 to 7, integer range 0 to 15) of unsigned(7 downto 0);

constant R_table : color_table := (

 ( "01100010", "11101101", "11110111", "11100011", "11010111", "10110111", "10100000", "11001010",
   "01100110", "10000011", "00110001", "01010001", "10100000", "10110101", "00010010", "11111111"),--0:bg
 ( "00000000", "11011000", "11101001", "00010101", "00110000", "00100110", "00011110", "00001101", 
   "01011101", "10001000", "11101000", "11010011", "10100101", "10111100", "01011110", "11111111"),--1:p1
 ( "00000000", "01100010", "01110001", "00010101", "00110000", "00100110", "00011110", "00001101", 
   "01011101", "01011000", "11101000", "11010011", "10100101", "01101100", "01011110", "11111111"),--2:p2  
 ( "11101100", "11111111", "11111111", "11111111", "11111111", "11111111", "11111111", "11111111",
   "11111111", "00110101", "11111111", "11111111", "11111111", "11111111", "11111111", "11111111"),--3:orange
 ( "00000000", "00000000", "00000000", "00000000", "00000000", "00000000", "00000000", "00000000", 
   "00000000", "00000000", "00000000", "00000000", "00000000", "00000000", "00000000", "00000000"),--4:black
 ( "11111111", "11111111", "11111111", "11111111", "11111111", "11111111", "11111111", "11111111", 
   "11111111", "11111111", "11111111", "11111111", "11111111", "11111111", "11111111", "11111111"),--5:white
 ( "10001110", "11011110", "11000011", "11111001", "11111000", "11110100", "11111000", "11110000",
   "11101010", "11101000", "11101011", "11110000", "11110101", "11100111", "10100011", "11111111"),--6:fire1
 ( "00000011", "00000000", "00000000", "00000000", "00000000", "00000000", "00000000", "00000000",
   "00000000", "00000000", "00000000", "00000000", "00000000", "00000000", "00000000", "11111111" )--7:fire2
   );

constant G_table : color_table := (
 ( "10001100", "11100111", "11110010", "11011000", "11001001", "10101011", "10010100", "10111010",
   "01011110", "01111010", "00101011", "01000001", "01011001", "01101110", "00010001", "11111111"),--0:bg
 ( "00000000", "11011010", "11101100", "00011000", "01000100", "00110100", "00100110", "00001111", 
   "01100011", "10001111", "10101000", "10000001", "01100100", "10110101", "01000000", "11111111"),--1:p1
 ( "00000000", "00100001", "01000101", "00011000", "01000100", "00110100", "00100110", "00001111", 
   "01100011", "00010110", "10101000", "10000001", "01100100", "00101010", "01000000", "11111111"),--2:p2   
 ( "11101100", "11111001", "11110100", "11101110", "11101001", "11100011", "11011010", "11001010",
   "10111000", "00101110", "10100001", "10001001", "01101101", "01010100", "00111101", "11111111"),--3:orange
 ( "00000000", "00000000", "00000000", "00000000", "00000000", "00000000", "00000000", "00000000", 
   "00000000", "00000000", "00000000", "00000000", "00000000", "00000000", "00000000", "00000000"),--4:black
 ( "11111111", "11111111", "11111111", "11111111", "11111111", "11111111", "11111111", "11111111", 
   "11111111", "11111111", "11111111", "11111111", "11111111", "11111111", "11111111", "11111111"),--5:white
 ( "00000100", "00001011", "00000100", "11101100", "11001100", "10100110", "11101000", "10000101",
   "01100100", "01101100", "10000010", "10011011", "10111001", "00111000", "00111100", "11111111"),--6:fire1
 ( "00010011", "01001000", "01110001", "10010110", "01101011", "10101111", "10010001", "11000010",
   "11010111", "11101111", "10101101", "11000111", "11100001", "10001110", "01000110", "11111111" )--7:fire2
   );

constant B_table : color_table := (
 ( "10011110", "11011000", "11101001", "11000111", "10110110", "10011011", "10000111", "10101001",
   "01010110", "01110001", "00101000", "00111011", "01001010", "01011110", "00010001", "11111111"),--0:bg
 ( "00000000", "11100011", "11110010", "00011010", "01010001", "00111101", "00101010", "00001111", 
   "01100010", "10001011", "01111001", "01011000", "01000110", "10110010", "00110101", "11111111"), --1:p1
 ( "00000000", "00110100", "01011001", "00011010", "01010001", "00111101", "00101010", "00001111", 
   "01100010", "00110010", "01111001", "01011000", "01000110", "01001100", "00110101", "11111111"), --2:p2   
 ( "11101110", "00000000", "00000000", "00000000", "00000000", "00000000", "00000000", "00000000",
   "00000000", "00011110", "00000000", "00000000", "00000000", "00000000", "00000000", "11111111"),--3:orange
 ( "00000000", "00000000", "00000000", "00000000", "00000000", "00000000", "00000000", "00000000", 
   "00000000", "00000000", "00000000", "00000000", "00000000", "00000000", "00000000", "00000000"),--4:black
 ( "11111111", "11111111", "11111111", "11111111", "11111111", "11111111", "11111111", "11111111", 
   "11111111", "11111111", "11111111", "11111111", "11111111", "11111111", "11111111", "11111111"),--5:white
 ( "00000110", "00010011", "00001110", "01010010", "00010011", "00001110", "11010100", "00010000",
   "00010001", "00111010", "01011001", "01111010", "10100101", "00011000", "00111011", "11111111"),--6:fire1
 ( "11101011", "11100111", "11100010", "11100001", "10011011", "11011011", "10101100", "11001100",
   "11010100", "11001001", "10001111", "10010110", "10100111", "01101001", "00110001", "11111111" )--7:fire2
   );
begin

  x_count <= x_cnt;

  process (clk_50)
  begin
    if rising_edge(clk_50) then
      clk <= not clk;
	  if reset = '1' then
		x_cnt <= (others => '0');	
	  elsif EndOfLine = '1' then
		x_cnt <= (others => '0');
	  else
		x_cnt <= x_cnt + 1;
    end if; end if;
  end process;

  -- Horizontal and vertical counters
  HCounter : process (clk)
  begin
    if rising_edge(clk) then      
      if reset = '1' then
        Hcount <= (others => '0');
      elsif EndOfLine = '1' then
        Hcount <= (others => '0');
      else
        Hcount <= Hcount + 1;
    end if;  end if;
  end process HCounter;

  EndOfLine <= '1' when Hcount = HTOTAL - 1 else '0';
  
  VCounter: process (clk)
  begin
    if rising_edge(clk) then      
      if reset = '1' then
        Vcount <= (others => '0');
      elsif EndOfLine = '1' then
        if EndOfField = '1' then
          Vcount <= (others => '0');
        else
          Vcount <= Vcount + 1;
        end if;  end if;  end if;
  end process VCounter;

  EndOfField <= '1' when Vcount = VTOTAL - 1 else '0';

  -- State machines to generate HSYNC, VSYNC, HBLANK, and VBLANK
  HSyncGen : process (clk)
  begin
    if rising_edge(clk) then     
      if reset = '1' or EndOfLine = '1' then
        vga_hsync <= '1';
      elsif Hcount = HSYNC - 1 then
        vga_hsync <= '0';
      end if;  end if;
  end process HSyncGen;
  
  HBlankGen : process (clk)
  begin
    if rising_edge(clk) then
      if reset = '1' then
        vga_hblank <= '1';
      elsif Hcount = HSYNC + HBACK_PORCH then
        vga_hblank <= '0';
      elsif Hcount = HSYNC + HBACK_PORCH + HACTIVE then
        vga_hblank <= '1';
      end if; end if;
  end process HBlankGen;

  VSyncGen : process (clk)
  begin
    if rising_edge(clk) then
      if reset = '1' then
        vga_vsync <= '1';
      elsif EndOfLine ='1' then
        if EndOfField = '1' then
          vga_vsync <= '1';
        elsif Vcount = VSYNC - 1 then
          vga_vsync <= '0';
        end if;  end if;  end if;
  end process VSyncGen;

  VBlankGen : process (clk)
  begin
    if rising_edge(clk) then    
      if reset = '1' then
        vga_vblank <= '1';
      elsif EndOfLine = '1' then
        if Vcount = VSYNC + VBACK_PORCH - 1 then
          vga_vblank <= '0';
        elsif Vcount = VSYNC + VBACK_PORCH + VACTIVE - 1 then
          vga_vblank <= '1';
        end if; end if; end if;
  end process VBlankGen;

  FLAGS : process(clk)
  begin
	if rising_edge(clk) then
		x_index <= Hcount - HSYNC - HBACK_PORCH + 1;
		y_index <= Vcount - VSYNC - VBACK_PORCH + 1;
		y_blank <= vga_vblank;
	end if;
  end process FLAGS;

  -- Registered video signals going to the video DAC
  VideoOut: process (clk, reset)
  begin
    if reset = '1' then
      VGA_R <= "0000000000";
      VGA_G <= "0000000000";
      VGA_B <= "0000000000";
    elsif rising_edge(clk) then
		if  vga_hblank = '0' and vga_vblank = '0'then		
				VGA_R(1 downto 0) <= "00";
				VGA_G(1 downto 0) <= "00";
				VGA_B(1 downto 0) <= "00";
				VGA_R(9 downto 2) <= R_table( to_integer(table), to_integer(pixel) );
				VGA_G(9 downto 2) <= G_table( to_integer(table), to_integer(pixel) );
				VGA_B(9 downto 2) <= B_table( to_integer(table), to_integer(pixel) );
		else
			VGA_R <= "0000000000";
			VGA_G <= "0000000000";
			VGA_B <= "0000000000";
    end if; end if;
  end process VideoOut;

  VGA_CLK <= clk;
  VGA_HS <= not vga_hsync;
  VGA_VS <= not vga_vsync;
  VGA_SYNC <= '0';
  VGA_BLANK <= not (vga_hsync or vga_vsync);

end rtl;
