library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;
--E: VGA Counter
entity vga_counter is
  
  port (
    reset : in std_logic;
    clk   : in std_logic;                    -- Should be 25.125 MHz

    VGA_CLK,                         -- Clock
    VGA_HS,                          -- H_SYNC
    VGA_VS,                          -- V_SYNC
    VGA_BLANK,                       -- BLANK
    VGA_SYNC : out std_logic;        -- SYNC


    VGA_X	 : out unsigned(9 downto 0);
	VGA_Y	 : out unsigned(8 downto 0);
	VGA_32	 : out std_logic;
	VGA_16	 : out std_logic
    );
end vga_counter;

architecture vga_counter_rtl of vga_counter is
  -- Video parameters
  
  constant HTOTAL       : integer := 800;
  constant HSYNC        : integer := 96;
  constant HBACK_PORCH  : integer := 48;
  constant HACTIVE      : integer := 640;
  constant HFRONT_PORCH : integer := 16;
  
  constant VTOTAL       : integer := 525;
  constant VSYNC        : integer := 2;
  constant VBACK_PORCH  : integer := 33;
  constant VACTIVE      : integer := 480;
  constant VFRONT_PORCH : integer := 10;

  constant HZONE		: integer := 384;
  constant LAG			: integer := 5;

  -- Signals for the video controller
  signal Hcount, HCarryCount : unsigned(9 downto 0);  -- Horizontal position (0-800)
  signal Vcount : unsigned(9 downto 0);  -- Vertical position (0-524)
  signal EndOfLine, EndOfField : std_logic;

  signal vga_hblank, vga_hsync,
		vga_vblank, vga_vsync, 
		vga_32_zone, vga_16_zone : std_logic;  -- Sync. signals
		
begin
  HCounter : process (clk)
  begin
    if rising_edge(clk) then      
      if reset = '1' then
        Hcount <= (others => '0');
      elsif EndOfLine = '1' then
        Hcount <= (others => '0');
      else
        Hcount <= Hcount + 1;
      end if;      
    end if;
  end process HCounter;

  EndOfLine <= '1' when Hcount = HTOTAL - 1 else '0';
  
  VCounter: process (clk)
  begin
    if rising_edge(clk) then      
      if reset = '1' then
        Vcount <= (others => '0');
      elsif EndOfLine = '1' then
        if EndOfField = '1' then
          Vcount <= (others => '0');
        else
          Vcount <= Vcount + 1;
        end if;
      end if;
    end if;
  end process VCounter;

  EndOfField <= '1' when Vcount = VTOTAL - 1 else '0';

  -- State machines to generate HSYNC, VSYNC, HZONE

  HSyncGen : process (clk)
  begin
    if rising_edge(clk) then     
      if reset = '1' or EndOfLine = '1' then
        vga_hsync <= '1';
      elsif Hcount = HSYNC - 1 then
        vga_hsync <= '0';
      end if;
    end if;
  end process HSyncGen;


  VSyncGen : process (clk)
  begin
    if rising_edge(clk) then
      if reset = '1' then
        vga_vsync <= '1';
      elsif EndOfLine ='1' then
        if EndOfField = '1' then
          vga_vsync <= '1';
        elsif Vcount = VSYNC - 1 then
          vga_vsync <= '0';
        end if;
      end if;      
    end if;
  end process VSyncGen;

  HZoneGen : process (clk)
  begin
    if rising_edge(clk) then
      if reset = '1' then
        vga_32_zone <= '0';
		vga_16_zone <= '0';
      elsif Hcount = HSYNC + HBACK_PORCH - LAG then
        vga_32_zone <= '1';
		vga_16_zone <= '0';
	  elsif Hcount = HSYNC + HBACK_PORCH + HZONE - LAG then
        vga_32_zone <= '0';
		vga_16_zone <= '1';
      elsif Hcount = HSYNC + HBACK_PORCH + HACTIVE - LAG then
        vga_32_zone <= '0';
		vga_16_zone <= '0';
      end if;
	  HCarryCount <= HCount;
    end if;
  end process HZoneGen;


--  HBlankGen : process (clk)
--  begin
--    if rising_edge(clk) then
--      if reset = '1' then
--        vga_hblank <= '1';
--      elsif Hcount = HSYNC + HBACK_PORCH then
--        vga_hblank <= '0';
--      elsif Hcount = HSYNC + HBACK_PORCH + HACTIVE then
--        vga_hblank <= '1';
--      end if;      
--    end if;
--  end process HBlankGen;

  VBlankGen : process (clk)
  begin
    if rising_edge(clk) then    
      if reset = '1' then
        vga_vblank <= '1';
      elsif EndOfLine = '1' then
        if Vcount = VSYNC + VBACK_PORCH - 1 then
          vga_vblank <= '0';
        elsif Vcount = VSYNC + VBACK_PORCH + VACTIVE - 1 then
          vga_vblank <= '1';
        end if;
      end if;
    end if;
  end process VBlankGen;




  VGA_CLK <= clk;
  VGA_HS <= not vga_hsync;
  VGA_VS <= not vga_vsync;
  VGA_SYNC <= '0';
  VGA_BLANK <= not (vga_hsync or vga_vsync);

  VGA_X <= unsigned(HCarryCount - HSYNC - HBACK_PORCH + LAG);
  VGA_Y <= unsigned(VCount - VSYNC - VBACK_PORCH)(8 downto 0);
  VGA_32 <= vga_32_zone AND NOT vga_vblank;
  VGA_16 <= vga_16_zone AND NOT vga_vblank;

end vga_counter_rtl;
-----------------------------------------
library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;
entity ctoa_converter_32 is
	port(
		clk			: in	std_logic;
		c_vert,
		c_horiz		: in	unsigned(4 downto 0);
		ctrl_in		: in	std_logic_vector(1 downto 0);
		x_in, y_in	: in	unsigned(4 downto 0);
		ctrl_out	: out	std_logic_vector(1 downto 0);
		x_out, y_out: out	unsigned(4 downto 0);
		addr		: out   unsigned(10 downto 0)
	);
end ctoa_converter_32;

architecture ctoa_32_arch of ctoa_converter_32 is
--	signal  vert_8_32, vert_4_32, horiz_0_32,
--			vert_16_16, horiz_0_16  : unsigned(10 downto 0) := "000" & X"00";
--	signal addr_32, addr_16 		: unsigned(10 downto 0) := "000" & X"00";
--	signal ctrl 					: std_logic_vector(1 downto 0);
	signal addr_32, addr_16 		: unsigned(10 downto 0) := "000" & X"00";
	signal x_temp, y_temp			: unsigned(4 downto 0);
	signal ctrl 					: std_logic_vector(1 downto 0);
begin
	process(clk)
	begin
		if rising_edge(clk) then
			if ctrl_in = "10" then
				addr	<= ("00" &(("0" & c_vert(4 downto 1) & "0") + ("00" & c_vert(4 downto 1)) + ("0000" & c_horiz(4 downto 3))) & c_horiz(2 downto 1) & "0");
			elsif ctrl_in = "01" then
				addr	<= (("0" & c_vert & (NOT c_horiz(3)) & c_horiz(2)) + 45) & c_horiz(1 downto 0) & "0";
			else
				addr	<= "00000000000";
			end if;	
			x_out	<= x_in;
			y_out	<= y_in;
			ctrl_out<= ctrl_in;
		end if;
	end process;

--	process(clk)
--	begin
--		if rising_edge(clk) then
--				addr_32	<= ("00" &(("0" & c_vert(4 downto 1) & "0") + ("00" & c_vert(4 downto 1)) + ("0000" & c_horiz(4 downto 3))) & c_horiz(2 downto 1) & "0");
--				addr_16	<= (("0" & c_vert & (NOT c_horiz(3)) & c_horiz(2)) + 45) & c_horiz(1 downto 0) & "0";		
--				
--				x_temp	<= x_in;
--				y_temp	<= y_in;
--				
--				ctrl	<= ctrl_in;
--		end if;
--	end process;
--	x_out <= x_temp;
--	y_out <= y_temp;
--	ctrl_out <= ctrl;
--	with ctrl select addr <=
--		addr_32 when "10",
--		addr_16 when "01",
--		"00000000000" when others;
end ctoa_32_arch;

-----------------------------------------

library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;
entity ram_2_wrapper is
	port(
		clk, 
		reset,
		read,
		write,
		chipselect	:	in std_logic;
		address		:	in std_logic_vector(9 downto 0);
		readdata	:	out std_logic_vector(15 downto 0);
		writedata	:	in std_logic_vector(15 downto 0);
		
		read_addr	:	in std_logic_vector(9 downto 0);
		ctrl_in		:	in 	std_logic_vector(1 downto 0);
		ctrl_out	:	out	std_logic_vector(1 downto 0);
		sprite_addr	:	out std_logic_vector(9 downto 0);
		
		x_in, y_in	:	in 	unsigned(4 downto 0);
		x_out, y_out:	out unsigned(4 downto 0)
    );
end ram_2_wrapper;

architecture ram_2_wrap_rtl of ram_2_wrapper is
	signal ram_out	:	std_logic_vector(15 downto 0);
	signal ctrl		:	std_logic_vector(1 downto 0);
	signal tmp_ctrl :	std_logic_vector(1 downto 0);
	signal tmp_x,
			tmp_y	:	unsigned(4 downto 0);
	
	signal we, re, oe	: 	std_logic;
	signal trd, rd	:	std_logic_vector(15 downto 0);
begin
	oe <= tmp_ctrl(1) OR tmp_ctrl(0);

	process(chipselect, read, write, oe)
	begin
		if chipselect = '1' then
			if write = '1' then
				ctrl <= "11";
			elsif read = '1' then
				ctrl <= "10";
			else
				ctrl <= "00";
			end if;
		elsif oe = '1' then
			ctrl <= "01";
		else
			ctrl <= "00";
		end if;
	end process;
	
	we <= '1' when ctrl = "11" else '0';
	
	
	MRAM: entity work.ram_2_port port map(
		clock 		 => clk,
		address_a	 => std_logic_vector(address),
		data_a		 => writedata,
		wren_a		 => we,
		q_a		 	 => trd,
		address_b	 => std_logic_vector(read_addr),
		data_b		 => X"0000",
		wren_b		 => '0',
		q_b			 => ram_out
	);
	process(clk)
	begin
		if rising_edge(clk) then
			ctrl_out <= tmp_ctrl;
			tmp_ctrl <= ctrl_in;
			
			x_out <= tmp_x;
			y_out <= tmp_y;
			tmp_x <= x_in;
			tmp_y <= y_in;
			
			re <= read;
		end if;
	end process;
	
	rd <= trd when re = '1' else rd;
	readdata <= rd;
	with oe select sprite_addr <=
		ram_out(9 downto 0) when '1',
		X"00" & "00" when '0';

end ram_2_wrap_rtl;
--------------------------------------

library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;
entity rotate_32 is
	port(
		clk			: in	std_logic;
		SPRITE_IN	: in	std_logic_vector(9 downto 0);
		CTRL_IN		: in	std_logic_vector(1 downto 0);
		VERT_IN,
		HORIZ_IN	: in	unsigned(4 downto 0);
		RT			: in	std_logic_vector(1 downto 0);
		SPRITE_OUT	: out	std_logic_vector(9 downto 0);
		CTRL_OUT	: out	std_logic_vector(1 downto 0);
		VERT_OUT,
		HORIZ_OUT	: out	unsigned(4 downto 0)
	);
end rotate_32;

architecture rotate_32_rtl of rotate_32 is
begin
	process(clk)
	begin
		if rising_edge(clk) then
			SPRITE_OUT <= SPRITE_IN;
			CTRL_OUT <= CTRL_IN;
			if RT = "11" then
				VERT_OUT <= NOT HORIZ_IN;
				HORIZ_OUT <= VERT_IN;
			elsif RT = "10" then
				VERT_OUT <= NOT VERT_IN;
				HORIZ_OUT <= NOT HORIZ_IN;
			elsif RT = "01" then
				VERT_OUT <= HORIZ_IN;
				HORIZ_OUT <= NOT VERT_IN;
			else
				VERT_OUT <= VERT_IN;
				HORIZ_OUT <= HORIZ_IN;
			end if;
		end if;
	end process;	
end rotate_32_rtl;

----------------------------------

library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;
entity sprite_rom_p is
	port(
		clk,
		reset		: in	std_logic;
		s_addr		: in	unsigned(6 downto 0);
		c_vert,
		c_horiz		: in	unsigned(4 downto 0);
		invert		: in	std_logic;
		s16			: in	std_logic;
		addr_out	: out	unsigned(18 downto 0);
		RGB_IN		: in	std_logic_vector(7 downto 0);
		RGB_OUT		: out   std_logic_vector(7 downto 0)
	);
end sprite_rom_p;


architecture sprite_arch_p of sprite_rom_p is
	signal p_addr : unsigned(18 downto 0);
	signal int_invert, int_reset : std_logic;
begin
	--p_addr <= "0100" & s_addr & c_vert(3 downto 0) & c_horiz(3 downto 0) when s16 = '1'
	--			else "00" & s_addr & c_vert & c_horiz;
	process(clk)
	begin
		if rising_edge(clk) then
			if s16 = '1' then
				addr_out <= "0100" & s_addr & c_vert(3 downto 0) & c_horiz(3 downto 0);
			else
				addr_out <= "00" & s_addr & c_vert & c_horiz;
			end if;
			--addr_out <= std_logic_vector(p_addr);
			int_invert <= invert;
			int_reset <= reset;
			if int_reset = '0' then
				RGB_OUT <= X"00";
			elsif int_invert = '1' then
				RGB_OUT <= NOT RGB_IN;
			else
				RGB_OUT <= RGB_IN;
			end if;
		end if;
	end process;
end sprite_arch_p;


----------------------------------


library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;
entity rgb_to_vga is
	port(
		RGB_HIGH	: in	std_logic;
		RGB_IN		: in	std_logic_vector(2 downto 0);
		VGA_OUT		: out	unsigned(9 downto 0)
	);
end rgb_to_vga;

architecture rgb_to_vga_rtl of rgb_to_vga is
begin
	with RGB_HIGH select VGA_OUT <=
		(others => '1') when '1',
		unsigned(RGB_IN & "0000000") when '0';
end rgb_to_vga_rtl;

----------------------------------




library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;
entity vga_micro_driver is

  port (                     -- External Clock
    clk		   : in  std_logic;
    reset_n    : in  std_logic;
    read       : in  std_logic;
    write      : in  std_logic;
    chipselect : in  std_logic;
    address    : in  unsigned(9 downto 0);
    readdata   : out std_logic_vector(15 downto 0);
    writedata  : in  std_logic_vector(15 downto 0);

	-- Loop around signals
	sprite_addr_out : out std_logic_vector(18 downto 0);
	rgb_vector_in	: in  std_logic_vector(7 downto 0);

    -- VGA output
    
    VGA_CLK,                                            -- Clock
    VGA_HS,                                             -- H_SYNC
    VGA_VS,                                             -- V_SYNC
    VGA_BLANK,                                          -- BLANK
    VGA_SYNC : out std_logic;                           -- SYNC
    VGA_R,                                              -- Red[9:0]
    VGA_G,                                              -- Green[9:0]
    VGA_B : out unsigned(9 downto 0)                    -- Blue[9:0]
    );
  
end vga_micro_driver;


architecture datapath of vga_micro_driver is

	signal clk50 : std_logic := '0';
	signal clk25 : std_logic := '0';

	signal coord_x : unsigned(9 downto 0);
	signal coord_y : unsigned(8 downto 0);
	signal pixel_x, pixel_y,
		   pixel_temp_x, pixel_temp_y: unsigned(4 downto 0);
	signal enable_32, enable_16,
			temp_32, temp_16	: std_logic;
	
	signal 	ctrl_2,
			ctrl_3,
			ctrl_4  : std_logic_vector(1 downto 0);
	signal 	pix_x_2,
			pix_y_2	: unsigned(4 downto 0);
	
	
	
	signal mem_addr: unsigned(10 downto 0);
	signal out_val, out_val_2: std_logic_vector(15 downto 0) := X"0000";
	signal sig_RGB_32, sig_RGB_16, sig_RGB, sig_RGB_PRC: std_logic_vector(7 downto 0);
	signal 	sig_red, sig_green, sig_blue,
			sig_red_x, sig_green_x, sig_blue_x,
			sig_red_y, sig_green_y, sig_blue_y	: unsigned(9 downto 0);

begin

	clk50 <= clk;
	
	CLK_25: process(clk)
	begin
		if rising_edge(clk) then
			clk25 <= NOT clk25;
		end if;
	end process CLK_25;

--2 Clock Cycles (due to clk25)
	VC: entity work.vga_counter port map(
		reset 		=> '0' ,
		clk 		=> clk25,

		VGA_CLK 	=> VGA_CLK,
		VGA_HS 		=> VGA_HS,
		VGA_VS 		=> VGA_VS,
		VGA_BLANK 	=> VGA_BLANK,
		VGA_SYNC 	=> VGA_SYNC,
		
		VGA_X 		=> coord_x,
		VGA_Y 		=> coord_y,
		VGA_32 		=> enable_32,
		VGA_16		=> enable_16
    );

--Clock Cycle
	ctoa: entity work.ctoa_converter_32 port map(
		clk			=> clk50,
	
		c_vert 		=> coord_y(8 downto 4),
		c_horiz 	=> coord_x(8 downto 4),
		ctrl_in		=> enable_32 & enable_16,
		ctrl_out	=> ctrl_2,
		x_in		=> coord_x(4 downto 0),
		y_in		=> coord_y(4 downto 0),
		x_out		=> pix_x_2,
		y_out		=> pix_y_2,
		addr 		=> mem_addr
	);

--2 Clock Cycles
	PRAM: entity work.ram_2_wrapper port map(
		clk				=> clk50,
		reset			=> reset_n,
		read			=> read,
		write			=> write, 
		chipselect		=> chipselect,
		address			=> std_logic_vector(address),
		readdata		=> readdata,
		writedata		=> writedata,

		read_addr		=> std_logic_vector(mem_addr(10 downto 1)),
		ctrl_in			=> ctrl_2,
		ctrl_out		=> ctrl_3,
		sprite_addr		=> out_val(9 downto 0),
		x_in			=> pix_x_2,
		y_in			=> pix_y_2,
		x_out			=> pixel_temp_x,
		y_out			=> pixel_temp_y
    );

--Clock Cycle
	rotate: entity work.rotate_32 port map(
		clk 		=> clk50,
		VERT_IN		=> pixel_temp_y,
		HORIZ_IN	=> pixel_temp_x,
		SPRITE_IN	=> out_val(9 downto 0),
		CTRL_IN		=> ctrl_3,
		RT => out_val(8 downto 7),
		SPRITE_OUT	=> out_val_2(9 downto 0),
		CTRL_OUT	=> ctrl_4,
		VERT_OUT	=> pixel_y,
		HORIZ_OUT	=> pixel_x
	);
	
--2 Clock Cycles
	sprite_rom_p: entity work.sprite_rom_p port map (
		clk => clk50,
		reset => ctrl_4(1) OR ctrl_4(0),
		s_addr => unsigned(out_val_2(6 downto 0)),
		c_vert => pixel_y,
		c_horiz => pixel_x,
		invert => out_val_2(9),
		s16 => ctrl_4(0),
		std_logic_vector(addr_out) => sprite_addr_out,
		RGB_IN => rgb_vector_in,
		RGB_OUT => sig_RGB_PRC
	);
	
	vga_red: entity work.rgb_to_vga port map (
		RGB_IN => sig_RGB_PRC(7 downto 5),
		RGB_HIGH => sig_RGB_PRC(7) AND sig_RGB_PRC(6) AND sig_RGB_PRC(5),
		VGA_OUT => sig_red
	);
	vga_green: entity work.rgb_to_vga port map (
		RGB_IN => sig_RGB_PRC(4 downto 2),
		RGB_HIGH => sig_RGB_PRC(4) AND sig_RGB_PRC(3) AND sig_RGB_PRC(2),
		VGA_OUT => sig_green
	);
	vga_blue: entity work.rgb_to_vga port map (
		RGB_IN => '0' & sig_RGB_PRC(1 downto 0),
		RGB_HIGH => sig_RGB_PRC(1) AND sig_RGB_PRC(0),
		VGA_OUT => sig_blue
	);
	
	--2 Clock Cycle
	STALL: process(clk50)
	begin
		if rising_edge(clk50) then
			VGA_R <= sig_red_x;
			VGA_G <= sig_green_x;
			VGA_B <= sig_blue_x;
			sig_red_x <= sig_red_y;
			sig_green_x <= sig_green_y;
			sig_blue_x <= sig_blue_y;
			sig_red_y <= sig_red;
			sig_green_y <= sig_green;
			sig_blue_y <= sig_blue;
		end if;
	end process STALL;
	
	
--	VGA_R <= sig_red;
--	VGA_G <= sig_green;
--	VGA_B <= sig_blue;
end datapath;


--RAM Megafunction


LIBRARY ieee;
USE ieee.std_logic_1164.all;

LIBRARY altera_mf;
USE altera_mf.all;

ENTITY ram_2_port IS
	PORT
	(
		address_a		: IN STD_LOGIC_VECTOR (9 DOWNTO 0);
		address_b		: IN STD_LOGIC_VECTOR (9 DOWNTO 0);
		clock		: IN STD_LOGIC ;
		data_a		: IN STD_LOGIC_VECTOR (15 DOWNTO 0);
		data_b		: IN STD_LOGIC_VECTOR (15 DOWNTO 0);
		wren_a		: IN STD_LOGIC  := '1';
		wren_b		: IN STD_LOGIC  := '1';
		q_a		: OUT STD_LOGIC_VECTOR (15 DOWNTO 0);
		q_b		: OUT STD_LOGIC_VECTOR (15 DOWNTO 0)
	);
END ram_2_port;


ARCHITECTURE SYN OF ram_2_port IS

	SIGNAL sub_wire0	: STD_LOGIC_VECTOR (15 DOWNTO 0);
	SIGNAL sub_wire1	: STD_LOGIC_VECTOR (15 DOWNTO 0);



	COMPONENT altsyncram
	GENERIC (
		address_reg_b		: STRING;
		clock_enable_input_a		: STRING;
		clock_enable_input_b		: STRING;
		clock_enable_output_a		: STRING;
		clock_enable_output_b		: STRING;
		indata_reg_b		: STRING;
		intended_device_family		: STRING;
		lpm_type		: STRING;
		numwords_a		: NATURAL;
		numwords_b		: NATURAL;
		operation_mode		: STRING;
		outdata_aclr_a		: STRING;
		outdata_aclr_b		: STRING;
		outdata_reg_a		: STRING;
		outdata_reg_b		: STRING;
		power_up_uninitialized		: STRING;
		read_during_write_mode_mixed_ports		: STRING;
		widthad_a		: NATURAL;
		widthad_b		: NATURAL;
		width_a		: NATURAL;
		width_b		: NATURAL;
		width_byteena_a		: NATURAL;
		width_byteena_b		: NATURAL;
		wrcontrol_wraddress_reg_b		: STRING
	);
	PORT (
			wren_a	: IN STD_LOGIC ;
			clock0	: IN STD_LOGIC ;
			wren_b	: IN STD_LOGIC ;
			address_a	: IN STD_LOGIC_VECTOR (9 DOWNTO 0);
			address_b	: IN STD_LOGIC_VECTOR (9 DOWNTO 0);
			q_a	: OUT STD_LOGIC_VECTOR (15 DOWNTO 0);
			q_b	: OUT STD_LOGIC_VECTOR (15 DOWNTO 0);
			data_a	: IN STD_LOGIC_VECTOR (15 DOWNTO 0);
			data_b	: IN STD_LOGIC_VECTOR (15 DOWNTO 0)
	);
	END COMPONENT;

BEGIN
	q_a    <= sub_wire0(15 DOWNTO 0);
	q_b    <= sub_wire1(15 DOWNTO 0);

	altsyncram_component : altsyncram
	GENERIC MAP (
		address_reg_b => "CLOCK0",
		clock_enable_input_a => "BYPASS",
		clock_enable_input_b => "BYPASS",
		clock_enable_output_a => "BYPASS",
		clock_enable_output_b => "BYPASS",
		indata_reg_b => "CLOCK0",
		intended_device_family => "Cyclone II",
		lpm_type => "altsyncram",
		numwords_a => 1024,
		numwords_b => 1024,
		operation_mode => "BIDIR_DUAL_PORT",
		outdata_aclr_a => "NONE",
		outdata_aclr_b => "NONE",
		outdata_reg_a => "CLOCK0",
		outdata_reg_b => "CLOCK0",
		power_up_uninitialized => "FALSE",
		read_during_write_mode_mixed_ports => "DONT_CARE",
		widthad_a => 10,
		widthad_b => 10,
		width_a => 16,
		width_b => 16,
		width_byteena_a => 1,
		width_byteena_b => 1,
		wrcontrol_wraddress_reg_b => "CLOCK0"
	)
	PORT MAP (
		wren_a => wren_a,
		clock0 => clock,
		wren_b => wren_b,
		address_a => address_a,
		address_b => address_b,
		data_a => data_a,
		data_b => data_b,
		q_a => sub_wire0,
		q_b => sub_wire1
	);



END SYN;