library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;

entity de2_wm8731_audio is
port (
    clk : in std_logic;       --  Audio CODEC Chip Clock AUD_XCK (18.43 MHz)
    reset_n : in std_logic;
    
    write      : in  std_logic;
	read       : in  std_logic;
    chipselect : in  std_logic;
    address    : in  unsigned(4 downto 0);
    writedata  : in  unsigned(15 downto 0);
    readdata   : out unsigned(15 downto 0);
  
    -- Audio interface signals
    AUD_ADCLRCK  : out  std_logic;   --    Audio CODEC ADC LR Clock
    AUD_ADCDAT   : in   std_logic;   --    Audio CODEC ADC Data
    AUD_DACLRCK  : out  std_logic;   --    Audio CODEC DAC LR Clock
    AUD_DACDAT   : out  std_logic;   --    Audio CODEC DAC Data
    AUD_BCLK     : inout std_logic  --    Audio CODEC Bit-Stream Clock
  );
end  de2_wm8731_audio;

architecture rtl of de2_wm8731_audio is     

    signal lrck : std_logic;
    signal bclk : std_logic;
    signal xck  : std_logic;
    
    signal lrck_divider : unsigned(11 downto 0); 
    signal bclk_divider : unsigned(3 downto 0);
    
    signal set_bclk : std_logic;
    signal set_lrck : std_logic;
    signal clr_bclk : std_logic;
    signal lrck_lat : std_logic;
    
    signal shift_out : unsigned(15 downto 0);

    signal sin_out     : unsigned(15 downto 0);
    signal sin_counter : unsigned(11 downto 0);    

-----------
type rom_type is array(920 downto 0) of unsigned(15 downto 0);
    
    constant ROM1 : rom_type :=
(

  X"2c21",
    X"ba0e",
    X"9602",
    X"34a4",
    X"7347",
    X"f0b0",
    X"8621",
    X"f2b1",
    X"75dd",
    X"2889",
    X"9364",
    X"bcbe",
    X"5b11",
    X"5cb1",
    X"c1a9",
    X"8e13",
    X"ff91",
    X"74ab",
    X"3488",
    X"9b9e",
    X"b06b",
    X"520e",
    X"61e3",
    X"c495",
    X"8fc2",
    X"1f31",
    X"7840",
    X"054c",
    X"8835",
    X"e464",
    X"6670",
    X"486a",
    X"b44d",
    X"ab11",
    X"4b4e",
    X"62ba",
    X"d430",
    X"8ef7",
    X"e630",
    X"68a5",
    X"43f3",
    X"ab9e",
    X"a6a3",
    X"4062",
    X"67a7",
    X"d739",
    X"8fcd",
    X"0bf4",
    X"73a3",
    X"1967",
    X"9aa3",
    X"d308",
    X"5cd5",
    X"3edf",
    X"bbd3",
    X"a4e1",
    X"fc3b",
    X"5f68",
    X"e964",
    X"a3ce",
    X"04b5",
    X"52d9",
    X"39c9",
    X"cd35",
    X"b59c",
    X"23c5",
    X"4d59",
    X"f290",
    X"b1ba",
    X"fa1f",
    X"49bc",
    X"1942",
    X"c1bf",
    X"d912",
    X"34aa",
    X"3087",
    X"dbae",
    X"c8c2",
    X"05f6",
    X"3968",
    X"163a",
    X"d075",
    X"de98",
    X"2446",
    X"2591",
    X"e6c1",
    X"d5ea",
    X"0c4c",
    X"2b0e",
    X"fe78",
    X"d9f1",
    X"f638",
    X"1fab",
    X"107e",
    X"e4b3",
    X"e8e8",
    X"167a",
    X"1b23",
    X"f384",
    X"e414",
    X"fe15",
    X"1d6a",
    X"104e",
    X"ec99",
    X"ede7",
    X"0d85",
    X"128d",
    X"f916",
    X"ed73",
    X"01d7",
    X"13d6",
    X"01ef",
    X"f1bc",
    X"fc09",
    X"0bf0",
    X"097c",
    X"f748",
    X"f6ed",
    X"041f",
    X"0a34",
    X"fc28",
    X"fc7a",
    X"f7e3",
    X"0968",
    X"efe6",
    X"2db1",
    X"fae9",
    X"95ce",
    X"9945",
    X"d84f",
    X"66b1",
    X"6870",
    X"17b0",
    X"91e5",
    X"a14c",
    X"b5dd",
    X"5b32",
    X"6702",
    X"3a7f",
    X"98b1",
    X"a4fe",
    X"a259",
    X"5396",
    X"5b85",
    X"6dbe",
    X"dc36",
    X"985c",
    X"9c08",
    X"32b1",
    X"6443",
    X"6bc2",
    X"fc21",
    X"9541",
    X"9a18",
    X"d48d",
    X"65ac",
    X"67f3",
    X"1c82",
    X"9161",
    X"a3a4",
    X"b13a",
    X"610a",
    X"5ad5",
    X"64e8",
    X"bfeb",
    X"9a8c",
    X"abed",
    X"4bbd",
    X"5a36",
    X"68e2",
    X"e246",
    X"9bdd",
    X"9ef1",
    X"2cd0",
    X"613e",
    X"66d0",
    X"fdfa",
    X"98da",
    X"9f1c",
    X"d260",
    X"607a",
    X"63c6",
    X"1f26",
    X"96ae",
    X"a72e",
    X"b3d6",
    X"5c70",
    X"5745",
    X"6172",
    X"c364",
    X"9db8",
    X"adee",
    X"4b08",
    X"5a29",
    X"693d",
    X"e8ae",
    X"a54a",
    X"a25c",
    X"f070",
    X"5c71",
    X"5bf6",
    X"02d4",
    X"a3ae",
    X"ae62",
    X"d64d",
    X"4fca",
    X"52f8",
    X"1d83",
    X"ac41",
    X"ba02",
    X"c2ff",
    X"4893",
    X"4172",
    X"4a00",
    X"d564",
    X"b5a9",
    X"c0ec",
    X"328b",
    X"4067",
    X"4a64",
    X"ee1f",
    X"bdf5",
    X"bbed",
    X"f3e8",
    X"427f",
    X"3de7",
    X"0327",
    X"bf28",
    X"c8ac",
    X"dddb",
    X"3b14",
    X"3337",
    X"32d8",
    X"d92a",
    X"cc99",
    X"da57",
    X"2dfa",
    X"2d54",
    X"3113",
    X"e770",
    X"cff0",
    X"d6c6",
    X"1d7f",
    X"2857",
    X"2d83",
    X"f635",
    X"d85a",
    X"d7f5",
    X"f7cf",
    X"2847",
    X"2395",
    X"02c1",
    X"dc99",
    X"e0dd",
    X"edd2",
    X"1bf6",
    X"1cca",
    X"0bd9",
    X"e198",
    X"e64a",
    X"e8d8",
    X"1af3",
    X"186d",
    X"1c09",
    X"f2ff",
    X"e8c0",
    X"e574",
    X"0259",
    X"1b45",
    X"1ae7",
    X"fbbd",
    X"e949",
    X"ebf4",
    X"fa2a",
    X"15ca",
    X"11df",
    X"018c",
    X"ee53",
    X"f0a3",
    X"f604",
    X"1088",
    X"0ed2",
    X"0e60",
    X"f4e2",
    X"f5b1",
    X"f866",
    X"0b10",
    X"0b07",
    X"0caa",
    X"fbb8",
    X"f599",
    X"f658",
    X"04e2",
    X"078b",
    X"094a",
    X"fe17",
    X"fba1",
    X"f5e0",
    X"11a6",
    X"b980",
    X"96f2",
    X"37e1",
    X"713c",
    X"e88e",
    X"869e",
    X"f431",
    X"76ca",
    X"2dd1",
    X"a4a2",
    X"9a13",
    X"2444",
    X"78c3",
    X"f40a",
    X"87c1",
    X"ee6f",
    X"75c4",
    X"2d7e",
    X"97c8",
    X"b2f7",
    X"5372",
    X"60f7",
    X"c269",
    X"9147",
    X"242f",
    X"779d",
    X"fc47",
    X"8934",
    X"e454",
    X"6a75",
    X"4181",
    X"b8ac",
    X"9554",
    X"1030",
    X"75ec",
    X"0710",
    X"8e33",
    X"dd8e",
    X"6ab4",
    X"3d5a",
    X"a768",
    X"a8bb",
    X"41c3",
    X"6700",
    X"d507",
    X"90df",
    X"12f0",
    X"73eb",
    X"131d",
    X"99c0",
    X"d4cc",
    X"5dda",
    X"3d31",
    X"babb",
    X"a5db",
    X"fc9b",
    X"6259",
    X"15bb",
    X"a994",
    X"d7a1",
    X"4b9a",
    X"3997",
    X"c7f5",
    X"b839",
    X"23f9",
    X"4db5",
    X"eeff",
    X"b2dc",
    X"fde7",
    X"4c9d",
    X"137b",
    X"c7b3",
    X"c5ff",
    X"1cb2",
    X"3634",
    X"d6e2",
    X"cbfb",
    X"071d",
    X"39e6",
    X"1545",
    X"cf7d",
    X"e01c",
    X"25f9",
    X"275f",
    X"e87b",
    X"d548",
    X"0da8",
    X"2a70",
    X"fdd4",
    X"d9a9",
    X"f728",
    X"204b",
    X"10ca",
    X"e852",
    X"e2b3",
    X"08e7",
    X"2097",
    X"0256",
    X"e38f",
    X"ffa0",
    X"1d3c",
    X"1016",
    X"ec19",
    X"ef02",
    X"0ea6",
    X"1215",
    X"f8d2",
    X"ed6a",
    X"0246",
    X"13c0",
    X"0147",
    X"f17b",
    X"fd09",
    X"0c05",
    X"0a29",
    X"f9c1",
    X"f4ab",
    X"ff9c",
    X"09db",
    X"0294",
    X"f990",
    X"fb69",
    X"0078",
    X"ff3f",
    X"0136",
    X"fdfc",
    X"0338",
    X"fdbb",
    X"0e05",
    X"259b",
    X"9c25",
    X"a192",
    X"9f8d",
    X"9f74",
    X"340b",
    X"6e96",
    X"53ab",
    X"728e",
    X"fa1e",
    X"8f83",
    X"a96a",
    X"9678",
    X"b81a",
    X"4f77",
    X"65d4",
    X"5b02",
    X"66fe",
    X"dcb0",
    X"9176",
    X"aaef",
    X"913e",
    X"d419",
    X"6121",
    X"453f",
    X"519e",
    X"8000",
    X"eb1c",
    X"7c1d",
    X"9449",
    X"9de6",
    X"2165",
    X"6771",
    X"38fd",
    X"6ced",
    X"5139",
    X"abba",
    X"8412",
    X"9046",
    X"a591",
    X"3cd2",
    X"6e2a",
    X"3535",
    X"6f12",
    X"372b",
    X"9d5f",
    X"8ce4",
    X"88e1",
    X"c5e5",
    X"6381",
    X"5891",
    X"3e15",
    X"7662",
    X"0277",
    X"8a96",
    X"91f1",
    X"8e50",
    X"f7ef",
    X"7065",
    X"40d2",
    X"48b3",
    X"5e91",
    X"e2e5",
    X"996c",
    X"9131",
    X"8c2f",
    X"1f8d",
    X"7982",
    X"3a61",
    X"55ca",
    X"4d57",
    X"c61b",
    X"959c",
    X"8aa6",
    X"a32f",
    X"44f7",
    X"6dde",
    X"386a",
    X"64e1",
    X"173d",
    X"a5ae",
    X"9852",
    X"8b57",
    X"be73",
    X"5a38",
    X"6c81",
    X"4001",
    X"6417",
    X"183e",
    X"9d5e",
    X"a322",
    X"879b",
    X"ed86",
    X"7023",
    X"5435",
    X"49aa",
    X"5bd5",
    X"e695",
    X"9edf",
    X"a92c",
    X"9261",
    X"ffbd",
    X"7097",
    X"4880",
    X"509b",
    X"4177",
    X"c3f6",
    X"b0d3",
    X"aa96",
    X"a7d6",
    X"27e8",
    X"6102",
    X"3d0d",
    X"5238",
    X"1c31",
    X"b412",
    X"b9f0",
    X"a7a8",
    X"c41a",
    X"476c",
    X"513e",
    X"3c24",
    X"4a03",
    X"f449",
    X"b812",
    X"c4ea",
    X"ac1a",
    X"e451",
    X"477e",
    X"43a7",
    X"3bc8",
    X"42c4",
    X"f2e7",
    X"bafb",
    X"c92e",
    X"b73d",
    X"00aa",
    X"4778",
    X"33e0",
    X"403f",
    X"19df",
    X"c988",
    X"c9a9",
    X"c9f8",
    X"c472",
    X"0a87",
    X"4103",
    X"2cf4",
    X"4ced",
    X"0b39",
    X"b72e",
    X"ba40",
    X"d8c2",
    X"eb34",
    X"3582",
    X"27fb",
    X"114e",
    X"1bb7",
    X"ff3c",
    X"e525",
    X"f1ec",
    X"cb55",
    X"d240",
    X"107f",
    X"20bf",
    X"4053",
    X"3e11",
    X"fc6d",
    X"c16f",
    X"c9c3",
    X"c7ca",
    X"171b",
    X"3e7c",
    X"357d",
    X"10df",
    X"01d6",
    X"cb95",
    X"e84b",
    X"fbec",
    X"f65d",
    X"f23b",
    X"07d3",
    X"fd32",
    X"1fc6",
    X"1fc2",
    X"fa44",
    X"f1e7",
    X"cab9",
    X"d166",
    X"0dfb",
    X"3790",
    X"2e46",
    X"2f24",
    X"f004",
    X"cc92",
    X"dc15",
    X"fef6",
    X"05a8",
    X"2dd1",
    X"062c",
    X"0341",
    X"fba5",
    X"f6d0",
    X"036a",
    X"06ce",
    X"de5a",
    X"e2d4",
    X"fbb6",
    X"118f",
    X"2b31",
    X"255f",
    X"000e",
    X"d668",
    X"da3f",
    X"e7f6",
    X"1da7",
    X"2842",
    X"215c",
    X"fd3a",
    X"f265",
    X"d92f",
    X"0346",
    X"0cc0",
    X"0b26",
    X"f9d2",
    X"f9a8",
    X"f4af",
    X"12c7",
    X"154e",
    X"086d",
    X"eff1",
    X"e254",
    X"e7d1",
    X"0741",
    X"1b11",
    X"1994",
    X"112f",
    X"ebf9",
    X"e6f5",
    X"f529",
    X"0ac7",
    X"0c8f",
    X"0ceb",
    X"f23c",
    X"ef0b",
    X"f6bd",
    X"049f",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000",
	X"0000"

  
 

);
constant ROM2 : rom_type :=
(

   X"ff9d",
    X"d2c4",
    X"f9ea",
    X"f73e",
    X"ee39",
    X"ed68",
    X"fc1c",
    X"f0c5",
    X"297e",
    X"fbe9",
    X"44e1",
    X"064f",
    X"2624",
    X"f409",
    X"ef34",
    X"e789",
    X"c9af",
    X"01e7",
    X"d6b2",
    X"3b26",
    X"f5c3",
    X"4a4a",
    X"00be",
    X"2b18",
    X"ebef",
    X"e947",
    X"e75d",
    X"c400",
    X"04c4",
    X"cdc9",
    X"2c0f",
    X"04b1",
    X"3b0e",
    X"1e83",
    X"1549",
    X"15f9",
    X"db04",
    X"f2b7",
    X"b6f6",
    X"fefe",
    X"d47c",
    X"2d1b",
    X"0265",
    X"3877",
    X"267d",
    X"1811",
    X"1353",
    X"d582",
    X"0230",
    X"b7be",
    X"f462",
    X"db23",
    X"139e",
    X"15fc",
    X"19d2",
    X"3879",
    X"05ea",
    X"2a1d",
    X"cf60",
    X"0147",
    X"b43d",
    X"0048",
    X"cfbf",
    X"1a6a",
    X"0d58",
    X"23d4",
    X"f20b",
    X"de43",
    X"da44",
    X"c9d9",
    X"f1d5",
    X"d634",
    X"2cb6",
    X"fdae",
    X"4da2",
    X"0c27",
    X"353d",
    X"02d0",
    X"ef36",
    X"e0e9",
    X"bfea",
    X"f82f",
    X"c5b2",
    X"2810",
    X"fdc0",
    X"4188",
    X"20fb",
    X"2947",
    X"2209",
    X"e6e5",
    X"f4f3",
    X"b219",
    X"f8b6",
    X"c710",
    X"2011",
    X"f842",
    X"2e3e",
    X"264a",
    X"2105",
    X"2334",
    X"e021",
    X"0432",
    X"b897",
    X"fefa",
    X"c48f",
    X"129b",
    X"0185",
    X"2dea",
    X"287a",
    X"1685",
    X"24ca",
    X"e363",
    X"f2eb",
    X"c225",
    X"ec20",
    X"cbab",
    X"f28e",
    X"0998",
    X"03e7",
    X"4739",
    X"0988",
    X"4fc0",
    X"ec8d",
    X"1795",
    X"c120",
    X"e551",
    X"c851",
    X"ea39",
    X"0814",
    X"0ba4",
    X"46a3",
    X"0812",
    X"4466",
    X"d003",
    X"034d",
    X"b474",
    X"e600",
    X"c44f",
    X"ec81",
    X"1d51",
    X"0e88",
    X"511d",
    X"1132",
    X"445f",
    X"e434",
    X"070f",
    X"c595",
    X"d714",
    X"d416",
    X"e482",
    X"1ac1",
    X"ff71",
    X"4eac",
    X"0f17",
    X"41e4",
    X"ead6",
    X"0636",
    X"cd20",
    X"cec3",
    X"e1c6",
    X"d484",
    X"1d68",
    X"f902",
    X"49e2",
    X"10c3",
    X"3aa7",
    X"0912",
    X"f914",
    X"ea1c",
    X"ca2a",
    X"f181",
    X"c791",
    X"1bd0",
    X"f298",
    X"4176",
    X"129f",
    X"3209",
    X"efb2",
    X"ea1e",
    X"cd05",
    X"d005",
    X"e3c0",
    X"dd25",
    X"212a",
    X"04f6",
    X"4594",
    X"1dac",
    X"32f8",
    X"fd9c",
    X"eedb",
    X"df8b",
    X"ca65",
    X"ea8d",
    X"d4a4",
    X"2308",
    X"0490",
    X"43df",
    X"15a6",
    X"2d84",
    X"090d",
    X"e4dc",
    X"ddda",
    X"c506",
    X"f1f6",
    X"cee4",
    X"1895",
    X"05b5",
    X"3ad6",
    X"21e6",
    X"2364",
    X"0aef",
    X"eec4",
    X"ea39",
    X"c9c1",
    X"f139",
    X"d60d",
    X"1d81",
    X"037a",
    X"355d",
    X"1afc",
    X"2014",
    X"154f",
    X"e4a1",
    X"f287",
    X"cdad",
    X"f531",
    X"d0d2",
    X"072f",
    X"083f",
    X"1e2f",
    X"2c65",
    X"13ab",
    X"28f8",
    X"eb15",
    X"0711",
    X"ccd4",
    X"f309",
    X"dacc",
    X"05fb",
    X"00e9",
    X"1ab9",
    X"24ed",
    X"139c",
    X"1a3d",
    X"e540",
    X"fbcb",
    X"d011",
    X"f258",
    X"daff",
    X"01e5",
    X"0d2d",
    X"14c5",
    X"2bb8",
    X"0c9e",
    X"2822",
    X"eb45",
    X"00af",
    X"d74d",
    X"eede",
    X"e373",
    X"f980",
    X"0e25",
    X"0d68",
    X"2a74",
    X"0e1f",
    X"20b1",
    X"eda7",
    X"fff7",
    X"d9aa",
    X"edb0",
    X"e2f1",
    X"f053",
    X"00f0",
    X"f4f3",
    X"1312",
    X"f901",
    X"2107",
    X"101d",
    X"1e60",
    X"0724",
    X"0d7b",
    X"041f",
    X"dbb8",
    X"ef09",
    X"ded3",
    X"0264",
    X"e644",
    X"139b",
    X"1488",
    X"1fb3",
    X"097a",
    X"14dd",
    X"094c",
    X"dbb4",
    X"eb55",
    X"e25b",
    X"fdc3",
    X"e17a",
    X"07be",
    X"16bb",
    X"21f8",
    X"11d5",
    X"1018",
    X"0e2b",
    X"ec8c",
    X"f006",
    X"da61",
    X"fca1",
    X"dfed",
    X"06bc",
    X"14ec",
    X"1b44",
    X"1566",
    X"1313",
    X"15d0",
    X"ed04",
    X"f038",
    X"dd51",
    X"f89e",
    X"e11e",
    X"f7ec",
    X"080e",
    X"1df8",
    X"1a11",
    X"1297",
    X"1e61",
    X"f809",
    X"edc9",
    X"d828",
    X"f719",
    X"dffc",
    X"f372",
    X"115e",
    X"1f07",
    X"1ac3",
    X"1266",
    X"1cb4",
    X"f59c",
    X"e97e",
    X"d0a4",
    X"e76b",
    X"dc62",
    X"e744",
    X"1f15",
    X"2df9",
    X"34a0",
    X"1d14",
    X"3562",
    X"094a",
    X"e699",
    X"c57a",
    X"dc06",
    X"d59f",
    X"dfd5",
    X"0c61",
    X"28b1",
    X"3251",
    X"1f40",
    X"34d8",
    X"04b6",
    X"e07f",
    X"c925",
    X"d872",
    X"da07",
    X"d60d",
    X"0ca1",
    X"2cdc",
    X"37ad",
    X"223b",
    X"3394",
    X"157d",
    X"f030",
    X"cbd1",
    X"d1e9",
    X"d8c8",
    X"d422",
    X"0855",
    X"2891",
    X"3940",
    X"2272",
    X"3587",
    X"1c9a",
    X"e9c5",
    X"c69a",
    X"c98c",
    X"d00e",
    X"c217",
    X"fb05",
    X"3074",
    X"47ac",
    X"316f",
    X"40ab",
    X"308d",
    X"f2b3",
    X"c349",
    X"be6b",
    X"cd8b",
    X"bc47",
    X"ef89",
    X"2a88",
    X"493e",
    X"3567",
    X"42e0",
    X"2bc4",
    X"eed2",
    X"be6b",
    X"bcc3",
    X"c810",
    X"ba33",
    X"ea9c",
    X"2b95",
    X"4b13",
    X"3c8d",
    X"431f",
    X"3ffa",
    X"0530",
    X"c829",
    X"b8c1",
    X"c801",
    X"bb45",
    X"dd9d",
    X"1487",
    X"47f3",
    X"3f00",
    X"4164",
    X"3f67",
    X"0194",
    X"c5d7",
    X"b527",
    X"ccad",
    X"b9bc",
    X"dc13",
    X"2105",
    X"4b64",
    X"3c5e",
    X"3e34",
    X"4600",
    X"0f72",
    X"cc19",
    X"b619",
    X"cd4d",
    X"bdf6",
    X"cff2",
    X"0b75",
    X"43d7",
    X"3de2",
    X"38f2",
    X"44d3",
    X"1abf",
    X"d429",
    X"b772",
    X"c8da",
    X"bebf",
    X"ce2b",
    X"0aa7",
    X"40ea",
    X"4714",
    X"38fe",
    X"4990",
    X"1a82",
    X"d646",
    X"b3c8",
    X"c6f4",
    X"c06f",
    X"c695",
    X"fc26",
    X"3b25",
    X"447c",
    X"3995",
    X"4457",
    X"23e7",
    X"e2ac",
    X"b769",
    X"c54d",
    X"c2b8",
    X"c5b4",
    X"fe3d",
    X"3951",
    X"478e",
    X"3635",
    X"4525",
    X"2f4d",
    X"ed9c",
    X"b7e4",
    X"c5be",
    X"c741",
    X"c345",
    X"ef1c",
    X"2eec",
    X"4605",
    X"3632",
    X"3fb2",
    X"2cb6",
    X"ed03",
    X"b9ef",
    X"c50f",
    X"c689",
    X"bfd7",
    X"ecf9",
    X"317c",
    X"4806",
    X"36fa",
    X"414e",
    X"360f",
    X"f9f3",
    X"c13e",
    X"c39e",
    X"c9c6",
    X"c1c2",
    X"e482",
    X"1f44",
    X"4289",
    X"3665",
    X"3af8",
    X"3137",
    X"fa2e",
    X"c5f0",
    X"c3a5",
    X"caf1",
    X"c29c",
    X"e51b",
    X"2402",
    X"4494",
    X"3643",
    X"3863",
    X"3473",
    X"0595",
    X"caec",
    X"c59a",
    X"d060",
    X"c6eb",
    X"e02c",
    X"1322",
    X"3b2e",
    X"333b",
    X"343f",
    X"338d",
    X"0d5c",
    X"d525",
    X"c7d0",
    X"d4ac",
    X"ca93",
    X"df95",
    X"14f7",
    X"3a17",
    X"2f75",
    X"2fb8",
    X"3273",
    X"0c2c",
    X"d524",
    X"cb69",
    X"d8ef",
    X"d038",
    X"ddf4",
    X"0b91",
    X"3179",
    X"2f0c",
    X"2a29",
    X"2e79",
    X"10c1",
    X"e0f6",
    X"cf7d",
    X"da3e",
    X"d28c",
    X"e0f7",
    X"0b4e",
    X"2d7e",
    X"2be9",
    X"25f3",
    X"2e61",
    X"1306",
    X"dedc",
    X"d2d4",
    X"dcae",
    X"d8e4",
    X"dee6",
    X"00a7",
    X"28d6",
    X"289b",
    X"24aa",
    X"28d8",
    X"14ea",
    X"eb9c",
    X"d5c5",
    X"de56",
    X"d995",
    X"de85",
    X"0315",
    X"2655",
    X"2824",
    X"214f",
    X"2921",
    X"180a",
    X"f19b",
    X"d7ac",
    X"e0ba",
    X"df04",
    X"df9f",
    X"fa49",
    X"1e24",
    X"256e",
    X"2120",
    X"240c",
    X"0ce3",
    X"deb1",
    X"c526",
    X"c544",
    X"c1e1",
    X"e0ed",
    X"2320",
    X"44e1",
    X"45e5",
    X"546f",
    X"3821",
    X"fe9c",
    X"bf66",
    X"bb7b",
    X"b07d",
    X"b8fb",
    X"decc",
    X"213f",
    X"4786",
    X"49a5",
    X"5775",
    X"2f71",
    X"f7f8",
    X"c1ff",
    X"bec6",
    X"b1a0",
    X"affa",
    X"dc9c",
    X"2234",
    X"4c87",
    X"434a",
    X"5233",
    X"3b78",
    X"08e6",
    X"c5cc",
    X"bd3f",
    X"b838",
    X"b3bf",
    X"dd58",
    X"1bfd",
    X"4874",
    X"3f01",
    X"5042",
    X"3d59",
    X"0d66",
    X"ccb1",
    X"bb7e",
    X"bfba",
    X"b00f",
    X"d374",
    X"06de",
    X"4a80",
    X"3f31",
    X"4df2",
    X"3c46",
    X"18f6",
    X"ddc4",
    X"b38c",
    X"c5d6",
    X"b11e",
    X"d4e7",
    X"04b0",
    X"437f",
    X"3c1e",
    X"4a01",
    X"3caa",
    X"1434",
    X"d57e",
    X"b3a0",
    X"bdd7",
    X"a663",
    X"db80",
    X"096c",
    X"5ce8",
    X"4290",
    X"555c",
    X"5281",
    X"17f3",
    X"d69a",
    X"9e1f",
    X"c11a",
    X"99ff",
    X"c8c8",
    X"00c7",
    X"512e",
    X"4aae",
    X"52b1",
    X"54ba",
    X"1540",
    X"d1df",
    X"a45d",
    X"c1fc",
    X"9cd8",
    X"c049",
    X"0496",
    X"4f3f",
    X"50f0",
    X"463b",
    X"6532",
    X"1d95",
    X"eb4e",
    X"9ff8",
    X"c106",
    X"a690",
    X"bb67",
    X"f0b4",
    X"3738",
    X"5b57",
    X"3f23",
    X"67bb",
    X"170d",
    X"f0d7",
    X"9825",
    X"b80c",
    X"9836",
    X"ac20",
    X"f74d",
    X"40de",
    X"70af",
    X"4590",
    X"7a38",
    X"2aab",
    X"f1cc",
    X"96b2",
    X"aaa6",
    X"9e26",
    X"9e9a",
    X"f5c2",
    X"36a4",
    X"7a81",
    X"43ea",
    X"8000",
    X"2ce8",
    X"ee69",
    X"92f7",
    X"a590",
    X"9e77",
    X"97f4",
    X"ee27",
    X"2ac6",
    X"7f49",
    X"49fe",
    X"7c70",
    X"4b67",
    X"018c",
    X"ae5c",
    X"9123",
    X"b066",
    X"8534",
    X"fbfc",
    X"1744",
    X"7e8d",
    X"4a84",
    X"7899",
    X"4a60",
    X"f76e",
    X"b52b",
    X"9118",
    X"ba57",
    X"8221",
    X"ebde",
    X"1747",
    X"7fde",
    X"449e",
    X"6f2d",
    X"55c4",
    X"0428",
    X"c493",
    X"8828",
    X"c7cb",
    X"82f3",
    X"dd09",
    X"04ef",
    X"6785",
    X"528c",
    X"5cad",
    X"5f89",
    X"fdc2",
    X"ded8",
    X"89ee",
    X"c77f",
    X"7fef",
    X"e86f",
    X"0d31",
    X"6056",
    X"56d4",
    X"53f1",
    X"6e0d",
    X"fc5f",
    X"db3d",
    X"86b4",
    X"c913",
    X"8a45",
    X"c995",
    X"09aa",
    X"51ee",
    X"63a2",
    X"49d6",
    X"6db7",
    X"01f7",
    X"ece8",
    X"8907",
    X"c48d",
    X"9105",
    X"cf02",
    X"0f8b",
    X"4ae4",
    X"6930",
    X"3f53",
    X"7885",
    X"0caf",
    X"f34e",
    X"9587",
    X"bcdc",
    X"a45b",
    X"acf7",
    X"0d57",
    X"2cc0",
    X"6ef2",
    X"3464",
    X"78f6",
    X"0edc",
    X"f0f6",
    X"a3d7",
    X"b61e",
    X"a8ae",
    X"af16",
    X"095a",
    X"2fff",
    X"71f7",
    X"362b",
    X"7812",
    X"11f5",
    X"ffe7",
    X"a10b",
    X"b032",
    X"b551",
    X"a418",
    X"0b08",
    X"1ad9",
    X"6bc5",
    X"3838",
    X"692b",
    X"2493",
    X"ed74",
    X"b90a",
    X"ab6c",
    X"bcd5",
    X"9e20",
    X"0764",
    X"1fe5",
    X"6a02",
    X"3310",
    X"66d0",
    X"3520",
    X"fb83",
    X"bda4",
    X"a6c5",
    X"cc1b",
    X"9c87",
    X"0042",
    X"0e65",
    X"628f",
    X"358c",
    X"58fa",
    X"328b",
    X"f729",
    X"d3f0",
    X"a804",
    X"cf84",
    X"a115",
    X"04b5",
    X"0d91",
    X"5339",
    X"3513",
    X"4e15",
    X"3d33",
    X"f791",
    X"d88e",
    X"ae19",
    X"d9b1",
    X"a952",
    X"f4f4",
    X"08e4",
    X"4950",
    X"35b2",
    X"3f0b",
    X"368f",
    X"fa5a",
    X"e599",
    X"ae7a",
    X"de4d",
    X"ad40",
    X"f4db",
    X"08fc",
    X"4269",
    X"3231",
    X"3ba1",
    X"3be4",
    X"f5d4",
    X"e389",
    X"b77a",
    X"e29a",
    X"b52d",
    X"eb1d",
    X"0aba",
    X"2e40",
    X"37e2",
    X"2c87",
    X"4046",
    X"fb26",
    X"ef56",
    X"bc1b",
    X"df9b",
    X"ba9b",
    X"e954",
    X"0992",
    X"2b1a"
 
);

   signal test_mode: std_logic :='1';                  -- Output a sine wave
   signal audio_request: std_logic:='1';
   signal data: unsigned(15 downto 0):= "0000000000000000";
   signal audio_clock : unsigned(1 downto 0) := "00";
   signal data_w : unsigned (15 downto 0);
   signal output_flag : std_logic:='0';
   signal output_flag2 : std_logic := '0';
   signal num: integer;
---------------
begin
process (clk)
  begin
    if rising_edge(clk) then
      audio_clock <= audio_clock + "1";
    end if;
  end process;
  
    -- LRCK divider 
    -- Audio chip main clock is 18.432MHz / Sample rate 48KHz
    -- Divider is 18.432 MHz / 48KHz = 192 (X"C0")   10b
    -- Left justify mode set by I2C controller
    
  process (audio_clock(1))
  begin
    if rising_edge(audio_clock(1)) then
      if reset_n = '0' then 
        lrck_divider <= (others => '0');
      elsif lrck_divider = "100001000000" THEN--X"BF"  then        -- "C0" minus 1
           lrck_divider <= "000000000000";
      else 
        lrck_divider <= lrck_divider + 1;
      end if;
    end if;   
  end process;

  process (clk)        --changed for clock frequency
  begin
    if rising_edge(audio_clock(1)) then      
      if reset_n = '0' then 
        bclk_divider <= (others => '0');
      elsif bclk_divider = X"B" or set_lrck = '1'  then  
        bclk_divider <= X"0";
      else 
        bclk_divider <= bclk_divider + 1;
      end if;
    end if;
  end process;

  set_lrck <= '1' when lrck_divider = X"BF" else '0';
    
  process (audio_clock(1))
  begin
    if rising_edge(audio_clock(1)) then
      if reset_n = '0' then
        lrck <= '0';
      elsif set_lrck = '1' then 
        lrck <= not lrck;
      end if;
    end if;
  end process;
    
  -- BCLK divider
  set_bclk <= '1' when bclk_divider(3 downto 0) = "0101" else '0';
  clr_bclk <= '1' when bclk_divider(3 downto 0) = "1011" else '0';
  
  process (audio_clock(1))
  begin
    if rising_edge(audio_clock(1)) then
      if reset_n = '0' then
        bclk <= '0';
      elsif set_lrck = '1' or clr_bclk = '1' then
        bclk <= '0';
      elsif set_bclk = '1' then 
        bclk <= '1';
      end if;
    end if;
  end process;

  -- Audio data shift output
  process (audio_clock(1))
  begin
    if rising_edge(audio_clock(1)) then
      if reset_n = '0' then
        shift_out <= (others => '0');
      elsif set_lrck = '1' then
        if test_mode = '1' then 
          shift_out <= sin_out;
        else 
          shift_out <= data;
        end if;
      elsif clr_bclk = '1' then 
        shift_out <= shift_out (14 downto 0) & '0';
      end if;
    end if;   
  end process;

    -- Audio outputs
    
    AUD_ADCLRCK  <= lrck;          
    AUD_DACLRCK  <= lrck;          
    AUD_DACDAT   <= shift_out(15); 
    AUD_BCLK     <= bclk;          

    -- Self test with Sin wave
--process(audio_clock(1))
--begin
--    if rising_edge(audio_clock(1)) then
--      if chipselect = '1' then         
--          if address = "11111"  then		         -- vertical 4
--             if write = '1' then
--              data_w<= writedata(15 downto 0) ;
--              num<=to_integer(data_w);
--              output_flag<='1';
--             end if;
--          end if;
--      end if;
--      if sin_counter =x"544" then
--         output_flag<1='0';
--      end if;
--    end if;
--end process;

    process(clk)      
    begin
      if rising_edge(clk) then
----------------------------------------------
    if chipselect = '1' then         
          if address = "11111"  then		         -- vertical 4
             if write = '1' then
              data_w <= writedata(15 downto 0) ;
              num<=to_integer(data_w);
              output_flag <= '1'; 
              
             end if;
          end if;
      end if; 
    if output_flag2 = '1' then
       output_flag <= '0';
    end if;
     end if;
  end process;
--------------------------------------
process(audio_clock(1))
   begin
    if rising_edge(audio_clock(1)) then  
      if reset_n = '0' then 
            sin_counter <= (others => '0');
        elsif output_flag = '1' then
         if lrck_lat = '1' and lrck = '0'  then  
--          if sin_counter = X"544" then 
--            sin_counter <= X"000";
--          else  
            sin_counter <= sin_counter + 1;
--          end if;
        end if;
      end if;
--------------------------------
      if sin_counter = x"398" then
         output_flag2 <= '1'; sin_counter <= x"000";
      elsif sin_counter /= x"398" then
         output_flag2 <= '0';
      end if;
----------------------------------
    end if;
    end process;

    process(audio_clock(1))
    begin
      if rising_edge(audio_clock(1)) then
        lrck_lat <= lrck;
      end if;
    end process;

    process (audio_clock(1)) 
    begin
      if rising_edge(audio_clock(1)) then 
        if lrck_lat = '1' and lrck = '0' then
          audio_request <= '1';
        else 
          audio_request <= '0';
        end if;
      end if;
    end process;
    with num select sin_out<=
	ROM1(TO_INTEGER(SIN_COUNTER)) when 1,
	ROM2(TO_INTEGER(SIN_COUNTER)) when 2,
	null when others;
--  with sin_counter select sin_out <=
--    X"0000" when "000000",
--    X"10b4" when "000001",
--    X"2120" when "000010",
--    X"30fb" when "000011",
--    X"3fff" when "000100",
--    X"4deb" when "000101",
--    X"5a81" when "000110",
--    X"658b" when "000111",
--    X"6ed9" when "001000",
--    X"7640" when "001001",
--    X"7ba2" when "001010",
--    X"7ee6" when "001011",
--    X"7fff" when "001100",
--    X"7ee6" when "001101",
--    X"7ba2" when "001110",
--    X"7640" when "001111",
--    X"6ed9" when "010000",
--    X"658b" when "010001",
--    X"5a81" when "010010",
--    X"4deb" when "010011",
--    X"3fff" when "010100",
--    X"30fb" when "010101",
--    X"2120" when "010110",
--    X"10b4" when "010111",
--    X"0000" when "011000",
--    X"ef4b" when "011001",
--    X"dee0" when "011010",
--    X"cf05" when "011011",
--    X"c001" when "011100",
--    X"b215" when "011101",
--    X"a57e" when "011110",
--    X"9a74" when "011111",
--    X"9127" when "100000",
--    X"89bf" when "100001",
--    X"845d" when "100010",
--    X"8119" when "100011",
--    X"8000" when "100100",
--    X"8119" when "100101",
--    X"845d" when "100110",
--    X"89bf" when "100111",
--    X"9127" when "101000",
--    X"9a74" when "101001",
--    X"a57e" when "101010",
--    X"b215" when "101011",
--    X"c000" when "101100",
--    X"cf05" when "101101",
--    X"dee0" when "101110",
--    X"ef4b" when "101111",
--    X"0000" when others;      

end architecture;


