--
-- circle generator
--

library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;


entity pong_circle is
	port (
		x,y				: in unsigned (9 downto 0);
		cx,cy			: in unsigned (9 downto 0);
		do				: out std_logic
	);
end pong_circle;

architecture rtl of pong_circle is
	type bitmap_t is array(0 to 14,0 to 14) of std_logic;
	constant bitmap : bitmap_t :=
	  ( "000001111100000",
	    "000111111111000",
        "001111111111100",
        "011111111111110",
        "011111111111110",
        "111111111111111",
        "111111111111111",
        "111111111111111",
        "111111111111111",
        "111111111111111",
        "011111111111110",
        "011111111111110",
        "001111111111100",
        "000111111111000",
        "000001111100000" );

    -- offset variables
	signal ix,iy : unsigned(9 downto 0);

	-- valid, in range of bitmap
	signal vx, vy : std_logic;
begin
	ix <= (x - (cx-7));
	iy <= (y - (cy-7));

	vx <= '1' when (x >= (cx-7) and x <= (cx+7)) else '0';
	vy <= '1' when (y >= (cy-7) and y <= (cy+7)) else '0';

	do <= vx and vy and bitmap(TO_INTEGER(ix),TO_INTEGER(iy));
end rtl;


--
-- paddle generator
--




library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;


entity pong_paddle is
	port (
		x,y,cx			: in unsigned (9 downto 0);
		a,b				: in unsigned (9 downto 0);
		do				: out std_logic
	);
end pong_paddle;

architecture rtl of pong_paddle is
  signal vx, vy : std_logic;
  signal vis : std_logic;
begin
  vx <= '1' when (x >= cx and x<(cx+5)) else '0';
  vy <= '1' when (y >= a and y <= b) else '0';

  vis <= '0' when (a = b) else '1';

  
  do <= vx and vy and vis;
end rtl;




--
-- Game Objects Generator
--

library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;

entity gobj_gen is
	port (
		--x,y 		: in unsigned (9 downto 0);
		visx, visy	: in unsigned (9 downto 0);
		ballx,
		bally,
		p1a,
		p1b,
		p2a,
		p2b			: in unsigned (9 downto 0);
		
		do			: out std_logic
		);
		
end gobj_gen;

architecture rtl of gobj_gen is
-- signals
  signal active_pixel,circle_pixel,padl1_pixel,padl2_pixel : std_logic;

begin


U1: entity work.pong_circle port map (
    x => visx,
    y => visy,
    cx => ballx, --"0001010000",
    cy => bally, -- "0001010000",
    do => circle_pixel
  );

  U2: entity work.pong_paddle port map (
    x => visx,
    y => visy,
    a => p1a, --"0011110000",
    b => p1b, --"0011111111",
    cx => "0000010100",	-- 20 pixel offset for left paddle
    do => padl1_pixel
  );

  U3: entity work.pong_paddle port map (
    x => visx,
    y => visy,
    a => p2a, --"0000001000",
    b => p2b, --"0000110000",
    cx => "1001100111",	-- VGA_X-(PADL_OFFSET+PADL_WIDTH)=640-20-5, give or take
    do => padl2_pixel
  );
  
  active_pixel <= circle_pixel or padl1_pixel or padl2_pixel; -- or bg_pixel;
  do <= active_pixel; 


end rtl;




--1001101111
-- Background Generator
--

library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;

entity bg_gen is
	port (
		x,y		: in unsigned (9 downto 0);
		--tx,ty	: in unsigned (9 downto 0);
		tile_map : in unsigned (255 downto 0);
		do		: out std_logic
		);
end bg_gen;

architecture rtl of bg_gen is
	
	--type bitmap_t is array(0 to 15,0 to 15) of std_logic;
	
	type bitmap_t is array (15 downto 0) of unsigned (15 downto 0);
	signal bitmap : bitmap_t;

		
		signal vx, vy : std_logic;
		signal bit_x,bit_y : unsigned(9 downto 0);
		signal current_line : unsigned (15 downto 0);
	--	signal count_x, count_y : unsigned (3 downto 0);
		
	begin

		bitmap (0) <= tile_map (255 downto 240);
		bitmap (1) <= tile_map (239 downto 224);
		bitmap (2) <= tile_map (223 downto 208);
		bitmap (3) <= tile_map (207 downto 192);
		bitmap (4) <= tile_map (191 downto 176);
		bitmap (5) <= tile_map (175 downto 160);
		bitmap (6) <= tile_map (159 downto 144);
		bitmap (7) <= tile_map (143 downto 128);
		bitmap (8) <= tile_map (127 downto 112);
		bitmap (9) <= tile_map (111 downto 96);
		bitmap (10) <= tile_map (95 downto 80);
		bitmap (11) <= tile_map (79 downto 64);
		bitmap (12) <= tile_map (63 downto 48);
		bitmap (13) <= tile_map (47 downto 32);
		bitmap (14) <= tile_map (31 downto 16);
		bitmap (15) <= tile_map (15 downto 0);
		
		current_line <= bitmap(TO_INTEGER(x(3 downto 0)));
		do <= '1' when current_line(TO_INTEGER(y(3 downto 0))) = '1' else '0';

--	do <= '1' when bitmap(TO_INTEGER(x(3 downto 0)),TO_INTEGER(y(3 downto 0)))='1' else '0';		

end rtl;




library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;


--
-- Foreground Generator
--

entity fg_gen is
	port (
			clk			: in std_logic;
			visx, visy 	: in unsigned (9 downto 0);
			rd_address 	: out unsigned (14 downto 0);
			data_in		: in unsigned (15 downto 0);
			do			: out std_logic
		);
		
end fg_gen;

architecture rtl of fg_gen is
	

	--signal byte_pos : unsigned (14 downto 0);
	signal bit_pos : unsigned (3 downto 0);
	
	-- new stuff
	
	signal data : unsigned (14 downto 0);
	--signal data_long : unsigned (19 downto 0);
	
	signal x,y : unsigned (14 downto 0);
	
	
	begin

	-- convert x,y to byte offset
	-- Position is calculated as => 80 * (visy) + (visx/8);
	
	--data_long <= (40 * visy) + (visx/16);
	
	x <= "000000000000000" + visx(9 downto 1);
	y <= "000000000000000" + visy(9 downto 1);
	
	data <= "000000000000000" + (((y sll 4) + (y sll 2)) + (x srl 4));
	
	--data <= data_long (14 downto 0);
	
	rd_address <= data;
	bit_pos <= x(3 downto 0); -- lower 4 bits of visx
	
	
	-- update on the clock
	
--	process (clk)
--		begin
--			if rising_edge(clk) then
--				if data_in(to_integer(bit_pos)) = '1' then
--					do <= '1';
--				else
--					do <= '0';
--				end if;
--			end if;
--		end process;
	
	do <= '1' when data_in(TO_INTEGER(bit_pos)) = '1' else '0';
	
end rtl;




--
-- Priority MUX
--

library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;

entity pmux is
port ( 
		en0, en1, en2				: in std_logic;
		-- incoming colors
		sc_color,
		bg_color,
		fg_color,
		go_color					: in unsigned (15 downto 0);
		vga_r, vga_g, vga_b			: out unsigned (9 downto 0);
		clk							: in std_logic
		);
end pmux;

architecture rtl of pmux is
-- signals
signal red, green, blue 	: unsigned (9 downto 0);

signal
	sc_r, bg_r, fg_r, go_r,
	sc_g, bg_g, fg_g, go_g,
	sc_b, bg_b, fg_b, go_b	: unsigned (9 downto 0);

begin

-- Screen Colors
sc_r <= sc_color(14 downto 10) & "11111" when sc_color(15) = '1' 
		 else sc_color(14 downto 10) & "00000";
sc_g <= sc_color(9 downto 5) & "11111" when sc_color(15) = '1' 
		 else sc_color(9 downto 5) & "00000";
sc_b <= sc_color(4 downto 0) & "11111" when sc_color(15) = '1' 
		 else sc_color(4 downto 0) & "00000";
	
-- Background Colors
bg_r <= bg_color(14 downto 10) & "11111" when bg_color(15) = '1' 
		 else bg_color(14 downto 10) & "00000";
bg_g <= bg_color(9 downto 5) & "11111" when bg_color(15) = '1' 
		 else bg_color(9 downto 5) & "00000";
bg_b <= bg_color(4 downto 0) & "11111" when bg_color(15) = '1' 
		 else bg_color(4 downto 0) & "00000";	

-- Foreground Colors
fg_r <= fg_color(14 downto 10) & "11111" when fg_color(15) = '1' 
		 else fg_color(14 downto 10) & "00000";
fg_g <= fg_color(9 downto 5) & "11111" when fg_color(15) = '1' 
		 else fg_color(9 downto 5) & "00000";
fg_b <= fg_color(4 downto 0) & "11111" when fg_color(15) = '1' 
		 else fg_color(4 downto 0) & "00000";

-- Game Object Colors
go_r <= go_color(14 downto 10) & "11111" when go_color(15) = '1' 
		 else go_color(14 downto 10) & "00000";
go_g <= go_color(9 downto 5) & "11111" when go_color(15) = '1' 
		 else go_color(9 downto 5) & "00000";
go_b <= go_color(4 downto 0) & "11111" when go_color(15) = '1' 
		 else go_color(4 downto 0) & "00000";	
		
		

red <= 	 go_r when en0 = '1' else -- game objects
		 fg_r when en1 = '1' else -- foreground
		 bg_r when en2 = '1' else -- background
		 sc_r;
		
green <= go_g when en0 = '1' else
		 fg_g when en1 = '1' else
		 bg_g when en2 = '1' else
		 sc_g;

blue <=  go_b when en0 = '1' else
		 fg_b when en1 = '1' else
		 bg_b when en2 = '1' else
		 sc_b;
		
	process (clk) begin
		if rising_edge(clk) then
			vga_r <= red;
			vga_g <= green;
			vga_b <= blue;
		end if;
	end process;
	
end rtl;


-------------------------------------------------------------------------------
--
-- Simple VGA raster display
--
-- Stephen A. Edwards
-- sedwards@cs.columbia.edu
--
-------------------------------------------------------------------------------
library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;

entity de2_vga_raster is
  
  port (
    reset : in std_logic;
    clk   : in std_logic;                    -- Should be 25.125 MHz

    -- circle position
    cx : in unsigned(9 downto 0);
    cy : in unsigned(9 downto 0);

	visx, visy : out unsigned(9 downto 0); -- visible screen x and y position
	pgen_vga_r, pgen_vga_g, pgen_vga_b : in unsigned (9 downto 0);
	
	vbi : out std_logic;		     -- anywhere in vertical blank interval
    
    -- vga interface

    VGA_CLK,                         -- Clock
    VGA_HS,                          -- H_SYNC
    VGA_VS,                          -- V_SYNC
    VGA_BLANK,                       -- BLANK
    VGA_SYNC : out std_logic;        -- SYNC
    VGA_R,                           -- Red[9:0]
    VGA_G,                           -- Green[9:0]
    VGA_B : out unsigned(9 downto 0) -- Blue[9:0]
    );

end de2_vga_raster;

architecture rtl of de2_vga_raster is
  
  -- Video parameters
  
  constant HTOTAL       : integer := 800;
  constant HSYNC        : integer := 96;
  constant HBACK_PORCH  : integer := 48;
  constant HACTIVE      : integer := 640;
  constant HFRONT_PORCH : integer := 16;
  
  constant VTOTAL       : integer := 525;
  constant VSYNC        : integer := 2;
  constant VBACK_PORCH  : integer := 33;
  constant VACTIVE      : integer := 480;
  constant VFRONT_PORCH : integer := 10;

  constant RECTANGLE_HSTART : integer := 100;
  constant RECTANGLE_HEND   : integer := 540;
  constant RECTANGLE_VSTART : integer := 100;
  constant RECTANGLE_VEND   : integer := 380;

  -- Signals for the video controller
  signal Hcount : unsigned(9 downto 0);  -- Horizontal position (0-800)
  signal Vcount : unsigned(9 downto 0);  -- Vertical position (0-524)
  signal EndOfLine, EndOfField : std_logic;
  signal visx_sig, visy_sig : unsigned(9 downto 0); -- visible screen x and y position

  signal vga_hblank, vga_hsync,
    vga_vblank, vga_vsync : std_logic;  -- Sync. signals

  signal active_pixel,circle_pixel,padl1_pixel,padl2_pixel : std_logic;

  --signal pgen_vga_r, pgen_vga_g, pgen_vga_b : unsigned (4 downto 0); -- 5-bit color output from pixel gen

  signal bg_pixel : std_logic;

begin

  -- Horizontal and vertical counters
  visx_sig <= b"0000000000" when Hcount<(HSYNC+HBACK_PORCH) else (Hcount - (HSYNC+HBACK_PORCH));
  visy_sig <= b"0000000000" when Vcount<(VSYNC + VBACK_PORCH+1) else (Vcount - (VSYNC+VBACK_PORCH+1));

  visx <= visx_sig;
  visy <= visy_sig;
  


  HCounter : process (clk)
  begin
    if rising_edge(clk) then      
      if reset = '1' then
        Hcount <= (others => '0');
      elsif EndOfLine = '1' then
        Hcount <= (others => '0');
      else
        Hcount <= Hcount + 1;
      end if;      
    end if;
  end process HCounter;

  EndOfLine <= '1' when Hcount = HTOTAL - 1 else '0';
  
  VCounter: process (clk)
  begin
    if rising_edge(clk) then      
      if reset = '1' then
        Vcount <= (others => '0');
      elsif EndOfLine = '1' then
        if EndOfField = '1' then
          Vcount <= (others => '0');
        else
          Vcount <= Vcount + 1;
        end if;
      end if;
    end if;
  end process VCounter;

  EndOfField <= '1' when Vcount = VTOTAL - 1 else '0';

  -- State machines to generate HSYNC, VSYNC, HBLANK, and VBLANK

  HSyncGen : process (clk)
  begin
    if rising_edge(clk) then     
      if reset = '1' or EndOfLine = '1' then
        vga_hsync <= '1';
      elsif Hcount = HSYNC - 1 then
        vga_hsync <= '0';
      end if;
    end if;
  end process HSyncGen;
  
  HBlankGen : process (clk)
  begin
    if rising_edge(clk) then
      if reset = '1' then
        vga_hblank <= '1';
      elsif Hcount = HSYNC + HBACK_PORCH then
        vga_hblank <= '0';
      elsif Hcount = HSYNC + HBACK_PORCH + HACTIVE then
        vga_hblank <= '1';
      end if;      
    end if;
  end process HBlankGen;

  VSyncGen : process (clk)
  begin
    if rising_edge(clk) then
      if reset = '1' then
        vga_vsync <= '1';
      elsif EndOfLine ='1' then
        if EndOfField = '1' then
          vga_vsync <= '1';
        elsif Vcount = VSYNC - 1 then
          vga_vsync <= '0';
        end if;
      end if;      
    end if;
  end process VSyncGen;

  VBlankGen : process (clk)
  begin
    if rising_edge(clk) then    
      if reset = '1' then
        vga_vblank <= '1';
      elsif EndOfLine = '1' then
        if Vcount = VSYNC + VBACK_PORCH - 1 then
          vga_vblank <= '0';
        elsif Vcount = VSYNC + VBACK_PORCH + VACTIVE - 1 then
          vga_vblank <= '1';
        end if;
      end if;
    end if;
  end process VBlankGen;
   
-- for testing purposes
active_pixel <= '1'; 

  -- Registered video signals going to the video DAC

  VideoOut: process (clk, reset)
  begin
    if reset = '1' then
      VGA_R <= "0000000000";
      VGA_G <= "0000000000";
      VGA_B <= "0000000000";
    elsif clk'event and clk = '1' then
      if (vga_hblank='1' or vga_vblank='1') then
        VGA_R <= "0000000000";
        VGA_G <= "0000000000";
        VGA_B <= "0000000000";    
      elsif (visx_sig <= 640) and (visy_sig <= 480) then
        VGA_R <= pgen_vga_r;
        VGA_G <= pgen_vga_g;
        VGA_B <= pgen_vga_b;
      else
        VGA_R <= "1111111111";
        VGA_G <= "0000000000";
        VGA_B <= "0000000000";    
      end if;
    end if;
  end process VideoOut;

  VGA_CLK <= clk;
  VGA_HS <= not vga_hsync;
  VGA_VS <= not vga_vsync;
  VGA_SYNC <= '0';
  VGA_BLANK <= not (vga_hsync or vga_vsync);

  vbi <= not vga_vblank;

end rtl;



library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;

entity vga_top is
  
  port (
    -- avalon interface
    clk        : in  std_logic;
    reset_n    : in  std_logic;
    read       : in  std_logic;
    write      : in  std_logic;
    chipselect : in  std_logic;
    address    : in  unsigned(15 downto 0);
    readdata   : out unsigned(15 downto 0);
    writedata  : in  unsigned(15 downto 0);

    -- vga interface

    CLOCK_50 : std_logic;
    
    VGA_CLK,                                            -- Clock
    VGA_HS,                                             -- H_SYNC
    VGA_VS,                                             -- V_SYNC
    VGA_BLANK,                                          -- BLANK
    VGA_SYNC : out std_logic;                           -- SYNC
    VGA_R,                                              -- Red[9:0]
    VGA_G,                                              -- Green[9:0]
    VGA_B : out unsigned(9 downto 0)                    -- Blue[9:0]

    
    );
  
end vga_top;

architecture rtl of vga_top is

  type ram_type is array(0 to 26) of unsigned(15 downto 0);
  signal RAM : ram_type;
  signal ram_address : unsigned(15 downto 0);
  signal clk25 : std_logic;
  signal vbi : std_logic;

  signal gobj_en, fg_en, bg_en : std_logic; -- enable signals for each layer
  signal pgen_vga_r, pgen_vga_g, pgen_vga_b : unsigned (9 downto 0);
  signal tile_map : unsigned (255 downto 0);

  signal 
		ballx,
		bally,
		p1a,
		p1b,
		p2a,
		p2b	: unsigned (9 downto 0);

-- color signals
  signal
		sc_color,
		bg_color,
		fg_color,
		go_color : unsigned (15 downto 0);
		
  signal visx, visy : unsigned(9 downto 0); -- visible screen x and y position

	-- VGA memory signals
	
	-- into memory
	signal address_avalon, address_vga : std_logic_vector (14 downto 0);
	signal write_avalon : std_logic; -- write signal for VGA hardwired to 0 since always read
	signal write_data : unsigned (15 downto 0);
	signal rd_address : unsigned (14 downto 0);
	
	-- from memory
	signal readdata_avalon, readdata_vga : unsigned (15 downto 0);
	
	signal mem_cs : std_logic;
	
	
	--test stuff	
	signal clk16  : std_logic; -- slow clock for VGA memory
	
	
	component ram_fg is
		port (
			
			-- avalon side
			clk 		: in std_logic; -- dual clocked
			addr		: in unsigned (14 downto 0);
			cs			: in std_logic; -- chipselect
			w_en		: in std_logic; -- write enable
			din			: in unsigned (15 downto 0); 
			dout		: out unsigned (15 downto 0);
			
			-- vga side
			clk2		: in std_logic; -- dual clocked
			addr2		: in unsigned (14 downto 0);
			dout2		: out unsigned (15 downto 0)
			
			);
end component;
	

begin

  --
  -- avalon interface
  --

  -- memory map: BASE:xpos BASE+2:ypos BASE+4:blank

  ram_address <= address(15 downto 0);
  write_avalon <= (write and address(15));
  
  process (clk)
  begin
    if rising_edge(clk) then
      if reset_n = '0' then
        readdata <= (others => '0');
      else
        if chipselect = '1' then
          if read = '1' then
			if ( address(15) = '0' ) then
				RAM(22) <= b"000000000000000" & vbi;
				readdata <= RAM(to_integer(ram_address(4 downto 0)));
				else
				readdata <= UNSIGNED(readdata_avalon);
		  end if;
		  elsif write = '1' then
				if address(15) = '0' then
				RAM(to_integer(ram_address)) <= writedata;
				else
				write_data <= writedata;
				end if;
		  end if;
        end if;

	  end if;
    end if;
  end process;


  --
  -- vga raster logic
  --
  
  process (CLOCK_50)
  begin
    if rising_edge(CLOCK_50) then
      clk25 <= not clk25;
    end if;
  end process;

 process (clk25)
	begin
		if rising_edge(clk25) then
			clk16 <= visx(4);
		end if;
	end process;

  V1: entity work.de2_vga_raster port map (
    reset => '0',
    clk => clk25,

	visx => visx,
	visy => visy,
	pgen_vga_r => pgen_vga_r,
	pgen_vga_g => pgen_vga_g,
	pgen_vga_b => pgen_vga_b,
	vbi => vbi,
	
    cx => RAM(0)(9 downto 0),
    cy => RAM(1)(9 downto 0),
    
    VGA_CLK => VGA_CLK,
    VGA_HS => VGA_HS,
	VGA_VS => VGA_VS,
    VGA_BLANK => VGA_BLANK,
    VGA_SYNC => VGA_SYNC,
    VGA_R => VGA_R,
    VGA_G => VGA_G,
    VGA_B => VGA_B
  );

		tile_map (255 downto 240) <= RAM(0);
		tile_map (239 downto 224) <= RAM(1);
		tile_map (223 downto 208) <= RAM(2);
		tile_map (207 downto 192) <= RAM(3);
		tile_map (191 downto 176) <= RAM(4);
		tile_map (175 downto 160) <= RAM(5);
		tile_map (159 downto 144) <= RAM(6);
		tile_map (143 downto 128) <= RAM(7);
		tile_map (127 downto 112) <= RAM(8);
		tile_map (111 downto 96) <= RAM(9);
		tile_map (95 downto 80) <= RAM(10);
		tile_map (79 downto 64) <= RAM(11);
		tile_map (63 downto 48) <= RAM(12);
		tile_map (47 downto 32) <= RAM(13);
		tile_map (31 downto 16) <= RAM(14);
		tile_map (15 downto 0) <= RAM(15);

		
		ballx <= RAM(16)(9 downto 0);
		bally <= RAM(17)(9 downto 0);
		p1a <= RAM(18)(9 downto 0);
		p1b <= RAM(19)(9 downto 0);
		p2a <= RAM(20)(9 downto 0);
		p2b	<= RAM(21)(9 downto 0);
		
		sc_color <= RAM(23);
		bg_color <= RAM(24);
		fg_color <= RAM(25);
		go_color <= RAM(26);
	

		U1: entity work.gobj_gen port map (
			visx => visx,
			visy => visy,
			ballx => ballx,
			bally => bally,
			p1a => p1a,
			p1b => p1b,
			p2a => p2a,
			p2b	=> p2b,
			do => gobj_en
		);
		
		
		U2: entity work.bg_gen port map (
			x => visx,
			y => visy,
			tile_map => tile_map,
			do => bg_en
		);

			
		U3: entity work.fg_gen port map(
			clk => clk25,
			visx => visx,
			visy => visy,
			rd_address => rd_address,
			data_in => readdata_vga,
			do => fg_en
			);
			
			
		U4: entity work.pmux port map (
			clk => clk25,
			en0 => gobj_en,
			en1 => fg_en, 
			en2 => bg_en,
			sc_color => sc_color,
			bg_color => bg_color,
			fg_color => fg_color,
			go_color => go_color,
			vga_r => pgen_vga_r,
			vga_g => pgen_vga_g,
			vga_b => pgen_vga_b
			);
			
			mem_cs <= std_logic(address(15)) and chipselect; -- used in ram_fg
			
		U5: ram_fg port map (
			
				-- avalon side
				clk => clk,
				addr => address(14 downto 0),
				cs => mem_cs,
				w_en => write_avalon,
				din => write_data,
				dout => readdata_avalon,
				
				-- vga side
				clk2 => clk,
				addr2 => rd_address,
				dout2 => readdata_vga
				
				);

end rtl;
