-------------------------------------------------------------------------------
--
-- Simple VGA raster display
--
-- Stephen A. Edwards
-- sedwards@cs.columbia.edu
--
-------------------------------------------------------------------------------
library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;

entity vga_raster is

  port (
    reset : in std_logic;
    clk : in std_logic;
	x : in unsigned(4 downto 0);
	y : in unsigned(3 downto 0);
    sil : in std_logic;
	ch_in	:in unsigned(11 downto 0);
	cv_in	:in unsigned(11 downto 0);
	h_in	:std_logic;
	r_in	:in unsigned(5 downto 0);
	dodge  :in std_logic;
	goalie :in std_logic;
	ninja  :in std_logic;
	
	HEX3,
	HEX2,
	HEX1,
	HEX0 : out std_logic_vector( 6 downto 0);
    VGA_CLK,                         -- Clock
    VGA_HS,                          -- H_SYNC
    VGA_VS,                          -- V_SYNC
    VGA_BLANK,                       -- BLANK
    VGA_SYNC : out std_logic;        -- SYNC
    VGA_R,                           -- Red[9:0]
    VGA_G,                           -- Green[9:0]
    VGA_B : out unsigned(9 downto 0) -- Blue[9:0]
    );

end vga_raster;

architecture rtl of vga_raster is

  -- Video parameters

  constant HTOTAL       : integer := 800;
  constant HSYNC        : integer := 96;
  constant HBACK_PORCH  : integer := 48;
  constant HACTIVE      : integer := 640;
  constant HFRONT_PORCH : integer := 16;

  constant VTOTAL       : integer := 525;
  constant VSYNC        : integer := 2;
  constant VBACK_PORCH  : integer := 33;
  constant VACTIVE      : integer := 480;
  constant VFRONT_PORCH : integer := 10;

  -- Signals for the video controller
  signal Hcount : unsigned(9 downto 0);  -- Horizontal position (0-800)
  signal Vcount : unsigned(9 downto 0);  -- Vertical position (0-524)
  signal EndOfLine, EndOfField : std_logic;

  signal vga_hblank, vga_hsync,
    vga_vblank, vga_vsync : std_logic;  -- Sync. signals

	signal Xcount : unsigned(9 downto 0);
	signal Ycount : unsigned(8 downto 0);
	signal silhouette : std_logic_vector(0 to 299) := (others => '0'); -- silhouette
	
	type ram_type is array(15 downto 0) of unsigned(15 downto 0);
	type array_t is array (15 downto 0) of unsigned(79 downto 0);
	type array_t2 is array (15 downto 0) of unsigned(127 downto 0);
	type array_t3 is array (15 downto 0) of unsigned(95 downto 0);
	type array_temp is array (15 downto 0) of unsigned(15 downto 0);

	signal CIRCLE_HSTART : unsigned(11 downto 0):= "000000010000";
	signal CIRCLE_VSTART : unsigned(11 downto 0):= "000000010000"; 
	signal CIRCLE_RADIUS : unsigned(5 downto 0) := "010000";
	signal CIRCLE_HIT: std_logic := '1';
	signal SPRITE_SCORE: array_t;
	signal SPRITE_LOSE: array_t2;
	signal SPRITE_HSCORE: array_t3;
	signal SPRITE0: array_temp;
	signal SPRITE1: array_temp;
	signal SPRITE2: array_temp;
	signal SPRITE3: array_temp;
	signal SPRITE4: array_temp;
	signal SPRITE5: array_temp;
	signal SPRITE6: array_temp;
	signal SPRITE7: array_temp;
	signal SPRITE8: array_temp;
	signal SPRITE9: array_temp;

	signal SCORE1 : unsigned(3 downto 0) := "0000";	--keeps track of the score of the current game
	signal SCORE2 : unsigned(3 downto 0) := "0000"; --1 is most significant and 4 is the least significant
	signal SCORE3 : unsigned(3 downto 0) := "0000";
	signal SCORE4 : unsigned(3 downto 0) := "0000";
	signal HSCORE1 : unsigned(3 downto 0) := "0000"; --keeps track of the highest score for all games.
	signal HSCORE2 : unsigned(3 downto 0) := "0000";
	signal HSCORE3 : unsigned(3 downto 0) := "0000";
	signal HSCORE4 : unsigned(3 downto 0) := "0000";
	
	signal increment: std_logic;
	signal nlose: std_logic := '1';
	signal restart: std_logic := '0';
	signal hit: std_logic := '0';
	
	signal touched: std_logic := '0';
	signal tempLose: std_logic := '0';
	signal ninja_hit: std_logic := '0';

	signal temp: unsigned(31 downto 0);
	signal circle : std_logic;
	signal circleEND : std_logic;
	
	

begin
  --Sprites for various messages in the game
  SPRITE_LOSE(0)  <= "00000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000";
  SPRITE_LOSE(1)  <= "00000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000";
  SPRITE_LOSE(2)  <= "00000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000";
  SPRITE_LOSE(3)  <= "00111100001111000001111111111000000110000001100000000000000000000001110000000000000111111111100000011111111110000001111111111000";
  SPRITE_LOSE(4)  <= "00011100001110000001111111111000000110000001100000000000000000000001110000000000000111111111100000011111111110000001111111111000";
  SPRITE_LOSE(5)  <= "00001110011100000001110000011000000110000001100000000000000000000001110000000000000111000011100000011000000000000001100000000000";
  SPRITE_LOSE(6)  <= "00000111111000000001110000011000000110000001100000000000000000000001110000000000000111000011100000011000000000000001100000000000";
  SPRITE_LOSE(7)  <= "00000011110000000001110000011000000110000001100000000000000000000001110000000000000111000011100000011111111110000001111111111000";
  SPRITE_LOSE(8)  <= "00000011110000000001110000011000000110000001100000000000000000000001110000000000000111000011100000011111111110000001111111111000";
  SPRITE_LOSE(9)  <= "00000011110000000001110000011000000110000001100000000000000000000001110000000000000111000011100000000000000110000001100000000000";
  SPRITE_LOSE(10) <= "00000011110000000001110000011000000110000001100000000000000000000001110000000000000111000011100000000000000110000001100000000000";
  SPRITE_LOSE(11) <= "00000011110000000001111111111000000111111111100000000000000000000001111111111000000111111111100000011111111110000001111111111000";
  SPRITE_LOSE(12) <= "00000011110000000001111111111000000111111111100000000000000000000001111111111000000111111111100000011111111110000001111111111000";
  SPRITE_LOSE(13) <= "00000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000";
  SPRITE_LOSE(14) <= "00000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000";
  SPRITE_LOSE(15) <= "00000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000";

  SPRITE_SCORE(0)  <= "00000000000000000000000000000000000000000000000000000000000000000000000000000000";
  SPRITE_SCORE(1)  <= "00000000000000000000000000000000000000000000000000000000000000000000000000000000";
  SPRITE_SCORE(2)  <= "00000000000000000000000000000000000000000000000000000000000000000000000000000000";
  SPRITE_SCORE(3)  <= "00001111111110000000111111111000000011111111000000011111111100000001111111111000";
  SPRITE_SCORE(4)  <= "00011111111110000001111111111000000111111111100000011111111110000001111111111000";
  SPRITE_SCORE(5)  <= "00011000000000000001110000000000000110000001100000011000000110000001100000000000";
  SPRITE_SCORE(6)  <= "00011000000000000001110000000000000110000001100000011000000110000001100000000000";
  SPRITE_SCORE(7)  <= "00011111111100000001110000000000000110000001100000011111111110000001111111111000";
  SPRITE_SCORE(8)  <= "00001111111110000001110000000000000110000001100000011111111100000001111111111000";
  SPRITE_SCORE(9)  <= "00000000000110000001110000000000000110000001100000011001100000000001100000000000";
  SPRITE_SCORE(10) <= "00000000000110000001110000000000000110000001100000011000110000000001100000000000";
  SPRITE_SCORE(11) <= "00011111111110000001111111111000000111111111100000011000011000000001111111111000";
  SPRITE_SCORE(12) <= "00011111111100000000111111111000000011111111000000011000001100000001111111111000";
  SPRITE_SCORE(13) <= "00000000000000000000000000000000000000000000000000000000000000000000000000000000";
  SPRITE_SCORE(14) <= "00000000000000000000000000000000000000000000000000000000000000000000000000000000";
  SPRITE_SCORE(15) <= "00000000000000000000000000000000000000000000000000000000000000000000000000000000";

  SPRITE_HSCORE(0)  <= "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000";
  SPRITE_HSCORE(1)  <= "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000";
  SPRITE_HSCORE(2)  <= "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000";
  SPRITE_HSCORE(3)  <= "000110000001100000001111111110000000111111111000000011111111000000011111111100000001111111111000";
  SPRITE_HSCORE(4)  <= "000110000001100000011111111110000001111111111000000111111111100000011111111110000001111111111000";
  SPRITE_HSCORE(5)  <= "000110000001100000011000000000000001110000000000000110000001100000011000000110000001100000000000";
  SPRITE_HSCORE(6)  <= "000110000001100000011000000000000001110000000000000110000001100000011000000110000001100000000000";
  SPRITE_HSCORE(7)  <= "000111111111100000011111111100000001110000000000000110000001100000011111111110000001111111111000";
  SPRITE_HSCORE(8)  <= "000111111111100000001111111110000001110000000000000110000001100000011111111100000001111111111000";
  SPRITE_HSCORE(9)  <= "000110000001100000000000000110000001110000000000000110000001100000011001100000000001100000000000";
  SPRITE_HSCORE(10) <= "000110000001100000000000000110000001110000000000000110000001100000011000110000000001100000000000";
  SPRITE_HSCORE(11) <= "000110000001100000011111111110000001111111111000000111111111100000011000011000000001111111111000";
  SPRITE_HSCORE(12) <= "000110000001100000011111111100000000111111111000000011111111000000011000001100000001111111111000";
  SPRITE_HSCORE(13) <= "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000";
  SPRITE_HSCORE(14) <= "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000";
  SPRITE_HSCORE(15) <= "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000";

  SPRITE0(0)  <= "0000000000000000";
  SPRITE0(1)  <= "0000000000000000";
  SPRITE0(2)  <= "0000000000000000";
  SPRITE0(3)  <= "0000111111110000";
  SPRITE0(4)  <= "0001111111111000";
  SPRITE0(5)  <= "0001100000011000";
  SPRITE0(6)  <= "0001100000011000";
  SPRITE0(7)  <= "0001100000011000";
  SPRITE0(8)  <= "0001100000011000";
  SPRITE0(9)  <= "0001100000011000";
  SPRITE0(10) <= "0001100000011000";
  SPRITE0(11) <= "0001111111111000";
  SPRITE0(12) <= "0000111111110000";
  SPRITE0(13) <= "0000000000000000";
  SPRITE0(14) <= "0000000000000000";
  SPRITE0(15) <= "0000000000000000";

  SPRITE1(0)  <= "0000000000000000";
  SPRITE1(1)  <= "0000000000000000";
  SPRITE1(2)  <= "0000000000000000";
  SPRITE1(3)  <= "0000000110000000";
  SPRITE1(4)  <= "0000000110000000";
  SPRITE1(5)  <= "0000000110000000";
  SPRITE1(6)  <= "0000000110000000";
  SPRITE1(7)  <= "0000000110000000";
  SPRITE1(8)  <= "0000000110000000";
  SPRITE1(9)  <= "0000000110000000";
  SPRITE1(10) <= "0000000110000000";
  SPRITE1(11) <= "0000000110000000";
  SPRITE1(12) <= "0000000110000000";
  SPRITE1(13) <= "0000000000000000";
  SPRITE1(14) <= "0000000000000000";
  SPRITE1(15) <= "0000000000000000";

  SPRITE2(0)  <= "0000000000000000";
  SPRITE2(1)  <= "0000000000000000";
  SPRITE2(2)  <= "0000000000000000";
  SPRITE2(3)  <= "0000111111110000";
  SPRITE2(4)  <= "0001111111111000";
  SPRITE2(5)  <= "0001110000111000";
  SPRITE2(6)  <= "0000000000111000";
  SPRITE2(7)  <= "0000000001110000";
  SPRITE2(8)  <= "0000000011100000";
  SPRITE2(9)  <= "0000000111000000";
  SPRITE2(10) <= "0000001110000000";
  SPRITE2(11) <= "0000111111111000";
  SPRITE2(12) <= "0001111111111000";
  SPRITE2(13) <= "0000000000000000";
  SPRITE2(14) <= "0000000000000000";
  SPRITE2(15) <= "0000000000000000";

  SPRITE3(0)  <= "0000000000000000";
  SPRITE3(1)  <= "0000000000000000";
  SPRITE3(2)  <= "0000000000000000";
  SPRITE3(3)  <= "0000111111110000";
  SPRITE3(4)  <= "0001111111111000";
  SPRITE3(5)  <= "0001100000011000";
  SPRITE3(6)  <= "0000000000011000";
  SPRITE3(7)  <= "0000000111110000";
  SPRITE3(8)  <= "0000000111110000";
  SPRITE3(9)  <= "0000000000011000";
  SPRITE3(10) <= "0001100000011000";
  SPRITE3(11) <= "0001111111111000";
  SPRITE3(12) <= "0000111111110000";
  SPRITE3(13) <= "0000000000000000";
  SPRITE3(14) <= "0000000000000000";
  SPRITE3(15) <= "0000000000000000";

  SPRITE4(0)  <= "0000000000000000";
  SPRITE4(1)  <= "0000000000000000";
  SPRITE4(2)  <= "0000000000000000";
  SPRITE4(3)  <= "0000000011111000";
  SPRITE4(4)  <= "0000000111111000";
  SPRITE4(5)  <= "0000001110011000";
  SPRITE4(6)  <= "0000011100011000";
  SPRITE4(7)  <= "0000111000011000";
  SPRITE4(8)  <= "0001111111111100";
  SPRITE4(9)  <= "0001111111111100";
  SPRITE4(10) <= "0000000000011000";
  SPRITE4(11) <= "0000000000011000";
  SPRITE4(12) <= "0000000000011000";
  SPRITE4(13) <= "0000000000000000";
  SPRITE4(14) <= "0000000000000000";
  SPRITE4(15) <= "0000000000000000";

  SPRITE5( 0) <= "0000000000000000";
  SPRITE5( 1) <= "0000000000000000";
  SPRITE5( 2) <= "0000000000000000";
  SPRITE5( 3) <= "0001111111111000";
  SPRITE5( 4) <= "0001111111111000";
  SPRITE5( 5) <= "0001100000000000";
  SPRITE5( 6) <= "0001100000000000";
  SPRITE5( 7) <= "0001111111110000";
  SPRITE5( 8) <= "0001111111111000";
  SPRITE5( 9) <= "0000000000011000";
  SPRITE5(10) <= "0001100000011000";
  SPRITE5(11) <= "0001111111111000";
  SPRITE5(12) <= "0000111111110000";
  SPRITE5(13) <= "0000000000000000";
  SPRITE5(14) <= "0000000000000000";
  SPRITE5(15) <= "0000000000000000";

  SPRITE6( 0) <= "0000000000000000";
  SPRITE6( 1) <= "0000000000000000";
  SPRITE6( 2) <= "0000000000000000";
  SPRITE6( 3) <= "0000111111110000";
  SPRITE6( 4) <= "0001111111111000";
  SPRITE6( 5) <= "0001100000011000";
  SPRITE6( 6) <= "0001100000000000";
  SPRITE6( 7) <= "0001111111110000";
  SPRITE6( 8) <= "0001111111111000";
  SPRITE6( 9) <= "0001100000011000";
  SPRITE6(10) <= "0001100000011000";
  SPRITE6(11) <= "0001111111111000";
  SPRITE6(12) <= "0000111111110000";
  SPRITE6(13) <= "0000000000000000";
  SPRITE6(14) <= "0000000000000000";
  SPRITE6(15) <= "0000000000000000";

  SPRITE7( 0) <= "0000000000000000";
  SPRITE7( 1) <= "0000000000000000";
  SPRITE7( 2) <= "0000000000000000";
  SPRITE7( 3) <= "0001111111111000";
  SPRITE7( 4) <= "0001111111111000";
  SPRITE7( 5) <= "0000000001110000";
  SPRITE7( 6) <= "0000000001110000";
  SPRITE7( 7) <= "0000000011100000";
  SPRITE7( 8) <= "0000000011100000";
  SPRITE7( 9) <= "0000000111000000";
  SPRITE7(10) <= "0000000111000000";
  SPRITE7(11) <= "0000001110000000";
  SPRITE7(12) <= "0000001110000000";
  SPRITE7(13) <= "0000000000000000";
  SPRITE7(14) <= "0000000000000000";
  SPRITE7(15) <= "0000000000000000";

  SPRITE8( 0) <= "0000000000000000";
  SPRITE8( 1) <= "0000000000000000";
  SPRITE8( 2) <= "0000000000000000";
  SPRITE8( 3) <= "0000111111110000";
  SPRITE8( 4) <= "0001111111111000";
  SPRITE8( 5) <= "0001100000011000";
  SPRITE8( 6) <= "0001100000011000";
  SPRITE8( 7) <= "0000111111110000";
  SPRITE8( 8) <= "0000111111110000";
  SPRITE8( 9) <= "0001100000011000";
  SPRITE8(10) <= "0001100000011000";
  SPRITE8(11) <= "0001111111111000";
  SPRITE8(12) <= "0000111111110000";
  SPRITE8(13) <= "0000000000000000";
  SPRITE8(14) <= "0000000000000000";
  SPRITE8(15) <= "0000000000000000";

  SPRITE9(0)  <= "0000000000000000";
  SPRITE9(1)  <= "0000000000000000";
  SPRITE9(2)  <= "0000000000000000";
  SPRITE9(3)  <= "0000111111110000";
  SPRITE9(4)  <= "0001111111111000";
  SPRITE9(5)  <= "0001100000011000";
  SPRITE9(6)  <= "0001100000011000";
  SPRITE9(7)  <= "0001111111111000";
  SPRITE9(8)  <= "0000111111111000";
  SPRITE9(9)  <= "0000000000111000";
  SPRITE9(10) <= "0000000000111000";
  SPRITE9(11) <= "0000000000111000";
  SPRITE9(12) <= "0000000000111000";
  SPRITE9(13) <= "0000000000000000";
  SPRITE9(14) <= "0000000000000000";
  SPRITE9(15) <= "0000000000000000";

  -- Horizontal and vertical counters

  HCounter : process (clk)
  begin
    if rising_edge(clk) then
      if reset = '1' then
        Hcount <= (others => '0');
      elsif EndOfLine = '1' then
        Hcount <= (others => '0');
      else
        Hcount <= Hcount + 1;
      end if;
    end if;
  end process HCounter;

  EndOfLine <= '1' when Hcount = HTOTAL - 1 else '0';

  VCounter: process (clk)
  begin
    if rising_edge(clk) then      
      if reset = '1' then
        Vcount <= (others => '0');
      elsif EndOfLine = '1' then
        if EndOfField = '1' then
          Vcount <= (others => '0');
        else
          Vcount <= Vcount + 1;
        end if;
      end if;
    end if;
  end process VCounter;

  EndOfField <= '1' when Vcount = VTOTAL - 1 else '0';

	-- X and Y Counters
	XCounter: process (clk)
	begin
		if rising_edge(clk) then
			if (Hcount > HSYNC + HBACK_PORCH and Xcount < HACTIVE - 1) then
				Xcount <= Xcount + 1;
			else
				Xcount <= (others => '0');
			end if;
		end if;
	end process XCounter;

	YCounter: process (clk)
	begin
		if rising_edge(clk) then
			if (Vcount > VSYNC + VBACK_PORCH and Ycount < VACTIVE) then
				if Hcount = 1 then
					Ycount <= Ycount + 1;
				end if;
			else
				Ycount <= (others => '0');
			end if;
		end if;
	end process YCounter;

  -- State machines to generate HSYNC, VSYNC, HBLANK, and VBLANK

  HSyncGen : process (clk)
  begin
    if rising_edge(clk) then     
      if reset = '1' or EndOfLine = '1' then
        vga_hsync <= '1';
      elsif Hcount = HSYNC - 1 then
        vga_hsync <= '0';
      end if;
    end if;
  end process HSyncGen;

  HBlankGen : process (clk)
  begin
    if rising_edge(clk) then
      if reset = '1' then
        vga_hblank <= '1';
      elsif Hcount = HSYNC + HBACK_PORCH then
        vga_hblank <= '0';
      elsif Hcount = HSYNC + HBACK_PORCH + HACTIVE then
        vga_hblank <= '1';
      end if;      
    end if;
  end process HBlankGen;

  VSyncGen : process (clk)
  begin
    if rising_edge(clk) then
      if reset = '1' then
        vga_vsync <= '1';
      elsif EndOfLine ='1' then
        if EndOfField = '1' then
          vga_vsync <= '1';
        elsif Vcount = VSYNC - 1 then
          vga_vsync <= '0';
        end if;
      end if;      
    end if;
  end process VSyncGen;

  VBlankGen : process (clk)
  begin
    if rising_edge(clk) then    
      if reset = '1' then
        vga_vblank <= '1';
      elsif EndOfLine = '1' then
        if Vcount = VSYNC + VBACK_PORCH - 1 then
          vga_vblank <= '0';
        elsif Vcount = VSYNC + VBACK_PORCH + VACTIVE - 1 then
          vga_vblank <= '1';
        end if;
      end if;
    end if;
  end process VBlankGen;

	-- Silhouette Updater
	SilUpdate : process (clk, y, x)
	begin
		if rising_edge(clk) then
			-- workaround begin
			if x = 0 then
				silhouette(to_integer(y)*20) <= sil;
			else
				silhouette(to_integer(y)*20 + 20 - to_integer(x)) <= sil;
			end if;
			-- workaround end
			-- line below should be the correct code, but does not work for some reason
--			silhouette(to_integer(y)*20 + 19 - to_integer(x)) <= sil;
		end if;
	end process SilUpdate;
	
	--Generates the balls on screen based on the vertical, horizontal coordinates and the size of the radius
	CircleGen : process (clk)
	begin
	  if rising_edge(clk) then
		CIRCLE_HSTART <= ch_in;
		CIRCLE_VSTART <= cv_in;
		CIRCLE_RADIUS <= r_in;
		if ((to_integer(Hcount)-HSYNC - HBACK_PORCH-to_integer(CIRCLE_HSTART))*(to_integer(Hcount)-HSYNC - HBACK_PORCH- to_integer(CIRCLE_HSTART)) + (to_integer(Vcount) -VSYNC - VBACK_PORCH + 1 -to_integer(CIRCLE_VSTART))*(to_integer(Vcount)-VSYNC - VBACK_PORCH + 1-to_integer(CIRCLE_VSTART))) < to_integer(CIRCLE_RADIUS)*to_integer(CIRCLE_RADIUS) then
		  circle <= '1';
		else
		  circle <= '0';
		end if;
	  end if;
	end process CircleGen;
	
  -- Registered video signals going to the video DAC

  VideoOut: process (clk, reset)
  variable SCORE_V : unsigned (9 downto 0);
  variable SCORE_H : unsigned (9 downto 0);
  variable SCORE_H2: unsigned (9 downto 0);
  variable LOSE_H3: unsigned (9 downto 0);
  variable HIGH_H4: unsigned (9 downto 0);
  variable SCORE_TEMP: unsigned(79 downto 0);
  variable LOSE_TEMP: unsigned(127 downto 0);
  variable HIGH_TEMP: unsigned(95 downto 0);
  variable SCORE1s: unsigned(15 downto 0);
  variable SCORE2s: unsigned(15 downto 0);
  variable SCORE3s: unsigned(15 downto 0);
  variable SCORE4s: unsigned(15 downto 0);
  variable HSCORE1s: unsigned(15 downto 0);
  variable HSCORE2s: unsigned(15 downto 0);
  variable HSCORE3s: unsigned(15 downto 0);
  variable HSCORE4s: unsigned(15 downto 0);

  begin
    if reset = '1' then
      VGA_R <= "0000000000";
      VGA_G <= "0000000000";
      VGA_B <= "0000000000";
	elsif clk'event and clk = '1' then
	  SCORE_V := (Vcount - (VSYNC + VBACK_PORCH)) AND "0000001111";
	  SCORE_H := (Hcount - (HSYNC + HBACK_PORCH)) AND "0001111111";
	  SCORE_H2:= (Hcount - (HSYNC + HBACK_PORCH)) AND "0000001111";
	  LOSE_H3 := (Hcount - (HSYNC + HBACK_PORCH));
	  HIGH_H4 := (Hcount - (HSYNC + HBACK_PORCH));
	
	  --Updates the score when increment is flagged
	  if to_integer(CIRCLE_RADIUS) < 45 and increment = '1' then
		if (to_integer(SCORE4) < 9) then
		  SCORE4 <= SCORE4 + 1;
		else
		  SCORE4 <= "0000";
		  if (to_integer(SCORE3) < 9) then
			SCORE3 <= SCORE3 + 1;
		  else
			SCORE3 <= "0000";
			if(to_integer(SCORE2) < 9) then
			  SCORE2 <= SCORE2 + 1;
			else
			  SCORE2 <= "0000";
			  if(to_integer(SCORE1) < 9) then
				SCORE1 <= SCORE1 + 1;
			  else
				SCORE1 <= "0000";
			  end if;
			end if;
		  end if;
		end if;
		increment <= '0';	
	  end if;
	  --Display the Score in LED
	  if to_integer(SCORE4) = 9 then
		HEX0 <= "0010000";
	  elsif to_integer(SCORE4) = 8 then
		HEX0 <= "0000000";
	  elsif to_integer(SCORE4) = 7 then
		HEX0 <= "1111000";
	  elsif to_integer(SCORE4) = 6 then
		HEX0 <= "0000010";
	  elsif to_integer(SCORE4) = 5 then
		HEX0 <= "0010010";
	  elsif to_integer(SCORE4) = 4 then
		HEX0 <= "0011001";
	  elsif to_integer(SCORE4) = 3 then
		HEX0 <= "0110000";
	  elsif to_integer(SCORE4) = 2 then
		HEX0 <= "0100100";
	  elsif to_integer(SCORE4) = 1 then
		HEX0 <= "1111001";
	  elsif to_integer(SCORE4) = 0 then
		HEX0 <= "1000000";
	  end if;
	  if to_integer(SCORE3) = 9 then
		HEX1 <= "0010000";
	  elsif to_integer(SCORE3) = 8 then
		HEX1 <= "0000000";
	  elsif to_integer(SCORE3) = 7 then
		HEX1 <= "1111000";
	  elsif to_integer(SCORE3) = 6 then
		HEX1 <= "0000010";
	  elsif to_integer(SCORE3) = 5 then
		HEX1 <= "0010010";
	  elsif to_integer(SCORE3) = 4 then
		HEX1 <= "0011001";
	  elsif to_integer(SCORE3) = 3 then
		HEX1 <= "0110000";
	  elsif to_integer(SCORE3) = 2 then
		HEX1 <= "0100100";
	  elsif to_integer(SCORE3) = 1 then
		HEX1 <= "1111001";
	  elsif to_integer(SCORE3) = 0 then
		HEX1 <= "1000000";
	  end if;
	  if to_integer(SCORE2) = 9 then
		HEX2 <= "0010000";
	  elsif to_integer(SCORE2) = 8 then
		HEX2 <= "0000000";
	  elsif to_integer(SCORE2) = 7 then
		HEX2 <= "1111000";
	  elsif to_integer(SCORE2) = 6 then
		HEX2 <= "0000010";
	  elsif to_integer(SCORE2) = 5 then
		HEX2 <= "0010010";
	  elsif to_integer(SCORE2) = 4 then
		HEX2 <= "0011001";
	  elsif to_integer(SCORE2) = 3 then
		HEX2 <= "0110000";
	  elsif to_integer(SCORE2) = 2 then
		HEX2 <= "0100100";
	  elsif to_integer(SCORE2) = 1 then
		HEX2 <= "1111001";
	  elsif to_integer(SCORE2) = 0 then
		HEX2 <= "1000000";
	  end if;	
	  if to_integer(SCORE1) = 9 then
		HEX3 <= "0010000";
	  elsif to_integer(SCORE1) = 8 then
		HEX3 <= "0000000";
	  elsif to_integer(SCORE1) = 7 then
		HEX3 <= "1111000";
	  elsif to_integer(SCORE1) = 6 then
		HEX3 <= "0000010";
	  elsif to_integer(SCORE1) = 5 then
		HEX3 <= "0010010";
	  elsif to_integer(SCORE1) = 4 then
		HEX3 <= "0011001";
	  elsif to_integer(SCORE1) = 3 then
		HEX3 <= "0110000";
	  elsif to_integer(SCORE1) = 2 then
		HEX3 <= "0100100";
	  elsif to_integer(SCORE1) = 1 then
		HEX3 <= "1111001";
	  elsif to_integer(SCORE1) = 0 then
		HEX3 <= "1000000";
	  end if;
	  --Sets the different modes of the game
	  if goalie = '1' then
		restart <= '1';
		hit <= '1';
		ninja_hit <= '0';
	  elsif dodge = '1' then
		restart <= '1';
		hit <= '0';
		ninja_hit <= '0';
	  elsif ninja = '1' then
		restart <= '1';
		ninja_hit <= '1';
	  end if;
	  if ninja_hit = '1' then
		hit <= h_in;
	  end if;
	  --restarts the game when the corresponding key is pressed
	  if nlose = '0' and restart = '1' and to_integer(CIRCLE_RADIUS) < 5  then
		SCORE4 <= "0000";
		SCORE3 <= "0000";
		SCORE2 <= "0000";
		SCORE1 <= "0000";
		nlose <= '1';
		tempLose <= '0';
		restart <= '0';
	  end if;
      if vga_hblank = '1' or vga_vblank = '1' then
        VGA_R <= "0000000000";
        VGA_G <= "0000000000";
        VGA_B <= "0000000000";
	  elsif nlose = '0' then
			--Screen displayed when the player loses, and saves high score if the current score is higher than saved score
			if to_integer(Hcount- (HSYNC + HBACK_PORCH)) < 384 and to_integer(Hcount- (HSYNC + HBACK_PORCH)) > 257 and to_integer(Vcount - (VSYNC + VBACK_PORCH)) < 96 and to_integer(Vcount- (VSYNC + VBACK_PORCH)) > 79 then --and to_integer(Hcount) < 465 and to_integer(Vcount) < 640 and to_integer(Vcount) > 624 then
				LOSE_TEMP := SPRITE_LOSE(to_integer(unsigned(SCORE_V)));
				if (LOSE_TEMP(to_integer(unsigned(127 - LOSE_H3))) = '1') then
					VGA_R <= "1111111111";
					VGA_G <= "0000000000";
					VGA_B <= "0000000000";
				else
					VGA_R <= "0000000000";
					VGA_G <= "0000000000";
					VGA_B <= "0000000000";
				end if;
			end if;	
			if (to_integer(HSCORE1) < to_integer(SCORE1)) then
				HSCORE1 <= SCORE1;
				HSCORE2 <= SCORE2;
				HSCORE3 <= SCORE3;
				HSCORE4 <= SCORE4;
			elsif (to_integer(HSCORE1) = to_integer(SCORE1)) then
				if (to_integer(HSCORE2) < to_integer(SCORE2)) then
					HSCORE1 <= SCORE1;
					HSCORE2 <= SCORE2;
					HSCORE3 <= SCORE3;
					HSCORE4 <= SCORE4;
				elsif (to_integer(HSCORE2) = to_integer(SCORE2)) then
					if (to_integer(HSCORE3) < to_integer(SCORE3)) then
						HSCORE1 <= SCORE1;
						HSCORE2 <= SCORE2;
						HSCORE3 <= SCORE3;
						HSCORE4 <= SCORE4;
					elsif (to_integer(HSCORE3) = to_integer(SCORE3)) then
						if (to_integer(HSCORE4) < to_integer(SCORE4)) then
							HSCORE1 <= SCORE1;
							HSCORE2 <= SCORE2;
							HSCORE3 <= SCORE3;
							HSCORE4 <= SCORE4;
						end if;
					end if;
				end if;
			end if;
		restart <= '0';
	  --Display the HSCORE text
	  elsif to_integer(Hcount - (HSYNC + HBACK_PORCH)) > 0 and to_integer(Hcount - (HSYNC + HBACK_PORCH)) < 96 and to_integer(Vcount - (VSYNC + VBACK_PORCH)) < 464 and to_integer(Vcount- (VSYNC + VBACK_PORCH)) > 447 then
		HIGH_TEMP := SPRITE_HSCORE(to_integer(unsigned(SCORE_V)));
		if (HIGH_TEMP(to_integer(unsigned(96 - HIGH_H4))) = '1') then
		  VGA_R <= "1111111111";
		  VGA_G <= "0000000000";
		  VGA_B <= "0000000000";
	    else
		  if circle = '1' and silhouette(to_integer(Ycount(8 downto 5))*20 + to_integer(Xcount(9 downto 5))) = '0' and hit = '1' then
			VGA_R <= "0000000000";
			VGA_G <= "1000000000";
			VGA_B <= "0000000000";
		    if CIRCLE_RADIUS > 46 and touched = '0' then
			  tempLose <= '1';
		    end if;
		  elsif circle = '1' and silhouette(to_integer(Ycount(8 downto 5))*20 + to_integer(Xcount(9 downto 5))) = '0' and hit = '0' then
			VGA_R <= "1000000000";
			VGA_G <= "0000000000";
			VGA_B <= "0000000000";
		  elsif circle = '1'and silhouette(to_integer(Ycount(8 downto 5))*20 + to_integer(Xcount(9 downto 5))) = '1' and hit = '1'  then
			VGA_R <= "0000000000";
			VGA_G <= "1111111111";
			VGA_B <= "0000000000";
			if CIRCLE_RADIUS > 46 then
				touched <= '1';
				tempLose <= '0';
			end if;
		  elsif circle = '1'and silhouette(to_integer(Ycount(8 downto 5))*20 + to_integer(Xcount(9 downto 5))) = '1' and hit = '0'  then
			VGA_R <= "1111111111";
			VGA_G <= "0000000000";
			VGA_B <= "0000000000";
			if CIRCLE_RADIUS > 46 then
				nlose <= '0';
			end if;
		  elsif silhouette(to_integer(Ycount(8 downto 5))*20 + to_integer(Xcount(9 downto 5))) = '1' then
			VGA_R <= "1111111111";
			VGA_G <= "1111111111";
			VGA_B <= "1111111111";
		  else
			VGA_R <= "0000000000";
			VGA_G <= "1111111111";
			VGA_B <= "1111111111";
		  end if;
		end if;
	  --Display the SCORE text
	  elsif to_integer(Hcount- (HSYNC + HBACK_PORCH)) > 559 and to_integer(Vcount - (VSYNC + VBACK_PORCH)) < 464 and to_integer(Vcount- (VSYNC + VBACK_PORCH)) > 447 then --and to_integer(Hcount) < 465 and to_integer(Vcount) < 640 and to_integer(Vcount) > 624 then
		SCORE_TEMP := SPRITE_SCORE(to_integer(unsigned(SCORE_V)));
	    if (SCORE_TEMP(to_integer(unsigned(127 - SCORE_H))) = '1') then
		  VGA_R <= "1111111111";
		  VGA_G <= "0000000000";
		  VGA_B <= "0000000000";
	    else
		  if circle = '1' and silhouette(to_integer(Ycount(8 downto 5))*20 + to_integer(Xcount(9 downto 5))) = '0' and hit = '1' then
			VGA_R <= "0000000000";
			VGA_G <= "1000000000";
			VGA_B <= "0000000000";
		    if CIRCLE_RADIUS > 46 and touched = '0' then
			  tempLose <= '1';
		    end if;
		  elsif circle = '1' and silhouette(to_integer(Ycount(8 downto 5))*20 + to_integer(Xcount(9 downto 5))) = '0' and hit = '0' then
			VGA_R <= "1000000000";
			VGA_G <= "0000000000";
			VGA_B <= "0000000000";
		  elsif circle = '1'and silhouette(to_integer(Ycount(8 downto 5))*20 + to_integer(Xcount(9 downto 5))) = '1' and hit = '1'  then
			VGA_R <= "0000000000";
			VGA_G <= "1111111111";
			VGA_B <= "0000000000";
			if CIRCLE_RADIUS > 46 then
				touched <= '1';
				tempLose <= '0';
			end if;
		  elsif circle = '1'and silhouette(to_integer(Ycount(8 downto 5))*20 + to_integer(Xcount(9 downto 5))) = '1' and hit = '0'  then
			VGA_R <= "1111111111";
			VGA_G <= "0000000000";
			VGA_B <= "0000000000";
			if CIRCLE_RADIUS > 46 then
				nlose <= '0';
			end if;
		  elsif silhouette(to_integer(Ycount(8 downto 5))*20 + to_integer(Xcount(9 downto 5))) = '1' then
			VGA_R <= "1111111111";
			VGA_G <= "1111111111";
			VGA_B <= "1111111111";
		  else
			VGA_R <= "0000000000";
			VGA_G <= "1111111111";
			VGA_B <= "1111111111";
		  end if;
	    end if;
	  --determine whether the player gets points or loses
	  elsif to_integer(Vcount - (VSYNC + VBACK_PORCH)) > 478 and (to_integer(Hcount)-HSYNC - HBACK_PORCH) > 638 then
		if CIRCLE_RADIUS > 46 and hit = '0' then
			increment <= '1';
		elsif hit = '1'and tempLose = '1'  then
			nlose <= '0';
		elsif hit = '1' and CIRCLE_RADIUS > 46 then
			increment <= '1';
			touched <= '0';
		end if;
		
	  elsif to_integer(Hcount - (HSYNC+ HBACK_PORCH)) > 0 and to_integer(Hcount - (HSYNC+ HBACK_PORCH)) < 16 and to_integer(Vcount - (VSYNC + VBACK_PORCH)) < 480 and to_integer(Vcount - (VSYNC + VBACK_PORCH)) > 463 then
		if to_integer(HSCORE1) = 9 then
		  HSCORE1s := SPRITE9(to_integer(unsigned(SCORE_V)));
		elsif to_integer(HSCORE1) = 8 then
		  HSCORE1s := SPRITE8(to_integer(unsigned(SCORE_V)));
		elsif to_integer(HSCORE1) = 7 then
		  HSCORE1s := SPRITE7(to_integer(unsigned(SCORE_V)));
		elsif to_integer(HSCORE1) = 6 then
		  HSCORE1s := SPRITE6(to_integer(unsigned(SCORE_V)));
		elsif to_integer(HSCORE1) = 5 then
		  HSCORE1s := SPRITE5(to_integer(unsigned(SCORE_V)));
		elsif to_integer(HSCORE1) = 4 then
		  HSCORE1s := SPRITE4(to_integer(unsigned(SCORE_V)));
		elsif to_integer(HSCORE1) = 3 then
		  HSCORE1s := SPRITE3(to_integer(unsigned(SCORE_V)));
		elsif to_integer(HSCORE1) = 2 then
		  HSCORE1s := SPRITE2(to_integer(unsigned(SCORE_V)));
		elsif to_integer(HSCORE1) = 1 then
		  HSCORE1s := SPRITE1(to_integer(unsigned(SCORE_V)));
		elsif to_integer(HSCORE1) = 0 then
		  HSCORE1s := SPRITE0(to_integer(unsigned(SCORE_V)));
		end if;	
		if (HSCORE1s(to_integer(unsigned(15- SCORE_H2))) = '1') then
		  VGA_R <= "1111111111";
		  VGA_G <= "0000000000";
		  VGA_B <= "0000000000";
		else
		  if circle = '1' and silhouette(to_integer(Ycount(8 downto 5))*20 + to_integer(Xcount(9 downto 5))) = '0' and hit = '1' then
			VGA_R <= "0000000000";
			VGA_G <= "1000000000";
			VGA_B <= "0000000000";
		    if CIRCLE_RADIUS > 46 and touched = '0' then
			  tempLose <= '1';
		    end if;
		  elsif circle = '1' and silhouette(to_integer(Ycount(8 downto 5))*20 + to_integer(Xcount(9 downto 5))) = '0' and hit = '0' then
			VGA_R <= "1000000000";
			VGA_G <= "0000000000";
			VGA_B <= "0000000000";
		  elsif circle = '1'and silhouette(to_integer(Ycount(8 downto 5))*20 + to_integer(Xcount(9 downto 5))) = '1' and hit = '1'  then
			VGA_R <= "0000000000";
			VGA_G <= "1111111111";
			VGA_B <= "0000000000";
			if CIRCLE_RADIUS > 46 then
				touched <= '1';
				tempLose <= '0';
			end if;
		  elsif circle = '1'and silhouette(to_integer(Ycount(8 downto 5))*20 + to_integer(Xcount(9 downto 5))) = '1' and hit = '0'  then
			VGA_R <= "1111111111";
			VGA_G <= "0000000000";
			VGA_B <= "0000000000";
			if CIRCLE_RADIUS > 46 then
				nlose <= '0';
			end if;
		  elsif silhouette(to_integer(Ycount(8 downto 5))*20 + to_integer(Xcount(9 downto 5))) = '1' then
			VGA_R <= "1111111111";
			VGA_G <= "1111111111";
			VGA_B <= "1111111111";
		  else
			VGA_R <= "0000000000";
			VGA_G <= "1111111111";
			VGA_B <= "1111111111";
		  end if;
		end if;
	  elsif to_integer(Hcount - (HSYNC+ HBACK_PORCH)) > 15 and to_integer(Hcount - (HSYNC+ HBACK_PORCH)) < 32  and to_integer(Vcount - (VSYNC + VBACK_PORCH)) < 480 and to_integer(Vcount - (VSYNC + VBACK_PORCH)) > 463 then
		if to_integer(HSCORE2) = 9 then
		  HSCORE2s := SPRITE9(to_integer(unsigned(SCORE_V)));
		elsif to_integer(HSCORE2) = 8 then
		  HSCORE2s := SPRITE8(to_integer(unsigned(SCORE_V)));
		elsif to_integer(HSCORE2) = 7 then
		  HSCORE2s := SPRITE7(to_integer(unsigned(SCORE_V)));
		elsif to_integer(HSCORE2) = 6 then
		  HSCORE2s := SPRITE6(to_integer(unsigned(SCORE_V)));
		elsif to_integer(HSCORE2) = 5 then
		  HSCORE2s := SPRITE5(to_integer(unsigned(SCORE_V)));
		elsif to_integer(HSCORE2) = 4 then
		  HSCORE2s := SPRITE4(to_integer(unsigned(SCORE_V)));
		elsif to_integer(HSCORE2) = 3 then
		  HSCORE2s := SPRITE3(to_integer(unsigned(SCORE_V)));
		elsif to_integer(HSCORE2) = 2 then
		  HSCORE2s := SPRITE2(to_integer(unsigned(SCORE_V)));
		elsif to_integer(HSCORE2) = 1 then
		  HSCORE2s := SPRITE1(to_integer(unsigned(SCORE_V)));
		elsif to_integer(HSCORE2) = 0 then
		  HSCORE2s := SPRITE0(to_integer(unsigned(SCORE_V)));
		end if;	
		if (HSCORE2s(to_integer(unsigned(15- SCORE_H2))) = '1') then
			VGA_R <= "1111111111";
			VGA_G <= "0000000000";
			VGA_B <= "0000000000";
		else
		  if circle = '1' and silhouette(to_integer(Ycount(8 downto 5))*20 + to_integer(Xcount(9 downto 5))) = '0' and hit = '1' then
			VGA_R <= "0000000000";
			VGA_G <= "1000000000";
			VGA_B <= "0000000000";
		    if CIRCLE_RADIUS > 46 and touched = '0' then
			  tempLose <= '1';
		    end if;
		  elsif circle = '1' and silhouette(to_integer(Ycount(8 downto 5))*20 + to_integer(Xcount(9 downto 5))) = '0' and hit = '0' then
			VGA_R <= "1000000000";
			VGA_G <= "0000000000";
			VGA_B <= "0000000000";
		  elsif circle = '1'and silhouette(to_integer(Ycount(8 downto 5))*20 + to_integer(Xcount(9 downto 5))) = '1' and hit = '1'  then
			VGA_R <= "0000000000";
			VGA_G <= "1111111111";
			VGA_B <= "0000000000";
			if CIRCLE_RADIUS > 46 then
				touched <= '1';
				tempLose <= '0';
			end if;
		  elsif circle = '1'and silhouette(to_integer(Ycount(8 downto 5))*20 + to_integer(Xcount(9 downto 5))) = '1' and hit = '0'  then
			VGA_R <= "1111111111";
			VGA_G <= "0000000000";
			VGA_B <= "0000000000";
			if CIRCLE_RADIUS > 46 then
				nlose <= '0';
			end if;
		  elsif silhouette(to_integer(Ycount(8 downto 5))*20 + to_integer(Xcount(9 downto 5))) = '1' then
			VGA_R <= "1111111111";
			VGA_G <= "1111111111";
			VGA_B <= "1111111111";
		  else
			VGA_R <= "0000000000";
			VGA_G <= "1111111111";
			VGA_B <= "1111111111";
		  end if;
		end if;
		elsif to_integer(Hcount - (HSYNC+ HBACK_PORCH)) > 31 and to_integer(Hcount - (HSYNC+ HBACK_PORCH)) < 48  and to_integer(Vcount - (VSYNC + VBACK_PORCH)) < 480 and to_integer(Vcount - (VSYNC + VBACK_PORCH)) > 463 then
		  if to_integer(HSCORE3) = 9 then
			HSCORE3s := SPRITE9(to_integer(unsigned(SCORE_V)));
		  elsif to_integer(HSCORE3) = 8 then
			HSCORE3s := SPRITE8(to_integer(unsigned(SCORE_V)));
		  elsif to_integer(HSCORE3) = 7 then
			HSCORE3s := SPRITE7(to_integer(unsigned(SCORE_V)));
		  elsif to_integer(HSCORE3) = 6 then
			HSCORE3s := SPRITE6(to_integer(unsigned(SCORE_V)));
		  elsif to_integer(HSCORE3) = 5 then
			HSCORE3s := SPRITE5(to_integer(unsigned(SCORE_V)));
		  elsif to_integer(HSCORE3) = 4 then
			HSCORE3s := SPRITE4(to_integer(unsigned(SCORE_V)));
		  elsif to_integer(HSCORE3) = 3 then
			HSCORE3s := SPRITE3(to_integer(unsigned(SCORE_V)));
		  elsif to_integer(HSCORE3) = 2 then
			HSCORE3s := SPRITE2(to_integer(unsigned(SCORE_V)));
		  elsif to_integer(HSCORE3) = 1 then
			HSCORE3s := SPRITE1(to_integer(unsigned(SCORE_V)));
		  elsif to_integer(HSCORE3) = 0 then
			HSCORE3s := SPRITE0(to_integer(unsigned(SCORE_V)));
		  end if;	
		  if (HSCORE3s(to_integer(unsigned(15- SCORE_H2))) = '1') then
			VGA_R <= "1111111111";
			VGA_G <= "0000000000";
			VGA_B <= "0000000000";
		  else
		  if circle = '1' and silhouette(to_integer(Ycount(8 downto 5))*20 + to_integer(Xcount(9 downto 5))) = '0' and hit = '1' then
			VGA_R <= "0000000000";
			VGA_G <= "1000000000";
			VGA_B <= "0000000000";
		    if CIRCLE_RADIUS > 46 and touched = '0' then
			  tempLose <= '1';
		    end if;
		  elsif circle = '1' and silhouette(to_integer(Ycount(8 downto 5))*20 + to_integer(Xcount(9 downto 5))) = '0' and hit = '0' then
			VGA_R <= "1000000000";
			VGA_G <= "0000000000";
			VGA_B <= "0000000000";
		  elsif circle = '1'and silhouette(to_integer(Ycount(8 downto 5))*20 + to_integer(Xcount(9 downto 5))) = '1' and hit = '1'  then
			VGA_R <= "0000000000";
			VGA_G <= "1111111111";
			VGA_B <= "0000000000";
			if CIRCLE_RADIUS > 46 then
				touched <= '1';
				tempLose <= '0';
			end if;
		  elsif circle = '1'and silhouette(to_integer(Ycount(8 downto 5))*20 + to_integer(Xcount(9 downto 5))) = '1' and hit = '0'  then
			VGA_R <= "1111111111";
			VGA_G <= "0000000000";
			VGA_B <= "0000000000";
			if CIRCLE_RADIUS > 46 then
				nlose <= '0';
			end if;
		  elsif silhouette(to_integer(Ycount(8 downto 5))*20 + to_integer(Xcount(9 downto 5))) = '1' then
			VGA_R <= "1111111111";
			VGA_G <= "1111111111";
			VGA_B <= "1111111111";
		  else
			VGA_R <= "0000000000";
			VGA_G <= "1111111111";
			VGA_B <= "1111111111";
		  end if;
		end if;
		elsif to_integer(Hcount - (HSYNC+ HBACK_PORCH)) > 47 and to_integer(Hcount - (HSYNC+ HBACK_PORCH)) < 64  and to_integer(Vcount - (VSYNC + VBACK_PORCH)) < 480 and to_integer(Vcount - (VSYNC + VBACK_PORCH)) > 463 then
		  if to_integer(HSCORE4) = 9 then
			HSCORE4s := SPRITE9(to_integer(unsigned(SCORE_V)));
		  elsif to_integer(HSCORE4) = 8 then
			HSCORE4s := SPRITE8(to_integer(unsigned(SCORE_V)));
		  elsif to_integer(HSCORE4) = 7 then
			HSCORE4s := SPRITE7(to_integer(unsigned(SCORE_V)));
		  elsif to_integer(HSCORE4) = 6 then
			HSCORE4s := SPRITE6(to_integer(unsigned(SCORE_V)));
		  elsif to_integer(HSCORE4) = 5 then
			HSCORE4s := SPRITE5(to_integer(unsigned(SCORE_V)));
		  elsif to_integer(HSCORE4) = 4 then
			HSCORE4s := SPRITE4(to_integer(unsigned(SCORE_V)));
		  elsif to_integer(HSCORE4) = 3 then
			HSCORE4s := SPRITE3(to_integer(unsigned(SCORE_V)));
		  elsif to_integer(HSCORE4) = 2 then
			HSCORE4s := SPRITE2(to_integer(unsigned(SCORE_V)));
		  elsif to_integer(HSCORE4) = 1 then
			HSCORE4s := SPRITE1(to_integer(unsigned(SCORE_V)));
		  elsif to_integer(HSCORE4) = 0 then
			HSCORE4s := SPRITE0(to_integer(unsigned(SCORE_V)));
		  end if;	
		if (HSCORE4s(to_integer(unsigned(15- SCORE_H2))) = '1') then
			VGA_R <= "1111111111";
			VGA_G <= "0000000000";
			VGA_B <= "0000000000";
		else
		  if circle = '1' and silhouette(to_integer(Ycount(8 downto 5))*20 + to_integer(Xcount(9 downto 5))) = '0' and hit = '1' then
			VGA_R <= "0000000000";
			VGA_G <= "1000000000";
			VGA_B <= "0000000000";
		    if CIRCLE_RADIUS > 46 and touched = '0' then
			  tempLose <= '1';
		    end if;
		  elsif circle = '1' and silhouette(to_integer(Ycount(8 downto 5))*20 + to_integer(Xcount(9 downto 5))) = '0' and hit = '0' then
			VGA_R <= "1000000000";
			VGA_G <= "0000000000";
			VGA_B <= "0000000000";
		  elsif circle = '1'and silhouette(to_integer(Ycount(8 downto 5))*20 + to_integer(Xcount(9 downto 5))) = '1' and hit = '1'  then
			VGA_R <= "0000000000";
			VGA_G <= "1111111111";
			VGA_B <= "0000000000";
			if CIRCLE_RADIUS > 46 then
				touched <= '1';
				tempLose <= '0';
			end if;
		  elsif circle = '1'and silhouette(to_integer(Ycount(8 downto 5))*20 + to_integer(Xcount(9 downto 5))) = '1' and hit = '0'  then
			VGA_R <= "1111111111";
			VGA_G <= "0000000000";
			VGA_B <= "0000000000";
			if CIRCLE_RADIUS > 46 then
				nlose <= '0';
			end if;
		  elsif silhouette(to_integer(Ycount(8 downto 5))*20 + to_integer(Xcount(9 downto 5))) = '1' then
			VGA_R <= "1111111111";
			VGA_G <= "1111111111";
			VGA_B <= "1111111111";
		  else
			VGA_R <= "0000000000";
			VGA_G <= "1111111111";
			VGA_B <= "1111111111";
		  end if;
		end if;	
	  elsif to_integer(Hcount - (HSYNC+ HBACK_PORCH)) > 576 and to_integer(Hcount - (HSYNC+ HBACK_PORCH)) < 593  and to_integer(Vcount - (VSYNC + VBACK_PORCH)) < 480 and to_integer(Vcount - (VSYNC + VBACK_PORCH)) > 463 then
		if to_integer(SCORE1) = 9 then
		  SCORE1s := SPRITE9(to_integer(unsigned(SCORE_V)));
		elsif to_integer(SCORE1) = 8 then
		  SCORE1s := SPRITE8(to_integer(unsigned(SCORE_V)));
		elsif to_integer(SCORE1) = 7 then
		  SCORE1s := SPRITE7(to_integer(unsigned(SCORE_V)));
		elsif to_integer(SCORE1) = 6 then
		  SCORE1s := SPRITE6(to_integer(unsigned(SCORE_V)));
		elsif to_integer(SCORE1) = 5 then
		  SCORE1s := SPRITE5(to_integer(unsigned(SCORE_V)));
		elsif to_integer(SCORE1) = 4 then
		  SCORE1s := SPRITE4(to_integer(unsigned(SCORE_V)));
		elsif to_integer(SCORE1) = 3 then
		  SCORE1s := SPRITE3(to_integer(unsigned(SCORE_V)));
		elsif to_integer(SCORE1) = 2 then
		  SCORE1s := SPRITE2(to_integer(unsigned(SCORE_V)));
		elsif to_integer(SCORE1) = 1 then
		  SCORE1s := SPRITE1(to_integer(unsigned(SCORE_V)));
		elsif to_integer(SCORE1) = 0 then
		  SCORE1s := SPRITE0(to_integer(unsigned(SCORE_V)));
		end if;	
		if (SCORE1s(to_integer(unsigned(15- SCORE_H2))) = '1') then
		  VGA_R <= "1111111111";
		  VGA_G <= "0000000000";
		  VGA_B <= "0000000000";
		else
		  if circle = '1' and silhouette(to_integer(Ycount(8 downto 5))*20 + to_integer(Xcount(9 downto 5))) = '0' and hit = '1' then
			VGA_R <= "0000000000";
			VGA_G <= "1000000000";
			VGA_B <= "0000000000";
		    if CIRCLE_RADIUS > 46 and touched = '0' then
			  tempLose <= '1';
		    end if;
		  elsif circle = '1' and silhouette(to_integer(Ycount(8 downto 5))*20 + to_integer(Xcount(9 downto 5))) = '0' and hit = '0' then
			VGA_R <= "1000000000";
			VGA_G <= "0000000000";
			VGA_B <= "0000000000";
		  elsif circle = '1'and silhouette(to_integer(Ycount(8 downto 5))*20 + to_integer(Xcount(9 downto 5))) = '1' and hit = '1'  then
			VGA_R <= "0000000000";
			VGA_G <= "1111111111";
			VGA_B <= "0000000000";
			if CIRCLE_RADIUS > 46 then
				touched <= '1';
				tempLose <= '0';
			end if;
		  elsif circle = '1'and silhouette(to_integer(Ycount(8 downto 5))*20 + to_integer(Xcount(9 downto 5))) = '1' and hit = '0'  then
			VGA_R <= "1111111111";
			VGA_G <= "0000000000";
			VGA_B <= "0000000000";
			if CIRCLE_RADIUS > 46 then
				nlose <= '0';
			end if;
		  elsif silhouette(to_integer(Ycount(8 downto 5))*20 + to_integer(Xcount(9 downto 5))) = '1' then
			VGA_R <= "1111111111";
			VGA_G <= "1111111111";
			VGA_B <= "1111111111";
		  else
			VGA_R <= "0000000000";
			VGA_G <= "1111111111";
			VGA_B <= "1111111111";
		  end if;
		end if;
	  elsif to_integer(Hcount - (HSYNC+ HBACK_PORCH)) > 592 and to_integer(Hcount - (HSYNC+ HBACK_PORCH)) < 608  and to_integer(Vcount - (VSYNC + VBACK_PORCH)) < 480 and to_integer(Vcount - (VSYNC + VBACK_PORCH)) > 463 then
		if to_integer(SCORE2) = 9 then
		  SCORE2s := SPRITE9(to_integer(unsigned(SCORE_V)));
		elsif to_integer(SCORE2) = 8 then
		  SCORE2s := SPRITE8(to_integer(unsigned(SCORE_V)));
		elsif to_integer(SCORE2) = 7 then
		  SCORE2s := SPRITE7(to_integer(unsigned(SCORE_V)));
		elsif to_integer(SCORE2) = 6 then
		  SCORE2s := SPRITE6(to_integer(unsigned(SCORE_V)));
		elsif to_integer(SCORE2) = 5 then
		  SCORE2s := SPRITE5(to_integer(unsigned(SCORE_V)));
		elsif to_integer(SCORE2) = 4 then
		  SCORE2s := SPRITE4(to_integer(unsigned(SCORE_V)));
		elsif to_integer(SCORE2) = 3 then
		  SCORE2s := SPRITE3(to_integer(unsigned(SCORE_V)));
		elsif to_integer(SCORE2) = 2 then
		  SCORE2s := SPRITE2(to_integer(unsigned(SCORE_V)));
		elsif to_integer(SCORE2) = 1 then
		  SCORE2s := SPRITE1(to_integer(unsigned(SCORE_V)));
		elsif to_integer(SCORE2) = 0 then
		  SCORE2s := SPRITE0(to_integer(unsigned(SCORE_V)));
		end if;	
		if (SCORE2s(to_integer(unsigned(15- SCORE_H2))) = '1') then
			VGA_R <= "1111111111";
			VGA_G <= "0000000000";
			VGA_B <= "0000000000";
		else
		  if circle = '1' and silhouette(to_integer(Ycount(8 downto 5))*20 + to_integer(Xcount(9 downto 5))) = '0' and hit = '1' then
			VGA_R <= "0000000000";
			VGA_G <= "1000000000";
			VGA_B <= "0000000000";
		    if CIRCLE_RADIUS > 46 and touched = '0' then
			  tempLose <= '1';
		    end if;
		  elsif circle = '1' and silhouette(to_integer(Ycount(8 downto 5))*20 + to_integer(Xcount(9 downto 5))) = '0' and hit = '0' then
			VGA_R <= "1000000000";
			VGA_G <= "0000000000";
			VGA_B <= "0000000000";
		  elsif circle = '1'and silhouette(to_integer(Ycount(8 downto 5))*20 + to_integer(Xcount(9 downto 5))) = '1' and hit = '1'  then
			VGA_R <= "0000000000";
			VGA_G <= "1111111111";
			VGA_B <= "0000000000";
			if CIRCLE_RADIUS > 46 then
				touched <= '1';
				tempLose <= '0';
			end if;
		  elsif circle = '1'and silhouette(to_integer(Ycount(8 downto 5))*20 + to_integer(Xcount(9 downto 5))) = '1' and hit = '0'  then
			VGA_R <= "1111111111";
			VGA_G <= "0000000000";
			VGA_B <= "0000000000";
			if CIRCLE_RADIUS > 46 then
				nlose <= '0';
			end if;
		  elsif silhouette(to_integer(Ycount(8 downto 5))*20 + to_integer(Xcount(9 downto 5))) = '1' then
			VGA_R <= "1111111111";
			VGA_G <= "1111111111";
			VGA_B <= "1111111111";
		  else
			VGA_R <= "0000000000";
			VGA_G <= "1111111111";
			VGA_B <= "1111111111";
		  end if;
		end if;
		elsif to_integer(Hcount - (HSYNC+ HBACK_PORCH)) > 607 and to_integer(Hcount - (HSYNC+ HBACK_PORCH)) < 624  and to_integer(Vcount - (VSYNC + VBACK_PORCH)) < 480 and to_integer(Vcount - (VSYNC + VBACK_PORCH)) > 463 then
		  if to_integer(SCORE3) = 9 then
			SCORE3s := SPRITE9(to_integer(unsigned(SCORE_V)));
		  elsif to_integer(SCORE3) = 8 then
			SCORE3s := SPRITE8(to_integer(unsigned(SCORE_V)));
		  elsif to_integer(SCORE3) = 7 then
			SCORE3s := SPRITE7(to_integer(unsigned(SCORE_V)));
		  elsif to_integer(SCORE3) = 6 then
			SCORE3s := SPRITE6(to_integer(unsigned(SCORE_V)));
		  elsif to_integer(SCORE3) = 5 then
			SCORE3s := SPRITE5(to_integer(unsigned(SCORE_V)));
		  elsif to_integer(SCORE3) = 4 then
			SCORE3s := SPRITE4(to_integer(unsigned(SCORE_V)));
		  elsif to_integer(SCORE3) = 3 then
			SCORE3s := SPRITE3(to_integer(unsigned(SCORE_V)));
		  elsif to_integer(SCORE3) = 2 then
			SCORE3s := SPRITE2(to_integer(unsigned(SCORE_V)));
		  elsif to_integer(SCORE3) = 1 then
			SCORE3s := SPRITE1(to_integer(unsigned(SCORE_V)));
		  elsif to_integer(SCORE3) = 0 then
			SCORE3s := SPRITE0(to_integer(unsigned(SCORE_V)));
		  end if;	
		  if (SCORE3s(to_integer(unsigned(15- SCORE_H2))) = '1') then
			VGA_R <= "1111111111";
			VGA_G <= "0000000000";
			VGA_B <= "0000000000";
		  else
		  if circle = '1' and silhouette(to_integer(Ycount(8 downto 5))*20 + to_integer(Xcount(9 downto 5))) = '0' and hit = '1' then
			VGA_R <= "0000000000";
			VGA_G <= "1000000000";
			VGA_B <= "0000000000";
		    if CIRCLE_RADIUS > 46 and touched = '0' then
			  tempLose <= '1';
		    end if;
		  elsif circle = '1' and silhouette(to_integer(Ycount(8 downto 5))*20 + to_integer(Xcount(9 downto 5))) = '0' and hit = '0' then
			VGA_R <= "1000000000";
			VGA_G <= "0000000000";
			VGA_B <= "0000000000";
		  elsif circle = '1'and silhouette(to_integer(Ycount(8 downto 5))*20 + to_integer(Xcount(9 downto 5))) = '1' and hit = '1'  then
			VGA_R <= "0000000000";
			VGA_G <= "1111111111";
			VGA_B <= "0000000000";
			if CIRCLE_RADIUS > 46 then
				touched <= '1';
				tempLose <= '0';
			end if;
		  elsif circle = '1'and silhouette(to_integer(Ycount(8 downto 5))*20 + to_integer(Xcount(9 downto 5))) = '1' and hit = '0'  then
			VGA_R <= "1111111111";
			VGA_G <= "0000000000";
			VGA_B <= "0000000000";
			if CIRCLE_RADIUS > 46 then
				nlose <= '0';
			end if;
		  elsif silhouette(to_integer(Ycount(8 downto 5))*20 + to_integer(Xcount(9 downto 5))) = '1' then
			VGA_R <= "1111111111";
			VGA_G <= "1111111111";
			VGA_B <= "1111111111";
		  else
			VGA_R <= "0000000000";
			VGA_G <= "1111111111";
			VGA_B <= "1111111111";
		  end if;
		end if;
		elsif to_integer(Hcount - (HSYNC+ HBACK_PORCH)) > 623 and to_integer(Hcount - (HSYNC+ HBACK_PORCH)) < 640  and to_integer(Vcount - (VSYNC + VBACK_PORCH)) < 480 and to_integer(Vcount - (VSYNC + VBACK_PORCH)) > 463 then
		  if to_integer(SCORE4) = 9 then
			SCORE4s := SPRITE9(to_integer(unsigned(SCORE_V)));
		  elsif to_integer(SCORE4) = 8 then
			SCORE4s := SPRITE8(to_integer(unsigned(SCORE_V)));
		  elsif to_integer(SCORE4) = 7 then
			SCORE4s := SPRITE7(to_integer(unsigned(SCORE_V)));
		  elsif to_integer(SCORE4) = 6 then
			SCORE4s := SPRITE6(to_integer(unsigned(SCORE_V)));
		  elsif to_integer(SCORE4) = 5 then
			SCORE4s := SPRITE5(to_integer(unsigned(SCORE_V)));
		  elsif to_integer(SCORE4) = 4 then
			SCORE4s := SPRITE4(to_integer(unsigned(SCORE_V)));
		  elsif to_integer(SCORE4) = 3 then
			SCORE4s := SPRITE3(to_integer(unsigned(SCORE_V)));
		  elsif to_integer(SCORE4) = 2 then
			SCORE4s := SPRITE2(to_integer(unsigned(SCORE_V)));
		  elsif to_integer(SCORE4) = 1 then
			SCORE4s := SPRITE1(to_integer(unsigned(SCORE_V)));
		  elsif to_integer(SCORE4) = 0 then
			SCORE4s := SPRITE0(to_integer(unsigned(SCORE_V)));
		  end if;	
		  if (SCORE4s(to_integer(unsigned(15- SCORE_H2))) = '1') then
			VGA_R <= "1111111111";
			VGA_G <= "0000000000";
			VGA_B <= "0000000000";
		  else
			if circle = '1' and silhouette(to_integer(Ycount(8 downto 5))*20 + to_integer(Xcount(9 downto 5))) = '0' and hit = '1' then
			  VGA_R <= "0000000000";
			  VGA_G <= "1000000000";
			  VGA_B <= "0000000000";
			  if CIRCLE_RADIUS > 46 and touched = '0' then
			    tempLose <= '1';
		      end if;
			elsif circle = '1' and silhouette(to_integer(Ycount(8 downto 5))*20 + to_integer(Xcount(9 downto 5))) = '0' and hit = '0' then
			  VGA_R <= "1000000000";
			  VGA_G <= "0000000000";
			  VGA_B <= "0000000000";
			elsif circle = '1'and silhouette(to_integer(Ycount(8 downto 5))*20 + to_integer(Xcount(9 downto 5))) = '1' and hit = '1'  then
			  VGA_R <= "0000000000";
			  VGA_G <= "1111111111";
			  VGA_B <= "0000000000";
			  if CIRCLE_RADIUS > 46 then
				touched <= '1';
				tempLose <= '0';
			  end if;
			elsif circle = '1'and silhouette(to_integer(Ycount(8 downto 5))*20 + to_integer(Xcount(9 downto 5))) = '1' and hit = '0'  then
			  VGA_R <= "1111111111";
			  VGA_G <= "0000000000";
			  VGA_B <= "0000000000";
			  if CIRCLE_RADIUS > 46 then
				nlose <= '0';
			  end if;
			elsif silhouette(to_integer(Ycount(8 downto 5))*20 + to_integer(Xcount(9 downto 5))) = '1' then
			  VGA_R <= "1111111111";
			  VGA_G <= "1111111111";
			  VGA_B <= "1111111111";
		    else
			  VGA_R <= "0000000000";
			  VGA_G <= "1111111111";
			  VGA_B <= "1111111111";
		    end if;
		  end if;
	  elsif circle = '1' and silhouette(to_integer(Ycount(8 downto 5))*20 + to_integer(Xcount(9 downto 5))) = '0' and hit = '1' then
		VGA_R <= "0000000000";
		VGA_G <= "1000000000";
		VGA_B <= "0000000000";
		if CIRCLE_RADIUS > 46 and touched = '0' then
			tempLose <= '1';
		end if;
	  elsif circle = '1' and silhouette(to_integer(Ycount(8 downto 5))*20 + to_integer(Xcount(9 downto 5))) = '0' and hit = '0' then
		VGA_R <= "1000000000";
		VGA_G <= "0000000000";
		VGA_B <= "0000000000";
	  elsif circle = '1'and silhouette(to_integer(Ycount(8 downto 5))*20 + to_integer(Xcount(9 downto 5))) = '1' and hit = '1'  then
		VGA_R <= "0000000000";
		VGA_G <= "1111111111";
		VGA_B <= "0000000000";
		if CIRCLE_RADIUS > 46 then
			touched <= '1';
			tempLose <= '0';
		end if;
	  elsif circle = '1'and silhouette(to_integer(Ycount(8 downto 5))*20 + to_integer(Xcount(9 downto 5))) = '1' and hit = '0'  then
		VGA_R <= "1111111111";
		VGA_G <= "0000000000";
		VGA_B <= "0000000000";
		if CIRCLE_RADIUS > 46 then
			nlose <= '0';
		end if;
	  elsif silhouette(to_integer(Ycount(8 downto 5))*20 + to_integer(Xcount(9 downto 5))) = '1' then
		VGA_R <= "1111111111";
        VGA_G <= "1111111111";
        VGA_B <= "1111111111";

      else
        VGA_R <= "0000000000";
        VGA_G <= "1111111111";
        VGA_B <= "1111111111";
      end if;
    end if;
  end process VideoOut;

  VGA_CLK <= clk;
  VGA_HS <= not vga_hsync;
  VGA_VS <= not vga_vsync;
  VGA_SYNC <= '0';
  VGA_BLANK <= not (vga_hsync or vga_vsync);

end rtl;
