library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;

entity GND_TOP is

	port (
		-- Clocks
		CLOCK_27,                                   -- 27 MHz
		CLOCK_50 : in std_logic;                    -- 50 MHz
		
		-- Buttons and switches
		KEY : in std_logic_vector(3 downto 0);      -- Push buttons
		
		-- LED displays
		HEX0, HEX1, HEX2, HEX3                      -- 7-segment displays
		: out std_logic_vector(6 downto 0);         -- (active low)
		
		-- I2C bus
		I2C_SDAT : inout std_logic;                 -- I2C Data
		I2C_SCLK : out std_logic;                   -- I2C Clock
		
		-- SRAM
		SRAM_DQ : inout std_logic_vector(15 downto 0); -- Data bus 16 Bits
		SRAM_ADDR : out std_logic_vector(17 downto 0); -- Address bus 18 Bits
		SRAM_UB_N,                                     -- High-byte Data Mask
		SRAM_LB_N,                                     -- Low-byte Data Mask
		SRAM_WE_N,                                     -- Write Enable
		SRAM_CE_N,                                     -- Chip Enable
		SRAM_OE_N : out std_logic;                     -- Output Enable
		
		-- VGA output
		VGA_CLK,                                    -- Clock
		VGA_HS,                                     -- H_SYNC
		VGA_VS,                                     -- V_SYNC
		VGA_BLANK,                                  -- BLANK
		VGA_SYNC : out std_logic;                   -- SYNC
		VGA_R,                                      -- Red[9:0]
		VGA_G,                                      -- Green[9:0]
		VGA_B : out unsigned(9 downto 0);           -- Blue[9:0]
		
		-- Video Decoder
		TD_DATA : in std_logic_vector(7 downto 0);  -- Data bus 8 bits
		TD_HS,                                      -- H_SYNC
		TD_VS : in std_logic;                       -- V_SYNC
		TD_RESET : out std_logic                    -- Reset
	);

end GND_TOP;

architecture arch of GND_TOP is

	component I2C_AV_Config port(
		iCLK : in std_logic;
		iRST_N : in std_logic;
		I2C_SCLK : out std_logic;
		I2C_SDAT : inout std_logic
	);
	end component;
	
	component TV_to_VGA port (
		OSC_27 : in std_logic;
		RESET : in std_logic;
		VGA_BLANK : out std_logic;
		VGA_SYNC : out std_logic;
		VGA_CLOCK : out std_logic;
		VGA_HS : out std_logic;
		VGA_VS : out std_logic;
		x640 : out unsigned(9 downto 0);
		y480 : out unsigned(8 downto 0);
		Y : out unsigned(7 downto 0);
		Cb : out unsigned(7 downto 0);
		Cr : out unsigned(7 downto 0);
		TD_D : in std_logic_vector(7 downto 0);
		TD_HS : in std_logic;
		TD_VS : in std_logic
	);
	end component;
	
	-- DECODER output signals
	signal vga_blank_sig : std_logic;
	signal vga_sync_sig : std_logic;
	signal vga_clock_sig : std_logic;
	signal vga_hs_sig : std_logic;
	signal vga_vs_sig : std_logic;
	
	signal x640 : unsigned(9 downto 0);
	signal y480 : unsigned(8 downto 0);
	signal Y : unsigned(7 downto 0);
	signal Cb : unsigned(7 downto 0);
	signal Cr : unsigned(7 downto 0);
	
	-- DOWNSCALER output signals
	signal x80 : unsigned(6 downto 0);
	signal y60 : unsigned(5 downto 0);
	signal data : unsigned(23 downto 0);
	
	-- BACKGROUND_RAM output signal
	signal bg : unsigned(23 downto 0);
	
	-- FOREGROUND_RAM output signal
	signal fg : unsigned(23 downto 0);
	
	-- SILHOUETTE_GENERATOR output signals
	signal rx : unsigned(6 downto 0);
	signal ry : unsigned(5 downto 0);
	signal sil_x : unsigned(4 downto 0);
	signal sil_y : unsigned(3 downto 0);
	signal sil : std_logic;
	
	-- signals for SRAM
	signal h  : std_logic;
	signal ch : std_logic_vector(11 downto 0);
	signal cv : std_logic_vector(11 downto 0);
	signal r  : std_logic_vector(5 downto 0);

begin
	
	TD_RESET <= '1'; -- necessary for CLOCK_27
	
	I2C : I2C_AV_Config port map ( -- necessary for TV Decoder to get video data
		iCLK => CLOCK_50,
		iRST_N => '1',
		I2C_SCLK => I2C_SCLK,
		I2C_SDAT => I2C_SDAT
	);
	
	DECODER : TV_to_VGA port map (
		OSC_27 => CLOCK_27, -- must be clocked at 27 MHz
		RESET => '0',
		VGA_BLANK => vga_blank_sig,
		VGA_SYNC => vga_sync_sig,
		VGA_CLOCK => vga_clock_sig,
		VGA_HS => vga_hs_sig,
		VGA_VS => vga_vs_sig,
		x640 => x640,
		y480 => y480,
		Y => Y,
		Cb => Cb,
		Cr => Cr,
		TD_D => TD_DATA,
		TD_HS => TD_HS,
		TD_VS => TD_VS
	);
	
	DOWNSCALER : entity work.downscaler port map (
		clk => vga_clock_sig,
		blank => vga_blank_sig,
		sync => vga_sync_sig,
		hs => vga_hs_sig,
		vs => vga_vs_sig,
		x640 => x640,
		y480 => y480,
		Y => Y,
		Cb => Cb,
		Cr => Cr,
		x80 => x80,
		y60 => y60,
		YCbCr => data
	);
	
	BACKGROUND_RAM : entity work.ram_img port map (
		wclk => CLOCK_50,
		rclk => CLOCK_50,
		we => not KEY(0),
		wx => x80,
		wy => y60,
		di => data,
		rx => rx,
		ry => ry,
		do => bg
	);
	
	FOREGROUND_RAM : entity work.ram_img port map (
		wclk => CLOCK_50,
		rclk => CLOCK_50,
		we => '1',
		wx => x80,
		wy => y60,
		di => data,
		rx => rx,
		ry => ry,
		do => fg
	);
	
	SILHOUETTE_GENERATOR : entity work.silhouette_generator port map (
		clk => CLOCK_27, -- must be clocked at 27 MHz to be in sync with TV_to_VGA and VGA_RASTER
		fg => fg,
		bg => bg,
		rx => rx,
		ry => ry,
		xo => sil_x,
		yo => sil_y,
		do => sil
	);
	
	VGA_RASTER : entity work.vga_raster port map (
		reset => '0',
		clk => CLOCK_27, -- must be 27 MHz to be in sync with TV_to_VGA (25 MHz does not work well)
		x => sil_x,
		y => sil_y,
		sil => sil,
		
		ch_in => unsigned(ch),
		cv_in => unsigned(cv),
		h_in => h,
		r_in => unsigned(r),
		
		goalie => not KEY(2),
		dodge => not KEY(1),
		ninja => not KEY(3),
		HEX0 => HEX0,
		HEX1 => HEX1,
		HEX2 => HEX2,
		HEX3 => HEX3,
		
		VGA_CLK => VGA_CLK,
		VGA_HS => VGA_HS,
		VGA_VS => VGA_VS,
		VGA_BLANK => VGA_BLANK,
		VGA_SYNC => VGA_SYNC,
		VGA_R => VGA_R,
		VGA_G => VGA_G,
		VGA_B => VGA_B
	);

	nios: entity work.nios_system port map(
		SRAM_ADDR_from_the_sram => SRAM_ADDR,
		SRAM_CE_N_from_the_sram => SRAM_CE_N,
		SRAM_DQ_to_and_from_the_sram => SRAM_DQ,
		SRAM_LB_N_from_the_sram => SRAM_LB_N,
		SRAM_OE_N_from_the_sram => SRAM_OE_N,
		SRAM_UB_N_from_the_sram => SRAM_UB_N,
		SRAM_WE_N_from_the_sram => SRAM_WE_N,
		ch_from_the_transfer => ch,
		cv_from_the_transfer => cv,
		h_from_the_transfer => h,
		r_from_the_transfer => r,
		clk => CLOCK_50,
		reset_n => '1'
	);

end arch;
