library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;

entity audio_driver is 
port(
	clock_50 : in std_logic;
	clock_18 : in std_logic;
	
	clk        : in  std_logic;
	reset_n1   : in  std_logic;
	read       : in  std_logic;
	write      : in  std_logic;
	chipselect : in  std_logic;
	address    : in  unsigned(4 downto 0);
	readdata   : out unsigned(31 downto 0);
	--writedata  : in  unsigned(31 downto 0);
	cpu_cmd : in std_logic_vector(31 downto 0);

	-- Audio interface signals
    AUD_ADCLRCK  : out  std_logic;   --    Audio CODEC ADC LR Clock
    AUD_ADCDAT   : in   std_logic;   --    Audio CODEC ADC Data
    AUD_DACLRCK  : out  std_logic;   --    Audio CODEC DAC LR Clock
    AUD_DACDAT   : out  std_logic;   --    Audio CODEC DAC Data
    AUD_BCLK     : inout std_logic  --    Audio CODEC Bit-Stream Clock
);
end audio_driver;


architecture behavior of audio_driver is

signal disable : std_logic; -- when '1' disable audio module
signal reset_n : std_logic; -- when '0' reset audio module
signal sound_sel : std_logic_vector(3 downto 0); -- when "0001", play "fire", when "0010", play "explosion"
signal play_finish1 : std_logic;-- exp
signal play_finish2 : std_logic;-- fire
signal play_finish3 : std_logic;-- falling down
signal reset_sm : std_logic; -- when '1' reset state machine

component de2_wm8731_audio is
port (
    clk : in std_logic;       --  Audio CODEC Chip Clock AUD_XCK (18.43 MHz)
    reset_n : in std_logic;
    clk_50 : in std_logic;
    disable : in std_logic;
    sound : in std_logic_vector(3 downto 0); -- select which sound will be played
    sound_finish1 : out std_logic;-- exp
    sound_finish2 : out std_logic;-- fire
    sound_finish3 : out std_logic;-- fall
	--sound_finish4 : out std_logic;-- begin
  
    -- Audio interface signals
    AUD_ADCLRCK  : out  std_logic;   --    Audio CODEC ADC LR Clock
    AUD_ADCDAT   : in   std_logic;   --    Audio CODEC ADC Data
    AUD_DACLRCK  : out  std_logic;   --    Audio CODEC DAC LR Clock
    AUD_DACDAT   : out  std_logic;   --    Audio CODEC DAC Data
    AUD_BCLK     : inout std_logic  --    Audio CODEC Bit-Stream Clock
  );
end  component;

signal audio_request : std_logic;

type state is (s0, bullet, explode, falling);
signal fsm_state : state;

begin

reset_sm <= cpu_cmd(28);

-- state machine of player
process(clock_50,reset_sm, cpu_cmd, fsm_state)
begin
	if rising_edge(clock_50) then
		if(reset_sm = '1') then
			fsm_state <= s0;
		elsif fsm_state = s0 then
			if cpu_cmd = x"00000061" then
				fsm_state <= bullet;
			elsif cpu_cmd = x"00000062" then
				fsm_state <= explode;
			elsif cpu_cmd = x"00000063" then
				fsm_state <= falling;
			else	
				fsm_state <= s0;
			end if;
		elsif fsm_state = bullet then
			if cpu_cmd = x"00000062" then
				fsm_state <= explode;
			elsif cpu_cmd = x"00000063" then
				fsm_state <= falling;
			elsif play_finish2 = '1' then
				fsm_state <= s0;
			else
				fsm_state <= bullet;
			end if;
		elsif fsm_state = explode then
			if cpu_cmd = x"00000061" then
				fsm_state <= bullet;
			elsif cpu_cmd = x"00000063" then
				fsm_state <= falling;
			elsif play_finish1 = '1' then
				fsm_state <= s0;
			else
				fsm_state <= explode;
			end if;
		elsif fsm_state = falling then
			if cpu_cmd = x"00000061" then
				fsm_state <= bullet;
			elsif cpu_cmd = x"00000062" then
				fsm_state <= explode;
			elsif play_finish3 = '1' then
				fsm_state <= s0;
			else
				fsm_state <= falling;
			end if;		
		end if;
	end if;
end process;

reset_n <= '0' when fsm_state = s0 else
		   '1' when fsm_state = bullet else
		   '1' when fsm_state = explode else
		   '1' when fsm_state = falling else
		   '1';

disable <= '1' when fsm_state = s0 else
           '0' when fsm_state = bullet else
		   '0' when fsm_state = explode else
		   '0' when fsm_state = falling else
		   '0';

sound_sel <= "0001" when fsm_state = bullet else
			 "0010" when fsm_state = explode else
			 "0100" when fsm_state = falling else
			 "0000";

--port map to the wm8731 module
audio: de2_wm8731_audio port map(

    clk => clock_18,
    reset_n => reset_n,
    clk_50 =>clock_50,
    disable => disable,
    sound => sound_sel,
    sound_finish1 => play_finish1,
    sound_finish2 => play_finish2,
    sound_finish3 => play_finish3,
  
    -- Audio interface signals
    AUD_ADCLRCK  => AUD_ADCLRCK,
    AUD_ADCDAT => AUD_ADCDAT,
    AUD_DACLRCK => AUD_DACLRCK,
    AUD_DACDAT =>  AUD_DACDAT,  
    AUD_BCLK => AUD_BCLK    
);

end architecture;