-------------------------------------------------------------------------------
--
-- Simple Zapper Controller
--
-- Scott Rogowski
-- smr2167@columbia.edu
--
-------------------------------------------------------------------------------
library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;

entity de2_zapper_controller is
  port (
    reset : in std_logic;
    clk   : in std_logic;                    -- Should be 25.125 MHz
    read       : in  std_logic;
    write      : in  std_logic;
    chipselect : in  std_logic;
    address    : in  unsigned(4 downto 0);
    readdata   : out unsigned(15 downto 0);
    writedata  : in  unsigned(15 downto 0);

	 --Pins
    GPIO_1: inout std_logic_vector(35 downto 0);

	 --To the VGA
	 flash_zapper : out std_logic;
	 triggerpull_zapper : out std_logic
    );

end de2_zapper_controller;

architecture rtl of de2_zapper_controller is
	--Translate GPIO into human readable signals
	signal trigger : std_logic:='0';
	signal ison : std_logic:='0';
	
	--Three signals to keep track of the trigger state
	signal triggerBeingHeld :std_logic :='0';
	signal waitForTriggerBounce: integer :=0;
	signal triggerAction: std_logic :='0'; --Once we finally decide to issue a trigger interrupt
	
	--amount of cycles to wait before we're sure the switch was released. 
	--not a concrete number, but it eliminates the bounce problem and is not noticably long
	constant SWITCHBOUNCETIME : integer := 200000;  	

	--A running count of the shots
	signal shots : unsigned(4 downto 0) := "00000";
	
	signal clk25 : std_logic := '0';
	
	begin
	
 process (clk) begin
	if rising_edge(clk) then
		clk25 <= not clk25;
		end if;
	end process;
	

	HumanReadable : process (clk25) begin
		if GPIO_1(0) = '1' then
			ison <= '1';
		else
			ison <= '0';
			end if;
		if GPIO_1(1) = '0' then    --the trigger is a high signal until the trigger pulls it low
			trigger <= '1';
		else
			trigger <= '0';
			end if;
		end process;
		
	--the purpose of this process is to prevent the screen from staying blanked while the trigger is held down
	--So, while the trigger is held, do nothing, upon release, wait for the switchbouncetime
	ZapperInterface : process (clk25) begin

		
		if rising_edge(clk25) then
			--Reset shot count if needed
			if chipselect = '1' and write = '1' then 
				if writedata(0) = '1' then
					shots <= "00000";
					end if;
			--Store when the trigger is being held
			elsif trigger='1' then 										 
				triggerBeingHeld <= '1';
			--Capture the moment the trigger goes from 1 to 0
			--start a timer to minimize bounce effects
			elsif triggerBeingHeld='1' and trigger = '0' then 
				waitForTriggerBounce <= SWITCHBOUNCETIME;					 
				triggerBeingHeld <= '0';
			--Upon the completion of bounce time, start the trigger action
			elsif waitForTriggerBounce = 1 then								 
				triggerAction <= '1';
				waitForTriggerBounce <= 0;
				shots <= shots + 1;
			--Immediately, upon the next clock cycle, the trigger is over
			elsif triggerAction = '1' then
				triggerAction <= '0';
			--Decrement the timer
			else 
				waitForTriggerBounce <= waitForTriggerBounce-1;
				end if;
			end if;
		end process;

	flash_zapper <= ison;
	triggerpull_zapper <= triggerAction;
	
	GPIO_1(35 downto 2) <= "0000000000000000000000000000000000";
	
	readdata <= "00000000000" & shots;
	
	end rtl;