-------------------------------------------------------------------------------
--
-- VGA Raster Display using Run-Length Encoding and Sprites stored in a ROM
-- Originally by Professor Stephen Edwards
-- Heavily Edited by the Duck Feed Group
-------------------------------------------------------------------------------
library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;

entity de2_vga_raster is

	port (
		reset : in std_logic;
		clk   : in std_logic;                    -- Should be 25.125 MHz
		read       : in  std_logic;
		write      : in  std_logic;
		chipselect : in  std_logic;
		address    : in  unsigned(4 downto 0);
		readdata   : out unsigned(15 downto 0);
		writedata  : in  unsigned(15 downto 0);

		VGA_CLK,                        				-- Clock
		VGA_HS,                          			-- H_SYNC
		VGA_VS,                          			-- V_SYNC
		VGA_BLANK,                      				-- BLANK
		VGA_SYNC : out std_logic;  			 		-- SYNC
		VGA_R,                           			-- Red[9:0]
		VGA_G,                           			-- Green[9:0]
		VGA_B : out std_logic_vector(9 downto 0); -- Blue[9:0]

		flash_vga : in std_logic;
		triggerpull_vga : in std_logic;

		red_pixel_vga: in std_logic_vector(9 downto 0)
		);

end de2_vga_raster;

architecture rtl of de2_vga_raster is
 
	-- Video parameters
 
	constant HTOTAL       : integer := 800;
	constant HSYNC        : integer := 96;
	constant HBACK_PORCH  : integer := 48;
	constant HACTIVE      : integer := 640;
	constant HFRONT_PORCH : integer := 16;
 
	constant VTOTAL       : integer := 525;
	constant VSYNC        : integer := 2;
	constant VBACK_PORCH  : integer := 33;
	constant VACTIVE      : integer := 480;
	constant VFRONT_PORCH : integer := 10;

--*10 just for debugging
	constant BLACKSCREENTIME : integer := 332500;       -- was 450000;  	--this is the time from end of field to a full screen blackening.
	constant SHOWTARGETTIME : integer := 450000;--450000;
	constant RADIUS       : integer := 40;
	constant SCOREHEIGHT  : integer := 40;
	constant SCOREWIDTH : integer := 40;
--	constant FishHEIGHT  : integer := 20;
--	constant FishWIDTH : integer := 20;
	

	-- Signals for the video controller
	signal Hcount : unsigned(9 downto 0) := "0000000000";  -- Horizontal position (0-800)
	signal Vcount : unsigned(9 downto 0) := "0000000000";  -- Vertical position (0-524)
	signal EndOfLine, EndOfField : std_logic := '0';

	signal vga_hblank, vga_hsync,
		vga_vblank, vga_vsync : std_logic := '0';  -- Sync. signals

	--Secondary clocks for VGA timing
	signal clk25 : std_logic := '0';
	
	--The duck position storage variables
	type position is array(0 to 7) of integer;
	signal hcenter : position := (others => 0);
	signal vcenter : position := (others => 0);
	signal old_hcenter : position := (others => 0);
	
	signal hScore : position := (600,650,700,0,0,0,0,0);
	signal vScore : position := (others => 430);
	
----	type fishPosition is array(0 to 15) of integer;
----
----	signal hFish : fishposition := (110,140,170,200,230,260,290,320,350,380,410,440,470,500,530,580);
----	signal vFish : fishposition := (others => 420);
	


	-- square at current pixel boolean
--	type squaretype is array (0 to 7) of std_logic;
	type squaretype is array (0 to 8) of std_logic;
--	signal square: squaretype := "00000000";
	signal square: squaretype := "000000000";
	signal squareScore: squaretype := "000000000";
----	type squareFishtype is array (0 to 15) of std_logic;
----	signal squareFish: squareFishtype := "0000000000000000";
	signal fishSquare : std_logic := '0';
	signal fishAddress : integer := 0;
	signal fishOpaque : std_logic := '0';

	signal duckHit: unsigned (7 downto 0) := "00000000";
	signal duckVisible: unsigned(7 downto 0) := "11111111";
	
	signal score : integer;
	
	signal gameReset : std_logic := '1';
	signal levelReset : std_logic := '1';
	
	--zapper signals
	signal hold_2: std_logic := '0';
	signal waitFor: integer := 0; --For 525*800
	signal waitForBlackScreen: integer :=0; --For 525*800
	signal waitForTarget: integer :=0;
	signal startTargetFlash : std_logic :='0';
	signal blackScreen : std_logic := '0';
	signal showTarget : integer := 0;
	signal blankOnNextScreen : std_logic :='0';
	signal targetFindingComplete : std_logic :='0';

	
	signal anyHit: std_logic := '0';
	signal targetFours: std_logic := '0';
	signal targetTwos: std_logic := '0';
	signal targetOnes: std_logic := '0';
	signal targetHit: integer := 0;

	signal backgroundPixelsRemaining : integer := 1;
	signal rCurrent : std_logic_vector (9 downto 0) := "1111111111";
	signal gCurrent : std_logic_vector (9 downto 0) := "0000000000";
	signal bCurrent : std_logic_vector (9 downto 0) := "1111111111";	
	signal opaque : std_logic := '1';
	

--	type pixel_values is array (7 downto 0) of std_logic_vector (9 downto 0);
	type pixel_values is array (8 downto 0) of std_logic_vector (9 downto 0);
	signal rCurrentSprite : pixel_values := (others => "1111111111");
	signal gCurrentSprite : pixel_values := (others => "1111111111");
	signal bCurrentSprite : pixel_values := (others => "1111111111");	
	signal opaqueSprite : squaretype := (others => '1');

	signal rCurrentScore : pixel_values := (others => "1111111111");
	signal gCurrentScore : pixel_values := (others => "1111111111");
	signal bCurrentScore : pixel_values := (others => "1111111111");	
	signal opaqueScore : squaretype := (others => '1');
	
----	type fish_pixel_values is array (15 downto 0) of std_logic_vector (9 downto 0);
----	signal rFish : fish_pixel_values := (others => "1111111111");
----	signal gFish : fish_pixel_values := (others => "1111111111");
----	signal bFish : fish_pixel_values := (others => "1111111111");	
----	signal opaqueFish : squareFishtype := (others => '1');
	
-- background and sprite display
	signal sr_instructionAddress 		: unsigned(15 downto 0) := "0000000000000000";
	signal sr_spriteAddress0 : integer := 0;
	signal sr_spriteAddress1 : integer := 0;
	signal sr_spriteAddress2 : integer := 0;
	signal sr_spriteAddress3 : integer := 0;
	signal sr_spriteAddress4 : integer := 0;
	signal sr_spriteAddress5 : integer := 0;
	signal sr_spriteAddress6 : integer := 0;
	signal sr_spriteAddress7 : integer := 0;
	
	signal sr_addressNumber : integer := 0;
	
	signal wing_position :std_logic := '1';
	signal frame_count : integer := 0;
--	signal sr_spriteAddress : position := (others => 0);
	signal read_sr_red	: unsigned(15 downto 0) := "0000000000000000";
	signal currColor      : unsigned (3 downto 0) := "0000";
	signal numElements    : integer := 0;
	signal numInstructions : integer := 8;
	signal nextInstruction : unsigned(15 downto 0) := "0000000000000000";
	signal incrementedAddress : std_logic := '0';
	
	
	type sprite_array is array (1599 downto 0) of std_logic_vector (9 downto 0);
	signal index : position := (others => 0);
	signal row : position := (others => 0);
	signal col : position := (others => 0);
	
	signal indexScore : position := (others => 0);
	signal rowScore : position := (others => 0);
	signal colScore : position := (others => 0);

----	signal indexfish : fishposition := (others => 0);
----	signal rowFish : fishposition := (others => 0);
----	signal colFish : fishposition := (others => 0);
	
	type curr_color_array is array (7 downto 0) of unsigned (3 downto 0);
	type score_color_array is array (2 downto 0) of std_logic;
----	type fish_color_array is array (15 downto 0) of std_logic

--	type curr_color_array is array (8 downto 0) of unsigned (3 downto 0);
	signal read_currSpriteColor : curr_color_array := (others => "0000");
	signal read_currScoreColor : score_color_array := (others => '0');
----	signal read_currFishColor : fish_color_array := (others => '0');
	signal read_currFishColor : std_logic;
--	signal fishIndex : integer := 0;

	signal timer : integer := 0;
	signal shotsRemaining : integer := 0;



begin


spriteStore: entity work.sprite_rom port map (
	Clk => clk25,
	spriteAddress0 => sr_spriteAddress0,
	spriteAddress1 => sr_spriteAddress1,
	spriteAddress2 => sr_spriteAddress2,
	spriteAddress3 => sr_spriteAddress3,
	spriteAddress4 => sr_spriteAddress4,
	spriteAddress5 => sr_spriteAddress5,
	spriteAddress6 => sr_spriteAddress6,
	spriteAddress7 => sr_spriteAddress7,
	up_wing => wing_position,
	
	addressNumber0 => indexScore(0),
	addressNumber1 => indexScore(1),
	addressNumber2 => indexScore(2),

	whichNumber => score,

	duck0 => read_CurrSpriteColor(0),
	duck1 => read_CurrSpriteColor(1),
	duck2 => read_CurrSpriteColor(2),
	duck3 => read_CurrSpriteColor(3),
	duck4 => read_CurrSpriteColor(4),
	duck5 => read_CurrSpriteColor(5),
	duck6 => read_CurrSpriteColor(6),
	duck7 => read_CurrSpriteColor(7),
	
	number0Pixel => read_CurrScoreColor(0),
	number1Pixel => read_CurrScoreColor(1),
	number2Pixel => read_CurrScoreColor(2),
	
	fishPixel => fishOpaque,
--	fishPixel1 => read_CurrFishColor(1),
--	fishPixel2 => read_CurrFishColor(2),
--	fishPixel3 => read_CurrFishColor(3),
--	fishPixel4 => read_CurrFishColor(4),
--	fishPixel5 => read_CurrFishColor(5),
--	fishPixel6 => read_CurrFishColor(6),
--	fishPixel7 => read_CurrFishColor(7),
--	fishPixel8 => read_CurrFishColor(8),
--	fishPixel9 => read_CurrFishColor(9),
--	fishPixel10 => read_CurrFishColor(10),
--	fishPixel11 => read_CurrFishColor(11),
--	fishPixel12 => read_CurrFishColor(12),
--	fishPixel13 => read_CurrFishColor(13),
--	fishPixel14 => read_CurrFishColor(14),
--	fishPixel15 => read_CurrFishColor(15),
--	
--	
	fishAddress => fishAddress
--	fishAddress1 => Indexfish(1),
--	fishAddress2 => Indexfish(2),
--	fishAddress3 => Indexfish(3),
--	fishAddress4 => Indexfish(4),
--	fishAddress5 => Indexfish(5),
--	fishAddress6 => Indexfish(6),
--	fishAddress7 => Indexfish(7),
--	fishAddress8 => Indexfish(8),
--	fishAddress9 => Indexfish(9),
--	fishAddress10 => Indexfish(10),
--	fishAddress11 => Indexfish(11),
--	fishAddress12 => Indexfish(12),
--	fishAddress13 => Indexfish(13),
--	fishAddress14 => Indexfish(14),
--	fishAddress15 => Indexfish(15)
	);
	
backgroundStore: entity work.background_rom port map (
	Clk => clk25,
	instructionAddress =>nextInstruction,
	data => read_sr_red,	
	arrayLength => numElements
	);	
	
	
HalfClock : process (clk) begin
	if rising_edge(clk) then
		clk25 <= not clk25;
		end if;
	end process;

	
--When the processor writes to this peripheral
SoftwareInterface : process (clk) begin
	if rising_edge(clk) then
		if chipselect = '1' and write = '1' then
			if writedata(15 downto 13) = "000" then --Update horizont
				old_hcenter <= hcenter;
				hcenter(to_integer(writedata(12 downto 10))) <= to_integer(writedata(9 downto 0));
			elsif writedata(15 downto 13) = "001" then --Update verical
				vcenter(to_integer(writedata(12 downto 10))) <= to_integer(writedata(9 downto 0));
			elsif writedata(15 downto 13) = "010" then --update duck visibility
				duckVisible <= writedata(7 downto 0);
			elsif writedata(15 downto 13) = "011" then --reset level
				if writedata (3 downto 0) = "0101" then
					levelReset <= '1';
				else
					levelReset <= '0';
					end if;
			elsif writedata(15 downto 13) = "100" then -- update timer
				timer <= to_integer(writedata(9 downto 0));
			elsif writedata(15 downto 13) = "101" then -- update shot counter
				shotsRemaining <= to_integer(writedata(5 downto 0));
			elsif writedata(15 downto 13) = "110" then -- reset game
				if writedata(3 downto 0) = "0101" then
					gameReset <= '1';
				else
					gameReset <= '0';
					end if;
				end if;	
			end if;  
		end if;
	end process;          
	 
--set a short timer to blank the screen  
process (clk25) begin
	if rising_edge(clk25) then
	
		if triggerpull_vga='1' then
			blankOnNextScreen <= '1';
			waitForBlackScreen <= -1; --so it never hits the case where blackScreenTime =1 or =0 before we reach endOfField		
		elsif blankOnNextScreen = '1' and EndOfField = '1' then			--set up a mechanism so blacking the screen only begins at a new frame.
			blankOnNextScreen <= '0';
			blackScreen <= '1';
			waitForBlackScreen <= BLACKSCREENTIME;
		elsif waitForBlackScreen = 1 then     --flicker variable "startTargetFlash" to enter the target flashing process
			startTargetFlash <= '1';
			waitForBlackScreen <= 0;
		elsif waitForBlackScreen = 0 then    
			blackScreen <= '0';
			startTargetFlash <= '0';
		elsif waitForBlackScreen > 1 then
			waitForBlackScreen <= waitForBlackScreen-1;
			end if;
		end if;
	end process;

--next stage, set a timer to show targets on a black screen
process (clk25) begin
	if rising_edge(clk25) then
		if gameReset = '1' then
			score <= 0;
			duckHit <= "00000000";
		elsif levelReset ='1' then 
			duckHit <= "00000000";
			showTarget <= 0;
			anyHit <= '0';
			--score <= score + 100;
		--Reset everything upon start of target flash
		elsif startTargetFlash = '1' then 
			showTarget <= 4;
			waitForTarget <= SHOWTARGETTIME;
			anyHit <= '0';
			targetFours <= '0';
			targetTwos <= '0';
			targetOnes <= '0';
			targetHit <= 0;
			targetFindingComplete<='0';
		--If we have waited through the target, but this is not the last target
		--then decrement and reset the target time
		elsif waitForTarget = 0 and showTarget>0 then
			showTarget <= showTarget - 1;
			waitForTarget <= SHOWTARGETTIME;
		--If we have gone through all of the targets, and atleast one target has been hit, update which duck was hit
		elsif showTarget=0 and anyHit='1' and targetFindingComplete='0' then
			targetHit <= to_integer(targetFours&targetTwos&targetOnes);
			duckHit(to_integer(targetFours&targetTwos&targetOnes)) <= '1';
			score <= score + 1;
			targetFindingComplete<='1';
		--Otherwise, decrement the timer and check for target hits
		else
			waitForTarget <= waitForTarget-1;
			
			if flash_vga='1' then
				if showTarget=4 then
					anyHit <= '1';
				elsif showTarget=3 then
					targetFours <= '1';
				elsif showTarget=2 then
					targetTwos <= '1';
				elsif showTarget=1 then
					targetOnes <= '1';
					end if;
				end if;
			end if;
		end if;
	end process;
	

-- Horizontal and vertical counters
HCounter : process (clk25) begin
	if rising_edge(clk25) then     
		if reset = '1' then
			Hcount <= (others => '0');
		elsif EndOfLine = '1' then
			Hcount <= (others => '0');
		else
			Hcount <= Hcount + 1;
			end if;     
		end if;
	end process HCounter;

EndOfLine <= '1' when Hcount = HTOTAL - 1 else '0';
 
VCounter: process (clk25) begin
	if rising_edge(clk25) then     
		if reset = '1' then
			Vcount <= (others => '0');
		elsif EndOfLine = '1' then
			if EndOfField = '1' then
				Vcount <= (others => '0');
			else
				Vcount <= Vcount + 1;
				end if;
			end if;
		end if;
	end process VCounter;

EndOfField <= '1' when Vcount = VTOTAL - 1 else '0';



-- State machines to generate HSYNC, VSYNC, HBLANK, and VBLANK
HSyncGen : process (clk25) begin
	if rising_edge(clk25) then    
		if reset = '1' or EndOfLine = '1' then
			vga_hsync <= '1';
		elsif Hcount = HSYNC - 1 then
			vga_hsync <= '0';
			end if;
		end if;
	end process HSyncGen;
 
HBlankGen : process (clk25) begin
	if rising_edge(clk25) then
		if reset = '1' then
			vga_hblank <= '1';
		elsif Hcount = HSYNC + HBACK_PORCH then
			vga_hblank <= '0';
		elsif Hcount = HSYNC + HBACK_PORCH + HACTIVE then
			vga_hblank <= '1';
			end if;     
		end if;
	end process HBlankGen;

VSyncGen : process (clk25) begin
	if rising_edge(clk25) then
		if reset = '1' then
			vga_vsync <= '1';
		elsif EndOfLine ='1' then
			if EndOfField = '1' then
				vga_vsync <= '1';
			elsif Vcount = VSYNC - 1 then
				vga_vsync <= '0';
				end if;
			end if;     
		end if;
	end process VSyncGen;

VBlankGen : process (clk25) begin
	if rising_edge(clk25) then   
		if reset = '1' then
			vga_vblank <= '1';
		elsif EndOfLine = '1' then
			if Vcount = VSYNC + VBACK_PORCH - 1 then
				vga_vblank <= '0';
			elsif Vcount = VSYNC + VBACK_PORCH + VACTIVE - 1 then
				vga_vblank <= '1';
				end if;
			end if;
		end if;
	end process VBlankGen;

--For all of the ducks, are we on a visible or invisible pixel
SquareGen : process (clk) begin
	if rising_edge(clk) then
		for i in 7 downto 0 loop --squareOff(i) = '0' and 
			if (duckVisible(i)='1' and vcenter(i)<Vcount and Vcount<vcenter(i)+RADIUS and hcenter(i)<HCount and Hcount<hcenter(i)+RADIUS) then
				square(i) <= '1';
			else
				square(i) <= '0';
				end if;
			end loop;
		end if;
	end process squareGen;
	
--For all of the score squares, are we on a visible or invisible pixel
ScoreGen : process (clk) begin
	if rising_edge(clk) then
		for i in 3 downto 0 loop
			if (vScore(i)<Vcount and Vcount<vScore(i)+SCOREHEIGHT and hScore(i)<HCount and Hcount<hScore(i)+SCOREWIDTH) then
				squareScore(i) <= '1';
			else
				squareScore(i) <= '0';
				end if;
			end loop;
		end if;
	end process ScoreGen;

fishGen: process (clk) begin
	if rising_edge(clk) then
		if(420<vcount and vcount<440) then
			if (200<hcount and hcount<220 and shotsRemaining>0) then
				fishSquare <= '1';
				fishAddress <= (to_integer(vcount) - 420)*20+to_integer(hcount)-200;
			elsif (230<hcount and hcount<250 and shotsRemaining>1) then
				fishSquare <= '1';
				fishAddress <= (to_integer(vcount) - 420)*20+to_integer(hcount)-230;
			elsif (260<hcount and hcount<280 and shotsRemaining>2) then
				fishSquare <= '1';
				fishAddress <= (to_integer(vcount) - 420)*20+to_integer(hcount)-260;
			elsif (290<hcount and hcount<310 and shotsRemaining>3) then
				fishSquare <= '1';
				fishAddress <= (to_integer(vcount) - 420)*20+to_integer(hcount)-290;
			elsif (320<hcount and hcount<340 and shotsRemaining>4) then
				fishSquare <= '1';
				fishAddress <= (to_integer(vcount) - 420)*20+to_integer(hcount)-320;
			elsif (350<hcount and hcount<370 and shotsRemaining>5) then
				fishSquare <= '1';
				fishAddress <= (to_integer(vcount) - 420)*20+to_integer(hcount)-350;
			elsif (380<hcount and hcount<400 and shotsRemaining>6) then
				fishSquare <= '1';
				fishAddress <= (to_integer(vcount) - 420)*20+to_integer(hcount)-380;
			elsif (410<hcount and hcount<430 and shotsRemaining>7) then
				fishSquare <= '1';
				fishAddress <= (to_integer(vcount) - 420)*20+to_integer(hcount)-410;
			else
				fishSquare <= '0';
				fishAddress <= 0;
				end if;
		elsif (460<vcount and vcount<480) then
			if (200<hcount and hcount<220 and shotsRemaining>8) then
				fishSquare <= '1';
				fishAddress <= (to_integer(vcount) - 460)*20+to_integer(hcount)-200;
			elsif (230<hcount and hcount<250 and shotsRemaining>9) then
				fishSquare <= '1';
				fishAddress <= (to_integer(vcount) - 460)*20+to_integer(hcount)-230;
			elsif (260<hcount and hcount<280 and shotsRemaining>10) then
				fishSquare <= '1';
				fishAddress <= (to_integer(vcount) - 460)*20+to_integer(hcount)-260;
			elsif (290<hcount and hcount<310 and shotsRemaining>11) then
				fishSquare <= '1';
				fishAddress <= (to_integer(vcount) - 460)*20+to_integer(hcount)-290;
			elsif (320<hcount and hcount<340 and shotsRemaining>12) then
				fishSquare <= '1';
				fishAddress <= (to_integer(vcount) - 460)*20+to_integer(hcount)-320;
			elsif (350<hcount and hcount<370 and shotsRemaining>13) then
				fishSquare <= '1';
				fishAddress <= (to_integer(vcount) - 460)*20+to_integer(hcount)-350;
			elsif (380<hcount and hcount<400 and shotsRemaining>14) then
				fishSquare <= '1';
				fishAddress <= (to_integer(vcount) - 460)*20+to_integer(hcount)-380;
			elsif (410<hcount and hcount<430 and shotsRemaining>15) then
				fishSquare <= '1';
				fishAddress <= (to_integer(vcount) - 460)*20+to_integer(hcount)-410;
			else
				fishSquare <= '0';
				fishAddress <= 0;
				end if;	
		else
			fishSquare <= '0';
			fishAddress <= 0;
			end if;
		end if;
	end process fishGen;	

--
WingFlap: process (EndofField) begin
	if EndofField = '1' then
		if frame_count < 9 then
			frame_count <= frame_count + 1;
		else
			wing_position <= not wing_position;
			frame_count <= 0;
			end if;
		end if;
	--wing_position <= not wing_position when frame_count = 5 else wing_position;
	end process wingFlap;


-- These should be the correct values for colors 0 - 9
-- The colors 10 - 15 will be added when we figure out the colors
-- needed for the ducks.
colorList: process (currColor, clk) begin
	case currColor is		
		when "0000" => rCurrent <= x"41" & "11";
						gCurrent <= x"40" & "11";
						bCurrent <= x"2b" & "11";
						opaque <= '0';
		when "0001" => rCurrent <= x"30" & "11";
						gCurrent <= x"73" & "11";
						bCurrent <= x"11" & "11";
						opaque <= '1';
		when "0010" =>  rCurrent <= x"2b" & "11";
						gCurrent <= x"cb" & "11";
						bCurrent <= x"ff" & "11";
						opaque <= '1';						
		when "0011" =>  rCurrent <= x"8b" & "11";
						gCurrent <= x"d2" & "11";
						bCurrent <= x"65" & "11";
						opaque <= '1';   
		when "0100" =>  rCurrent <= x"51" & "11";
						gCurrent <= x"ba" & "11";
						bCurrent <= x"c6" & "11";
						opaque <= '1';   
		when "0101" =>  rCurrent <= x"84" & "11";
						gCurrent <= x"1a" & "11";
						bCurrent <= x"05" & "11";
						opaque <= '1';   			
		when "0110" =>  rCurrent <= x"b4" & "11";
						gCurrent <= x"f7" & "11";
						bCurrent <= x"15" & "11";
						opaque <= '1';   
		when "0111" =>  rCurrent <= x"5b" & "11";
						gCurrent <= x"c0" & "11";
						bCurrent <= x"9f" & "11";
						opaque <= '1';   
		when "1000" =>  rCurrent <= x"ad" & "11";
						gCurrent <= x"7d" & "11";
						bCurrent <= x"00" & "11";
						opaque <= '1';   
		when "1001" =>  rCurrent <= x"8c" & "11";
						gCurrent <= x"d4" & "11";
						bCurrent <= x"2d" & "11";
						opaque <= '1';   
		when "1010" =>  rCurrent <= "0010111100";
						gCurrent <= "0011010100";
						bCurrent <= "0000101101";
						opaque <= '1';   
		when "1011" =>  rCurrent <= "0011111111";
						gCurrent <= "0010001111";
						bCurrent <= "0000011111";
						opaque <= '1';   
		when "1100" =>  rCurrent <= "0010001111";
						gCurrent <= "0011111000";
						bCurrent <= "0011111111";
						opaque <= '1';   
		when "1101" =>  rCurrent <= "0011111111";
						gCurrent <= "0000000111";
						bCurrent <= "0011111000";
						opaque <= '1';   				
		when "1110" =>  rCurrent <= "0010011101";
						gCurrent <= "0011100001";
						bCurrent <= "0011111111";
						opaque <= '1';   				
		when "1111" =>  rCurrent <= "0011111111";
						gCurrent <= "0011111100";
						bCurrent <= "0011100011";
						opaque <= '1';   				 		
						
		when others =>	rCurrent <= "1111111111";
						gCurrent <= "1111111111";
						bCurrent <= "1111111111";
						opaque <= '1';   		
		end case;
	end process colorList;
	
	
	spriteColorList: process (read_currSpriteColor, clk) begin
		for i in 7 downto 0 loop
--		for i in 8 downto 0 loop
			case read_currSpriteColor(i) is		
				when "0000" => rCurrentSprite(i) <= x"00" & "11"; --1111   
								gCurrentSprite(i) <= x"09" & "11";					--black <-- orange
								bCurrentSprite(i) <= x"02" & "11";
								opaqueSprite(i) <= '1';
				when "0001" => rCurrentSprite(i) <= x"ff" & "11";  --1110   
								gCurrentSprite(i) <= x"b4" & "11";					--orange <-- black
								bCurrentSprite(i) <= x"60" & "11";
								opaqueSprite(i) <= '1';
				when "0010" =>  rCurrentSprite(i) <= x"fa" & "11"; --1101   
								gCurrentSprite(i) <= x"fe" & "11";					--white <-- green
								bCurrentSprite(i) <= x"ec" & "11";
								opaqueSprite(i) <= '1';						
				when "0011" =>  rCurrentSprite(i) <= x"5c" & "11";  --1101    --blue
								gCurrentSprite(i) <= x"a4" & "11";
								bCurrentSprite(i) <= x"f4" & "11";
								opaqueSprite(i) <= '0';   --Blue represents background.
				when "0100" =>  rCurrentSprite(i) <= x"11" & "11"; --1011 
								gCurrentSprite(i) <= x"56" & "11";					--green <-- white
								bCurrentSprite(i) <= x"04" & "11";
								opaqueSprite(i) <= '1'; 
				when others =>	rCurrentSprite(i) <= "1111111111";
								gCurrentSprite(i) <= "1111111111";
								bCurrentSprite(i) <= "1111111111";
								opaqueSprite(i) <= '1';
			end case;
		end loop;	
	end process spriteColorList;
	
	
scoreColorList: process (read_currSpriteColor, clk) begin
	for i in 2 downto 0 loop
		case read_currScoreColor(i) is		
			when '0' =>	
							opaqueScore(i) <= '0';
			when others =>	
							rCurrentScore(i) <= "1111111111";
							gCurrentScore(i) <= "1111111111";
							bCurrentScore(i) <= "1111111111";
							opaqueScore(i) <= '1';
			end case;
		end loop;	
	end process scoreColorList;
	
----		fishColorList: process (read_currFishColor, clk) begin
----		for i in 15 downto 0 loop
------		for i in 8 downto 0 loop
----			case read_currFishColor(i) is		
----				when '0' => rFish(i) <= x"00" & "11"; --1111   
----								gFish(i) <= x"09" & "11";					--black <-- orange
----								bFish(i) <= x"02" & "11";
----								opaqueFish(i) <= '0';
----				when others =>	rFish(i) <= "1111111111";
----								gFish(i) <= "1111111111";
----								bFish(i) <= "1111111111";
----								opaqueFish(i) <= '1';
----			end case;
----		end loop;	
----	end process fishColorList;


nextInstruction <= sr_instructionAddress+1;


CounterThing : process(clk25, hcount, vcount)
begin
if rising_edge(clk25) then

		for i in 3 downto 0 loop 
		 col(i) <= to_integer(Vcount) - vcenter(i) ;
			if wing_position ='1' then
				row(i) <= to_integer(Hcount) - hcenter(i) + 3;
				index(i) <=  col(i)*40 + row(i); --facing right
			else
				row(i) <= to_integer(Hcount) - hcenter(i) + 4;
				index(i) <=  col(i)*40 - row(i); --facing left
			end if;
		end loop;
		for i in 7 downto 4 loop 
				 col(i) <= to_integer(Vcount) - vcenter(i) ;
			if wing_position ='0' then
				row(i) <= to_integer(Hcount) - hcenter(i) + 3;
				index(i) <=  col(i)*40 + row(i); --facing right
			else
				row(i) <= to_integer(Hcount) - hcenter(i) + 4;
				index(i) <=  col(i)*40 - row(i); --facing left
			end if;
		end loop;
		
		 sr_spriteAddress0 <= index(0);
		 sr_spriteAddress1 <= index(1);
		 sr_spriteAddress2 <= index(2);
		 sr_spriteAddress3 <= index(3);
		 sr_spriteAddress4 <= index(4);
		 sr_spriteAddress5 <= index(5);
		 sr_spriteAddress6 <= index(6);
		 sr_spriteAddress7 <= index(7);
		
		
		for i in 7 downto 0 loop
			colScore(i) <= to_integer(Vcount) - vScore(i) ;
			rowScore(i) <= to_integer(Hcount) - hScore(i) + 3;
			indexscore(i) <=  colScore(i)*40 + rowScore(i); 	 
		end loop;
--		for i in 15 downto 0 loop
--			colFish(i) <= to_integer(Vcount) - vFish(i) ;
--			rowFish(i) <= to_integer(Hcount) - hFish(i) + 3;
--			indexFish(i) <=  colFish(i)*20 + rowFish(i); 	 
--		end loop;

end if;
end process;


-- Registered video signals going to the video DAC
VideoOut: process (clk25, reset) begin
	--Asyncronous reset, pixels are black and all instructions are zero
	if reset = '1' then
		VGA_R <= "0000000000";
		VGA_G <= "0000000000";
		VGA_B <= "0000000000";
		backgroundPixelsRemaining <= 0;
		sr_instructionAddress <= "0000000000000000";
		numInstructions <= numElements;
		
	elsif rising_edge(clk25) then
		--If there are no instructions left, reset the instruction counter
		if numInstructions = 0 then
			sr_instructionAddress <= "0000000000000000";
			end if;
		
		--If we are in the actual pixel part of the screen
		if vga_hblank = '0' and vga_vblank = '0' then					
			--Handle the background run length encoding stuff.
			--Backgroundpixelsremaining refers to remaining pixels in the current instruction
			--numInstructios refers to the number of instructions left
			--read_sr_red is the actual color coming out of the ROM
			if backgroundPixelsRemaining = 2 then
				sr_instructionAddress <= sr_instructionAddress + 1;
				backgroundPixelsRemaining <= 1;
				incrementedAddress <= '1';
			elsif backgroundPixelsRemaining = 1 then
					if incrementedAddress = '0' then
						sr_instructionAddress <= sr_instructionAddress + 1;
						end if;
					incrementedAddress <= '0';
					backgroundPixelsRemaining <= 0;
			elsif backgroundPixelsRemaining = 0 then
				currColor <= read_sr_red(15 downto 12);
				backgroundPixelsRemaining <= to_integer(read_sr_red(11 downto 0))-1;
				numInstructions <= numInstructions -1;
			else
				backgroundPixelsRemaining <= backgroundPixelsRemaining-1;
				end if;
			
			if blackScreen = '1' then
				VGA_R <= "0000000000";
				VGA_G <= "0000000000";
				VGA_B <= "0000000000";
			elsif showTarget=4 then
				if	(square(0)='1' and opaqueSprite(0)='1')
						or (square(1)='1' and opaqueSprite(1)='1')
						or (square(2)='1' and opaqueSprite(2)='1')
						or (square(3)='1' and opaqueSprite(3)='1')
						or (square(4)='1' and opaqueSprite(4)='1')
						or (square(5)='1' and opaqueSprite(5)='1')
						or (square(6)='1' and opaqueSprite(6)='1')
						or (square(7)='1' and opaqueSprite(7)='1') 
						then
					VGA_R <= "1111111111";
					VGA_G <= "1111111111";
					VGA_B <= "1111111111";
				else 
					VGA_R <= "0000000000";
					VGA_G <= "0000000000";
					VGA_B <= "0000000000";
					end if;
			elsif showTarget=3 then
				if (square(4)='1' and opaqueSprite(4)='1')
						or (square(5)='1' and opaqueSprite(5)='1')
						or (square(6)='1' and opaqueSprite(6)='1')
						or (square(7)='1' and opaqueSprite(7)='1') 
						then
					VGA_R <= "1111111111";
					VGA_G <= "1111111111";
					VGA_B <= "1111111111";
				else
					VGA_R <= "0000000000";
					VGA_G <= "0000000000";
					VGA_B <= "0000000000";
					end if;
			elsif showTarget=2 then
				if	(square(2)='1' and opaqueSprite(2)='1')
						or (square(3)='1' and opaqueSprite(3)='1')
						or (square(6)='1' and opaqueSprite(6)='1')
						or (square(7)='1' and opaqueSprite(7)='1')  
						then
					VGA_R <= "1111111111";
					VGA_G <= "1111111111";
					VGA_B <= "1111111111";
				else 
					VGA_R <= "0000000000";
					VGA_G <= "0000000000";
					VGA_B <= "0000000000";
					end if;
			elsif showTarget=1 then
				if	(square(1)='1' and opaqueSprite(1)='1')
						or (square(3)='1' and opaqueSprite(3)='1')
						or (square(5)='1' and opaqueSprite(5)='1')
						or (square(7)='1' and opaqueSprite(7)='1')  
						then
					VGA_R <= "1111111111";
					VGA_G <= "1111111111";
					VGA_B <= "1111111111";
				else 
					VGA_R <= "0000000000";
					VGA_G <= "0000000000";
					VGA_B <= "0000000000";
					end if;
			elsif showTarget = 0 then
					if square(0)= '1' and opaqueSprite(0) ='1' then
						VGA_R <= rCurrentSprite(0);
						VGA_G <= gCurrentSprite(0);
						VGA_B <= bCurrentSprite(0);
					elsif square(1)='1' and opaqueSprite(1) ='1' then
						VGA_R <= rCurrentSprite(1);
						VGA_G <= gCurrentSprite(1);
						VGA_B <= bCurrentSprite(1);		
					elsif square(2)='1' and opaqueSprite(2) ='1' then
						VGA_R <= rCurrentSprite(2);
						VGA_G <= gCurrentSprite(2);
						VGA_B <= bCurrentSprite(2);
					elsif square(3)='1' and opaqueSprite(3) ='1' then
						VGA_R <= rCurrentSprite(3);
						VGA_G <= gCurrentSprite(3);
						VGA_B <= bCurrentSprite(3);
					elsif square(4)='1' and opaqueSprite(4) ='1' then
						VGA_R <= rCurrentSprite(4);
						VGA_G <= gCurrentSprite(4);
						VGA_B <= bCurrentSprite(4);
					elsif square(5)='1' and opaqueSprite(5) ='1' then
						VGA_R <= rCurrentSprite(5);
						VGA_G <= gCurrentSprite(5);
						VGA_B <= bCurrentSprite(5);
					elsif square(6)='1'  and opaqueSprite(6) ='1' then
						VGA_R <= rCurrentSprite(6);
						VGA_G <= gCurrentSprite(6);
						VGA_B <= bCurrentSprite(6);
					elsif square(7)='1' and opaqueSprite(7) ='1' then
						VGA_R <= rCurrentSprite(7);
						VGA_G <= gCurrentSprite(7);
						VGA_B <= bCurrentSprite(7);
					elsif squareScore(0) = '1' and opaqueScore(0) = '1' then
						VGA_R <= rCurrentScore(0);
						VGA_G <= gCurrentScore(0);
						VGA_B <= bCurrentScore(0);
					elsif squareScore(1) = '1' and opaqueScore(1) = '1' then
						VGA_R <= rCurrentScore(1);
						VGA_G <= gCurrentScore(1);
						VGA_B <= bCurrentScore(1);
					elsif squareScore(2) = '1' and opaqueScore(2) = '1' then
						VGA_R <= rCurrentScore(2);
						VGA_G <= gCurrentScore(2);
						VGA_B <= bCurrentScore(2);
					elsif fishOpaque='1' and fishSquare = '1' then
						VGA_R <= "0000000000";
						VGA_G <= "0000000000";
						VGA_B <= "1111111111";
						
--					elsif squareFish(0)='1' and opaqueFish(0) = '1'then
--						VGA_R <= rFish(0);
--						VGA_G <= gFish(0);
--						VGA_B <= bfish(0);
--					elsif squareFish(1)='1' and opaqueFish(1) = '1'then
--						VGA_R <= rFish(1);
--						VGA_G <= gFish(1);
--						VGA_B <= bfish(1);					
--					elsif squareFish(2)='1'and opaqueFish(2) = '1' then
--						VGA_R <= rFish(2);
--						VGA_G <= gFish(2);
--						VGA_B <= bfish(2);					
--					elsif squareFish(3)='1' and opaqueFish(3) = '1'then
--						VGA_R <= rFish(3);
--						VGA_G <= gFish(3);
--						VGA_B <= bfish(3);					
--					elsif squareFish(4)='1' and opaqueFish(4) = '1'then
--						VGA_R <= rFish(4);
--						VGA_G <= gFish(4);
--						VGA_B <= bfish(4);					
--					elsif squareFish(5)='1' and opaqueFish(5) = '1'then
--						VGA_R <= rFish(5);
--						VGA_G <= gFish(5);
--						VGA_B <= bfish(5);
--					elsif squareFish(6)='1' and opaqueFish(6) = '1'then
--						VGA_R <= rFish(6);
--						VGA_G <= gFish(6);
--						VGA_B <= bfish(6);
--					elsif squareFish(7)='1' and opaqueFish(7) = '1'then
--						VGA_R <= rFish(7);
--						VGA_G <= gFish(7);
--						VGA_B <= bfish(7);
--					elsif squareFish(8)='1' and opaqueFish(8) = '1'then
--						VGA_R <= rFish(8);
--						VGA_G <= gFish(8);
--						VGA_B <= bfish(8);					
--					elsif squareFish(9)='1' and opaqueFish(9) = '1'then
--						VGA_R <= rFish(9);
--						VGA_G <= gFish(9);
--						VGA_B <= bfish(9);					
--					elsif squareFish(10)='1' and opaqueFish(10) = '1'then
--						VGA_R <= rFish(10);
--						VGA_G <= gFish(10);
--						VGA_B <= bfish(10);					
--					elsif squareFish(11)='1' and opaqueFish(11) = '1'then
--						VGA_R <= rFish(11);
--						VGA_G <= gFish(11);
--						VGA_B <= bfish(11);		
--					elsif squareFish(12)='1' and opaqueFish(12) = '1'then
--						VGA_R <= rFish(12);
--						VGA_G <= gFish(12);
--						VGA_B <= bfish(12);				
--					elsif squareFish(13)='1' and opaqueFish(13) = '1'then
--						VGA_R <= rFish(13);
--						VGA_G <= gFish(13);
--						VGA_B <= bfish(13);					
--					elsif squareFish(14)='1' and opaqueFish(14) = '1'then
--						VGA_R <= rFish(14);
--						VGA_G <= gFish(14);
--						VGA_B <= bfish(14);
--					elsif squareFish(15)='1' and opaqueFish(15) = '1'then
--						VGA_R <= rFish(15);
--						VGA_G <= gFish(15);
--						VGA_B <= bfish(15);						
																		
					elsif vCount > VSYNC + VBACK_PORCH + VACTIVE - 12 and hCount < HSYNC + HBACK_PORCH + timer then
						VGA_R <= "1111111111";
						VGA_G <= "1111111111";
						VGA_B <= "1111111111";
					else
						--otherwise, just print the background
						VGA_R <= rCurrent;
						VGA_G <= gCurrent;
						VGA_B <= bCurrent;
						end if; 
			end if; --blackscreen
		elsif EndOfField = '1' then
			--This completely resets upon a new screen
			backgroundPixelsRemaining <= 0;    
			sr_instructionAddress <= "0000000000000000";
			VGA_R <= "0000000000";
			VGA_G <= "0000000000";
			VGA_B <= "0000000000";
		else
			VGA_R <= "0000000000";
			VGA_G <= "0000000000";
			VGA_B <= "0000000000";
		end if; -- vgablankl end if
	end if;	-- reset and clock end if
	end process VideoOut;

	
VGA_CLK <= clk25;
VGA_HS <= not vga_hsync;
VGA_VS <= not vga_vsync;
VGA_SYNC <= '0';
VGA_BLANK <= not (vga_hsync or vga_vsync);

readData <= "00000000" & duckHit;

end rtl;
		