--
-- DE2 top-level module that includes the simple VGA raster generator
--
-- Stephen A. Edwards, Columbia University, sedwards@cs.columbia.edu
--
-- From an original by Terasic Technology, Inc.
-- (DE2_TOP.v, part of the DE2 system board CD supplied by Altera)
--

library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;

entity top is

  port (
    -- Clocks
    
    CLOCK_27,                                      -- 27 MHz
    CLOCK_50,                                      -- 50 MHz
    EXT_CLOCK : in std_logic;                      -- External Clock

    -- Buttons and switches
    
    KEY : in std_logic_vector(3 downto 0);         -- Push buttons
    SW : in std_logic_vector(17 downto 0);         -- DPDT switches

    -- LED displays

    HEX0, HEX1, HEX2, HEX3, HEX4, HEX5, HEX6, HEX7 -- 7-segment displays
       : out std_logic_vector(6 downto 0);
    LEDG : out std_logic_vector(8 downto 0);       -- Green LEDs
    LEDR : out std_logic_vector(17 downto 0);      -- Red LEDs

    -- RS-232 interface

    UART_TXD : out std_logic;                      -- UART transmitter   
    UART_RXD : in std_logic;                       -- UART receiver

    -- IRDA interface

--    IRDA_TXD : out std_logic;                      -- IRDA Transmitter
    IRDA_RXD : in std_logic;                       -- IRDA Receiver

    -- SDRAM
   
    DRAM_DQ : inout std_logic_vector(15 downto 0); -- Data Bus
    DRAM_ADDR : out std_logic_vector(11 downto 0); -- Address Bus    
    DRAM_WE_N,                                     -- Write Enable
    DRAM_CAS_N,                                    -- Column Address Strobe
    DRAM_RAS_N,                                    -- Row Address Strobe
    DRAM_CS_N,                                     -- Chip Select
    DRAM_CLK,                                      -- Clock
    DRAM_CKE : out std_logic;                      -- Clock Enable
    DRAM_LDQM,                                     -- Low-byte Data Mask 
    DRAM_UDQM,                                     -- High-byte Data Mask
    DRAM_BA_0,                                     -- Bank Address 0
    DRAM_BA_1: out std_logic;                   -- Bank Address 0

    -- FLASH
    
    FL_DQ : inout std_logic_vector(7 downto 0);      -- Data bus
    FL_ADDR : out std_logic_vector(21 downto 0);  -- Address bus
    FL_WE_N,                                         -- Write Enable
    FL_RST_N,                                        -- Reset
    FL_OE_N,                                         -- Output Enable
    FL_CE_N : out std_logic;                         -- Chip Enable

    -- SRAM
    
    SRAM_DQ : inout std_logic_vector(15 downto 0); -- Data bus 16 Bits
    SRAM_ADDR : out std_logic_vector(17 downto 0); -- Address bus 18 Bits
    SRAM_UB_N,                                     -- High-byte Data Mask 
    SRAM_LB_N,                                     -- Low-byte Data Mask 
    SRAM_WE_N,                                     -- Write Enable
    SRAM_CE_N,                                     -- Chip Enable
    SRAM_OE_N : out std_logic;                     -- Output Enable

    -- USB controller
    
    OTG_DATA : inout std_logic_vector(15 downto 0); -- Data bus
    OTG_ADDR : out std_logic_vector(1 downto 0);    -- Address
    OTG_CS_N,                                       -- Chip Select
    OTG_RD_N,                                       -- Write
    OTG_WR_N,                                       -- Read
    OTG_RST_N,                                      -- Reset
    OTG_FSPEED,                     -- USB Full Speed, 0 = Enable, Z = Disable
    OTG_LSPEED : out std_logic;     -- USB Low Speed, 0 = Enable, Z = Disable
    OTG_INT0,                                       -- Interrupt 0
    OTG_INT1,                                       -- Interrupt 1
    OTG_DREQ0,                                      -- DMA Request 0
    OTG_DREQ1 : in std_logic;                       -- DMA Request 1   
    OTG_DACK0_N,                                    -- DMA Acknowledge 0
    OTG_DACK1_N : out std_logic;                    -- DMA Acknowledge 1

    -- 16 X 2 LCD Module
    
    LCD_ON,                     -- Power ON/OFF
    LCD_BLON,                   -- Back Light ON/OFF
    LCD_RW,                     -- Read/Write Select, 0 = Write, 1 = Read
    LCD_EN,                     -- Enable
    LCD_RS : out std_logic;     -- Command/Data Select, 0 = Command, 1 = Data
    LCD_DATA : inout std_logic_vector(7 downto 0); -- Data bus 8 bits

    -- SD card interface
    
    SD_DAT,                     -- SD Card Data
    SD_DAT3,                    -- SD Card Data 3
    SD_CMD : inout std_logic;   -- SD Card Command Signal
    SD_CLK : out std_logic;     -- SD Card Clock

    -- USB JTAG link
    
    TDI,                        -- CPLD -> FPGA (data in)
    TCK,                        -- CPLD -> FPGA (clk)
    TCS : in std_logic;         -- CPLD -> FPGA (CS)
    TDO : out std_logic;        -- FPGA -> CPLD (data out)

    -- I2C bus
    
    I2C_SDAT : inout std_logic; -- I2C Data
    I2C_SCLK : out std_logic;   -- I2C Clock

    -- PS/2 port

    PS2_DAT,                    -- Data
    PS2_CLK : in std_logic;     -- Clock

    -- VGA output
    
    VGA_CLK,                                            -- Clock
    VGA_HS,                                             -- H_SYNC
    VGA_VS,                                             -- V_SYNC
    VGA_BLANK,                                          -- BLANK
    VGA_SYNC : out std_logic;                           -- SYNC
    VGA_R,                                              -- Red[9:0]
    VGA_G,                                              -- Green[9:0]
    VGA_B : out unsigned(9 downto 0);                   -- Blue[9:0]

    --  Ethernet Interface
    
    ENET_DATA : inout std_logic_vector(15 downto 0);    -- DATA bus 16Bits
    ENET_CMD,           -- Command/Data Select, 0 = Command, 1 = Data
    ENET_CS_N,                                          -- Chip Select
    ENET_WR_N,                                          -- Write
    ENET_RD_N,                                          -- Read
    ENET_RST_N,                                         -- Reset
    ENET_CLK : out std_logic;                           -- Clock 25 MHz
    ENET_INT : in std_logic;                            -- Interrupt
    
    -- Audio CODEC
    
    AUD_ADCLRCK : inout std_logic;                      -- ADC LR Clock
    AUD_ADCDAT : in std_logic;                          -- ADC Data
    AUD_DACLRCK : inout std_logic;                      -- DAC LR Clock
    AUD_DACDAT : out std_logic;                         -- DAC Data
    AUD_BCLK : inout std_logic;                         -- Bit-Stream Clock
    AUD_XCK : out std_logic;                            -- Chip Clock
    
    -- Video Decoder
    
    TD_DATA : in std_logic_vector(7 downto 0);  -- Data bus 8 bits
    TD_HS,                                      -- H_SYNC
    TD_VS : in std_logic;                       -- V_SYNC
    TD_RESET : out std_logic;                   -- Reset
    
    -- General-purpose I/O
    
    GPIO_0,                                      -- GPIO Connection 0
    GPIO_1 : inout std_logic_vector(35 downto 0) -- GPIO Connection 1   
    );
  
end top;


	
architecture datapath of top is

	COMPONENT sdram_pll
		PORT ( inclk0 : IN STD_LOGIC;
				c0 : OUT STD_LOGIC;
				c1: OUT STD_LOGIC );
	END COMPONENT;
	
	component I2C_AV_Config 
		port ( iCLK : in std_logic;
			   iRST_N : in std_logic;
			   I2C_SCLK : out std_logic;
			   I2C_SDAT: inout std_logic
			   );
	end component;
	
	component TD_Detect 
		port ( oTD_Stable: out std_logic;
			   iTD_VS,iTD_HS,iRST_N : in std_logic	
			  );
	end component;
	
	component DIV
		port ( aclr,clock : in std_logic;
				denom: in std_logic_vector(3 downto 0);
				numer: in std_logic_vector (9 downto 0);
				quotient: out std_logic_vector (9 downto 0);
				remain: out std_logic_vector (3 downto 0)
				);
	end component;
	
	component Reset_Delay
		port ( iCLK, 
			   iRST: in std_logic;
			   oRST_0, 
			   oRST_1, 
			   oRST_2: out std_logic
			 );
	end component;
	
	component ITU_656_Decoder
		port ( iTD_DATA: in std_logic_vector (7 downto 0);
			   oTV_X,opixelno,oTV_Y: out std_logic_vector (9 downto 0);
			   oYCbCr: out std_logic_vector (3 downto 0);
			   oDVAL: out std_logic;
			   oField: out std_logic;
			   iSwap_CbCr,iSkip,iRST_N,iCLK_27: in std_logic
			  );
	end component; 
			
	signal clk25 : std_logic := '0';
	
	--TD_DETECT OUTPUT SIGNALS
	signal myTD_Stable: std_logic;
	
	-- RESET DELAY OUTPUT SIGNALS
	signal myDLY0,myDLY1,myDLY2: std_logic;
	
	-- DIVIDER OUTPUT SIGNALS
	signal myQuotient:std_logic_vector(9 downto 0);
	signal myRemain: std_logic_vector(3 downto 0);
	
	-- ITU DECODER OUTPUT SIGNALS
	signal myTV_X: std_logic_vector (9 downto 0);
	signal mypixelno: std_logic_vector (9 downto 0);
	signal myTV_Y: std_logic_vector (9 downto 0);
	signal myYCbCr: std_logic_vector (3 downto 0);
	signal myDVAL: std_logic;
	signal myField: std_logic;
	
	-- SDRAM INTERMEDIARY SIGNALS
	signal BA : STD_LOGIC_VECTOR(1 DOWNTO 0);
	signal DQM : STD_LOGIC_VECTOR(1 DOWNTO 0); 
	
	-- FOR ITU
	signal mySkip: std_logic;
	
	-- ARBITRATOR OUTPUT / VGA INPUT
	signal myRed, myGreen, myBlue: std_logic_vector (9 downto 0);
	signal myPixelCLK: std_logic;
	
	-- FROM VGA Controller
	signal myHCount, myVcount: unsigned (9 downto 0);
	
	signal enetClk: std_logic;
	
	-- DMA Controller STUFF
	signal busy_to_dma_from_arbitrator: std_logic;
	signal data_to_dma_from_linebuffer: std_logic_vector(15 downto 0);
	signal linebufferfull_to_dma_from_linebuffer: std_logic;
	signal address_from_dma_to_arbitrator: std_logic_vector(17 downto 0);
	signal address_from_dma_to_linebuffer: std_logic_vector(6 downto 0); 
    signal data_from_dma_to_arbitrator: std_logic_vector(15 downto 0);
    signal data_ready_from_dma_to_arbitrator: std_logic;
	signal field_from_linebuffer_to_dma: std_logic;
	signal linenumber_from_linebuffer_to_dma: std_logic_vector (9 downto 0);
	
	signal clk100: std_logic;
	
--	signal myDVAL,myRESET,myCLK: std_logic;
--	signal myY: std_logic_vector (7 downto 0);
--	
--	signal oYCbCr: std_logic_vector (3 downto 0);
--
--	signal oTV_Cont: std_logic_vector (31 downto 0);
--	signal oDVAL: std_logic;
--	

--	
--	signal myRed, myGreen, myBlue: std_logic_vector (9 downto 0);
--	signal myOVAL: std_logic;
	
	-- 
--	signal myTD_Stable,DLY0,DLY1,DLY2: std_logic;

	
begin
	
	DRAM_BA_1 <= BA(1);
	DRAM_BA_0 <= BA(0);
	
	DRAM_UDQM <= DQM(1);
	DRAM_LDQM <= DQM(0);

  process (CLOCK_50)
  begin
    if rising_edge(CLOCK_50) then
      clk25 <= not clk25;
    end if;
  end process;
  
ENET_CLK <= clk25;


u1: I2C_AV_Config port map (iCLK=> CLOCK_50, 
						    iRST_N => KEY(0),
							I2C_SCLK => I2C_SCLK, 
							I2C_SDAT => I2C_SDAT);

u2: TD_Detect port map(myTD_Stable, 
					   TD_VS, 
					   TD_HS,
					   KEY(0));

u3: Reset_Delay port map (CLOCK_50, 
						  myTD_Stable, 
						  myDLY0, 
						  myDLY1, 
						  myDLY2);

u4: DIV port map (not myDLY0, 
				  CLOCK_27,
				  "1001",
				  myTV_X,
				  myQuotient,
				  myRemain);

u5: ITU_656_Decoder port map (TD_DATA, 
							  myTV_X, 
							  mypixelno,
							  myTV_Y, 
							  myYCbCr, 
							  myDVAL,
							  myField,
							  myQuotient(0),
							  mySkip,
							  myDLY1,
							  CLOCK_27);

neg_3ns: sdram_pll PORT MAP (CLOCK_50, clk100, DRAM_CLK);
LEDG (7 downto 0) <= TD_DATA;
mySkip <= '1' when myRemain = "0000" else '0';
--
vga: work.de2_vga_raster PORT MAP (reset => '0', clk => CLOCK_50, pixel_clock => myPixelCLK, RED => unsigned(myRed), GREEN => unsigned(myGreen), BLUE => unsigned(myBlue), 
							  VGA_CLK => VGA_CLK, VGA_HS => VGA_HS,VGA_VS => VGA_VS,
							  VGA_BLANK => VGA_BLANK,VGA_SYNC => VGA_SYNC,VGA_R => VGA_R,
							  VGA_G => VGA_G,VGA_B => VGA_B, HCOUNT_OUT => myHCount, VCOUNT_OUT => myVCount);
							
dmactrl: work.DMACONTROLLER PORT MAP (clk_50 => CLOCK_50, linebufferfull_from_lb => linebufferfull_to_dma_from_linebuffer,
									  data_from_lb => data_to_dma_from_linebuffer, 
									  busy_from_arb =>  busy_to_dma_from_arbitrator,
									  address_to_lb => address_from_dma_to_linebuffer,
									  address_to_arb => address_from_dma_to_arbitrator,
									  data_to_arb => data_from_dma_to_arbitrator,
									  data_ready_to_arb => data_ready_from_dma_to_arbitrator,
									  field_from_lb => field_from_linebuffer_to_dma,
									  lineNumber_from_lb => linenumber_from_linebuffer_to_dma);

  a1 : entity work.nios_0
    port map(
      ENET_CMD_from_the_DM9000A => ENET_CMD,
      ENET_CS_N_from_the_DM9000A => ENET_CS_N,
      ENET_DATA_to_and_from_the_DM9000A => ENET_DATA,
      ENET_RD_N_from_the_DM9000A => ENET_RD_N,
      ENET_RST_N_from_the_DM9000A => ENET_RST_N,
      ENET_WR_N_from_the_DM9000A => ENET_WR_N,
	  ENET_INT_to_the_DM9000A => ENET_INT,
	  --iRST_N_to_the_DM9000A => '1',
      BLUE_from_the_arbitrator_inst => myBlue,
      GREEN_from_the_arbitrator_inst => myGreen,
      RED_from_the_arbitrator_inst => myRed,
      SRAM_ADDR_from_the_arbitrator_inst => SRAM_ADDR,
      SRAM_CE_N_from_the_arbitrator_inst => SRAM_CE_N,
      SRAM_DQ_to_and_from_the_arbitrator_inst => SRAM_DQ,
      SRAM_LB_N_from_the_arbitrator_inst => SRAM_LB_N,
      SRAM_OE_N_from_the_arbitrator_inst => SRAM_OE_N,
      SRAM_UB_N_from_the_arbitrator_inst => SRAM_UB_N,
      SRAM_WE_N_from_the_arbitrator_inst => SRAM_WE_N,
	  pixel_clock_from_the_arbitrator_inst => myPixelCLK,
      zs_addr_from_the_sdram => DRAM_ADDR,
      zs_ba_from_the_sdram => BA,
      zs_cas_n_from_the_sdram => DRAM_CAS_N,
      zs_cke_from_the_sdram => DRAM_CKE,
      zs_cs_n_from_the_sdram => DRAM_CS_N,
      zs_dq_to_and_from_the_sdram => DRAM_DQ,
      zs_dqm_from_the_sdram => DQM,
      zs_ras_n_from_the_sdram => DRAM_RAS_N,
      zs_we_n_from_the_sdram => DRAM_WE_N,
	  Field_to_the_linebuffer_inst => myField,
      HCOUNT_to_the_arbitrator_inst => std_logic_vector(myHCount),
      VCOUNT_to_the_arbitrator_inst => std_logic_vector(myVCount),
      clk => CLOCK_50,
      clk_27_to_the_linebuffer_inst => CLOCK_27,
      datavalid_to_the_linebuffer_inst => myDVAL,
      reset_n => '1',
      xpos_to_the_linebuffer_inst => mypixelno,
      y_to_the_linebuffer_inst => myYCbCr,
      ypos_to_the_linebuffer_inst => myTV_Y,
clk_1 => clk100,
clk_100_to_the_linebuffer_inst => clk100,
      busy_to_dma_from_the_arbitrator_inst => busy_to_dma_from_arbitrator,
      data_to_dma_from_the_linebuffer_inst => data_to_dma_from_linebuffer,
      linebufferfull_to_dma_from_the_linebuffer_inst => linebufferfull_to_dma_from_linebuffer,
	  address_from_dma_to_the_arbitrator_inst => address_from_dma_to_arbitrator,
      address_from_dma_to_the_linebuffer_inst => address_from_dma_to_linebuffer,
      data_from_dma_to_the_arbitrator_inst => data_from_dma_to_arbitrator,
      data_ready_from_dma_to_the_arbitrator_inst => data_ready_from_dma_to_arbitrator,
      field_to_dma_from_the_linebuffer_inst => field_from_linebuffer_to_dma,
	  linenumber_to_dma_from_the_linebuffer_inst => linenumber_from_linebuffer_to_dma
    );

--  HEX7     <= "0001001"; -- Leftmost
--  HEX6     <= "0000110";
--  HEX5     <= "1000111";
--  HEX4     <= "1000111";
--  HEX3     <= "1000000";
--  HEX2     <= (others => '1');
--  HEX1     <= (others => '1');
--  HEX0     <= (others => '1');          -- Rightmost
  --LEDG     <= (others => '1');
  LEDR     <= (others => '1');
  LCD_ON   <= '1';
  LCD_BLON <= '1';
  LCD_RW <= '1';
  LCD_EN <= '0';
  LCD_RS <= '0';

  SD_DAT3 <= '1';  
  SD_CMD <= '1';
  SD_CLK <= '1';

--  SRAM_DQ <= (others => 'Z');
--  SRAM_ADDR <= (others => '0');
--  SRAM_UB_N <= '1';
--  SRAM_LB_N <= '1';
--  SRAM_CE_N <= '1';
--  SRAM_WE_N <= '1';
--  SRAM_OE_N <= '1';

  UART_TXD <= '0';
--  DRAM_ADDR <= (others => '0');
--  DRAM_LDQM <= '0';
--  DRAM_UDQM <= '0';
--  DRAM_WE_N <= '1';
--  DRAM_CAS_N <= '1';
--  DRAM_RAS_N <= '1';
--  DRAM_CS_N <= '1';
--  DRAM_BA_0 <= '0';
--  DRAM_BA_1 <= '0';
--  DRAM_CLK <= '0';
--  DRAM_CKE <= '0';
  FL_ADDR <= (others => '0');
  FL_WE_N <= '1';
  FL_RST_N <= '0';
  FL_OE_N <= '1';
  FL_CE_N <= '1';
  OTG_ADDR <= (others => '0');
  OTG_CS_N <= '1';
  OTG_RD_N <= '1';
  OTG_RD_N <= '1';
  OTG_WR_N <= '1';
  OTG_RST_N <= '1';
  OTG_FSPEED <= '1';
  OTG_LSPEED <= '1';
  OTG_DACK0_N <= '1';
  OTG_DACK1_N <= '1';

  TDO <= '0';

--  ENET_CMD <= '0';
--  ENET_CS_N <= '1';
--  ENET_WR_N <= '1';
--  ENET_RD_N <= '1';
--  ENET_RST_N <= '1';
--  ENET_CLK <= '0';
  
  TD_RESET <= KEY(0); -- IMPORTANT !!!!!!
  
  --I2C_SCLK <= '1';

  AUD_DACDAT <= '1';
  AUD_XCK <= '1';
  
  -- Set all bidirectional ports to tri-state
  DRAM_DQ     <= (others => 'Z');
  FL_DQ       <= (others => 'Z');
  --SRAM_DQ     <= (others => 'Z');
  OTG_DATA    <= (others => 'Z');
  LCD_DATA    <= (others => 'Z');
  SD_DAT      <= 'Z';
  I2C_SDAT    <= 'Z';
  ENET_DATA   <= (others => 'Z');
  AUD_ADCLRCK <= 'Z';
  AUD_DACLRCK <= 'Z';
  AUD_BCLK    <= 'Z';
  GPIO_0      <= (others => 'Z');
  GPIO_1      <= (others => 'Z');


end datapath;
