library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;
--use ieee.std_logic_arith.all;

entity linebuffer is
  
  port (
    reset : in std_logic;
    clk_27   : in std_logic;
	clk_50: in std_logic;
	clk_100: in std_logic;
	
	xpos: in std_logic_vector(9 downto 0);
	ypos: in std_logic_vector(9 downto 0);
	datavalid:in std_logic ;
	y: in std_logic_vector (3 downto 0);
	Field: in std_logic;

    -- These inputs & outputs make the component avalon bus 
	-- compliant.
	read : in std_logic;
	write : in std_logic;
	chipselect : in std_logic;
	address : in std_logic_vector(7 downto 0);
	readdata : out std_logic_vector(15 downto 0);
	writedata : in std_logic_vector(15 downto 0);

	-- TO DMA CONTROLLER
	linebufferfull_to_dma : out std_logic;
	data_to_dma: out std_logic_vector(15 downto 0);
	field_to_dma: out std_logic;
	linenumber_to_dma: out std_logic_vector (9 downto 0);
	
	-- FROM DMA CONTROLLER
	address_from_dma: in std_logic_vector(6 downto 0)
	
    );
  
end linebuffer;

architecture behavior of linebuffer is

type ram_type is array ( 80 downto 0) of std_logic_vector(15 downto 0);
signal RAM : ram_type;
signal linebufferfull: std_logic;
signal lineNumber: std_logic_vector(9 downto 0);

begin

	process(clk_27)

	variable tmpaddress: integer range 0 to 79:=0 ;
	variable pixel_no: integer range 0 to 4:=0 ;
	variable tmpdata: std_logic_vector(15 downto 0);
	
		begin
			if clk_27'event and clk_27='1' then 
				linebufferfull <= '0';
				
				-- xpos(0) = '0' ensures every even pixel is taken (and thus every odd pixel is skipped)
				-- ypos(0) = '0' ensures every even line is taken (and odd line is skipped)
				
				-- The ITU decoder outputs 480 lines with 640 pixels per line. Using our sampling
				-- technique decimates the resolution to 320x240
				if (ypos(0)='0' and datavalid='1' and ypos > "0000010001" and 
				    ypos < "0011111110" ) then 
					if xpos(2 downto 0)= "000" then
						tmpdata(15 downto 12) := y;
						pixel_no := pixel_no + 1;
					elsif xpos(2 downto 0)= "010" then
						tmpdata(11 downto 8) := y;
						pixel_no := pixel_no + 1;
					elsif xpos(2 downto 0)= "100" then
						tmpdata(7 downto 4) := y;
						pixel_no := pixel_no + 1;
					elsif xpos(2 downto 0)= "110" then
						tmpdata(3 downto 0) := y;
						pixel_no := pixel_no + 1;
					end if;
		 
					-- If we've gotten four pixels, then: reset the pixel index, update the
					-- write address of the local line buffer, clear the four pixel array, and
					-- finally write the line buffer
					if pixel_no = 4 then 
						pixel_no := 0;
						RAM(tmpaddress)<= tmpdata;
						tmpdata := "0000000000000000"; 
						
						-- If we've filled addresses 0-79, it's time to flush the buffer out to the
						-- NIOS.
						if (tmpaddress = 79) then 
							linebufferfull <= '1';
							tmpaddress:=0;	
							lineNumber <= std_logic_vector(unsigned(ypos)-18);
							RAM(80) <= Field & "0000" & lineNumber & '1';
						else
							tmpaddress := tmpaddress + 1;
						end if;
					end if;


					
				end if;
			end if;	
	end process;  
	

	process (clk_50)	

	begin
		if rising_edge(clk_50) then
			linebufferfull_to_dma <= linebufferfull;
			data_to_dma <= RAM(to_integer(unsigned(address_from_dma)));
			field_to_dma <= Field;
			linenumber_to_dma <= lineNumber;
		end if;
	end process;
	
	-- Avalon stuff. WRITE = NIOS->HERE
	--				 READ  = HERE->NIOS
	process (clk_100)	
	begin
		if rising_edge(clk_100) then
			if reset = '1' then
				readdata <= (others => '0');
			else
				if chipselect = '1' then
					if read = '1' then
						if address = "01010001" then
							readdata <= datavalid & "00000" & xpos;
						elsif address = "01010010" then
							readdata <= datavalid & "00000" & ypos;
						else
							readdata <= RAM(to_integer(unsigned(address)));
						end if;
					elsif write = '1' then
						-- Do not care
					end if;
				end if;
			end if;
		end if;
	end process;

end behavior;



