library ieee;

use ieee.std_logic_1164.all;
use ieee.numeric_std.all;


entity arbitrator is
  
  port (
    reset : in std_logic;
    clk   : in std_logic;			-- 50 Mhz

    -- These inputs & outputs make the component avalon bus 
	-- compliant.
	read : in std_logic;
	write : in std_logic;
	chipselect : in std_logic;
	address : in std_logic_vector(15 downto 0);
	readdata : out std_logic_vector(15 downto 0);
	writedata : in std_logic_vector(15 downto 0);
	waitrequest: out std_logic;
	
	-- SRAM Connection
	SRAM_DQ   : inout std_logic_vector(15 downto 0);
    SRAM_ADDR : out std_logic_vector(17 downto 0);
    SRAM_UB_N, SRAM_LB_N : out std_logic;
    SRAM_WE_N, SRAM_CE_N : out std_logic;
    SRAM_OE_N            : out std_logic;

	-- From the VGA Controller
	HCOUNT, VCOUNT: unsigned (9 downto 0);
	
	-- To the VGA Controller
	RED, GREEN, BLUE : out std_logic_vector (9 downto 0);
	pixel_clock: out std_logic;
	
	-- TO DMA
	busy_to_dma: out std_logic;
	
	-- FROM DMA
	address_from_dma: in std_logic_vector(17 downto 0);
	data_from_dma: in std_logic_vector(15 downto 0);
	data_ready_from_dma: in std_logic
    );
  
end arbitrator;

architecture behavior of arbitrator is

	signal clk25 : std_logic := '0';
	
	signal SRAM_ADDRESS_BUFFER			: std_logic_vector(17 downto 0) := "000000000000000000";
	signal SRAM_WRITEDATA_BUFFER		: std_logic_vector(15 downto 0) := "0000000000000000";
	signal SRAM_READDATA_BUFFER         : std_logic_vector(15 downto 0) := "0000000000000000";  
	signal SRAM_WRITE_ENABLE_BUFFER_N   : std_logic := '0';	
	signal read_flag					: std_logic := '0';
	signal write_flag					: std_logic := '0';
	signal RED_BUFFER, GREEN_BUFFER, 
	       BLUE_BUFFER					: std_logic_vector (9 downto 0) := "0000000000";
	
	signal sramWriteAddressFromNIOS     : std_logic_vector(15 downto 0) := "0000000000000000";
	signal pixelDataFromNIOS            : std_logic_vector(15 downto 0) := "0000000000000000";
	
begin

process (clk)
	begin
		if rising_edge(clk) then
			clk25 <= not clk25;
		end if;
end process;

SRAM_DQ <= SRAM_WRITEDATA_BUFFER when SRAM_WRITE_ENABLE_BUFFER_N = '0' else (others => 'Z');
SRAM_READDATA_BUFFER <= SRAM_DQ;
SRAM_ADDR <= SRAM_ADDRESS_BUFFER;
SRAM_WE_N <= SRAM_WRITE_ENABLE_BUFFER_N;
SRAM_UB_N <= '0';
SRAM_LB_N <= '0';
SRAM_CE_N <= '0';
SRAM_OE_N <= '0';

pixel_clock <= clk25;


process(clk,clk25)
	
	variable pixelIndex: integer := -1;
	variable addr: std_logic_vector (19 downto 0) := "00000000000000000000";
	variable tmp: unsigned (19 downto 0) := "00000000000000000000";
	variable tmp2: unsigned (19 downto 0) := "00000000000000000000";
	variable fourPixels: std_logic_vector(15 downto 0) := "0000000000000000";
		
	begin
		-- CLK_50 Positive Edge
		if clk'event and clk='1' then 
		busy_to_dma <= '0';
			-- CLK_50 Positive Edge, CLK_25 Negative Edge
			if clk25 = '0' then
				-- Are we about to be in the active region? Add 1 to HCOUNT because it's old
				if (HCOUNT+1 > (8 + 96 + 40 + 8 + 160 - 1) and HCOUNT+1 < (8 + 96 + 40 + 8 + 160 + 320)
					and VCOUNT > (2 + 2 + 25 + 8 - 1) and VCOUNT < (2 + 2 + 25 + 8 + 480) ) then
				
					-- If our four pixel buffer is empty
					if pixelIndex = -1 then
						-- First, lets caculate the desired SRAM address based on HCOUNT & VCOUNT.
						tmp := ((VCOUNT - 37) * 80);
						tmp2 := "0000000000" & ((HCOUNT+1+160)-152);
						tmp2 := "00" & tmp2 (19 downto 2); -- Divide by 4
						tmp := tmp + tmp2;
						SRAM_ADDRESS_BUFFER <= std_logic_vector(tmp) (17 downto 0);
						SRAM_WRITE_ENABLE_BUFFER_N <= '1';
						read_flag <= '1';
						waitrequest <= '1';
						
					elsif chipselect = '1' and write = '1' then
						SRAM_ADDRESS_BUFFER <= "00" & address;
						SRAM_WRITE_ENABLE_BUFFER_N <= '0';
						SRAM_WRITEDATA_BUFFER <= writedata;
						write_flag <= '1';	
						waitrequest <= '0';			
						

					elsif data_ready_from_dma = '1' then
						SRAM_ADDRESS_BUFFER <= address_from_dma;
						SRAM_WRITE_ENABLE_BUFFER_N <= '0';
						SRAM_WRITEDATA_BUFFER <= data_from_dma;
						write_flag <= '1';
						waitrequest <= '1';
					end if;								

				elsif chipselect = '1' and write = '1' then
					SRAM_ADDRESS_BUFFER <= "00" & address;
					SRAM_WRITE_ENABLE_BUFFER_N <= '0';
					SRAM_WRITEDATA_BUFFER <= writedata;
					write_flag <= '1';
					waitrequest <= '0';	
				

				elsif data_ready_from_dma = '1' then
					SRAM_ADDRESS_BUFFER <= address_from_dma;
					SRAM_WRITE_ENABLE_BUFFER_N <= '0';
					SRAM_WRITEDATA_BUFFER <= data_from_dma;
					write_flag <= '1';
					waitrequest <= '1';
						
				end if;
				
				
			-- CLK_50 Positive Edge, CLK_25 Positive Edge
			elsif clk25 = '1' then
				-- If we're in the active region
				if (HCOUNT > (8 + 96 + 40 + 8 + 160 - 1) and HCOUNT < (8 + 96 + 40 + 8 + 160 + 320)
					and VCOUNT > (2 + 2 + 25 + 8 - 1) and VCOUNT < (2 + 2 + 25 + 8 + 480) ) then
			
					if chipselect = '1' and write = '1' then
						SRAM_ADDRESS_BUFFER <= "00" & address;
						SRAM_WRITE_ENABLE_BUFFER_N <= '0';
						SRAM_WRITEDATA_BUFFER <= writedata;
						write_flag <= '1';
						waitrequest <= '0';	
										
					elsif data_ready_from_dma = '1' then
						SRAM_ADDRESS_BUFFER <= address_from_dma;
						SRAM_WRITE_ENABLE_BUFFER_N <= '0';
						SRAM_WRITEDATA_BUFFER <= data_from_dma;
						write_flag <= '1';
						waitrequest <= '1';
					end if;					

										
					-- Read the SRAM contents & buffer it locally *if* we just initiated a read
					if read_flag = '1' then
						fourPixels := SRAM_READDATA_BUFFER;
						--SRAM_WRITE_ENABLE_BUFFER_N <= '0';
						pixelIndex := 0;
						read_flag <= '0';
						waitrequest <= '0';
					end if;
					
					-- To the VGA Controller
					case pixelIndex is
						when 0 =>
							RED <= fourPixels (15 downto 12) & "000000";
							GREEN <= fourPixels (15 downto 12) & "000000";
							BLUE <= fourPixels (15 downto 12) & "000000";
						when 1 =>
							RED <= fourPixels (11 downto 8) & "000000";
							GREEN <= fourPixels (11 downto 8) & "000000";
							BLUE <= fourPixels (11 downto 8) & "000000";
						when 2 =>
							RED <= fourPixels (7 downto 4) & "000000";
							GREEN <= fourPixels (7 downto 4) & "000000";
							BLUE <= fourPixels (7 downto 4) & "000000";
						when 3 =>
							RED <= fourPixels (3 downto 0) & "000000";
							GREEN <= fourPixels (3 downto 0) & "000000";
							BLUE <= fourPixels (3 downto 0) & "000000";	
									
							busy_to_dma <= '1';							
						when others =>
							RED <= "0000000000";
							GREEN <= "0000000000";
							BLUE <= "0000000000"; 
					end case;
					
					-- If all four pixels have been read, time to signify that the pixel
					-- buffer is empty so a new SRAM read can be initiated
					if pixelIndex = 3 then
						pixelIndex := -1;
					else
						pixelIndex := pixelIndex + 1;
					end if;
					
				-- If not, output black
				else
					RED <= "0000000000";
					GREEN <= "0000000000";
					BLUE <= "0000000000"; 
					
					if chipselect = '1' and write = '1' then
						SRAM_ADDRESS_BUFFER <= "00" & address;
						SRAM_WRITE_ENABLE_BUFFER_N <= '0';
						SRAM_WRITEDATA_BUFFER <= writedata;
						write_flag <= '1';
						waitrequest <= '0';	
										
					elsif data_ready_from_dma = '1' then
						SRAM_ADDRESS_BUFFER <= address_from_dma;
						SRAM_WRITE_ENABLE_BUFFER_N <= '0';
						SRAM_WRITEDATA_BUFFER <= data_from_dma;
						write_flag <= '1';
						waitrequest <= '1';
					end if;

										
				end if;

			end if;
		
		
		-- CLK_50 Negative Edge
		elsif clk'event and clk='0' then
		
		end if;

end process;	


	-- Avalon stuff. WRITE = NIOS->HERE
	--				 READ  = HERE->NIOS
	process (clk)	
	begin
		if clk'event and clk='1' then
			if reset = '1' then
				readdata <= (others => '0');
			else
				if chipselect = '1' then
						if read = '1' then
							if address = "0000000000000000" then
								readdata <= std_logic_vector("000000" & HCOUNT);
								--readdata <= testSig;
							elsif address = "0000000000000001" then
								readdata <= SRAM_READDATA_BUFFER;
							elsif address = "0000000000000010" then
								readdata <= SRAM_WRITEDATA_BUFFER;								
							elsif address = "0000000000000100" then
								readdata <= "000000000000000" & SRAM_WRITE_ENABLE_BUFFER_N;		
							elsif address = "0000000000001000" then
								readdata <= SRAM_ADDRESS_BUFFER (15 downto 0);
							end if;

						end if;
				end if;
			end if;
		end if;
	end process;

end behavior;
