-------------------------------------------------------------------------------
--
-- Simple VGA raster display
--
-- Stephen A. Edwards
-- sedwards@cs.columbia.edu
--
-------------------------------------------------------------------------------
library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;

entity VGA is

generic 
	(
		DATA_WIDTH : natural := 64;
		ADDR_WIDTH : natural := 10
	);
  
  port (
    reset_n : in std_logic;
    clk27   : in std_logic;                    
	clk50   : in std_logic; 
    VGA_CLK,                         -- Clock
    VGA_HS,                          -- H_SYNC
    VGA_VS,                          -- V_SYNC
    VGA_BLANK,                       -- BLANK
    VGA_SYNC : out std_logic;        -- SYNC
    VGA_R,                           -- Red[9:0]
    VGA_G,                           -- Green[9:0]
    VGA_B : out std_logic_vector(9 downto 0); -- Blue[9:0]
	readin:in std_logic_vector(14 downto 0 );
	--iG,iR,iB: in std_logic_vector(7 downto 0);
	TD_HS,TD_HS_edge,
	TD_VS: in std_logic;
	vga_vsync_edge: in std_logic;
	data_valid: in std_logic;
	
	bluecount, greencount : in std_logic_vector(9 downto 0);
	x_blue, x_green : in std_logic_vector(9 downto 0);
	Linecount_vga : in std_logic_vector(9 downto 0);	--@@
			-- Avalon_signals

		 signal address : IN STD_LOGIC_VECTOR (15 DOWNTO 0);
		 signal chipselect : IN STD_LOGIC;
		 signal read : IN STD_LOGIC;
		 signal write : IN STD_LOGIC;
		 signal writedata : IN STD_LOGIC_VECTOR (15 DOWNTO 0);		
		 signal readdata : OUT STD_LOGIC_VECTOR (15 DOWNTO 0)	
);

end VGA;

architecture rtl of VGA is

component ram_dual
port (
		rclk	: in std_logic;
		wclk	: in std_logic;
		raddr	: in natural range 0 to 2**ADDR_WIDTH - 1;
		waddr	: in natural range 0 to 2**ADDR_WIDTH - 1;
		data	: in std_logic_vector((DATA_WIDTH-1) downto 0);
		we		: in std_logic := '1';
		q		: out std_logic_vector((DATA_WIDTH -1) downto 0)
	);

end component;
  

  constant HTOTAL       : integer := 858;
  constant HSYNC        : integer := 103;
  constant HBACK_PORCH  : integer := 76;
  constant HACTIVE      : integer := 640;
  constant HFRONT_PORCH : integer := 39;

  constant VTOTAL       : integer := 524;
  constant VSYNC        : integer := 2;
  constant VBACK_PORCH  : integer := 17;		--34;
  constant VACTIVE      : integer := 500;--478;
  constant VFRONT_PORCH : integer := 5;--10;

  -- Signals for the video controller
  signal Hcount : unsigned(10 downto 0);  -- Horizontal position (0-800)
  signal Vcount : unsigned(9 downto 0);  -- Vertical position (0-524)
  signal EndOfLine, EndOfField : std_logic;
  --signal vga_x : std_logic := '0';

  signal vga_hblank, vga_hsync,
    vga_vblank, vga_vsync, flag_TD_HS_edge,flag_TD_VS_edge,flag_TD_HS_edge_reset : std_logic;  -- Sync. signals
                           
type ROMType is array(0 to 640)of std_logic_vector(14 downto 0);
  signal ROM1, ROM2:ROMType := (others => "000000000000000");
signal Read_HS_flag : std_logic := '0';
signal a,b,c: std_logic;
signal INDEX : integer range 0 to 800;
signal regx1, regx2, inner_regx1, inner_regx2 : integer := 0; 			--range 0 to 640;
signal x_info_from_nios, x_info_to_vga : std_logic_vector(63 downto 0);
signal ram_index : integer range 0 to 500:=0; 


begin		


VGA_RAM : ram_dual port map(
		rclk => clk27,	
		wclk	=> clk50,
		raddr	=>(to_integer(Vcount)-VSYNC - VBACK_PORCH),
		waddr	=> ram_index,
		data	=> x_info_from_nios,
		q => x_info_to_vga		
	);

ReadfromRAM : process(clk27)
begin
if rising_edge(clk27) then
regx1 <= to_integer(unsigned(x_info_to_vga(15 downto 0)));
regx2 <= to_integer(unsigned(x_info_to_vga(31 downto 16)));
inner_regx1<=to_integer(unsigned(x_info_to_vga(47 downto 32)));
inner_regx2<=to_integer(unsigned(x_info_to_vga(63 downto 48)));
end if;
end process ReadfromRAM;	
		
  HCounter : process (clk27)
  begin
    if rising_edge(clk27) then      
      if reset_n = '0'  then-- or  
        Hcount <= (others => '0');
      elsif TD_HS='1'then
		Hcount <=(others =>'0');
		else
		Hcount <= Hcount+1;	
	  end if;
	end if;
  end process HCounter;

  VCounter: process (clk27)
  begin
    if rising_edge(clk27) then      
      if reset_n = '0'then 
        Vcount <= (others => '0');
	  elsif vga_vsync_edge='1' then
		Vcount <= (others => '0');	
		elsif TD_HS='1' then
		Vcount<=Vcount+1;
      end if;
    end if;
  end process VCounter;

  -- State machines to generate HSYNC, VSYNC, HBLANK, and VBLANK

  HSyncGen : process (clk27)
  begin
    if rising_edge(clk27) then     
      if reset_n = '0' then
        vga_hsync <= '1';
		elsif TD_HS='1' then
		 vga_hsync <= '1';
		elsif Hcount=Htotal-1 then
		 vga_hsync <= '1';
      elsif Hcount = HSYNC - 1 or Hcount=HTOTAL+HSYNC-1 then
        vga_hsync <= '0';
      end if;
    end if;
  end process HSyncGen;
  
  HBlankGen : process (clk27)
  begin
    if rising_edge(clk27) then
      if reset_n = '0' then
        vga_hblank <= '1';
      elsif Hcount = HSYNC + HBACK_PORCH-1 or Hcount = HTOTAL+HSYNC + HBACK_PORCH-1 then
        vga_hblank <= '0';
      elsif Hcount = HSYNC + HBACK_PORCH + HACTIVE-1 or 
			Hcount = HTOTAL + HSYNC + HBACK_PORCH + HACTIVE-1 then
        vga_hblank <= '1';
      end if;      
    end if;
  end process HBlankGen;

  VSyncGen : process (clk27)
  begin
if rising_edge(clk27) then	
	if reset_n = '0' then
        vga_vsync <= '1';
    elsif vga_vsync_edge='1' then
		vga_vsync<='1';
	elsif Vcount = VSYNC - 1 then
          vga_vsync <= '0';

      end if;
end if;
end process VSyncGen;



  VBlankGen : process (clk27)
  begin
    if rising_edge(clk27) then    
      if reset_n = '0' then
        vga_vblank <= '1';
        if Vcount = VSYNC+VBACK_PORCH - 1 then
          vga_vblank <= '0';
        elsif Vcount =VSYNC+VBACK_PORCH + VACTIVE - 1 then
          vga_vblank <= '1';
        end if;
      end if;
    end if;
  end process VBlankGen;

Read_HS: process(TD_HS)
begin
		if TD_HS'event and TD_HS = '1' then
			Read_HS_flag <= not Read_HS_flag;
		end if;
end process Read_HS;

fill_buffer: process(clk27,data_valid)
variable i,j : integer range 0 to 720 := 0;
begin
	if rising_edge(clk27) then
		if Read_HS_flag = '1' and data_valid='1' then
			ROM1 ( i ) <= readin ;
			i := i+ 1;
			if i = 639 then 
				i := 0;
			elsif TD_HS='1' then 
				i:=0;
			end if;
		elsif Read_HS_flag='0' and data_valid='1' then
			ROM2 ( j ) <= readin ;
			j := j+1;
			if j = 639 then 
				j := 0;
			elsif TD_HS='1' then 
				j:=0;
			end if;
		end if;
	end if;	
end process fill_buffer;

---------------------------------------------------------------------------------------------
-----------------------------------NIOS Communication----------------------------------------
--bus_comm: process (clk50)
--  begin
--    if rising_edge(clk50) then
--		if (chipselect = '1')then
--			if address(4 downto 0) = "00000" then
--                if write = '1' then
--                      regx1 <= to_integer(unsigned(writedata));--x1
--                end if;
--			elsif address(4 downto 0) = "00010" then
--                if write = '1' then
--                      regx2 <= to_integer(unsigned(writedata));--x2
--                 end if;
--				end if;
--		
--		end if;
--    end if;
--  end process bus_comm;
------------------------------------------------------------------------------------------------------
bus_comm: process (clk50)
  begin
    if rising_edge(clk50) then
		if (chipselect = '1')then
			if address(4 downto 0) = "00000" then
                if write = '1' then
                    x_info_from_nios(15 downto 0) <= writedata;--x1
                end if;
			elsif address(4 downto 0) = "00010" then
                if write = '1' then
                    x_info_from_nios(31 downto 16) <= writedata;--x2
                 end if;
			elsif address(4 downto 0) = "11110" then  --- using address 30 inner x1
                if write = '1' then
                    x_info_from_nios(47 downto 32) <= writedata;
                 end if;
			elsif address(4 downto 0) = "10010" then  --- using address 18 inner x2
                if write = '1' then
                    x_info_from_nios(63 downto 48) <= writedata;
                 end if;

			elsif address(4 downto 0) = "11000" then   -- unused address 24
                if write = '1' then
                    ram_index <= to_integer(unsigned(writedata));
                 end if;
            
				end if;
				
		
		end if;
    end if;
  end process bus_comm;
-----------------------------------------------------------------------------------------------------
Read_process: process(clk50)
 begin
    if rising_edge(clk50) then
		if vga_vsync_edge = '1' then
			flag_TD_VS_edge <= '1';
		end if;
		if TD_HS='1' then 
			flag_TD_HS_edge<='1';
		end if;	

	
	if (chipselect = '1')then	
-----------------------------------VSFlagwrite(ADDR 20)---------------------------------------------------		
			if address(4 downto 0) = "10100" then
                if write = '1' then
                    flag_TD_VS_edge <= '0';
					end if;
                end if;
-----------------------------------HS FlagWrite(ADDR 30)----------------------------------------------------------
--			if address(4 downto 0) = "11110" then
--                if write = '1' then
--                    flag_TD_HS_edge <= '0';
--					end if;
--                end if;
			
		

-------------------------------------------------------VGA_HS(ADDR 4)----------------------------------
			if address(4 downto 0) = "00100" then
                if read = '1' then
                    if vga_hsync='1' then
							readdata<="0000000000000001";		-- We send VGA_HS in form of 1 or 0
						else
							readdata<=(others=>'0');
					end if;
                end if;
			 end if;
			
	----------------------ROW_NO (ADDR 6)--------------------------------------------		
			if address(4 downto 0) = "00110" then
                if read = '1' then
					if vga_hsync='0' then
						if  (to_integer(Vcount)-VSYNC - VBACK_PORCH)>0 and (to_integer(Vcount)-VSYNC - VBACK_PORCH) < 478 then		-- line number is sent to NIOS
							readdata <= std_logic_vector(to_unsigned((to_integer(Vcount)-VSYNC - VBACK_PORCH), 16));			-- 
						else 
							readdata<=(others=>'0');
						end if;
					end if;
				end if;
			 end if;

	---------------------------Blue count(ADDR 8)-------------------------------------------		
				if address(4 downto 0) = "01000" then
					if read = '1' then
						--if TD_HS = '0' then
						readdata <= ("000000" & bluecount);
						end if;
						end if;
						--end if;
 -------------------------------Xblue(ADDR 10)-----------------------------------------------						
				if address(4 downto 0) = "01010" then
					if read = '1' then
				   -- if TD_HS = '0' then
				  readdata <= ("000000" & x_blue);				-- software must find x_blue - bluecount 
					end if;
					end if;
					--end if;
----------------------------------Green count(ADDR 12)--------------------------------------					
				if address(4 downto 0) = "01100" then
						if read = '1' then
						--if TD_HS = '0' then
				  readdata <= ("000000" & greencount);
					end if;
					end if;
----------------------------------Xgreen(ADDR 14)------------------------------------------------					
				if address(4 downto 0) = "01110" then
					if read = '1' then
				  readdata <= ("000000" & x_green);				-- software must find x_green - greencount 
					end if;
					end if;
				
				
------------------------VGA_VS(ADDR 16)------------------------------------------------		

			if address(4 downto 0) = "10000" then
				if read ='1' then
					if vga_vsync='1' then
							readdata<="0000000000000001";		-- We send VGA_VS in form of 1 or 0
						else
							readdata<=(others=>'0');
					end if;
					end if;
				end if;
-----------------------TD_HS(ADDR 18)------------------------------------------------
--		if address(4 downto 0) ="10010" then
--			if read='1' then
--				if TD_HS ='1'then
--				readdata<="0000000000000001";		-- We send TD_HS in form of 1 or 0
--				else
--				readdata<=(others=>'0');
--				end if;
--				end if;
--				end if;

-----------------------------TD_HS_edge(ADDR 22)---------------------------------------------------------	
--		if address(4 downto 0) ="10110" then
--			if read='1' then
--				if flag_TD_HS_edge = '1' then
--				readdata<= "0000000000000001";		-- We send TD_HS in form of 1 or 0
--				else
--				readdata<=(others=>'0');
--				end if;
--				end if;
--				end if;

-----------------------------Linecount_vga(ADDR 26)---------------------------------------------------------	
		if address(4 downto 0) ="11010" then
			if read='1' then
				--if TD_HS = '1' then
				readdata<= "000000" & Linecount_vga;		-- send line count 
				end if;
				end if;

--------------------------------VS_edge(ADDR 28)---------------------------------------------------------				
		if address(4 downto 0) ="11100" then
			if read='1' then
				if flag_TD_VS_edge = '1' then
				readdata<= "0000000000000001";		-- We send TD_VS_edge
				else
				readdata<=(others=>'0');
				end if;
				end if;
				end if;
		end if;
end if;
end process Read_process;


VideoOut: process (clk27, reset_n)
  begin
 if rising_edge(clk27) then
	if(Hcount<HTOTAL) then
	INDEX<=to_integer(HCOUNT-HSYNC-HBACK_PORCH);		-- 
	else
	INDEX<=to_integer(HCOUNT-HSYNC-HBACK_PORCH-HTOTAL);		-- 
	end if;	
			if reset_n = '0' then
			  VGA_R <= "0000000000";
			  VGA_G <= "0000000000";
			  VGA_B <= "0000000000";
			else
			if vga_hblank = '0' and vga_vblank ='0' then
			  if Read_HS_flag = '1' then
				if INDEX > regx1 and INDEX < regx2 then
					if( INDEX > inner_regx1 and INDEX < inner_regx2) then
					VGA_R <= "1111111111";
					VGA_G <= "1111111111";   -- making a plain green light saber
					VGA_B <= "1111111111";
					else
					VGA_R <= ROM2(INDEX)(14 downto 10) & "11111";
					VGA_G <= ROM2(INDEX)(9) & "111111111";  -- transparent light saber
					VGA_B <= ROM2(INDEX)(4 downto 0)& "11111"; 
					end if;
				else
				VGA_R <= ROM2(INDEX)(14 downto 10) & "00000";
				VGA_G <= ROM2(INDEX)(9 downto 5)& "00000";
				VGA_B <= ROM2(INDEX)(4 downto 0)& "00000"; 
				end if;
			  elsif Read_HS_flag = '0' then
				if INDEX > regx1 and INDEX < regx2 then
					if( INDEX > inner_regx1 and INDEX < inner_regx2) then
					VGA_R <= "1111111111";
					VGA_G <= "1111111111";   -- making a plain green light saber
					VGA_B <= "1111111111";
					else
					VGA_R <= ROM1(INDEX)(14 downto 10) & "11111";
					VGA_G <= ROM1(INDEX)(9) & "111111111";  -- transparent light saber
					VGA_B <= ROM1(INDEX)(4 downto 0)& "11111"; 
					end if;
				else
				VGA_R <= ROM1(INDEX)(14 downto 10) & "00000";
				VGA_G <= ROM1(INDEX)(9 downto 5)& "00000";
				VGA_B <= ROM1(INDEX)(4 downto 0)& "00000"; 
				end if;
	else
        VGA_R <= "0000000000";
        VGA_G <= "0000000000";
        VGA_B <= "0000000000";    
    end if;
  end if;
  end if;
  end if;
  end process VideoOut;

  VGA_CLK <= clk27;
  VGA_HS <= not vga_hsync;
  VGA_VS <= not vga_vsync;
  VGA_SYNC <= '0';
  VGA_BLANK <= not (vga_hsync or vga_vsync);

end rtl;
