#include <io.h>
#include <system.h>
#include <stdio.h>
#include "sys/alt_irq.h"
#include <alt_types.h>
#include <math.h>
#include <unistd.h>
#include <fcntl.h>

//READ/WRITE REGISTERS
#define STEER_W(data) IOWR_16DIRECT(PWM_UNIT2_BASE, 0, data)    
#define THROTTLE_W(data) IOWR_16DIRECT(PWM_UNIT2_BASE, 2 , data)   
#define STEER_R() IORD_16DIRECT(PWM_UNIT2_BASE, 0)
#define THROTTLE_R() IORD_16DIRECT(PWM_UNIT2_BASE, 2)
#define SERVO_W(data) IOWR_16DIRECT(PWM_UNIT2_BASE, 4 , data)   
#define SERVO_R() IORD_16DIRECT(PWM_UNIT2_BASE, 4)

//SENSOR REGISTERS
#define TIMER_R() IORD_16DIRECT(SENSOR_BASE, 4)
#define CLEAR_IRQ() IOWR_16DIRECT(SENSOR_BASE, 2, 0)
#define READ_SENSOR() IORD_16DIRECT(SENSOR_BASE, 0) 

// TEST SETTINGS
#define KEYBOARD_CONTROL 0
#define CALIBRATE 0
#define SENSOR_TEST 0
#define FEEDBACK_CONTROL 0
#define STEER_CONTROL 1

//CONSTANTS
#define CAL_COUNT 101
#define RECORDING_LENGTH 2000
#define THROTTLE_REFRESH_FREQ 4 //every fifth cycle

#define RIGHT    1
#define LEFT    -1
#define STRAIGHT 0


//FUNCTION PROTOTYPES
void steer_control_monitor();
void feedback_monitor();
void delay(int);
void initialize();
void forward(int);
void backward(int);
void left(int);
void right(int);
void halt();
int set_throttle(int, short);
int set_steer(int, short);
static void sensor_interrupt_handler();
void calibrate();
void keyboard_control();
void sensor_test();
int feedback_control();
void steer(int);

//speed, distance, and flag variables in handler
int breaking_distance = 20;

int distance_reading = 0;
int distance = 0;
int delta_distance;
int last_distance = -1;
int desired_dist = 40;
int speed = 0;
int time_msec = 0;
int elapsed_time = 0;
short start = 0;
short success = 0;
short stationary_count = 0;
short fourcount = 0;
short similar_count = 0;
short turn_cycles = 0;
short last_turn_cycles = 8;
short direction = STRAIGHT;
int dist_on_turn; //distance from wall during last turn

//calibration variables
int counter = 0;
int calibrated = 0;
int diff;
int mult;

//global throttle, steering, and servo variables
int max_throttle = 800;//820;
int min_throttle = 540;
int max_steer = 800;
int min_steer = 550;
int neutral_steer = 700;
int neutral_throttle = 750;
int current_throttle;
int current_steer;
int previous_throttle;
int previous_steer;
int max_throttle_increase = 5;
int max_throttle_decrease = 7;
int max_steer_increase = 30;
int max_steer_decrease = 30;
int servo_straight_pwm = 920;
int servo_right_pwm = 420;
int residual_throttle;

//feedback variables
int error;
int K_gain = 1;
int K_rate = -2;
int gain_out;
int rate_out;
int throttle_correction;
int error_in_pwm;
int rate_in_pwm;

//data logging
FILE *fp;
short recorded_time[RECORDING_LENGTH];
short filtered_distance[RECORDING_LENGTH];
short dirty_distance[RECORDING_LENGTH];
short distance_delta[RECORDING_LENGTH];
short throttle[RECORDING_LENGTH];
short gain[RECORDING_LENGTH];
short rate[RECORDING_LENGTH];
short record_index;
short latest_results[2][6];

//SENSOR INTERRUPT HANDLER
static void sensor_interrupt_handler() {
    
    distance_reading = READ_SENSOR();
    time_msec = TIMER_R();
    elapsed_time += time_msec;
    
    latest_results[0][5] = latest_results[0][4];
    latest_results[0][4] = latest_results[0][3];
    latest_results[0][3] = latest_results[0][2];
    latest_results[0][2] = latest_results[0][1];
    latest_results[0][1] = latest_results[0][0];
    latest_results[0][0] = distance_reading;
    
    latest_results[1][3] = latest_results[1][2];
    latest_results[1][2] = latest_results[1][1];
    latest_results[1][1] = latest_results[1][0];
    latest_results[1][0] = elapsed_time;
          
    if( ((latest_results[0][1] > latest_results[0][0]) && (latest_results[0][1] > latest_results[0][2])) ||
        ((latest_results[0][1] < latest_results[0][0]) && (latest_results[0][1] < latest_results[0][2])) )
    {
        latest_results[0][1] = (latest_results[0][2] + latest_results[0][0]) / 2;
    }
   
    distance = latest_results[0][1]; //filtered
    last_distance = latest_results[0][2]; //filtered  
    delta_distance = (int)(10*(
                            0.1 * (latest_results[0][5]-latest_results[0][4]) + 
                            0.2 * (latest_results[0][4]-latest_results[0][3]) +
                            0.3 * (latest_results[0][3]-latest_results[0][2]) +
                            0.4 * (latest_results[0][2]-latest_results[0][1]) )); //filtered
    
    error = distance - desired_dist;
    
    if (start == 1)
    {
        if(STEER_CONTROL)
        {
            if(fourcount < 4)
            {
                set_throttle(771, 0);
                fourcount++;
            }
            else if(fourcount >= 4 && fourcount < 10)
            {
                set_throttle(neutral_throttle, 0);
                fourcount++;
            }
            if (fourcount == 10)
                fourcount = 0;
            
            if (turn_cycles == 0)
            {       
                if (error >= 1 && dist_on_turn <= distance)
                {
                    turn_cycles = error;
                    /*
                    if(error < 1)
                    {
                        turn_cycles = 2;
                    }
                    else if(error < 2)
                    {
                        turn_cycles = 3;
                    }
                    else if(error < 3)
                    {
                        turn_cycles = 4;
                    }
                    else if(error < 4)
                    {
                        turn_cycles = 5;
                    }
                    else if (error < 6)
                    {
                        turn_cycles = 6;
                    }
                    else if(error >= 6 && error < 12)
                    {
                        turn_cycles = 8;
                    }
                    else if(error >= 12 && error < 24)
                    {
                        turn_cycles = 12;
                    }
                    else
                    {
                        turn_cycles = 16;
                    }
                    */
                    if(turn_cycles > 20)
                        turn_cycles = 20;
                    
                    last_turn_cycles = turn_cycles;
                    direction = RIGHT; 
                    dist_on_turn = distance;     
                }
                else if(error <= -1 && dist_on_turn >= distance)
                {
                    turn_cycles = fabs(error);
                    /*
                    if(error > -1)
                    {
                        turn_cycles = 2;
                    }
                    else if(error > -2)
                    {
                        turn_cycles = 3;
                    }
                    else if(error > -3)
                    {
                        turn_cycles = 4;
                    }
                    else if(error > -4)
                    {
                        turn_cycles = 5;
                    }
                    else if (error > -6)
                    {
                        turn_cycles = 6;
                    }
                    else if(error <= -6 && error > -12)
                    {
                        turn_cycles = 8;
                    }
                    else if(error <= -12 && error > -24)
                    {
                        turn_cycles = 12;
                    }
                    else
                    {
                        turn_cycles = 16;
                    }
                    */
                    if(turn_cycles > 20)
                        turn_cycles = 20;
                    
                    last_turn_cycles = turn_cycles;
                    direction = LEFT;
                    dist_on_turn = distance;
                }
                else
                {
                    turn_cycles = 4 ;
                    last_turn_cycles = turn_cycles;
                    direction = STRAIGHT;
                    dist_on_turn = distance;
                }              
            }
            
            if(turn_cycles > 0)
            {
                steer(direction);
                turn_cycles--;
            }
        }
        
        if (FEEDBACK_CONTROL) //if the experiemnt is running
        {    
            if (fourcount == 0) 
            {
                fourcount = THROTTLE_REFRESH_FREQ;
                throttle_correction = feedback_control();
                residual_throttle = set_throttle(current_throttle + throttle_correction, 1);
            }
            else 
            {
                residual_throttle = set_throttle(current_throttle + residual_throttle, 1);
                fourcount--;
            }
                                                                             
            if (record_index < RECORDING_LENGTH)
            {
                recorded_time[record_index] = latest_results[1][1];
                filtered_distance[record_index] = latest_results[0][1];
                dirty_distance[record_index] = distance_reading;
                distance_delta[record_index] = delta_distance;
                throttle[record_index] = current_throttle >> 2;
                gain[record_index] = gain_out;
                rate[record_index] = rate_out; 
                
                record_index++;
            }
            
            if (last_distance <= distance && distance < (desired_dist + breaking_distance))
            {
                similar_count++;            
            }
            else
            {     
                similar_count = 0;   
            }
            
            if (similar_count == 5) //end the experiment
            { 
                halt();  
                start = 2;
            }       
        }
    }
    else if(start == 2)
    {    
        halt();
        
        if(FEEDBACK_CONTROL)
        {             
            if(distance == last_distance) //collect data due to inertia after the experiment is terminated
            {
                stationary_count++;
            }
            else
            {
                stationary_count = 0;
            }
            
            if(stationary_count == 10)
            {                   
                success = 1;
                start = 0;
            } 
        }
    }
    
    CLEAR_IRQ();   // clear the interrupt   
}

steer(int direction)
{
    if (direction == RIGHT)
        set_steer(max_steer, 0);
    else if (direction == LEFT)
        set_steer(min_steer + 50, 0);
    else
        set_steer(neutral_steer, 0);
}

int main()
{   
    printf("entering ...\n");

    initialize(); //registers sensor interrupt and other stuff
 
    if (CALIBRATE)
        calibrate();
  
    if (KEYBOARD_CONTROL)
        keyboard_control();
 
    if (SENSOR_TEST) 
        while(1)
            sensor_test();
      
    if (FEEDBACK_CONTROL)
        feedback_monitor();
        
    if (STEER_CONTROL)
        steer_control_monitor();
  
    printf("exiting ...\n");
  
    return 0;
}

void steer_control_monitor()
{    
    int w;
    printf("Challenge start. Press 'g' when ready! 'v' to kill!\n");
    while(getchar() != 'g') 
    {
        dist_on_turn = distance;
        sensor_test();
    }
    
    start = 1; //launch experiment
    
    int j = 5000;
    while(1) 
    {     
        if(success == 1) 
        {
            printf("Challenge completed! record_index %d\n", record_index);
            
            printf("Dumping data to file\n");
            
            for(w = 0; w < record_index; w++)
            {
                fprintf(fp,"%d \t %d \t %d \t %d \t %d \t %d \t %d\n ",recorded_time[w],filtered_distance[w],dirty_distance[w],distance_delta[w],throttle[w],rate[w],gain[w]);
            }     
            fclose(fp);
            
            while(1)
                sensor_test();
                
            return;        
        }
        
        j--;
        if(j==0) 
        {
            printf("dist: %d; direction: %d; delta_dist: %d; error %d\n", \
                distance, direction, delta_distance, error);
            j = 5000;
        }
        
        if(getchar() == 'v') //kill the experiment
        {
            start = 2;
            printf("Experiment killed from keyboard\n");
        }       
    }
}


void feedback_monitor()
{    
    int w;
    printf("Challenge start. Press 'g' when ready! 'v' to kill!\n");
    while(getchar() != 'g') 
    {
        sensor_test();
    }
    
    start = 1; //launch experiment
    
    int j = 5000;
    while(1) 
    {     
        if(success == 1) 
        {
            printf("Challenge completed! record_index %d\n", record_index);
            
            printf("Dumping data to file\n");
            
            for(w = 0; w < record_index; w++)
            {
                fprintf(fp,"%d \t %d \t %d \t %d \t %d \t %d \t %d\n ",recorded_time[w],filtered_distance[w],dirty_distance[w],distance_delta[w],throttle[w],rate[w],gain[w]);
            }     
            fclose(fp);
            
            while(1)
                sensor_test();
                
            return;        
        }
        
        j--;
        if(j==0) 
        {
            printf("dist: %d; throt: %d; delta_dist: %d; error %d; gain %d; rate %d\n", \
                distance, current_throttle, delta_distance, error, gain_out, rate_out);
            j = 5000;
        }
        
        if(getchar() == 'v') //kill the experiment
        {
            start = 2;
            printf("Experiment killed from keyboard\n");
        }       
    }
}

int feedback_control() 
{         
    int correction;         
    error = distance - desired_dist;
    int error_abs = fabs(error);
    
    
    //error_in_pwm = (error * 100) >> 7; //error as percentage of 128 (2^7) +. results in 0 to .1% duty cycle
    /*
    if(delta_distance >= 0) 
    {        
        if (error != 0)
            rate_in_pwm =  (delta_distance * 100) / (error_abs);
        else
            rate_in_pwm =   (delta_distance * 100) / 1;
    }
    else 
    {
        rate_in_pwm = 0;  
    }
    
    gain_out = K_gain * error_in_pwm;
    rate_out = K_rate * rate_in_pwm;
       
    correction = gain_out + rate_out; 
    return correction;
    */
    
    static short breaking = 0;
    if (error > breaking_distance && breaking == 0)
        return max_throttle - current_throttle; //speed up as much as u can
    else
    {
        breaking = 1;
        return min_throttle - current_throttle; //full break
    }   
}

void calibrate() 
{
    while(1) 
    {
        printf("Dist: %d, neutral throttle %d, read throt %d \n", distance, neutral_throttle, THROTTLE_R());
        if (calibrated == 1) 
        {
            printf("Calibrated. %d \n", neutral_throttle); 
            break;
        }
        
        int j = 0;
        while (j != 100000) 
        {
            j++;
        }
    }
}

void keyboard_control() 
{
    short steer_incr = 5;
    short throttle_incr = 1;
    printf("a, d, w, s, v: \n");
    char c;
    //fcntl(STDIN_FILENO, F_SETFL, O_NONBLOCK);
  
    for(;;) 
    {    
        switch(c = getchar()) 
        {
            case 'a':   
                left(steer_incr); 
                break;
            case 'd':
                right(steer_incr);   
                break;
            case 'w': 
                forward(throttle_incr);  
                break; 
            case 's':
                backward(throttle_incr);
                break;
            case 'v':
                halt();
                break;  
            case 'q':
                SERVO_W(servo_straight_pwm);
                break;
            case 'e':
                SERVO_W(servo_right_pwm);
                break;
            default:
                continue;      
        }
        printf("Steering: %d; throttle: %d; filt_dist: %d\n", current_steer, current_throttle, distance);
    }
}

void sensor_test() 
{
    delay(1000000);
    printf("D %d; Filtered D %d; T %d; S %d\n", distance_reading, distance, time_msec, speed);
}

int set_throttle(int t, short limit_incr) 
{
    if ( (t > previous_throttle + max_throttle_increase) && limit_incr == 1) 
        current_throttle = previous_throttle + max_throttle_increase;
    else if ( (t < previous_throttle - max_throttle_decrease) && limit_incr == 1) 
        current_throttle = previous_throttle - max_throttle_decrease;
    else current_throttle = t;
    
    if (current_throttle > max_throttle) 
        current_throttle = max_throttle;
    else if (current_throttle < min_throttle) 
        current_throttle = min_throttle;
        
    previous_throttle = current_throttle;
    THROTTLE_W(current_throttle);
    
    return (t - current_throttle);
}

int set_steer(int s, short limit_incr) 
{
    if ( (s > previous_steer + max_steer_increase) && limit_incr == 1) 
        current_steer = previous_steer + max_steer_increase;
    else if ( (s < previous_steer - max_steer_decrease) && limit_incr == 1) 
        current_steer = previous_steer - max_steer_decrease;
    else current_steer = s;
    
    if (current_steer > max_steer) 
        current_steer = max_steer;
    else if (current_steer < min_steer) 
        current_steer = min_steer;
        
    previous_steer = current_steer;
    STEER_W(current_steer);
    
    return (s - current_steer);
}

void forward(int i) 
{
    set_throttle(current_throttle += i, 1);
}

void backward(int i) 
{
    set_throttle(current_throttle -= i, 1);
}

void left(int i) 
{
    set_steer(current_steer -= i, 1);
}

void right(int i) 
{
    set_steer(current_steer += i, 1);   
}

void halt() 
{
    previous_throttle = neutral_throttle;
    previous_steer - neutral_steer;
    set_throttle(current_throttle = neutral_throttle, 0);
    set_steer(current_steer = neutral_steer, 0);
}

void initialize() 
{
    int r = alt_irq_register(SENSOR_IRQ, NULL, (void*) sensor_interrupt_handler);
    printf("registered interrupt handler: %d\n", r);
    
    latest_results[0][5] = 0;
    latest_results[0][4] = 0; 
    latest_results[0][3] = 0;
    latest_results[0][2] = 0;
    latest_results[0][1] = 0;
    latest_results[0][0] = 0;
    latest_results[1][3] = 0;
    latest_results[1][2] = 0;
    latest_results[1][1] = 0;
    latest_results[1][0] = 0;
  
    fp = fopen("/mnt/host/data_log.txt","w");
    fprintf(fp,"time\t filtered_distance\t dirty_distance\t distance_delta\t trottle\t rate\t gain\n");
    
    fcntl(STDIN_FILENO, F_SETFL, O_NONBLOCK); //non-blocking
    
    previous_throttle = neutral_throttle;
    previous_steer = neutral_steer;  
    SERVO_W(servo_straight_pwm);
    halt();
    
    delay(1000000);
}

void delay(int i) 
{
    while(i--);
}
