library IEEE;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;

entity tv_controller is
  port (
    clk           : in std_logic;         -- 50 MHz clock (system)
    reset_n       : in std_logic;
    read          : in std_logic;
    write         : in std_logic;
    chipselect    : in std_logic;
    address       : in unsigned(10 downto 0);
    readdata      : out unsigned(15 downto 0);
    writedata     : in unsigned(15 downto 0);
    clk27         : in std_logic;       -- 27 MHz clock (video)
    td_data       : in unsigned(7 downto 0);
    td_hs         : in std_logic;
    td_vs         : in std_logic
  );

end tv_controller;

architecture rtl of tv_controller is

  component adv_interface is
    port (
      clk27     : in  std_logic;
      reset     : in  std_logic;
      out_data  : out unsigned(31 downto 0);
      data_good : out std_logic;
      field_out : out std_logic;
      hcount    : out unsigned(10 downto 0);
      vcount    : out unsigned(9 downto 0);
      td_data   : in  unsigned(7 downto 0);
      td_hs     : in  std_logic;	
      td_vs     : in  std_logic
    );
  end component;

  component yuv2rgb is
    port (
      yuv   : in unsigned (31 downto 0); 
      rgb   : out unsigned (15 downto 0)
      );
  end component;


  component line_buffer is
    port (
      reset           : in std_logic;
      wclk	      : in std_logic;
      rclk	      : in std_logic;
      write_enable    : in std_logic;
      write_data      : in unsigned(15 downto 0);
      write_address   : in unsigned(8 downto 0);
      read_data       : out unsigned(15 downto 0);
      read_address    : in unsigned(8 downto 0);
      page            : in std_logic
    );
  end component;

  -- the 32 bits at a time (in YUV format) that the interface sends out
  signal yuv : unsigned(31 downto 0);
  -- the converted 16 RGB bits
  signal rgb : unsigned(15 downto 0);
  -- tells the line buffer when to accept the data from the video interface
  signal data_good : std_logic;

  signal hcount : unsigned(15 downto 0);
  signal vcount : unsigned(15 downto 0);
  
  signal buffer_out : unsigned(15 downto 0);
  signal field_out : unsigned(15 downto 0);

begin

  adv_interface0 : adv_interface port map (
    clk27     => clk27,
    reset     => not reset_n,
    out_data  => yuv,
    data_good => data_good,
    field_out => field_out(0),
    hcount    => hcount(10 downto 0),
    vcount    => vcount(9 downto 0),
    td_data   => td_data,
    td_hs     => td_hs,
    td_vs     => td_vs
  );

  yuv2rgb0 : yuv2rgb port map (
    yuv => yuv,
    rgb => rgb
  );

  line_buffer0 : line_buffer port map (
    reset         => not reset_n,
    wclk          => clk27,
    rclk          => clk,
    write_enable  => data_good,
    write_data    => rgb,
    write_address => hcount(10 downto 2),
    read_data     => buffer_out,
    read_address  => address(8 downto 0),
    page          => vcount(0)        -- flip the buffers in each
                                      -- vertical line
  );

  readdata <= buffer_out when address(10 downto 9) = "00" else
              field_out when address(10 downto 9) = "01" else
              hcount when address(10 downto 9) = "10" else
              vcount;
end rtl;



    
